///////////////////////////////////////////////////////////////////////////////
// $Id: mapdocument.cpp,v 1.12 2004/11/30 15:37:18 krake Exp $
//
// Package:   MOAGG Edit - Level Editor for MOAGG
// Copyright: Kevin Krammer, 2003
//
///////////////////////////////////////////////////////////////////////////////
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License version 2 as
// published by the Free Software Foundation.
//
///////////////////////////////////////////////////////////////////////////////

/*! \file    mapdocument.cpp
    \author  Kevin Krammer, kevin.krammer@gmx.at
    \brief   Implementation of the map document class
*/

// Qt includes
#include <qfile.h>
#include <qpainter.h>
#include <qpixmap.h>

// local includes
#include "mapdocument.h"
#include "maplayer.h"
#include "tilegroup.h"
#include "tileset.h"

///////////////////////////////////////////////////////////////////////////////

MapDocument::MapDocument() :
	m_fileVersion(MapLayer::MAX_VERSION),
	m_modified(false),
	m_rows(0),
	m_cols(0),
	m_tileSet(0),
	m_currentLayerIndex(0),
	m_cachePix(0)
{
}

///////////////////////////////////////////////////////////////////////////////

MapDocument::~MapDocument()
{
	delete m_tileSet;
}

///////////////////////////////////////////////////////////////////////////////

bool MapDocument::init(const QString& tilesetfile)
{
	if (m_tileSet != 0) return false; // already initialized
	m_tileSet = new TileSet();
	return m_tileSet->load(tilesetfile);
}

///////////////////////////////////////////////////////////////////////////////

bool MapDocument::init(const TileSet& tileset)
{
	if (m_tileSet != 0) return false; // already initialized

	m_tileSet = new TileSet(tileset);

	return true;
}

///////////////////////////////////////////////////////////////////////////////

bool MapDocument::init(int rows, int cols, const QString& tilesetfile)
{
	if (!init(tilesetfile)) return false;

	if (rows <= 0 || cols <= 0) return false;

	m_rows = rows;
	m_cols = cols;

	return true;
}

///////////////////////////////////////////////////////////////////////////////

bool MapDocument::openFile(const QString& filename)
{
	qDebug("MapDocument::openFile(%s)", (const char*) filename);
	if (filename.isEmpty()) return false;
	if (m_tileSet == 0) return false;

	QFile file(filename);
	if (!file.open(IO_ReadOnly)) return false;

	bool result = readFromDevice(&file);

	file.close();

	if (result)
		m_fileName = filename;

	return result;
}

///////////////////////////////////////////////////////////////////////////////

bool MapDocument::saveFile()
{
	if (m_fileName.isEmpty()) return false;

	return saveFileAs(m_fileName);
}

///////////////////////////////////////////////////////////////////////////////

bool MapDocument::saveFileAs(const QString& filename)
{
	qDebug("MapDocument::saveFileAs(%s)", (const char*) filename);
	if (filename.isEmpty()) return false;

	if (m_layers.count() == 0) return false;

	QFile file(filename);
	if (!file.open(IO_WriteOnly)) return false;

	bool result = writeToDevice(&file);

	if (result)
	{
		clearModifies();
		m_fileName = filename;
	}

	file.close();

	return result;
}

///////////////////////////////////////////////////////////////////////////////

bool MapDocument::readFromDevice(QIODevice* device)
{
	if (device == 0) return false;

	int version = device->getch();
	if (version < 1 || version > MapLayer::MAX_VERSION)
	{
		qWarning("Mapfile version %d not supported", version);
		return false;
	}
	qDebug("Mapfile version %d", version);

	char header[5];
	if (device->readBlock(header, 5) != 5)
	{
		return false;
	}

	int cols = ((header[0] & 0xFF) << 8) + (header[1] & 0xFF);
	int rows = ((header[2] & 0xFF) << 8) + (header[3] & 0xFF);
	int layers = header[4];

	qDebug("Header: 0x%x, 0x%x, 0x%x, 0x%x, 0x%x",
	       0xFF & header[0], 0xFF & header[1],
	       0xFF & header[2], 0xFF & header[3], 0xFF & header[4]);
	qDebug("cols: %d, rows: %d, layers: %d", cols, rows, layers);

	long expectedFileSize = MapLayer::byteSize(version, rows, cols) * layers + 6;
	if (device->size() != static_cast<QIODevice::Offset>(expectedFileSize))
	{
		qWarning("Buffersize(%ld) differs from expected size(%ld)",
		         device->size(), expectedFileSize);
		return false;
	}

	m_layers.clear();
	MapLayer* layer = 0;
	for (int i = 0; i < layers; ++i)
	{
		layer = new MapLayer();
		layer->init(rows, cols);

		int groupIndex = device->getch();
		if (groupIndex != -1)
		{
			uint index = static_cast<uint>(groupIndex);

			TileGroup* group = m_tileSet->group(index);
			if (group == 0)
			{
				m_layers.clear();
				return false;
			}
			layer->setTileGroup(group);
			layer->setGroupIndex(index);

			if (!layer->readFromDevice(device, version))
			{
				m_layers.clear();
				return false;
			}
		}
		m_layers.append(layer);
	}
	
	// set version to current version used by layers
	version = MapLayer::MAX_VERSION;

	m_fileVersion = static_cast<uchar>(version & 0xFF);
	m_rows = rows;
	m_cols = cols;

	return true;
}

///////////////////////////////////////////////////////////////////////////////

bool MapDocument::writeToDevice(QIODevice* device) const
{
	if (device == 0) return false;

	QPtrListIterator<MapLayer> it(m_layers);

	char header[5];
	int rows = it.current()->numRows();
	int cols = it.current()->numCols();

	header[0] = static_cast<char>((cols >> 8) & 0xFF);
	header[1] = static_cast<char>(cols & 0xFF);
	header[2] = static_cast<char>((rows >> 8) & 0xFF);
	header[3] = static_cast<char>(rows & 0xFF);
	header[4] = static_cast<char>(m_layers.count() & 0xFF);

	qDebug("cols: %d, rows: %d, layers: %d", cols, rows, m_layers.count());
	qDebug("Header: 0x%x, 0x%x, 0x%x, 0x%x, 0x%x",
	       0xFF & header[0], 0xFF & header[1],
	       0xFF & header[2], 0xFF & header[3], 0xFF & header[4]);

	if (device->putch(m_fileVersion) == -1) return false;

	if (device->writeBlock(header, 5) != 5) return false;

	for (; it.current() != 0; ++it)
	{
		if (device->putch(static_cast<char>(it.current()->groupIndex())) == -1)
			return false;

		if (!it.current()->writeToDevice(device)) return false;
	}

	return true;
}

///////////////////////////////////////////////////////////////////////////////

void MapDocument::setCurrentLayer(uint index)
{
	if (index >= m_layers.count()) return;

	m_currentLayerIndex = index;
}

///////////////////////////////////////////////////////////////////////////////

int MapDocument::groupForLayer(uint index) const
{
	if (index >= m_layers.count()) return -1;

	QPtrListIterator<MapLayer> it(m_layers);
	it += index;
	if (it.current() == 0) return -1;

	return it.current()->groupIndex();
}

///////////////////////////////////////////////////////////////////////////////

void MapDocument::insertLayer(uint groupindex, int layerindex)
{
	// background always first
	if (groupindex == 0) layerindex = 0;

	if (layerindex < 0) layerindex = m_layers.count();
	if (static_cast<uint>(layerindex) >= m_tileSet->count()) return;

	// allow only one background layer
	if (groupindex == 0 && groupForLayer(0) == 0) return;

	TileGroup* group = m_tileSet->group(groupindex);
	if (group == 0) return;

	MapLayer* layer = new MapLayer();
	layer->setGroupIndex(groupindex);
	layer->setTileGroup(group);
	layer->init(numRows(), numCols());

	m_layers.insert(layerindex, layer);
	
	if (static_cast<uint>(layerindex) >= m_currentLayerIndex)
		m_currentLayerIndex++;

	m_modified = true;
}

///////////////////////////////////////////////////////////////////////////////

void MapDocument::removeLayer(uint layerindex)
{
	if (layerindex >= m_layers.count()) return;

	m_layers.remove(layerindex);
	if (layerindex == m_currentLayerIndex)
		m_currentLayerIndex = 0;

	m_modified = true;
}

///////////////////////////////////////////////////////////////////////////////

void MapDocument::moveLayer(uint fromindex, uint toindex)
{
	if (fromindex >= m_layers.count() || toindex >= m_layers.count()) return;

	m_layers.insert(toindex, m_layers.take(fromindex));
	if (fromindex == m_currentLayerIndex)
		m_currentLayerIndex = toindex;

	m_modified = true;
}

///////////////////////////////////////////////////////////////////////////////

bool MapDocument::isLayerVisible(uint layerindex) const
{
	if (layerindex >= m_layers.count()) return false;

	QPtrListIterator<MapLayer> it(m_layers);
	it += layerindex;

	if (it.current() != 0)
		return it.current()->isVisible();
	else
		return false;
}

///////////////////////////////////////////////////////////////////////////////

void MapDocument::setLayerVisible(uint layerindex, bool on)
{
	if (layerindex >= m_layers.count()) return;

	if (m_layers.at(layerindex) != 0)
		m_layers.at(layerindex)->setVisible(on);
}

///////////////////////////////////////////////////////////////////////////////

bool MapDocument::isLayerBackground(uint layerindex) const
{
	if (layerindex >= m_layers.count()) return false;

	QPtrListIterator<MapLayer> it(m_layers);
	it += layerindex;

	if (it.current() != 0)
		return it.current()->isBackground();
	else
		return false;
}

///////////////////////////////////////////////////////////////////////////////

void MapDocument::setLayerBackground(uint layerindex, bool on)
{
	if (layerindex >= m_layers.count()) return;

	if (m_layers.at(layerindex) != 0)
		m_layers.at(layerindex)->setBackground(on);
}

///////////////////////////////////////////////////////////////////////////////

void MapDocument::setLayerImageParameters(int alpha, int brightness)
{
	if (m_layers.count() == 0) return;

	if (m_layers.at(m_currentLayerIndex) != 0)
		m_layers.at(m_currentLayerIndex)->setImageParameters(alpha, brightness);
}

///////////////////////////////////////////////////////////////////////////////

void MapDocument::getLayerImageParameters(int& alpha, int& brightness) const
{
	if (m_layers.count() == 0) return;

	QPtrListIterator<MapLayer> it(m_layers);
	it += m_currentLayerIndex;

	if (it.current() != 0)
		it.current()->getImageParameters(alpha, brightness);
}

///////////////////////////////////////////////////////////////////////////////

void MapDocument::setCurrentTile(uint tileindex)
{
	if (m_layers.count() == 0) return;

	m_layers.at(m_currentLayerIndex)->setCurrentTile(tileindex);
}

///////////////////////////////////////////////////////////////////////////////

void MapDocument::drawCurrentTileAt(int row, int col)
{
	if (m_layers.count() == 0) return;

	m_layers.at(m_currentLayerIndex)->drawTile(row, col);

	m_modified = m_modified || m_layers.at(m_currentLayerIndex)->isModified();
}

///////////////////////////////////////////////////////////////////////////////

QPixmap* MapDocument::pixmap(int row, int col)
{
	if (row < 0 || col < 0 || row >= m_rows || col >= m_cols) return 0;

	if (m_tileSet == 0) return 0;

	if (m_cachePix == 0)
	{
		m_cachePix = new QPixmap(16, 16);
	}

	QPainter painter;
	painter.begin(m_cachePix);
	painter.fillRect(0, 0, m_cachePix->width(), m_cachePix->height(), Qt::black);
	painter.end();

	for (uint l = 0; l < m_layers.count(); ++l)
	{
		QPixmap* pix = m_layers.at(l)->pixmap(row, col);
		if (pix != 0)
		{
			bitBlt(m_cachePix, 0, 0,
			       pix, 0, 0, pix->width(), pix->height(),
			       Qt::CopyROP);
		}
	}

	return m_cachePix;
}

///////////////////////////////////////////////////////////////////////////////

void MapDocument::resize(int rows, int cols)
{
	if (rows == m_rows && cols == m_cols) return;
	if (rows <= 0 || cols <= 0) return;

	for (uint i = 0; i < m_layers.count(); ++i)
	{
		MapLayer* layer = m_layers.at(i);
		if (layer == 0) continue;

		layer->resize(rows, cols);
	}

	m_modified = true;
	m_rows = rows;
	m_cols = cols;
}

///////////////////////////////////////////////////////////////////////////////

bool MapDocument::copyData(const QRect& area, int layer,
	                       const MapDocument& source, const QPoint& destpoint,
	                       bool createlayer, bool allowresize)
{
	// sanity checks
	if (source.layerCount() == 0) return false; // no layer -> no data
	if (source.numRows() == 0 || source.numCols() == 0) return false; // no size
	if (layer >= static_cast<int>(source.layerCount())) return false;
	if (destpoint.x() < 0 || destpoint.y() < 0) return false;

	QRect sourceRect = QRect(0, 0, source.numCols(), source.numRows());
	if (!area.isNull() && sourceRect.contains(area))
		sourceRect = area;

	uint sourceStartLayer = layer < 0 ? 0 : layer;
	uint sourceEndLayer = layer < 0 ? source.layerCount() - 1 : layer;

	if (layerCount() < (sourceEndLayer - sourceStartLayer + 1)
		&& !createlayer) return false;

	QRect destRect = sourceRect;
	destRect.moveTopLeft(destpoint);

	QRect rect(0, 0, numCols(), numRows());
	if (!rect.contains(destRect))
	{
		if (allowresize)
		{
			int r = QMAX(numRows(), destRect.bottom() + 1);
			int c = QMAX(numCols(), destRect.right() + 1);
			resize(r, c);
		}
		else
		{
			destRect = rect.intersect(destRect);
			sourceRect.setWidth(destRect.width());
			sourceRect.setHeight(destRect.height());
		}
	}

	// copy data
	uint destLayerIndex = currentLayer();
	QPtrListIterator<MapLayer> it(source.m_layers);
	it += sourceStartLayer;
	for(uint l = sourceStartLayer; l <= sourceEndLayer; ++it, ++l, ++destLayerIndex)
	{
		MapLayer* sourceLayer = it.current();
		MapLayer* destLayer = 0;
		if (destLayerIndex == layerCount())
		{
			// create new layer
			int destGroup = source.groupForLayer(l);
			if (destGroup < 0 || destGroup >= static_cast<int>(m_tileSet->count()))
				destGroup = m_tileSet->count() - 1;
                
			// insertLayer creates a new layer if destLayerIndex is not
			// valid yet
			insertLayer(destGroup, destLayerIndex);
			destLayer = m_layers.at(destLayerIndex);
		}
		else
			destLayer = m_layers.at(destLayerIndex);

		if (sourceLayer == 0 || destLayer == 0) return false;

		if (!destLayer->copyData(sourceRect, *sourceLayer, destRect.topLeft()))
			return false;
	}

	return true;
}

///////////////////////////////////////////////////////////////////////////////

void MapDocument::clearModifies()
{
	m_modified = false;

	QPtrListIterator<MapLayer> it(m_layers);
	for (; it.current() != 0; ++it)
		it.current()->clearModified();
}

// End of file
