#include <stack>
#include <list>
#include <sstream>

#include <expat.h>

#include "Tools.h"
#include "StringTools.h"
#include "File.h"

#include "XMLException.h"
#include "XMLTools.h"
#include "XMLElements.h"
#include "XMLVisitors.h"
#include "XMLParser.h"


//============================================================================
// Implementation of XMLException.h
//============================================================================

//----------------------------------------------------------------------------
XMLException::XMLException(const char *error)
        : Exception(error)
{
}

//----------------------------------------------------------------------------
XMLException::XMLException(const std::string &error)
        : Exception(error)
{
}

//----------------------------------------------------------------------------
XMLException::~XMLException()
{
}



//============================================================================
// Implementation of XMLTools.h
//============================================================================

//----------------------------------------------------------------------------
bool XML_TOOLS::popUnsignedFromPropertyTokenString(XMLProperty *p,
                                                   const unsigned value)
{
    std::string s = p->getValue();
    bool rc = STRING_TOOLS::popUnsignedFromTokenString(s, value);
    if (rc)
    {
        p->setValue(s);
    }

    return rc;
}

//----------------------------------------------------------------------------
bool XML_TOOLS::popFirstUnsignedFromPropertyTokenString(XMLProperty *p,
                                                        const unsigned value)
{
    std::string s = p->getValue();
    bool rc = STRING_TOOLS::popFirstUnsignedFromTokenString(s, value);
    if (rc)
    {
        p->setValue(s);
    }

    return rc;
}



//============================================================================
// Implementation of XMLElements.h
//============================================================================

//----------------------------------------------------------------------------
XMLElement::~XMLElement()
{
}



//----------------------------------------------------------------------------
XMLProperty::XMLProperty(const std::string &name, const bool value)
        : XMLElement(name)
{
    setValue(value);
}

//----------------------------------------------------------------------------
XMLProperty::XMLProperty(const std::string &name, const int value)
        : XMLElement(name)
{
    setValue(value);
}

//----------------------------------------------------------------------------
XMLProperty::XMLProperty(const std::string &name, const unsigned value)
        : XMLElement(name)
{
    setValue(value);
}

//----------------------------------------------------------------------------
XMLProperty::XMLProperty(const std::string &name, const char *value)
        : XMLElement(name)
{
    setValue(value);
}

//----------------------------------------------------------------------------
XMLProperty::XMLProperty(const std::string &name, const std::string &value)
        : XMLElement(name)
{
    setValue(value);
}

//----------------------------------------------------------------------------
XMLProperty::~XMLProperty()
{
}

//----------------------------------------------------------------------------
XMLProperty *XMLProperty::clone() const
{
    return new XMLProperty(getName(), m_value);
}


//----------------------------------------------------------------------------
void XMLProperty::setValue(const bool value)
{
    m_value = value ? "1" : "0";
}

//----------------------------------------------------------------------------
void XMLProperty::setValue(const int value)
{
    std::ostringstream s;
    s << value;
    m_value = s.str();
}

//----------------------------------------------------------------------------
void XMLProperty::setValue(const unsigned value)
{
    std::ostringstream s;
    s << value;
    m_value = s.str();
}

//----------------------------------------------------------------------------
void XMLProperty::setValue(const char *value)
{
    m_value = value;
}

//----------------------------------------------------------------------------
void XMLProperty::setValue(const std::string &value)
{
    m_value = value;
}


//----------------------------------------------------------------------------
bool XMLProperty::getAsBool() const
{
    try
    {
        return STRING_TOOLS::toBool(getValue());
    }
    catch (Exception &e)
    {
        throw XMLException(
            std::string("The property '")
            .append(getName()).append("' is not a boolean"));
    }
}

//----------------------------------------------------------------------------
int XMLProperty::getAsInt() const
{
    try
    {
        return STRING_TOOLS::toInt(getValue());
    }
    catch (Exception &e)
    {
        throw XMLException(
            std::string("The property '")
            .append(getName()).append("' is not an integer"));
    }
}

//----------------------------------------------------------------------------
unsigned XMLProperty::getAsUnsigned() const
{
    try
    {
        return STRING_TOOLS::toUnsigned(getValue());
    }
    catch (Exception &e)
    {
        throw XMLException(
            std::string("The property '")
            .append(getName()).append("' is not an integer"));
    }
}

//----------------------------------------------------------------------------
DECLARE_XML_VISITOR_API_BODY(XMLProperty)



//----------------------------------------------------------------------------
XMLText::XMLText(const std::string &text)
        : XMLElement(), m_text(text)
{
}

//----------------------------------------------------------------------------
XMLText::~XMLText()
{
}

//----------------------------------------------------------------------------
DECLARE_XML_VISITOR_API_BODY(XMLText)



//----------------------------------------------------------------------------
class XMLNode::PImpl
{
public:
    //------------------------------------------------------------------------
    typedef std::list<XMLProperty*> XMLProperties;
    typedef XMLProperties::iterator XMLPropertyIter;
    typedef XMLProperties::const_iterator XMLPropertyCIter;

    //------------------------------------------------------------------------
    typedef std::list<XMLText*> XMLTexts;
    typedef XMLTexts::iterator XMLTextIter;
    typedef XMLTexts::const_iterator XMLTextCIter;

    //------------------------------------------------------------------------
    typedef std::list<XMLNode*> XMLNodes;
    typedef XMLNodes::iterator XMLNodeIter;
    typedef XMLNodes::const_iterator XMLNodeCIter;

    //------------------------------------------------------------------------
    typedef std::list<XMLElement*> XMLChilds;
    typedef XMLChilds::iterator XMLChildIter;
    typedef XMLChilds::const_iterator XMLChildCIter;


    //------------------------------------------------------------------------
    PImpl() {}
    ~PImpl() { clear(); }

    //------------------------------------------------------------------------
    void clear();
    inline bool empty() const { return m_childs.empty(); }

    //------------------------------------------------------------------------
    void addProperty(XMLProperty *p);
    void removeProperty(const XMLProperty *p);
    XMLProperty *getMandatoryProperty(const std::string &name);
    const XMLProperty *getMandatoryProperty(const std::string &name) const;
    XMLProperty *getProperty(const std::string &name);
    const XMLProperty *getProperty(const std::string &name) const;
    bool hasProperty(const std::string &name) const;
    inline bool hasProperties() const { return !m_properties.empty(); }
    inline size_t getNumberOfProperties() const { return m_properties.size(); }

    //------------------------------------------------------------------------
    void addText(XMLText *t);
    void removeText(const XMLText *t);
    inline bool hasText() const { return !m_texts.empty(); }
    const std::string &getText() const;

    //------------------------------------------------------------------------
    void addNode(XMLNode *n);
    void removeNode(const XMLNode *n);
    XMLNode *getMandatoryNode(const std::string &name);
    const XMLNode *getMandatoryNode(const std::string &name) const;
    XMLNode *getNode(const std::string &name);
    const XMLNode *getNode(const std::string &name) const;
    inline XMLNode *getFirstNode()
    {
        return !m_nodes.empty() ? *m_nodes.begin() : NULL;
    }
    inline const XMLNode *getFirstNode() const
    {
        return !m_nodes.empty() ? *m_nodes.begin() : NULL;
    }
    XMLNode *popNode(const std::string &name);
    XMLNode *popFirstNode();
    bool hasNode(const std::string &name) const;
    inline bool hasNodes() const { return !m_nodes.empty(); }
    inline size_t getNumberOfNodes() const { return m_nodes.size(); }

    //------------------------------------------------------------------------
    const std::string &getStringProperty(const std::string &name) const;
    const std::string &getStringProperty(const std::string &name,
                                         const std::string &defaultValue) const;
    bool getBoolProperty(const std::string &name) const;
    bool getBoolProperty(const std::string &name, bool defaultValue) const;
    int getIntProperty(const std::string &name) const;
    int getIntProperty(const std::string &name, int defaultValue) const;
    unsigned getUnsignedProperty(const std::string &name) const;
    unsigned getUnsignedProperty(const std::string &name,
                                 unsigned defaultValue) const;

    //------------------------------------------------------------------------
    void acceptAllProperties(XMLVisitor &v);
    void acceptAllProperties(XMLConstVisitor &v) const;
    void acceptAllNodes(XMLVisitor &v);
    void acceptAllNodes(XMLConstVisitor &v) const;
    void acceptAllChilds(XMLVisitor &v);
    void acceptAllChilds(XMLConstVisitor &v) const;

private:
    //------------------------------------------------------------------------
    void removeChild(const XMLElement *e);

    //------------------------------------------------------------------------
    XMLProperties m_properties;
    XMLTexts m_texts;
    XMLNodes m_nodes;
    XMLChilds m_childs;
};


//----------------------------------------------------------------------------
XMLNode::XMLNode()
{
    m_pImpl = new PImpl();
}

//----------------------------------------------------------------------------
XMLNode::XMLNode(const std::string &name) : XMLElement(name)
{
    m_pImpl = new PImpl();
}

//----------------------------------------------------------------------------
XMLNode::~XMLNode()
{
    ZAP_POINTER(m_pImpl);
}

//----------------------------------------------------------------------------
void XMLNode::clear()
{
    m_pImpl->clear();
}

//----------------------------------------------------------------------------
void XMLNode::PImpl::clear()
{
    for (XMLPropertyIter
             iter = m_properties.begin(),
             end = m_properties.end();
         iter != end; ++iter)
    {
        delete *iter;
    }
    m_properties.clear();

    for (XMLNodeIter
             iter = m_nodes.begin(),
             end = m_nodes.end();
         iter != end; ++iter)
    {
        delete *iter;
    }
    m_nodes.clear();

    for (XMLTextIter
             iter = m_texts.begin(),
             end = m_texts.end();
         iter != end; ++iter)
    {
        delete *iter;
    }
    m_texts.clear();

    m_childs.clear();
}

//----------------------------------------------------------------------------
bool XMLNode::empty() const
{
    return m_pImpl->empty();
}

//----------------------------------------------------------------------------
void XMLNode::addProperty(XMLProperty *p)
{
    m_pImpl->addProperty(p);
}

//----------------------------------------------------------------------------
void XMLNode::PImpl::addProperty(XMLProperty *p)
{
    m_properties.push_back(p);
}

//----------------------------------------------------------------------------
void XMLNode::removeProperty(const XMLProperty *p)
{
    m_pImpl->removeProperty(p);
}

//----------------------------------------------------------------------------
void XMLNode::PImpl::removeProperty(const XMLProperty *p)
{
    for (XMLPropertyIter
             iter = m_properties.begin(),
             end = m_properties.end();
         iter != end; ++iter)
    {
        if (*iter == p)
        {
            m_properties.erase(iter);
            break;
        }
    }
}

//----------------------------------------------------------------------------
XMLProperty *XMLNode::getMandatoryProperty(const std::string &name)
{
    return m_pImpl->getMandatoryProperty(name);
}

//----------------------------------------------------------------------------
XMLProperty *XMLNode::PImpl::getMandatoryProperty(const std::string &name)
{
    XMLProperty *prop = getProperty(name);
    if (!prop)
    {
        throw XMLException(
            std::string("The mandatory property '")
            .append(name).append("' is missing"));
    }

    return prop;
}

//----------------------------------------------------------------------------
const XMLProperty *XMLNode::getMandatoryProperty(const std::string &name) const
{
    const XMLProperty *prop = getProperty(name);
    if (!prop)
    {
        throw XMLException(
            std::string("The mandatory property '")
            .append(name).append("' is missing"));
    }

    return prop;
}

//----------------------------------------------------------------------------
XMLProperty *XMLNode::getProperty(const std::string &name)
{
    return m_pImpl->getProperty(name);
}

//----------------------------------------------------------------------------
XMLProperty *XMLNode::PImpl::getProperty(const std::string &name)
{
    for (XMLPropertyCIter
             iter = m_properties.begin(),
             end = m_properties.end();
         iter != end; ++iter)
    {
        if ((*iter)->getName() == name)
        {
            return *iter;
        }
    }

    return NULL;
}

//----------------------------------------------------------------------------
const XMLProperty *XMLNode::getProperty(const std::string &name) const
{
    return m_pImpl->getProperty(name);
}

//----------------------------------------------------------------------------
const XMLProperty *XMLNode::PImpl::getProperty(const std::string &name) const
{
    for (XMLPropertyCIter
             iter = m_properties.begin(),
             end = m_properties.end();
         iter != end; ++iter)
    {
        if ((*iter)->getName() == name)
        {
            return *iter;
        }
    }

    return NULL;
}

//----------------------------------------------------------------------------
bool XMLNode::hasProperty(const std::string &name) const
{
    return m_pImpl->hasProperty(name);
}

//----------------------------------------------------------------------------
bool XMLNode::PImpl::hasProperty(const std::string &name) const
{
    for (XMLPropertyCIter
             iter = m_properties.begin(),
             end = m_properties.end();
         iter != end; ++iter)
    {
        if ((*iter)->getName() == name)
        {
            return true;
        }
    }

    return false;
}

//----------------------------------------------------------------------------
bool XMLNode::hasProperties() const
{
    return m_pImpl->hasProperties();
}

//----------------------------------------------------------------------------
size_t XMLNode::getNumberOfProperties() const
{
    return m_pImpl->getNumberOfProperties();
}


//----------------------------------------------------------------------------
void XMLNode::addText(XMLText *t)
{
    m_pImpl->addText(t);
}

//----------------------------------------------------------------------------
void XMLNode::PImpl::addText(XMLText *t)
{
    m_texts.push_back(t);
    m_childs.push_back(t);
}

//----------------------------------------------------------------------------
void XMLNode::removeText(const XMLText *t)
{
    m_pImpl->removeText(t);
}

//----------------------------------------------------------------------------
void XMLNode::PImpl::removeText(const XMLText *t)
{
    for (XMLTextIter
             iter = m_texts.begin(),
             end = m_texts.end();
         iter != end; ++iter)
    {
        if (*iter == t)
        {
            m_texts.erase(iter);
            break;
        }
    }

    removeChild(t);
}

//----------------------------------------------------------------------------
bool XMLNode::hasText() const
{
    return m_pImpl->hasText();
}

//----------------------------------------------------------------------------
const std::string &XMLNode::getText() const
{
    return m_pImpl->getText();
}

//----------------------------------------------------------------------------
const std::string &XMLNode::PImpl::getText() const
{
    for (XMLTextCIter
             iter = m_texts.begin(),
             end = m_texts.end();
         iter != end; ++iter)
    {
        return (*iter)->getText();
    }

    throw XMLException("The mandatory text is missing");
}


//----------------------------------------------------------------------------
void XMLNode::addNode(XMLNode *n)
{
    m_pImpl->addNode(n);
}

//----------------------------------------------------------------------------
void XMLNode::PImpl::addNode(XMLNode *n)
{
    m_nodes.push_back(n);
    m_childs.push_back(n);
}

//----------------------------------------------------------------------------
void XMLNode::removeNode(const XMLNode *n)
{
    m_pImpl->removeNode(n);
}

//----------------------------------------------------------------------------
void XMLNode::PImpl::removeNode(const XMLNode *n)
{
    for (XMLNodeIter
             iter = m_nodes.begin(),
             end = m_nodes.end();
         iter != end; ++iter)
    {
        if (*iter == n)
        {
            m_nodes.erase(iter);
            break;
        }
    }

    removeChild(n);
}

//----------------------------------------------------------------------------
XMLNode *XMLNode::getMandatoryNode(const std::string &name)
{
    return m_pImpl->getMandatoryNode(name);
}

//----------------------------------------------------------------------------
XMLNode *XMLNode::PImpl::getMandatoryNode(const std::string &name)
{
    XMLNode *node = getNode(name);
    if (!node)
    {
        throw XMLException(
            std::string("The mandatory node '")
            .append(name).append("' is missing"));
    }

    return node;
}

//----------------------------------------------------------------------------
const XMLNode *XMLNode::getMandatoryNode(const std::string &name) const
{
    return m_pImpl->getMandatoryNode(name);
}

//----------------------------------------------------------------------------
const XMLNode *XMLNode::PImpl::getMandatoryNode(const std::string &name) const
{
    const XMLNode *node = getNode(name);
    if (!node)
    {
        throw XMLException(
            std::string("The mandatory node '")
            .append(name).append("' is missing"));
    }

    return node;
}

//----------------------------------------------------------------------------
XMLNode *XMLNode::getNode(const std::string &name)
{
    return m_pImpl->getNode(name);
}

//----------------------------------------------------------------------------
XMLNode *XMLNode::PImpl::getNode(const std::string &name)
{
    for (XMLNodeCIter
             iter = m_nodes.begin(),
             end = m_nodes.end();
         iter != end; ++iter)
    {
        if ((*iter)->getName() == name)
        {
            return *iter;
        }
    }

    return NULL;
}

//----------------------------------------------------------------------------
const XMLNode *XMLNode::getNode(const std::string &name) const
{
    return m_pImpl->getNode(name);
}

//----------------------------------------------------------------------------
const XMLNode *XMLNode::PImpl::getNode(const std::string &name) const
{
    for (XMLNodeCIter
             iter = m_nodes.begin(),
             end = m_nodes.end();
         iter != end; ++iter)
    {
        if ((*iter)->getName() == name)
        {
            return *iter;
        }
    }

    return NULL;
}

//----------------------------------------------------------------------------
XMLNode *XMLNode::getFirstNode()
{
    return m_pImpl->getFirstNode();
}

//----------------------------------------------------------------------------
const XMLNode *XMLNode::getFirstNode() const
{
    return m_pImpl->getFirstNode();
}

//----------------------------------------------------------------------------
XMLNode *XMLNode::popNode(const std::string &name)
{
    return m_pImpl->popNode(name);
}

//----------------------------------------------------------------------------
XMLNode *XMLNode::PImpl::popNode(const std::string &name)
{
    XMLNode *node = getNode(name);
    if (node)
    {
        removeNode(node);
    }

    return node;
}

//----------------------------------------------------------------------------
XMLNode *XMLNode::popFirstNode()
{
    return m_pImpl->popFirstNode();
}

//----------------------------------------------------------------------------
XMLNode *XMLNode::PImpl::popFirstNode()
{
    XMLNode *node = getFirstNode();
    if (node)
    {
        removeNode(node);
    }

    return node;
}

//----------------------------------------------------------------------------
bool XMLNode::hasNode(const std::string &name) const
{
    return m_pImpl->hasNode(name);
}

//----------------------------------------------------------------------------
bool XMLNode::PImpl::hasNode(const std::string &name) const
{
    for (XMLNodeCIter
             iter = m_nodes.begin(),
             end = m_nodes.end();
         iter != end; ++iter)
    {
        if ((*iter)->getName() == name)
        {
            return true;
        }
    }

    return false;
}

//----------------------------------------------------------------------------
bool XMLNode::hasNodes() const
{
    return m_pImpl->hasNodes();
}

//----------------------------------------------------------------------------
size_t XMLNode::getNumberOfNodes() const
{
    return m_pImpl->getNumberOfNodes();
}


//----------------------------------------------------------------------------
const std::string &XMLNode::getStringProperty(const std::string &name) const
{
    return m_pImpl->getStringProperty(name);
}

//----------------------------------------------------------------------------
const std::string &XMLNode::PImpl::getStringProperty(const std::string &name) const
{
    const XMLProperty *prop = getProperty(name);
    if (!prop)
    {
        throw XMLException(
            std::string("The mandatory attribute '")
            .append(name).append("' is missing"));
    }

    return prop->getValue();
}

//----------------------------------------------------------------------------
const std::string &XMLNode::getStringProperty(const std::string &name,
                                              const std::string &defaultValue) const
{
    return m_pImpl->getStringProperty(name, defaultValue);
}

//----------------------------------------------------------------------------
const std::string &XMLNode::PImpl::getStringProperty(const std::string &name,
                                                     const std::string &defaultValue) const
{
    const XMLProperty *prop = getProperty(name);
    return prop ? prop->getValue() : defaultValue;
}

//----------------------------------------------------------------------------
bool XMLNode::getBoolProperty(const std::string &name) const
{
    return m_pImpl->getBoolProperty(name);
}

//----------------------------------------------------------------------------
bool XMLNode::PImpl::getBoolProperty(const std::string &name) const
{
    const XMLProperty *prop = getProperty(name);
    if (!prop)
    {
        throw XMLException(
            std::string("The mandatory attribute '")
            .append(name).append("' is missing"));
    }

    return prop->getAsBool();
}

//----------------------------------------------------------------------------
bool XMLNode::getBoolProperty(const std::string &name,
                              const bool defaultValue) const
{
    return m_pImpl->getBoolProperty(name, defaultValue);
}

//----------------------------------------------------------------------------
bool XMLNode::PImpl::getBoolProperty(const std::string &name,
                                     const bool defaultValue) const
{
    const XMLProperty *prop = getProperty(name);
    return prop ? prop->getAsBool() : defaultValue;
}

//----------------------------------------------------------------------------
int XMLNode::getIntProperty(const std::string &name) const
{
    return m_pImpl->getIntProperty(name);
}

//----------------------------------------------------------------------------
int XMLNode::PImpl::getIntProperty(const std::string &name) const
{
    const XMLProperty *prop = getProperty(name);
    if (!prop)
    {
        throw XMLException(
            std::string("The mandatory attribute '")
            .append(name).append("' is missing"));
    }

    return prop->getAsInt();
}

//----------------------------------------------------------------------------
int XMLNode::getIntProperty(const std::string &name,
                            const int defaultValue) const
{
    return m_pImpl->getIntProperty(name, defaultValue);
}

//----------------------------------------------------------------------------
int XMLNode::PImpl::getIntProperty(const std::string &name,
                                   const int defaultValue) const
{
    const XMLProperty *prop = getProperty(name);
    return prop ? prop->getAsInt() : defaultValue;
}

//----------------------------------------------------------------------------
unsigned XMLNode::getUnsignedProperty(const std::string &name) const
{
    return m_pImpl->getUnsignedProperty(name);
}

//----------------------------------------------------------------------------
unsigned XMLNode::PImpl::getUnsignedProperty(const std::string &name) const
{
    const XMLProperty *prop = getProperty(name);
    if (!prop)
    {
        throw XMLException(
            std::string("The mandatory attribute '")
            .append(name).append("' is missing"));
    }

    return prop->getAsUnsigned();
}

//----------------------------------------------------------------------------
unsigned XMLNode::getUnsignedProperty(const std::string &name,
                                      const unsigned defaultValue) const
{
    return m_pImpl->getUnsignedProperty(name, defaultValue);
}

//----------------------------------------------------------------------------
unsigned XMLNode::PImpl::getUnsignedProperty(const std::string &name,
                                             const unsigned defaultValue) const
{
    const XMLProperty *prop = getProperty(name);
    return prop ? prop->getAsUnsigned() : defaultValue;
}


//----------------------------------------------------------------------------
void XMLNode::acceptAllProperties(XMLVisitor &v)
{
    m_pImpl->acceptAllProperties(v);
}

//----------------------------------------------------------------------------
void XMLNode::PImpl::acceptAllProperties(XMLVisitor &v)
{
    for (XMLPropertyIter
             iter = m_properties.begin(),
             end = m_properties.end();
         iter != end; ++iter)
    {
        (*iter)->accept(v);
    }
}

//----------------------------------------------------------------------------
void XMLNode::acceptAllProperties(XMLConstVisitor &v) const
{
    m_pImpl->acceptAllProperties(v);
}

//----------------------------------------------------------------------------
void XMLNode::PImpl::acceptAllProperties(XMLConstVisitor &v) const
{
    for (XMLPropertyCIter
             iter = m_properties.begin(),
             end = m_properties.end();
         iter != end; ++iter)
    {
        (*iter)->accept(v);
    }
}

//----------------------------------------------------------------------------
void XMLNode::acceptAllNodes(XMLVisitor &v)
{
    m_pImpl->acceptAllNodes(v);
}

//----------------------------------------------------------------------------
void XMLNode::PImpl::acceptAllNodes(XMLVisitor &v)
{
    for (XMLNodeIter
             iter = m_nodes.begin(),
             end = m_nodes.end();
         iter != end; ++iter)
    {
        (*iter)->accept(v);
    }
}

//----------------------------------------------------------------------------
void XMLNode::acceptAllNodes(XMLConstVisitor &v) const
{
    m_pImpl->acceptAllNodes(v);
}

//----------------------------------------------------------------------------
void XMLNode::PImpl::acceptAllNodes(XMLConstVisitor &v) const
{
    for (XMLNodeCIter
             iter = m_nodes.begin(),
             end = m_nodes.end();
         iter != end; ++iter)
    {
        (*iter)->accept(v);
    }
}

//----------------------------------------------------------------------------
void XMLNode::acceptAllChilds(XMLVisitor &v)
{
    m_pImpl->acceptAllChilds(v);
}

//----------------------------------------------------------------------------
void XMLNode::PImpl::acceptAllChilds(XMLVisitor &v)
{
    for (XMLChildIter
             iter = m_childs.begin(),
             end = m_childs.end();
         iter != end; ++iter)
    {
        (*iter)->accept(v);
    }
}

//----------------------------------------------------------------------------
void XMLNode::acceptAllChilds(XMLConstVisitor &v) const
{
    m_pImpl->acceptAllChilds(v);
}

//----------------------------------------------------------------------------
void XMLNode::PImpl::acceptAllChilds(XMLConstVisitor &v) const
{
    for (XMLChildCIter
             iter = m_childs.begin(),
             end = m_childs.end();
         iter != end; ++iter)
    {
        (*iter)->accept(v);
    }
}

//----------------------------------------------------------------------------
void XMLNode::PImpl::removeChild(const XMLElement *e)
{
    for (XMLChildIter
             iter = m_childs.begin(),
             end = m_childs.end();
         iter != end; ++iter)
    {
        if (*iter == e)
        {
            m_childs.erase(iter);
            break;
        }
    }
}

//----------------------------------------------------------------------------
DECLARE_XML_VISITOR_API_BODY(XMLNode)



//============================================================================
// Implementation of XMLVisitors.h
//============================================================================

//----------------------------------------------------------------------------
XMLWriteToStreamVisitor::XMLWriteToStreamVisitor(std::ostream &os)
        : m_os(os)
{
    m_level = -1;
}

//----------------------------------------------------------------------------
XMLWriteToStreamVisitor::~XMLWriteToStreamVisitor()
{
}

//----------------------------------------------------------------------------
void XMLWriteToStreamVisitor::do_visit(const XMLProperty *p)
{
    m_os << " " << p->getName() << "=\"" << p->getValue() << "\"";
}

//----------------------------------------------------------------------------
void XMLWriteToStreamVisitor::do_visit(const XMLText *t)
{
    m_os << t->getText();
}

//----------------------------------------------------------------------------
void XMLWriteToStreamVisitor::do_visit(const XMLNode *n)
{
    if (m_level >= 0)
    {
        writeIndentation();
        m_os << "<" << n->getName();
        n->acceptAllProperties(*this);
    }

    if (n->getNumberOfNodes() > 0)
    {
        if (m_level >= 0)
        {
            m_os << ">\n";
        }

        m_level++;
        n->acceptAllChilds(*this);
        m_level--;

        if (m_level >= 0)
        {
            writeIndentation();
            m_os << "</";
            m_os << n->getName();
        }
    }
    else
    {
        if (m_level >= 0)
        {
            m_os << "/";
        }
    }

    if (m_level >= 0)
    {
        m_os << ">\n";
    }
}

//----------------------------------------------------------------------------
void XMLWriteToStreamVisitor::writeIndentation()
{
    for (int i=0; i<m_level; i++)
    {
        m_os << "  ";
    }
}



//============================================================================
// Implementation of XMLParser.h
//============================================================================

//----------------------------------------------------------------------------
class XMLParser::PImpl
{
    //------------------------------------------------------------------------
    typedef std::stack<XMLNode*> XMLNodeStack;

public:
    //------------------------------------------------------------------------
    PImpl();
    ~PImpl();

    //------------------------------------------------------------------------
    XMLNode *parse(const char *file);
    void parse(const char *file, XMLNode &root);

private:
    //------------------------------------------------------------------------
    void resetParser();

    //------------------------------------------------------------------------
    void do_parse(const char *file, XMLNode *root);

    //------------------------------------------------------------------------
    static void startElement(void *userData,
                             const char *name,
                             const char **atts);

    static void endElement(void *userData, const char *name);

    static void textElement(void *userData, const XML_Char *s, int len);

    //------------------------------------------------------------------------
    void startElement(const char *name, const char **atts);
    void endElement(const char *name);
    void textElement(const char *text, int len);

    void handleText();

    //------------------------------------------------------------------------
    XML_Parser m_xmlParser;
    XMLNodeStack m_stack;
    bool m_deleteRootOnError;
    std::string m_text;
};

//----------------------------------------------------------------------------
XMLParser::PImpl::PImpl()
{
    m_xmlParser = XML_ParserCreate(NULL);
    m_deleteRootOnError = false;
}

//----------------------------------------------------------------------------
XMLParser::PImpl::~PImpl()
{
    XML_ParserFree(m_xmlParser);
    m_xmlParser = NULL;

    while (m_stack.size() > 1)
    {
        delete m_stack.top();
        m_stack.pop();
    }

    if (m_deleteRootOnError && (m_stack.size() == 1))
    {
        delete m_stack.top();
        m_stack.pop();
    }
}

//----------------------------------------------------------------------------
XMLNode *XMLParser::PImpl::parse(const char *file)
{
    XMLNode *root = new XMLNode();
    m_deleteRootOnError = true;

    do_parse(file, root);
    return root;
}

//----------------------------------------------------------------------------
void XMLParser::PImpl::parse(const char *file, XMLNode &root)
{
    root.clear();
    m_deleteRootOnError = false;

    do_parse(file, &root);
}

//----------------------------------------------------------------------------
void XMLParser::PImpl::resetParser()
{
    if (!XML_ParserReset(m_xmlParser, NULL))
    {
        throw XMLException("Error resetting XML parser");
    }

    XML_SetUserData(m_xmlParser, this);
    XML_SetElementHandler(m_xmlParser, startElement, endElement);
    XML_SetCharacterDataHandler(m_xmlParser, textElement);
}

//----------------------------------------------------------------------------
void XMLParser::PImpl::do_parse(const char *file, XMLNode *root)
{
    resetParser();

    m_stack.push(root);

    char buffer[1024];
    File f(file, "r");
    int done;

    do
    {
        size_t len = f.read(buffer, sizeof(buffer));
        done = len < sizeof(buffer);
        if (XML_Parse(m_xmlParser, buffer, len, done) == XML_STATUS_ERROR)
        {
            std::ostringstream s;
            s << "XML parse error of file '" << f.getName()
              << "' at line " << XML_GetCurrentLineNumber(m_xmlParser)
              << ": " << XML_ErrorString(XML_GetErrorCode(m_xmlParser));
            throw XMLException(s.str());
        }
    }
    while (!done);

    m_stack.pop();
}


//----------------------------------------------------------------------------
void XMLParser::PImpl::startElement(void *userData,
                                    const char *name,
                                    const char **atts)
{
    static_cast<PImpl*>(userData)->startElement(name, atts);
}

//----------------------------------------------------------------------------
void XMLParser::PImpl::endElement(void *userData, const char *name)
{
    static_cast<PImpl*>(userData)->endElement(name);
}

//----------------------------------------------------------------------------
void XMLParser::PImpl::textElement(void *userData, const XML_Char *s, int len)
{
    static_cast<PImpl*>(userData)->textElement(s, len);
}

//----------------------------------------------------------------------------
void XMLParser::PImpl::startElement(const char *name, const char **atts)
{
    handleText();

    XMLNode *node = new XMLNode(name);
    for (int i=0; atts[i]; i+=2)
    {
        node->addProperty(new XMLProperty(atts[i], atts[i+1]));
    }

    m_stack.push(node);
}

//----------------------------------------------------------------------------
void XMLParser::PImpl::endElement(const char *name)
{
    handleText();

    XMLNode *node = m_stack.top();
    m_stack.pop();

    XMLNode *top = m_stack.top();
    top->addNode(node);
}

//----------------------------------------------------------------------------
void XMLParser::PImpl::textElement(const char *text, int len)
{
    m_text.append(text, len);
}

//----------------------------------------------------------------------------
void XMLParser::PImpl::handleText()
{
    const char *spaces = " \t\r\n";
    size_t index = m_text.find_first_not_of(spaces);
    m_text.erase(0, index);
    index = m_text.find_last_not_of(spaces);
    m_text.erase(index == std::string::npos ? 0 : index+1);

    if (m_text.length() > 0)
    {
        m_stack.top()->addText(new XMLText(m_text));
        m_text.clear();
    }
}


//----------------------------------------------------------------------------
XMLParser::XMLParser()
{
    m_pImpl = new PImpl();
}

//----------------------------------------------------------------------------
XMLParser::~XMLParser()
{
    ZAP_POINTER(m_pImpl);
}

//----------------------------------------------------------------------------
XMLNode *XMLParser::parse(const char *file)
{
    return m_pImpl->parse(file);
}

//----------------------------------------------------------------------------
void XMLParser::parse(const char *file, XMLNode &node)
{
    m_pImpl->parse(file, node);
}
