;; mh-alias.el - MH mail alias expansion and substitution.
;;
;; Copyright (C) 1994, 1995, 1996, 1997, 2001 Peter S. Galbraith
 
;; Author:    Peter S. Galbraith <psg@debian.org>
;; Created:   16 June 1994.
;; Version:   2.15 (12 Oct 2001)
;; Keywords:  mh-e, mail, alias, emacs, xemacs

;; RCS $Id: mh-alias.el,v 1.9 2002/01/25 14:26:48 psg Exp $
;; Note: RCS version number does not correspond to release number.

;; This program is free software; you can redistribute it and/or modify
;; it under the terms of the GNU General Public License as published by
;; the Free Software Foundation; either version 2, or (at your option)
;; any later version.
;; 
;; This program is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;; GNU General Public License for more details.
;; 
;; You should have received a copy of the GNU General Public License
;; along with this program; if not, write to the Free Software
;; Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

;;; Commentary:

;; New versions of this package (if they exist) may be found at:
;;  http://cvs.sourceforge.net/cgi-bin/viewcvs.cgi/mh-e/contrib/mh-alias.el
;; and, possibly, at:
;;  http://people.debian.org/~psg/elisp/mh-alias.el

;; Description:
;;
;;  This packages makes [TAB] do completion in minibuffer To: and CC:
;;  prompts on mail aliases (and optionally local usernames) and will
;;  substitute these aliases in the MH-letter buffer.  You may enter
;;  multiple addressees by separating them with a comma, which (by default)
;;  will flash the alias translation of the previous address.
;;
;;  This package also makes [C-c C-f TAB] and Meta-[TAB] (aka M-\t) do
;;  completion in the letter header itself.  The Meta-[TAB] feature is
;;  useful when you want to add an addressee as an afterthought while
;;  editing a letter, or add an addresse to a reply.
;;
;; Installation instructions:
;;
;;  All you need to do is add this line to your .emacs file
;;   (require 'mh-alias)
;;
;;  By default, completion is case insensitive.  Should you want to change 
;;  this, you can set the following variable:
;;   (defvar mh-alias-completion-ignore-case nil)
;;  This is useful, for example, if you set all people aliases to lowercase
;;  like 
;;    p.galbraith: Peter Galbraith <GalbraithP@dfo-mpo.gc.ca>
;;  and lists in uppercase, like: 
;;    MH-E: mh-e mailing list <mh-e@x.org>
;;  Note that this variable affects minibuffer completion only.  If you
;;  have an alias for P.Galbraith and type in p.galbraith at the prompt, it
;;  will still be substituted in the letter buffer because these are
;;  identical aliases as far as MH and mh-alias are concerned.
;;
;;  By default, when you press the [comma] key at the To: or Cc: prompts,
;;  the previous mail alias translation is flashed (This feature is
;;  currently broken if you use the `multi-prompt' package described
;;  below).  To inhibit this, add the following to your ~/.emacs file
;;  *before* you load mh-alias.
;;
;;   (setq mh-alias-flash-on-comma nil)
;;
;;  By default, after the mail prompt all the aliases that you have entered
;;  are translated to their alias expansions.  To inhibit this (and thus
;;  use this package only to get completion at the mail prompt) set this
;;  variable:
;;
;;   (setq mh-alias-substitute-aliases nil)
;;
;;  Completion and substitutions are also done on usernames extracted from
;;  your /etc/passwd file.  This can be a handy tool on a machine where
;;  you and co-workers exchange messages, but should probably be disabled
;;  on a system with 100 users whom you don't know.  This feature is
;;  disabled by adding the following to your ~/.emacs file:
;;
;;   (setq mh-alias-local-users nil)
;;
;;  You may also set mh-alias-local-users to a string to be executed to
;;  generate the password file.  This could be "ypcat passwd" if you use
;;  yellow pages.
;;
;;  If you do use the mh-alias-local-users feature, check that the
;;  variable mh-alias-hostname is set to a string containing @ followed by your
;;  hostname for better results (e.g. C-h v mh-alias-hostname).  This should be
;;  set correctly after mh-alias is loaded, but if it isn't do it in your
;;  ~/.emacs file using, for example
;;
;;   (setq mh-alias-hostname "@mixing.qc.dfo.ca")
;;
;;  By default, mh-alias `learns' mail aliases the first time you send
;;  mail and get the "To:" prompt, and subsequently whenever the mail alias
;;  file and password file are out of date with-respect-to its internal
;;  list.  Its internal list timestamp is compared against files set in the
;;  variable mh-alias-timestamp.  To add your own alias file, you might use a 
;;  list like:
;;
;;   (setq mh-alias-timestamp 
;;         '("~/Mail/mailaliases" "/usr/lib/mh/MailAliases" "/etc/passwd"))
;;
;;  You may want to answer the mail prompt with more than a single email
;;  address or alias.  To do this completion on comma-separated aliases,
;;  you must either load the package `complete.el' *before* you load
;;  mh-alias, or obtain the `multi-prompt.el' package.  The package
;;  complete.el affects minibuffer completion in everyday use, and you
;;  might not like it.  However, `multi-prompt.el' was written exactly for
;;  this reason so I suggest you use it; it's part of AUC TeX:
;;      ftp://sunsite.auc.dk/packages/auctex/auctex.tar.gz
;;
;;  Little quirks using the `multi-prompt' package:
;;   - The mh-alias-flash-on-comma feature is broken.
;;   - You may not precede a alias to complete with a space 
;;     e.g. you must enter  `To: p.galbraith,p.abrahamsen' 
;;          and not         `To: p.galbraith, p.abrahamsen'
;;     (This may change with future versions of multi-prompt)
;; ----------------------------------------------------------------------------
;;; Change log:
;; V1.00 16Jun94 Peter S Galbraith - Created as mh-aliases.el
;; V2.00 20Jul95 PSG - new version called mh-alias.el 
;;  - get aliases from MH's ali command.
;;  - allow for comma separated list.
;;  - optionally flash aliases when [comma] is pressed.
;;  - [C-c C-f \t] or [M-\t] in MH-Letter header expands/substitutes aliases
;; V2.01 24Jul95 PSG - Added e-mh-alias-completion-ignore-case
;; V2.02 24Jul95 PSG - 
;;  - called e-mh-alias.el because mh- prefix reserved for mh-e packages.
;;  - e-mh-alias-flash-on-comma may be unset after package loaded and can take 
;;    values nil, 1 or t. (Eric Ding <ericding@San-Jose.ate.slb.com>)
;;  - initialize mh-alias-hostname (Stephen Gildea <gildea@x.org>)
;;  - shows completions buffer immediately on \M-\t in MH-Letter-buffer
;; V2.03 27Jul95 PSG - Made truly case insensitive for substitution. 
;;    Thanks to Christopher Lott <lott@informatik.uni-kl.de> for bug reports.
;; V2.04 29Sep95 PSG - Added e-mh-alias-use-complete for xemacs compatibility.
;; V2.05 03Jul96 PSG - Support yellow pages for e-mh-alias-local-users
;;                     /etc/passwd entries don't overwrite existing aliases
;;                     Updates to new alias files automatically.
;;                     Added e-mh-alias-substitute-aliases.
;;                     First entry in alias file superseeds others
;; V2.06 26Aug96 Jeffrey C Honig <jch@bsdi.com>  (RCS 1.2)
;;  - Better end of header regexp.
;; V2.07 26Nov96 Matthew Swift <swift@bu.edu>  (RCS 1.3)
;;  - bug fix in e-mh-learn-aliases
;; V2.08 03Jan97 PSG  (RCS 1.4)
;;  - Removed e-mh-alias-use-complete and allow use of multi-prompt.el
;; V2.09 12Feb97 (RCS 1.5) Ed Reingold <emr@black.wisdom.weizmann.ac.il>
;;  - e-mh-learn-local-users handles commas in name field correctly.
;; V2.10 20Mar97 (RCS 1.6) Ed Reingold <emr@black.wisdom.weizmann.ac.il>
;;  - e-mh-learn-local-users handles gecos correctly.
;; V2.11 10Jan01 (RCS 1.8) PSG
;;  - implement blind list correctly (See mh-alias(5) for details).
;; V2.12 11Jan01 (RCS 1.10) PSG
;;  - add e-mh-alias-display-blind-name-on-completion for blind lists.
;; V2.13 25Jun01 PSG
;;  - uploaded to contrib in the mh-e project on Sourceforge
;;    (All RCS numbers mentionned above don't apply!)
;;  - rename all e-mh* variables and functions to mh-alias*
;;  - change license to GPL, change home to SF and debian web sites.
;;  - changed all defvar to defcustom.
;; V2.14 12Oct01 PSG
;;  - clean up old code written for emacs19 (buffer-substring -> match-string)
;; V2.15 12Oct01 PSG
;;  - fix one match-string -> match-string-no-properties.
;; V2.16 23Oct01 PSG
;;  - change customization group to mh (instead of mh-e)
;; ----------------------------------------------------------------------------
;;; Code:

(defgroup mh-alias nil
  "MH mail alias expansion and substitution" :group 'mh)

(defcustom mh-alias-substitute-aliases t
  "*if t, mh-alias will substitute aliases entered in the mail prompts
with their expansions."
  :group 'mh-alias
  :type 'boolean)

(defcustom mh-alias-timestamp
  '("/etc/nmh/MailAliases" "/usr/lib/mh/MailAliases" "/etc/passwd")
  "*mh-alias-learn-aliases runs automatically if one of these files is updated.
This will keep the mh-alias database up-to-date.  The modification timestamp
of these files is compared against a created file named ~/.mh-alias

Set this to nil if you don't want the internal database to be updated 
automatically in this fashion.  You can always update it using
 M-x mh-alias-learn-aliases"
  :group 'mh-alias
  :type '(choice (file) (repeat file)))

(defcustom mh-alias-completion-ignore-case t
  "*non-nil means don't consider case significant in MH alias completion.
This is the default in plain MH, so it is the default here as well.
But you can change it usefully if, for example, you use lowercase aliases
for people and uppercase for lists."
  :group 'mh-alias
  :type 'boolean)

(defcustom mh-alias-flash-on-comma t
  "*if non-nil, alias translation displayed when [comma] pressed in mh queries
t   flash alias translation but don't warn if there is no translation. 
1   flash alias translation and warn if there is no translation.
nil don't flash alias translation or warn if there is no translation."
  :group 'mh-alias
  :type '(choice (const :tag "Flash but don't warn if no translation" t)
		 (const :tag "Flash and warn if no translation" 1)
		 (const :tag "Don't flash or warn if no translation" nil)))

(defcustom mh-alias-local-users t
  "*If t, local users are completed-to and expanded by MH To: and Cc: prompts.
The fake aliases are usually extracted from /etc/passwd for UID >= 200.
However, if you set this variable to a string, it will be executed to generate
the password file.  Use this for yellow pages, e.g.

 (setq mh-alias-local-users \"ypcat passwd\")"
  :group 'mh-alias
  :type '(choice (boolean) (string)))

(defcustom mh-alias-display-blind-name-on-completion t
  "*Display text string that is substituted for blind aliases on completion.
e.g. for an alias entry like:

blind-friends: All my friends: buddy@yahoo.com, mary@school.edu, etc

The string `All my friends' would be used and displayed on completion of
the mail To: prompt."
  :group 'mh-alias
  :type 'boolean)

(defcustom mh-alias-hostname 
  (concat "@"
          (or (and (boundp 'mail-host-address)
                   mail-host-address)
              (system-name)))
  "*String to append to local usernames from /etc/passwd to make addresses.
This should be \"@\" your fully qualified hostname (e.g. \"@mixing.qc.dfo.ca\")
See variable mh-alias-local-users."
  :group 'mh-alias
  :type 'string)

(defvar mh-alias-alist nil "Alist of MH mailaliases.")
(defvar mh-alias-blind-alist nil 
  "Alist of MH mailaliases that should be treated as blind.")
(defvar mh-alias-lowercase-alist nil 
  "Alist of lowercased MH mailaliases with translations as cdr.")

(defvar mh-alias-read-address-map nil)
(if mh-alias-read-address-map
    ()
  (setq mh-alias-read-address-map 
	(copy-keymap minibuffer-local-completion-map))
  (if (featurep 'complete)
      (define-key mh-alias-read-address-map 
	"\t" 'mh-alias-PC-complete-address))
  (if mh-alias-flash-on-comma
      (define-key mh-alias-read-address-map 
	"," 'mh-alias-minibuffer-confirm-address))
  (define-key mh-alias-read-address-map " " 'self-insert-command))

(eval-after-load "mh-comp"
  '(defun mh-read-address (prompt &optional initial-address)
     "Read a To: or Cc: address, prompting in the minibuffer with PROMPT."
     (cond
      ((not mh-alias-alist)		; Doesn't exist, so create it.
       (mh-alias-learn-aliases))
      ((and mh-alias-timestamp		; Out of date, so recreate it.
	    (car (memq t
		       (mapcar
			(function 
			 (lambda (x) (file-newer-than-file-p x "~/.mh-alias")))
			mh-alias-timestamp))))
       (mh-alias-learn-aliases)))
     (if (not mh-alias-alist)		; If still no aliases, just prompt
	 (read-string prompt)
       (let* ((minibuffer-local-completion-map mh-alias-read-address-map)
	      (completion-ignore-case mh-alias-completion-ignore-case)
	      ;; initial-address is nil with minibuffer To: and CC: prompts
	      ;;                 maybe "" in MH-letter-mode
	      (unread-command-char (cond ((or (not initial-address)
					      (string-equal "" 
							    initial-address)) 
					  -1)
					 (t
					  ??)))
	      (the-answer 
	       (or (if (and (featurep 'multi-prompt)
			    (not (featurep 'complete)))
		       (mapconcat 
			(function (lambda (x) (format "%s" x)))
			(multi-prompt "," nil
				      prompt mh-alias-alist nil nil 
				      initial-address)
			", ")
		     (completing-read 
		      prompt mh-alias-alist nil nil initial-address))
		   "")))
	 (cond 
	  (mh-alias-substitute-aliases
	   ;; Here I find all comma-whitespace-delimited words to test
	   ;; as aliases.
	   (let ((the-aliases)
		 (case-fold-search t))
	     (while (and the-answer
			 ;; Catches comma-delimited address with trailing space
			 (string-match "^[ \t,]*\\([^,]+\\)" the-answer))
	       (let* ((the-index (match-end 0))
		      (the-match (substring the-answer 
					    (match-beginning 1)(match-end 1)))
		      ;; Now trim off trailing whitespace
		      (trim-match (substring 
				   the-match 0 
				   (string-match "[ \t]*$" the-match))))
		 (cond
		  ((member (list trim-match) mh-alias-blind-alist)
		   (if (string-match "\\(.*\\):" trim-match)
		       (setq trim-match (match-string 1 trim-match)))
		   (setq the-aliases (concat the-aliases ",\n " trim-match)))
		  (t
		   (setq the-aliases
			 (concat 
			  the-aliases ",\n "
			  (if (assoc (downcase trim-match)     ; Translates to 
				     mh-alias-lowercase-alist) ; alias?
			      (mapconcat 
			       (function (lambda (x) (format "%s" x)))
			       (cdr (assoc (downcase trim-match) 
					   mh-alias-lowercase-alist))
			       ",\n ")
			    trim-match))))) ; Does not translate 
		 (setq the-answer (substring the-answer the-index))))
	     ;; remove leading comma
	     (if the-aliases
		 (substring the-aliases 3)
	       "")))
	  (t
	   ;; Don't do any further processing...
	   the-answer))))))

(defun mh-alias-PC-complete-address ()
  "Called by pressing [TAB] while in mh TO: or CC: prompts.
Uses the complete.el package."
  (interactive)
  (PC-do-completion nil (save-excursion (skip-chars-backward "^ \t,")(point))))

(defun mh-alias-minibuffer-confirm-address ()
  "Called by pressing [comma] if mh-alias-flash-on-comma is t." 
  (interactive)
  (if (not mh-alias-flash-on-comma)
      ()
    (save-excursion
      (let* ((case-fold-search t)
             (the-name (buffer-substring
                        (progn (skip-chars-backward " \t")(point))
                        ;; This moves over to previous comma, if any
                        (progn (or (and (not (= 0 (skip-chars-backward "^,")))
                                        ;; the skips over leading whitespace
                                        (skip-chars-forward " "))
                                   ;; no comma, then to beginning of word
                                   (skip-chars-backward "^ \t"))
                               ;; In Emacs21, the beginning of the prompt
                               ;; line is accessible, which wasn't the case
                               ;; in emacs20.  Skip over it.
                               (if (looking-at "^[^ \t]+:")
                                   (skip-chars-forward "^ \t"))
                               (skip-chars-forward " ")
                               (point)))))
        (if (assoc (downcase the-name) mh-alias-lowercase-alist)
            (message "%s -> %s" the-name
                     (mapconcat (function (lambda (x) (format "%s" x)))
                                (cdr (assoc (downcase the-name) 
                                            mh-alias-lowercase-alist))
                                ", "))
          ;; Check if if was a single word likely to be an alias
          (if (and (equal mh-alias-flash-on-comma 1)
                   (not (string-match " " the-name)))
              (message "No alias for %s" the-name))))))
    (self-insert-command 1))

(defun mh-alias-learn-aliases ()
  "Learn MH aliases, building an alist to complete with, and substitute with.
This can be called interactively to rebuild from updated files."
;;; Could be done using an obarray and `intern' to create it.
;;; elisp info doesn't say if completing-read is more efficient with alists
;;;  or obarrays.
  (interactive)
  (let ((mailalias-buffer (get-buffer-create " *mh-alias-ali-command*")))
    (save-excursion
      (set-buffer mailalias-buffer)
      (erase-buffer)
      (insert (expand-file-name "ali" mh-progs) " -list\n")
      (write-region (point-min) (point-max) "/tmp/mh-alias-ali-command" nil nil)
      (message "building list of MH mail aliases...")
      (erase-buffer)
      (call-process "/bin/sh" "/tmp/mh-alias-ali-command" t nil)
      (delete-file "/tmp/mh-alias-ali-command")
      (setq mh-alias-alist nil
            mh-alias-lowercase-alist nil)
      (goto-char (point-min))
      (let ((zero 0))
        (while (= zero 0)
          (cond
           ((looking-at "\\(\\(.+\\): .+\\): .*$")
            ;; A new -blind- MH alias
            (let* ((username (match-string 2))
                   (fullname (match-string 1))
                   (nameused (if mh-alias-display-blind-name-on-completion
                                 fullname
                               username)))
              (cond
               ;; Don't overwrite existing aliases
               ((not (assoc (downcase nameused)
                            mh-alias-lowercase-alist))
                (setq mh-alias-alist (cons (list nameused) mh-alias-alist))
                (setq mh-alias-blind-alist 
                      (cons (list nameused) mh-alias-blind-alist))
                (setq mh-alias-lowercase-alist 
                      (cons (list (downcase nameused))
                            mh-alias-lowercase-alist))))
              ;; Skip any continuation line
              (let ((zero 0))
                (while (and (eq 0 zero)
                            (setq zero (forward-line 1))
                            (looking-at " ")))
                (if (eq zero 0)
                    (forward-line -1)))))
           ((looking-at "\\(.*\\): *\\(.*\\)$") ; A new MH alias
            (let ((username (match-string 1))
                  (realname (match-string 2)))
              (cond
               ;; Don't overwrite existing aliases
               ((not (assoc (downcase username) 
                            mh-alias-lowercase-alist))
                (setq mh-alias-alist (cons (list username) mh-alias-alist))
                (setq mh-alias-lowercase-alist
                      (cons (list (downcase username) realname)
                            mh-alias-lowercase-alist)))
               (t                       
                ;; This alias already exists
                ;; Skip any continuation line
                (let ((zero 0))
                  (while (and (eq 0 zero)
                              (setq zero (forward-line 1))
                              (looking-at " ")))
                  (if (eq zero 0)
                      (forward-line -1)))))))
           ((looking-at "[ ]+\\(.*\\)$") ; An alias continued
            (let ((the-first (car mh-alias-lowercase-alist)) 
                  ;;^^The previous alias list entered ("NEW" "schneider@awi")
                  (the-rest (cdr mh-alias-lowercase-alist)) 
                  ;;^^The others before
                  (the-new (match-string 1)))
              (setq mh-alias-lowercase-alist
                    (cons
                     (append            ;  ("NEW" "budeus@awi" "schneider@awi")
                      (list (car the-first)) ; ("NEW")
                      (list the-new)    ;      ("budeus@awi")
                      (cdr the-first))  ;      ("schneider@awi")
                     the-rest)))))
          (setq zero (forward-line 1)))))
    (kill-buffer mailalias-buffer))
  (if mh-alias-local-users
      (mh-alias-learn-local-users))
  (if mh-alias-timestamp
      (let ((the-buffer (create-file-buffer "~/.mh-alias")))
        (save-excursion
          (set-buffer the-buffer)
          (set-buffer-modified-p t)
          (set-visited-file-name "~/.mh-alias")
          (set-buffer-modified-p t)
          (save-buffer 0))
        (kill-buffer the-buffer)))
  (message "building list of MH mail aliases... done."))

(defun mh-alias-learn-local-users ()
  "Add local users from /etc/passwd to mh-alias-alist"
  (let ((mailalias-buffer (get-buffer-create " *mailalias*")))
    (save-excursion
      (set-buffer mailalias-buffer)
      (cond
       ((eq mh-alias-local-users t)
        (if (file-readable-p "/etc/passwd")
            (insert-file-contents "/etc/passwd")))
       (mh-alias-local-users
        (insert mh-alias-local-users "\n")
        (write-region (point-min) (point-max) "/tmp/mh-alias-command" nil nil)
        (erase-buffer)
        (call-process "/bin/sh" "/tmp/mh-alias-command" t nil)
        (goto-char (point-min))
        (delete-file "/tmp/mh-alias-command")))
      (let ((zero 0))
        (while (= zero 0)
          (cond
           ((looking-at "\\([^:]*\\):[^:]*:\\([^:]*\\):[^:]*:\\([^:,]*\\)[:,]")
            (if (> (string-to-int (match-string 2)) 200)
                (let* ((username (match-string 1))
                       (gecos-name (match-string 3))
                       (realname
                        (if (string-match "&" gecos-name)
                            (concat
                             (substring gecos-name 0 (match-beginning 0))
                             (capitalize username)
                             (substring gecos-name (match-end 0)))
                          gecos-name)))
                  (cond
                   ;; Don't overwrite existing aliases
                   ((not (assoc (downcase username) 
                                mh-alias-lowercase-alist))
                    (setq mh-alias-lowercase-alist
                          (cons (list 
                                 (downcase username)
                                 (if (string-equal "" realname)
                                     (concat "<" username mh-alias-hostname ">")
                                   (concat realname 
                                           " <" username mh-alias-hostname ">")))
                                mh-alias-lowercase-alist))
                    (setq mh-alias-alist
                          (cons (list username) mh-alias-alist))))))))
          (setq zero (forward-line 1)))))
    (kill-buffer mailalias-buffer)))

;;; --------------------------------------------------------------------------
;;; This part of mh-alias adds [M-TAB] expansion/substitution in letter head

(eval-after-load "mh-comp"
  '(progn
     (define-key mh-letter-mode-map "\C-c\C-f\t" 'mh-alias-letter-expand-alias)
     (cond
      ((string< "19.28.9" emacs-version)
       ;; wanted also to test (emacs-type) but this is unbound in -batch mode !
       (setq mh-alias-expand-alias-map (copy-keymap mh-letter-mode-map))
       (define-key mh-alias-expand-alias-map "\M-\t" 
	 'mh-alias-letter-expand-alias)
      
       (defun mh-alias-letter-mode-expand-alias-hook ()
	 "Make M-[TAB] expand mail alias within letter header"
	 (save-excursion
	   ;; Take extra precautions to not return errors if regexp not found.
	   (add-text-properties (progn (goto-char (point-min))(point))
				(progn (re-search-forward "^-*$" nil t)(point))
				(list 'local-map mh-alias-expand-alias-map))))

       (add-hook 'mh-letter-mode-hook 
		 'mh-alias-letter-mode-expand-alias-hook)))
     ))

(defun mh-alias-letter-expand-alias ()
  "expand/convert mail alias before/under point."
  (interactive)
  (save-excursion
    (skip-chars-backward "^ ,:")
    (let ((the-match)(init-address "")(the-address))
      (if (looking-at "[^ ,;\n]+")
          (setq the-match (match-data)
                init-address (match-string-no-properties 0)))
      (if (assoc (try-completion (downcase init-address) 
                                 mh-alias-lowercase-alist) 
                 mh-alias-lowercase-alist)
          ;; init-address is "p.gal" which has unique completion to 
          ;; "p.galbraith", then substitute its alias, possibly multi-line.
          (cond
           ((member (list
                     (try-completion (downcase init-address) mh-alias-alist))
                    mh-alias-blind-alist)
            (let ((blindname (try-completion (downcase init-address)
                                                    mh-alias-alist)))
              (setq the-address 
                    (substring blindname 0 (string-match ":" blindname)))))
           (t
            (setq the-address
                  (mapconcat 
                   (function (lambda (x) (format "%s" x)))
                   (cdr (assoc (try-completion (downcase init-address) 
                                               mh-alias-lowercase-alist) 
                               mh-alias-lowercase-alist))
                   ",\n "))))
        (setq the-address (mh-read-address "Address: " init-address)))
      (if (string-equal "" the-address)
          ()
        (if (not the-match)
            (insert the-address)
          (set-match-data the-match)
          (replace-match the-address t t))))))

(provide 'mh-alias)
;;; mh-alias.el ends here
