/* sel-tables.c
 *
 * Copyright (C) 2004 Vivien Malerba
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include "sel-tables.h"
#include <libgnomedb/libgnomedb.h>
#include "mg-database.h"
#include "mg-db-table.h"
#include "sel-onetable.h"

/*
 *
 * Module for tables
 *
 */
static void         module_tables_fill_model (Module *module);
static void         module_tables_free (Module *module);
static const gchar *module_tables_col_name (Module *module, guint colno);
static Module      *module_tables_obj_manager (Module *module, GtkTreeIter *iter, GObject *object);
static void         module_tables_model_store_data (Module *module, GtkTreeIter *iter);

Module *sel_module_tables_new (MgSelector *mgsel, gboolean insert_header, 
			       GtkTreeIter *iter, gpointer data)
{
	Module *module;

	module = g_new0 (Module, 1);
	module->selector = mgsel;
	module->fill_model = module_tables_fill_model;
	module->free = module_tables_free;
	module->col_name = module_tables_col_name;
	module->obj_manager = module_tables_obj_manager;
	module->model_store_data = module_tables_model_store_data;
	module->mod_data = NULL;
	module->iter = NULL;
	module->parent_module = NULL;
	module->sub_modules = NULL;

	if (insert_header) {
		GdkPixbuf *pixbuf = NULL;
		GtkTreeModel *model = mgsel->priv->model;

		pixbuf = gnome_db_stock_get_icon_pixbuf (GNOME_DB_STOCK_TABLES);
		module->iter = g_new0 (GtkTreeIter, 1);
		gtk_tree_store_append (GTK_TREE_STORE (model), module->iter, iter);
		gtk_tree_store_set (GTK_TREE_STORE (model), module->iter, NAME_COLUMN, 
				    _("Tables & views"), PIXBUF_COLUMN, pixbuf, 
				    CONTENTS_COLUMN, CONTENTS_TOP_CATEGORY, 
				    SUB_MODULE_COLUMN, NULL, -1);		
	}
	else {
		if (iter)
			module->iter = gtk_tree_iter_copy (iter);
	}

	return module;
}


static GSList *module_tables_get_objects_list (Module *module);
static gchar  *module_tables_get_extended_name (GObject *obj);
static void
module_tables_fill_model (Module *module)
{
	MgDatabase *db;
	GtkTreeModel *model;
	GdkPixbuf *pixbuf_table = NULL;

	db = mg_conf_get_database (module->selector->priv->conf);
	pixbuf_table = gnome_db_stock_get_icon_pixbuf_file ("gnome-db-table_16x16.png");

	/* Module's private data */
	module->mod_data = g_new0 (ModNameGroupData, 1);
	GROUP_DATA (module)->manager = G_OBJECT (db);
	GROUP_DATA (module)->manager_weak_refed = FALSE;
	GROUP_DATA (module)->obj_pixbuf = pixbuf_table;
	GROUP_DATA (module)->get_objects_list = module_tables_get_objects_list;
	GROUP_DATA (module)->get_extended_name = module_tables_get_extended_name;
	
	/* Initial model filling */
	model = module->selector->priv->model;
	name_group_init_model_fill (module, model);

	/* Signals handlers */
	g_signal_connect (G_OBJECT (db), "table_added",
			  G_CALLBACK (name_group_obj_added_cb), module);
	g_signal_connect (G_OBJECT (db), "table_removed",
			  G_CALLBACK (name_group_obj_removed_cb), module);
	g_signal_connect (G_OBJECT (db), "table_updated",
			  G_CALLBACK (name_group_obj_updated_cb), module);
	g_signal_connect (G_OBJECT (db), "data_update_started",
			  G_CALLBACK (name_group_update_started_cb), module);
	g_signal_connect (G_OBJECT (db), "data_update_finished",
			  G_CALLBACK (name_group_update_finished_cb), module);
}

static GSList *
module_tables_get_objects_list (Module *module)
{
	g_return_val_if_fail (GROUP_DATA (module)->manager, NULL);
	g_return_val_if_fail (IS_MG_DATABASE (GROUP_DATA (module)->manager), NULL);

	return mg_database_get_tables (MG_DATABASE (GROUP_DATA (module)->manager));
}

static gchar *
module_tables_get_extended_name (GObject *obj)
{
	g_return_val_if_fail (obj && IS_MG_DB_TABLE (obj), NULL);

	return g_strdup (mg_base_get_name (MG_BASE (obj)));
}

static void
module_tables_free (Module *module)
{
	gpointer db = GROUP_DATA (module)->manager;
	GSList *list = module->sub_modules;

	/* Free the sub modules for individual tables contents */
	while (list) {
		(MODULE (list->data)->free) (MODULE (list->data));
		g_free (list->data);
		list = g_slist_next (list);
	}
	if (module->sub_modules) {
		g_slist_free (module->sub_modules);
		module->sub_modules = NULL;
	}

	/* free this module */
	if (db) {
		g_signal_handlers_disconnect_by_func (G_OBJECT (db),
						      G_CALLBACK (name_group_obj_added_cb), module);
		g_signal_handlers_disconnect_by_func (G_OBJECT (db),
						      G_CALLBACK (name_group_obj_removed_cb), module);
		g_signal_handlers_disconnect_by_func (G_OBJECT (db),
						      G_CALLBACK (name_group_obj_updated_cb), module);
		g_signal_handlers_disconnect_by_func (G_OBJECT (db),
						      G_CALLBACK (name_group_update_started_cb), module);
		g_signal_handlers_disconnect_by_func (G_OBJECT (db),
						      G_CALLBACK (name_group_update_finished_cb), module);
	}
	if (module->iter)
		gtk_tree_iter_free (module->iter);
	name_group_free_mod_data (module);
	g_free (module->mod_data);
	module->mod_data = NULL;
}


static const gchar *
module_tables_col_name (Module *module, guint colno)
{
	switch (colno) {
	case 0:
		return _("Table or View");
		break;
	case EXTRA1_COLUMN:
		return _("Type");
		break;
	default:
		return NULL;
		break;
	}
}


/*
 * module_tables_obj_manager
 *
 * This function is called when a new MgDbTable is created. It simply creates a new
 * Module (which will be appended to the list of modules of the module for the tables)
 * for the fields of that particular table.
 */
static Module *
module_tables_obj_manager (Module *module, GtkTreeIter *iter, GObject *object)
{
	Module *sub_module = NULL;

	g_assert (object && IS_MG_DB_TABLE (object));

	if (module->selector->priv->mode & MG_SELECTOR_FIELDS) 
		sub_module = sel_module_onetable_new (module->selector, FALSE, iter, object);

	return sub_module;
}

static void
module_tables_model_store_data (Module *module, GtkTreeIter *iter)
{
	GObject *obj;
	GtkTreeModel *model;

	model = module->selector->priv->model;
	gtk_tree_model_get (model, iter, OBJ_COLUMN, &obj, -1);

	if (obj && IS_MG_DB_TABLE (obj)) {
		MgDbTable *table;
		const gchar *str1;

		table = MG_DB_TABLE (obj);

		if (mg_db_table_is_view (table))
			str1 = _("View");
		else
			str1 = _("Table");

		gtk_tree_store_set (GTK_TREE_STORE (model), iter, 
				    EXTRA1_COLUMN, str1,
				    -1);
	}
}

