/* lib.c - MemTest-86  Version 2.9
 *
 * Released under version 2 of the Gnu Public License.
 * By Chris Brady, cbrady@sgi.com
 */
#include "io.h"
#include "serial.h"
#include "test.h"
#include "config.h"
#include "screen_buffer.h"

extern struct vars *v;

int slock = 0, lsr = 0;
short serial_cons = SERIAL_CONSOLE_DEFAULT;
char buf[18];

struct ascii_map_str {
        int ascii;
        int keycode;
};

char *codes[] = {
	"  Divide",
	"   Debug",
	"     NMI",
	"  Brkpnt",
	"Overflow",
	"   Bound",
	"  Inv_Op",
	" No_Math",
	"Double_Fault",
	"Seg_Over",
	" Inv_TSS",
	"  Seg_NP",
	"Stack_Fault",
	"Gen_Prot",
	"Page_Fault",
	"   Resvd",
	"     FPE",
	"Alignment",
	" Mch_Chk",
	"SIMD FPE"
};

struct eregs {
	ulong eax;
	ulong ebx;
	ulong ecx;
	ulong edx;
	ulong edi;
	ulong esi;
	ulong ebp;
	ulong esp;
	ulong vect;
	ulong eflag;
	ulong cs;
	ulong eip;
	ulong code;
};

struct eregs trap_regs;

/*
 * Scroll the error message area of the screen as needed
 * Starts at line LINE_SCROLL and ends at line 23
 */
void scroll() {
	int i, j;
	char *s, tmp;

	/* Only scroll if at the bottom of the screen */
	if (v->msg_line < 23) {
		v->msg_line++;
	} else {
		/* If scroll lock is on, loop till it is cleared */
		while (slock) {
			check_input();
		}
	        for (i=LINE_SCROLL; i<23; i++) {
			s = (char *)(SCREEN_ADR + ((i+1) * 160));
			for (j=0; j<160; j+=2, s+=2) {
				*(s-160) = *s;
                                tmp = get_scrn_buf(i+1, j/2);
                                set_scrn_buf(i, j/2, tmp);
			}
		}
		/* Clear the newly opened line */
		s = (char *)(SCREEN_ADR + (23 * 160));
		for (j=0; j<80; j++) {
			*s = ' ';
                        set_scrn_buf(23, j, ' ');
			s += 2;
		}
                tty_print_region(LINE_SCROLL, 0, 23, 79);
        }
}

/*
 * Print characters on screen
 */
void cprint(int y, int x, char *text)
{
	register int i;
	char *dptr;

	dptr = (char *)(SCREEN_ADR + (160*y) + (2*x));
	for (i=0; text[i]; i++) {
		*dptr = text[i];
		dptr += 2;
                set_scrn_buf(y, x+i, text[i]);
        }
        tty_print_line(y, x, i);
}

void itoa(char s[], int n) 
{
  int i, sign;

  if((sign = n) < 0)
    n = -n;
  i=0;
  do {
    s[i++] = n % 10 + '0';
  } while ((n /= 10) > 0);
  if(sign < 0)
    s[i++] = '-';
  s[i] = '\0';
  reverse(s);
}

void reverse(char s[])
{
  int c, i, j;

  for(i=0, j = strlen(s) -1; i < j; i++, j--) {
    c = s[i];
    s[i] = s[j];
    s[j] = c;
  }
}

/*
 * Print a people friendly address
 */
void aprint(int y, int x, ulong val)
{
	if (val < 10000) {
		dprint(y, x, val, 5, 0);
	} else if (val < 1024*9999) {
		dprint(y, x, val/1024, 4, 0);
		cprint(y, x+4, "K");
	} else {
		dprint(y, x, val/(1024*1024), 4, 0);
		cprint(y, x+4, "M");
	}
}

/*
 * Print a decimal number on screen
 */
void dprint(int y, int x, ulong val, int len, int right)
{
	ulong j, k;
	int i, flag=0;

	if (val > 999999999 || len > 9) {
		return;
	}
	for(i=0, j=1; i<len-1; i++) {
		j *= 10;
	}
	if (!right) {
		for (i=0; j>0; j/=10) {
			k = val/j;
			if (k > 9) {
				j *= 100;
				continue;
			}
			if (flag || k || j == 1) {
				buf[i++] = k + '0';
				flag++;
			} else {
				buf[i++] = ' ';
			}
			val -= k * j;
		}
	} else {
		for(i=0; i<len; j/=10) {
			if (j) {
				k = val/j;
					if (k > 9) {
					j *= 100;
					len++;
					continue;
				}
				if (k == 0 && flag == 0) {
					continue;				
				}
				buf[i++] = k + '0';
				val -= k * j;
			} else {
                                if (flag == 0 &&  i < len-1) {
                                        buf[i++] = '0';
                                } else {
                                        buf[i++] = ' ';
                                }
			}
			flag++;
		}
	}
	buf[i] = 0;
	cprint(y,x,buf);
}

/*
 * Print a hex number on screen
 */
void hprint(int y,int x, ulong val)
{
	ulong j;
	int i, idx, flag = 0;

        for (i=0, idx=0; i<8; i++) {
                j = val >> (28 - (4 * i));
		j &= 0xf;
		if (j < 10) {
			if (flag || j || i == 7) {
		                buf[idx++] = j + '0';
				flag++;
			} else {
				buf[idx++] = '0';
			}
		} else {
			buf[idx++] = j + 'a' - 10;
			flag++;
		}
        }
        buf[idx] = 0;
	cprint(y,x,buf);
}

/*
 * Print an address in 0000m0000k0000 notation
 */
void xprint(int y,int x, ulong val)
{
        ulong j;

	j = (val & 0xffc00000) >> 20;
	dprint(y, x, j, 4, 0);
	cprint(y, x+4, "m");
	j = (val & 0xffc00) >> 10;
	dprint(y, x+5, j, 4, 0);
	cprint(y, x+9, "k");
	j = val & 0x3ff;
	dprint(y, x+10, j, 4, 0);
}
	
/* Handle an interrupt */
void inter()
{
	int i, line;
	char *pp;

#ifdef PARITY_MEM

	/* Check for a parity error */
	if (trap_regs.vect == 2) {
		parity_err(trap_regs.edi, trap_regs.esi);
		return;
	}
#endif

	/* clear scrolling region */
        pp=(char *)(SCREEN_ADR+(2*80*(LINE_SCROLL-2)));
        for(i=0; i<2*80*(24-LINE_SCROLL-2); i++, pp+=2) {
                *pp = ' ';
        }
	line = LINE_SCROLL-2;

	cprint(line, 0, "Unexpected Interrupt - Halting");
	cprint(line+2, 0, " Type: ");
	if (trap_regs.vect <= 19) {
		cprint(line+2, 7, codes[trap_regs.vect]);
	} else {
		hprint(line+2, 7, trap_regs.vect);
	}
	cprint(line+3, 0, "   PC: ");
	hprint(line+3, 7, trap_regs.eip);
	cprint(line+4, 0, "   CS: ");
	hprint(line+4, 7, trap_regs.cs);
	cprint(line+5, 0, "Eflag: ");
	hprint(line+5, 7, trap_regs.eflag);
	cprint(line+6, 0, " Code: ");
	hprint(line+6, 7, trap_regs.code);

	cprint(line+2, 20, "eax: ");
	hprint(line+2, 25, trap_regs.eax);
	cprint(line+3, 20, "ebx: ");
	hprint(line+3, 25, trap_regs.ebx);
	cprint(line+4, 20, "ecx: ");
	hprint(line+4, 25, trap_regs.ecx);
	cprint(line+5, 20, "edx: ");
	hprint(line+5, 25, trap_regs.edx);
	cprint(line+6, 20, "edi: ");
	hprint(line+6, 25, trap_regs.edi);
	cprint(line+7, 20, "esi: ");
	hprint(line+7, 25, trap_regs.esi);
	cprint(line+8, 20, "ebp: ");
	hprint(line+8, 25, trap_regs.ebp);
	cprint(line+9, 20, "esp: ");
	hprint(line+9, 25, trap_regs.esp);

	cprint(line+1, 38, "Stack:");
	for (i=0; i<12; i++) {
		hprint(line+2+i, 38, trap_regs.esp+(4*i));
		hprint(line+2+i, 47, *(ulong*)(trap_regs.esp+(4*i)));
		hprint(line+2+i, 57, trap_regs.esp+(4*(i+12)));
		hprint(line+2+i, 66, *(ulong*)(trap_regs.esp+(4*(i+12))));
	}

	while(1);
}

void set_cache(int val) {
	switch(val) {
	case 0:
		cache_off();	
		if (v->cache_flag) {
			cprint(LINE_INFO, COL_CACHE, "OFF");
		} else {
			cprint(LINE_INFO, COL_CACHE, "off");
		}
		break;
	case 1:
		cache_on();
		if (v->cache_flag) {
			cprint(LINE_INFO, COL_CACHE, " ON");
		} else {
			cprint(LINE_INFO, COL_CACHE, " on");
		}
		break;
	}
}

int get_key() {
	int c;
	
	c = inb(0x64);
	if ((c & 1) == 0) {
		if (serial_cons) {
			int comstat;
			comstat = serial_echo_inb(UART_LSR);
			if (comstat & UART_LSR_DR) {
				c = serial_echo_inb(UART_RX);
				/* Pressing '.' has same effect as 'c'
				   on a keyboard.
				   Oct 056   Dec 46   Hex 2E   Ascii .
				*/
				return (ascii_to_keycode(c));
			}
		}
		return(0);
	}
	c = inb(0x60);
	return((c));
}

void check_input()
{
	unsigned char c;

	if ((c = get_key())) {
		switch(c & 0x7f) {
		case 1:	
			/* "ESC" key was pressed, bail out.  */
			cprint(LINE_RANGE, COL_MID+23, "Halting... ");

			/* tell the BIOS to do a warm start */
			*((unsigned short *)0x472) = 0x1234;
			cache_on();
			outb(0xfe,0x64);
			break;
		case 46:
			/* c - Configure */
			get_config();
			break;
		case 28:
			/* CR - clear scroll lock */
			slock = 0;
			footer();
			break;
		case 57:
			/* SP - set scroll lock */
			slock = 1;
			footer();
			break;
		}
	}
}

void footer()
{
	cprint(24, 0, "(ESC)exit  (c)configuration  (SP)scroll_lock  (CR)scroll_unlock");
	if (slock) {
		cprint(24, 74, "Locked");
	} else {
		cprint(24, 74, "      ");
	}
}

ulong getval(x, y)
{
	int flag=0, c, n=0;
	ulong i, val, mult;
	char buf[16];
	char str[2];

	str[1] = 0;
	wait_keyup();
	while (!flag) {
		c = get_key();

		/* If CR and something has been entered, bail */
		if (c == 28 && n) {
			break;
		}

		/* M */
		if (c == 50 && n) {
			cprint(x, y+n, "m");
			buf[n] = 'm';
			if (n < 12) {
				n++;
			}
			continue;
		}

		/* K */
		if (c == 37 && n) {
			cprint(x, y+n, "k");
			buf[n] = 'k';
			if (n < 12) {
				n++;
			}
			continue;
		}

		/* Backspace */
		if (c == 14 && n) {
			n--;
			str[0] = ' ';
			cprint(x, y+n, str);
		}

		c--;
		if (c < 1 || c > 10) {
			continue;
		}
		if (c == 10) {
			c = 0;
		}

		str [0] = '0'+c;
		cprint(x, y+n, str);
		buf[n] = c;
		if (n < 12) {
			n++;
		}
	}

	/* Convert stored values to an integer */
	i = 1;
	val = 0;
	mult = 1;
	while (n--) {
		if (buf[n] == 'm') {
			mult = 1048576;
			continue;
		}
		if (buf[n] == 'k') {
			mult = 1024;
			continue;
		}

		val += buf[n] * i;
		i *= 10;
	}
	return(val * mult);
}

void ttyprint(int y, int x, char *p)
{
       static char sx[3];
       static char sy[3];

       sx[0]='\0';
       sy[0]='\0';
       x++; y++;
       itoa(sx, x);
       itoa(sy, y);
       serial_echo_print("[");
       serial_echo_print(sy);
       serial_echo_print(";");
       serial_echo_print(sx);
       serial_echo_print("H");
       serial_echo_print(p);
}

#if ((115200%SERIAL_BAUD_RATE) != 0)
#error Bad ttys0 baud rate
#endif

#define SERIAL_DIV     (115200/SERIAL_BAUD_RATE)

void serial_echo_init()
{
	int comstat, hi, lo;
	
	/* read the Divisor Latch */
	comstat = serial_echo_inb(UART_LCR);
	serial_echo_outb(comstat | UART_LCR_DLAB, UART_LCR);
	hi = serial_echo_inb(UART_DLM);
	lo = serial_echo_inb(UART_DLL);
	serial_echo_outb(comstat, UART_LCR);

	/* now do hardwired init */
	serial_echo_outb(0x03, UART_LCR); /* No parity, 8 data bits, 1 stop */
	serial_echo_outb(0x83, UART_LCR); /* Access divisor latch */
	serial_echo_outb(SERIAL_DIV & 0xff, UART_DLL);  /* baud rate divisor */
	serial_echo_outb((SERIAL_DIV>> 8) & 0xff, UART_DLM);
	serial_echo_outb(0x03, UART_LCR); /* Done with divisor */

	/* Prior to disabling interrupts, read the LSR and RBR
	 * registers */
	comstat = serial_echo_inb(UART_LSR); /* COM? LSR */
	comstat = serial_echo_inb(UART_RX);	/* COM? RBR */
	serial_echo_outb(0x00, UART_IER); /* Disable all interrupts */

        clear_screen_buf();

	return;
}

void serial_echo_print(char *p)
{
	if (!serial_cons) {
		return;
	}
	/* Now, do each character */
	while (*p) {
		WAIT_FOR_XMITR;

		/* Send the character out. */
		serial_echo_outb(*p, UART_TX);
		if(*p==10) {
			WAIT_FOR_XMITR;
			serial_echo_outb(13, UART_TX);
		}
		p++;
	}
}

/* This map needs to be updated whenever a new keycode
   is searched for.  I don't have a complete keycode
   map, so I don't know how to fill it all in!
*/

struct ascii_map_str ser_map[] =
/*ascii keycode     ascii  keycode*/
{ {  99,     46}, /* c    -> c    */
  {  67,     46}, /* C    -> c    */
  {  27,      1}, /* ESC  -> ESC  */
  {  10,     28}, /* NL   -> CR   */
  {  13,     28}, /* CR   -> CR   */
  {  32,     28}, /* SPACE-> CR   */
  { 109,     50}, /* m    -> m    */
  {  77,     50}, /* M    -> M    */
  { 110,     49}, /* n    -> n    */
  {  78,     49}, /* N    -> n    */
  { 107,     37}, /* k    -> k    */
  {  75,     37}, /* K    -> K    */
  {  87,     21}, /* y    -> y    */
  { 119,     21}, /* Y    -> y    */
  {   8,     14}, /* BS   -> BS   */
  { 127,     14}, /* DEL  -> BS   */
  {  48,     11}, /* 0    -> 0    */
  {  49,      2}, /* 1    -> 1    */
  {  50,      3}, /* 2    -> 2    */
  {  51,      4}, /* 3    -> 3    */
  {  52,      5}, /* 4    -> 4    */
  {  53,      6}, /* 5    -> 5    */
  {  54,      7}, /* 6    -> 6    */
  {  55,      8}, /* 7    -> 7    */
  {  56,      9}, /* 8    -> 8    */
  {  57,     10}, /* 9    -> 9    */
  {  19,     57}, /* ^S   -> SP   */
  {  17,     28}, /* ^Q   -> CR   */
  {   0,      0}
};

/*
 * Given an ascii character, return the keycode
 *
 * Uses ser_map definition above.
 *
 * It would be more efficient to use an array of 255 characters
 * and directly index into it.
 */
int ascii_to_keycode (int in)
{
	struct ascii_map_str *p;
	for (p = ser_map; p->ascii; p++) {
		if (in ==p->ascii)
			return p->keycode;
	}
	return 0;
}

/*
 * Call this when you want to wait for the user to lift the
 * finger off of a key.  It is a noop if you are using a
 * serial console.
 */
void wait_keyup( void ) {
	/* Check to see if someone lifted the keyboard key */
	while (1) {
		if ((get_key() & 0x80) != 0) {
			return;
		}
		/* Trying to simulate waiting for a key release with
		 * the serial port is to nasty to let live.
		 * In particular some menus don't even display until
		 * you release the key that caused to to get there.
		 * With the serial port this results in double pressing
		 * or something worse for just about every key.
		 */
		if (serial_cons) {
			return;
		}
	}
}
#ifdef LP
#define DATA            0x00
#define STATUS          0x01
#define CONTROL         0x02

#define LP_PBUSY        0x80  /* inverted input, active high */
#define LP_PERRORP      0x08  /* unchanged input, active low */

#define LP_PSELECP      0x08  /* inverted output, active low */
#define LP_PINITP       0x04  /* unchanged output, active low */
#define LP_PSTROBE      0x01  /* short high output on raising edge */

#define DELAY 0x10c6ul

void lp_wait(ulong xloops)
{
	int d0;
	__asm__("mull %0"
		:"=d" (xloops), "=&a" (d0)
		:"1" (xloops),"0" (current_cpu_data.loops_per_sec));
	__delay(xloops);
}
static void __delay(ulong loops)
{
	int d0;
	__asm__ __volatile__(
		"\tjmp 1f\n"
		".align 16\n"
		"1:\tjmp 2f\n"
		".align 16\n"
		"2:\tdecl %0\n\tjns 2b"
		:"=&a" (d0)
		:"0" (loops));
}

put_lp(char c, short port)
{
	unsigned char status;

	/* Wait for printer to be ready */
	while (1) {
		status = inb(STATUS(port));
		if (status & LP_PERRORP) {
			if (status & LP_PBUSY) {
				break;
			}
		}
	}

	outb(d, DATA(c));
	lp_wait(DELAY);
	outb((LP_PSELECP | LP_PINITP | LP_PSTROBE), CONTROL(port));
	lp_wait(DELAY);
	outb((LP_PSELECP | LP_PINITP), CONTROL(port));
	lp_wait(DELAY);
}
#endif

