/* melon.c -- a mailbox flag for X 
 *
 * Copyright(C) 2001-2002 Elisa Manara <e@entropika.net>
 * This code is released under the GPL License version 2 */

#include <gtk/gtk.h>
#include <fcntl.h>
#include <sys/stat.h>
#include <sys/wait.h>
#include <unistd.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <dirent.h>
#include "melon.h"

/* globals */
struct widgets_data	w_data;
struct pixmaps_data	p_data[MAX_STYLE_DIRS];
struct config_data	cfg_data;

char active_mbox[MAX_MBOX][MAX_PLEN];
int ignore_pid = 0;
//int pixmap_width = 0, pixmap_height = 0;

static int check_pixmaps(void);
static gint buttons_handler(GtkWidget *widget, GdkEvent *event);
static void load_config(void);
static void mboxes_init(void);
static void set_default_cfg (void);
static void set_field(char *buf, char *field);
static void set_config_file(char *config_file_path);
static void write_pid_file(void);
static void parse_command_line (int argc, char **argv);
static void extract_coordinates(char *arg_string);

char *set_melon_dir(void);
void save_status(void);
int retrieve_mbox_status(char *mbox, int i);
void melon_signals_connect(void);
void melon_init_control(int argc, char **argv);
void set_pid_file(char *pid_file_path);
void melon_end(void);
pid_t check_pid(void);
gint melon_handle_signals(gpointer data);
void pidfile_action_dialog (void);
void set_melon_position (void);
void melon_usage(void);

#define MELON_VERSION_BANNER  \
"Melon version %s\n" \
"Copyright(C) 2001-2002 Elisa Manara <e@entropika.net>\n\n" \
"PIXMAP_ROOT: %s\n" \
"AUDIO_ROOT (au): %s\n" \
"AUDIO_ROOT (voc): %s\n"

int main (int argc, char *argv[])
{
	/* GtkWidget is the storage type for widgets */
	GtkWidget *window, *pixmap, *fixed;
	GdkPixmap *gdk_pixmap, *new_gdk_pixmap;
	GdkBitmap *mask1, *mask2;
	GtkStyle *style;
	GdkGC *gc;
	int pixmap_width, pixmap_height;
	int c;

	opterr = 0;
	while((c = getopt(argc, argv, "vph")) != EOF) {
		switch(c) {
		case 'v':
			printf(MELON_VERSION_BANNER, MELON_VERSION, PIXMAPS_DIR, 
				BEEP_DIR_AU, BEEP_DIR_VOC);
			exit(0);
		case 'p':
			ignore_pid = 1;
			break;
		case 'h':
			melon_usage();
			break;
		}
	}

	/* create melon dir */
	set_melon_dir();
	/* set signals */
	melon_signals_connect();
	/* check melon process */
	if (!ignore_pid) {
		melon_init_control(argc, argv);
	}
	if (!ignore_pid) {
		/* write pid file */
		write_pid_file();
	}

	/* initialize values of p_data struct */
	check_pixmaps();
	/* load configuration */
	load_config();
	/* initialize mboxes */
	mboxes_init();

	gtk_init (&argc, &argv);
	parse_command_line(argc, argv);

	/* set default style */
	set_style();

	/* Create the main window */
	window = gtk_window_new( GTK_WINDOW_POPUP );
	gtk_window_set_title(GTK_WINDOW( window ), "melon");
	w_data.window = window;

	gtk_signal_connect (GTK_OBJECT (window), "delete_event",
		GTK_SIGNAL_FUNC (close_application), NULL);

	/* press button */
	gtk_signal_connect_object(GTK_OBJECT(window), "button_press_event",
		GTK_SIGNAL_FUNC (buttons_handler), NULL);

	gtk_widget_show (window);

	/* pixmap and pixmap widget */
	style = gtk_widget_get_default_style();
	gc = style->black_gc;

	/* check if pixmap files exist*/
	if(!pixmap_exist(cfg_data.no_mail_pix) || !pixmap_exist(cfg_data.new_mail_pix)) {
		melon_error("Cannot load pixmap files:\n   %s\n   %s\n", 
			cfg_data.no_mail_pix, cfg_data.new_mail_pix);
		exit(1);
	}

	gdk_pixmap = gdk_pixmap_create_from_xpm( window->window, &mask1,
		&style->bg[GTK_STATE_NORMAL],
		cfg_data.no_mail_pix);

	new_gdk_pixmap = gdk_pixmap_create_from_xpm( window->window, &mask2,
		&style->bg[GTK_STATE_NORMAL],
		cfg_data.new_mail_pix);

	pixmap = gtk_pixmap_new( gdk_pixmap, mask1 );

	/* fill struct */
	w_data.pixmap = pixmap;
	w_data.gdk_pixmap = gdk_pixmap;
	w_data.new_gdk_pixmap = new_gdk_pixmap;
	w_data.mask1 = mask1;
	w_data.mask2 = mask2;

	gtk_widget_show( pixmap );

	/* get the pixmap size */
	gdk_window_get_size (w_data.gdk_pixmap, &pixmap_width, &pixmap_height); 

	/* Fixed widget to place the pixmap */
	fixed = gtk_fixed_new();
	gtk_widget_set_usize( fixed, pixmap_width, pixmap_height );
	gtk_fixed_put( GTK_FIXED(fixed), pixmap, 0, 0 );
	gtk_container_add( GTK_CONTAINER(window), fixed );
	gtk_widget_show( fixed );
	
	/* This masks out everything except for the image itself */
	gtk_widget_shape_combine_mask( window, mask1, 0, 0 );

	/* Set window position */
	set_melon_position();
	/* show the window */
	gtk_widget_show( window );

	gtk_timeout_add (cfg_data.interval, check_mbox, NULL);
	gtk_timeout_add (250, melon_handle_signals, NULL);

	gtk_main ();

	return(0);
}

/* When invoked (via signal delete_event), terminates the application */
gint close_application( GtkWidget *widget,
                        GdkEvent  *event,
                        gpointer   data )
{
	melon_end();
	gtk_main_quit();
	return(FALSE);
}

gint buttons_handler(GtkWidget *widget, GdkEvent *event)
{
	GdkEventButton *event_button;

	if (event->type == GDK_BUTTON_PRESS)
	{
		event_button = (GdkEventButton *) event;
		/* press button 1 */
		if (event_button->button == 1) {
			active_mbox_menu(event_button);
		/* press button 3 */
		} else if (event_button->button == 3) {
			pop_menu(event_button);
			
			return TRUE;
		}
	} 
	return FALSE;
}

int check_pixmaps(void)
{
	DIR *dir_entries, *style_dir;
	struct dirent *dr, *px;
	char styles_dir_path[MAX_PLEN], style_pix[MAX_PLEN];
	char *error_string;
	int c = 0;

	if((dir_entries = opendir(PIXMAPS_DIR)) == NULL) {
		error_string=strerror(errno);
		melon_error("Cannot open %s: %s\n", PIXMAPS_DIR, error_string);
		exit(1);
	}

	while((dr = readdir(dir_entries)) != NULL)
	{
		//if (!strcmp(dr->d_name, ".") || !strcmp(dr->d_name, ".."))
		if (!strncmp(dr->d_name, ".", 1))
			continue;

		strncpy(p_data[c].pix_name, dr->d_name, MAX_PLEN);

		/* path of every style dir in pixmaps_dir */
		snprintf(styles_dir_path, MAX_PLEN, "%s/%s", PIXMAPS_DIR, dr->d_name);

		if((style_dir = opendir(styles_dir_path)) == NULL) {
			error_string=strerror(errno);
			melon_error("Cannot open %s: %s\n", styles_dir_path, error_string);
			continue;
		}

		while((px = readdir(style_dir)) != NULL)
		{
			char *p;

			if (!strcmp(px->d_name, ".") || !strcmp(px->d_name, ".."))
				continue;
			if ((p = strrchr(px->d_name, '.')) == NULL || strcmp(p, ".xpm"))
				continue;
			/* path of every pix in style pix dir */
			snprintf(style_pix, MAX_PLEN, "%s/%s", styles_dir_path, px->d_name);

			if (!strncmp(px->d_name, "1", 1)) {
				strncpy(p_data[c].no_mail_pix, style_pix, MAX_PLEN);
			} else if (!strncmp(px->d_name, "2", 1)) {
				strncpy(p_data[c].new_mail_pix, style_pix, MAX_PLEN);
			}
		}
		closedir(style_dir);
		c++;
	}
	closedir(dir_entries);

	/* initialize to void strings remaining data of struct */
        while (c < MAX_STYLE_DIRS)
        {
		p_data[c].pix_name[0] = '\0';
		p_data[c].no_mail_pix[0] = '\0';
		p_data[c].new_mail_pix[0] = '\0';
		c++;
        }

	return 0;
}

void config_update (void)
{
	FILE *fp;
	int i;
	char config_file[MAX_PLEN];

	set_config_file(config_file);
	if((fp = fopen(config_file, "w")) == NULL) {
		char msg[1024];
		snprintf(msg, 1024, " Cannot open %s for writing ", config_file);
		dialog_message (msg);
		return;
	}

	fprintf(fp, "NO_MAIL_PIX:%s\n", cfg_data.no_mail_pix);
	fprintf(fp, "NEW_MAIL_PIX:%s\n", cfg_data.new_mail_pix);
	fprintf(fp, "BEEP:%s\n", cfg_data.beep_file);
	fprintf(fp, "BEEP_DEVICE:%s\n", cfg_data.beep_device);
	fprintf(fp, "BEEP_MODE:%d\n", cfg_data.beep_mode);
	fprintf(fp, "INTERVAL:%d\n", cfg_data.interval);
	fprintf(fp, "XPOSITION:%s\nYPOSITION:%s\n", cfg_data.xposition, cfg_data.yposition);
	fprintf(fp, "SAVE_STATUS:%d\n", cfg_data.save_status);
	fprintf(fp, "MAILER:%s\n", cfg_data.mailer);
	if((strcmp(cfg_data.color_style, "gtk_default")) == 0) {
		fprintf(fp, "COLOR_STYLE:gtk_default\n");
	} else {
		fprintf(fp, "BGCOLOR:%04x %04x %04x\n", 
			cfg_data.bgcolor.red, cfg_data.bgcolor.green, cfg_data.bgcolor.blue);
		fprintf(fp, "FGCOLOR:%04x %04x %04x\n", 
			cfg_data.fgcolor.red, cfg_data.fgcolor.green, cfg_data.fgcolor.blue);
		fprintf(fp, "ABGCOLOR:%04x %04x %04x\n", 
			cfg_data.abgcolor.red, cfg_data.abgcolor.green, cfg_data.abgcolor.blue);
		fprintf(fp, "AFGCOLOR:%04x %04x %04x\n", 
			cfg_data.afgcolor.red, cfg_data.afgcolor.green, cfg_data.afgcolor.blue);
	}

	for(i = 0; i < MAX_MBOX; i++)
	{
		if(cfg_data.mbox[i][0] == '\0')
			break;

		if(cfg_data.is_a_mbox[i])
			fprintf(fp,"MBOX:%s\n", cfg_data.mbox[i]);
		else
			fprintf(fp,"FILE:%s\n", cfg_data.mbox[i]);

	}
	fclose(fp);
}

void load_config(void)
{
	FILE *fp;
	char buf[1024];
	int i;
	char config_file[MAX_PLEN];
	char *fields[] = { "NO_MAIL_PIX:", "NEW_MAIL_PIX:", "BEEP:",
			   "BEEP_MODE:", "BEEP_DEVICE:", "INTERVAL:", 
			   "XPOSITION:", "YPOSITION:", "SAVE_STATUS:", 
			   "MAILER:", "COLOR_STYLE:", "BGCOLOR:", "FGCOLOR:", 
			   "ABGCOLOR:", "AFGCOLOR:", "MBOX:", "FILE:", NULL };

	set_config_file(config_file);
	/* set default configuration and initialize structs*/
	set_default_cfg();

	if((fp = fopen(config_file, "r")) == NULL)
		return;
	
	while((fgets(buf, 1024, fp)) != NULL)
	{
		int l = strlen(buf);
		if (l && buf[l-1] == '\n')
			buf[l-1] = '\0';
		for(i = 0; fields[i]; i++)
			set_field(buf, fields[i]);
	}
	fclose(fp);

	if (cfg_data.mbox[0][0] == '\0')
		melon_error("No configured mailbox\n");
}

void set_field(char *buf, char *field)
{
	char *st = NULL;
	static int mcount = 0;
	int dest_buf_size = MAX_PLEN;

	if((strncmp(buf, field, strlen(field))) == 0) {
		buf+=strlen(field);
		if((strcmp(field, "NO_MAIL_PIX:")) == 0) {
			st = cfg_data.no_mail_pix;
		} else if ((strcmp(field, "NEW_MAIL_PIX:")) == 0) {
			st = cfg_data.new_mail_pix;
		} else if ((strcmp(field, "BEEP:")) == 0) {
			st = cfg_data.beep_file;
		} else if ((strcmp(field, "BEEP_DEVICE:")) == 0) {
			st = cfg_data.beep_device;
		} else if ((strcmp(field, "BEEP_MODE:")) == 0) {
			cfg_data.beep_mode = atoi(buf);
			return;
		} else if ((strcmp(field, "INTERVAL:")) == 0) {
			cfg_data.interval = atoi(buf);
			if (cfg_data.interval <= 0) {
				fprintf(stderr, "Invalid interval value <= 0\n");
				melon_end();
				exit(1);
			}
			return;
		} else if ((strcmp(field, "XPOSITION:")) == 0) {
			st = cfg_data.xposition;
			dest_buf_size = 16;
		} else if ((strcmp(field, "YPOSITION:")) == 0) {
			st = cfg_data.yposition;
			dest_buf_size = 16;
		} else if ((strcmp(field, "SAVE_STATUS:")) == 0) {
			cfg_data.save_status = atoi(buf);
			return;
		} else if ((strcmp(field, "MAILER:")) == 0) {
			st = cfg_data.mailer;
		} else if ((strcmp(field, "COLOR_STYLE:")) == 0) {
			st = cfg_data.color_style;
		} else if ((strcmp(field, "BGCOLOR:")) == 0) {
			sscanf(buf, "%hx %hx %hx\n", 
			&cfg_data.bgcolor.red, &cfg_data.bgcolor.green, &cfg_data.bgcolor.blue);
			return;
		} else if ((strcmp(field, "FGCOLOR:")) == 0) {
			sscanf(buf, "%hx %hx %hx\n", 
			&cfg_data.fgcolor.red, &cfg_data.fgcolor.green, &cfg_data.fgcolor.blue);
			return;
		} else if ((strcmp(field, "ABGCOLOR:")) == 0) {
			sscanf(buf, "%hx %hx %hx\n", 
			&cfg_data.abgcolor.red, &cfg_data.abgcolor.green, &cfg_data.abgcolor.blue);
			return;
		} else if ((strcmp(field, "AFGCOLOR:")) == 0) {
			sscanf(buf, "%hx %hx %hx\n", 
			&cfg_data.afgcolor.red, &cfg_data.afgcolor.green, &cfg_data.afgcolor.blue);
			return;
		} else if ((strcmp(field, "MBOX:")) == 0) {
			st = cfg_data.mbox[mcount];
			cfg_data.is_a_mbox[mcount] = 1;
			mcount++;
		} else if ((strcmp(field, "FILE:")) == 0) {
			st = cfg_data.mbox[mcount];
			cfg_data.is_a_mbox[mcount] = 0;
			mcount++;
		} else {
			melon_error("Unknown keyword '%s' in config file\n", field);
			return;
		}
		strncpy(st, buf, dest_buf_size);
	}
}

void set_default_cfg (void)
{
	int i;
	char *mbox;

	set_melon_def_colors();

	strncpy(cfg_data.no_mail_pix, NO_MAIL_PIX, MAX_PLEN);
	strncpy(cfg_data.new_mail_pix, NEW_MAIL_PIX, MAX_PLEN);
	cfg_data.beep_file[0] = '\0';
	cfg_data.beep_device[0] = '\0';

	cfg_data.beep_mode = 0;
	cfg_data.interval = 5000;
	cfg_data.save_status = 0;
	strncpy(cfg_data.xposition, "-0", 16);
	strncpy(cfg_data.yposition, "-0", 16);

	for (i = 0; i < MAX_MBOX; i++) {
		/* initialize cfg_data.mbox */
		cfg_data.mbox[i][0] = '\0';
		/* initialize active_mbox */
		active_mbox[i][0] = '\0';
		/* initialize is_a_mbox */
		cfg_data.is_a_mbox[i] = 0;
	}
	if((mbox = getenv("MAIL"))) {
		strncpy(cfg_data.mbox[0], mbox, MAX_PLEN);
		cfg_data.is_a_mbox[0] = 1;
	}

	cfg_data.mailer[0] = '\0';
}

int pixmap_exist (char *path)
{
	int retval;
	struct stat buf;
	
	if((retval = stat(path, &buf)) == -1)
		return 0;;
	if(!S_ISREG(buf.st_mode))
		return 0;

	return 1;
}

void mboxes_init(void)
{
	int i;
	int retval;
	struct stat buf;

	for (i = 0; i < MAX_MBOX; i++)
	{
		if (cfg_data.mbox[i][0] == '\0')
			break;

		w_data.beep[i] = 0;
		w_data.unread_msg[i] = 0;

		/* retrieve the saved status from status file */
		if(cfg_data.save_status == 1) {
			if((retval = retrieve_mbox_status(cfg_data.mbox[i], i)) == 1)
				continue;
		}
		/* otherwise, set w_data.m_time to the modification time */
		if((retval = stat(cfg_data.mbox[i], &buf)) == -1) {
			melon_error("Stat error on initializing '%s': %s\n", 
				cfg_data.mbox[i], strerror(errno));
			melon_end();
			exit(1);
		}
		w_data.m_time[i] = buf.st_mtime;
	}
}

char *set_melon_dir(void)
{
	char *home;
	char *error_string;
	char *melon_home_dir;
	char m_home_dir[MAX_PLEN];
	char old_cfg_file_path[MAX_PLEN];
	struct stat buf;
	int retval;

	if((home=getenv("HOME")) == NULL) {
		melon_error("Please, set correctly your HOME environment variable!\n");
		exit(1);
	}

	snprintf(m_home_dir, MAX_PLEN, "%s/%s", home, ".melon");


	if((retval = stat(m_home_dir, &buf)) == -1) {
		if(errno != ENOENT) {
			error_string=strerror(errno);
			melon_error("Stat dir %s: %s\n", m_home_dir, error_string);
			exit(1);
		} else {
			/* create melon dir ~/.melon if don't exist */
			if((retval=mkdir(m_home_dir, 0755)) < 0) {
				error_string=strerror(errno);
				melon_error("Cannot create dir %s: %s\n", 
					     m_home_dir, error_string);
				exit(1);
			}
			/* look for an old ~/.melon.cfg file */
			snprintf(old_cfg_file_path, MAX_PLEN, "%s/.melon.cfg", home);
			if((retval = stat(old_cfg_file_path, &buf)) == 0) {
				char msg[1024];
				snprintf(msg, 1024, "An old melon.cfg file was found in your "\
						    "home directory.\nPlease, don't forget to "\
						    "rename it to %s/melon.cfg if you "\
						    "means to use it.\n",
						      m_home_dir);


				melon_error(msg); 
			}
		}
	} else {
		if(!S_ISDIR(buf.st_mode)) {
			melon_error("Melon dir error: %s exists but it's not a directory.\n", 
				     m_home_dir);
			exit(1);
		}
	}

	if((melon_home_dir=malloc(MAX_PLEN)) == NULL) {
		melon_error("Malloc fails");
		exit(1);
	}
	strncpy(melon_home_dir, m_home_dir, MAX_PLEN);
	return melon_home_dir;
}
	
pid_t check_pid(void)
{
	FILE *p_file;
	char pid_file[MAX_PLEN];
	pid_t m_pid;

	set_pid_file(pid_file);

	/* check if a pid_file exists, i.e. the program is already running */
	if((p_file=fopen(pid_file, "r")) == NULL) {
		if(errno != ENOENT) {
			melon_error("Pid file %s: %s\n", pid_file, strerror(errno));
			exit(1);
		} else {
			/* no other melon is running */
			return 0;
		}
	} else {
		/* an other melon is already running */
		/* read the pid from file */
		/* TODO: controlla! */
		fscanf(p_file, "%d\n", &m_pid);
		fclose(p_file);
		return m_pid;
	}
}

void write_pid_file(void)
{
	pid_t melon_pid;
	FILE *p_file;
	char pid_file[MAX_PLEN];
	char *error_string;

	set_pid_file(pid_file);

	melon_pid=getpid();

	if((p_file=fopen(pid_file, "w")) == NULL) {
		error_string=strerror(errno);
		melon_error("Cannot open %sfor writing: %s\n", pid_file, error_string);
		exit(1);
	}

	fprintf(p_file, "%d\n", melon_pid);
	fclose(p_file);
}

void set_pid_file(char *pid_file_path)
{
	char *melon_home_dir=set_melon_dir();

	snprintf(pid_file_path, MAX_PLEN, "%s/%s", melon_home_dir, "melon.pid");
	free(melon_home_dir);
}
	
void set_config_file(char *config_file_path)
{
	char *melon_home_dir=set_melon_dir();
	snprintf(config_file_path, MAX_PLEN, "%s/%s", melon_home_dir, "melon.cfg");
	free(melon_home_dir);
}

int melon_error(const char *fmt, ...)
{
	int ret = 0;
	va_list ap;

	va_start(ap, fmt);

	ret = vfprintf(stderr, fmt, ap);
	va_end(ap);

	return ret;
}

void melon_end(void)
{
	char pid_file[MAX_PLEN];
	char *error_string;
	int retval;

	save_status();
	if(!ignore_pid) {
		set_pid_file(pid_file);
		if((retval=unlink(pid_file)) < 0) {
			if(errno == ENOENT)
				return;
			error_string=strerror(errno);
			melon_error("Error while unlinking pid file %s: %s\n", 
				pid_file, error_string);
		}
	}
}

void melon_usage(void)
{
	printf("Melon version %s\n"
	"Copyright(C) 2001-2002 Elisa Manara <e@entropika.net>\n\n",
	MELON_VERSION);

	printf("Usage: melon [OPTIONS]\n");
	printf("\n\t%-15.20s %-35.35s\n", "-p", "ignore pidfile");
	printf("\t%-15.20s %-35.35s\n", "-g, -geometry", "specify windows coordinates");
	printf("\t%-15.20s %-35.35s\n", "", "example: melon -geometry -1+10");
	printf("\t%-15.20s %-35.35s\n", "-v", "show version and paths");
	printf("\t%-15.20s %-35.35s\n", "-h", "print this help");
	exit(0);
}

void parse_command_line (int argc, char **argv)
{
	int arg_options;

	for(arg_options = 0; arg_options < argc; arg_options++)
		if(((strcmp(argv[arg_options], "-geometry")) == 0) ||
			(strcmp(argv[arg_options], "-g") == 0)) {
			if(++arg_options == argc) {
				melon_error("Geometry not specified: option ignored.\n");
				break;
			}
			extract_coordinates(argv[arg_options]);
			/*
			if((sscanf(argv[arg_options], "%d %d", 
				&cfg_data.xposition, &cfg_data.yposition)) < 2)
					melon_error("Geometry format not valid: option ignored.\n");
			*/
			/* no other options recognized yet */
			break;
		}
}

void extract_coordinates (char *arg_string)
{
	char *ypos;
	char xpos[16];
	char *p;
	int i;
	int ignore = 0;

	xpos[0] = '\0';
	if((ypos=strrchr(arg_string, '+')) == NULL)
		if((ypos=strrchr(arg_string, '-')) == NULL) {
			melon_error("Geometry format not valid: option ignored.\n");
			ignore = 1;
		}

	for(i = 0, p = arg_string; p < ypos; i++, p++)
		xpos[i] = *p;
	xpos[i] = '\0';

	if(xpos[0] == '\0') {
		melon_error("Geometry format not valid: option ignored.\n");
		ignore = 1;
	}
	if(!ignore) {
		strncpy(cfg_data.xposition, xpos, 16);
		strncpy(cfg_data.yposition, ypos, 16);
	}
}
