/* mbox.c -- mailbox status handling
 *
 * Copyright(C) 2001-2002 Elisa Manara <e@entropika.net>
 * This code is released under the GPL License version 2 */

#include <gtk/gtk.h>
#include <sys/stat.h>
#include <unistd.h>
#include <stdlib.h>
#include <utime.h>
#include <stdio.h>
#include <sys/wait.h>
#include <string.h>
#include "melon.h"

static void mbox_file_check (int index, time_t m_time, time_t a_time, off_t size);
static void generic_file_check (int index, time_t m_time, off_t size);
static void mbox_set_active (char *mbox);
static void mbox_set_inactive (char *mbox);
static void clear_entry(GtkWidget *widget, gpointer user_data);
static void set_mailer (GtkWidget *widget, gpointer user_data);
char *my_strsep(char **stringp, const char *delim);
GtkWidget *entry;

extern char **environ;

gint check_mbox (gpointer data)
{
	int i;
	int retval;
	struct stat buf;

	/* check mailbox and choice pixmap to show */
	for (i = 0; i < MAX_MBOX; i++)
	{
		if (cfg_data.mbox[i][0] == '\0')
			break;
		if((retval = stat(cfg_data.mbox[i], &buf)) == -1) {
			melon_error("Stat error for mailbox %s: %s\n", 
				cfg_data.mbox[i], strerror(errno));
			break;
		} 

		/* if the modification time or file size are changed... */
		if (buf.st_mtime > w_data.m_time[i] || buf.st_size != w_data.size[i]) {
			if(cfg_data.is_a_mbox[i])
				/* mailbox file - check is based upon
				 * the status flag updated by MUA
				 */
				mbox_file_check(i, buf.st_mtime, buf.st_atime, buf.st_size);
			else
				/* generic file - check is based only upon
				 * the change time
				 */
				generic_file_check(i, buf.st_mtime, buf.st_size);
		} 
	} /* end of for */

	return 1;
}

void mbox_file_check (int index, time_t m_time, time_t a_time, off_t size)
{
	FILE *mailbox;
	char buf[1024];
	int header = 0;
	struct utimbuf ut;
	int start_header = 1;
	int msg_count = 0, status_count = 0, unread = 0;

	ut.actime = a_time;
	ut.modtime = m_time;

	if ((mailbox = fopen(cfg_data.mbox[index], "r")) == NULL) {
		melon_error("Error on reading mailbox '%s': %s\n",
			cfg_data.mbox[index], strerror(errno));
	}

	while (fgets(buf, 1024, mailbox)) {
		if (buf[0] == '\n') {
			/* newline: end of header */
			if (header)
				header = 0;
			else
				start_header = 1;
		} else if ((strncmp(buf, "From ", 5)) == 0) {
			/* a new header begin */
			if (start_header)
				header = 1;
			if (header)
				msg_count++;
		} else {
			start_header = 0;
			if ((strncmp(buf, "Status: ", 8)) == 0) {
				if (strrchr(buf, 'R')) {
					if (header)
						status_count++;
				}
			}
		}
	}
	fclose(mailbox);
	utime(cfg_data.mbox[index], &ut);

	unread = msg_count - status_count;

	if (unread == 0 || unread < w_data.unread_msg[index])
		reset_mbox_status(cfg_data.mbox[index]);
	else if(unread > w_data.unread_msg[index])
		generic_file_check(index, m_time, size);

	w_data.unread_msg[index] = unread;
}

void generic_file_check (int index, time_t m_time, off_t size)
{
	/* config: play only one time */
	if (cfg_data.beep_mode == 0 && !w_data.beep[index]) {
		play_beep();
		mbox_set_active(cfg_data.mbox[index]);
	/* config: play for each new e-mail */
	} else if (cfg_data.beep_mode == 1) {
		play_beep();
		if (!w_data.beep[index])
			mbox_set_active(cfg_data.mbox[index]);
		w_data.m_time[index] = m_time;
		w_data.size[index] = size;
	}

	w_data.beep[index] = 1;
	gtk_pixmap_set( GTK_PIXMAP(w_data.pixmap), 
		w_data.new_gdk_pixmap, w_data.mask2 );
	gtk_widget_shape_combine_mask( w_data.window, w_data.mask2, 0, 0 );
}

#define ARGS_MAX 32
gint see_mail (GtkWidget *widget, char *mbox)
{
	int retval;
	int childpid;
	int i;
	char *tok;
	char *cmd_string;
	char *mailer_args[ARGS_MAX];
	char *path = NULL;

	/* TODO */
	if (!mbox)
		mbox = cfg_data.mbox[0];

	if(cfg_data.mailer[0] == '\0') {
		reset_mbox_status(mbox);
		return 1;
	}

	if ((childpid = fork()) == -1) {
		perror ("fork");
	}

	/* child */
	if (childpid == 0) {
		for(i = 0; i < ARGS_MAX; i++)
			mailer_args[i] = NULL;
			
		cmd_string = strdup(cfg_data.mailer);
		i = 0;

		while((tok = my_strsep(&cmd_string, " ")) != NULL)
		{
			if (i == ARGS_MAX) {
				melon_error("Too much arguments in mailer command line\n");
				exit(1);
			}
			if(tok[0] == '\0')
				continue;
			if (i == 0) {
				char *p = strrchr(tok, '/');
				path = strdup(tok);
				if (!p)
					mailer_args[i] = strdup(tok);
				else
					mailer_args[i] = strdup(p+1);
			} else {
				if(strcmp(tok, "%M") == 0) {
					mailer_args[i] = strdup(mbox);
				} else {
					mailer_args[i] = strdup(tok);
				}
			}
			i++;
		}
		if (i == 0) { exit(0); } 

		/*
		printf("path: %s\n", path);
		for (i = 0; mailer_args[i]; i++)
			printf("%s, ", mailer_args[i]);
		printf("\n");
		*/

		free(cmd_string);

		retval = execve (path, mailer_args, environ);
		for (i = 0; i < ARGS_MAX; i++)
			if (mailer_args[i])
				free(mailer_args[i]);
		free(path);
	
		if (retval == -1) {
			char *error_string;

			error_string=strerror(errno);
			melon_error ("Cannot execute mailer program: %s\n", error_string);
		}

		_exit(1);
	} else {
		wait(NULL);
		reset_mbox_status(mbox);
	}

	return 1;
}

void reset_mbox_status (char *mbox)
{	
	struct stat buf;
	int i;
	int retval;

	for (i = 0; i < MAX_MBOX; i++)
	{
		if (cfg_data.mbox[i][0] == '\0')
			break;
		if (!strcmp(cfg_data.mbox[i], mbox)) {
			if((retval = stat(cfg_data.mbox[i], &buf)) == -1) {
				melon_error ("Stat: error on reset status of %s\n", 
					cfg_data.mbox[i]);
				return;
			}
			w_data.m_time[i] = buf.st_mtime;
			w_data.size[i] = buf.st_size;
			w_data.beep[i] = 0;
			break;
		}
	}

	mbox_set_inactive(mbox);
		
	/* reset icon if no one mbox is active */
	if(none_active_mbox()) {
		gtk_pixmap_set( GTK_PIXMAP(w_data.pixmap),
			w_data.gdk_pixmap, w_data.mask1 );
		gtk_widget_shape_combine_mask( w_data.window, w_data.mask1, 0, 0 );
	}
}

void mbox_set_active (char *mbox)
{
	int c;

	for(c = 0; c < MAX_MBOX; c++)
	{
		if(active_mbox[c][0] == '\0') {
			strncpy(active_mbox[c], mbox, MAX_PLEN);
			return;
		}
	}
}

void mbox_set_inactive (char *mbox)
{
	int c;

	for(c = 0; c < MAX_MBOX; c++)
	{
		if (!strcmp(active_mbox[c], mbox)) {
			active_mbox[c][0] = '\0';
			return;
		}
	}
}

int none_active_mbox(void)
{
	int c;

	for(c = 0; c < MAX_MBOX; c++)
	{
		if (active_mbox[c][0] != '\0') {
			return 0;
		}
	}

	return 1;
}

gint mailer_config (GtkWidget *widget, gpointer data)
{
	GtkWidget *window;
	GtkWidget *vbox, *hbox2;
	GtkWidget *button, *label;

	/* window */
	window = gtk_window_new( GTK_WINDOW_DIALOG);
	gtk_window_set_position(GTK_WINDOW (window), GTK_WIN_POS_MOUSE);
	gtk_widget_set_usize( GTK_WIDGET (window), 200, 70);
	gtk_window_set_title(GTK_WINDOW (window), "Configure mailer command");
	gtk_widget_show(window);

	/* vbox */
	vbox = gtk_vbox_new( FALSE, 0 );
	gtk_container_add( GTK_CONTAINER( window ), vbox );
	gtk_widget_show( vbox );

	/* label */
	label = gtk_label_new ("Command line: ");
	gtk_box_pack_start (GTK_BOX (vbox), label, FALSE, FALSE, 2);
	gtk_widget_show(label);

	/* entry */
	entry = gtk_entry_new();
	gtk_entry_set_max_length (GTK_ENTRY( entry ), MAX_PLEN);

	if(cfg_data.mailer[0] != '\0')
		gtk_entry_set_text (GTK_ENTRY( entry ), cfg_data.mailer);
	gtk_box_pack_start (GTK_BOX (vbox), entry, FALSE, FALSE, 2);
	gtk_widget_show(entry);

	/* hbox2 */
	hbox2 = gtk_hbox_new( FALSE, 0 );
	gtk_container_add( GTK_CONTAINER( vbox ), hbox2);
	gtk_widget_show( hbox2 );

	/* APPLY button */
	button = gtk_button_new_with_label ("Apply");
	gtk_signal_connect_object (GTK_OBJECT (button), "clicked",
		GTK_SIGNAL_FUNC(set_mailer), NULL);
	gtk_box_pack_start (GTK_BOX (hbox2), button, TRUE, TRUE, 0);
	gtk_widget_show (button);

	/* CLEAR button */
	button = gtk_button_new_with_label ("Clear");
	gtk_signal_connect_object (GTK_OBJECT (button), "clicked",
		GTK_SIGNAL_FUNC(clear_entry), NULL);
	gtk_box_pack_start (GTK_BOX (hbox2), button, TRUE, TRUE, 0);
	gtk_widget_show (button);

	/* CLOSE button */
	button = gtk_button_new_with_label ("Close");
	gtk_signal_connect_object (GTK_OBJECT (button), "clicked",
		GTK_SIGNAL_FUNC(gtk_widget_destroy),
		GTK_OBJECT (window));
	gtk_box_pack_start (GTK_BOX (hbox2), button, TRUE, TRUE, 0);
	gtk_widget_show (button);

	return 1;

}

void clear_entry(GtkWidget *widget, gpointer user_data)
{
	gtk_entry_set_text(GTK_ENTRY( entry ), "");
	return;
}

void set_mailer(GtkWidget *widget, gpointer user_data)
{
	char *text;

	text = gtk_editable_get_chars(GTK_EDITABLE( entry ), 0, -1);
	if (text[0] != '\0') {
		strncpy(cfg_data.mailer, text, MAX_PLEN);
	} else {
		cfg_data.mailer[0] = '\0';
	}

	g_free(text);

	dialog_message (" Mailer command configured ");
	config_update();

	return;
}
