#   MailScanner - SMTP E-Mail Virus Scanner
#   Copyright (C) 2001  Julian Field
#
#   $Id: sendmail.pl,v 1.64 2002/03/02 10:41:40 jkf Exp $
#
#   This program is free software; you can redistribute it and/or modify
#   it under the terms of the GNU General Public License as published by
#   the Free Software Foundation; either version 2 of the License, or
#   (at your option) any later version.
#
#   This program is distributed in the hope that it will be useful,
#   but WITHOUT ANY WARRANTY; without even the implied warranty of
#   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#   GNU General Public License for more details.
#
#   You should have received a copy of the GNU General Public License
#   along with this program; if not, write to the Free Software
#   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#
#   The author, Julian Field, can be contacted by email at
#      Jules@JulianField.net
#   or by paper mail at
#      Julian Field
#      Dept of Electronics & Computer Science
#      University of Southampton
#      Southampton
#      SO17 1BJ
#      United Kingdom
#

# Scan the incoming sendmail queue and find all the messages ready for
# scanning.

use strict;
use Socket;
use FileHandle;
use MIME::Entity;

package Sendmail;

# Only do this if we want to use SpamAssassin and therefore have it installed.
# Justin Mason advises only creating 1 Mail::SpamAssassin object, so I do it
# here while we are starting up.
my($SAspamtest);
if ($Config::SpamAssassin) {
  require Mail::SpamAssassin;
  # JKF 2/3/2002 Commented out next line for SpamAssassin 2.1
  #require Mail::SpamAssassin::MyMailAudit;
  $SAspamtest = new Mail::SpamAssassin();
  # JKF 7/1/2002 Commented out due to it causing false positives
  #$SAspamtest->compile_now(); # Saves me recompiling all the modules every time
}

my($SendmailOptions) = "-t -oi -oem -F MailScanner -f $Config::LocalPostmaster";

# List all the message id's of the files in the incoming queue that are ready
# to be scanned. Will always return at least 1 message if available.
# Parameters are:
#  Sendmail incoming queue directory
#  Scanner incoming work directory
#  Ref to fill with list of definitely-clean messages
#  Ref to fill with list of possibly-dirty messages
#  Ref to fill with hash of sender+\0+recipient+\0+subject info about messages
#  Ref to fill with number of possibly-dirty messages
#  Ref to fill with total size of possibly-dirty messages
#  Optional max number of definitely-clean messages to process
#  Optional max number of definitely-dirty messages to process
#  Optional max size of definitely-clean messages to process
#  Optional max size of possibly-dirty messages to process
sub FindMessagesToProcess {
  my($InQueueDir, $OutDir, $RClean, $RUnscanned, $RDirty, $MessagesInfo,
     $Headers, $RDirtyMsgs, $RDirtyBytes) = @_;
  local(*QUEUE);
  my($file, $id, $RHeaders, $MsgInfo);
  my($CleanMsgs, $DirtyMsgs, $CleanBytes, $DirtyBytes);
  my($HitLimit1, $HitLimit2, $HitLimit3, $HitLimit4);
  my(%ModDate, @SortedFiles);
  # necessary because otherwise we need to pass a reference to
  # *QFFILE before it exists... apparently this is OK in perl 5.6,
  # but not in 5.005...
  my $QfFile = new FileHandle;

  $$RDirtyMsgs = 0;
  $$RDirtyBytes = 0;
  @$RClean = ();
  @$RUnscanned = ();
  @$RDirty = ();
  %$MessagesInfo = {};
  %$Headers = {};

  opendir QUEUE, $InQueueDir or return undef;

  # Read in all the modification dates of the qf files, and use them in date order
  while($file = readdir QUEUE) {
    next unless -f "$InQueueDir/$file";
    next unless $file =~ /$MTA::HFileRegexp/;
    $ModDate{$file} = (stat("$InQueueDir/$file"))[9]; # 9 = mtime
  }
  @SortedFiles = sort { $ModDate{$a} <=> $ModDate{$b} } keys %ModDate;

  # Keep going until end of dir or have reached every imposed limit.
  # This now processes the files oldest first to make for fairer queue cleanups.
  while(($file = shift @SortedFiles) &&
        $HitLimit1+$HitLimit2+$HitLimit3+$HitLimit4<1) {
    # must separate next two lines or $1 gets re-tainted by being part of
    # same expression as $file [mumble mumble grrr mumble mumble]
    #next unless -f "$InQueueDir/$file";
    next unless $file =~ /$MTA::HFileRegexp/;
    $id = $1;
    # Try to open qf file for reading and shared-lock it.
    # Skip it if either fails.
    Lock::openlock($QfFile, "$InQueueDir/$file", "r") or next;
    
    # See if we need to archive the message
    # Should really work out where "cp" is, but this path will work okay.
    system("/bin/cp $InQueueDir/" . MTA::HFileName($id) . " $InQueueDir/" .
           MTA::DFileName($id) . " $Config::ArchiveMailDir")
      if $Config::ArchiveMail;
    
    ($RHeaders, $MsgInfo) = ReadQf($QfFile); # Read the headers into an array of lines
    $MessagesInfo->{$id} = $MsgInfo;
    $Headers->{$id} = $RHeaders; # Store all the headers for SpamAssassin
    if (DefinitelyClean($RHeaders)) {
	push @$RClean, $id;
        $CleanMsgs++;
        $CleanBytes += -s "$InQueueDir/" . MTA::HFileName($id);
        $CleanBytes += -s "$InQueueDir/" . MTA::DFileName($id);
        $HitLimit1 = 1 if $CleanMsgs>=$Config::MaxCleanPerRun;
        $HitLimit2 = 1 if $CleanBytes>=$Config::MaxCleanBytes;
    } elsif (DontScan($MsgInfo)) {
        push @$RUnscanned, $id;
        $CleanMsgs++;
        $CleanBytes += -s "$InQueueDir/" . MTA::HFileName($id);
        $CleanBytes += -s "$InQueueDir/" . MTA::DFileName($id);
        $HitLimit1 = 1 if $CleanMsgs>=$Config::MaxCleanPerRun;
        $HitLimit2 = 1 if $CleanBytes>=$Config::MaxCleanBytes;
    } else {
        push @$RDirty, $id;
        $DirtyMsgs++;
        $DirtyBytes += -s "$InQueueDir/" . MTA::HFileName($id);
        $DirtyBytes += -s "$InQueueDir/" . MTA::DFileName($id);
        $HitLimit3 = 1 if $DirtyMsgs>=$Config::MaxDirtyPerRun;
        $HitLimit4 = 1 if $DirtyBytes>=$Config::MaxDirtyBytes;
        WriteHeaderFile($OutDir, $id, $RHeaders); # Write the file of headers
    }
    Lock::unlockclose($QfFile);
  }
  closedir QUEUE;

  Log::InfoLog("Forwarding $CleanMsgs clean messages, $CleanBytes bytes")
    if $CleanMsgs;
  Log::InfoLog("Scanning $DirtyMsgs messages, $DirtyBytes bytes")
    if $DirtyMsgs;
  $$RDirtyMsgs = $DirtyMsgs;
  $$RDirtyBytes = $DirtyBytes;
}

# Look up all the mail relays that sent us all these messages. Find which
# are in any of the RBL domains I have been given, and write appropriate
# %SpamInfo entries for them.
# For more info on the RBL domains, read www.orbs.org and www.mail-abuse.org.
# Passed in a ref to a hash of messageID-->messageinfo as input, and a ref
# to a hash of messageID-->spamtext as output.
# Now also use SpamAssassin to deduce likely-looking spam.
#Sendmail::FindSpammers(\%MessagesInfo, \%Headers, \%SpamInfo);
sub FindSpammers {
  my($MsgInfo, $Headers, $SpamText) = @_;
  my($mID, $Info, $Text, @RBLs, @IPwords);
  my($from, $to, $subject, $relay, $fromdomain);
  my($i, $j, $RBLEntry, $SkipChecks);

  while (($mID, $Info) = each %$MsgInfo) {
    ($from, $to, $subject, $relay) = split(/\0/, $Info);
    # Check to ensure that the relay isn't in the acceptable list of spam relays.
    $SkipChecks = 0;
    foreach $i (@Config::AcceptSpamFrom) {
      $j = $i; # Needed for bug in Perl 5.6.1
      $j =~ s#\.#\\.#g; # Escape all the dots in the IP addresses
      $SkipChecks = 1 if $relay =~ /^$j/;
    }
    next if $SkipChecks;

    # Check to ensure the sender address isn't in the white list
    $from = lc($from);
    $from =~ s/^<//; # Delete leading and
    $from =~ s/>$//; # trailing <>
    $fromdomain = $from;
    $fromdomain =~ s/^[^@]*@//; # Delete everything up to and including the @ sign
    next if $Config::SpamWhiteList{"$from"} ||
            $Config::SpamWhiteList{"$fromdomain"};

    # Implement *.domain.com wildcards in the spam white list
    my($key, $wildcard, $skipthis);
    $skipthis = 0;
    foreach $key (keys %Config::SpamWhiteList) {
      $wildcard = $key;
      next unless $wildcard =~ s/^\*\.//;
      #print STDERR "Testing $domain against $key\n";
      #print STDERR "Found a match with $key\n" if $domain =~ /\Q$wildcard\E$/i;
      if ($fromdomain =~ /\Q$wildcard\E$/i) {
        $skipthis = 1;
        last;
      }
    }
    next if $skipthis;

    # Reverse the $relay IP address
    @IPwords = split(/\./, $relay);
    $relay = join('.', reverse @IPwords);
    @RBLs = ();
    for ($i=0; $i<@Config::SpamNames; $i++) {
      # Look up $relay in each of the @Config::SpamDomains we have
      $RBLEntry = gethostbyname("$relay." . $Config::SpamDomains[$i]);
      if ($RBLEntry) {
        $RBLEntry = Socket::inet_ntoa($RBLEntry);
        push @RBLs, $Config::SpamNames[$i]
          if ($RBLEntry =~ /^127\.[01]\.0\.[1234567]$/);
      }
    }
    $SpamText->{$mID} = join(', ', @RBLs) if @RBLs;

    #
    # SpamAssassin: Run the message, with its headers, through SpamAssassin
    # and deduce likely looking spam. Then destroy the SpamAssassin data
    # structures and add to the end of $SpamText->{$mID}, separating with a
    # comma if there is already content there.
    #
    next unless $Config::SpamAssassin;

    my($spammy);
    $spammy = SpamAssassinChecks($Headers, $mID);
    if ($spammy) {
      $SpamText->{$mID} .= ", " if $SpamText->{$mID};
      $SpamText->{$mID} .= "SpamAssassin ($spammy hits)";
      ## Make a copy of the spam for later analysis and checking
      #system("/bin/cp $dfilename /export/2/var/spam/queue");
    }
  }
}

# Do SpamAssassin checks on the message number $mID. $Headers refers to
# a hash of arrays of message headers, saves me reading the qf file again.
sub SpamAssassinChecks {
  my($Headers, $mID) = @_;

  local(*DF);
  my($dfilename, $dfile, $dsize, @WholeMessage, $SAResult);

  # Also only do this if the message is reasonably small.
  $dfilename = $Config::InQueueDir . '/' . MTA::DFileName($mID);
  $dsize = (stat($dfilename))[7];
  return 0 if $dsize > $Config::MaxAssassinSize; # Max spam message size check

  # Construct the array of lines of the header and body of the message
  # JKF 30/1/2002 Don't chop off the line endings. Thanks to Andreas Piper
  #               for this.
  @WholeMessage = ();
  my($H) = $Headers->{$mID};
  return 0 unless $H;
  foreach $_ (@$H) {
    #chomp;
    push(@WholeMessage, $_);
  }
  push(@WholeMessage, "\n");
  open(DF, $dfilename) or return 0;
  while(<DF>) {
    #chomp;
    push(@WholeMessage, $_);
  }
  close(DF);

  #print STDERR "Whole message is this:\n";
  #print STDERR "----------------------\n";
  #print STDERR @WholeMessage . "\n";
  #print STDERR "---------------\n";
  #print STDERR "End of message.\n";

  # Now construct the SpamAssassin object
  my $spammail = Mail::SpamAssassin::NoMailAudit->new('data'=>\@WholeMessage);

  # Test it for spam-ness
  # Need to wrap this in a fork().
  #my $spamness = $SAspamtest->check ($spammail);
  # If it is spam, tag it as such
  #$SAResult = 1 if $spamness->is_spam();
  # Destroy the status result
  #$spamness->finish();
  #print STDERR "Doing test for $mID\n";
  $SAResult = SAForkAndTest($SAspamtest, $spammail);
  #print STDERR "Done test for $mID\n";
  #Log::WarnLog("Done SAForkAndTest\n");
  return $SAResult;
}

# Fork and test with SpamAssassin. This implements a timeout on the execution
# of the SpamAssassin checks, which occasionally take a *very* long time to
# terminate due to regular expression backtracking and other nasties.
sub SAForkAndTest {
  my($Test, $Mail) = @_;

  my $PipeReturn = 0;

  my $pid = fork();
  die "Can't fork: $!" unless defined($pid);

  if ($pid == 0) {
    # In the child
    POSIX::setsid();
    # Do the actual tests and work out the integer result
    my($spamness) = $Test->check($Mail);
    my($SAResult) = ($spamness->is_spam())?1:0;
    $SAResult = int($spamness->get_hits()) if $SAResult;
    # Destroy the status result -- should be unnecessary
    $spamness->finish();
    exit $SAResult;
  }

  eval {
    local $SIG{ALRM} = sub { die "Command Timed Out" };
    alarm $Config::SpamAssassinTimeout;
    wait;
    $PipeReturn = $?;
    alarm 0;
    $pid = 0;
  };
  alarm 0;


  # Note to self: I only close the KID in the parent, not in the child.

  # Catch failures other than the alarm
  Log::DieLog("SpamAssassin failed with real error: $@")
    if $@ and $@ !~ /Command Timed Out/;

  # In which case any failures must be the alarm
  #if ($@ or $pid>0) {
  if ($pid>0) {
    Log::WarnLog("SpamAssassin timed out and was killed");
    # Kill the running child process
    my($i);
    kill -15, $pid;
    # Wait for up to 10 seconds for it to die
    for $i (1 .. 10) {
      sleep 1;
      ($pid=0),last unless kill(0, $pid);
      kill -15, $pid;
    }
    # And if it didn't respond to 11 nice kills, we kill -9 it
    kill -9, $pid if $pid;
    wait; # 2.53
  }
  #Log::WarnLog("8 PID is $pid");

  # The return from the pipe is a measure of how spammy it was
  Log::DebugLog("SpamAssassin returned $PipeReturn");

  return $PipeReturn>>8;
}


# List all the message ids present in the sendmail queue.
# This is just used at startup to clean up from previous runs which were
# killed half way through doing something important.
# Passed in the queue directory.
# Returns a ref to a list of message ids which are in the queue directory.
sub ListWholeQueue {
  my($QDir) = @_;
  my($results, $file, $id);
  local(*QDIR);
  my $Qf = new FileHandle;

  @$results = ();

  opendir QDIR, $QDir or return $results;
  # Keep going until end of dir
  while($file = readdir QDIR) {
    next unless -f "$QDir/$file" && $file =~ /$MTA::HFileRegexp/;
    $id = $1;
    # Try to open qf file for reading and share-lock it.
    # Skip it if either fails.
    Lock::openlock ($Qf, "$QDir/$file", "r") or next;
    push @$results, $id;
    Lock::unlockclose($Qf);
  }
  closedir QDIR;

  Log::InfoLog("Startup: found " . scalar(@$results) . " messages waiting")
    if @$results;

  return $results;
}

# Read a envelope queue file (sendmail qf) and build
# an array of lines which together form all the mail headers.
sub ReadQf {
  # Call with our args and return its returns
  &MTA::ReadQf;
}

# Write a fake email header file for the given message id.
sub WriteHeaderFile {
  my($OutDir, $id, $RHeaders) = @_;
  my $Header = new FileHandle;

  Lock::openlock($Header, ">$OutDir/$id.header", "w")
    or Log::DieLog("Cannot create + lock headers file $OutDir / $id.header, %s", $!);
# if debugging not desired
#  print $Header @$RHeaders;
# if debugging desired
  foreach(@$RHeaders) {
    tr/\r/\n/; # Work around Outlook [Express] bug allowing viruses in headers
    #Log::DebugLog("Output header ($_)\n");
    print $Header $_;
  }
  print $Header "\n";
  Lock::unlockclose($Header);
}

# Delete messages which are present in both the incoming and outgoing queues
# from the outgoing queue. This will happen if I am interrupted half way
# through posting a message to the outgoing queue. Safer to start again.
# Passed refs to the lists of messages in both queues, and the location
# of the outgoing queue.
sub ClearOutQueue {
  my($InMsgs, $OutQueue) = @_;
  local(*OUTQ);
  my($file, $id, $counter);

  #print "Removing duplicates from outgoing queue:\n";
  $counter = 0;
  opendir OUTQ, $OutQueue or return undef;
  while($file = readdir OUTQ) {
    next unless -f "$OutQueue/$file";
    next unless $file =~ /$MTA::QueueFileRegexp/; # Is it a queue file of some sort?
    $id = $1;
    next unless grep /^$id$/, @$InMsgs;
    #print "$OutQueue/$file\n";
    unlink "$OutQueue/$file";
    $counter++;
  }
  closedir OUTQ;
  Log::InfoLog("Startup: removed $counter duplicated files from outgoing queue")
    if $counter;
}

# Add an "infected" header, according to the config file requirement
sub AddUnscannedHeader {
  my($headers) = @_;

  return MTA::AddHeader($headers, $Config::MailHeader,
                        $Config::UnscannedHeader)
    if $Config::MultipleHeaders =~ /add/i;
  return MTA::ReplaceHeader($headers, $Config::MailHeader,
                        $Config::UnscannedHeader)
    if $Config::MultipleHeaders =~ /replace/i;
  return MTA::AppendHeader($headers, $Config::MailHeader,
                        $Config::UnscannedHeader, ', ')
    if $Config::MultipleHeaders =~ /append/i;
}

# Add an "infected" header, according to the config file requirement
sub AddInfectedHeader {
  my($headers) = @_;

  return MTA::AddHeader($headers, $Config::MailHeader,
                        $Config::InfectedHeader)
    if $Config::MultipleHeaders =~ /add/i;
  return MTA::ReplaceHeader($headers, $Config::MailHeader,
                        $Config::InfectedHeader)
    if $Config::MultipleHeaders =~ /replace/i;
  return MTA::AppendHeader($headers, $Config::MailHeader,
                        $Config::InfectedHeader, ', ')
    if $Config::MultipleHeaders =~ /append/i;
}

# Add a "clean" header, according to the config file requirement
sub AddCleanHeader {
  my($headers) = @_;

  return MTA::AddHeader($headers, $Config::MailHeader,
                        $Config::CleanHeader)
    if $Config::MultipleHeaders =~ /add/i;
  return MTA::ReplaceHeader($headers, $Config::MailHeader,
                        $Config::CleanHeader)
    if $Config::MultipleHeaders =~ /replace/i;
  return MTA::AppendHeader($headers, $Config::MailHeader,
                        $Config::CleanHeader, ', ')
    if $Config::MultipleHeaders =~ /append/i;
}


# Deliver any old message ids.
sub DeliverIds {
  my($Ids, $entities, $MsgInfo, $IsSpam, $InQ, $OutQ, $Clean) = @_;
  #my($Ids, $entities, $InQ, $OutQ) = @_;
  my $Qf = new FileHandle;
  my $Tf = new FileHandle;
  my $DfOut = new FileHandle;
  my $DfIn = new FileHandle;
  my($id, $dfile, $hfile, $tfile, $newheaders, $newenvelope);
  my($info, $from, $fromdomain, $relay);

  foreach $id (@$Ids) {
    $dfile = MTA::DFileName($id);
    $hfile = MTA::HFileName($id);
    $tfile = MTA::TFileName($id);

    # Find the From: sender address, as we don't want to send anything if
    # the message came from a local domain if that option is switched on.
    $info = $MsgInfo->{$id};
    $from = (split(/\0/, $info))[0];
    $from =~ s/^<//; # Delete leading and
    $from =~ s/>$//; # trailing <>
    $relay = (split(/\0/, $info))[3]; # Get the SMTP client host
    $fromdomain = $from;
    $fromdomain =~ s/^[^@]*@//; # Delete everything up to and including the @

    if ($Clean ne 'clean') {
      # Just delete the message if it came from a local domain/address
      if (!$Config::DeliverFromLocal &&
          ($Config::LocalDomains{"$from"} ||
           $Config::LocalDomains{"$fromdomain"})) {
        Log::WarnLog("Virus originated from Internal LAN: User is $from, Host is $relay");
        unlink "$InQ/$hfile";
        unlink "$InQ/$dfile";
        next;
      }
    }

    # If there is no data structure for this message (i.e. the parsing failed)
    # then just delete it, don't attempt to deliver it.
    unless ($entities->{$id}) {
      # 2.53 Delete the original copies from the incoming sendmail queue
      Log::WarnLog("Deleting unparsable message $id from queue");
      unlink "$InQ/$hfile";
      unlink "$InQ/$dfile";
      next;
    }

    # Exim locks queue data file to indicate working on
    # that message... so we need to too.
    $Config::MTA eq "exim" and (Lock::openlock($DfIn,"$InQ/$dfile","r") or next);

    # Write the body first
    Lock::openlock($DfOut, ">$OutQ/$dfile", "w")
      or Log::DieLog("Cannot create + lock queue data file $OutQ/$dfile, %s", $!);
    $Config::MTA eq "exim" and print $DfOut "$dfile\n";
    $entities->{$id}->print_body($DfOut);
    Lock::unlockclose($DfOut);
    #undef $DfOut;

    # Construct all the new headers
    $newheaders = MTA::ConstructHeaders($entities->{$id}->stringify_header);
    $newheaders = AddCleanHeader($newheaders) if $Clean eq 'clean';
    $newheaders = AddInfectedHeader($newheaders) if $Clean eq 'dirty';
    $newheaders = AddUnscannedHeader($newheaders) if $Clean eq 'unscanned';
    if (defined($IsSpam->{$id})) {
      $newheaders = MTA::AddHeader($newheaders, $Config::SpamHeader,
                                   $IsSpam->{$id});
      $newheaders = MTA::PrependHeader($newheaders, "Subject:",
                                       $Config::SpamSubjectText, ' ')
        if $Config::SpamPrependSubject &&
           !MTA::TextStartsHeader($newheaders, "Subject:",
                                  $Config::SpamSubjectText);
      Log::InfoLog("Message $id is spam according to " . $IsSpam->{$id});
    }

    # Copy qf file from incoming queue except for H lines
    Lock::openlock($Tf, ">$OutQ/$tfile", "w")
      or Log::DieLog("Cannot create + lock temp queue file $OutQ/$tfile, %s", $!);
    # Shouldn't we lock this? -- nwp
    open($Qf, "$InQ/$hfile")
      or Log::DieLog("Cannot read envelope queue file $InQ/$hfile, %s", $!);
    # JKF 3/8/2001 Added $newheaders as 2nd parameters, fix Nick's bug
    $newenvelope = MTA::MergeEnvelopeParts($Qf, $newheaders);
    #Log::DebugLog("Headers3:\n$newheaders\n");
    close $Qf;
    print $Tf $newenvelope
      or Log::DieLog("Failed to write envelope and headers for message $id, %s", $!);
    Lock::unlockclose($Tf);
    #undef $Tf;
    rename "$OutQ/$tfile", "$OutQ/$hfile"
      or Log::DieLog("Cannot rename $OutQ/$tfile to $OutQ/$hfile, %s", $!);

    # Delete the original copies from the incoming sendmail queue
    unlink "$InQ/$hfile";
    unlink "$InQ/$dfile";

    # Close and unlock the (now unlinked) original data file
    $Config::MTA eq "exim" and Lock::unlockclose($DfIn);
  }

  Sendmail::TellAbout($Ids);
}

# Delete specified messages from the incoming queue. Used when not
# delivering messages to recipients at all if they had a virus in them.
sub DeleteIds {
  my($Ids, $InQ) = @_;
  my($id);

  foreach $id (@$Ids) {
    unlink "$InQ/" . MTA::DFileName($id);
    unlink "$InQ/" . MTA::HFileName($id);
  }
}

# Kick MTA into delivering each of the messages I'm passed.
sub TellAbout {
  my($Ids) = @_;
  my($id, $cmd, @ListCopy, @ThisBatch, $BatchSize);

  @ListCopy = @$Ids;
  Log::DebugLog("About to deliver " . scalar(@$Ids) . " messages") if @$Ids;

  # If we aren't kicking sendmail at all, then just return
  return if $Config::DeliveryMethod =~ /queue/i;

  if ($Config::DeliveryMethod =~ /batch/i) {
    # Deliver up to 20 messages per invocation of sendmail
    while (@ListCopy) {
      @ThisBatch = splice @ListCopy, $[, 20;
      MTA::KickMessage(@ThisBatch);
    }
  } else {
    # They must want each message delivered individually
    foreach $id (@$Ids) {
      MTA::KickMessage("$id");
    }
  }
}

# Deliver the clean messages into the outgoing sendmail.
# Pass in the Incoming queue dir, outgoing queue dir, outgoing sendmail
# binary, list of message ids which are clean.
# Need to add a hash of headers to add to each one.
sub MoveToOutgoingQueue {
  my($Ids, $entities, $IsSpam, $InQ, $OutQ) = @_;
  #my($Ids, $entities, $InQ, $OutQ) = @_;
  my($id, $dfile, $tfile, $hfile, $envelope, $headers);
  my $Df = new FileHandle;
  my $Qf = new FileHandle;
  my $Tf = new FileHandle;

  foreach $id (@$Ids) {

    $dfile = MTA::DFileName($id);
    $tfile = MTA::TFileName($id);
    $hfile = MTA::HFileName($id);

    # Exim locks queue data file to indicate working on
    # that message... so we need to too.
    $Config::MTA eq "exim" and (Lock::openlock($Df,"$InQ/$dfile","r") or next);

    # Link the queue data file
    # If the link fails for some reason (usually caused by sendmail calling
    # 2 messages the same thing in a very short time), then just skip this
    # message and move on to the next one. This one will get delivered when
    # the previous one with the same name has been delivered.
    unless (link "$InQ/$dfile", "$OutQ/$dfile") {
      # The link failed, so get the inode numbers of the two files
      my($ininode, $outinode);
      $ininode = (stat "$InQ/$dfile")[1];
      $outinode = (stat "$OutQ/$dfile")[1];
      # If the files are the same, then just quietly delete the incoming one
      $Config::MTA eq "exim" and Lock::unlockclose($Df);
      if ($ininode == $outinode) {
        unlink "$InQ/$hfile";
        unlink "$InQ/$dfile";
      } else {
        Log::WarnLog("Failed to link message body between queues ($OutQ/$dfile --> $InQ/$dfile)");
      }
      next;
    }

    # Add the signature to the outgoing queue file. If we die before deleting
    # the incoming queue file, there is a chance of double-signing a message,
    # but that is reckoned to be okay and very unlikely.
    # Only do this if we are delivering plain text messages. MIME messages will
    # have already been signed by the time they get here.
    if ($Config::SignCleanMessages && $entities eq 'unused') {
      local(*DOUT);
      open(DOUT, ">>$OutQ/$dfile") or Log::DieLog("Failed to create outgoing message body $OutQ/$dfile for clean text message");
      print DOUT $Config::InlineTextSig;
      close DOUT;
    }

    # Copy the queue envelope file, adding extra X- header to show we checked it.
    Lock::openlock($Tf, ">$OutQ/$tfile", "w")
      or Log::DieLog("Cannot create + lock clean tempfile $OutQ/$tfile, %s", $!);
    # Shouldn't we lock this? -- nwp
    unless (open($Qf, "$InQ/$hfile")) {
      Log::WarnLog("Cannot read qf file $InQ/$hfile, %s", $!);
      $Config::MTA eq "exim" and Lock::unlockclose($Df); # Remember to unlock it!
      next;
    }
    ($envelope,$headers) = MTA::SplitEnvelope(MTA::ReadEnvelope($Qf));
    close $Qf;
    if ($Config::SignUnscannedMessages && $entities eq 'unscanned') {
      $headers = AddUnscannedHeader($headers);
    } else {
      $headers = AddCleanHeader($headers);
    }
    # Delete any old content-length: headers if we are modifying the message body
    $headers = MTA::DeleteHeader($headers, "Content-length:")
      if $Config::SignCleanMessages && $entities eq 'unused';
    if (defined($IsSpam->{$id})) {
      $headers = MTA::AddHeader($headers, $Config::SpamHeader,
                                $IsSpam->{$id});
      $headers = MTA::PrependHeader($headers, "Subject:",
                                    $Config::SpamSubjectText, ' ')
        if $Config::SpamPrependSubject &&
           !MTA::TextStartsHeader($headers, "Subject:",
                                  $Config::SpamSubjectText);
      Log::InfoLog("Message $id is spam according to " . $IsSpam->{$id});
    }
    print $Tf &MTA::MergeEnvelope($envelope,$headers)
      or Log::DieLog("Failed to write headers for clean message $id, %s", $!);
    Lock::unlockclose($Tf);
    #undef $Tf;
    rename "$OutQ/$tfile", "$OutQ/$hfile"
      or Log::DieLog("Cannot rename clean $OutQ/$tfile to $OutQ/$hfile, %s", $!);

    # Delete the original copies from the incoming sendmail queue
    unlink "$InQ/$hfile";
    unlink "$InQ/$dfile";

    # Close and unlock the (now moved) data file
    $Config::MTA eq "exim" and Lock::unlockclose($Df);
  }

  Sendmail::TellAbout($Ids);
}

# Given a bunch of envelope information, work out if the message is
# to be scanned or not. Return 1 if the message is NOT to be scanned,
# return 0 otherwise.
sub DontScan {
  my($Envelope) = @_;
  my($from, $to, $subject, $relay, @addresses, $address, $addr, $domain);

  return 0 unless $Config::ScanningByDomain; # Do we want this feature?

  ($from, $to, $subject, $relay) = split(/\0/, $Envelope);

  $from = lc($from);
  $to   = lc($to);

  @addresses = split(/,\s+/, $to);
  push(@addresses, $from);

  # Search for every address in the %Config::DomainsToScan
  foreach $address (@addresses) {
    $addr = lc($address); # To handle bugs in perl
    $addr =~ s/^<//; # Delete leading and
    $addr =~ s/>$//; # trailing <>
    $addr = $1 if $addr =~ /\s([^\s]+)$/; # Extract the last word
    $domain = $addr;
    $domain =~ s/^[^@]*@//; # Delete all up to and including the @
    return 0 if $Config::DomainsToScan{$domain} ||
                $Config::DomainsToScan{$addr};
    my($key, $wildcard);
    foreach $key (keys %Config::DomainsToScan) {
      $wildcard = $key;
      next unless $wildcard =~ s/^\*\.//;
      #print STDERR "Testing $domain against $key\n";
      #print STDERR "Found a match with $key\n" if $domain =~ /\Q$wildcard\E$/i;
      return 0 if $domain =~ /\Q$wildcard\E$/i;
    }
  }
  return 1;
}

# Given a ref to a list of header lines, work out if the message is
# 100% definitely clean and non-corruptible.
# For now, this means messages that either have no "Content-type:"
# header at all, or ones that say "Content-type: text/plain"
# Return true or false.
sub DefinitelyClean {
  my($RHeaders) = @_;
  my(@Headers, $Header, $IsClean);

  # JKF 28/2/2002 Added new config variable to force it to scan
  # plain-text messages
  # Default will be yes.
  return 0 if $Config::ScanAllMessages;

  # JKF 23/4/2001 If the config option "Skip Messages If Already Scanned"
  # is set, and the MailHeader is already there, then mark this message
  # as definitely clean so it doesn't get scanned again.
  # Warning: The MailHeader could have been faked!
  @Headers = grep /^$Config::MailHeader:/i, @$RHeaders;
  return 1 if $Config::SkipAlreadyScanned && @Headers;

  @Headers = grep /^content-type:\s+/i, @$RHeaders;
  return 0 if $#Headers>0; # Catch multiple content-type headers!
  return 1 unless @Headers; # No content-type header at all
  $Header = $Headers[0];
  chomp $Header;
  $Header =~ s/^Content-Type:\s+//i;
  #print "Header is now \"$Header\"\n";
  return 1 if $Header =~ /^text$/i;
  return 1 if $Header =~ /^text\/plain$/i;
  return 1 if $Header =~ /^text\/plain;/i;
  return 0;
}

# Notify the sender of each message that their system probably has a virus.
sub WarnSenders {
  my($Ids, $MessagesInfo, $Reports, $InfectionTypes) = @_;
  my($id, $from, $to, $subject, $relay, $type, $type1);
  my($counter, $parts, $report, $filename);
  my($filename, $output, $result);
  local(*SENDMAIL, *TEXT);

  return unless @$Ids; # Do nothing if no infections
  #return unless $Config::TellSenders; # or we're not supposed to tell them

  my($date) = scalar localtime;
  foreach $id (@$Ids) {
    ($from, $to, $subject, $relay) = split(/\0/, $MessagesInfo->{$id});
    #$from =~ s/^\s*\<(.+)\>\s*$/$1/;
    #$to   =~ s/^\s*\<(.+)\>\s*$/$1/;
    $parts = $Reports->{$id};
    $type1 = $InfectionTypes->{$id};
    $type  = join("", values %$type1);
    $report = join("Report: ", values %$parts);

    # Don't send a message to "" or "<>"
    next if $from eq "" || $from eq "<>";

    open(SENDMAIL, "|$Config::Sendmail $SendmailOptions")
      or Log::WarnLog("Could not notify senders"), return;

    # Set the report filename dependent on what triggered MailScanner, be it
    # a virus, a filename trap, a Denial Of Service attack, or an parsing error.
    if ($type =~ /v/i) { $filename = $Config::SenderVirusReportText; }
    elsif ($type =~ /f/i) { $filename = $Config::SenderFilenameReportText; }
    elsif ($type =~ /e/i) { $filename = $Config::SenderErrorReportText; }
    else { $filename = $Config::SenderVirusReportText; }

    # Process the text file, substituting variable names as we go
    open(TEXT, $filename)
      or Log::WarnLog("Cannot open message file $filename");
    $output = "";
    while (<TEXT>) {
      chomp;
      s#"#\\"#g;
      # Boring untainting again...
      /(.*)/;
      $result = eval "\"$1\"";
      $output .= $result . "\n";
    }
    print SENDMAIL $output;
    close SENDMAIL;
    $counter++;
  }

  Log::InfoLog("Notified senders about $counter infections") if $counter;
}

# Notify the local postmaster that virus-infected email messages have been
# found.
# Pass in a ref to a list of infected message ids,
# and a ref to a hash of message ids --> "$from\0$to\0$subject",
# and a ref to a hash of lists of message headers.
sub WarnLocalPostmaster {
  my($Ids, $MessagesInfo, $Reports, $Headers) = @_;
  my($id, $info, $from, $to, $subject, $relay, $counter);
  my($parts, $report, $H);
  local(*SENDMAIL);

  return unless @$Ids; # Do nothing if no infections

  open(SENDMAIL, "|$Config::Sendmail $SendmailOptions")
    or Log::WarnLog("Could not notify local postmaster"), return;
  print SENDMAIL <<EONOTE1;
From: "MailScanner" <$Config::LocalPostmaster>
To: $Config::LocalPostmaster
Subject: Warning: E-mail viruses detected

The following e-mail messages were found to have viruses in them:
EONOTE1
  foreach $id (@$Ids) {
    #next if $id eq "";
    $info = $MessagesInfo->{$id};
    ($from, $to, $subject, $relay) = split(/\0/, $info);

    # Find every report for this message id
    $parts = $Reports->{$id};
    $report = join("   Report: ", values %$parts);

    print SENDMAIL <<EONOTE2;

   Sender: $from
Recipient: $to
  Subject: $subject
MessageID: $id
   Report: $report
EONOTE2
    $counter++;

    # New for 2.70. Optionally print full message headers in the postie msg.
    if ($Config::PostmasterFullHeaders) {
      $H = $Headers->{$id};
      next unless $H;
      print SENDMAIL "Full headers are:\n";
      print SENDMAIL " " . join(' ', @$H) . "\n";
    }
  }

  print SENDMAIL <<EONOTE3;
--
MailScanner
Email Virus Scanner
EONOTE3
  close SENDMAIL;

  Log::InfoLog("Notified $Config::LocalPostmaster about $counter infections") if $counter;
}

# Send a message completely contained in a MIME::Entity structure.
sub SendEntity {
  my($top) = @_;
  local(*SENDMAIL);

  # Send message
  # It is a good idea to have "hardwired" error text in seemingly
  # generic function, as below? -nwp
  open SENDMAIL, "|$Config::Sendmail $SendmailOptions"
    or Log::WarnLog("Could not send disinfected attachments"), return;
  $top->print(\*SENDMAIL);
  close SENDMAIL;
}

1;
