/******************************** LICENSE ********************************

 Copyright 2007 European Centre for Medium-Range Weather Forecasts (ECMWF)

 Licensed under the Apache License, Version 2.0 (the "License");
 you may not use this file except in compliance with the License.
 You may obtain a copy of the License at 

    http://www.apache.org/licenses/LICENSE-2.0

 Unless required by applicable law or agreed to in writing, software
 distributed under the License is distributed on an "AS IS" BASIS,
 WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 See the License for the specific language governing permissions and
 limitations under the License.

 ******************************** LICENSE ********************************/

/*! \file Data.h
    \brief Definition of the Abstract template class Data.
    
    Magics Team - ECMWF 2004
    
    Started: Fri 16-Jan-2004
    
    Changes:
    
*/

#ifndef Data_H
#define Data_H

#include "magics.h"
#include "MagException.h"

#include "MatrixHandler.h"
#include "PointsHandler.h"
#include "RasterData.h"
#include "CustomisedPoint.h"
#include "RasterData.h"
#include "Transformation.h"
#include "Layer.h"
#include "DateTime.h"
#include "MagicsEvent.h"
#include "ThinningMethod.h"
#include "DataStats.h"
#include "BinningObject.h"

namespace magics {

class LegendVisitor;
class TextVisitor;
class MetaDataVisitor; 
class MagnifierVisitor; 
class LayerNode;
class AnimationRules;
class AnimationStep;


template <class P>
struct MatchCriteria 
{
    MatchCriteria() {}
    virtual ~MatchCriteria() {}
    virtual bool verify(const P&, const string&, const string&) { return false; }
};

class DataInMemory : public MagicsException
{
public:
	 DataInMemory():
		MagicsException("Data in Memory..." ) {}
};     


template <class P>
class Data: public MetviewIcon{

public:
	Data(): name_("no_name"), binning_(0) {}
	virtual ~Data() { if ( binning_ ) delete binning_; }
    //! Method to access the data as a matrix Used by pcont action routine
    virtual MatrixHandler<P>& matrix() { throw MethodNotYetImplemented("Data<P>::matrix"); }
    virtual MatrixHandler<P>& matrix(const Transformation&) { throw MethodNotYetImplemented("Data<P>::matrix"); }
    

    void binning(BinningObject* binning) { binning_ = binning; }

  

    //! Method to access the data as a list of points : Used by psymb.
    virtual PointsHandler<P>& points() { throw MethodNotYetImplemented("Data::points"); }
    virtual PointsHandler<P>& points(const Transformation&) { throw MethodNotYetImplemented("Data::points"); }
    virtual void release() {}
    
    virtual void getReady(const Transformation&) {}
    virtual void visit(Transformation&)          {}
    
    //! Methods to access the data as a file used by pplot action routine
    virtual string path() { throw DataInMemory(); }
    //! Methods to access the data as a RasterData used by pimage action routine.
    virtual RasterData<P>& raster(const Transformation&) { throw MethodNotYetImplemented("Data<P>::raster"); }
    
    virtual void getInfo(const std::set<string>&, multimap<string, string>&) 
    { throw MethodNotYetImplemented("Data::getInfo"); }
    
    virtual void points(vector<P>&)
    { throw MethodNotYetImplemented("Data::points"); }
    virtual void customisedPoints(const Transformation&, const std::set<string>&, CustomisedPointsList& ) 
        { throw MethodNotYetImplemented("Data::customisedPoints-Transformation"); }
    virtual void customisedPoints(const std::set<string>&, CustomisedPointsList& )
           { throw MethodNotYetImplemented("Data::customisedPoints-Transformation"); }
    virtual void customisedPoints(const AutomaticThinningMethod&, const Transformation& transformation, const std::set<string>& need , CustomisedPointsList& out)
           { customisedPoints(transformation, need, out); }
    virtual void customisedPoints(const BasicThinningMethod&, const Transformation& transformation, const std::set<string>& need , CustomisedPointsList& out) 
             { customisedPoints(transformation, need, out); }
    virtual void visit(TextVisitor&) {}
    virtual void visit(LegendVisitor&) {}
    virtual void visit(AnimationRules&) {}
    virtual void visit(AnimationStep&) {}
    virtual void visit(MetaDataVisitor&) {} 
    virtual void visit(Layer& layer) { 
    	MetviewIcon::visit(layer);
    	layer.name(name());
    	layer.valid(from(), to());
    }
    virtual void visit(MetaDataCollector& collector)  {MetviewIcon::visit(collector);}  
    virtual void visit(ValuesCollector&) {}
    virtual void visit(MagnifierCollector& magnifier) {
    	PointsHandler<P>& list = this->points(); 
    	const Transformation& transformation = magnifier.transformation();
    	list.setToFirst();
    	while (list.more()) {   		
    		magnifier.push_back(transformation(list.current()));
    		list.advance();
    	}  
    }
    virtual void initInfo() {MetviewIcon::initInfo();}
    string legend() { return legend_; }
    
    // Information needed fron layer management!
    virtual string layerId()  {  return  ( layerId_.empty() ) ?  iconName_ + "/ " + iconClass_  : layerId_  ; }
    virtual string name()  { return  ( iconName_.empty() ) ? name_ : iconName_; }
    virtual const DateTime& from()  { return from_; }
    virtual const DateTime& to()  { return to_; }

protected:
     //! Method to print string about this class on to a stream of type ostream (virtual).
    virtual void print(ostream& out) const { out << "Data<P>"; }
    virtual void computeStats();
    
     VectorOfPointers<vector<PointsHandler<P>* > > pointsHandlers_; 
     // The objects are put in a list, they will be automaticcaly destructed 
     VectorOfPointers<vector<MatrixHandler<P>* > > matrixHandlers_; 

	string legend_;
	
	string name_;
	string layerId_;
	DateTime from_;
	DateTime to_;
	map<string,vector<double> > stats_;

	BinningObject* binning_;

private:
    //! Copy constructor - No copy allowed
	Data(const Data&);
    //! Overloaded << operator to copy - No copy allowed
	Data& operator=(const Data&);
    
// -- Friends
    //! Overloaded << operator to call print().
	friend ostream& operator<<(ostream& s,const Data<P>& p)
		{ p.print(s); return s; }
};

template <class P>
void Data<P>::computeStats()
{
	map<string,vector<double> >::iterator itX, itY, itV;	
	itX=stats_.find("x");
	itY=stats_.find("y");
	itV=stats_.find("value");

	//X and Y for scatterplots
	if(info("statsType") == "scatter" && 
	   itX != stats_.end() && itY != stats_.end())
	{
		DataStats stX(itX->second);
		DataStats stY(itY->second);	

		std::map<string,DataStats*> st;
		st["_x"]=&stX;
		st["_y"]=&stY;
	  	
		for(std::map<string,DataStats*>::iterator it=st.begin(); it != st.end(); it++)
		{
		 	setInfo("stats::points" + it->first,tostring(it->second->num()));
			setInfo("stats::min" + it->first, tostring(it->second->min()));	
			setInfo("stats::max" + it->first, tostring(it->second->max()));
			setInfo("stats::avg" + it->first, tostring(it->second->mean()));
			if(it->second->hasStDev()) setInfo("stats::stdev" + it->first,tostring(it->second->stDev()));
				else setInfo("stats::stdev" + it->first,"-");
		}
		
		setInfo("stats::correlation",tostring(DataStats::correlation(itX->second,itY->second,stX,stY)));
	}
	
	else if(info("statsType") == "vector" && 
	        itX != stats_.end() && itY != stats_.end())
	{		
		DataStats stX(itX->second);
		DataStats stY(itY->second);
	  	
		std::map<string,DataStats*> st;
		st["_x"]=&stX;
		st["_y"]=&stY;
	  	
		for(std::map<string,DataStats*>::iterator it=st.begin(); it != st.end(); it++)
		{
		 	setInfo("stats::points" + it->first,tostring(it->second->num()));
			setInfo("stats::min" + it->first, tostring(it->second->min()));	
			setInfo("stats::max" + it->first, tostring(it->second->max()));
			setInfo("stats::avg" + it->first, tostring(it->second->mean()));
			setInfo("stats::stdev" + it->first,"-");
			setInfo("stats::skewness" + it->first,"-");		
			setInfo("stats::kurtosis" + it->first,"-");	
	
			if(it->second->hasStDev())
			{
				setInfo("stats::stdev" + it->first,tostring(it->second->stDev()));
				if(it->second->hasThirdMoment())
				{
					setInfo("stats::skewness" + it->first,tostring(it->second->skewness()));
					setInfo("stats::kurtosis" + it->first,tostring(it->second->kurtosis()));
				}
			}	
		}
	}

        //Value		
	if(itV != stats_.end())
	{
		if(itV->second.size() == 0)	
		{
			setInfo("stats::min","");
    			setInfo("stats::max","");
			setInfo("stats::avg","");
    			setInfo("stats::points","");
			return;
		}	

		DataStats st(itV->second);
		setInfo("stats::points",tostring(st.num()));
		setInfo("stats::min", tostring(st.min()));
    		setInfo("stats::max", tostring(st.max()));
		setInfo("stats::avg", tostring(st.mean()));
		setInfo("stats::stdev","-");
		setInfo("stats::skewness","-");		
		setInfo("stats::kurtosis","-");

		if(st.hasStDev())
		{
			setInfo("stats::stdev",tostring(st.stDev()));
			if(st.hasThirdMoment())
			{
				setInfo("stats::skewness",tostring(st.skewness()));
				setInfo("stats::kurtosis",tostring(st.kurtosis()));
			}
		}
	}

	stats_.clear();	
}


template <class P>
class DataLoop: public MetviewIcon
{ 
public:
	DataLoop() {}
	virtual ~DataLoop() {}
	virtual void set(const map<string, string>&) {}
	virtual void set(const XmlNode&) {}
	virtual void set(LayerNode&) {}
	
	virtual void setToFirst() {}
	virtual Data<P>* current() = 0;
	virtual bool         hasMore() = 0;
	virtual void         next() = 0;
	virtual void add(Data<P>*) {}
	virtual string layerId()  {  return   iconName_ + "/ " + iconClass_   ; }
	virtual string name()  { return  iconName_; }
	virtual void visit(Transformation&) {}
	void visit(Layer& layer) { MetviewIcon::visit(layer); }
};
} // namespace magics


#endif
