/******************************** LICENSE ********************************

 Copyright 2007 European Centre for Medium-Range Weather Forecasts (ECMWF)

 Licensed under the Apache License, Version 2.0 (the "License");
 you may not use this file except in compliance with the License.
 You may obtain a copy of the License at 

    http://www.apache.org/licenses/LICENSE-2.0

 Unless required by applicable law or agreed to in writing, software
 distributed under the License is distributed on an "AS IS" BASIS,
 WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 See the License for the specific language governing permissions and
 limitations under the License.

 ******************************** LICENSE ********************************/

/*! \file NetcdfGeopointsInterpretor.cc
    \brief Implementation of the Template class NetcdfGeopointsInterpretor.
    
    Magics Team - ECMWF 2004
    
    Started: Tue 17-Feb-2004
    
    Changes:
    
*/

#include "NetcdfGeopointsInterpretor.h"
#include "Factory.h"
#include "Netcdf.h"
#include <limits>

using namespace magics;

NetcdfGeopointsInterpretor::NetcdfGeopointsInterpretor() 
{}


NetcdfGeopointsInterpretor::~NetcdfGeopointsInterpretor() 
{}

void NetcdfGeopointsInterpretor::interpretAsPoints(PointsList<GeoPoint>& list) 
{
	Netcdf netcdf(path_);
	
	// get the data ...
	try
	{
		vector<double> latitudes;
		vector<double> longitudes;
		vector<double> values;
		map<string, string> first, last;
		setDimensions(dimension_, first, last);
		
		netcdf.get(field_, values, first, last);
		netcdf.get(longitude_, longitudes, first, last);
		netcdf.get(latitude_, latitudes, first, last);
		
		vector<double>::iterator lat = latitudes.begin();
		vector<double>::iterator lon = longitudes.begin();
		vector<double>::const_iterator val = values.begin();
		
		//If the lat-lon units is specified as "radians" convert lat-lon 
		//to degrees. By default the units are sipposed to be "degrees"
		const char *units;
		if ( magCompare(netcdf.getVariableAttribute(latitude_,"units",units), "radians") )
		{			
			while ( lat!= latitudes.end()) {
			  *lat=DEG(*lat);
			  lat++;
			}  			
		}
		
		if ( magCompare(netcdf.getVariableAttribute(longitude_,"units",units), "radians") )
		{			
			while ( lon!= longitudes.end()) {
			  *lon=DEG(*lon);
			  lon++;
			}  			
		}		

		lat = latitudes.begin();		
		lon = longitudes.begin();
		while ( lat != latitudes.end() && lon != longitudes.end() && val != values.end() ) {
			list.push_back(GeoPoint(*lon,*lat,*val));		
			lon++;
			lat++;
			val++;
		}
		
	}
    
	catch (MagicsException& e)
	{
		Log::error() << e << "\n";
	}
}

/*
 Class information are given to the output-stream.
*/		
void NetcdfGeopointsInterpretor::print(ostream& out)  const
{
	out << "NetcdfGeopointsInterpretor[";
	NetcdfInterpretor::print(out);
	out << "]";
}

static SimpleObjectMaker<NetcdfGeopointsInterpretor, NetcdfInterpretor> netcdf_geopoints_interpretor("geopoints");
