/*
 * @(#)MarkedInstructionUTest.java
 *
 * Copyright (C) 2003 Matt Albrecht
 * groboclown@users.sourceforge.net
 * http://groboutils.sourceforge.net
 *
 *  Permission is hereby granted, free of charge, to any person obtaining a
 *  copy of this software and associated documentation files (the "Software"),
 *  to deal in the Software without restriction, including without limitation
 *  the rights to use, copy, modify, merge, publish, distribute, sublicense,
 *  and/or sell copies of the Software, and to permit persons to whom the 
 *  Software is furnished to do so, subject to the following conditions:
 *
 *  The above copyright notice and this permission notice shall be included in 
 *  all copies or substantial portions of the Software. 
 *
 *  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR 
 *  IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, 
 *  FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL 
 *  THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER 
 *  LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING 
 *  FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER 
 *  DEALINGS IN THE SOFTWARE.
 */

package net.sourceforge.groboutils.codecoverage.v2.compiler;

import junit.framework.Test;
import junit.framework.TestCase;
import junit.framework.TestSuite;
import net.sourceforge.groboutils.autodoc.v1.AutoDoc;
import net.sourceforge.groboutils.codecoverage.v2.BCELCreatorUtil;

import org.apache.bcel.generic.InstructionHandle;
import org.apache.bcel.generic.InstructionList;


/**
 * Tests the MarkedInstruction class.
 *
 * @author    Matt Albrecht <a href="mailto:groboclown@users.sourceforge.net">groboclown@users.sourceforge.net</a>
 * @version   $Date: 2004/04/15 05:48:28 $
 * @since     January 13, 2003
 */
public class MarkedInstructionUTest extends TestCase
{
    //-------------------------------------------------------------------------
    // Standard JUnit Class-specific declarations
    
    private static final Class THIS_CLASS = MarkedInstructionUTest.class;
    private static final AutoDoc DOC = new AutoDoc( THIS_CLASS );
    
    public MarkedInstructionUTest( String name )
    {
        super( name );
    }


    //-------------------------------------------------------------------------
    // Tests
    
    private static final int MARK_INSTRUCTION_COUNT = 5;
    
    
    public void testConstructor1()
    {
        try
        {
            new MarkedInstruction( (short)0, 0, 0, null );
            fail( "Did not throw IllegalArgumentException." );
        }
        catch (IllegalArgumentException e)
        {
            // test failure?
        }
    }
    
    
    public void testGetInstruction1()
    {
        InstructionHandle ih = createInstructionHandle();
        MarkedInstruction mi = new MarkedInstruction(
            (short)0, 0, 0, ih );
        assertEquals(
            "Did not return expected instruction.",
            ih.getInstruction(),
            mi.getHandle().getInstruction() );
    }
    
    
    public void testAddMark1()
    {
        InstructionHandle ih = createInstructionHandle();
        MarkedInstruction mi = new MarkedInstruction(
            (short)0, 0, 0, ih );
        mi.addMark( (short)1, Short.MIN_VALUE );
    }
    
    
    public void testAddMark2()
    {
        InstructionHandle ih = createInstructionHandle();
        MarkedInstruction mi = new MarkedInstruction(
            (short)0, 0, 0, ih );
        mi.addMark( Short.MIN_VALUE, (short)1 );
    }
    
    
    public void testAddMark3()
    {
        InstructionHandle ih = createInstructionHandle();
        MarkedInstruction mi = new MarkedInstruction(
            (short)0, 0, 0, ih );
        mi.addMark( (short)1, Short.MAX_VALUE );
    }
    
    
    public void testAddMark4()
    {
        InstructionHandle ih = createInstructionHandle();
        MarkedInstruction mi = new MarkedInstruction(
            (short)0, 0, 0, ih );
        mi.addMark( Short.MAX_VALUE, (short)1 );
    }
    
    
    public void testGetMarkedList1()
    {
        InstructionHandle ih = createInstructionHandle();
        MarkedInstruction mi = new MarkedInstruction(
            (short)0, 0, 0, ih );
        InstructionList il = mi.getMarkedList();
        assertNull(
            "Did not return a null marked list.",
            il );
    }
    
    
    public void testGetMarkedList2()
    {
        InstructionHandle ih = createInstructionHandle();
        MarkedInstruction mi = new MarkedInstruction(
            (short)0, 0, 0, ih );
        mi.addMark( (short)1, (short)1 );
        InstructionList il = mi.getMarkedList();
        assertEquals(
            "Did not return a marked list of the expected size.",
            MARK_INSTRUCTION_COUNT,
            il.size() );
    }
    
    
    public void testGetMarkedList3()
    {
        InstructionHandle ih = createInstructionHandle();
        MarkedInstruction mi = new MarkedInstruction(
            (short)0, 0, 0, ih );
        mi.addMark( (short)1, (short)1 );
        mi.addMark( (short)1, (short)1 );
        mi.addMark( (short)1, (short)1 );
        InstructionList il = mi.getMarkedList();
        assertEquals(
            "Did not return a marked list of the expected size.",
            MARK_INSTRUCTION_COUNT*3,
            il.size() );
    }
    
    
    public void testGetMarkedList_state1()
    {
        InstructionHandle ih = createInstructionHandle();
        MarkedInstruction mi = new MarkedInstruction(
            (short)0, 0, 0, ih );
        mi.getMarkedList();
        
        // repeat again without any mark changes
        InstructionList il = mi.getMarkedList();
        assertNull(
            "Did not return a null marked list.",
            il );
    }
    
    
    public void testGetMarkedList_state2()
    {
        InstructionHandle ih = createInstructionHandle();
        MarkedInstruction mi = new MarkedInstruction(
            (short)0, 0, 0, ih );
        mi.addMark( (short)1, (short)1 );
        mi.addMark( (short)1, (short)1 );
        mi.getMarkedList();
        
        // repeat again without any mark changes
        InstructionList il = mi.getMarkedList();
        assertNull(
            "Did not return a null marked list.",
            il );
    }
    
    
    public void testGetMarkedList_state3()
    {
        InstructionHandle ih = createInstructionHandle();
        MarkedInstruction mi = new MarkedInstruction(
            (short)0, 0, 0, ih );
        mi.getMarkedList();
        
        mi.addMark( (short)1, (short)1 );
        mi.addMark( (short)1, (short)1 );
        InstructionList il = mi.getMarkedList();
        assertEquals(
            "Did not return a marked list of the expected size.",
            MARK_INSTRUCTION_COUNT*2,
            il.size() );
    }
    
    
    public void testGetMarkedList_state4()
    {
        InstructionHandle ih = createInstructionHandle();
        MarkedInstruction mi = new MarkedInstruction(
            (short)0, 0, 0, ih );
        mi.addMark( (short)0, (short)0 );
        mi.addMark( (short)0, (short)1 );
        mi.addMark( (short)0, (short)2 );
        mi.getMarkedList();
        
        mi.addMark( (short)1, (short)0 );
        mi.addMark( (short)1, (short)1 );
        InstructionList il = mi.getMarkedList();
        assertEquals(
            "Did not return a marked list of the expected size.",
            MARK_INSTRUCTION_COUNT*2,
            il.size() );
    }
    
    
    public void testGetHandle1()
    {
        InstructionHandle ih = createInstructionHandle();
        MarkedInstruction mi = new MarkedInstruction(
            (short)0, 0, 0, ih );
        assertSame(
            "Did not return same handle.",
            ih,
            mi.getHandle() );
    }
    
    
    //-------------------------------------------------------------------------
    // Helpers
    
    
    protected InstructionHandle createInstructionHandle()
    {
        return BCELCreatorUtil.createInstructionHandle();
    }
    
    
    
    //-------------------------------------------------------------------------
    // Standard JUnit declarations
    
    
    public static Test suite()
    {
        TestSuite suite = new TestSuite( THIS_CLASS );
        
        return suite;
    }
    
    public static void main( String[] args )
    {
        String[] name = { THIS_CLASS.getName() };
        
        // junit.textui.TestRunner.main( name );
        // junit.swingui.TestRunner.main( name );
        
        junit.textui.TestRunner.main( name );
    }
    
    
    /**
     * 
     * @exception Exception thrown under any exceptional condition.
     */
    protected void setUp() throws Exception
    {
        super.setUp();
        
        // set ourself up
    }
    
    
    /**
     * 
     * @exception Exception thrown under any exceptional condition.
     */
    protected void tearDown() throws Exception
    {
        // tear ourself down
        
        
        super.tearDown();
    }
}

