/*******************************************************************************************************************************************
 cmetamoduleimporter.c
*******************************************************************************************************************************************/

#include "cmetamoduleimporter.h"

//------------------------------------------------------------------------------------------------------------------------------------------
// metaclass code resolution
//------------------------------------------------------------------------------------------------------------------------------------------
RESOLVE_CAPSULE_METACLASS (CMetaModuleImporter);

//------------------------------------------------------------------------------------------------------------------------------------------
// constructor
//------------------------------------------------------------------------------------------------------------------------------------------
CMetaModuleImporter::CMetaModuleImporter (const CString &inLibPath, const CMetaClass *inRequestedAPI)
		    :CClass		 (),
		     m_LibName		 (inLibPath),
		     m_LibHandle	 (NULL),
		     m_LibMetaClass	 (NULL),
		     m_LibMetaModules	 ()
{
	if (inRequestedAPI == NULL) 
		throw new CException (m_LibName + " : bad parameter, requested API can\'t be NULL.", __exception(NULLPARAMETER));
	if (!CMetaClass::MetaClassIs (__metaclass(CMetaModule), inRequestedAPI))
		throw new CException (m_LibName + " : bad parameter, requested API must be derived from CMetaModule.", 
				      __exception(BADPARAMETER));
	if ((m_LibHandle = ::dlopen (m_LibName.Get(), RTLD_NOW)) != NULL)
	{
		void *inGetModuleMetaClass = ::dlsym (m_LibHandle, "_Z18GetModuleMetaClassv");
		if (inGetModuleMetaClass != NULL)
		{
			m_LibMetaClass = (CMetaClass *) ((const CMetaClass * (*) ()) (inGetModuleMetaClass)) ();
			if (m_LibMetaClass == NULL)
			{	
				::dlclose (m_LibHandle);
				m_LibHandle = NULL;
				throw new CException (m_LibName + " : the module\'s exported metaclass can\'t be NULL.", 
						      __exception(NULLEXPORT));
			}
			if (m_LibMetaClass -> MetaClassType != METACLASS_DYNAMIC || m_LibMetaClass -> ClassInstanciate == NULL ||
			   !CMetaClass::MetaClassIs (inRequestedAPI, m_LibMetaClass))
			{
				CString inClassName = m_LibMetaClass -> ClassName;
				::dlclose (m_LibHandle);
				m_LibHandle = NULL;
				m_LibMetaClass = NULL;
				throw new CException (m_LibName + " : " + inClassName + " can\'t be exported as dynamic derived from " + 
						      inRequestedAPI -> ClassName + ".", __exception(BADEXPORT));
			}
		}
		else
		{
			::dlclose (m_LibHandle);
			m_LibHandle = NULL;
			throw new CException (m_LibName + " : no export point defined in the module.", __exception(NOEXPORT));
		}	
	}
	else
	{
		throw new CException (CString(::dlerror()));
	}
}

//------------------------------------------------------------------------------------------------------------------------------------------
// destructor
//------------------------------------------------------------------------------------------------------------------------------------------
CMetaModuleImporter::~CMetaModuleImporter ()
{
	for (size_t i=m_LibMetaModules.GetLength(); i>0; i--) delete *m_LibMetaModules[i-1];
	if (m_LibHandle != NULL) ::dlclose (m_LibHandle);
}

//------------------------------------------------------------------------------------------------------------------------------------------
// lib name access
//------------------------------------------------------------------------------------------------------------------------------------------
CString CMetaModuleImporter::GetLibName () const
{
	return m_LibName;
}

//------------------------------------------------------------------------------------------------------------------------------------------
// lib handle
//------------------------------------------------------------------------------------------------------------------------------------------
void * CMetaModuleImporter::GetLibHandle () const
{
	return m_LibHandle;
}

//------------------------------------------------------------------------------------------------------------------------------------------
// metaclass access
//------------------------------------------------------------------------------------------------------------------------------------------
const CMetaClass * CMetaModuleImporter::GetModuleMetaClass () const
{
	return m_LibMetaClass;
}

//------------------------------------------------------------------------------------------------------------------------------------------
// instanciate the metamodule module
//------------------------------------------------------------------------------------------------------------------------------------------
CMetaModule * CMetaModuleImporter::InstanciateModule ()
{
	if (m_LibMetaClass == NULL || m_LibMetaClass -> ClassInstanciate == NULL) return NULL;
	CMetaModule *newMetaModule = static_cast <CMetaModule *> (m_LibMetaClass -> ClassInstanciate ());
	if (newMetaModule != NULL) m_LibMetaModules += newMetaModule;
	return newMetaModule;
}

//------------------------------------------------------------------------------------------------------------------------------------------
// delete the module
//------------------------------------------------------------------------------------------------------------------------------------------
void CMetaModuleImporter::DeleteModule (CMetaModule *&ioMetaModule)
{
	if (ioMetaModule != NULL)
	{
		m_LibMetaModules -= ioMetaModule;
		delete ioMetaModule;
		ioMetaModule = NULL;
	}
}

static CStrings LibNames;

static int PathWalker (const char *inFileName, const struct stat *inStat, int, struct FTW *)
{
	if (S_ISREG(inStat->st_mode) && CString(inFileName).Find (CString(".so")))
		LibNames += CString(inFileName);
	return 0;
}

//------------------------------------------------------------------------------------------------------------------------------------------
// lib names of specified path
//------------------------------------------------------------------------------------------------------------------------------------------
CStrings CMetaModuleImporter::GetLibNames (const CString &inPath)
{
	LibNames.Delete (0, LibNames.GetLength());
	::nftw (inPath.Get(), ::PathWalker, 10, FTW_DEPTH|FTW_PHYS);
	return LibNames;
}
