/*******************************************************************************************************************************************
 cchunk.c
*******************************************************************************************************************************************/

#include "cchunk.h"

//------------------------------------------------------------------------------------------------------------------------------------------
// metaclass code resolution
//------------------------------------------------------------------------------------------------------------------------------------------
RESOLVE_CAPSULE_METACLASS (CChunk);

//------------------------------------------------------------------------------------------------------------------------------------------
// chunk reallocation
//------------------------------------------------------------------------------------------------------------------------------------------
bool CChunk::ReallocateChunk (CChunk &ioChunk, const SInt32 inDeltaSize)
{
	if (!ioChunk.m_HandleData) return false;
	if ((ioChunk.m_Size+inDeltaSize) < 0) return false;
	void *newChunk = new char [ioChunk.m_Size+inDeltaSize];
	if (ioChunk.m_Chunk != NULL)
	{
		::memcpy (newChunk, ioChunk.m_Chunk, inDeltaSize > 0L ? ioChunk.m_Size : ioChunk.m_Size+inDeltaSize);
		delete [] reinterpret_cast <char *> (ioChunk.m_Chunk);
	}
	ioChunk.m_Chunk = newChunk;
	ioChunk.m_Size += inDeltaSize;
	return true;
}

//------------------------------------------------------------------------------------------------------------------------------------------
// constructor
//------------------------------------------------------------------------------------------------------------------------------------------
CChunk::CChunk       ()
       :m_Chunk      (NULL),
        m_Size       (0L),
	m_Offset     (0L),
	m_iData	     (0L),
	m_DataList   (),
	m_HandleData (true)
{ }

//------------------------------------------------------------------------------------------------------------------------------------------
// constructor
//------------------------------------------------------------------------------------------------------------------------------------------
CChunk::CChunk       (void *inData, const UInt32 inSize)
       :m_Chunk      (inData),
        m_Size       (inSize),
        m_Offset     (0L),
        m_iData      (0L),
        m_DataList   (),
	m_HandleData (true)
{ }

//------------------------------------------------------------------------------------------------------------------------------------------
// constructor
//------------------------------------------------------------------------------------------------------------------------------------------
CChunk::CChunk       (void *inData, const UInt32 inSize, const TDataList &inDataList, const bool inHandleData)
       :m_Chunk	     (inData),
	m_Size	     (inSize),
	m_Offset     (0L),
	m_iData	     (0L),
	m_DataList   (inDataList),
	m_HandleData (inHandleData)
{ }

//------------------------------------------------------------------------------------------------------------------------------------------
// copy constructor
//------------------------------------------------------------------------------------------------------------------------------------------
CChunk::CChunk       (const CChunk &inChunk)
       :m_Chunk      (NULL),
        m_Size       (0L),
	m_Offset     (0L),
	m_iData	     (0L),
	m_DataList   (),
	m_HandleData (true)
{ 
	if (inChunk.m_Chunk != NULL)
	{
		m_Size  = inChunk.m_Size;
		m_Chunk = new char [m_Size];
		::memcpy (m_Chunk, inChunk.m_Chunk, m_Size);
		m_Offset = inChunk.m_Size;
		m_DataList = inChunk.m_DataList;
		m_iData = inChunk.m_iData;
	}
}

//------------------------------------------------------------------------------------------------------------------------------------------
// destructor
//------------------------------------------------------------------------------------------------------------------------------------------
CChunk::~CChunk ()
{ 
	if (m_Chunk != NULL && m_HandleData) delete [] reinterpret_cast <char *> (m_Chunk);
}

//------------------------------------------------------------------------------------------------------------------------------------------
// chunk data access
//------------------------------------------------------------------------------------------------------------------------------------------
void * CChunk::GetChunk () const
{
	return m_Chunk;
}

//------------------------------------------------------------------------------------------------------------------------------------------
// data size
//------------------------------------------------------------------------------------------------------------------------------------------
UInt32 CChunk::GetSize () const
{
	return m_Size;
}

//------------------------------------------------------------------------------------------------------------------------------------------
// data type analyse
//------------------------------------------------------------------------------------------------------------------------------------------
TData CChunk::NextDataIs () const
{
	return m_iData < m_DataList.GetLength() ? *m_DataList[m_iData] : static_cast <TData> (-1);
}

//------------------------------------------------------------------------------------------------------------------------------------------
// data list access
//------------------------------------------------------------------------------------------------------------------------------------------
TDataList CChunk::GetDataList () const
{
	return m_DataList;
}

//------------------------------------------------------------------------------------------------------------------------------------------
// chunk write
//------------------------------------------------------------------------------------------------------------------------------------------
#define WRITEFUNC(typename,typedesc) void CChunk::Write##typename (const typename inValue)					\
{																\
	if (CChunk::ReallocateChunk (*this, sizeof(typename)))									\
	{															\
		*reinterpret_cast <typename *> (reinterpret_cast <char *> (m_Chunk) + m_Size - sizeof(typename)) = inValue;	\
		m_DataList += typedesc;												\
	}															\
}										

WRITEFUNC (Bool,    BOOL)
WRITEFUNC (SInt8,   SINT8)
WRITEFUNC (UInt8,   UINT8)
WRITEFUNC (SInt16,  SINT16)
WRITEFUNC (UInt16,  UINT16)
WRITEFUNC (SInt32,  SINT32)
WRITEFUNC (UInt32,  UINT32)
WRITEFUNC (SInt64,  SINT64)
WRITEFUNC (UInt64,  UINT64)
WRITEFUNC (Float32, FLOAT32)
WRITEFUNC (Float64, FLOAT64)

#undef WRITEFUNC

void CChunk::WriteString (const SInt8 *inBuffer)
{
	if (inBuffer != NULL)
	{
		size_t inLength = ::strlen(inBuffer) + 1;
		if (inLength > 0)
		{
			if (CChunk::ReallocateChunk (*this, inLength))
			{
				::strncpy (reinterpret_cast <char *> (m_Chunk) + m_Size - inLength, inBuffer, inLength);
				m_DataList += STRING;
			}
		}
	}
}

void CChunk::WritePVoid (const void *inBuffer, const UInt32 inLength)
{
	if (inBuffer != NULL && inLength > 0L)
	{
		if (CChunk::ReallocateChunk (*this, inLength+sizeof(UInt32)))
		{
			*reinterpret_cast <UInt32 *> (reinterpret_cast <char *> (m_Chunk) + m_Size - inLength - sizeof(UInt32)) = inLength;
			::memcpy (reinterpret_cast <char *> (m_Chunk) + m_Size - inLength, inBuffer, inLength);
			m_DataList += PVOID;
		}
	}
}

//------------------------------------------------------------------------------------------------------------------------------------------
// chunk read
//------------------------------------------------------------------------------------------------------------------------------------------
#define READFUNC(typename) bool CChunk::Read##typename (typename &outValue)							\
{																\
	if (m_Offset + sizeof(typename) > m_Size) return false;									\
	outValue = *reinterpret_cast <typename *> (reinterpret_cast <char *> (m_Chunk) + m_Offset);				\
	m_iData++;														\
	m_Offset += sizeof(typename);												\
	return true;														\
}

READFUNC (Bool)
READFUNC (SInt8)
READFUNC (UInt8)
READFUNC (SInt16)
READFUNC (UInt16)
READFUNC (SInt32)
READFUNC (UInt32)
READFUNC (SInt64)
READFUNC (UInt64)
READFUNC (Float32)
READFUNC (Float64)

#undef READFUNC

bool CChunk::ReadString (SInt8 *&outBuffer)
{
	if (outBuffer != NULL) delete [] outBuffer;
	size_t inLength = ::strlen (reinterpret_cast <char *> (m_Chunk) + m_Offset) + 1;
	if (m_Offset + inLength > m_Size) return false;
	outBuffer = new char [inLength];
	::strncpy (outBuffer, reinterpret_cast <char *> (m_Chunk) + m_Offset, inLength);
	m_Offset += inLength;
	m_iData++;
	return true;
}

bool CChunk::ReadPVoid (void *&outBuffer, UInt32 *outLength)
{
	if (m_Offset + sizeof(UInt32) > m_Size) return false;
	if (outBuffer != NULL) delete [] reinterpret_cast <char *> (outBuffer);
	UInt32 inLength = *reinterpret_cast <UInt32 *> (reinterpret_cast <char *> (m_Chunk) + m_Offset);
	m_Offset += sizeof(UInt32);
	if (m_Offset + inLength > m_Size) return false;
	outBuffer = new char [inLength];
	::memcpy (outBuffer, reinterpret_cast <char *> (m_Chunk) + m_Offset, inLength);
	m_Offset += inLength;
	if (outLength != NULL) *outLength = inLength;
	m_iData++;
	return true;
}

#define WRITEFUNC(typename) CChunk & CChunk::operator << (const typename inValue)						\
{																\
	Write##typename (inValue);												\
	return *this;														\
}

WRITEFUNC (SInt8)
WRITEFUNC (UInt8)
WRITEFUNC (SInt16)
WRITEFUNC (UInt16)
WRITEFUNC (SInt32)
WRITEFUNC (UInt32)
WRITEFUNC (SInt64)
WRITEFUNC (UInt64)
WRITEFUNC (Float32)
WRITEFUNC (Float64)

CChunk & CChunk::operator << (const SInt8 *inBuffer)
{
	WriteString (inBuffer);
	return *this;
}

#undef WRITEFUNC

#define READFUNC(typename) CChunk & CChunk::operator >> (typename &outValue)							\
{																\
	Read##typename (outValue);												\
	return *this;														\
}

READFUNC (SInt8)
READFUNC (UInt8)
READFUNC (SInt16)
READFUNC (UInt16)
READFUNC (SInt32)
READFUNC (UInt32)
READFUNC (SInt64)
READFUNC (UInt64)
READFUNC (Float32)
READFUNC (Float64)

CChunk & CChunk::operator >> (SInt8 *&outBuffer)
{
	ReadString (outBuffer);
	return *this;
}

#undef READFUNC

CChunk & CChunk::operator ++ ()
{
	switch (NextDataIs())
	{
		case BOOL    : m_iData++; m_Offset += sizeof(Bool);    break;
		case SINT8   :
		case UINT8   : m_iData++; m_Offset += sizeof(SInt8);   break;
		case SINT16  : 
		case UINT16  : m_iData++; m_Offset += sizeof(SInt16);  break;
		case SINT32  :
		case UINT32  : m_iData++; m_Offset += sizeof(SInt32);  break;
		case SINT64  : 
		case UINT64  : m_iData++; m_Offset += sizeof(SInt64);  break;
		case FLOAT32 : m_iData++; m_Offset += sizeof(Float32); break;
		case FLOAT64 : m_iData++; m_Offset += sizeof(Float64); break;
		case STRING  : m_iData++; m_Offset += (::strlen (reinterpret_cast <char *> (m_Chunk) + m_Offset) + 1); break;
		case PVOID   :
		{
			if (m_Offset + sizeof(UInt32) <= m_Size)
			{
				m_Offset += sizeof(UInt32) + *reinterpret_cast <UInt32 *> (reinterpret_cast <char *> (m_Chunk) + m_Offset);
				m_iData++;
			}
		}
		break;
	}
	return *this;
}

CChunk & CChunk::operator ++ (int)
{
	return operator ++ ();
}

CChunk & CChunk::operator -- ()
{
	if (m_iData > 0)
	{
		switch (*m_DataList[m_iData-1])
		{
			case BOOL    : m_iData--; m_Offset -= sizeof(Bool);    break;
			case SINT8   :
			case UINT8   : m_iData--; m_Offset -= sizeof(SInt8);   break;
			case SINT16  : 
			case UINT16  : m_iData--; m_Offset -= sizeof(SInt16);  break;
			case SINT32  :
			case UINT32  : m_iData--; m_Offset -= sizeof(SInt32);  break;
			case SINT64  : 
			case UINT64  : m_iData--; m_Offset -= sizeof(SInt64);  break;
			case FLOAT32 : m_iData--; m_Offset -= sizeof(Float32); break;
			case FLOAT64 : m_iData--; m_Offset -= sizeof(Float64); break;
			case STRING  : 
			case PVOID   :
			{
				UInt32 toIndex = m_iData-1;
				m_iData  = 0;
				m_Offset = 0;
				for (UInt32 i=0; i<toIndex; i++) operator ++ ();
			}
			break;
		}
	}
	return *this;
}

CChunk & CChunk::operator -- (int)
{ 
	return operator -- ();
}

