/*******************************************************************************************************************************************
 cobject.h
 
 CClass				CClass
 |--CSerialized			|--CSerialized
    |--CMetaModule		   |--CMetaModule
       |--CObject		      |--CObjectListener

 Basic event notion definition. Any class derived from CObject has a potential associated listener that receives notifications the
 programmer can overwrite to perform specific actions.
*******************************************************************************************************************************************/

#ifndef __COBJECT_H__
#define __COBJECT_H__

#include "cmetamodule.h"
#include "cexception.h"

// defined classes
class CObject;
class CObjectListener;

// object list declaration
typedef NServices::TBuffer <CObject *> CObjects;

//-----------------------------------------------------------------------------------------------------------------------------------------
// CObjectListener class
//-----------------------------------------------------------------------------------------------------------------------------------------
class CObjectListener : public CMetaModule
{
	// instanciation section
	public :

		CObjectListener			();
		virtual ~CObjectListener	() =0;

	// listener section
	public : 

		// called when CObject constructor is executed on inSender (except when CObject is instanciated from serialization process)
		virtual void			OnConstruct		(CObject *inSender)	{ }

		// called when CObject destructor is executed on inSender, don't forget that only CObject's memory stack is available 
		// at this point !
		virtual void			OnDestruct		(CObject *inSender)	{ }

	// private section
	private :

		// the listener associated owners
		CObjects			m_Owners;

		// friend object class
		friend class			CObject;

		// generic metaclass association
		SECTION_GENERIC_METACLASS;
};

// metaclass and class tag declarations
DECLARE_GENERIC_METACLASS ('_obj', CObjectListener, CMetaModule);

//-----------------------------------------------------------------------------------------------------------------------------------------
// CObject class
//-----------------------------------------------------------------------------------------------------------------------------------------
class CObject : public CMetaModule
{
	// instanciation section
	public :

		CObject				(const CObjectListener *inListener =NULL);
		virtual ~CObject		() =0;

	// protected section
	protected :

		// object expected listener class, should be overwritten whenever the derived object is attached to a specific listener
		virtual const CMetaClass *	ListenerMustBe		() const;

	// specific functions
	public :

		// listener affectation, reader and removal (do not force class type caller on AssignListener : to increase performance, 
		// types may be checked only while assignation in overwritten functions, not before calling specific listener functions; 
		// if you force, you could generate segmentation faults)
		virtual Bool			AssignListener		(const CObjectListener *inListener);
		virtual CObjectListener *	RemoveListener		();
		CObjectListener *		GetListener		() const;

	// serialization redefinition
	public :

		// object serialization, stores or instanciates associated listener if any, archive handling
		virtual void			Serialize		(CArchive &ioArchive) THROWABLE;

		// object serialization, stores or instanciates associated listener if any, xml handling
		virtual void			Serialize		(CXMLElementNode *&ioXMLElementNode, const int inMode) THROWABLE;

	// protected attributes
	protected :
		
		// the object listener
		CObjectListener *		m_Listener;

		// the serialization handle of the object
		Bool				m_SerializeListener;

		// friend object listener class
		friend class			CObjectListener;

		// generic metaclass specifications
		SECTION_GENERIC_METACLASS;
};

// generic metaclass and class tag declarations
DECLARE_GENERIC_METACLASS ('objt', CObject, CMetaModule);

#endif
