// -*- C++ -*-
#include <iostream>
#include <cmath>
#include <sstream>

#include <ept/cache/apt/state.h>
#include <ept/cache/version.h>
#include <ept/error.h>

#ifndef EPT_CACHE_APT_STATE_TCC
#define EPT_CACHE_APT_STATE_TCC
#include <ept/cache/apt/algorithms.tcc>

namespace ept {
namespace t {
namespace cache {
namespace apt {

template< typename P >
bool State< P >::isInstalled( const Package &p ) const
{
    if ( p.pointer().package()->CurrentState
         == pkgCache::State::ConfigFiles ) return false;
    if ( p.pointer().package()->CurrentState
         == pkgCache::State::NotInstalled ) return false;
    return true;
}

template< typename P >
typename State< P >::Package::State State< P >::packageState( const Package &p ) const
{
    assert( p.id() >= 0 );
    if ( m_states.size() <= static_cast< unsigned >( p.id() ) )
        m_states.resize( p.id() + 1, 0 );
    if ( m_states[ p.id() ] != 0 )
       return m_states[ p.id() ];
    typename Package::State s = 0;
    if ( isInstalled( p ) )
        s |= Package::State::Installed;
    if ( isInstalled( p ) && p.candidateVersion() != p.installedVersion() )
        s |= Package::State::Upgradable;
    pkgDepCache::StateCache S = aptState()[ aptIterator( p ) ];
    if ( S.Install() )
        s |= Package::State::Install;
    if ( ( S.iFlags & pkgDepCache::ReInstall ) == pkgDepCache::ReInstall )
        s |= Package::State::ReInstall;
    if ( S.Keep() )
        s |= Package::State::Keep;
    if ( S.Delete() )
        s |= Package::State::Remove;
    if ( ( S.iFlags & pkgDepCache::Purge ) == pkgDepCache::Purge )
        s |= Package::State::Purge;
    if ( S.NowBroken() )
        s |= Package::State::NowBroken;
    if ( S.InstBroken() )
        s |= Package::State::WillBreak;
    m_states[ p.id() ] = s;
    return s;
}

/*
State::State( const State &s )
    : Implementation<State>( s ), m_policy( s.m_policy )
{
    Cache = s.Cache;
    if ( Cache ) {
        // std::cerr << "deep-copying State!" << std::endl;
        // std::cerr << "Cache = " << Cache << std::endl;
        PkgState = new StateCache[Head().PackageCount];
        DepState = new unsigned char[Head().DependsCount];
        std::copy( s.PkgState, s.PkgState+
                   Head().PackageCount, PkgState );
        std::copy( s.DepState, s.DepState+
                   Head().DependsCount, DepState );
        LocalPolicy = &policy();
        delLocalPolicy = 0;
    }
}
*/

template< typename P >
void State< P >::action( Package p, Action a, bool notify ) {
    /* if ( notify )
       notifyPreChange(); */
    Resolver< DepCache > fix( &aptState() );
    if ( a == AInstall || a == AReInstall ) {
        // std::cerr << "marking for install: " << p.name() << std::endl;
        fix.Clear( aptIterator( p ) );
        fix.Protect( aptIterator( p ) );
        aptState().MarkInstall( aptIterator( p ), true );
        fix.InstallProtect();
        if ( a == AReInstall )
            aptState().SetReInstall( aptIterator( p ), true );
    } else if ( a == ARemove || a == APurge ) {
        // std::cerr << "marking for removal: " << p.name() << std::endl;
        fix.Clear( aptIterator( p ) );
        fix.Protect( aptIterator( p ) );
        fix.Remove( aptIterator( p ) );
        aptState().MarkDelete( aptIterator( p ), a == APurge ? true : false );
    } else if ( a == AKeep ) {
        // std::cerr << "marking for keeping: " << p.name() << std::endl;
        fix.Clear( aptIterator( p ) );
        fix.Protect( aptIterator( p ) );
        aptState().MarkKeep( aptIterator( p ), true );
    }
    fix.Resolve( true );
    /* if ( notify )
       notifyPostChange(); */
}

template< typename P >
void State< P >::revert( wibble::Range< Package > r ) {
    typename wibble::Range< Package >::iterator it;
    for ( it = r.begin(); it != r.end(); ++it ) {
        if ( ( !it->markedKeep() ) || it->markedPurge() ) {
            pkgDepCache::StateCache &S = aptState()[ aptIterator( *it ) ];
            aptState().MarkKeep( aptIterator( *it ) );
            S.iFlags &= ~pkgDepCache::Purge;
            S.iFlags &= ~pkgDepCache::ReInstall;
        }
    }
}

template< typename P >
void State< P >::packageChanged( Package p ) {
    packages().packageChanged.emit( p );
}

template< typename P >
void State< P >::replay( RequestList l ) {
    // notifyPreChange();
    while ( l != l.end() ) {
        action( l->first, l->second, false );
        l.advance();
    }
    // notifyPostChange();
}

template< typename P >
State< P >::State( Aggregator &p )
    : m_packages( p )
{
    open();
}

template< typename P >
void State< P >::open() {
    m_policy = new PackagePolicy< P >( index() );
    m_removeCount = m_newInstallCount = m_upgradeCount = m_reInstallCount = 0;
    m_installedCount = m_upgradableCount = m_availableCount = 0;

    m_aptState = new DepCache( &m_packages, this, &index().aptCache(), &policy() );

    // states need to be created already since the DepCache->Init tries
    // to access that
    m_states.clear();
    m_states.resize( index().packageCount() );

    m_aptState->Init( 0 ); // XXX pass progress around
    checkGlobalError( "error constructing package policy" );

    if ( ReadPinFile( policy() )  == false )
        throw wibble::exception::System( "error reading pin file" );
}

template< typename P >
void State< P >::close() {
    delete m_policy;
    delete m_aptState;
    m_policy = 0;
    m_aptState = 0;
}

template< typename P >
State< P >::~State ()
{
    close();
}

template< typename P >
bool State< P >::changed() const {
    return (removeCount() || newInstallCount() || upgradeCount() || reInstallCount());
}

template< typename P >
std::string State< P >::sizeString( double d )
{
    const char *post = "BKMG";
    while ( std::fabs( d ) > 4096 && post[0] != 'G' )
        (d /= 1024), (++post);
    std::ostringstream s;
    d = (d - std::floor( d ) > 0.5) ? std::ceil( d ) : std::floor( d );
    s << int( d ) << post[0];
    return s.str();
}

}
}
}
}

#endif
