/** -*- C++ -*-
    @file cache/apt/index.h
    @author Peter Rockai <me@mornfall.net>
*/

#include <apt-pkg/pkgcache.h>
#include <wibble/range.h>
#include <ept/forward.h>
#include <ept/cache/component.h>

#include <map>

#ifndef EPT_CACHE_APT_INDEX_H
#define EPT_CACHE_APT_INDEX_H

namespace ept {
namespace t {
namespace cache {
namespace apt {

template< typename _ > struct PackagePolicy;

template< typename T >
struct Indirector {

    Indirector( T h ) : helper( h ), m_firstFreeRuntimeId( 1 ) {}

    int ondiskToRuntime( int ondisk ) const {
        assert( ondisk >= 0 );
        if ( m_ondiskToRuntime.size() <= static_cast< unsigned >( ondisk ) )
            m_ondiskToRuntime.resize( ondisk + 1, 0 );
        int nid = m_ondiskToRuntime[ ondisk ];
        if ( nid == 0 ) { // not yet seen since invalidation
            nid = m_nameToRuntime[ helper.ondiskToName( ondisk ) ];
            if ( nid == 0 ) { // never seen
                nid = m_firstFreeRuntimeId;
                if ( m_runtimeToName.size() <= static_cast< unsigned >( nid ) )
                    m_runtimeToName.resize( nid + 1, "" );
                m_nameToRuntime[ helper.ondiskToName( ondisk ) ] = nid;
                m_runtimeToName[ nid ] = helper.ondiskToName( ondisk );
                m_firstFreeRuntimeId += 1;
            }
            m_ondiskToRuntime[ ondisk ] = nid;
            if ( m_runtimeToOndisk.size() <= static_cast< unsigned >( nid ) )
                m_runtimeToOndisk.resize( nid + 1, helper.invalidOndisk() );
            m_runtimeToOndisk[ nid ] = ondisk;
        }
        return nid;
    }

    int runtimeToOndisk( int runtime ) const {
        assert( runtime >= 0 );
        if ( m_runtimeToOndisk.size() <= static_cast< unsigned >( runtime ) )
            m_runtimeToOndisk.resize( runtime + 1, helper.invalidOndisk() );
        int ondisk = m_runtimeToOndisk[ runtime ];
        if ( ondisk == helper.invalidOndisk() ) {
            std::string name = m_runtimeToName[ runtime ];
            ondisk = helper.nameToOndisk( name );
            if ( ondisk != helper.invalidOndisk() ) {
                m_runtimeToOndisk[ runtime ] = ondisk;
                if ( m_ondiskToRuntime.size() <= static_cast< unsigned >( ondisk ) )
                    m_ondiskToRuntime.resize( ondisk + 1, 0 );
                m_ondiskToRuntime[ ondisk ] = runtime;
            } else; // XXX?
        }
        return ondisk;
    }

    void invalidate() {
        m_ondiskToRuntime.clear();
        m_runtimeToOndisk.clear();
    }

    // also used as idToName lookup
    mutable std::vector< int > m_ondiskToRuntime;
    mutable std::vector< int > m_runtimeToOndisk;
    mutable std::vector< std::string > m_runtimeToName;
    mutable std::map< std::string, int > m_nameToRuntime;
    T helper;
    mutable int m_firstFreeRuntimeId;

};

template< typename C >
struct Index {
public:
    typedef typename C::Package Package;
    typedef typename C::Version Version;
    typedef typename C::Relation Relation;

    typedef typename C::State State;
    typedef typename C::Aggregator Aggregator;

    typedef wibble::Range< Version > VersionRange;
    typedef wibble::Range< Relation > RelationRange;

    struct PackagePointer : Pointer< C > {
        friend struct apt::Index< C >;
        friend struct apt::State< C >;
        friend struct Relation::Provides; // XXX
        long id() const { return m_id; }
        bool valid() const { return Pointer< C >::valid() && m_id != 0; } // ERR
        PackagePointer() : m_id( 0 ) {}

    protected:
        PackagePointer( const Aggregator &c, int id )
            : Pointer< C >( c ), m_id( id ) {}
        long hashIndex() { return this->aggregator().index().hashIndex( id() ); }
        pkgCache::Package *package() { return this->aggregator().index().aptPackageById( id() ); }
        // XXX should be const... really
        pkgCache::Package *package() const {
            return this->aggregator().index().aptPackageById( id() ); }
        long m_id;
    };

    struct VersionPointer : Pointer< C > {
        friend struct apt::Index< C >;
        friend struct apt::State< C >;
        friend struct Relation::Provides; // XXX
        long id() const { return m_version->ID; }
        bool valid() const { return Pointer< C >::valid()
                && m_version && m_version != this->aggregator().index().verPtr(); }
        VersionPointer() : m_version( 0 ) {}

    protected:
        VersionPointer( const Aggregator &c, pkgCache::Version *v )
            : Pointer< C >( c ), m_version( v ) {}
        pkgCache::Version *version() const { return m_version; }
        pkgCache::Version *m_version;
    };

    struct RelationPointer : Pointer< C > {
        bool valid() const { return Pointer< C >::valid()
                && m_dependency && m_dependency != this->aggregator().index().depPtr(); }
        RelationPointer() : m_dependency( 0 ) {}
        RelationPointer( const Aggregator &c, pkgCache::Dependency *d )
            : Pointer< C >( c ), m_dependency( d ) {}
        pkgCache::Dependency *dependency() const { return m_dependency; }
        pkgCache::Dependency *m_dependency;
    };

    struct IndirectorHelper {
        IndirectorHelper( Aggregator&a ) : aggregator( a ) {}
        int nameToOndisk( std::string s ) const {
            if ( s == "-invalid-" ) return invalidOndisk();
            return aggregator.index().aptPackageByName( s )->ID;
        }
        std::string ondiskToName( int id ) const {
            if ( id == invalidOndisk() ) return "-invalid-";
            return aggregator.index().strPtr( aggregator.index().m_ondiskToPointer[ id ]->Name );
        }
        int invalidOndisk() const { return aggregator.index().packageCount(); }
        Aggregator &aggregator;
    };

    // this is like this until the time comes i figure how to
    // SWIG-enable the commented out version (which is a bit faster in c++)
    // update: or no
    typedef wibble::Range< Package > PackageRange;
    /* typedef wibble::GeneratedRange< Package,
                                   void(*)( Package & ),
                                   bool(*)( const Package & ) >
                                   PackageRange; */

    void reload();

    static time_t currentTimestamp();
    time_t timestamp() const { return m_timestamp; }

    /// Get the number of packages in the cache
    int packageCount() const { return aptCache().HeaderP->PackageCount; }

    /// return a package of a given name
    Package createPackage( pkgCache::Package *p, long h ) const;
    Package packageByName( const std::string& name ) const;
    pkgCache::Package *aptPackageByName( const std::string& name ) const;
    Index( Aggregator &a );

    typedef typename Index::PackageRange::iterator iterator;

    iterator begin() { return range().begin(); }
    iterator end() { return range().end(); }

    PackageRange range(); // { return m_range; }
    wibble::Range< Package > sorted();

// ** UNSAFE METHODS FOLLOW

    pkgCache::VerFileIterator fileList( const Version &v ) const {
        return pkgCache::VerFileIterator( aptCache(), verFilePtr( v.pointer() ) );
    }

    struct PackageData {
        PackageData() : valid(false) {}
        std::string name;
        bool valid:1;
    };

    static void advancePackageInCache( Package &p );

    static void advanceVersionInCache( Version &v ) {
        v.checkValid();
        v = Version( VersionPointer( v.aggregator(),
                                     v.aggregator().index().verPtr(
                                         v.pointer().version()->NextVer ) ) );
    }

    static bool lastVersionInCache( const Version &v ) {
        return v == Version();
    }

    // somewhat friendlier mmap-ed area interface
    const char *strPtr( map_ptrloc offset = 0 ) const {
        return offset ? aptCache().StrP + offset : ""; }
    pkgCache::Package *pkgPtr( map_ptrloc offset = 0 ) const { return aptCache().PkgP + offset; }
    pkgCache::Version *verPtr( map_ptrloc offset = 0 ) const { return aptCache().VerP + offset; }
    pkgCache::Dependency *depPtr( map_ptrloc offset = 0 ) const { return aptCache().DepP + offset; }
    pkgCache::VerFile *verFilePtr( VersionPointer v ) const {
        return aptCache().VerFileP + v.version()->FileList; }
    pkgCache::Provides *providesPtr( map_ptrloc offset = 0 ) const {
        return aptCache().ProvideP + offset; }
    map_ptrloc hashOffset (int idx) const { return aptCache().HeaderP->HashTable[idx]; }

    VersionRange versions( Package p ) const {
        pkgCache::Version *f = verPtr( p.pointer().package()->VersionList );
        return wibble::generatedRange( Version( VersionPointer( p.pointer().aggregator(), f ) ),
                                       advanceVersionInCache, lastVersionInCache );
    }

    // RelationRange relations( const Version &v );

    Version currentVersion( Package p ) const {
        return Version( VersionPointer( p.pointer().aggregator(),
                                        verPtr( p.pointer().package()->CurrentVer ) ) );
    }

    int installedSize( const Version &v ) const { return v.pointer().version()->InstalledSize; }
    std::string architecture( const Version &v ) const {
        return strPtr( v.pointer().version()->Arch ); }
    std::string section( const Version &v ) const {
        return strPtr( v.pointer().version()->Section ); }

    Package packageForVersion( Version ) const;

    int hashSize() const;

    // pkgCache::Package *lookupRedirect( int ) const;
    // int lookupUnredirect( pkgCache::Package * ) const;

    /// Get the name of a package
    std::string packageName( const Package &p ) const {
        return std::string( strPtr( p.pointer().package()->Name ) );
    }

    std::string packageSection( const Package &p ) const {
        return std::string( strPtr( p.pointer().package()->Section ) );
    }

    std::string versionString( Version v ) const {
        return strPtr( v.pointer().version()->VerStr );
    }

    pkgCache &aptCache() const { return *m_aptCache; }
    // const pkgCache &aptCache() const { return *m_aptCache; }

    long hashIndex( int id ) const { return m_hashIndex[ id ]; }

    Package packageFromDependency( pkgCache::Dependency *d ) const { // XXX
        return createPackage( pkgPtr( d->Package ), -1 );
    }

    Version versionOwningDependency( pkgCache::Dependency *d ) const { // XXX
        return Version( VersionPointer( m_aggregator, verPtr( d->ParentVer ) ) );
    }

    Version versionFromProvides( pkgCache::Provides *p ) const { // XXX
        return Version( VersionPointer( m_aggregator, verPtr( p->Version ) ) );
    }

    Relation firstRelationForVersion( Version v ) const { // XXX
        return Relation( RelationPointer( m_aggregator,
                                          depPtr( v.pointer().version()->DependsList ) ) );
    }

    pkgCache::Package *aptPackageById( int id ) const {
        return m_ondiskToPointer[ m_indirector.runtimeToOndisk( id ) ];
    }

    int ondiskId( Package p ) const {
        return m_indirector.runtimeToOndisk( p.id() );
    }

protected:
    void open();
    void close();
    pkgCache *m_aptCache;
    Aggregator &m_aggregator;
    Indirector< IndirectorHelper > m_indirector;
    time_t m_timestamp;

    mutable std::vector< long > m_hashIndex; // hashindex cache
    std::vector< pkgCache::Package * > m_ondiskToPointer;

    mutable std::set< Package, bool(*)( const Package &,
                                        const Package & ) > m_sorted;
};

}
}
}
}

#endif
