#pragma implementation
#include <stdio.h>
#include <stdlib.h>
#include <ctype.h>
#include <string.h>
#include <configf.h>
#include <misc.h>
#include <confdb.h>
#include "quota.h"
#include "fstab.h"
#include "fstab.m"
#include "../paths.h"
#include <pwd.h>
#include <grp.h>
#include <userconf.h>
#ifdef __GLIBC__
	#include <asm/types.h>
	#include <sys/types.h>
	#include <sys/quota.h>
	#include <unistd.h>
#else
	#include <linux/types.h>
	#include <linux/quota.h>
#endif
#include <dialog.h>

static FSTAB_HELP_FILE help_quota ("quota");
static CONFIG_FILE f_quota (ETC_QUOTA_CONF,help_quota
	,CONFIGF_MANAGED|CONFIGF_OPTIONNAL
	,"root","root",0600);

static const char K_USER[]="user";
static const char K_USERDEF[]="userdef";
static const char K_USERDEF_G[]="userdef_g";
static const char K_GROUP[]="group";
static const char K_GROUPDEF[]="groupdef";

// Default for grace period is seven days.
#define GRACE_DEF	(7*24*60*60)


/* #Specification: Quota management / strategy
	The files /quota.user and /quota.group are both configuration file
	and  "work" files for the kernel. As such, their format is heavily
	ditacted by the performance requirement of the kernel itself.

	Mostly, these file are binary files and hold one record per potential
	users. As a configuration file, they are pretty poor. There is
	one record per user and it must be filled to be of any use. This
	defeats administration strategy such as providing defaults setup.

	While one can copy the setup of one user to another, as such providing
	some mecanism to apply a default, this is a one way situation: Once
	done, you can' tell if this user has a specific setup which has the
	same values as the default or simply inherited the default values.

	The strategy used by linuxconf does not managed the quota.user and
	quota.group files as configuration file. Linuxconf uses other
	file located in /etc to record the specifications and defaults. Using
	these new files, linuxconf is able to update the /quota.user and
	/quota.group. This new strategy allows one to

	#
		-leave a user account empty quota wise. The user account
		 will receive some defaults values.
		-Change the default values and this will ripple to all
		 users without specific setups.
		-Allows the definition of a per group default setup. (this
		 is completly different from group quota).
	#
*/
/* #Specification: quota management / values
	All values stored in the quota.conf file may be in 3 states

	#
	 0: No limit
	-1: Inherit from defaults or group defaults
	>0: Use this limit
	#
*/
#
PRIVATE void QUOTA_SPEC::init ()
{
	soft_maxk = soft_maxf = hard_maxk = hard_maxf = -1;
	grace_k.setfrom (GRACE_DEF);
 	grace_f.setfrom (GRACE_DEF);
}

/*
	Reset to "open" (no limit) values
*/
PUBLIC void QUOTA_SPEC::reset ()
{
	soft_maxk = soft_maxf = hard_maxk = hard_maxf = 0;
	grace_k.setfrom (GRACE_DEF);
	grace_f.setfrom (GRACE_DEF);
}

/*
	Return true if any parameter do differ between two QUOTA_SPEC
*/
PUBLIC bool QUOTA_SPEC::isdiff(const QUOTA_SPEC &s) const
{
	return soft_maxk != s.soft_maxk
			|| soft_maxf != s.soft_maxf
			|| hard_maxk != s.hard_maxk
			|| hard_maxf != s.hard_maxf
			|| grace_k.seconds != s.grace_k.seconds
			|| grace_f.seconds != s.grace_f.seconds;
}


PUBLIC QUOTA_SPEC::QUOTA_SPEC()
{
	init();
}

PUBLIC QUOTA_SPEC::QUOTA_SPEC(const char *val)
{
	init();
	setfrom (val);
}

PUBLIC QUOTA_SPEC::QUOTA_SPEC(const QUOTA_SPEC *s)
{
	setfrom (s);
}	

PUBLIC void QUOTA_SPEC::setfrom(const QUOTA_SPEC *s)
{
	name.setfrom (s->name);
	soft_maxk = s->soft_maxk;
	soft_maxf = s->soft_maxf;
	hard_maxk = s->hard_maxk;
	hard_maxf = s->hard_maxf;
	grace_k.setfrom (s->grace_k);
	grace_f.setfrom (s->grace_f);
}	

PUBLIC void QUOTA_SPEC::setfrom (const char *val)
{
	if (val != NULL){
		val = name.copyword (val);
		long gk,gf;
		sscanf (val,"%d %d %d %d %ld %ld",&soft_maxk,&hard_maxk
			,&soft_maxf,&hard_maxf,&gk,&gf);
		grace_k.setfrom (gk);
		grace_f.setfrom (gf);
	}
}

PUBLIC void QUOTA_SPEC::write (
	CONFDB &conf,
	const char *key,
	const char *device)
{
	char buf[200];
	sprintf (buf,"%s %d %d %d %d %ld %ld",name.get()
		,soft_maxk,hard_maxk,soft_maxf,hard_maxf
		,grace_k.seconds,grace_f.seconds);
	conf.add (key,device,buf);
}

PUBLIC QUOTA_SPECS::QUOTA_SPECS(
	CONFDB &conf,
	const char *key,		// user quota, group quota or group defaults
	const char *device)		//`Device on which those quota apply
{
	SSTRINGS tb;
	int n = conf.getall (key,device,tb,0);
	for (int i=0; i<n; i++){
		const char *val = tb.getitem(i)->get();
		add (new QUOTA_SPEC(val));
	}
}

PUBLIC QUOTA_SPECS::QUOTA_SPECS()
{
}

PUBLIC void QUOTA_SPECS::write (
	CONFDB &conf,
	const char *key,
	const char *device)
{
	conf.removeall (key,device);
	int n = getnb();
	for (int i=0; i<n; i++){
		QUOTA_SPEC *s = getitem(i);
		s->write (conf,key,device);
	}
}


PUBLIC QUOTA_SPEC* QUOTA_SPECS::getitem (int no) const
{
	return (QUOTA_SPEC*) ARRAY::getitem(no);
}

/*
	Locate one spec by name.
	Return NULL if not found
*/
PUBLIC QUOTA_SPEC* QUOTA_SPECS::getitem (const char *name) const
{
	QUOTA_SPEC *ret = NULL;
	int n = getnb();
	for (int i=0; i<n; i++){
		QUOTA_SPEC *sp = getitem (i);
		if (sp->name.cmp(name)==0){
			ret = sp;
			break;
		}
	}
	return ret;
}

PUBLIC QUOTA_DEV::QUOTA_DEV (CONFDB &conf, const char *device)
	:
	userdef_g(conf,K_USERDEF_G,device),
	users (conf,K_USER,device),
	groups (conf,K_GROUP,device)
{
	this->device.setfrom (device);
	userdef.setfrom (conf.getval (K_USERDEF,device));
	userdef.name.setfrom ("none");
	groupdef.setfrom (conf.getval(K_GROUPDEF,device));
	groupdef.name.setfrom ("none");
}

PUBLIC void QUOTA_DEV::write (CONFDB &conf)
{
	const char *dev = device.get();
	conf.removeall (K_USERDEF,dev);
	userdef.write (conf,K_USERDEF,dev);
	conf.removeall (K_GROUPDEF,dev);
	groupdef.write (conf,K_GROUPDEF,dev);
	users.write (conf,K_USER,dev);
	userdef_g.write (conf,K_USERDEF_G,dev);
	groups.write (conf,K_GROUP,dev);
}

static void quota_override (QUOTA_SPEC &eff, QUOTA_SPEC *one)
{
	if (one != NULL){
		if (one->soft_maxk != -1) eff.soft_maxk = one->soft_maxk;
		if (one->soft_maxf != -1) eff.soft_maxf = one->soft_maxf;
		if (one->hard_maxk != -1) eff.hard_maxk = one->hard_maxk;
		if (one->hard_maxf != -1) eff.hard_maxf = one->hard_maxf;
		if (one->grace_k.seconds != 0){
			eff.grace_k.setfrom (one->grace_k);
		}
		if (one->grace_f.seconds != 0){
			eff.grace_f.setfrom (one->grace_f);
		}
	}
}

/*
	Get the quota spec applicable to a user.
	Always return something useful, ultimatly, the device wide defaults.
*/
PUBLIC void QUOTA_DEV::geteffuserspec (
	const char *name,
	const char *group,
	QUOTA_SPEC &eff)		// Will contain the effective values
{
	/* #Specification: Quota / user default management / strategy
		A given user may be in one of the following state (for
		the configuration of the user quota). He may

		#
		-have some specific quota values
		-not have any specific values, but there is a default defined
		 for group he beloong.
		-not have any specific values and there is no default for
		 his group, so the defaults are used.
		#

		Further the strategy is applied item per item of the quota record.
		This means that a user may have a soft quota on disk space but
		may inherit the other limits from the group defaults or
		the device defaults.
	*/
	eff.reset();
	quota_override (eff,&userdef);
	quota_override (eff,userdef_g.getitem(group));
	quota_override (eff,users.getitem(name));
}

/*
	Get the quota spec applicable to a group.
	Always return something useful, ultimatly, the device wide defaults.
*/
PUBLIC void QUOTA_DEV::geteffgroupspec (
	const char *group,
	QUOTA_SPEC &eff)
{
	/* #Specification: Quota / group default management / strategy
		A given group may be in one of the following state (for
		the configuration of the group quota). He may

		#
		-have some specific quota values
		-not have any specific values, so the defaults are picked.
		#
	*/
	eff.reset();
	quota_override (eff,&groupdef);
	quota_override (eff,groups.getitem(group));
}

/*
	Return the quota specification for a user or NULL if the user do not
	have a specific record.
*/
PUBLIC QUOTA_SPEC *QUOTA_DEV::getuserspec (const char *name)
{
	return users.getitem(name);
}
/*
	Return the quota specification for a group or NULL if the user do not
	have a specific record.
*/
PUBLIC QUOTA_SPEC *QUOTA_DEV::getgroupspec (const char *name)
{
	return groups.getitem(name);
}

/*
	Return the quota specification for a per group defaults or NULL
	if there is not default for that group.
*/
PUBLIC QUOTA_SPEC *QUOTA_DEV::getuserdef_gspec (const char *name)
{
	return userdef_g.getitem(name);
}

/*
	Delete the quota specification for a user
*/
PUBLIC void QUOTA_DEV::deluserspec (const char *name)
{
	users.remove_del (getuserspec(name));
}
/*
	Delete the quota specification for a group.
*/
PUBLIC void QUOTA_DEV::delgroupspec (const char *name)
{
	groups.remove_del (getgroupspec(name));
}

/*
	Delete the quota specification for the per group defaults.
*/
PUBLIC void QUOTA_DEV::deluserdef_gspec (const char *name)
{
	userdef_g.remove_del (getuserdef_gspec(name));
}

#ifndef __GLIBC__
#if defined (__alpha__)
	#include <errno.h>
	#include <syscall.h>
	#include <asm/unistd.h>

	int quotactl(int cmd, const char * special, int id, caddr_t addr)
	{
		return syscall(__NR_quotactl, cmd, special, id, addr);
	}
#else
	#define __LIBRARY__
	#include <linux/unistd.h>

	_syscall4(int, quotactl, int, cmd, const char *, special, int, id, caddr_t, addr);
#endif
#endif



static int quota_apply (
	QUOTA_SPEC &spec,
	int id,
	int type,
	const char *device)
{
	struct dqblk dq;
	// Translation from K to block is not done!!!!!
	dq.dqb_bhardlimit = spec.hard_maxk;
	dq.dqb_bsoftlimit = spec.soft_maxk;

	dq.dqb_ihardlimit = spec.hard_maxf;
	dq.dqb_isoftlimit = spec.soft_maxf;

	dq.dqb_btime = spec.grace_k.seconds;
	dq.dqb_itime = spec.grace_f.seconds;
	return quotactl (QCMD(Q_SETQLIM,type),device,id,(caddr_t)&dq);
}
/*
	Install the quota for a given user
*/
PUBLIC void QUOTA_DEV::applyone (struct passwd *p)
{
	if (!user_isadmin(p->pw_name)){
		struct group *g = getgrgid (p->pw_gid);
		if (g != NULL){
			QUOTA_SPEC eff;
			geteffuserspec (p->pw_name,g->gr_name,eff);
			quota_apply (eff,p->pw_uid,USRQUOTA,realdev.get());
		}
	}
}
/*
	Transform a LABEL= specification into a device
	Return -1 if this can't be done.
*/
PUBLIC int QUOTA_DEV::setrealdevice()
{
	int ret = 0;
	const char *dev = device.get();
	realdev.setfrom (dev);
	if (strncasecmp(dev,"LABEL=",6)==0){
		dev = partition_findfromlabel(dev+6);
		if (dev == NULL){
			xconf_error (MSG_U(E_TRANSLATELABEL
				,"Can't translate the file system label %s to any device.\n"
				 "Won't apply quota"),device.get());
			ret = -1;;
		}else{
			realdev.setfrom (dev);
		}
	}
	return ret;
}

/*
	Fill the user and group quota for all users and group on the system
	It basically apply map /etc/quota.conf on this device
*/
PUBLIC void QUOTA_DEV::applyall()
{
	if (setrealdevice()!=-1){
		const char *devstr = realdev.get();
		setpwent();
		struct passwd *p;
		while ((p=getpwent())!=NULL){
			applyone (p);
		}
		setgrent ();
		struct group *g;
		while ((g=getgrent())!=NULL){
			QUOTA_SPEC eff;
			geteffgroupspec (g->gr_name,eff);
			quota_apply (eff,g->gr_gid,GRPQUOTA,devstr);
		}
	}
}

PUBLIC QUOTA_DEV *QUOTA_DEVS::getitem (int no) const
{
	return (QUOTA_DEV*)ARRAY::getitem(no);
}

PUBLIC QUOTACTL::QUOTACTL()
{
	conf = new CONFDB (f_quota);
	FSTAB fs;
	for (int i=0; i<fs.getnb(); i++){
		FSTAB_ENTRY *e = fs.getitem(i);
		if (e->has_quota_u() || e->has_quota_g()){
			devs.add (new QUOTA_DEV (*conf,e->getsource()));
		}
	}
}

PUBLIC QUOTACTL::~QUOTACTL()
{
	delete conf;
}

PUBLIC int QUOTACTL::write (PRIVILEGE *priv)
{
	for (int i=0; i<devs.getnb(); i++){
		devs.getitem(i)->write (*conf);
	}
	return conf->save(priv);
}

PUBLIC void QUOTA_SPEC::setupdia(DIALOG &dia)
{
	static const char *tb[]={MSG_U(I_NOLIMIT,"No limit"),NULL};
	static const int tbv[]={-1,0};

	dia.newf_chkm_num (MSG_U(F_SOFTK,"Disk space soft limit"),soft_maxk
		,tbv,tb);
	dia.newf_chkm_num (MSG_U(F_HARDK,"Disk space hard limit"),hard_maxk
		,tbv,tb);
	dia.newf_str (MSG_U(F_GRACEK,"Disk space grace period"),grace_k);
	dia.newf_title ("","");
	dia.newf_chkm_num (MSG_U(F_SOFTF,"Files soft limit"),soft_maxf
		,tbv,tb);
	dia.newf_chkm_num (MSG_U(F_HARDF,"Files hard limit"),hard_maxf
		,tbv,tb);
	dia.newf_str (MSG_U(F_GRACEF,"Files grace period"),grace_f);
}



PUBLIC void QUOTACTL::editdef ()
{
	if (devs.getnb()==0){
		xconf_error (MSG_U(E_NODEVQUOTA,
			"There is currently no partition with disk quota enabled.\n"
			"Visit the \"Access local drive\" menu and enable\n"
			"disk quota on selected volumes."));
	}else{
		DIALOG dia;
		for (int i=0; i<devs.getnb(); i++){
			QUOTA_DEV *d = devs.getitem(i);
			const char *device = d->device.get();
			dia.newf_title (device,1,"",device);
			QUOTA_SPEC *s = &d->userdef;
			dia.newf_title (MSG_U(T_USERDEF,"User default"),2,"",MSG_R(T_USERDEF));
			s->setupdia (dia);
			s = &d->groupdef;
			dia.newf_title (MSG_U(T_GROUPDEF,"Group default"),2,"",MSG_R(T_GROUPDEF));
			s->setupdia (dia);
		}
		int nof = 0;
		while (1){
			MENU_STATUS code = dia.edit (MSG_U(T_QUOTADEF
				,"Default quota for users and groups")
				,MSG_U(I_QUOTADEF,"")
				,help_quota
				,nof);
			if (code == MENU_CANCEL || code == MENU_ESCAPE){
				dia.restore();
				break;
			}else{
				write(NULL);
				applyall();
				break;
			}
		}
	}
}

static QUOTACTL *ctl;
static short int nbuse=0;

static void quota_allocctl()
{
	if (ctl == NULL){
		ctl = new QUOTACTL;
		nbuse = 1;
	}else{
		nbuse++;
	}
}

static void quota_freectl()
{
	nbuse--;
	if (nbuse == 0){
		delete ctl;
		ctl = NULL;
	}
}

/*
	Edit the defaults for users and groups
*/
void quota_editdef ()
{
	quota_allocctl();
	if (ctl != NULL) ctl->editdef();
	quota_freectl();
}

// Run through all users and group to update their quota
// This is not efficient but will do the job
PUBLIC void QUOTACTL::applyall ()
{
	for (int i=0; i<devs.getnb(); i++){
		devs.getitem(i)->applyall();
	}
}

// Run through all users and group to update their quota on a given device
PUBLIC void QUOTACTL::applyall (const char *dev)
{
	for (int i=0; i<devs.getnb(); i++){
		QUOTA_DEV *d = devs.getitem(i);
		if (d->device.cmp(dev)==0){
			d->applyall();
			break;
		}
	}
}

// Install the disk quota for one user on all devices
PUBLIC int QUOTACTL::applyone (const char *user)
{
	int ret = -1;
	struct passwd *p = getpwnam (user);
	if (p != NULL){
		for (int i=0; i<devs.getnb(); i++){
			QUOTA_DEV *dev = devs.getitem(i);
			if (dev->setrealdevice()!=-1) dev->applyone(p);
		}
		ret = 0;
	}
	return ret;
}

/*
	Install the disk quota for one user on all devices
*/
int quota_applyone (const char *user)
{
	int ret = -1;
	quota_allocctl();
	if (ctl != NULL) ret = ctl->applyone (user);
	quota_freectl();
	return ret;
}

/*
	Set the quota limits for all users and groups on a device.
*/
void quota_applyall (const char *dev)
{
	quota_allocctl();
	if (ctl != NULL) ctl->applyall (dev);
	quota_freectl();
}

/*
	Walk all device with quota and update the quota record for all users
	This is used after having added some users without linuxconf (useradd)
	so the quota defaults apply correctly.
*/
void quota_applyall ()
{
	FSTAB fstab;
	for (int i=0; i<fstab.getnb(); i++){
		FSTAB_ENTRY *e = fstab.getitem(i);
		if (e->has_quota_u() || e->has_quota_g()){
			const char *dev = e->getsource();
			quota_applyall (dev);
		}
	}
}

/*
	Extract the quota spec associated with the name.
	Return NULL if not found
*/
PRIVATE QUOTA_SPEC *QUOTA_EDIT::getspec	(QUOTA_DEV *d)
{
	QUOTA_SPEC *u = NULL;
	const char *nam = name.get();
	if (nam[0] != '\0'){
		if (type == QUOTA_USER){
			u = d->getuserspec (nam);
		}else if (type == QUOTA_GROUP){
			u= d->getgroupspec (nam);
		}else if (type == QUOTA_USERDEF_G){
			u= d->getuserdef_gspec (nam);
		}
	}
	return u;
}

/*
	Remove the user from the quota.conf file
*/
PUBLIC int QUOTA_EDIT::deluser(PRIVILEGE *priv)
{
	int ret = -1;
	const char *nam = name.get();
	if (nam[0] != '\0'){
		for (int i=0; i<specs.getnb(); i++){
			QUOTA_DEV *d = ctl->devs.getitem(i);
			if (type == QUOTA_USER){
				d->deluserspec (nam);
			}else if (type == QUOTA_GROUP){
				d->delgroupspec (nam);
			}else if (type == QUOTA_USERDEF_G){
				d->deluserdef_gspec (nam);
			}
		}
		ret = ctl->write(priv);	
	}
	return ret;
}

/*
	Extract the effective quota spec associated with the name.
*/
PRIVATE void QUOTA_EDIT::geteffspec	(QUOTA_DEV *d, QUOTA_SPEC &eff)
{
	const char *nam = name.get();
	if (type == QUOTA_USER){
		d->geteffuserspec (nam,group.get(),eff);
	}else if (type == QUOTA_GROUP){
		d->geteffgroupspec (nam,eff);
	}else if (type == QUOTA_USERDEF_G){
		d->geteffuserspec ("",nam,eff);
	}
}
	

/*
	Prepare the edition of disk quota for a user account or group
*/
PUBLIC QUOTA_EDIT::QUOTA_EDIT (const char *_name, QUOTA_TYPE type)
{
	this->type = type;
	setname(_name);
	is_new = name.is_empty();
	quota_allocctl();
	for (int i=0; i<ctl->devs.getnb(); i++){
		QUOTA_DEV *d = ctl->devs.getitem(i);
		QUOTA_SPEC *u = getspec(d);
		if (u == NULL){
			u = new QUOTA_SPEC();
			u->name.setfrom (name);
		}else{
			u = new QUOTA_SPEC(u);
		}
		specs.add (u);
	}
}

PUBLIC void QUOTA_EDIT::setname (const char *_name)
{
	name.setfrom (_name);
}

PUBLIC void QUOTA_EDIT::setgroup (const char *grp)
{
	group.setfrom (grp);
}

PUBLIC QUOTA_EDIT::~QUOTA_EDIT ()
{
	quota_freectl();
}

/*
	Accept the changes done for this user or group
*/
PUBLIC int QUOTA_EDIT::save(PRIVILEGE *priv)
{
	const char *nam = name.get();
	for (int i=0; i<specs.getnb(); i++){
		QUOTA_DEV *d = ctl->devs.getitem(i);
		QUOTA_SPEC *s = specs.getitem(i);
		s->name.setfrom (nam);
		QUOTA_SPEC old_eff;		// Current effective state of the quota
								// spec for this user or group
		geteffspec (d,old_eff);
		bool empty = s->soft_maxk == -1
				&& s->soft_maxf == -1
				&& s->hard_maxk == -1
				&& s->hard_maxf == -1;
		QUOTA_SPEC *u = getspec(d);
		if (u == NULL){
			if (!empty){
				QUOTA_SPEC *ns = new QUOTA_SPEC(s);
				if (type == QUOTA_USER){
					d->users.add (ns);
				}else if (type == QUOTA_GROUP){
					d->groups.add (ns);
				}else if (type == QUOTA_USERDEF_G){
					d->userdef_g.add (ns);
				}
			}
		}else{
			if (empty){
				if (type == QUOTA_USER){
					d->deluserspec (nam);
				}else if (type == QUOTA_GROUP){
					d->delgroupspec (nam);
				}else if (type == QUOTA_USERDEF_G){
					d->deluserdef_gspec (nam);
				}
			}else{
				u->setfrom (s);
			}
		}
		QUOTA_SPEC new_eff;
		geteffspec (d,new_eff);

		if (is_new || old_eff.isdiff (new_eff)){
			if (d->setrealdevice()!=-1){
				const char *device = d->realdev.get();
				if (type == QUOTA_USER){
					struct passwd *p = getpwnam (nam);
					if (p != NULL){
						quota_apply (new_eff,p->pw_uid,USRQUOTA,device);
					}
				}else if (type == QUOTA_GROUP){
					struct group *g = getgrnam (nam);
					if (g != NULL){
						quota_apply (new_eff,g->gr_gid,GRPQUOTA,device);
					}
				}else if (type == QUOTA_USERDEF_G){
					// We have to walk all member of the group and maybe
					// update their quota
					struct group *g = getgrnam (nam);
					if (g != NULL){
						gid_t gid = g->gr_gid;
						setpwent();
						struct passwd *p;
						while ((p=getpwent())!=NULL){
							if (p->pw_gid == gid){
								QUOTA_SPEC eff;
								d->geteffuserspec (p->pw_name,nam,eff);
								quota_apply (eff,p->pw_uid,USRQUOTA,device);
							}
						}
					}
				}
			}
		}
	}
	return ctl->write(priv);
}

/*
	Add the different fields in the dialog for that user or group
*/
PUBLIC void QUOTA_EDIT::setupdia (DIALOG &dia, const char *maintitle)
{
	int n = specs.getnb();
	if (n > 0){
		dia.newf_title (maintitle,1,"",maintitle);
		for (int i=0; i<n; i++){
			QUOTA_SPEC *s = specs.getitem(i);
			QUOTA_DEV *d = ctl->devs.getitem(i);
			const char *title = d->device.get();
			dia.newf_title (title,2,"",title);
			// dia.newf_title ("",title);
			static const char *tbvals[]={
				MSG_U(I_DEFAULT,"Default"),
				MSG_R(I_NOLIMIT),
				NULL
			};
			static int vals[]={-1,0};
			dia.newf_chkm_num (MSG_R(F_SOFTK),s->soft_maxk,vals,tbvals);
			dia.newf_chkm_num (MSG_R(F_HARDK),s->hard_maxk,vals,tbvals);
			dia.newf_str (MSG_R(F_GRACEK),s->grace_k);

			dia.newf_title ("","");
			dia.newf_chkm_num (MSG_R(F_SOFTF),s->soft_maxf,vals,tbvals);
			dia.newf_chkm_num (MSG_R(F_HARDF),s->hard_maxf,vals,tbvals);
			dia.newf_str (MSG_R(F_GRACEF),s->grace_f);
		}
	}
}

PUBLIC QUOTA_DEV *QUOTACTL::getdev (const char *device)
{
	QUOTA_DEV *ret = NULL;
	for (int i=0; i<devs.getnb(); i++){
		QUOTA_DEV *dev = devs.getitem(i);
		if (dev->device.cmp(device)==0){
			ret = dev;
			break;
		}
	}
	return ret;
}

static void quota_setif (const char *val, int &field)
{
	if (val != NULL){
		if (strcmp(val,"default")==0){
			field = -1;
		}else if(!isdigit(val[0])){
			fprintf (stderr
				,MSG_U(E_IVLDQUOTAVAL
					,"Invalid quota value %s: expect default or a numeric value\n")
				,val);
		}else{
			field = atoi(val);
		}
	}
}

/*
	Command line support to set disk quota
*/
int quota_setquota (
	bool is_group,		// Update group or user quota
	const char *name,	// Group or user name
	int argc,
	char *argv[])
{
	int ret = -1;
	bool some_errors = false;
	const char *device = NULL;
	const char *softk = NULL;
	const char *hardk = NULL;
	const char *softf = NULL;
	const char *hardf = NULL;
	int i;
	for (i=0; i<argc-1; i+=2){
		const char *opt = argv[i];
		const char *arg = argv[i+1];
		if (strcmp(opt,"--device")==0){
			device = arg;
		}else if (strcmp(opt,"--softk")==0){
			softk = arg;
		}else if (strcmp(opt,"--hardk")==0){
			hardk = arg;
		}else if (strcmp(opt,"--softf")==0){
			softf = arg;
		}else if (strcmp(opt,"--hardf")==0){
			hardf = arg;
		}else{
			fprintf (stderr,MSG_U(E_IVLDQUOTAOPT,"Invalid setquota option %s\n")
				,opt);
			some_errors = true;
			break;
		}
	}
	if (i!=argc){
		if (!some_errors){
			fprintf (stderr,MSG_R(E_IVLDQUOTAOPT),argv[i]);
		}
	}else{
		if (device == NULL){
			fprintf (stderr,MSG_U(E_MISSDEVICE,"Missing --device option\n"));
		}else{
			QUOTACTL ctl;
			QUOTA_DEV *dev = ctl.getdev(device);
			if (device == NULL){
				fprintf (stderr,MSG_U(E_NOQUOTADEV
					,"Quota not available for device %s\n"),device);
			}else{
				QUOTA_SPECS *specs = is_group ? &dev->groups : &dev->users;
				QUOTA_SPEC *spec = specs->getitem(name);
				if (spec == NULL){
					spec = new QUOTA_SPEC;
					spec->name.setfrom (name);
					specs->add (spec);
				}
				quota_setif (softk,spec->soft_maxk);
				quota_setif (hardk,spec->hard_maxk);
				quota_setif (softf,spec->soft_maxf);
				quota_setif (hardf,spec->hard_maxf);
				ret = ctl.write(NULL);
				if (ret == 0) quota_applyall();
			}
		}
	}
	return ret;
	return ret;
}

