/*
 * $Id: ntfsd.c,v 1.7 2001/12/15 05:13:08 antona Exp $
 * 
 * ntfsd.c - Writer daemon. Part of the Linux-NTFS project.
 *
 * Copyright (c) 2001 Anton Altaparmakov.
 *
 * This program/include file is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as published
 * by the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program/include file is distributed in the hope that it will be 
 * useful, but WITHOUT ANY WARRANTY; without even the implied warranty 
 * of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program (in the main directory of the Linux-NTFS 
 * distribution in the file COPYING); if not, write to the Free Software
 * Foundation,Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <sys/time.h>
#include <signal.h>
#include <string.h>
#include <stdio.h>
#include <errno.h>
#include <sched.h>

#include "volume.h"
#include "ntfsd.h"

/*
 * Keep track of mounted volumes so we can walk the list and sync them all.
 */
LIST_HEAD(ntfs_mounted_volumes);
unsigned long ntfs_nr_mounted_volumes = 0;

/*
 * Private global flags for NTFS.
 *
 * NF_need_sync bit is set by ntfsd every 5 seconds and cleared by ntfs_flush(),
 * when it is called.
 * 
 * NF_ntfsd_running is set by ntfs_flush() when it initiates the sync to disk
 * and cleared when it finishes. This prevents reentry and loops where
 * ntfs_flush() will call a function which will have an ntfs_flush() call in
 * its code path.
 */
static unsigned long ntfs_flags = 0;

#define NF_need_sync		0
#define NF_ntfsd_running	1
				/* bits 2-31 reserved for future use */

#define NtfsNeedSync(n)		test_bit(NF_need_sync, (n))
#define SetNtfsNeedSync(n)	set_bit(NF_need_sync, (n))
#define ClearNtfsNeedSync(n)	clear_bit(NF_need_sync, (n))

#define NtfsdRunning(n)		test_bit(NF_ntfsd_running, (n))
#define SetNtfsdRunning(n)	set_bit(NF_ntfsd_running, (n))
#define ClearNtfsdRunning(n)	clear_bit(NF_ntfsd_running, (n))

/*
 * Keep track of how many times the timer has been started.
 */
static unsigned long ntfsd_times_started = 0;

/*
 * We need this global to preserve the old sigalrm_action between calls to
 * ntfs_timer_start() and ntfs_timer_stop().
 */
static struct sigaction ntfsd_old_sigalrm_action;

/* Convenience strings. */
static const char *ntfsd_internal_error = "Internal error.";
static const char *ntfsd_sigalrm_not_supported = "This system does not support "
						 "the SIGALRM signal.";

static void ntfs_flush_all_volumes(void)
{
	struct list_head *tmp;
	ntfs_volume *vol;

	if (!ntfs_nr_mounted_volumes)
		return;
	list_for_each(tmp, &ntfs_mounted_volumes) {
		vol = list_entry(tmp, ntfs_volume, v_list);
		if (ntfs_sync_volume(vol) < 0) {
#ifdef DEBUG
			fprintf(stderr, "DEBUG Linux-NTFS: ntfs_sync_volume() "
					"failed for volume %s (%s): %s",
					vol->vol_name, vol->dev_name,
					strerror(errno));
#endif
		}
	}
}

__inline__ void ntfs_flush(void)
{
	if (test_and_clear_bit(NF_need_sync, ntfs_flags) &&
	    !test_and_set_bit(NF_ntfsd_running, ntfs_flags)) {
		ntfs_flush_all_volumes();
		ClearNtfsdRunning(ntfs_flags);
	}
}

/*
 * The signal handler which sets ntfs_need_sync to true every time it is
 * called. (Every 5 seconds. Only if it is started and there are some mounted
 * volumes.)
 */
static void ntfsd_thread(int i)
{
	if (ntfsd_times_started && ntfs_nr_mounted_volumes)
		SetNtfsNeedSync(ntfs_flags);
}
	
BOOL ntfs_timer_start(void)
{	
	int tries, err;
	const char *ntfsd_start_error = "Linux-NTFS: failed to start ntfsd.";
	struct sigaction action;
	struct itimerval timer_val;

	if (ntfsd_times_started + 1) {
		fprintf(stderr, "Linux-NTFS: ntfsd started counter "
			       "would overflow. Not starting.\n");
		err = EMFILE;
		goto ret_false;
	}
	if (++ntfsd_times_started > 1)
		return TRUE;
	memset(&ntfsd_old_sigalrm_action, 0, sizeof(struct sigaction));
	tries = 0;
retry_get_action:
	if (sigaction(SIGALRM, NULL, &action)) {
		switch (err = errno) {
		case EINTR:
			if (tries++ < 2)
				goto retry_get_action;
			perror(ntfsd_start_error);
			break;
		case EINVAL:
			fprintf(stderr, "%s %s\n", ntfsd_start_error,
					ntfsd_sigalrm_not_supported);
			err = ENOTSUP;
			break;
		case EFAULT:
			fprintf(stderr, "%s %s\n", ntfsd_start_error,
					ntfsd_internal_error);
			break;
		default:
			perror(ntfsd_start_error);
			err = EFAULT;
		}
		goto ret_false;
	}
	if (action.sa_handler == &ntfsd_thread)
		goto set_timer_only;
	memset(&action, 0, sizeof(struct sigaction));
	tries = 0;
retry_set_action:
	action.sa_handler = &ntfsd_thread;
	if (sigaction(SIGALRM, &action, &ntfsd_old_sigalrm_action)) {
		switch (err = errno) {
		case EINTR:
			if (tries++ < 2)
				goto retry_set_action;
			perror(ntfsd_start_error);
			break;
		case EINVAL:
			fprintf(stderr, "%s %s\n", ntfsd_start_error,
					ntfsd_sigalrm_not_supported);
			err = ENOTSUP;
			break;
		case EFAULT:
			fprintf(stderr, "%s %s\n", ntfsd_start_error,
					ntfsd_internal_error);
			break;
		default:
			perror(ntfsd_start_error);
			err = EFAULT;
		}
		goto ret_false;
	}
set_timer_only:
	memset(&timer_val, 0, sizeof(struct itimerval));
	timer_val.it_interval.tv_sec = 5;
	timer_val.it_value.tv_sec = 5;
	if (!setitimer(ITIMER_REAL, &timer_val, NULL))
		return TRUE;
	switch (errno) {
	case EINVAL:
	case EFAULT:
		fprintf(stderr, "%s %s\n", ntfsd_start_error,
				ntfsd_internal_error);
		break;
	default:
		perror(ntfsd_start_error);
	}
	err = EFAULT;
	tries = 0;
retry_reset_action:
	if (sigaction(SIGALRM, &ntfsd_old_sigalrm_action, NULL)) {
		if (errno == EINTR && tries++ < 2)
			goto retry_reset_action;
		err = ECANCELED;
	}
ret_false:
	ntfsd_times_started--;
	errno = err;
	return FALSE;
}

BOOL ntfs_timer_stop(void)
{
	int tries, err;
	const char *ntfs_timer_stop_error = "Linux-NTFS: failed to stop timer.";
	struct itimerval timer_val;

	if (ntfsd_times_started > 0 && --ntfsd_times_started)
		return TRUE;
	ntfsd_times_started = 0;
	memset(&timer_val, 0, sizeof(struct itimerval));
	timer_val.it_interval.tv_sec = 0;
	timer_val.it_value.tv_sec = 0;
	if (setitimer(ITIMER_REAL, &timer_val, NULL)) {
		switch (errno) {
		case EFAULT:
		case EINVAL:
			fprintf(stderr, "%s %s\n", ntfs_timer_stop_error,
					ntfsd_internal_error);
			break;
		default:
			perror(ntfs_timer_stop_error);
		}
		err = EFAULT;
		ntfsd_times_started++;
		goto ret_false;
	}
	tries = 0;
retry_set_action:
	if (!sigaction(SIGALRM, &ntfsd_old_sigalrm_action, NULL))
		return TRUE;
	switch (err = errno) {
	case EINTR:
		if (tries++ < 2)
			goto retry_set_action;
		perror(ntfs_timer_stop_error);
		break;
	case EINVAL:
		fprintf(stderr, "%s %s\n", ntfs_timer_stop_error,
				ntfsd_sigalrm_not_supported);
		err = ENOTSUP;
		break;
	case EFAULT:
		fprintf(stderr, "%s %s\n", ntfs_timer_stop_error,
				ntfsd_internal_error);
		break;
	default:
		perror(ntfs_timer_stop_error);
		err = EFAULT;
	}
	timer_val.it_interval.tv_sec = 5;
	timer_val.it_value.tv_sec = 5;
	if (setitimer(ITIMER_REAL, &timer_val, NULL)) {
		err = ECANCELED;
		goto ret_false;
	}
ret_false:
	errno = err;
	return FALSE;
}

