/*
 * $Id: volume.h,v 1.15 2001/12/06 01:14:52 antona Exp $
 *
 * volume.h - Exports for NTFS volume handling. Part of the Linux-NTFS project.
 *
 * Copyright (c) 2000,2001 Anton Altaparmakov.
 *
 * This program/include file is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as published
 * by the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program/include file is distributed in the hope that it will be 
 * useful, but WITHOUT ANY WARRANTY; without even the implied warranty 
 * of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program (in the main directory of the Linux-NTFS 
 * distribution in the file COPYING); if not, write to the Free Software
 * Foundation,Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#ifndef VOLUME_H
#define VOLUME_H

#include <stdio.h>
#include <sys/mount.h>

#include "list.h"
#include "support.h"
#include "types.h"

int ntfs_sync_volume(ntfs_volume *vol);

/**
 * ntfs_mount - open ntfs volume
 * @name:	name of device/file to open
 * @rwflag:	optional mount flags
 *
 * This function mounts an ntfs volume. @name should contain the name of the
 * device/file to mount as the ntfs volume.
 *
 * @rwflags is an optional second parameter. The same flags are used as for
 * the mount system call (man 2 mount). Currently only the following flags
 * are implemented:
 *	MS_RDONLY	- mount volume read-only
 * 
 * The function opens the device or file @name and verifies that it contains a
 * valid bootsector. Then, it allocates an ntfs_volume structure and initializes
 * some of the values inside the structure from the information stored in the
 * bootsector. It proceeds to load the necessary system files and completes
 * setting up the structure. 
 *
 * ntfs_mount() returns a pointer to the allocated ntfs_volume structure.
 * On error, the return value is NULL and errno is set appropriately.
 *
 * Note, that a copy is made of @name, and hence it can be discarded as
 * soon as the function returns.
 *
 * FIXME: Should document all error codes that can be returned and what each
 * means.
 */
ntfs_volume *ntfs_mount(const char *name, unsigned long rwflag);

/**
 * ntfs_umount - close ntfs volume
 * @vol: address of ntfs_volume structure of volume to close
 * @force: if true force close the volume even if it is busy
 * 
 * Flush buffers, close all files and deallocate all structures (including @vol
 * itself) associated with the ntfs volume @vol.
 *
 * Return true on success. On error return false with errno set appropriately
 * (most likely to one of EAGAIN, EBUSY or EINVAL). The EAGAIN error means that
 * an operation is in progress and if you try the close later the operation
 * might be completed and the close succeed.
 *  
 * If @force is true (i.e. not zero) this function will close the volume even
 * if this means that data might be lost. In this case the function always
 * returns 1.
 *
 * @vol must have previously been returned by a call to ntfs_mount().
 *
 * @vol itself is deallocated and should no longer be referenced after this
 * function returns success. If it returns an error then nothing has been done
 * so it is safe to continue using @vol.
 *
 * FIXME: Should document all error codes that can be returned and what each
 * means.
 */
BOOL ntfs_umount(ntfs_volume *vol, const int force);

#endif /* defined VOLUME_H */

