#ifndef _KVI_SCRIPT_OBJECT_H_INCLUDED_
#define _KVI_SCRIPT_OBJECT_H_INCLUDED_

// =============================================================================
//
//   This file is part of the KVIrc IRC client distribution
//   Copyright (C) 1999-2000 Szymon Stefanek (stefanek@tin.it)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc, 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
// =============================================================================

#include <qasciidict.h>
#include <qobject.h>
#include <qptrlist.h>

#include "kvi_string.h"

class KviScriptObject;
class KviScriptObjectClassDefinition;
class KviScriptObjectController;
class KviVariableCache;
class KviWindow;

typedef struct KviScriptObjectSlot
{
	KviScriptObject *pObject;     // Destination object
	KviStr           szFunction;  // Slot function name
};

typedef QPtrList<KviScriptObjectSlot> KviScriptObjectSlotList;

typedef struct KviScriptObjectSignal
{
	KviStr           szSignalName;
	KviStr           szSlotName;
	KviScriptObject *pObject;
};

/**
 * A script object : base class
 */
class KviScriptObject : public QObject
{
	Q_OBJECT
public:
	KviScriptObject(KviScriptObjectController *, KviScriptObject *, const char *name, KviScriptObjectClassDefinition *);
	~KviScriptObject();
private:
	KviStr                               m_szId;              // Unique ID string (@id)
	QPtrList<KviScriptObject>           *m_pChildList;        // List of children objects
	KviScriptObjectController           *m_pController;       // Object controller (unique for each frame)
	KviVariableCache                    *m_pVarCache;         // Object member variables
	QAsciiDict<KviStr>                  *m_pEventDict;        // Object event handlers
	QPtrList<KviStr>                    *m_pInheritanceList;  // List of inherited classes
	KviScriptObjectClassDefinition      *m_pClassDefinition;  // Pointer to the global class definition
	bool                                 m_bInDelayedDestroy; // In DelayedDestroy state!
	bool                                 m_bDying;            // In the destructor?
	bool                                 m_bNeedRecreate;     // Need to rise from the ashes after being deleted?

	QAsciiDict<KviScriptObjectSlotList> *m_pSlotDict;         // Dictionary of slots connected to the signals of
	                                                          // this object (key = signal name)
	QPtrList<KviScriptObjectSignal>     *m_pSignalList;
private:
	void     registerSlot(const char *signalName, const char *slotName, KviScriptObject *object);
	void   unregisterSlot(const char *signalName, const char *slotName, KviScriptObject *object);
	void   registerSignal(const char *signalName, const char *slotName, KviScriptObject *object);
	void unregisterSignal(const char *signalName, const char *slotName, KviScriptObject *object);
public:
	static void connectSignalSlot(KviScriptObject *src, const char *signalName, KviScriptObject *dst, const char *slotName);
	static void disconnectSignalSlot(KviScriptObject *src, const char *signalName, KviScriptObject *dst, const char *slotName);
public:
	KviScriptObjectController *controller();

	void die(bool bDelayed);
	void dieAndRecreate(bool bDelayed);
	/** Sets m_bInDelayedDestroy to true. */
	void dieOutOfThisEventStep();
	bool inDelayedDestroy();

	KviVariableCache       *varCache();
	KviScriptObject        *parentObject();
	QPtrList<KviScriptObject> *childrenList();
	KviScriptObject        *childByName(const char *szName);

	const char *getName();
	const char *getClass();
	const char *id();

	KviScriptObjectClassDefinition *classDefinition();

	KviScriptObject *findObjectByName(const char *szName);
	KviScriptObject *findObjectByClass(const char *szClass);
	KviScriptObject *findObjectById(const char *szId);

	void triggerEvent(const char *evName, const KviStr &parms);
	bool emitSignal(const char *sigName, QPtrList<KviStr> *params, KviStr &buffer);
	void removeEventHandler(const char *evName);
	void setEventHandler(const char *evName, const char *evBuffer);
	bool hasEventHandler(const char *evName);
	int  callFunction(const char *fncName, QPtrList<KviStr> *params, KviStr &buffer, const char *classOverride = 0);
	void dump(KviWindow *wnd, const char *margin = 0);

	static void initializeClassDefinition(KviScriptObjectClassDefinition *);

	int               builtinFunction_NAME(QPtrList<KviStr> *params, KviStr &buffer);
	int              builtinFunction_CLASS(QPtrList<KviStr> *params, KviStr &buffer);
	int             builtinFunction_PARENT(QPtrList<KviStr> *params, KviStr &buffer);
	int           builtinFunction_INHERITS(QPtrList<KviStr> *params, KviStr &buffer);
	int   builtinFunction_INHERITEDCLASSES(QPtrList<KviStr> *params, KviStr &buffer);
	int           builtinFunction_CHILDREN(QPtrList<KviStr> *params, KviStr &buffer);
	int    builtinFunction_FINDDIRECTCHILD(QPtrList<KviStr> *params, KviStr &buffer);
	int          builtinFunction_FINDCHILD(QPtrList<KviStr> *params, KviStr &buffer);
	int               builtinFunction_EMIT(QPtrList<KviStr> *params, KviStr &buffer);

private:
	void addChild(KviScriptObject *);
	void removeChild(KviScriptObject *, bool bDelete = true);
private slots:
	virtual void delayedDie();
};

#endif // _KVI_SCRIPT_OBJECT_H_INCLUDED_
