// =============================================================================
//
//   This file is part of the KVIrc IRC client distribution
//   Copyright (C) 1999-2000 Szymon Stefanek (stefanek@tin.it)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc, 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
// =============================================================================

#define _KVI_DEBUG_CHECK_RANGE_
#define _KVI_DEBUG_CLASS_NAME_ "KviScriptComboBox"

#include "kvi_combobox.h"
#include "kvi_error.h"
#include "kvi_script_combobox.h"
#include "kvi_script_objectclassdefinition.h"

/*
	@class: combobox
	@short:
		A handy selection widget
	@inherits:
		[class]object[/class]<br>
		[class]widget[/class]
	@functions:
		!fn: $insertItem(&lt;text&gt;, &lt;index&gt)
		Inserts a text item at the specified index.
		!fn: $clear()
		Clears the combobox.
		!fn: $currentText()
		Returns the text of currently selected item.
		!fn: $text(&lt;index&gt;)
		Returns the text of an item at given index.
		!fn: $removeItem(&lt;index&gt;)
		Removes item at given index.

	@description:
		This widget is a great widget to display or get user selection.
		It is somewhat similar to a button combined with a popup menu:
		when user clicks a button placed on the right edge of the
		combobox, a popup menu with availible options/selections will
		pop up.

	@examples:
		Let's say you have a script that can run in two modes: Normal
		and Advanced. One of the way to show current mode, or to let
		user change it is putting a combobox in your script's config
		dialog:
		<example>
		# we assume that %my_cfg_dlg is a widget (config dialog)<br>
		%cb = $new(combobox, %my_cfg_dlg);
		%cb->$insertItem("Normal");
		%cb->$insertItem("Advanced");
		# now after user makes a choice you can obtain it like this:<br>
		%sMode = %cb->$currentText;
		echo Current script mode is: %sMode;
		</example>


	@seealso:
		class [class]object[/class], <br>
		class [class]widget[/class], <br>
		<a href="syntax_objects.kvihelp">Objects documentation</a><br>
*/

/**
 * COMBOBOX class
 */
void KviScriptComboBox::initializeClassDefinition(KviScriptObjectClassDefinition *d)
{
	d->addBuiltinFunction("insertItem",  (scriptObjectFunction) &KviScriptComboBox::builtinFunction_INSERTITEM);
	d->addBuiltinFunction("clear",       (scriptObjectFunction) &KviScriptComboBox::builtinFunction_CLEAR);
	d->addBuiltinFunction("currentText", (scriptObjectFunction) &KviScriptComboBox::builtinFunction_CURRENTTEXT);
	d->addBuiltinFunction("text",        (scriptObjectFunction) &KviScriptComboBox::builtinFunction_TEXT);
	d->addBuiltinFunction("removeItem",  (scriptObjectFunction) &KviScriptComboBox::builtinFunction_REMOVEITEM);
}

KviScriptComboBox::KviScriptComboBox(
	KviScriptObjectController *cntrl, KviScriptObject *p, const char *name, KviScriptObjectClassDefinition *pDef)
	: KviScriptWidget(cntrl, p, name, pDef)
{
	// Nothing here
}

KviScriptComboBox::~KviScriptComboBox()
{
	// Nothing here
}

bool KviScriptComboBox::init(QPtrList<KviStr> *params)
{
	if( parent() ) {
		if( parent()->inherits("KviScriptWidget") ) {
			m_pWidget = new KviComboBox(((KviScriptWidget *) parent())-> m_pWidget, name());
		}
	}

	if( !m_pWidget )
		m_pWidget = new KviComboBox(0, name());
	m_bAutoDestroyControlledWidget = true;
	m_pWidget->installEventFilter(this);
	connect(m_pWidget, SIGNAL(destroyed()),                this,      SLOT(widgetDestroyed()));
	connect(m_pWidget, SIGNAL(activated(const QString &)), m_pWidget, SLOT(__slotActivated(const QString &)));
	return true;
}

int KviScriptComboBox::builtinFunction_INSERTITEM(QPtrList<KviStr> *params, KviStr &buffer)
{
	if( params ) {
		KviStr *pS = params->first(); // Text
		int val = -1;

		bool bOk;
		if( params->count() > 1 ) {
			KviStr *pS2 = params->at(1); // Index
			val = pS2->toInt(&bOk);
			if( !bOk )
				return KVI_ERROR_InvalidParameter;
		}
		((KviComboBox *) m_pWidget)->insertItem(pS->ptr(), val);
		return KVI_ERROR_Success;
	}
	return KVI_ERROR_MissingParameter;
}

int KviScriptComboBox::builtinFunction_CLEAR(QPtrList<KviStr> *, KviStr &)
{
	((KviComboBox *) m_pWidget)->clear();
	return KVI_ERROR_Success;
}

int KviScriptComboBox::builtinFunction_CURRENTTEXT(QPtrList<KviStr> *, KviStr &buffer)
{
	buffer.append(((KviComboBox *) m_pWidget)->currentText());
	return KVI_ERROR_Success;
}

int KviScriptComboBox::builtinFunction_TEXT(QPtrList<KviStr> *params, KviStr &buffer)
{
	if( !params )
		return KVI_ERROR_MissingParameter;

	KviStr *pS = params->first();
	if( !pS )
		return KVI_ERROR_MissingParameter;

	bool bOk;
	int val = pS->toInt(&bOk);
	if( !bOk )
		return KVI_ERROR_InvalidParameter;
	buffer.append(((KviComboBox *) m_pWidget)->text(val));
	return KVI_ERROR_Success;
}

int KviScriptComboBox::builtinFunction_REMOVEITEM(QPtrList<KviStr> *params, KviStr &buffer)
{
	if( params ) {
		KviStr *pS = params->first();
		if( pS ) {
			bool bOk;
			int val = pS->toInt(&bOk);
			if( !bOk )
				return KVI_ERROR_InvalidParameter;
			((KviComboBox *) m_pWidget)->removeItem(val);
			return KVI_ERROR_Success;
		}
	}
	return KVI_ERROR_MissingParameter;
}

void KviScriptComboBox::__slotActivated(const QString &str)
{
	KviStr parms(KviStr::Format, "%s", (const char *) str.utf8());
	triggerEvent("OnActivated", parms);
}

#include "m_kvi_script_combobox.moc"
