// =============================================================================
//
//      --- kvi_raweventeditor.cpp ---
//
//   This file is part of the KVIrc IRC client distribution
//   Copyright (C) 1999-2000 Szymon Stefanek (stefanek@tin.it)
//   Copyright (C) 1999-2000 Till Busch (buti@geocities.com)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
// =============================================================================

#define _KVI_DEBUG_CHECK_RANGE_
#define _KVI_DEBUG_CLASS_NAME_ "KviRawEventEditor"

#include <qsplitter.h>
#include <qvbox.h>

#include "kvi_listview.h"
#include "kvi_locale.h"
#include "kvi_messagebox.h"
#include "kvi_rawevent.h"
#include "kvi_raweventeditor.h"
#include "kvi_pushbutton.h"
#include "kvi_script_datalistview.h"
#include "kvi_script_editor.h"

// TODO: Totally missing quick help!

// Declared in kvi_app.cpp:
extern KviRawEventManager *g_pRawEventManager;

KviStr g_szLastSelectedRawEvent = "";

/*
	@quickhelp: KviRawEventEditor
	@widget: Raw event editor
		Here, you can edit the <a href="syntax_events.kvihelp">raw event handlers</a>.<br>
		The left list view displays the list of the currently defined handlers:<br>
		the left column in that list displays the raw server command that the handler
		reacts to and the right column shows the source mask for that handler.<br>
		By clicking on "Add Handler, " you will add an empty handler for the UNKNOWN literal reply
		with a full wildcard source mask.<br>
		Double-click on the left column of the newly added item to edit the command name.<br>
		You can insert a numeric value from 0 to 999, or a literal command such as 'WALLOPS' or 'KICK'.<br>
		Double-click on the right column of the item to edit the source mask.<br>
		The source mask is a string that can contain '*' wildcards: something as
		pragma!*@* or *.it (remember that the source of a server message may be either a user mask
		or a server name; some messages may also arrive from IRC services).<br>
		The single "full" wildcard ('*') matches all the sources.<br>
		In the source editor on the right you can edit your handler for the message.<br>
		For example, if you want to filter all privmsgs that arrive from the server:<br>
		- Click on "Add Handler"<br>
		- Double-click on the string, 'UNKNOWN', that just appeared in the list view.<br>
		- Change 'UNKNOWN' to 'PRIVMSG'
		- Leave the source mask as is ('*') so that it will match all sources.<br>
		- Edit your handler.<br>
		If you want to react only to privmsgs sent by users from France:<br>
		- Double-click on the "Source Mask" column of the last added item<br>
		- Change '*' to '*.fr'
		More handlers for the same command are accepted, and will be triggered in an
		arbitrary (unspecified) order.<br>
		See also the <a href="syntax_events.kvihelp">event scripting syntax documentation</a>.<br>
*/

/**
 *
 * RAW EVENTS EDITOR.
 *
 * This widget allows editing of the raw events.
 *
 */
KviRawEventEditor::KviRawEventEditor(QWidget *parent, const char *name)
	: QWidget(parent, name)
{
	m_pCurItem  = 0;
	m_pSplitter = new QSplitter(this);
	m_pSplitter->setOpaqueResize();

	QVBox *box  = new QVBox(m_pSplitter);
	m_pListView = new KviListView(box, "raw_commands", true);
	m_pListView->addColumn(_i18n_("Raw Command"));
	m_pListView->addColumn(_i18n_("Source Mask (RegExp)"));
	m_pNew    = new KviPushButton(_i18n_("&Add Handler"), box);
	m_pDelete = new KviPushButton(_i18n_("&Remove Handler"), box);
	m_pDelete->setEnabled(false);
	m_pClear  = new KviPushButton(_i18n_("&Clear Handlers"), box);

	m_pScriptEditor = new KviScriptEditor(m_pSplitter);
	m_pScriptEditor->setEnabled(false);

	fillEventListView();
	connect(m_pListView,
		SIGNAL(selectionChanged(KviListViewItem *)), this, SLOT(listViewSelectionChanged(KviListViewItem *))
	);
	connect(m_pNew,    SIGNAL(clicked()), this, SLOT(makeNewEvent()));
	connect(m_pDelete, SIGNAL(clicked()), this, SLOT(deleteCurEvent()));
	connect(m_pClear,  SIGNAL(clicked()), this, SLOT(clearAllEvents()));

	KviScriptDataListViewItem *it = findEventItem(g_szLastSelectedRawEvent.ptr());
	if( it )
		m_pListView->setSelected(it, true);
}

KviRawEventEditor::~KviRawEventEditor()
{
	// Nothing here
}

void KviRawEventEditor::makeNewEvent()
{
	KviScriptDataListViewItem *it = new KviScriptDataListViewItem(m_pListView, "", _i18n_("UNKNOWN"), "*");
	m_pListView->setSelected(it, true);
	m_pScriptEditor->setFocus();
}

void KviRawEventEditor::deleteCurEvent()
{
	if( !m_pCurItem ) return;

	KviScriptDataListViewItem *it = m_pCurItem;
	m_pCurItem = 0;
	delete it;
	m_pScriptEditor->setText(0);
	it = (KviScriptDataListViewItem *) m_pListView->currentItem();
	if( it )
		m_pListView->setSelected(it, true);
}

void KviRawEventEditor::clearAllEvents()
{
	if( KviMessageBox::warningYesNo(
		_i18n_("Are you sure you want to remove all raw event handlers?"),
		_i18n_("Clear Handlers"),
		this) == KviMessageBox::Yes
	) {
		m_pCurItem = 0;
		m_pListView->clear();
	}
}

void KviRawEventEditor::saveCurrentItem()
{
	if( !m_pCurItem )return;
	m_pCurItem->m_szBuffer = m_pScriptEditor->text();
}

KviScriptDataListViewItem *KviRawEventEditor::findEventItem(const char *item_key)
{
	for( KviScriptDataListViewItem *it = (KviScriptDataListViewItem *) m_pListView->firstChild()
	   ; it
	   ; it = (KviScriptDataListViewItem *) it->nextSibling()
	) {
		KviStr tmp;
		it->makeKey(tmp);
		if( kvi_strEqualCI(tmp.ptr(), item_key) )
			return it;
	}
	return 0;
}

void KviRawEventEditor::commit()
{
	if( m_pCurItem )
		m_pCurItem->makeKey(g_szLastSelectedRawEvent);
	saveCurrentItem();
	g_pRawEventManager->cleanup();
	for( KviScriptDataListViewItem *it = (KviScriptDataListViewItem *) m_pListView->firstChild()
	   ; it
	   ; it = (KviScriptDataListViewItem *) it->nextSibling()
	) {
		g_pRawEventManager->addHandler(it->text(0).ascii(), it->text(1).ascii(), it->m_szBuffer.ptr());
	}
}

void KviRawEventEditor::listViewSelectionChanged(KviListViewItem *it)
{
	saveCurrentItem();
	m_pCurItem = 0;
	if( it ) {
		m_pScriptEditor->setText(((KviScriptDataListViewItem *) it)->m_szBuffer.ptr());
		m_pScriptEditor->setFocus();
	} else {
		m_pScriptEditor->setText("");
	}
	m_pCurItem = (KviScriptDataListViewItem *) it;
	m_pScriptEditor->setEnabled(it);
	m_pDelete->setEnabled(it);
}

void KviRawEventEditor::fillEventListView()
{
	KviScriptDataListViewItem *item = 0;

	for( int i = 0; i < 1000; i++ ) {
		if( g_pRawEventManager->numericEvent(i) ) {
			for( KviRawEvent *e = g_pRawEventManager->numericEvent(i)->first()
			   ; e
			   ; e = g_pRawEventManager->numericEvent(i)->next()
			) {
				KviStr tmp(KviStr::Format, "%d", i);
				item = new KviScriptDataListViewItem(m_pListView, e->szBuffer.ptr(), tmp.ptr(), e->szSource.ptr());
			}
		}
	}

	if( g_pRawEventManager->literalEventsDict() ) {
		QAsciiDictIterator<KviRawEventList> it(*(g_pRawEventManager->literalEventsDict()));
		QPtrList<KviRawEvent> *l = 0;
		while( (l = it.current()) ) {
			for(KviRawEvent *e = l->first(); e; e = l->next() ) {
				item = new KviScriptDataListViewItem(m_pListView, e->szBuffer.ptr(), it.currentKey(), e->szSource.ptr());
			}
			++it;
		}
	}
}

void KviRawEventEditor::resizeEvent(QResizeEvent *)
{
	m_pSplitter->setGeometry(0, 0, width(), height());
}

#include "m_kvi_raweventeditor.moc"
