// =============================================================================
//
//      --- kvi_slaveio.cpp ---
//
//   This file is part of the KVIrc IRC client distribution
//   Copyright (C) 1999-2000 Szymon Stefanek (stefanek@tin.it)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc, 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
// =============================================================================

#define _KVI_DEBUG_CHECK_RANGE_
#define _KVI_DEBUG_CLASS_NAME_ "KviSlaveIO"

#include "kvi_console.h"
#include "kvi_debug.h"
#include "kvi_frame.h"
#include "kvi_locale.h"
#include "kvi_options.h"
#include "kvi_process.h"
#include "kvi_slaveio.h"
#include "kvi_userparser.h"

KviSlaveIO::KviSlaveIO(KviFrame *pFrm, KviUserParser *pParser)
	: QObject()
{
	m_pFrm        = pFrm;
	m_pProcList   = new QPtrList<KviSlaveProcEntry>;
	m_pProcList->setAutoDelete(true);
	m_pUserParser = pParser;
}

KviSlaveIO::~KviSlaveIO()
{
	killAll();
	delete m_pProcList;
	m_pProcList = 0;
}

bool KviSlaveIO::exec(
	const char *commandline, KviWindow *w, bool bExecInSubshell, bool bQuiet, const char *asyncCmd, const char *magic)
{
	KviProcess *p = new KviProcess();
	if( !p->run(commandline, true, true, bExecInSubshell) ) {
		delete p;
		return false;
	}
	KviSlaveProcEntry *e = new KviSlaveProcEntry();
	e->proc            = p;
	e->uPid            = (unsigned long int) p->pid();
	e->commandline     = commandline;
	e->bRunsInSubshell = bExecInSubshell;
	e->bQuiet          = bQuiet;
	e->bAsyncCmd       = asyncCmd != 0;
	e->window          = w->caption();
	if( asyncCmd ) {
		e->asyncCmd = asyncCmd;
		if( magic )
			e->magic = magic;
	}

	if( !bQuiet ) {
		KviWindow *pOut = g_pOptions->m_bProcessOutputToConsole ? m_pFrm->m_pConsole : m_pFrm->activeWindow();
		pOut->output(KVI_OUT_STDIN,
			_i18n_("Process (%s) started: %s %u"), e->commandline.ptr(), bExecInSubshell ? "shell pid" : "pid", e->uPid
		);
	}
	m_pProcList->append(e);
	connect(p, SIGNAL(processExited(KviProcess *)),               this, SLOT(procExited(KviProcess *)));
	connect(p, SIGNAL(receivedStdout(KviProcess *, char *, int)), this, SLOT(procStdout(KviProcess *, char *, int)));
	connect(p, SIGNAL(receivedStderr(KviProcess *, char *, int)), this, SLOT(procStderr(KviProcess *, char *, int)));
	return true;
}

void KviSlaveIO::procExited(KviProcess *proc)
{
	KviSlaveProcEntry *e = findProcByPtr(proc);
	if( !e ){
		debug("Unexpected process termination signal for process %p", proc);
		return;
	}
	__range_valid(e->uPid == (unsigned long int) proc->pid());
	if( !e->bQuiet ) {
		KviWindow *pOut = g_pOptions->m_bProcessOutputToConsole ? m_pFrm->m_pConsole : m_pFrm->activeWindow();
		pOut->output(KVI_OUT_STDIN, _i18n_("Process %u has exited (%s)"), e->uPid, e->commandline.ptr());
	}
	if( e->bAsyncCmd ) {
		int status = e->proc->exitStatus();
		m_pUserParser->slaveAsyncProcessExited(
			e->asyncCmd, e->commandline, e->stdoutBuffer, e->stderrBuffer, e->magic, e->window, e->uPid, status
		);
	}
	killProcess(e->uPid);
}

void KviSlaveIO::procStdout(KviProcess *proc, char *buffer, int buflen)
{
	KviSlaveProcEntry *e = findProcByPtr(proc);
	if( !e ) {
		debug("Unexpected standard output for process %p", proc);
		return;
	}
	__range_valid(e->uPid == (unsigned long int) proc->pid());
	KviWindow *pOut = g_pOptions->m_bProcessOutputToConsole ? m_pFrm->m_pConsole : m_pFrm->activeWindow();
	KviStr tmp(buffer, buflen);
	if( e->bAsyncCmd )
		e->stdoutBuffer.append(tmp);
	else {
		KviStr line;
		while( tmp.getLine(line) )
			pOut->output(KVI_OUT_STDOUT, "[%u]: %s", e->uPid, line.ptr());
	}
}

void KviSlaveIO::procStderr(KviProcess *proc, char *buffer, int buflen)
{
	KviSlaveProcEntry *e = findProcByPtr(proc);
	if( !e ) {
		debug("Unexpected standard error for process %p", proc);
		return;
	}
	__range_valid(e->uPid == (unsigned long int) proc->pid());
	KviWindow *pOut = g_pOptions->m_bProcessOutputToConsole ? m_pFrm->m_pConsole : m_pFrm->activeWindow();
	KviStr tmp(buffer, buflen);
	if( e->bAsyncCmd )
		e->stderrBuffer.append(tmp);
	else {
		KviStr line;
		while( tmp.getLine(line) )
			pOut->output(KVI_OUT_STDERR, "[%u]: %s", e->uPid, line.ptr());
	}
}

KviSlaveProcEntry *KviSlaveIO::findProcByPid(unsigned long int pid)
{
	for( KviSlaveProcEntry *e = m_pProcList->first(); e; e = m_pProcList->next() ) {
		if( e->uPid == pid )
			return e;
	}
	return 0;
}

KviSlaveProcEntry *KviSlaveIO::findProcByPtr(KviProcess *proc)
{
	for( KviSlaveProcEntry *e = m_pProcList->first(); e; e = m_pProcList->next() ) {
		if( e->proc == proc )
			return e;
	}
	return 0;
}

bool KviSlaveIO::killProcess(unsigned long int pid)
{
	KviSlaveProcEntry *p = findProcByPid(pid);
	if( !p ) return false;

	delete p->proc;
	m_pProcList->removeRef(p);
	return true;
}

int KviSlaveIO::killAll()
{
	int killed = 0;
	while( m_pProcList->first() ) {
		killProcess(m_pProcList->first()->uPid);
		killed++;
	}
	return killed;
}

bool KviSlaveIO::writeProcess(KviSlaveProcEntry *e, const char *buffer, int len)
{
	if( len == -1 )
		len = strlen(buffer);
	return e->proc->writeStdin(buffer, len);
}

void KviSlaveIO::outputProcessList(KviWindow *w)
{
	if( !m_pProcList->first() ) {
		w->outputNoFmt(KVI_OUT_STDOUT, _i18n_("No processes running"));
		return;
	}
	int i = 0;
	for( KviSlaveProcEntry *e = m_pProcList->first(); e; e = m_pProcList->next() ) {
		KviStr cmd;
		if( e->bRunsInSubshell ) {
			cmd.append(e->proc->findShell());
			cmd.append(" -c \"");
		}
		cmd.append(e->commandline);
		if( e->bRunsInSubshell )
			cmd.append('"');
		w->output(KVI_OUT_STDIN, _i18n_("\t%d: %s (pid:%u)"), i, cmd.ptr(), e->uPid);
		i++;
	}
	w->output(KVI_OUT_STDIN, _i18n_("Total %d %s running"), i, ((i == 1) ? _i18n_("process") : _i18n_("processes")));
}

#include "m_kvi_slaveio.moc"
