// =============================================================================
//
//      --- kvi_app_setup.cpp ---
//
//   This file is part of the KVIrc IRC client distribution
//   Copyright (C) 1999-2000 Szymon Stefanek (stefanek@tin.it)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
// =============================================================================

#define _KVI_DEBUG_CHECK_RANGE_
#define _KVI_DEBUG_CLASS_NAME_ "KviAppSetup"

#include <stdlib.h>
#include <unistd.h>

#include "kvi_settings.h"

#include <qdir.h>

#ifdef COMPILE_WITH_KDE
	#include <kconfig.h>
	#include <kstddirs.h>
#endif

#include "kvi_app.h"
#include "kvi_defines.h"
#include "kvi_fileutils.h"
#include "kvi_locale.h"
#include "kvi_messagebox.h"

/**
 * Things launched at startup:
 * - Attempt to find the global KVIrc directory
 * - Attempt to find the local KVIrc directory
 *   and if it is not found, ask the user to choose one
 */
bool KviApp::checkGlobalKVIrcDirectory(const char *dir)
{
	// First check if the help subdir exists
	KviStr szDir2(KviStr::Format, "%s/config", dir);
	if( !kvi_directoryExists(szDir2.ptr()) ) return false;
	// Then check if the pics subdir exists
	KviStr szDir(KviStr::Format, "%s/msgcolors", dir);
	if( !kvi_directoryExists(szDir.ptr()) ) return false;
	// Now make sure that it is the dir that we are looking for.
	// Check for an image file that we need.
	szDir.ensureLastCharIs('/');
	szDir += "default.kviclr";
	return kvi_fileIsReadable(szDir.ptr());
}

bool KviApp::checkLocalKVIrcDirectory(const char *dir)
{
	return kvi_directoryIsWritable(dir);
}

const char *usualKVIrcGlobalPrePath[] =
{
	"/usr/local",    "/usr",           "/usr/etc",
	"/usr/X11",      "/opt",           "/opt/kde",
	"/opt/kde2",     "/usr/X11R6",     "/",
	"/lib",          "/lib/X11",       "/etc",
	"/etc/X11",      "/usr/local/etc", "/usr/local/X11",
	"/home",         "/usr/local/kde", "/usr/kde",
	0
};

const char *usualKVIrcGlobalDir[] =
{
	"/share/kvirc2",             "/share/apps/kvirc2",
	"/apps/kvirc2",              "/kvirc2",
	"/kvirc2/share/apps/kvirc2", "/kvirc2/share/kvirc2",
	"/kvirc2/share",              0
};

void KviApp::findGlobalKVIrcDirectory()
{
	// Look for the installation directory
	// with the image and help files inside
#ifdef GLOBAL_KVIRC_DIR
	m_szGlobalKVIrcDir = GLOBAL_KVIRC_DIR;
	if( checkGlobalKVIrcDirectory(m_szGlobalKVIrcDir.ptr()) ) return;
#endif
	// Since I had many problems with it
	// because of strange distributions or KDEDIRS
	// I do it in that way...
#ifdef COMPILE_WITH_KDE
	// KDE compilation...
	// The things usually go installed into $KDEDIR/share/apps/kvirc2
	// Look in the main KDE directory
	KStandardDirs *d = dirs();
	if( d ) {
		m_szGlobalKVIrcDir = locate("appdata", "");
		if( checkGlobalKVIrcDirectory(m_szGlobalKVIrcDir.ptr()) ) return;
		#ifdef HAVE_GETENV
			// KDEDIR sanity check...
			m_szGlobalKVIrcDir  = getenv("KDEDIR");
			m_szGlobalKVIrcDir += "/share/apps/kvirc2";
			if( checkGlobalKVIrcDirectory(m_szGlobalKVIrcDir.ptr()) ) return;
		#endif
	}
	// FAILED? Check the usual way...
#endif // COMPILE_WITH_KDE
	// Non KDE compilation, or not found under $KDEDIR/share/apps/kvirc2
	// Check usual directories...
	for( int j = 0; usualKVIrcGlobalPrePath[j] != 0; j++ ) {
		for( int i = 0; usualKVIrcGlobalDir[i] != 0; i++ ) {
			m_szGlobalKVIrcDir  = usualKVIrcGlobalPrePath[j];
			m_szGlobalKVIrcDir += usualKVIrcGlobalDir[i];
			if( checkGlobalKVIrcDirectory(m_szGlobalKVIrcDir.ptr()) ) return;
		}
	}
	// Last resource, try $HOME and $HOME/.kde
	for( int k = 0; usualKVIrcGlobalDir[k] != 0; k++ ) {
		m_szGlobalKVIrcDir  = QDir::homeDirPath();
		m_szGlobalKVIrcDir += usualKVIrcGlobalDir[k];
		if( checkGlobalKVIrcDirectory(m_szGlobalKVIrcDir.ptr()) ) return;
	}
	for( int k = 0; usualKVIrcGlobalDir[k] != 0; k++ ) {
		m_szGlobalKVIrcDir  = QDir::homeDirPath();
		m_szGlobalKVIrcDir += "./kde";
		m_szGlobalKVIrcDir += usualKVIrcGlobalDir[k];
		if( checkGlobalKVIrcDirectory(m_szGlobalKVIrcDir.ptr()) ) return;
	}

	m_szGlobalKVIrcDir = "";
	// DO NOT TRANSLATE THIS
	// THE TRANSLATION DIRECTORY WAS NOT FOUND YET
	// AND THE LOCALE IS NOT INITIALIZED AT ALL!
#ifdef COMPILE_WITH_KDE
	warningBox("Unable to find the shared KVIrc directory.\n"\
			"The usual path for this directory is $KDEDIR/share/apps/kvirc2.\n"\
			"Are you sure that 'make install' worked correctly?\n"\
			"Please make sure that you have read permission for that directory\n"\
			"and you have set KDEDIR correctly. You may also try to rerun 'make install'.\n"\
			"A detailed explanation of the KVIrc directory system is in the INSTALL document\n"\
			"shipped with the KVIrc source distribution.\n"\
			"Trying to run anyway...");
#else
	warningBox("Unable to find the shared KVIrc directory.\n"\
			"The usual path for this directory is /usr/share/kvirc2.\n"\
			"Are you sure that 'make install' worked correctly ?\n"\
			"Please make sure that you have read permission for that directory.\n"\
			"You may also need to rerun 'make install'.\n"\
			"A detailed explanation of the KVIrc directory system is in the INSTALL document\n"\
			"shipped with the KVIrc source distribution.\n"\
			"Trying to run anyway...\n");
#endif // COMPILE_WITH_KDE
}

bool KviApp::findLocalKVIrcDirectory()
{
	// Here we check if we already did the setup
	// and we have the KVIrc local directory saved somewhere
#ifdef COMPILE_WITH_KDE
	if( !m_pConfigFile ) {
		// Do not do that if user supplied a config file :)
		// With KDE we save the dir by placing it in the app config file
		KConfig *cfg = config();
		if( cfg ) {
			if( cfg->getConfigState() == KConfig::ReadWrite ) {
				cfg->setGroup("LocalKVIrcDirectory");
				m_szLocalKVIrcDir = cfg->readEntry("LocalKVIrcDirectory", "");
				// If we have it, ok... done
				if( checkLocalKVIrcDirectory(m_szLocalKVIrcDir.ptr()) ) return true;
			}
		}
	}
#endif
	// Check if we have a special .kvircrc in $HOME
	KviStr szF(QDir::homeDirPath());
	if( m_pConfigFile ) {
		if( *m_pConfigFile != '/' ) {
			szF.ensureLastCharIs('/');
			szF += m_pConfigFile;
		} else szF = m_pConfigFile;
	} else {
		szF.ensureLastCharIs('/');
		KviStr newF(szF);
		szF += KVI_HOME_CONFIG_FILE_NAME;
		QFile f1(szF.ptr());
		if( !f1.exists() ) {
			newF += KVI_OLD_HOME_CONFIG_FILE_NAME;
			szF   = newF;
		}
	}
	// If the file exists, read the first non empty line.
	QFile f(szF.ptr());
	if( f.open(IO_ReadOnly) ) {
		QTextStream t(&f);
		if( !t.eof() ) {
			do {
				m_szLocalKVIrcDir = t.readLine();
			} while( (!t.eof()) && (m_szLocalKVIrcDir.isEmpty()) );
		}
		f.close();
	} else {
		if( m_pConfigFile && !m_pCreateConfig ) {
			debug("Cannot open config file %s: defaulting to ~/%s", szF.ptr(), KVI_HOME_CONFIG_FILE_NAME);
			m_pConfigFile = 0;
			return findLocalKVIrcDirectory();
		}
	}
	return checkLocalKVIrcDirectory(m_szLocalKVIrcDir.ptr());
}

bool KviApp::loadDirectories()
{
	// Init locale
	if( !kvi_initLocale(this) )
		debug("[KVIrc]: cannot initialize the localization engine");
	// First find the global (readable) one...
	findGlobalKVIrcDirectory();

	// Now look for the local (writable) one
	if( !findLocalKVIrcDirectory() ) {
		// We must do the setup... ask the user..
		KviStr tmp(KviStr::Format, _i18n_(
			"Welcome to the KVIrc setup program!\n"
			"You will be asked to choose an existing directory now.\n"
			"A '%s' subdirectory will be created inside. KVIrc will\n"
			"use it to store its configuration files.\n"
			"Make sure that you have write permission to that directory.\n"),
			KVI_LOCAL_KVIRC_SUBDIRECTORY_NAME
		);

		KviMessageBox::information(tmp.ptr(), _i18n_("KVIrc Setup"));
		KviStr home = QDir::homeDirPath();

		m_szLocalKVIrcDir = kvi_askForDirectoryName(home.ptr());
		if( m_szLocalKVIrcDir.isEmpty() ) {
			warningBox(_i18n_("Setup aborted."));
			return false;
		}
		m_szLocalKVIrcDir.ensureLastCharIs('/');
		m_szLocalKVIrcDir += KVI_LOCAL_KVIRC_SUBDIRECTORY_NAME;
		// Now create the dir...
		if( kvi_directoryExists(m_szLocalKVIrcDir.ptr()) ) {
			tmp.sprintf(_i18n_(
				"The '%s' subdirectory already exists,\n"
				"maybe from a previous KVIrc installation.\n"
				"If you experience any problems, try deleting the old directory."),
				KVI_LOCAL_KVIRC_SUBDIRECTORY_NAME
			);
			KviMessageBox::information(tmp.ptr(), _i18n_("KVIrc Setup"));
		} else if( !kvi_makeDir(m_szLocalKVIrcDir.ptr()) ) {
			// Failed to make the dir...
			warningBox(_i18n_(
				"Could not make the directory %s.\n"
				"Trying alternate default $HOME/%s"),
				m_szLocalKVIrcDir.ptr(), KVI_LOCAL_KVIRC_SUBDIRECTORY_NAME
			);
			m_szLocalKVIrcDir  = QDir::homeDirPath();
			m_szLocalKVIrcDir.ensureLastCharIs('/');
			m_szLocalKVIrcDir += KVI_LOCAL_KVIRC_SUBDIRECTORY_NAME;
			if( kvi_directoryExists(m_szLocalKVIrcDir.ptr()) ) {
				tmp.sprintf(_i18n_(
					"The '%s' subdirectory already exists, maybe from a previous KVIrc installation.\n"
					"It can be used, but if you experience any problems, try deleting the old directory\n."
					"and repeating the setup."),
					KVI_LOCAL_KVIRC_SUBDIRECTORY_NAME
				);
				KviMessageBox::information(tmp.ptr(), _i18n_("KVIrc Setup"));
			} else if( !kvi_makeDir(m_szLocalKVIrcDir.ptr()) ) {
				warningBox(_i18n_(
					"Could not make directory $HOME/%s.\n"
					"Bad luck... you have some problems with your environment, \n"
					"the $HOME/%s path is not writable or is not a directory.\n"
					"Setup aborted."),
					KVI_LOCAL_KVIRC_SUBDIRECTORY_NAME, KVI_LOCAL_KVIRC_SUBDIRECTORY_NAME
				);
				return false;
			}
		}
		if( !kvi_directoryIsWritable(m_szLocalKVIrcDir.ptr()) ) {
			warningBox(_i18n_(
				"Could not write to directory %s.\n"
				"Setup aborted."),
				m_szLocalKVIrcDir.ptr()
			);
			return false;
		} else {
			// Make global->local link
			KviStr localPath(KviStr::Format, "%s/global", m_szLocalKVIrcDir.ptr());
			symlink(m_szGlobalKVIrcDir.ptr(), localPath.ptr());
			// Now save it
			saveKVIrcDirectory();
		}
	}
	return true;
}
