#! /usr/bin/perl

use strict;
use warnings;

my $g_BGCOLOR = "#999999";

my %g_TITLES = (
	'description' => 'Description',
	'events'      => 'Events',
	'examples'    => 'Examples',
	'functions'   => 'Functions',
	'inherits'    => 'Inherits',
	'parameters'  => 'Parameters',
	'return'      => 'Return value',
	'seealso'     => 'See also',
	'signals'     => 'Signals',
	'syntax'      => 'Syntax',
	'window'      => 'Window',
);

my %g_DEFAULTS = (
	'return' => 'Not modified',
);

my $g_HELP_INSTALL_DIR = shift @ARGV;
my $FH_STDOUT          = select STDOUT;

################################################################################
#
##
sub make_single_line
{
	if( $_[0] ) {
		$_[0] =~ s/^\s*//g;
		$_[0] =~ s/[\s\n]*$//gs;
	}
}

################################################################################
#
##
sub process_body_line
{
	$_[0] =~ s/->/-&gt;/g;
	$_[0] =~ s/\@tab\@/\t/g;
	$_[0] =~ s/\t/\&nbsp\;\&nbsp\;\&nbsp\;\&nbsp\;/g;
	$_[0] =~ s/\[cmd\](\w+)\[\/cmd\]/<a href="$1.kvihelp">$1<\/a>/g;
	$_[0] =~ s/\[cmd:(\w+)\](\w+)\[\/cmd\]/<a href="$1.kvihelp">$2<\/a>/g;
	$_[0] =~ s/\[fnc\]\$(\w+)\[\/fnc\]/<a href="s_$1.kvihelp">\$$1<\/a>/g;
	$_[0] =~ s/\[class\](\w+)\[\/class\]/<a href="class_$1.kvihelp">$1<\/a>/g;
	$_[0] =~ s/\[classfnc:(\w+)\]\$(\w+)\[\/classfnc\]/<a href="class_$1.kvihelp#$2">\$$2<\/a>/g;
	$_[0] =~ s/\[classevent:(\w+)\](\w+)\[\/classevent\]/<a href="class_$1.kvihelp#$2">$2<\/a>/g;
	$_[0] =~ s/\[classsignal:(\w+)\](\w+)\[\/classsignal\]/<a href="class_$1.kvihelp#$2">$2<\/a>/g;
	$_[0] =~ s/\[event\](\w+)\[\/event\]/<a href="$1.kvihelp">$1<\/a>/g;
}

################################################################################
#
##
sub subst_node
{
	my $value  = $_[0];
	my $node   = $_[1];
	$value =~ s/:node:/$node/g;
	$value =~ s/:NODE:/\U$node\E/g;
	return $value;
}

################################################################################
#
##
sub print_document_header
{
	my $title = subst_node($_[0], $_[1]);
	print "<doctitle>$title</doctitle>\n";
	print "<docbody>\n";
	print "<hr>\n";
}

################################################################################
#
##
sub print_document_footer
{
	print "<hr>\n";
	print "<a href=\"index.kvihelp\">Main Index</a>$_[0]\n";
	print "</docbody>\n";
}

################################################################################
#
##
sub print_part
{
	my $title      = $_[0];
	my $body       = $_[1];
	my $blockquote = $_[2];
	if( $g_TITLES{$title} ) {
		$title = $g_TITLES{$title};
	}
	print "<docsubtitle>$title</docsubtitle>\n";
	if( $blockquote ) {
		print "<blockquote>\n";
	}
	if( $body ) {
		print "$body<br>\n";
	} else {
		if( $g_DEFAULTS{$_[0]} ) {
			print "$g_DEFAULTS{$_[0]}<br>";
		}
	}
	if( $blockquote ) {
		print "</blockquote>\n";
	}
}

################################################################################
#
##
sub parse_documentation_for_X
{
	my %params = @_;
	if( $params{'input'} ) {
		# Input file specified; read only that file
		$params{'input'} = "$params{'input'}";
		my %node_data = read_doc_from_file(%params);
		# Merge node data into the params hash
		$params{'node_data'} = ();
		for my $key (sort keys %node_data) {
			%{$params{'node_data'}{$key}} = %{$node_data{$key}};
		}
	} else {
		# Input file NOT specified; analyze all source files
		$params{'node_data'} = ();
		for my $input_file (`find -name '*.cpp'`) {
			chomp $input_file;
			$params{'input'} = "$input_file";
			my %node_data = read_doc_from_file(%params);
			# Merge node data from the current file into the params hash
			for my $key (sort keys %node_data) {
				%{$params{'node_data'}{$key}} = %{$node_data{$key}};
			}
		}
	}
	# Return the parameters in case the caller wants to do something with it
	return %params;
}

################################################################################
#
##
sub read_doc_from_file
{
	my %params     = @_;
	my $input_file = "$params{'input'}";
	my $type       = "$params{'type'}";
	if( !open(INPUT_FILE, "$input_file") ) {
		print "\nCannot open $input_file\n";
		print "Aborting!\n";
		exit 1;
	}
	my %node_data = ();
	my $node      = "";
	my $part      = "";
	my $partbody  = "";
	my $tabblock  = "";
	while( <INPUT_FILE> ) {
		if( /^\s*\@$type:[\sa-z_]*/ ) {
			$node = "$_";
			$node =~ s/\s+\@$type:\s*//;
			$node =~ s/\s*$//;
			INNER: while( <INPUT_FILE> ) {
				if( /^\s*\*\/\s*/ ) {
					if( ($part ne "") && ($partbody ne "") && ($partbody ne "\n") ) {
						$node_data{$node}{$part} = "$partbody";
					}
					last INNER;
				} else {
					# Inside a part
					if( /^\s*\@[a-z]*:\s*/ ) {
						# A new part title
						if( ($part ne "") && ($partbody ne "") && ($partbody ne "\n") ) {
							# Store the previous part
							$node_data{$node}{$part} = "$partbody";
						}
						# Start a new part
						$part =  "$_";
						$part =~ s/^\s*//;
						$part =~ s/\s*$//;
						$part =~ s/\@//g;
						# Split the line around the ':' character in case
						# the body starts right after the field name
						$partbody = $part;
						$part =~ s/:.*//g;
						chomp $part;
						$partbody =~ s/.*://g;
						$partbody =~ s/^\s*//;
						$partbody =~ s/\s*$//;
						if( $partbody ) {
							$node_data{$node}{$part} = "$partbody";
							$part     = "body";
							$partbody = "";
						}
					} else {
						# Somewhere in a part body
						if( ($_ ne "") && ($_ ne "\n") ) {
							if( $partbody eq "" ) {
								# If it is the first line of the part body
								# Extract the amount of tabs that the part has
								# We will use it to remove the C++ indentation
								$tabblock = "$_";
								$tabblock =~ s/^(\t*).*/$1/g;
								chomp $tabblock;
							}
							if( $tabblock ne "" ) {
								# If we have the initial tab block, remove it from the line (remove indentation)
								$_ =~ s/^$tabblock//;
							}
							process_body_line($_);
							$partbody = "$partbody$_";
						}
					}
				}
			}
		}
	}
	close(INPUT_FILE);
	return %node_data;
}

################################################################################
#
##
sub write_help_files
{
	my %params    = @_;
	my %node_data = %{$params{'node_data'}};
	for my $node (sort keys %node_data) {
		my $node_lower    = "\L$node\E";
		my $help_filename = subst_node("$g_HELP_INSTALL_DIR/$params{'help_file'}", $node_lower);
		if( open(DOCFILE, ">$help_filename") ) {
			print ".";
			select DOCFILE;
			if( $params{'titlefield'} ) {
				print_document_header($params{'title'}, $node_data{$node}{$params{'titlefield'}});
			} else {
				print_document_header($params{'title'}, $node);
			}
			foreach my $part (split / /, $params{'fields'}) {
				next unless ($node_data{$node}{$part} || $part eq "return");
				make_single_line($node_data{$node}{$part});
				print_part($part, $node_data{$node}{$part}, $params{'blockquote'});
			}
			if( $node_data{$node}{'body'} ) {
				print $node_data{$node}{'body'};
			}
			print_document_footer($params{'footer'});
		} else {
			print "\nCan't open $help_filename for writing\n";
		}
		select $FH_STDOUT;
		close(DOCFILE);
	}
}

################################################################################
#
##
sub generate_index
{
	my %params         = @_;
	my $index_filename = "$g_HELP_INSTALL_DIR/$params{'index'}";
	my %node_data      = %{$params{'node_data'}};
	if( open(INDEXFILE, ">$index_filename") ) {
		print ".";
		select INDEXFILE;
		print_document_header($params{'doctitle'});
		my $bgcolor = "$g_BGCOLOR";
		if( $params{'bgcolor'} ) {
			$bgcolor = $params{'background'};
		}
		print "<table bgcolor=\"$bgcolor\">\n";
		for my $node (sort keys %node_data) {
			my $node_lower  = "\L$node\E";
			my $link        = subst_node($params{'help_file'}, $node_lower);
			my $index_entry = $node;
			if( $params{'index_entry'} ) {
				if( $node_data{$node}{$params{'index_entry'}} ) {
					$index_entry = $node_data{$node}{$params{'index_entry'}};
				}
			}
			print "<tr><td><a href=\"$link\">$index_entry</a>";
			if( $node_data{$node}{short} ) {
				make_single_line($node_data{$node}{short});
				print "</td><td>$node_data{$node}{short}</td></tr>\n";
			} else {
				print "</td><td></td></tr>\n";
			}
		}
		print "</table>\n";
		print_document_footer($params{'footer'});
	} else {
		print "\nCan't open $index_filename for writing\n";
	}
	close(INDEXFILE);
	select $FH_STDOUT;
}

################################################################################
#
##
sub generate_table
{
	my $cells = $_[0];
	chomp $cells;
	my $end_block = "<br>\n</blockquote>\n</td></tr>";
	$cells =~ s
		{(.+)(\n<tr><td>)}
		{$1$end_block$2}g;
	$cells = "<table bgcolor=\"$g_BGCOLOR\">\n$cells$end_block\n</table>";
	return $cells;
}

################################################################################
#
##
sub generate_classes
{
	my %params = (
		'blockquote' => 'yes',
		'doctitle'   => 'Classes index',
		'fields'     => 'inherits functions events signals description examples seealso',
		'footer'     => ' <a href="classes.kvihelp">Class Index</a>',
		'help_file'  => 'class_:node:.kvihelp',
		'index'      => 'classes.kvihelp',
		'title'      => 'Class :node:',
		'type'       => 'class',
	);
	# Read the documentation data
	%params = parse_documentation_for_X(%params);
	# Process the functions table
	for my $class (sort keys %{$params{'node_data'}}) {
		my %node_data = %{$params{'node_data'}{$class}};
		for my $field (sort keys %node_data) {
			my $cells = $node_data{$field};
			next unless ($cells);
			if( $field eq "functions" ) { # Generate functions table
				$cells =~ s
					{\!fn: \$(\w+)(\(.*\))}
					{<tr><td>\n<b><a name="\L$1\E">\$$1$2</a></b><br>\n<blockquote>}g;
				$node_data{$field} = generate_table($cells);
			}
			if( $field eq "events" ) {    # Generate events table
				$cells =~ s
					{\!ev: (.*)}
					{<tr><td>\n<b>$1</b><br>\n<blockquote>}g;
				$node_data{$field} = generate_table($cells);
			}
			if( $field eq "signals" ) {   # Generate signals table
				$cells =~ s
					{\!sg: (\w+)(\(.*\))}
					{<tr><td>\n<b><a name="\L$1\E">$1$2</a></b><br>\n<blockquote>}g;
				$node_data{$field} = generate_table($cells);
			}
		}
		%{$params{'node_data'}{$class}} = %node_data;
	}
	# Write out all help files
	write_help_files(%params);
	$params{'footer'} = ' <a href="syntax_objects.kvihelp">Objects documentation</a>';
	# Generate index file
	generate_index(%params);
}

################################################################################
#
##
sub generate_commands
{
	my %params = (
		'doctitle'    => 'Supported commands',
		'fields'      => 'syntax return description examples seealso',
		'footer'      => ' <a href="commands.kvihelp">Command Index</a>',
		'help_file'   => ':node:.kvihelp',
		'index'       => 'commands.kvihelp',
		'input'       => 'src/kvirc/kvi_commands.cpp',
		'title'       => ':NODE: command',
		'type'        => 'command',
	);
	# Read the documentation data
	%params = parse_documentation_for_X(%params);
	# Write out all help files
	write_help_files(%params);
	# Generate index file
	$params{'footer'} = "";
	generate_index(%params);
}

################################################################################
#
##
sub generate_functions
{
	my %params = (
		'doctitle'    => 'Supported functions',
		'fields'      => 'syntax description examples seealso',
		'footer'      => ' <a href="functions.kvihelp">Function Index</a>',
		'help_file'   => 's_:node:.kvihelp',
		'index'       => 'functions.kvihelp',
		'input'       => 'src/kvirc/kvi_functions.cpp',
		'title'       => ':node: function',
		'type'        => 'function',
	);
	# Read the documentation data
	%params = parse_documentation_for_X(%params);
	# Write out all help files
	write_help_files(%params);
	# Generate index file
	$params{'footer'} = "";
	generate_index(%params);
}

################################################################################
#
##
sub generate_identifiers
{
	my %params = (
		'doctitle'    => 'Supported identifiers',
		'fields'      => 'syntax description examples seealso',
		'footer'      => ' <a href="identifiers.kvihelp">Identifier Index</a>',
		'help_file'   => 's_:node:.kvihelp',
		'index'       => 'identifiers.kvihelp',
		'input'       => 'src/kvirc/kvi_identifiers.cpp',
		'title'       => ':node: identifier',
		'type'        => 'identifier',
	);
	# Read the documentation data
	%params = parse_documentation_for_X(%params);
	# Write out all help files
	write_help_files(%params);
	# Generate index file
	$params{'footer'} = "";
	generate_index(%params);
}

################################################################################
#
##
sub generate_events
{
	my %params = (
		'doctitle'    => 'Supported events',
		'fields'      => 'parameters window description examples seealso',
		'footer'      => ' <a href="events.kvihelp">Event Index</a>',
		'help_file'   => ':node:.kvihelp',
		'index'       => 'events.kvihelp',
		'input'       => 'src/kvirc/kvi_event.cpp',
		'title'       => ':node: event',
		'type'        => 'event',
	);
	# Read the documentation data
	%params = parse_documentation_for_X(%params);
	# Write out all help files
	write_help_files(%params);
	# Generate index file
	$params{'footer'} = "";
	generate_index(%params);
}

################################################################################
#
##
sub generate_quickhelp
{
	my %params = (
		'doctitle'   => 'Widget index',
		'fields'     => '',
		'footer'     => ' <a href="quickhelp.kvihelp">Widget Index</a>',
		'help_file'  => 'qh_:node:.kvihelp',
		'index'      => 'quickhelp.kvihelp',
		'title'      => ':node:',
		'titlefield' => 'widget',
		'type'       => 'quickhelp',
	);
	# Read the documentation data
	%params = parse_documentation_for_X(%params);
	# Write out all help files
	write_help_files(%params);
	# Generate index file
	$params{'footer'} = "";
	generate_index(%params);
}

################################################################################
#
##
sub generate_misc
{
	my %params = (
		'doctitle'    => 'Misc documentation index',
		'fields'      => '',
		'footer'      => ' <a href="misc.kvihelp">Misc. Documentation Index</a>',
		'help_file'   => ':node:',
		'index'       => 'misc.kvihelp',
		'index_entry' => 'title',
		'title'       => ':node:',
		'titlefield'  => 'title',
		'type'        => 'document',
	);
	# Read the documentation data
	%params = parse_documentation_for_X(%params);
	# Write out all help files
	write_help_files(%params);
	# Generate index file
	$params{'footer'} = "";
	generate_index(%params);
}

################################################################################
# Main
######

print "\n### Generating help for commands...";
generate_commands;
print "\n### Generating help for functions...";
generate_functions;
print "\n### Generating help for identifiers...";
generate_identifiers;
print "\n### Generating help for events...";
generate_events;
print "\n### Generating quick help...";
generate_quickhelp;
print "\n### Generating class documentation...";
generate_classes;
print "\n### Generating misc docs...";
generate_misc;
print "\n";

exit;

#
################################################################################
