#ifndef PLUGIN_H
#define PLUGIN_H

#include <kore/kore.h>
#include <kore/module.h>

/*
 * Each Kore dll has to export a function with this signature, in order
 * to be used as a Kore plugin (this function is similar to main()):
 * extern "C" {
 *      Plugin* plugin(HMODULE libHandle, const char* libName, const char* libPath, int libFlags );
 * }
 * The default implementation for this function should look like:
 * Plugin* plugin(HMODULE libHandle, const char* libName, const char* libPath, int libFlags )
 * {
 *      return new Plugin( libHandle, libName, libPath, libFlags );
 * }
 * The PLUGIN_MAIN_HDR and PLUGIN_MAIN_BODY are convenience macros for
 * the interface and the implementation of the plugin(...) function.
 * "handle", "name", "path", "flags" are the NAMES of the corresponding parameters.
 * "Type" is the plugin class name (i.e. "MyPlugin").
 * I.e. for MyPlugin the user should include the following:
 * myplugin.h:
 * ==========
 * extern "C"
 * {
 *      PLUGIN_MAIN_HDR( handle, name, path, flags );
 * };
 * myplugin.cpp:
 * ============
 * PLUGIN_MAIN_BODY( MyPlugin, handle, name, path, flags );
 */
#define PLUGIN_MAIN_HDR(handle,name,path,flags) \
    Plugin* plugin(HMODULE handle, const char* name, const char* path, int flags)
#define PLUGIN_MAIN_BODY(Type,handle,name,path,flags) \
    PLUGIN_MAIN_HDR(handle,name,path,flags) \
    { \
        return new Type(handle,name,path,flags); \
    }

namespace kore
{

/**
 * class kore::Plugin - platform-independant representation of a dynamic
 * library (dll). One or more Kore Modules can be bundled into a dll
 * and dynamically loaded at run-time using the PluginLoader.
 * The kore::Plugin class provides the cross-platform abstraction for dll's,
 * as well as a few usefull callbacks related to dynamic loading.
 * The future versions of kore::Plugin will also provide introspective methods
 * (similar to rtti), such as hints about modules/services provided by the ddl.
 */
class KORE_API Plugin: public Module
{
public:
    /**
     * Default constructor. Creates an empty plugin instance.
     */
    Plugin();
    /**
     * Creates a plugin. This constructor is usually invoked by the plugin(...)
     * function exported by the dll:
     * Plugin* plugin(HMODULE handle, const char* name, const char* path, int flags)
     * { return new MyPlugin(handle,name,path,flags); }
     * @param handle - platform-dependant representation of the actual dl.l
     * @param libname - dll's name.
     * @param libpath - dll's path.
     * @param flags - flags used for opening the dll (platform dependant).
     */
    Plugin(HMODULE handle, const char* libname, const char* libpath, int flags);
    /**
     * Destructor.
     */
    virtual ~Plugin();

    /**
     * Callback triggered right AFTER the dll has been LOADED.
     */
    virtual void pluginLoaded();
    /**
     * Callback triggered right BEFORE UNLOADING the dll.
     */
    virtual void unloadingPlugin();
    /**
     * Callback triggered when a plugin gets "activated".
     * Here is the right place for registering any modules/services
     * provided by the dll. DO NOT use pluginLoaded() for that.
     */
    virtual void initPlugin();
    /**
     * Callback triggered when a plugin gets "deactivated".
     * Here is the right place to do cleanup/unregistering the
     * modules/services this plugin might have registered.
     */
    virtual void finalizePlugin();

    /**
     * Gets the platform-dependant representation of the dll.
     * @return - the dll "handle".
     */
    virtual HMODULE libHandle() const;
protected:
private:
    // common initialization routine for Plugin's constructors.
    void commonInit();
    // default Plugin version
    const Version* _pluginVersion;
    // Kernel API version required by this Plugin
    const Version* _pluginAPIVersion;
    // Plugin info
    const Info* _pluginInfo;
    // dll's name
    const char* _libName;
    // dll's path
    const char* _libPath;
    // flags used when opening the dll (platform dependant)
    int _libFlags;
    // platform-dependant representation of the actual dll
    HMODULE _libHandle;
};

};

#endif
