/***************************************************************************
 *                                                                         *
 *   $FILE$ - A NetworkManager frontend for KDE                            *
 *   Copyright (C) 2005 Novell, Inc.                                       *
 *                                                                         *
 *   Author: Will Stephenson <wstephenson@suse.de>                         *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 *                                                                         *
 ***************************************************************************/
#include <qdragobject.h>
#include <qheader.h>
#include <qtooltip.h>

#include <kdebug.h>
#include <kglobal.h>
#include <klocale.h>

#include "knetworkmanager-network.h"
#include "knetworkmanager-nminfo.h"

#include "knetworkmanager-ui-networklistview.h"

class NetworkGroupLVI : public QListViewItem
{
public:
	enum GroupType { Trusted, Preferred, Untrusted };
	GroupType groupType;
	NetworkGroupLVI( QListView * parent, GroupType gType, const QString & label ) : QListViewItem( parent, label ), groupType( gType )  
	{
		setDragEnabled( false );
		setExpandable( true );
	}
	virtual int rtti() const { return 1001; }
};

class NetworkLVI : public QListViewItem
{
public:
	Network * net;
	NetworkLVI( NetworkGroupLVI * parent, Network* nw, const QString & label ) : QListViewItem( parent, label ), net( nw ) { setExpandable( true ); KLocale * loc = KGlobal::locale(); setText( 1, loc->formatDateTime( nw->getTimestamp() ) ); }
	virtual int rtti() const { return 1002; }
};

class AccessPointLVI : public QListViewItem
{
public:
	AccessPointLVI( NetworkLVI * parent, const QString & label ) : QListViewItem( parent, label ) {}
	virtual int rtti() const { return 1003; }
};

KNetworkManagerNetworkListView::KNetworkManagerNetworkListView( NetworkManagerInfo * nmi, QWidget * parent, const char * name ) : KListView( parent, name )
{
	header()->setStretchEnabled( true, 0 );
	header()->setStretchEnabled( false, 1 );
	setDragEnabled( true );
	setAcceptDrops( true );
	setRootIsDecorated( true );
	setAllColumnsShowFocus( true );
	addColumn( i18n( "Networks" ), -1 );
	addColumn( i18n( "column header for the last time that a particular network was used", "Last Used" ), -100 );
	m_trustedGroup = new NetworkGroupLVI( this, NetworkGroupLVI::Trusted, i18n("Trusted") );
	i18n("Preferred" );
	//m_preferredGroup = new NetworkGroupLVI( this, NetworkGroupLVI::Preferred, i18n("Preferred" ) );
	m_untrustedGroup = new NetworkGroupLVI( this, NetworkGroupLVI::Untrusted, i18n("Untrusted" ) );
	m_trustedGroup->setOpen( true );
	//m_preferredGroup->setOpen( true );
	m_untrustedGroup->setOpen( true );
	//header()->setStretchEnabled( true, 0 );
	
	QStringList networks = nmi->getNetworks();
	QStringList::Iterator it = networks.begin();
	const QStringList::Iterator end = networks.end();
	for ( ; it != end; ++it )
	{
		kdDebug() << "populating network: " << *it << endl;
		Network * net = nmi->getNetworkProperties( *it );
		NetworkLVI * netLvi = new NetworkLVI( ( net->isTrusted() ? m_trustedGroup : m_untrustedGroup ), net, *it );

		QStringList hwAddrs = net->getHardwareAddresses( );
		QStringList::Iterator apIt = hwAddrs.begin();
		const QStringList::Iterator apEnd = hwAddrs.end();
		for ( ; apIt != apEnd; ++apIt )
				new AccessPointLVI( netLvi, *apIt );
	}
/*	Network * foo = new Network();
	foo->setEssid( "foo" );
	NetworkLVI * on = new NetworkLVI( m_trustedGroup, foo, "onenet" );
	NetworkLVI * bj =new NetworkLVI( m_trustedGroup, foo, "banjaxed" );
	NetworkLVI * kn =new NetworkLVI( m_preferredGroup, foo, "knetworkmanager" );
	NetworkLVI * bb =new NetworkLVI( m_untrustedGroup, foo, "badboyz" );
	new AccessPointLVI( on, "10:00:00:00:00:01" );
	new AccessPointLVI( bj, "20:00:00:00:00:01" );
	new AccessPointLVI( kn, "30:00:00:00:00:01" );
	new AccessPointLVI( bb, "66:66:66:66:66:66" );
	kdDebug() << k_funcinfo << endl;*/
	
	connect( this, SIGNAL( moved ( QListViewItem *, QListViewItem *, QListViewItem * ) ), SLOT( slotMoved( QListViewItem *, QListViewItem *, QListViewItem * ) ) );
}

QDragObject * KNetworkManagerNetworkListView::dragObject()
{
	if (!currentItem())
		return 0;
	QStoredDrag * dragObj = 0;
	NetworkLVI * netLvi = dynamic_cast< NetworkLVI * >( currentItem() );
	if ( netLvi )
	{
		dragObj = new QStoredDrag("knetworkmanager/network", viewport());
		dragObj->setEncodedData( netLvi->net->getEssid().utf8() );
	}
	else
	{
		AccessPointLVI * apLvi = dynamic_cast< AccessPointLVI * >( currentItem() );
		if ( apLvi )
		{
			dragObj = new QStoredDrag("knetworkmanager/accesspoint", viewport());
			dragObj->setEncodedData( apLvi->text( 0 ).utf8() );
		}
	}
	return dragObj;
	//return KListView::dragObject();
}

bool KNetworkManagerNetworkListView::acceptDrag(QDropEvent* e) const
{
	if ( acceptDrops() && itemsMovable() && (e->source()==viewport()) )
	{
		QListViewItem *source=currentItem();
		QListViewItem *parent;
		QListViewItem *afterme;
		const_cast<KNetworkManagerNetworkListView *>( this )->findDrop( e->pos(), parent, afterme );
		
		QListViewItem * dest = parent;
		NetworkGroupLVI * destGroup = dynamic_cast< NetworkGroupLVI * >( dest );
		NetworkLVI * destNetwork = dynamic_cast< NetworkLVI * >( dest );
		NetworkLVI * sourceNetwork = dynamic_cast< NetworkLVI * >( source );
		AccessPointLVI * destAp = dynamic_cast< AccessPointLVI * >( dest );
		AccessPointLVI * sourceAp = dynamic_cast< AccessPointLVI * >( source );
		// debug
		
		kdDebug() << "*** parent= " << parent << " afterme=" << afterme << endl;
		if ( destGroup )
			kdDebug() << "destination is group " << destGroup->text( 0 ) << endl;;
		if ( destNetwork )
			kdDebug() << "destination is network " << destNetwork->text( 0 ) << endl;
		if ( destAp ) 
			kdDebug() << "destination is access point " << destAp->text( 0 ) << endl;
		if ( sourceAp ) 
			kdDebug() << "source is access point " << sourceAp->text( 0 ) << endl;
		if ( sourceNetwork ) 
			kdDebug() << "source is access point " << sourceNetwork->text( 0 ) << "***" << endl;
		// end of debug
		
		
		if ( destNetwork != 0 && !( sourceNetwork == 0 || sourceAp == 0) )  	// dropping a network on a network, accept
			return true;										// or dropping an AP on a network, accept
		if ( destAp != 0 && sourceAp != 0 )	// accept if dropping an AP on an AP
			return true;
		if ( destNetwork != 0 && sourceAp != 0 )	// accept if dropping an AP on an AP
			return true;
		if ( destGroup != 0 && sourceNetwork != 0 ) // accept if dragging a network to a group
			return true;
	}
	return false;
}

void KNetworkManagerNetworkListView::slotMoved(QListViewItem * item, QListViewItem * after, QListViewItem * /*now*/ )
{
	kdDebug() << k_funcinfo << endl;
	NetworkLVI * netLvi;
	if ( ( netLvi = dynamic_cast< NetworkLVI * >( item ) ) )
	{
		kdDebug() << k_funcinfo << "An entire network was dropped on " << netLvi->parent()->text( 0 ) << endl;
		if ( netLvi->parent() == m_untrustedGroup )
			netLvi->net->setTrusted( false );
		if ( item->parent() == m_trustedGroup )
			netLvi->net->setTrusted( true );
		// save changes
		m_changedNetworks.append( netLvi->net );
	}
	else if ( AccessPointLVI * apLvi = dynamic_cast< AccessPointLVI * >( item ) )
	{
		kdDebug() << k_funcinfo << "An Access Point " << apLvi->text( 0 ) << " was dropped on " << apLvi->parent()->text( 0 ) << endl;
		netLvi = dynamic_cast< NetworkLVI * >( item->parent() );
		if ( netLvi )
		{
			QStringList hwAddrs = netLvi->net->getHardwareAddresses();
			netLvi->net->insertHardwareAddress( apLvi->text( 0 ) );
			kdDebug() << "new parent now has hwaddrs: " << netLvi->net->getHardwareAddresses( ) << endl;
			kdDebug() << "previous parent was: " << after->text( 0 ) << endl;
			NetworkLVI * oldParentLvi = dynamic_cast< NetworkLVI * >( after );
			if ( oldParentLvi )
			{
				oldParentLvi->net->removeHardwareAddress( apLvi->text( 0 ) );
				kdDebug() << "old parent now has hwaddrs: " << oldParentLvi->net->getHardwareAddresses( ) << endl;
			}
			//save changes
			m_changedNetworks.append( netLvi->net );
			if ( oldParentLvi )
				m_changedNetworks.append( oldParentLvi->net );
		}
	}
	else 
		kdDebug() << "something I don't like was dropped" << endl;
}


void KNetworkManagerNetworkListView::slotAddNetworkClicked()
{
	// prompt for essid
	// create a new network and add it to m_changedNetworks

}

void KNetworkManagerNetworkListView::slotAddAccessPointClicked()
{
	// prompt for hwaddr
	// find the selected network, add it, and add it to m_changedNetworks
}

void KNetworkManagerNetworkListView::slotRemoveItemClicked()
{
	if ( selectedItem() )
	{
		if ( NetworkLVI * netLvi = dynamic_cast<NetworkLVI * >( selectedItem() ) )
		{
			m_deletedNetworks.append( netLvi->net );
			delete selectedItem();
		}
		else if ( AccessPointLVI * apLvi = dynamic_cast<AccessPointLVI * >( selectedItem() ) )
		{
			NetworkLVI * netLvi = dynamic_cast< NetworkLVI * >( apLvi->parent() );
			Q_ASSERT( netLvi );
			netLvi->net->removeHardwareAddress( apLvi->text( 0 ) );
			m_changedNetworks.append( netLvi->net );
			delete selectedItem();
		}
	}
}

#include "knetworkmanager-ui-networklistview.moc"
