/*
 *  linux/arch/parisc/kernel/power.c
 *  HP PARISC powerfail and soft power switch support
 *
 *  Copyright (C) 2001 Helge Deller <deller@gmx.de>
 *
 */

/*
 * Status: EXPERIMENTAL
 *  
 * TODO:
 * - use tasklets instead of calling it from the timer-interrupt function
 * - after a special amount of time just turn the machine off (killing init may have failed!)
 * - ....
 */

#include <linux/init.h>
#include <linux/kernel.h>
#include <linux/string.h>
#include <linux/reboot.h>
#include <linux/sched.h>
#include <linux/interrupt.h>

#include <asm/gsc.h>
#include <asm/pdc.h>
#include <asm/irq.h>
#include <asm/io.h>
#include <asm/led.h>


/* For kernel debugging purposes it's sometimes better to have
 * the soft-power switch killing the power at once.
 * This may be reached by uncommenting the following line: */

//#define POWERSWITCH_DISABLED


/* local time-counter for shutdown */
static int shutdown_timer;

#define DIAG_CODE(code)		(0x14000000 + ((code)<<5))

/* this will go to processor.h or any other place... */
/* taken from PCXL ERS page 82 */
#define MFCPU_X(rDiagReg, t_ch, t_th, code) \
	(DIAG_CODE(code) + ((rDiagReg)<<21) + ((t_ch)<<16) + ((t_th)<<0) )
	
#define MTCPU(dr, gr)		MFCPU_X(dr, gr,  0, 0x12)       /* move value of gr to dr[dr] */
#define MFCPU_C(dr, gr)		MFCPU_X(dr, gr,  0, 0x30)	/* for dr0 and dr8 only ! */
#define MFCPU_T(dr, gr)		MFCPU_X(dr,  0, gr, 0xa0)	/* all dr except dr0 and dr8 */
	
#define __getDIAG(dr) ( { 			\
        register unsigned long __res asm("r28");\
	 __asm__ __volatile__ (			\
		".word %1\n nop\n" : "=&r" (__res) : "i" (MFCPU_T(dr,28)) \
	);					\
	__res;					\
} )




static void deferred_poweroff(void *dummy)
{
	extern int cad_pid;	/* kernel/sys.c */
	kill_proc(cad_pid, SIGINT, 1);
	/* machine_power_off(); */
}

/*
 * This function gets called from interrupt context.
 * As it's called within an interrupt, it wouldn't sync if we don't
 * use schedule_task().
 */

static void poweroff(void)
{
	static int powering_off;
	static struct tq_struct poweroff_tq = {
		routine: deferred_poweroff,
	};

	if (powering_off)
		return;

	powering_off++;
	schedule_task(&poweroff_tq);
}

/* check, give feedback and start shutdown after one second */
static void process_shutdown(void)
{
	if (shutdown_timer == 0)
		printk(KERN_INFO "Shutdown requested...\n");

	shutdown_timer++;
	
	/* wait until the button was pressed for 1 second */
	if (shutdown_timer == HZ) {
		static char msg[] = "Shutting down...";
		printk(KERN_INFO "%s\n", msg);
#ifdef CONFIG_CHASSIS_LCD_LED
		lcd_print(msg);
#endif
		poweroff();
	}
}


/* soft power switch function ptr */
void (*check_soft_power)(void);


/*
 * On gecko style machines (e.g. 712/xx and 715/xx) 
 * the power switch status is stored in Bit 0 ("the highest bit")
 * of CPU diagnose register 25.
 * 
 */
static void check_soft_power_gecko(void)
{
	if (__getDIAG(25) & 0x80000000) {
		/* power switch button not pressed or released again */
		shutdown_timer = 0;
	} else {
		process_shutdown();
	}
}



/*
 * Check the power switch status which is read from the
 * real I/O location at soft_power_reg.
 * Bit 31 ("the lowest bit) is the status of the power switch.
 */
static unsigned long soft_power_reg;

static void check_soft_power_polling(void)
{
        unsigned long current_status;

	current_status = gsc_readl(soft_power_reg);
	if (current_status & 0x1) {
		/* power switch button not pressed */
		shutdown_timer = 0;
	} else {
		process_shutdown();
	}
}


/*
 * powerfail interruption handler (irq IRQ_FROM_REGION(CPU_IRQ_REGION)+2) 
 */
#if 0
static void powerfail_interrupt(int code, void *x, struct pt_regs *regs)
{
	printk(KERN_CRIT "POWERFAIL INTERRUPTION !\n");
	poweroff();
}
#endif



static int __init power_init(void)
{
	unsigned long ret;

#if 0
	request_irq( IRQ_FROM_REGION(CPU_IRQ_REGION)+2, &powerfail_interrupt,
		0, "powerfail", NULL);
#endif

#if !defined(POWERSWITCH_DISABLED)
	/* enable the soft power switch if possible */
	ret = pdc_soft_power_info(&soft_power_reg);
	if (ret != PDC_OK)
		return 0;
	
	switch ((long) soft_power_reg) {
	case 0:		printk(KERN_INFO "Enabled gecko-style soft power switch.\n");
			check_soft_power = check_soft_power_gecko;
			break;
			
	case -1:	printk(KERN_INFO "No soft power switch support.\n");
			break;
	
	default:	printk(KERN_INFO "Enabled soft power switch (polling mode, io=0x%08lx).\n",
				soft_power_reg);
			check_soft_power = check_soft_power_polling;
	}
#endif
	return 0;
}

module_init(power_init);

