#include "test-widget.h"
#include <inti/gdk-pixbuf/pixbuf.h>
#include <inti/glib/error.h>
#include <inti/glib/iochannel.h>
#include <inti/gtk/accelgroup.h>
#include <inti/gtk/fileselection.h>
#include <inti/gtk/menu.h>
#include <inti/gtk/menubar.h>
#include <inti/gtk/checkmenuitem.h>
#include <inti/gtk/radiomenuitem.h>
#include <inti/gtk/messagedialog.h>
#include <inti/gtk/scrolledwindow.h>
#include <inti/pango/font.h>
#include <inti/bind.h>
#include <libgnomevfs/gnome-vfs-init.h>
#include <libgnomevfs/gnome-vfs-mime-utils.h>
#include <libgnomevfs/gnome-vfs-utils.h>
#include <libgnomeprintui/gnome-print-job-preview.h>
#include <algorithm>

const int READ_BUFFER_SIZE = 4096;
const char *MARKER_TYPE_1 = "one";
const char *MARKER_TYPE_2 = "two";

/*  SourceView
 */

SourceView::SourceView(Gtk::SourceBuffer& buffer)
: Gtk::SourceView(buffer)
{
	// set font
	Pointer<Pango::FontDescription> font_desc = new Pango::FontDescription("monospace 10");
	if (font_desc)
		modify_font(font_desc);

	// add marker pixbufs
	Gdk::Pixbuf *pixbuf = new Gdk::Pixbuf("/usr/share/pixmaps/apple-green.png");
	set_marker_pixbuf(MARKER_TYPE_1, pixbuf);
	pixbuf = new Gdk::Pixbuf("/usr/share/pixmaps/apple-red.png");
	set_marker_pixbuf(MARKER_TYPE_2, pixbuf);
	pixbuf->unref();
}

SourceView::~SourceView()
{
}

bool
SourceView::on_button_press_event(const Gdk::EventButton& event)
{
	// check that the click was on the left gutter
	if (get_show_line_markers() && event.window() == get_window(Gtk::TEXT_WINDOW_LEFT))
	{
		int y_buf;
		const char *marker_type;

		if (event.button() == 1)
			marker_type = MARKER_TYPE_1;
		else
			marker_type = MARKER_TYPE_2;

		window_to_buffer_coords(Gtk::TEXT_WINDOW_LEFT, (int)event.x(), (int)event.y(), 0, &y_buf);
		Gtk::TextIter line_start = get_line_at_y(y_buf);
		Gtk::TextIter line_end = line_start;
		line_end.forward_to_line_end();

		// get the markers already in the line
		Gtk::SourceBuffer *buffer = get_source_buffer();
		std::vector<Gtk::SourceMarker*> markers = buffer->get_markers(line_start, line_end);

		// search for the marker corresponding to the button pressed
		Gtk::SourceMarker *marker = 0;
		int count = markers.size();

		for (int i = 0; i < count; i++)
		{
			String tmp_type = markers[i]->get_marker_type();
			if (!tmp_type.compare(marker_type))
				marker = markers[i];
		}

		if (marker)
		{
			// a marker was found, so delete it
			buffer->delete_marker(*marker);
		}
		else
		{
			// no marker found -> create one
			marker = buffer->create_marker(0, marker_type, line_start);
		}
	}
	return Gtk::SourceView::on_button_press_event(event);
}

/*  MainWindow
 */

MainWindow::MainWindow()
{
	set_title("Inti SourceView Demo");
	set_default_size(500, 500);

	// vbox
	Gtk::VBox *vbox = new Gtk::VBox;
	add(*vbox);
	vbox->show();

	// scrolled window
	Gtk::ScrolledWindow *sw = new Gtk::ScrolledWindow;
	sw->set_shadow_type(Gtk::SHADOW_IN);
	vbox->pack_end(*sw);
	sw->show();

	// source view
	manager = new Gtk::SourceLanguagesManager;
	buffer = new Gtk::SourceBuffer;
	source_view = new SourceView(*buffer);
	sw->add(*source_view);
	source_view->set_show_line_markers(true);
	source_view->set_show_line_numbers(true);
	source_view->set_show_margin(true);
	source_view->set_auto_indent(true);
	source_view->set_tabs_width(8);
	views_list.push_back(source_view);
	source_view->show();

	// item factory
	Pointer<Gtk::AccelGroup> accel_group = new Gtk::AccelGroup;
	item_factory = new Gtk::ItemFactory(GTK_TYPE_MENU_BAR, "<main>", accel_group);
	add_accel_group(accel_group);
	item_factory->create_items(*this);

	// menubar
	Gtk::MenuBar *menu = item_factory->menu_bar();
	vbox->pack_start(*menu, false, false);
	menu->show();

	// preselect menu checkitems
	static_cast<Gtk::CheckMenuItem*>(item_factory->get_item("/View/Show Line Numbers"))->set_active(true);
	static_cast<Gtk::CheckMenuItem*>(item_factory->get_item("/View/Show Markers"))->set_active(true);
	static_cast<Gtk::CheckMenuItem*>(item_factory->get_item("/View/Show Margin"))->set_active(true);
	static_cast<Gtk::CheckMenuItem*>(item_factory->get_item("/View/Enable Auto Indent"))->set_active(true);
	static_cast<Gtk::RadioMenuItem*>(item_factory->get_item("/View/Tabs Width/8"))->set_active(true);

	// cursor position label
	pos_label = new Gtk::Label("label");
	vbox->pack_start(*pos_label, false, false);
	buffer->sig_mark_set().connect(slot(this, &MainWindow::on_move_cursor));
	buffer->sig_changed().connect(slot(this, &MainWindow::on_update_cursor_position));
	pos_label->show();
}

MainWindow::~MainWindow()
{
}

bool
MainWindow::on_view_delete_event(GdkEventAny *event, SourceView *view)
{
	std::vector<Gtk::SourceView*>::iterator i = std::find(views_list.begin(), views_list.end(), view);
	if (i != views_list.end())
	{
		views_list.erase(i);

		// Return false to destroy the window.
		return false;
	}
	return true;
}

void
MainWindow::on_move_cursor(const GtkTextIter *cursoriter, GtkTextMark *mark)
{
	if (mark != gtk_text_buffer_get_insert(*buffer))
		return;

	on_update_cursor_position();
}

void
MainWindow::on_update_cursor_position()
{
	Gtk::TextIter iter = buffer->get_iter_at_mark(*buffer->get_insert());
	int chars = iter.get_offset();
	int row = iter.get_line() + 1;

	Gtk::TextIter start = iter;
	start.set_line_offset(0);
	int col = 0;

	G::Unichar tab_char('\t');
	while (!iter.equal(start))
	{
		if (start.get_char() == tab_char)
		{
			int tab_stop = source_view->get_tabs_width();
			col += (tab_stop - (col % tab_stop));
		}
		else
			++col;

		start.forward_char();
	}

	String msg;
	msg.format("char: %d, line: %d, column: %d", chars, row, col);
	pos_label->set_text(msg);
}

namespace { // helper functions

void remove_all_markers(Gtk::SourceBuffer& buffer)
{
	Gtk::TextIter begin, end;
	buffer.get_bounds(begin, end);
	std::vector<Gtk::SourceMarker*> markers = buffer.get_markers(begin, end);

	int count = markers.size();
	for (int i = 0; i < count; i++)
	{
		buffer.delete_marker(*markers[i]);
	}
}

void error_dialog(const String& message)
{
	Gtk::MessageDialog dialog(Gtk::MESSAGE_ERROR, Gtk::BUTTONS_OK);
	dialog.set_message(message);
	dialog.run();
	dialog.dispose();
}

bool buffer_load_with_encoding(Gtk::SourceBuffer& source_buffer, const String& filename, const char *encoding, G::Error *error)
{
	g_return_val_if_fail (!filename.empty(), false);

	Pointer<G::IOChannel> io = G::IOChannel::create(filename, "r", error);

	String msg;
	if (error->get())
	{
		msg.format("%s\nFile %s", error->message(), filename.c_str());
		error_dialog(msg);
		return false;
	}

	if (io->set_encoding(encoding, error) != G::IO_STATUS_NORMAL)
	{
		msg.format("Failed to set encoding:\n%s\n%s", filename.c_str(), error->message());
		error_dialog(msg);
		return false;
	}

	source_buffer.begin_not_undoable_action();
	source_buffer.set_text("");
	char *buffer = new char[READ_BUFFER_SIZE];
	bool reading = true;
	Gtk::TextIter iter;
	while (reading)
	{
		size_t bytes_read;
		G::IOStatus status = io->read(buffer, READ_BUFFER_SIZE, &bytes_read, error);
		switch (status)
		{
			case G::IO_STATUS_EOF:
				reading = false;
				// fall through

			case G::IO_STATUS_NORMAL:
				if (bytes_read == 0)
				{
					continue;
				}

				iter = source_buffer.get_end_iter();
				source_buffer.insert(iter, buffer, bytes_read);
				break;

			case G::IO_STATUS_AGAIN:
				continue;

			case G::IO_STATUS_ERROR:
			default:
				msg.format("%s\nFile %s", filename.c_str(), error->message());
				error_dialog(msg);

				// because of error in input we clear already loaded text
				source_buffer.set_text("");
				reading = false;
				break;
		}
	}

	delete buffer;
	source_buffer.end_not_undoable_action();

	if (error->get())
		return false;

	source_buffer.set_modified(false);

	// move cursor to the beginning
	iter = source_buffer.get_start_iter();
	source_buffer.place_cursor(iter);
	return true;
}

} // namespace

bool
MainWindow::open_file(const String& filename)
{
	String uri;
	if (g_path_is_absolute(filename.c_str()))
	{
		uri = gnome_vfs_get_uri_from_local_path(filename.c_str());
	}
	else
	{
		String path(g_get_current_dir());
		path += "/";
		path += filename;
		uri = gnome_vfs_get_uri_from_local_path(path.c_str());
	}

	String mime_type = gnome_vfs_get_mime_type(uri.c_str());
	if (!mime_type.null())
	{
		Gtk::SourceLanguage *language = manager->get_language_from_mime_type(mime_type);

		if (!language)
		{
			g_print("No language found for mime type `%s'\n", mime_type.c_str());
			buffer->set_highlight(false);
		}
		else
		{
			buffer->set_highlight(true);
			buffer->set_language(language);
		}
	}
	else
	{
		buffer->set_highlight(false);
		g_warning ("Couldn't get mime type for file `%s'", filename.c_str());
	}

	remove_all_markers(*buffer);
	G::Error error;
	buffer_load_with_encoding(*buffer, filename, "utf-8", &error);
	current_file = filename;

	return error.get() ? false : true;
}

void
MainWindow::print_job_begin_page(Gtk::SourcePrintJob *job)
{
	g_print("Printing %.2f%%    \r", 100.0 * job->get_page() / job->get_page_count());
}

void MainWindow::print_job_finished(Gtk::SourcePrintJob *job)
{
	g_print ("\n");
	GnomePrintJob *gjob = job->get_print_job();
	GtkWidget *preview = gnome_print_job_preview_new(gjob, (unsigned char*)"test-widget print preview");
 	g_object_unref (gjob);
 	job->unref();

	gtk_widget_show(preview);
}

void
MainWindow::on_file_open()
{
	Gtk::FileSelection file_sel("Open file...");
	if (file_sel.run() == Gtk::RESPONSE_OK)
	{
		open_file(file_sel.get_filename());
	}
	file_sel.dispose();
}

void
MainWindow::on_file_print_preview()
{
	Gtk::SourcePrintJob *job = new Gtk::SourcePrintJob(*source_view);
	job->set_wrap_mode(Gtk::WRAP_CHAR);
	job->set_highlight(true);
	job->set_print_numbers(5);
	job->set_header_format("Printed on %A", 0, "%F", true);
	job->set_footer_format("%T", current_file, "Page %N/%Q", true);
	job->set_print_header(true);
	job->set_print_footer(true);

	Gtk::TextIter start, end;
	buffer->get_bounds(start, end);
	if (job->print_async(start, end))
	{
		job->sig_begin_page().connect(bind(slot(this, &MainWindow::print_job_begin_page), job));
		job->sig_finished().connect(bind(slot(this, &MainWindow::print_job_finished), job));
	}
	else
		g_warning("Async print failed");
}

void
MainWindow::on_file_quit()
{
	Inti::Main::quit();
}

void
MainWindow::on_view_new()
{
	Gtk::Window *window = new Gtk::Window;
	window->set_default_size(400, 400);
	window->set_title(get_title());
	window->set_border_width(0);

	// vbox
	Gtk::VBox *vbox = new Gtk::VBox;
	window->add(*vbox);
	vbox->show();

	// scrolled window
	Gtk::ScrolledWindow *sw = new Gtk::ScrolledWindow;
	sw->set_shadow_type(Gtk::SHADOW_IN);
	vbox->pack_end(*sw);
	sw->show();

	// source view
	SourceView *view = new SourceView(*buffer);
	sw->add(*view);
	view->set_show_line_markers(source_view->get_show_line_markers());
	view->set_show_line_numbers(source_view->get_show_line_numbers());
	view->set_show_margin(source_view->get_show_margin());
	view->set_auto_indent(source_view->get_auto_indent());
	view->set_insert_spaces_instead_of_tabs(source_view->get_insert_spaces_instead_of_tabs());
	view->set_tabs_width(source_view->get_tabs_width());
	views_list.push_back(view);
	view->show();

	window->sig_delete_event().connect(bind(slot(this, &MainWindow::on_view_delete_event), view));
	window->show();
}

void
MainWindow::on_view_show_markers()
{
	int count = views_list.size();
	bool check = static_cast<Gtk::CheckMenuItem*>(item_factory->get_item("/View/Show Markers"))->get_active();
	for (int i = 0; i < count; i++)
	{
		views_list[i]->set_show_line_markers(check);
	}
}

void
MainWindow::on_view_show_line_numbers()
{
	int count = views_list.size();
	bool check = static_cast<Gtk::CheckMenuItem*>(item_factory->get_item("/View/Show Line Numbers"))->get_active();
	for (int i = 0; i < count; i++)
	{
		views_list[i]->set_show_line_numbers(check);
	}
}

void
MainWindow::on_show_margin()
{
	int count = views_list.size();
	bool check = static_cast<Gtk::CheckMenuItem*>(item_factory->get_item("/View/Show Margin"))->get_active();
	for (int i = 0; i < count; i++)
	{
		views_list[i]->set_show_margin(check);
	}
}

void
MainWindow::on_view_enable_auto_indent()
{
	int count = views_list.size();
	bool check = static_cast<Gtk::CheckMenuItem*>(item_factory->get_item("/View/Enable Auto Indent"))->get_active();
	for (int i = 0; i < count; i++)
	{
		views_list[i]->set_auto_indent(check);
	}
}

void
MainWindow::on_view_insert_spaces()
{
	int count = views_list.size();
	bool check = static_cast<Gtk::CheckMenuItem*>(item_factory->get_item("/View/Insert Spaces Instead of Tabs"))->get_active();
	for (int i = 0; i < count; i++)
	{
		views_list[i]->set_insert_spaces_instead_of_tabs(check);
	}
}

void
MainWindow::on_view_tabs_width_4()
{
	int count = views_list.size();
	for (int i = 0; i < count; i++)
	{
		views_list[i]->set_tabs_width(4);
	}
}

void
MainWindow::on_view_tabs_width_6()
{
	int count = views_list.size();
	for (int i = 0; i < count; i++)
	{
		views_list[i]->set_tabs_width(6);
	}
}

void
MainWindow::on_view_tabs_width_8()
{
	int count = views_list.size();
	for (int i = 0; i < count; i++)
	{
		views_list[i]->set_tabs_width(8);
	}
}

void
MainWindow::on_view_tabs_width_10()
{
	int count = views_list.size();
	for (int i = 0; i < count; i++)
	{
		views_list[i]->set_tabs_width(10);
	}
}

void
MainWindow::on_view_tabs_width_12()
{
	int count = views_list.size();
	for (int i = 0; i < count; i++)
	{
		views_list[i]->set_tabs_width(12);
	}
}

BEGIN_ITEM_FACTORY_MAP(MainWindow)
	IFM_BRANCH("/_File"),
	IFM_STOCK_ITEM("/File/_Open", "<control>O", on_file_open, GTK_STOCK_OPEN),
	IFM_STOCK_ITEM("/File/_Print Preview", "<control>P", on_file_print_preview, GTK_STOCK_PRINT),
	IFM_SEPARATOR("/File/sep1"),
	IFM_STOCK_ITEM("/File/Quit", "<control>Q", on_file_quit, GTK_STOCK_QUIT),

	IFM_BRANCH("/_View"),
	IFM_STOCK_ITEM("/View/_New View", 0, on_view_new, GTK_STOCK_NEW),
	IFM_SEPARATOR("/View/sep1"),
	IFM_CHECK_ITEM("/View/Show _Line Numbers", 0, on_view_show_line_numbers),
	IFM_CHECK_ITEM("/View/Show _Markers", 0, on_view_show_markers),
	IFM_CHECK_ITEM("/View/Show M_argin", 0, on_show_margin),
	IFM_SEPARATOR("/View/sep2"),
	IFM_CHECK_ITEM("/View/Enable _Auto Indent", 0, on_view_enable_auto_indent),
	IFM_CHECK_ITEM("/View/Insert _Spaces Instead of Tabs", 0, on_view_insert_spaces),
	IFM_SEPARATOR("/View/sep3"),

	IFM_BRANCH("/_View/_Tabs Width"),
	IFM_RADIO_ITEM("/View/Tabs Width/4", 0, on_view_tabs_width_4),
	IFM_RADIO_ITEM_LINK("/View/Tabs Width/6", on_view_tabs_width_6, "/View/Tabs Width/4"),
	IFM_RADIO_ITEM_LINK("/View/Tabs Width/8", on_view_tabs_width_8, "/View/Tabs Width/4"),
	IFM_RADIO_ITEM_LINK("/View/Tabs Width/10", on_view_tabs_width_10, "/View/Tabs Width/4"),
	IFM_RADIO_ITEM_LINK("/View/Tabs Width/12", on_view_tabs_width_12, "/View/Tabs Width/4"),
END_ITEM_FACTORY_MAP

int main (int argc, char *argv[])
{
	using namespace Main;

	init(&argc, &argv);
	gnome_vfs_init ();

	MainWindow window;
	window.sig_destroy().connect(slot(&Inti::Main::quit));

	if (argc > 1)
		window.open_file(argv [1]);
	else
		window.open_file("../inti/gtk-sourceview/sourcebuffer.cc");

	window.show();

	run();
	gnome_vfs_shutdown ();
	return 0;
}


