/*
   -------------------------------------------------------------------------
   i2c-adap-ppc405.c i2c-hw access for the IIC peripheral on the IBM PPC 405
   -------------------------------------------------------------------------
  
   Ian DaSilva, MontaVista Software, Inc.
   idasilva@mvista.com or source@mvista.com

   Copyright 2000 MontaVista Software Inc.

   Changes made to support the IIC peripheral on the IBM PPC 405 


   ----------------------------------------------------------------------------
   This file was highly leveraged from i2c-elektor.c, which was created
   by Simon G. Vogl and Hans Berglund:

 
     Copyright (C) 1995-97 Simon G. Vogl
                   1998-99 Hans Berglund

   With some changes from Kysti Mlkki <kmalkki@cc.hut.fi> and even
   Frodo Looijaard <frodol@dds.nl>


    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
   ----------------------------------------------------------------------------
*/


#include <linux/kernel.h>
#include <linux/ioport.h>
#include <linux/module.h>
#include <linux/delay.h>
#include <linux/slab.h>
#include <linux/version.h>
#include <linux/init.h>
#include <asm/irq.h>
#include <asm/io.h>
#include <linux/i2c.h>
#include <linux/i2c-algo-ppc405.h>
#include <linux/i2c-ppc405adap.h>
#include "i2c-ppc405.h"

#ifdef MODULE_LICENSE
MODULE_LICENSE("GPL");
#endif

#define DEFAULT_BASE 0xEF600500
#define DEFAULT_IRQ      2 
#define DEFAULT_CLOCK   50
#define DEFAULT_OWN   0x55

static int base  = 0;
static int irq   = 0;
static int clock = 0;
static int own   = 0;
static int i2c_debug=0;
static struct iic_ppc405 gpi;
#if (LINUX_VERSION_CODE < 0x020301)
static struct wait_queue *iic_wait = NULL;
#else
static wait_queue_head_t iic_wait;
#endif
static int iic_pending;

/* ----- global defines -----------------------------------------------	*/
#define DEB(x)	if (i2c_debug>=1) x
#define DEB2(x) if (i2c_debug>=2) x
#define DEB3(x) if (i2c_debug>=3) x
#define DEBE(x)	x	/* error messages 				*/


/* --- Convenience defines for the i2c port:			*/
#define BASE	((struct iic_ppc405 *)(data))->iic_base

/* ----- local functions ----------------------------------------------	*/

//
// Description: Write a byte to IIC hardware
//
static void iic_ppc405_setbyte(void *data, int ctl, int val)
{
	// writeb resolves to a write to the specified memory location
	// plus a call to eieio.  eieio ensures that all instructions
	// preceding it are completed before any further stores are
	// completed.
	// Delays at this level (to protect writes) are not needed here.
        writeb(val, ctl);
}


//
// Description: Read a byte from IIC hardware
//
static int iic_ppc405_getbyte(void *data, int ctl)
{
	int val;

	DEB3(printk("i2c-ppc405_getbyte: Read IIC register\n"));
	val = readb(ctl);
	DEB3(printk("i2c-ppc405getbyte: Read Data 0x%02X\n", val));
	return (val);
}


//
// Description: Return our slave address.  This is the address
// put on the I2C bus when another master on the bus wants to address us
// as a slave
//
static int iic_ppc405_getown(void *data)
{
	return (gpi.iic_own);
}


//
// Description: Return the clock rate
//
static int iic_ppc405_getclock(void *data)
{
	return (gpi.iic_clock);
}


#if 0
static void iic_ppc405_sleep(unsigned long timeout)
{
	schedule_timeout( timeout * HZ);
}
#endif


//
// Description:  Put this process to sleep.  We will wake up when the
// IIC controller interrupts.
//
static void iic_ppc405_waitforpin(void) {

   int timeout = 2;

   //printk("iic_ppc405_waitforpin: At top of function\n");
   //
   // If interrupts are enabled (which they are), then put the process to
   // sleep.  This process will be awakened by two events -- either the
   // the IIC peripheral interrupts or the timeout expires. 
   //
   if (gpi.iic_irq > 0) {
      cli();
      if (iic_pending == 0) {
         //printk("iic_ppc405_waitforpin: calling interruptible_sleep_on_timeout\n");
  	 interruptible_sleep_on_timeout(&iic_wait, timeout*HZ );
      } else
 	 iic_pending = 0;
      sti();
   } else {
      //
      // If interrupts are not enabled then delay for a reasonable amount
      // of time and return.  We expect that by time we return to the calling
      // function that the IIC has finished our requested transaction and
      // the status bit reflects this.
      //
      // udelay is probably not the best choice for this since it is
      // the equivalent of a busy wait
      //
      udelay(100);
   }
   //printk("iic_ppc405_waitforpin: exitting\n");
}


//
// Description: The registered interrupt handler
//
static void iic_ppc405_handler(int this_irq, void *dev_id, struct pt_regs *regs) 
{
   int ret;
	
   iic_pending = 1;
   DEB2(printk("iic_ppc405_handler: in interrupt handler\n"));
   // Read status register
   ret = readb(0xef600508);
   DEB2(printk("iic_ppc405_handler: status = %x\n", ret));
   // Clear status register.  See IBM PPC 405 reference manual for details
   writeb(0x0a, 0xef600508);
   wake_up_interruptible(&iic_wait);
}


//
// Description: This function is very hardware dependent.  First, we lock
// the region of memory where out registers exist.  Next, we request our
// interrupt line and register its associated handler.  Our IIC peripheral
// uses interrupt number 2, as specified by the 405 reference manual.
//
static int iic_hw_resrc_init(void)
{
//	this is not a pci io region, don't use request_region()
//	if (check_region(gpi.iic_base, 17) < 0 ) {
// 	   return -ENODEV;
//	} else {
//	   request_region(gpi.iic_base, 17, "i2c (i2c bus adapter)");
//	}
	if (gpi.iic_irq > 0) {
	   if (request_irq(gpi.iic_irq, iic_ppc405_handler, 0, "PPC405 IIC", 0) < 0) {
	      printk(KERN_ERR "iic_hw_resrc_init: Request irq%d failed\n", gpi.iic_irq);
	      gpi.iic_irq = 0;
	   } else
	      DEB3(printk("iic_hw_resrc_init: Enabled interrupt\n"));
	      enable_irq(gpi.iic_irq);
	}
	return 0;
}


//
// Description: Release irq and memory
//
static void iic_ppc405_release(void)
{
	if (gpi.iic_irq > 0) {
		disable_irq(gpi.iic_irq);
		free_irq(gpi.iic_irq, 0);
	}
	release_region(gpi.iic_base , 2);
}


//
// Description: Does nothing
//
static int iic_ppc405_reg(struct i2c_client *client)
{
	return 0;
}


//
// Description: Does nothing
//
static int iic_ppc405_unreg(struct i2c_client *client)
{
	return 0;
}


//
// Description: If this compiled as a module, then increment the count
//
static void iic_ppc405_inc_use(struct i2c_adapter *adap)
{
#ifdef MODULE
	MOD_INC_USE_COUNT;
#endif
}


//
// Description: If this is a module, then decrement the count
//
static void iic_ppc405_dec_use(struct i2c_adapter *adap)
{
#ifdef MODULE
	MOD_DEC_USE_COUNT;
#endif
}


/* ------------------------------------------------------------------------
 * Encapsulate the above functions in the correct operations structure.
 * This is only done when more than one hardware adapter is supported.
 */
static struct i2c_algo_iic_data iic_ppc405_data = {
	NULL,
	iic_ppc405_setbyte,
	iic_ppc405_getbyte,
	iic_ppc405_getown,
	iic_ppc405_getclock,
	iic_ppc405_waitforpin,
	80, 80, 100,		/*	waits, timeout */
};

static struct i2c_adapter iic_ppc405_ops = {
	"PPC405 IIC adapter",
	I2C_HW_I_IIC,
	NULL,
	&iic_ppc405_data,
	iic_ppc405_inc_use,
	iic_ppc405_dec_use,
	iic_ppc405_reg,
	iic_ppc405_unreg,
};


//
// Description: Called when the module is loaded.  This function starts the
// cascade of calls up through the heirarchy of i2c modules (i.e. up to the
//  algorithm layer and into to the core layer)
//
static int __init iic_ppc405_init(void) 
{

	struct iic_ppc405 *piic = &gpi;


//	printk(KERN_INFO "Infinite loop\n");
//	// Soft reset
//	writeb(0x1, 0xef60050f);
//	while(1) {
//	   // Set SDA line low
//	   writeb(0x8, 0xef600510);
//	}

	printk(KERN_INFO "iic_ppc405_init: PPC 405 iic adapter module version %s (%s)\n", I2C_VERSION, I2C_DATE);
	if (base == 0)
		piic->iic_base = DEFAULT_BASE;
	else
		piic->iic_base = base;

	if (irq == 0)
		piic->iic_irq = DEFAULT_IRQ;
	else
		piic->iic_irq = irq;

	if (clock == 0)
		piic->iic_clock = DEFAULT_CLOCK;
	else
		piic->iic_clock = clock;

	if (own == 0)
		piic->iic_own = DEFAULT_OWN;
	else
		piic->iic_own = own;

	iic_ppc405_data.data = (void *)piic;
#if (LINUX_VERSION_CODE >= 0x020301)
	init_waitqueue_head(&iic_wait);
#endif
	if (iic_hw_resrc_init() == 0) {
		if (i2c_iic_add_bus(&iic_ppc405_ops) < 0)
			return -ENODEV;
	} else {
		return -ENODEV;
	}
	printk(KERN_INFO "iic_ppc405_init: found device at %#x.\n", piic->iic_base);
	return 0;
}


static void iic_ppc405_exit(void)
{
	i2c_iic_del_bus(&iic_ppc405_ops);
        iic_ppc405_release();
}

EXPORT_NO_SYMBOLS;

//
// If modules is NOT defined when this file is compiled, then the MODULE_*
// macros will resolve to nothing
//
MODULE_AUTHOR("MontaVista Software <www.mvista.com>");
MODULE_DESCRIPTION("I2C-Bus adapter routines for PPC 405 IIC bus adapter");

MODULE_PARM(base, "i");
MODULE_PARM(irq, "i");
MODULE_PARM(clock, "i");
MODULE_PARM(own, "i");
MODULE_PARM(i2c_debug,"i");


//
// Description: Called when module is loaded or when kernel is intialized.
// If MODULES is defined when this file is compiled, then this function will
// resolve to init_module (the function called when insmod is invoked for a
// module).  Otherwise, this function is called early in the boot, when the
// kernel is intialized.  Check out /include/init.h to see how this works.
//
module_init(iic_ppc405_init);



//
// Description: Resolves to module_cleanup when MODULES is defined.
//
module_exit(iic_ppc405_exit); 
