// ****************************************************************************
//  Project:        GUYMAGER
// ****************************************************************************
//  Programmer:     Guy Voncken
//                  Police Grand-Ducale
//                  Service de Police Judiciaire
//                  Section Nouvelles Technologies
// ****************************************************************************
//  Module:         Acquisition dialog
// ****************************************************************************

#include <QtGui>

#include "common.h"
#include "compileinfo.h"
#include "config.h"
#include "qtutil.h"
#include "main.h"
#include "dlgdirsel.h"
#include "dlgacquire.h"
#include "dlgacquire_private.h"

// -----------------------------
//           Constants
// -----------------------------

const char *DLG_ACQUIRE_DEFAULT_EMPTY_FILENAME  = "Out";  // Default image file name if - after removing special chars - the file name is empty.
const char *DLGACQUIRE_PROPERTY_SENDER_LINEEDIT = "SenderLineEdit";

// -----------------------------
//           Classes
// -----------------------------

class t_DlgAcquireLocal
{
   public:
      QRadioButton    *pRadioButtonFormatDD;
      QRadioButton    *pRadioButtonFormatEWF;
      QRadioButton    *pRadioButtonFormatAFF;

      QList<QWidget*>   EwfWidgetsList;                  // Used to comfortably enable / disable EWF related entry fields and labels

      QCheckBox       *pCheckBoxCalcHashes;
      QCheckBox       *pCheckBoxVerifySource;

      QPushButton     *pButtonOk;
      QPushButton     *pButtonCancel;

      QFileDialog     *pDlg;

      t_pDevice        pDevice;
};

static t_File::Format DlgAcquireLastUsedFormat = t_File::NotSet;  // For remembering the format that was used for the last acquisition

// -----------------------------
//    Field utility functions
// -----------------------------

static t_pCfgDlgAcquireField DlgAcquireGetField (const QString &Name)
{
   t_pCfgDlgAcquireFields pDlgAcquireFields;
   t_pCfgDlgAcquireField  pDlgAcquireField;
   int i;

   CHK_EXIT (CfgGetDlgAcquireFields (&pDlgAcquireFields))

   for (i=0; i<pDlgAcquireFields->count(); i++)
   {
      pDlgAcquireField = pDlgAcquireFields->at(i);
      if (pDlgAcquireField->FieldName.compare(Name, Qt::CaseInsensitive)==0)
         return pDlgAcquireField;
   }
   LOG_ERROR ("DlgAcquire field %s not found", QSTR_TO_PSZ(Name))
   return NULL;
}


static t_pDlgAcquireLineEdit DlgAcquireGetLineEdit (const QString &Name)
{
   return DlgAcquireGetField(Name)->pLineEdit;
}


static APIRET DlgAcquireResolveSpecialSequences (t_pDevice pDevice, bool ConsiderFields, const QString &In, QString &Out)
{
   t_pCfgDlgAcquireFields pDlgAcquireFields;
   t_pCfgDlgAcquireField  pDlgAcquireField;
   QDateTime               Now = QDateTime::currentDateTime();
   int                     i;
   QString                 Search;

   // Replace date/time values
   // ------------------------
   Out = In;
   Out.replace ("%d%"   , Now.toString("d"   ));  // Special sequences
   Out.replace ("%dd%"  , Now.toString("dd"  ));
   Out.replace ("%ddd%" , Now.toString("ddd" ));
   Out.replace ("%dddd%", Now.toString("dddd"));
   Out.replace ("%M%"   , Now.toString("M"   ));
   Out.replace ("%MM%"  , Now.toString("MM"  ));
   Out.replace ("%MMM%" , Now.toString("MMM" ));
   Out.replace ("%MMMM%", Now.toString("MMMM"));
   Out.replace ("%yy%"  , Now.toString("yy"  ));
   Out.replace ("%yyyy%", Now.toString("yyyy"));
   Out.replace ("%h%"   , Now.toString("h"   ));
   Out.replace ("%hh%"  , Now.toString("hh"  ));
   Out.replace ("%m%"   , Now.toString("m"   ));
   Out.replace ("%mm%"  , Now.toString("mm"  ));
   Out.replace ("%s%"   , Now.toString("s"   ));
   Out.replace ("%ss%"  , Now.toString("ss"  ));
   Out.replace ("%AP%"  , Now.toString("AP"  ));
   Out.replace ("%ap%"  , Now.toString("ap"  ));

   // Replace device values
   // ---------------------
   QString SizeHuman = t_Device::GetSizeHumanFrac (pDevice, true, 0, 10000).toString();
   SizeHuman.replace (MainGetpNumberLocale()->groupSeparator(), "");

   Out.replace ("%serial%" , pDevice->SerialNumber);
   Out.replace ("%model%"  , pDevice->Model       );
   Out.replace ("%size%"   , SizeHuman            );
   Out.replace ("%version%", pCompileInfoVersion  );

   // Replace field values
   // --------------------
   if (ConsiderFields)
   {
      CHK_EXIT (CfgGetDlgAcquireFields (&pDlgAcquireFields))
      for (i=0; i<pDlgAcquireFields->count(); i++)
      {
         pDlgAcquireField = pDlgAcquireFields->at(i);
         Search = "%" + pDlgAcquireField->FieldName + "%";
         Out.replace (Search, pDlgAcquireField->pLineEdit->text());
      }
   }

   return NO_ERROR;
}


static APIRET DlgAcquireGetFieldValue (t_pDevice pDevice, t_pCfgDlgAcquireField pDlgAcquireField, QString &Set)
{
   switch (pDlgAcquireField->EntryMode)
   {
      case CFG_ENTRYMODE_HIDE:
      case CFG_ENTRYMODE_SHOWLAST:
         Set = pDlgAcquireField->LastEnteredValue;
         if (Set.isNull())
         {
            Set = pDlgAcquireField->DefaultValue;
            CHK (DlgAcquireResolveSpecialSequences (pDevice, false, pDlgAcquireField->DefaultValue, Set))
         }
         break;

      case CFG_ENTRYMODE_SHOWDEFAULT:
         Set = pDlgAcquireField->DefaultValue;
         CHK (DlgAcquireResolveSpecialSequences (pDevice, false, pDlgAcquireField->DefaultValue, Set))
         break;

      default:
         CHK (ERROR_DLGACQUIRE_INVALID_ENTRYMODE)
   }

   return NO_ERROR;
}

// -----------------------------
//         t_DlgAcquire
// -----------------------------

APIRET t_DlgAcquire::AddField (t_pDevice pDevice, t_pCfgDlgAcquireField pField, QGridLayout *pLayout, int *pRow)
{
   t_pDlgAcquireLineEdit  pLineEdit     = NULL;
   QLabel                *pLabel        = NULL;
   QPushButton           *pButtonBrowse = NULL;
   QString                 Set;
   QSize                   MinButtonSize;

   pLabel    = new QLabel (tr(QSTR_TO_PSZ(pField->FieldName)), this);
   pLineEdit = new t_DlgAcquireLineEdit (this, pField->FieldName);
   if ((pField->DirField) && (pField->EntryMode != CFG_ENTRYMODE_HIDE))
      pButtonBrowse = new QPushButton (tr("...", "The directory browse button"), this);

   pField->pLineEdit = pLineEdit;
   if (pField->EntryMode == CFG_ENTRYMODE_HIDE)
   {
      pLabel   ->hide();
      pLineEdit->hide();
   }
   else
   {
      if (pButtonBrowse)
      {
         pLayout->addWidget (pLabel       , *pRow, 0);
         pLayout->addWidget (pButtonBrowse, *pRow, 1);
         MinButtonSize = pButtonBrowse->minimumSize();
         MinButtonSize.setWidth(20);
         pButtonBrowse->setSizePolicy (QSizePolicy(QSizePolicy::Ignored, QSizePolicy::Preferred));   // Allow to shrink horizontal size below preferred minimum
         pButtonBrowse->setMinimumSize(MinButtonSize);
      }
      else
      {
         pLayout->addWidget (pLabel, *pRow, 0, 1, 2);        // if there's no button then use the first 2 columns for the label
      }
      pLayout->addWidget (pLineEdit, *pRow, 2);
      if (pField->EwfField)
      {
         pOwn->EwfWidgetsList.append (pLabel   );
         pOwn->EwfWidgetsList.append (pLineEdit);
      }
      (*pRow)++;
   }
   CHK_EXIT (DlgAcquireGetFieldValue (pDevice, pField, Set))
   if (pField->DirField)
   {
      pLineEdit->setReadOnly (true);
      if (!Set.endsWith ("/"))
         Set += "/";
   }
   pLineEdit->setText (Set);
   CHK_QT_EXIT (connect (pLineEdit, SIGNAL (SignalTextEdited (t_DlgAcquireLineEdit *, const QString &)),
                              this, SLOT   (SlotTextEdited   (t_DlgAcquireLineEdit *, const QString &))))
   if (pButtonBrowse)
   {
      (void) pButtonBrowse->setProperty (DLGACQUIRE_PROPERTY_SENDER_LINEEDIT, qVariantFromValue ((void *)pLineEdit));
      CHK_QT_EXIT (connect (pButtonBrowse, SIGNAL (released()), this, SLOT(SlotBrowse())))
   }

   if (pField->DstField)
      CHK_QT_EXIT (connect (pLineEdit, SIGNAL (textChanged(const QString &)), this, SLOT(UpdateButtonState(const QString &))))

   return NO_ERROR;
}


t_DlgAcquire::t_DlgAcquire ()
{
   CHK_EXIT (ERROR_DLGACQUIRE_CONSTRUCTOR_NOT_SUPPORTED)
} //lint !e1401 pOwn not initialised

t_DlgAcquire::t_DlgAcquire (t_pDevice pDevice, QWidget *pParent, Qt::WFlags Flags)
   :QDialog (pParent, Flags)
{
   static bool             Initialised = false;
   t_pCfgDlgAcquireFields pDlgAcquireFields;
   t_pCfgDlgAcquireField  pDlgAcquireField;
   QVBoxLayout           *pLayout;
   QLabel                *pLabel;
   QString                 DefaultFilename;
   QString                 Path;
   QString                 Str;
   QString                 ButtonTextDD;
   QString                 ButtonTextEWF;
   QString                 ButtonTextAFF;
   QString                 Set;
   int                     Row;
   int                     i;

   if (!Initialised)
   {
      Initialised = true;
      CHK_EXIT (TOOL_ERROR_REGISTER_CODE (ERROR_DLGACQUIRE_CONSTRUCTOR_NOT_SUPPORTED))
      CHK_EXIT (TOOL_ERROR_REGISTER_CODE (ERROR_DLGACQUIRE_UNKNOWN_FILEDIALOG_SIZE))
      CHK_EXIT (TOOL_ERROR_REGISTER_CODE (ERROR_DLGACQUIRE_INVALID_ENTRYMODE))
      CHK_EXIT (TOOL_ERROR_REGISTER_CODE (ERROR_DLGACQUIRE_INVALID_FORMAT))
   }

   setWindowTitle (tr ("Acquisition parameters for %1", "Dialog title, %1 is the device (for instance /dev/hdc)") .arg(pDevice->LinuxDevice));

   pOwn    = new t_DlgAcquireLocal;
   pOwn->pDevice = pDevice;

   pLayout = new QVBoxLayout (this);
   setLayout (pLayout);

//   pLayout->addWidget (new QLabel(("<b>" + tr("Acquisition parameters for %1") + "</b>") .arg(pDevice->LinuxDevice) , this));

   // Format box (with EWF fields)
   // ----------------------------
   QGroupBox   *pGroupBoxFormat = new QGroupBox (tr("File format"), this);
   QVBoxLayout *pLayoutFormat   = new QVBoxLayout;
   QGridLayout *pLayoutEwf      = new QGridLayout ();

   pGroupBoxFormat->setLayout (pLayoutFormat);
   pLayout->addWidget (pGroupBoxFormat);
   pLayout->addStretch (1);

   CHK_EXIT (t_File::GetFormatDescription (t_File::DD , Str))            ButtonTextDD  = "&" + Str;
   CHK_EXIT (t_File::GetFormatDescription (t_File::EWF, Str))            ButtonTextEWF = "&" + Str;
   CHK_EXIT (t_File::GetFormatDescription (t_File::AFF, Str))            ButtonTextAFF = "&" + Str;
   CHK_EXIT (t_File::GetFormatExtension   (t_File::DD , NULL, &Str))     ButtonTextDD += " " + tr("(file extension %1)") .arg (Str);
   CHK_EXIT (t_File::GetFormatExtension   (t_File::EWF, NULL, &Str))     ButtonTextEWF+= " " + tr("(file extension %1)") .arg (Str);
   CHK_EXIT (t_File::GetFormatExtension   (t_File::AFF, NULL, &Str))     ButtonTextAFF+= " " + tr("(file extension %1)") .arg (Str);

   pOwn->pRadioButtonFormatDD  = new QRadioButton (ButtonTextDD , this);
   pOwn->pRadioButtonFormatEWF = new QRadioButton (ButtonTextEWF, this);
   pOwn->pRadioButtonFormatAFF = new QRadioButton (ButtonTextAFF, this);

   pLayoutFormat->addWidget  (pOwn->pRadioButtonFormatDD );
   pLayoutFormat->addWidget  (pOwn->pRadioButtonFormatEWF);
   pLayoutFormat->addWidget  (pOwn->pRadioButtonFormatAFF);
   pLayoutFormat->addLayout  (pLayoutEwf);

   CHK_EXIT (CfgGetDlgAcquireFields (&pDlgAcquireFields))

   Row = 0;
   for (i=0; i<pDlgAcquireFields->count(); i++)
   {
      pDlgAcquireField = pDlgAcquireFields->at(i);
      if (pDlgAcquireField->EwfField)
         CHK_EXIT (AddField (pDevice, pDlgAcquireField, pLayoutEwf, &Row))
   }
   pLayoutEwf->setColumnMinimumWidth (0, 20);

   // Destination box
   // ---------------
   QGroupBox   *pGroupBoxDest = new QGroupBox(tr("Destination"), this);
   QGridLayout *pLayoutDest   = new QGridLayout ();
   pGroupBoxDest->setLayout (pLayoutDest);
   pLayout->addWidget (pGroupBoxDest);
   pLayout->addStretch (1);

   Row = 0;
   if (CONFIG(WriteToDevNull))
   {
      pLabel = new QLabel(tr("Configuration flag WriteToDevNull is set!"), this);
      pLayoutDest->addWidget (pLabel, Row++, 0);
   }
   //lint -restore

   for (i=0; i<pDlgAcquireFields->count(); i++)
   {
      pDlgAcquireField = pDlgAcquireFields->at(i);
      if (pDlgAcquireField->DstField)
         CHK_EXIT (AddField (pDevice, pDlgAcquireField, pLayoutDest, &Row))
   }

   // Hash
   // ----
   QGroupBox   *pGroupBoxHash = new QGroupBox(tr("Hash computation"), this);
   QGridLayout *pLayoutHash   = new QGridLayout ();
   pGroupBoxHash->setLayout (pLayoutHash);
   pLayout->addWidget (pGroupBoxHash);
   pLayout->addStretch (1);

   pOwn->pCheckBoxCalcHashes   = new QCheckBox (tr("Calculate hashes (MD5 and SHA-256)"), this);
   pOwn->pCheckBoxVerifySource = new QCheckBox (tr("Re-read source after acquisition for verification (takes twice as long)"), this);
   pLayoutHash->addWidget (pOwn->pCheckBoxCalcHashes  , 0, 0);
   pLayoutHash->addWidget (pOwn->pCheckBoxVerifySource, 1, 0);

   // Dialog buttons
   // --------------
   QHBoxLayout *pLayoutButtons = new QHBoxLayout ();
   pLayout->addLayout (pLayoutButtons);

   pOwn->pButtonOk     = new QPushButton (QObject::tr("Ok"    ), this);
   pOwn->pButtonCancel = new QPushButton (QObject::tr("Cancel"), this);
   pLayoutButtons->addWidget (pOwn->pButtonOk    );
   pLayoutButtons->addWidget (pOwn->pButtonCancel);
   pOwn->pButtonOk->setDefault (true);

   // Set other defaults
   // ------------------
   if (DlgAcquireLastUsedFormat == t_File::NotSet)
      DlgAcquireLastUsedFormat = (t_File::Format) CONFIG (DefaultFormat);

   switch (DlgAcquireLastUsedFormat)
   {
      case t_File::DD : pOwn->pRadioButtonFormatDD ->setChecked (true); break;
      case t_File::EWF: pOwn->pRadioButtonFormatEWF->setChecked (true); break;
      case t_File::AFF: pOwn->pRadioButtonFormatAFF->setChecked (true); break;
      default         : CHK_EXIT (ERROR_DLGACQUIRE_INVALID_FORMAT)
   }
   pOwn->pCheckBoxCalcHashes->setChecked (true);

   UpdateCheckboxState (pOwn->pCheckBoxCalcHashes->isChecked() ? Qt::Checked : Qt::Unchecked);
   UpdateButtonState   ();

   // Connections
   // -----------
   CHK_QT_EXIT (connect (pOwn->pCheckBoxCalcHashes, SIGNAL (stateChanged(int)), this, SLOT(UpdateCheckboxState(int))))

   CHK_QT_EXIT (connect (pOwn->pRadioButtonFormatDD , SIGNAL (released()), this, SLOT(UpdateFieldState())))
   CHK_QT_EXIT (connect (pOwn->pRadioButtonFormatEWF, SIGNAL (released()), this, SLOT(UpdateFieldState())))
   CHK_QT_EXIT (connect (pOwn->pRadioButtonFormatAFF, SIGNAL (released()), this, SLOT(UpdateFieldState())))

   CHK_QT_EXIT (connect (pOwn->pButtonOk    , SIGNAL (released()), this, SLOT(SlotAccept())))
   CHK_QT_EXIT (connect (pOwn->pButtonCancel, SIGNAL (released()), this, SLOT(reject    ())))
}

void t_DlgAcquire::UpdateCheckboxState (int State)
{
   pOwn->pCheckBoxVerifySource->setEnabled (State == Qt::Checked);
}

void t_DlgAcquire::UpdateButtonState (const QString & /*NewText*/)
{
   t_pCfgDlgAcquireFields pFields;
   t_pCfgDlgAcquireField  pField;
   bool                    Enabled=true;
   int                     i;

   CHK_EXIT (CfgGetDlgAcquireFields (&pFields))

   for (i=0; i<pFields->count() && Enabled; i++)
   {
      pField = pFields->at(i);
      if (pField->DstField)
         Enabled = !pField->pLineEdit->text().isEmpty();
   }

   pOwn->pButtonOk->setEnabled (Enabled);
}

void t_DlgAcquire::UpdateFieldState (void)
{
   QWidget *pWidget;
   bool      Enabled;
   int       i;

   Enabled = (pOwn->pRadioButtonFormatEWF->isChecked() ||
              pOwn->pRadioButtonFormatAFF->isChecked());
   for (i = 0;
        i < pOwn->EwfWidgetsList.count();
        i++)
   {
      pWidget = pOwn->EwfWidgetsList.at(i);
      pWidget->setEnabled (Enabled);
   }
}

void t_DlgAcquire::SlotBrowse (void)
{
   t_pDlgAcquireLineEdit pLineEdit;
   QString                Path;
   bool                   OkPressed;

   pLineEdit = (t_pDlgAcquireLineEdit) sender()->property (DLGACQUIRE_PROPERTY_SENDER_LINEEDIT).value<void *>();

   if (CONFIG(UseFileDialogFromQt))
   {
      Path = QFileDialog::getExistingDirectory(this, tr("Select destination directory", "Dialog title"), pLineEdit->text(), QFileDialog::ShowDirsOnly);
      OkPressed = !Path.isNull();
   }
   else
   {
      Path = pLineEdit->text();
      t_DlgDirSel Dlg (&Path, this);

      switch (CONFIG(FileDialogSize))
      {
         case CFG_STARTUPSIZE_STANDARD  :                       break;
         case CFG_STARTUPSIZE_FULLSCREEN: Dlg.showFullScreen(); break;
         case CFG_STARTUPSIZE_MAXIMIZED : Dlg.showMaximized (); break;
         case CFG_STARTUPSIZE_MANUAL    : CHK_EXIT (QtUtilSetGeometryCentered (&Dlg, CONFIG(FileDialogSizeManualDx), CONFIG(FileDialogSizeManualDy))); break;
         default                        : CHK_EXIT (ERROR_DLGACQUIRE_UNKNOWN_FILEDIALOG_SIZE)
      }
      OkPressed = (Dlg.exec() == QDialog::Accepted);
   }

   if (OkPressed)
   {
      if (!Path.endsWith ("/"))
         Path += "/";
      pLineEdit->setText (Path); // setText doesn't emit a textEdited signal (that's completely right  in order to prevent endless
      pLineEdit->TextUpdated (); // update signals) and so we have the LineEdit emit a SignalTextEdit, thus triggering field updates.
   }
}

void t_DlgAcquire::SlotTextEdited (t_pDlgAcquireLineEdit pLineEdit, const QString &/*NewVal*/)
{
   t_pCfgDlgAcquireRules pDlgAcquireRules;
   t_pCfgDlgAcquireRule  pDlgAcquireRule;
   t_pDlgAcquireLineEdit pLineEditDest;
   int                    i;
   QString                Set;

//   printf ("\nNewVal in %s: %s", QSTR_TO_PSZ(pLineEdit->Name), QSTR_TO_PSZ(NewVal));

   CHK_EXIT (CfgGetDlgAcquireRules (&pDlgAcquireRules))

   for (i=0; i<pDlgAcquireRules->count(); i++)
   {
      pDlgAcquireRule = pDlgAcquireRules->at(i);
      if (pDlgAcquireRule->TriggerFieldName.compare (pLineEdit->Name, Qt::CaseInsensitive) == 0)
      {
         pLineEditDest = DlgAcquireGetLineEdit (pDlgAcquireRule->DestFieldName);

         CHK_EXIT (DlgAcquireResolveSpecialSequences (pOwn->pDevice, true, pDlgAcquireRule->Value, Set))
         pLineEditDest->setText(Set);
      }
   }
}

static APIRET DlgAcquireCheckFilename (const QString &In, bool &Clean, QString &Out)
{
   unsigned char Ch;
   int           i;
   bool          Ok;
   QString       SpecialChars = CONFIG (SpecialFilenameChars);

   Clean = true;
   Out   = "";
   for (i=0; i<In.length(); i++)
   {
      Ch = In[i].toAscii();
      Ok = ((Ch >= '0') && (Ch <= '9')) ||
           ((Ch >= 'a') && (Ch <= 'z')) ||
           ((Ch >= 'A') && (Ch <= 'Z')) ||
            (Ch == '_') || SpecialChars.contains(Ch);
      if (Ok)
           Out += Ch;
      else Clean = false;
   }
   if (Out.isEmpty())
      Out = DLG_ACQUIRE_DEFAULT_EMPTY_FILENAME;

   return NO_ERROR;
}


APIRET t_DlgAcquire::CheckWriteAccess (const QString &Path, const QString &Filename, bool &Ok)
{
   FILE       *pFile;
   const char *pStr = "Test file created by Guymager for checking write access.\r\nYou may delete this file";
   const int    StrLen = strlen (pStr);
   QString      TestFileName = Path + Filename + ".test";
   int          wr;

   pFile = fopen (QSTR_TO_PSZ(TestFileName), "w");
   Ok = (pFile != NULL);
   if (!Ok)
   {
      LOG_INFO ("Opening test file %s failed.", QSTR_TO_PSZ(TestFileName))
   }
   else
   {
      if ((wr = fprintf (pFile, "%s", pStr)) != StrLen)
      {
         Ok = false;
         LOG_INFO ("Writing to test file %s failed, %d of %d bytes written.", QSTR_TO_PSZ(TestFileName), wr, StrLen)
      }

      if (fclose (pFile) != 0)
      {
         Ok = false;
         LOG_INFO ("Closing test file %s failed", QSTR_TO_PSZ(TestFileName))
      }

      if (!QFile::remove (TestFileName))
      {
         Ok = false;
         LOG_INFO ("Removing the test file %s failed.", QSTR_TO_PSZ(TestFileName))
      }
   }

   if (!Ok)
      QMessageBox::information (this, tr ("Access denied", "Dialog title"),
                                      tr ("Guymager cannot write to the directory"
                                          "\n\t%1"
                                          "\nThis may be due to insufficient access rights. Please choose another directory.")
                                          .arg(Path), QMessageBox::Ok);
   return NO_ERROR;
}


void t_DlgAcquire::SlotAccept (void)
{
   t_Device::t_Acquisition      Acquisition;
   t_pDlgAcquireLineEdit       pLineEditDestImageFilename;
   t_pDlgAcquireLineEdit       pLineEditDestInfoFilename ;
   QMessageBox::StandardButton  Button;
   QString                      ImageFilenameCorrected;
   QString                      InfoFilenameCorrected;
   bool                         ImageFilenameClean;
   bool                         InfoFilenameClean;

   CHK_EXIT (GetParameters (Acquisition, false))

   // Check for strange characters in filenames
   // -----------------------------------------

   pLineEditDestImageFilename = DlgAcquireGetField (CFG_DLGACQUIRE_DEST_IMAGEFILENAME)->pLineEdit;
   pLineEditDestInfoFilename  = DlgAcquireGetField (CFG_DLGACQUIRE_DEST_INFOFILENAME )->pLineEdit;

   CHK_EXIT (DlgAcquireCheckFilename (pLineEditDestImageFilename->text(), ImageFilenameClean, ImageFilenameCorrected))
   CHK_EXIT (DlgAcquireCheckFilename (pLineEditDestInfoFilename ->text(), InfoFilenameClean,  InfoFilenameCorrected ))

   if (!ImageFilenameClean || !InfoFilenameClean)
   {
      LOG_INFO ("Unallowed characters in filenames found")
      Button = QMessageBox::question (this, tr ("Special characters", "Dialog title"),
                                            tr ("The filenames contain special characters which are not allowed. Guymager suggests the "
                                                "following changes:"
                                                "\n\tImage filename: %1"
                                                "\n\tInfo filename: %2"
                                                "\nDo you accept these changes?") .arg(ImageFilenameCorrected) .arg(InfoFilenameCorrected),
                                                QMessageBox::Yes | QMessageBox::No, QMessageBox::Yes);
      if (Button == QMessageBox::Yes)
      {
         LOG_INFO ("User accepts filename changes")
         pLineEditDestImageFilename->setText(ImageFilenameCorrected);
         pLineEditDestInfoFilename ->setText(InfoFilenameCorrected );
      }
      else
      {
         LOG_INFO ("User rejects filename changes")
         return;
      }
   }

   // Check if file can be created on destination path
   // ------------------------------------------------
   bool Ok;

   CHK_EXIT (t_DlgAcquire::CheckWriteAccess (Acquisition.InfoPath , Acquisition.InfoFilename , Ok))
   if (Ok)
      CHK_EXIT (t_DlgAcquire::CheckWriteAccess (Acquisition.ImagePath, Acquisition.ImageFilename, Ok))
   if (!Ok)
      return;

   // Check if image file already exists
   // ----------------------------------

   QDir    Dir (Acquisition.ImagePath);
   QString ExtensionImage;
   bool    Empty;
   QString NameFilter;

   CHK_EXIT (t_File::GetFormatExtension (Acquisition.Format, &ExtensionImage))
   NameFilter = Acquisition.ImageFilename + ExtensionImage;
   Empty = Dir.entryInfoList (QStringList(NameFilter), QDir::Files, QDir::Name).isEmpty();
   if (Empty)
   {
      Dir.setPath (Acquisition.InfoPath);
      NameFilter = Acquisition.InfoFilename + t_File::pExtensionInfo;
      Empty = Dir.entryInfoList (QStringList(NameFilter), QDir::Files, QDir::Name).isEmpty();
   }

   if (!Empty)
   {
      LOG_INFO ("Images files already exist")
      Button = QMessageBox::question (this, tr ("Images files exist", "Dialog title"),
                                            tr ("The image files already exist. Do you want to overwrite them?"),
                                                QMessageBox::Yes | QMessageBox::No, QMessageBox::No);
      if (Button == QMessageBox::Yes)
      {
         LOG_INFO ("User accepts to overwrite existing image")
      }
      else
      {
         LOG_INFO ("User rejects to overwrite existing image")
         return;
      }
   }
   accept();
}


APIRET t_DlgAcquire::GetParameters (t_Device::t_Acquisition &Acquisition, bool RememberLastUsedValues)
{
   t_pCfgDlgAcquireField pDlgAcquireField;

   Acquisition.CalcHashes   = pOwn->pCheckBoxCalcHashes  ->isChecked();
   Acquisition.VerifySource = pOwn->pCheckBoxVerifySource->isChecked();
   if      (pOwn->pRadioButtonFormatDD ->isChecked()) Acquisition.Format = t_File::DD;
   else if (pOwn->pRadioButtonFormatEWF->isChecked()) Acquisition.Format = t_File::EWF;
   else                                               Acquisition.Format = t_File::AFF;
   if (RememberLastUsedValues)
      DlgAcquireLastUsedFormat = Acquisition.Format;

   #define COPY_VALUE(Acq, FieldName)                    \
      pDlgAcquireField = DlgAcquireGetField (FieldName); \
      Acq = pDlgAcquireField->pLineEdit->text();         \
      if (RememberLastUsedValues)                        \
         pDlgAcquireField->LastEnteredValue = Acq;

   COPY_VALUE (Acquisition.CaseNumber    , CFG_DLGACQUIRE_EWF_CASENUMBER     )
   COPY_VALUE (Acquisition.EvidenceNumber, CFG_DLGACQUIRE_EWF_EVIDENCENUMBER )
   COPY_VALUE (Acquisition.Examiner      , CFG_DLGACQUIRE_EWF_EXAMINER       )
   COPY_VALUE (Acquisition.Description   , CFG_DLGACQUIRE_EWF_DESCRIPTION    )
   COPY_VALUE (Acquisition.Notes         , CFG_DLGACQUIRE_EWF_NOTES          )
   COPY_VALUE (Acquisition.ImagePath     , CFG_DLGACQUIRE_DEST_IMAGEDIRECTORY)
   COPY_VALUE (Acquisition.ImageFilename , CFG_DLGACQUIRE_DEST_IMAGEFILENAME )
   COPY_VALUE (Acquisition.InfoPath      , CFG_DLGACQUIRE_DEST_INFODIRECTORY )
   COPY_VALUE (Acquisition.InfoFilename  , CFG_DLGACQUIRE_DEST_INFOFILENAME  )
   #undef COPY_VALUE

   return NO_ERROR;
}

t_DlgAcquire::~t_DlgAcquire ()
{
   delete pOwn;
}

