/* GNOME Transcript
 * Copyright (C) 1999-2000 the Free Software Foundation
 * Authors : Matias Mutchinick
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <gnome.h>
#include <gtrans_ifase/gtrans_ifase.h>
#include "gtrans_error_dialog.h"
#include "gtrans_db_editor.h"
#include "gtrans_table_editor.h"





/**
 * gtrans_edit_table_error_msg:
 * @serv_err : The error message returned by the server
 * @tab_name : The name of the table being edited
 * @row      : The number of the row being edited
 *
 * Return an error message string.
 */
gchar *
gtrans_edit_table_error_msg(gchar *serv_err,
			    gchar *tab_name,
			    gint   row)
{
	gchar *msg;
	
	msg = g_strdup_printf("An error ocurred while commiting"
			      " the changes made in the\nrow %i"
			      " to the table '%s'.\nThe error message"
			      " returned by the server is the following"
			      ":\n\n%s\n"
			      ,row,tab_name,serv_err);
	
	return msg;
}





/**
 * gtrans_edit_table_commit_next:
 * @gte : The GTransTableEditor
 *
 * Commit the next change in the @gte.
 */
void
gtrans_edit_table_commit_next(GTransTableEditor *gte)
{
	GTransIFaseResult *res;
	gchar             *qstr, *msg, *serv_err;
	gint               row;


	if (!gtrans_table_editor_can_next(gte))
		return;
	
	qstr = gtrans_table_editor_pop_query(gte);
	
	if (qstr != NULL){
		
		res = gtrans_ifase_exec(gte->conn,qstr);
		if (gtrans_ifase_result_status(res)){
			serv_err = gtrans_ifase_result_error_msg(res);
			row = gtrans_table_editor_next_row(gte);
			msg = gtrans_edit_table_error_msg(serv_err,gte->tname,row);
			gtrans_error_dialog(msg);
			g_free(msg);
			return;
		}
		
		g_free(qstr);
	}
	
	gtrans_table_editor_sync(gte);
}




/**
 * gtrans_edit_table_commit_all:
 * @gte : The GTransTableEditor
 *
 * Commit the all changes made in the @gte.
 */
void
gtrans_edit_table_commit_all(GTransTableEditor *gte)
{
	GTransIFaseResult *res;
	gchar             *qstr, *msg, *serv_err;
	gint               row;
	
	while (gtrans_table_editor_can_next(gte)){
		
		qstr = gtrans_table_editor_pop_query(gte);

		if (qstr != NULL){
			
			res = gtrans_ifase_exec(gte->conn,qstr);
			
			if (gtrans_ifase_result_status(res)){
				serv_err = gtrans_ifase_result_error_msg(res);
				row = gtrans_table_editor_next_row(gte);
				msg = gtrans_edit_table_error_msg(serv_err,gte->tname,row);
				gtrans_error_dialog(msg);
				g_free(msg);
				return;
			}
			
			g_free(qstr);
		}
		
		gtrans_table_editor_sync(gte);
		
		while (gtk_events_pending())
			gtk_main_iteration();
		
		/* If the "stop" signal was emmited */
		if (gtrans_table_editor_interrupt(gte))
			return;
	}
}




/**
 * gtrans_edit_table_close
 * 
 */
void
gtrans_edit_table_close(GTransTableEditor *gte)
{
	gtrans_db_editor_remove_child(GTRANS_DB_EDITOR(gte->owner),GTK_WIDGET(gte));
	gtk_widget_destroy(GTK_WIDGET(gte));
}




/**
 * gtrans_edit_table
 * @gde : The Database Editor
 *
 * Open a table in a GTransTableEditor.
 */
void
gtrans_edit_table(GTransDbEditor  *gde)
{
	GTransIFaseResult  *res;
	GtkWidget          *gteditor;
	gchar              *table;
	
	
	table = gde->table_idata;
	if (!table){
		gtrans_error_dialog(GTransErrorMsg[GTRANS_ERROR_SELECT_TABLE_OPEN]);
		return;
	}
	
	res = gtrans_ifase_exec_table_select_all(gde->conn,table);
	if (gtrans_ifase_result_status(res)){
		gtrans_error_dialog(gtrans_ifase_result_error_msg(res));
		gtrans_ifase_result_free(res);
		return;
	}
	
	gteditor = gtrans_table_editor_new(gde->conn,res,table,GTK_WIDGET(gde));
	gtk_signal_connect_object(GTK_OBJECT(gteditor),"commit_all",
				  GTK_SIGNAL_FUNC(gtrans_edit_table_commit_all),
				  GTK_OBJECT(gteditor));
	
	gtk_signal_connect_object(GTK_OBJECT(gteditor),"commit_next",
				  GTK_SIGNAL_FUNC(gtrans_edit_table_commit_next),
				  GTK_OBJECT(gteditor));

	gtk_signal_connect_object(GTK_OBJECT(gteditor),"close",
				  GTK_SIGNAL_FUNC(gtrans_edit_table_close),
				  GTK_OBJECT(gteditor));
	
	/* Set the 'Db Editor' as the paren of the 'Table Editor' */
	gtrans_db_editor_add_child(gde,gteditor);

	gtk_widget_show(gteditor);
}


