#
# Copyright 2009 Martin Owens
#
# This program is free software: you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation, either version 3 of the License, or
#  (at your option) any later version.
#
#  This program is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program.  If not, see <http://www.gnu.org/licenses/>
#
"""App for looking at changes and confirming them."""

# Import standard python libs
import os
import gtk
import gtk.gdk
import logging

# Various required variables and locations
from GroundControl.gtkviews import (
    TreeView, Window,
    IconManager, GtkApp
)
from GroundControl.bugs import BugSelection
from bzrlib.plugins.gtk.diff import DiffView

change_types = (
    'added',         # (path, id, kind)
    'removed',       # (path, id, kind)
    'renamed',       # (oldpath, newpath, id, kind, text_modified, meta_mod)
    'kind_changed',  # (path, id, old_kind, new_kind)
    'modified',      # (path, id, kind, text_modified, meta_modified)
#    'unmodifed',     # (path, id, kind)
    'unversioned'    # (path, None, kind)
)
change_label = {
    'added'        : _('Added'),
    'removed'      : _('Deleted'),
    'renamed'      : _('Renamed'),
    'kind_changed' : _('Changed'),
    'modified'     : _('Edited'),
    'unversioned'  : _('New'),
    ''             : '',
}
change_icons = IconManager('chicons')


class CommitWindow(Window):
    """Select a project to import."""
    name = 'confirm'

    def  load(self, *args, **kwargs):
        """Load a commit window"""
        self.branch  = kwargs.pop('branch')
        self.project = kwargs.pop('project')
        self.fixing  = kwargs.pop('fixing', None)
        self.wtr     = self.branch.get_workingtree()
        self.prevt   = self.branch.get_oldtree()
        self.slist   = None
        self.child   = None
        self.result  = None
        self.comment = None
        self.newadd  = False
        self.dirs    = {}
        self.fixes   = []
        super(CommitWindow, self).load(*args, **kwargs)
        self.diffview = DiffView()
        #self.diffview.set_trees(self.prevt, self.wtr)
        self.diffview.set_trees(self.wtr, self.prevt)
        self.widget('viewchanges').add(self.diffview)
        self.slist = ChangesView(self.widget('changes'),
            selected=self.selected)
        all_changes = self.branch.get_changes()
        for atype in change_types:
            changes = getattr(all_changes, atype)
            if changes:
                for change in changes:
                    self.add_change(atype, change)
        # Remove any empty directories that aren't changing
        for iter in self.dirs.values():
            if not self.slist.get_item(iter).changes:
                self.slist.remove_item(target_iter=iter)
        # Hide the ignore new files feature if we have no new files.
        if not self.newadd:
            self.widget('ignorenew').hide()
            self.widget('ignorelabel').hide()
        # Auto add the bug we're fixing as expected
        if self.fixing:
            self.tie_bug(self.fixing)
            # If we ever add a feature to enable a user
            # To select what files should be commited, Then
            # disable it here since fixing bugs is incompatable

    def signals(self):
        return {
            'tie_bug' : self.get_bug,
        }

    def add_change(self, atype, change):
        """Return an added tree item node"""
        # Make our change item object for the list
        item   = ChangeItem(self.branch, atype, change)
        parent = self.get_folder(item.stem)
        # We want to track if we have any added files
        if atype == 'unversioned':
            # Don't add items that are going to be ignored anyway
            if self.wtr.is_ignored(item.path):
                return None
            self.newadd = True
        # Add item to visual tree view list
        iter1 = self.slist.add_item( item, parent )
        if item.isdir:
            self.dirs[item.stem] = iter1
        if parent:
            self.slist.expand_item(parent)
        if atype == 'unversioned' and item.isdir:
            # Also add in any sub files
            self.add_sub_files( item.stem )

    def get_folder(self, path):
        """Returns a folder tree item node"""
        if path == '':
            return None
        fdir = os.path.dirname(path)
        full_path = os.path.join(self.branch.path, fdir)
        if fdir and os.path.isdir(full_path):
            if not self.dirs.has_key(fdir):
                parent = self.get_folder(fdir)
                item = ChangeFolder(full_path)
                self.dirs[fdir] = self.slist.add_item( item, parent )
        return self.dirs.get(fdir, None)

    def add_sub_files(self, dir):
        """Add the subfiles of a directory"""
        path = self.branch.path
        full_path = os.path.join(path, dir)
        for res in os.walk(full_path):
            for sfile in (res[1]+res[2]):
                new_path = os.path.join(res[0], sfile).replace(path, '')
                self.add_change('added', (new_path, None, None))

    def is_valid(self):
        """Return true if inputs are correct."""
        return len(self.widget('comments').get_text()) > 2

    def get_bug(self, widget=None):
        """Load up the bug gui and let someone tie a bug to this report"""
        project = None
        if self.project:
            project = str(self.project.pid)
        BugSelection(project=project,
            callback=self.tie_bug,
            parent=self.window,
        )

    def tie_bug(self, bug=None, project=None):
        """Add the bug to our fixes list"""
        if type(bug) not in (str, unicode):
            bug = "lp:%s" % str(bug.id)
        else:
            bug = "lp:%s" % bug
        self.fixes.append(bug)
        # This button will allow us to remove the fixes.
        bugb = gtk.Button()
        bugb.set_label(bug)
        bugb.show()
        bugb.connect('clicked', self.remove_bug)
        self.widget('fixeslist').pack_start(bugb)
        self.widget('fixeslist').reorder_child(bugb, 0)

    def selected(self, item):
        """Show in the diff any items which we want"""
        if not item.isdir and item.type == 'modified':
            logging.debug("Showing diff for %s" % item.stem)
            self.diffview.show_diff([item.stem])
            self.widget("viewchanges").set_position(300)
            self.window.resize(800, 600)
            self.diffview.show()
        else:
            self.diffview.hide()

    def remove_bug(self, widget):
        """It doesn't fix this bug at all!"""
        bug = widget.get_label()
        self.fixes.remove(bug)
        widget.destroy()

    def get_args(self):
        """Do out commit here"""
        return {
            'message'   : self.widget('comments').get_text(),
            'ignore'    : self.widget('ignorenew').get_active(),
            'branch'    : self.branch,
            'fixes'     : self.fixes,
            'file_list' : None,
        }


class CommitChanges(GtkApp):
    """Graphical application for presenting the user with commit changes"""
    gtkfile = 'commit-branch.glade'
    windows = [ CommitWindow ]


class NodeItem(object):
    """Wrap all items in the review"""
    def __init__(self, path, atype=None):
        self._icon   = None
        self.changes = None
        self.path    = path
        self.type    = atype
        self.name    = os.path.basename(self.path)
        self.isdir   = os.path.isdir(self.path)

    def label(self):
        """Return the change type label"""
        return "<i>%s</i>" % self.name

    def icon(self):
        """Icon of the modeification"""
        if not self._icon:
            base = change_icons.get_icon('file').copy()
            if self.isdir:
                base = change_icons.get_icon('folder').copy()
            overlay = change_icons.get_icon(self.type).copy()
            overlay.composite( base, 0, 0, base.props.width, base.props.height,
                0, 0, 1.0, 1.0, gtk.gdk.INTERP_BILINEAR, 128 )
            self._icon = base
        return self._icon


class ChangeItem(NodeItem):
    """Wraps any change item from TreeDelta"""
    def __init__(self, branch, atype, changes):
        self.branch   = branch
        self.selected = False
        if atype == 'renamed':
            self.oldname = changes.pop(0)
        self.stem     = changes[0]
        path = os.path.join(self.branch.path, self.stem)
        super(ChangeItem, self).__init__(path, atype)
        self.changes  = changes
        # Force the correct type for deleted elements.
        if changes[2] == 'directory':
            self.isdir = True

    def set_selected(self, shouldi=True):
        """Choose to confirm the changes of this file"""
        self.selected = shouldi

    def label(self):
        """Return the simple label for this item"""
        if self.type == 'renamed':
            return "%s => %s" % (self.oldname, self.name)
        else:
            return self.name


class ChangeFolder(NodeItem):
    """Wraps a change type"""
    def has_changes(self):
        self.changes = 'folder'


class ChangesView(TreeView):
    """Configures the view for changes."""

    def add_item(self, item, parent=None):
        if parent:
            folder = self.get_item(parent)
            if not folder.changes:
                folder.has_changes()
        return super(ChangesView, self).add_item(item, parent)

    def setup(self):
        """Setup a treeview for showing services"""
        def text_cell_func(cell_layout, renderer, model, item_iter):
            """Render the text of the services tree view"""
            item = model.get_value(item_iter, 0)
            renderer.set_property("markup", item.label())

        def type_cell_func(cell_layout, renderer, model, item_iter):
            """Render the kind of change"""
            item = model.get_value(item_iter, 0)
            if item.type:
                renderer.set_property("markup", change_label[item.type])
            else:
                renderer.set_property("markup", '')

        def icon_cell_func(column, cell, model, item_iter):
            """Reender the icons of the services tree view"""
            item = model.get_value(item_iter, 0)
            cell.set_property("pixbuf", item.icon())
            cell.set_property("visible", True)

        svlist = super(ChangesView, self).setup()
        column = gtk.TreeViewColumn((_("List of Changes")))
        column.set_sizing(gtk.TREE_VIEW_COLUMN_AUTOSIZE)
        #column.set_sizing(gtk.TREE_VIEW_COLUMN_FIXED)
        column.set_expand(True)
        # The icon
        renderer_icon = gtk.CellRendererPixbuf()
        renderer_icon.set_property("ypad", 2)
        renderer_icon.set_property("xpad", 2)
        column.pack_start(renderer_icon, False)
        column.set_cell_data_func(renderer_icon, icon_cell_func)
        # The name
        renderer = gtk.CellRendererText()
        column.pack_start(renderer, True)
        column.set_cell_data_func(renderer, text_cell_func)
        # Add the required coluns to the treeview
        svlist.append_column(column)
        # The kind of change
        column2 = gtk.TreeViewColumn(("Kind"))
        column2.set_sizing(gtk.TREE_VIEW_COLUMN_AUTOSIZE)
        renderer = gtk.CellRendererText()
        column2.pack_start(renderer, True)
        column2.set_cell_data_func(renderer, type_cell_func)
        # Add the required coluns to the treeview
        svlist.append_column(column2)


class RevertWindow(Window):
    """Show a window that asks the user to confirm the revert."""
    name = 'revert'

    def  load(self, branch, *args, **kwargs):
        """Load a revert confirmation window"""
        self.branch  = branch
        super(RevertWindow, self).load(*args, **kwargs)

    def get_args(self):
        """Return the branch to a sucess response"""
        return { 'branch' : self.branch }

class RevertChanges(GtkApp):
    """An application to show some help to the user."""
    gtkfile = 'revert-branch.glade'
    windows = [ RevertWindow ]
