#
# Copyright 2009 Martin Owens
#
# This program is free software: you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation, either version 3 of the License, or
#  (at your option) any later version.
#
#  This program is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program.  If not, see <http://www.gnu.org/licenses/>
#
"""
Small app for selecting Launchpad branches
"""

# Import standard python libs
import gobject
import gtk
import gtk.gdk
import logging

# Various required variables and locations
from GroundControl.gtkviews import (
    GtkApp, ThreadedWindow, IconManager, TreeView
)
from GroundControl.launchpad import (
    LaunchpadProject, LaunchpadBranch, get_launchpad
)

project_icons = IconManager('project')
person_icons = IconManager('person')
MSG_TMPL = "<b><big>%s</big></b>\n\n<i><b>%s</b></i>"
LP_MESSAGE = MSG_TMPL % (_("Logging into Launchpad..."), _("Please Wait..."))
BR_MESSAGE = MSG_TMPL % (_("Loading Available Branches..."),
    _("Loading Branch: %s"))


class SelectionWindow(ThreadedWindow):
    """Select a branch to import."""
    name = 'select_branch'

    def load(self, oid, path, *args, **kwargs):
        """Load this window, initalise everything"""
        self.oid        = oid
        self.branchpath = path
        self.slist      = None
        self._selected  = None
        self._project   = None
        self._launchpad = None
        self.unselected()
        self.branch = kwargs.pop('branch', None)
        self.slist = BranchView(self.widget('branchlist'),
            selected=self.selected,
            unselected=self.unselected)
        self.update_status()
        # Special selection since we're given a workname
        workname  = kwargs.pop('workname', None)
        if workname:
            self.widget('options').hide()
            self.widget('workname').set_text(workname)
            if '#' not in workname:
                self.widget('message_box').show()
        self._olab = _(self.widget("lblworkname").get_text())
        super(SelectionWindow, self).load(*args, **kwargs)

    def inital_thread(self):
        """Load anything outside of the gtk call"""
        lpa = self.launchpad.launchpad
        if self.branch:
            # Load merge request branches
            self.merge_branches(self.branch)
        else:
            self.project_branches(self.project)
        # Now we add in extra variables which would take too long at init
        for row in self.slist._model:
            item = row[0]
            item.load_extra_attributes()
            # This is not redundent, it updates the treeview item.
            row[0] = item

    def project_branches(self, project):
        """Load project branches into the list"""
        logging.debug("Adding to the list...")
        # We need to be careful accessing items until we're finished because
        # launchpadlib isn't thread safe and multiple requests will kill it.
        self.call('loading_project')
        self.slist.project(project)
        self.call('load_complete')
        # Thread safe version, adds when results come in
        project.branches( callback=self.slist.add_item )

    def merge_branches(self, branch):
        """Load branch merge requests as branches"""
        lpa = self.launchpad
        url = branch.get_launchpad_name()
        branch = LaunchpadBranch(lpa.launchpad, url=url)
        self.call('load_complete')
        branch.merge_requests( callback=self.slist.add_item )

    def signals(self):
        """Add any extra gui signals"""
        return {
            'option_changed' : self.branch_option_changed,
        }

    def loading_project(self):
        """Show a message about loading project data"""
        self.widget('waittext').set_markup( "<big><b>%s</b></big>" % (
            _("Loading Project %s..." % self.oid)))

    def load_complete(self):
        """Show required widgets and list views of branches"""
        self.widget("branches").show()
        self.widget("wait").hide()
        # Only show options if we have not had them set for us.
        if self.widget("workname").get_text() == '':
            self.widget("options").show()

    def get_args(self):
        """Return all required vars to callback"""
        workname = None
        # If it's a branching operating then we should have
        # A workname to pass on to the caller.
        if self.widget("optionbranch").get_active():
            workname = self.widget("workname").get_text()
        # Do something different if we're downloading the
        # user's own branch, make sure push is the parent uri
        if self.widget("optionown").get_active():
            workname = self._selected.name.replace('lp:', '')
            workname = workname.split('/')[-1]
        return {
            'lpname'   : self._selected.name,
            'workname' : workname,
            'path'     : self.branchpath,
        }

    def is_valid(self):
        """Return true if a branch is selected"""
        if self._selected == None:
            return 'None Selected'
        if self.widget("optionbranch").get_active():
            if len(self.widget("workname").get_text()) < 4:
                return "No Workname"
        return True

    def is_valid_failed(self, reason):
        """Make user aware that the workname is required"""
        if 'Workname' in reason:
            label = self.widget("lblworkname")
            if "*" not in label.get_text():
                rec = _("Required")
                template = "<span foreground='red'>%s* %s</span>"
                label.set_markup(template % (rec, self._olab))
        else:
            logging.debug("This should never happen.")

    @property
    def launchpad(self):
        """Return a valid launchpad object"""
        if self._launchpad == None:
            logging.debug("Loading Launchpad object...")
            self._launchpad = get_launchpad()
        return self._launchpad

    @property
    def project(self):
        """Return a project object on demand"""
        if not self._project:
            lp = self.launchpad
            logging.debug("Loading project object...")
            self._project = LaunchpadProject(lp.launchpad, oid=self.oid)
        return self._project

    def update_status(self, project=None):
        """This simply keeps the user up to date"""
        logging.debug("Updating status to: %s" % str(project))
        if project:
            msg = BR_MESSAGE % project.name
        else:
            msg = LP_MESSAGE
        target = self.widget("waittext")
        if target:
            target.set_markup(msg)

    def branch_option_changed(self, widget):
        """Called when the branch option has changed"""
        res = self.widget("optionbranch").get_active()
        self.widget('lblworkname').set_sensitive(res)
        self.widget('lblworkname').set_markup(self._olab)
        self.widget('workname').set_sensitive(res)

    def selected(self, item):
        """An item has surely been selected."""
        self._selected = item
        self.widget('buttonok').set_sensitive(True)
        if not hasattr(item, '_sowner') or item._sowner.member:
            self.widget("optionown").set_sensitive(True)
        else:
            if self.widget("optionown").get_active():
                self.widget("optionbranch").set_active(True)
            self.widget("optionown").set_sensitive(False)

    def unselected(self):
        """All items unselected"""
        self._selected = None
        self.widget("optionown").set_sensitive(False)
        self.widget('buttonok').set_sensitive(False)


class BranchSelection(GtkApp):
    """Application for selecting a branch from a project"""
    gtkfile = 'branch-select.glade'
    windows = [ SelectionWindow ]


class BranchView(TreeView):
    """Controls and operates a table as a projects view."""
    template = """<span size="%s" weight="%s" style="%s">%s</span> <i>%s</i>
<span foreground="#888888" style="italic">%s</span>"""

    def __init__(self, widget, **args):
        self._locked = False
        self._project = None
        args['name'] = _("Project Branches") #self.project().name
        super(BranchView, self).__init__(widget, **args)

    def project(self, set_project=None):
        if set_project:
            self._project = set_project
        return self._project

    def add_item(self, item, parent=None):
        """Filter the items being added"""
        # We would filter out the branch we have here,
        # So as not to merge the same branch into it's self
        # But we can't because launchpad sucks.
        super(BranchView, self).add_item(item, parent)

    def setup(self):
        """Setup a treeview for showing services"""
        def text_cell_func(cell_layout, renderer, model, item_iter):
            """Render the text of the services tree view"""
            item = model.get_value(item_iter, 0)
            size = 'large'
            weight = 'normal'
            style = 'normal'
            whoby = ""
            comment = item.name
            if hasattr(item, '_request_message'):
                comment += _('\nTo Merge: %s') % item._request_message
            if hasattr(item, '_ownname'):
                whoby += _("by %s") % item._ownname
            if item.develop:
                whoby = _("(Development Focus)")
                size = 'x-large'
                weight = 'bold'
            elif item.series:
                name = item.series.title().replace("&", '&amp;')
                whoby = _("(%s Series Focus)") % name
                size = 'x-large'
                weight = 'bold'
                style = 'italic'
            sname = item.sname.replace('-', ' ').replace('_', ' ').title()
            sname = sname.replace('&', '&amp')
            markup = self.template % (size, weight, style,
                sname, whoby, comment)
            renderer.set_property("markup", markup)

        def icon_cell_func(column, cell, model, item_iter):
            """Reender the icons of the services tree view"""
            item = model.get_value(item_iter, 0)
            img = project_icons.get_icon('default')
            if item.develop or item.series:
                img = project_icons.get_icon(self.project().logo())
            else:
                if hasattr(item, '_fimage'):
                    img = person_icons.get_icon(item._fimage)
            img2 = img.scale_simple(64, 64, gtk.gdk.INTERP_BILINEAR)
            cell.set_property("pixbuf", img2)
            cell.set_property("visible", True)
            
        def sort_branches_func(model, iter1, iter2):
            """Sorts branches so that development series branch is first, followed
               by series focus branches"""
            FOCUS_NONE, FOCUS_SERIES, FOCUS_DEVEL = range(3)
            branches = (model.get_value(iter1, 0), model.get_value(iter2, 0))
            focuses = list()
            
            for branch in branches:
                if branch.develop:
                    focuses.append(FOCUS_DEVEL)
                elif branch.series:
                    focuses.append(FOCUS_SERIES)
                else:
                    focuses.append(FOCUS_NONE)

            return focuses[1] - focuses[0]
        
        svlist = super(BranchView, self).setup()
        model = svlist.get_model()
        model.set_sort_func(0, sort_branches_func)
        model.set_sort_column_id(0, gtk.SORT_ASCENDING)
        
        column = gtk.TreeViewColumn((_("%s Branches") % self._name))
        column.set_sizing(gtk.TREE_VIEW_COLUMN_AUTOSIZE)
        column.set_expand(True)
        # The icon
        renderer_icon = gtk.CellRendererPixbuf()
        renderer_icon.set_property("ypad", 8)
        renderer_icon.set_property("xpad", 8)
        column.pack_start(renderer_icon, False)
        column.set_cell_data_func(renderer_icon, icon_cell_func)
        # The name
        renderer = gtk.CellRendererText()
        column.pack_start(renderer, True)
        column.set_cell_data_func(renderer, text_cell_func)
        # Add the required coluns to the treeview
        svlist.append_column(column)

