/* -*- Mode: C; tab-width: 4; indent-tabs-mode: t; c-basic-offset: 4 -*- */
/*
  Gpredict: Real-time satellite tracking and orbit prediction program

  Copyright (C)  2001-2009  Alexandru Csete, OZ9AEC.

  Authors: Alexandru Csete <oz9aec@gmail.com>

  Comments, questions and bugreports should be submitted via
  http://sourceforge.net/projects/gpredict/
  More details can be found at the project home page:

  http://gpredict.oz9aec.net/
 
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, visit http://www.fsf.org/
*/
/** \brief RIG control window.
 *  \ingroup widgets
 *
 * The master radio control UI is implemented as a Gtk+ Widget in order
 * to allow multiple instances. The widget is created from the module
 * popup menu and each module can have several radio control windows
 * attached to it. Note, however, that current implementation only
 * allows one control window per module.
 *
 * TODO Duplex TRX
 * TODO Transponder passband display somewhere, below Sat freq?
 * 
 */
#include <gtk/gtk.h>
#include <glib/gi18n.h>
#include <math.h>
#include "compat.h"
#include "sat-log.h"
#include "predict-tools.h"
#include "gpredict-utils.h"
#include "sat-cfg.h"
#include "gtk-freq-knob.h"
#include "radio-conf.h"
#include "trsp-conf.h"
#ifdef HAVE_CONFIG_H
#  include <build-config.h>
#endif

/* NETWORK */
//#include <sys/types.h>
#include <sys/socket.h>     /* socket(), connect(), send() */
#include <netinet/in.h>     /* struct sockaddr_in */
#include <arpa/inet.h>      /* htons() */
#include <netdb.h>          /* gethostbyname() */
/* END */
#include "gtk-rig-ctrl.h"


#define AZEL_FMTSTR "%7.2f\302\260"
#define MAX_ERROR_COUNT 5
#define WR_DEL 5000 /* delay in usec to wait between write and read commands */


static void gtk_rig_ctrl_class_init (GtkRigCtrlClass *class);
static void gtk_rig_ctrl_init       (GtkRigCtrl      *list);
static void gtk_rig_ctrl_destroy    (GtkObject       *object);


static GtkWidget *create_downlink_widgets (GtkRigCtrl *ctrl);
static GtkWidget *create_uplink_widgets (GtkRigCtrl *ctrl);
static GtkWidget *create_target_widgets (GtkRigCtrl *ctrl);
static GtkWidget *create_conf_widgets (GtkRigCtrl *ctrl);
static GtkWidget *create_count_down_widgets (GtkRigCtrl *ctrl);

/* callback functions */
static void sat_selected_cb (GtkComboBox *satsel, gpointer data);
static void track_toggle_cb (GtkToggleButton *button, gpointer data);
static void delay_changed_cb (GtkSpinButton *spin, gpointer data);
static void primary_rig_selected_cb (GtkComboBox *box, gpointer data);
static void secondary_rig_selected_cb (GtkComboBox *box, gpointer data);
static void rig_engaged_cb (GtkToggleButton *button, gpointer data);
static void trsp_selected_cb (GtkComboBox *box, gpointer data);
static void trsp_tune_cb (GtkButton *button, gpointer data);
static void trsp_lock_cb (GtkToggleButton *button, gpointer data);
static gboolean rig_ctrl_timeout_cb (gpointer data);
static void downlink_changed_cb (GtkFreqKnob *knob, gpointer data);
static void uplink_changed_cb (GtkFreqKnob *knob, gpointer data);


/* radio control functions */
static void exec_rx_cycle (GtkRigCtrl *ctrl);
static void exec_tx_cycle (GtkRigCtrl *ctrl);
static void exec_trx_cycle (GtkRigCtrl *ctrl);
static void exec_duplex_cycle (GtkRigCtrl *ctrl);
static void exec_dual_rig_cycle (GtkRigCtrl *ctrl);
static gboolean set_freq_simplex (GtkRigCtrl *ctrl, radio_conf_t *conf, gdouble freq);
static gboolean get_freq_simplex (GtkRigCtrl *ctrl, radio_conf_t *conf, gdouble *freq);
static gboolean get_ptt (GtkRigCtrl *ctrl, radio_conf_t *conf);
static gboolean set_vfo (GtkRigCtrl *ctrl, vfo_t vfo);
static void update_count_down (GtkRigCtrl *ctrl, gdouble t);

/* misc utility functions */
static void load_trsp_list (GtkRigCtrl *ctrl);
static void store_sats (gpointer key, gpointer value, gpointer user_data);
static gboolean have_conf (void);
static void track_downlink (GtkRigCtrl *ctrl);
static void track_uplink (GtkRigCtrl *ctrl);
static gboolean is_rig_tx_capable (const gchar *confname);


static GtkVBoxClass *parent_class = NULL;

static GdkColor ColBlack = { 0, 0, 0, 0};
static GdkColor ColWhite = { 0, 0xFFFF, 0xFFFF, 0xFFFF};
static GdkColor ColRed =   { 0, 0xFFFF, 0, 0};
static GdkColor ColGreen = {0, 0, 0xFFFF, 0};


GType
gtk_rig_ctrl_get_type ()
{
    static GType gtk_rig_ctrl_type = 0;

    if (!gtk_rig_ctrl_type) {

        static const GTypeInfo gtk_rig_ctrl_info = {
            sizeof (GtkRigCtrlClass),
            NULL,  /* base_init */
            NULL,  /* base_finalize */
            (GClassInitFunc) gtk_rig_ctrl_class_init,
            NULL,  /* class_finalize */
            NULL,  /* class_data */
            sizeof (GtkRigCtrl),
            2,     /* n_preallocs */
            (GInstanceInitFunc) gtk_rig_ctrl_init,
        };

        gtk_rig_ctrl_type = g_type_register_static (GTK_TYPE_VBOX,
                                                    "GtkRigCtrl",
                                                    &gtk_rig_ctrl_info,
                                                     0);
    }

    return gtk_rig_ctrl_type;
}


static void
gtk_rig_ctrl_class_init (GtkRigCtrlClass *class)
{
    GObjectClass      *gobject_class;
    GtkObjectClass    *object_class;
    GtkWidgetClass    *widget_class;
    GtkContainerClass *container_class;

    gobject_class   = G_OBJECT_CLASS (class);
    object_class    = (GtkObjectClass*) class;
    widget_class    = (GtkWidgetClass*) class;
    container_class = (GtkContainerClass*) class;

    parent_class = g_type_class_peek_parent (class);

    object_class->destroy = gtk_rig_ctrl_destroy;
 
}



static void
gtk_rig_ctrl_init (GtkRigCtrl *ctrl)
{
    ctrl->sats = NULL;
    ctrl->target = NULL;
    ctrl->pass = NULL;
    ctrl->qth = NULL;
    ctrl->conf = NULL;
    ctrl->conf2 = NULL;
    ctrl->trsp = NULL;
    ctrl->trsplist = NULL;
    ctrl->trsplock = FALSE;
    ctrl->tracking = FALSE;
    ctrl->busy = FALSE;
    ctrl->engaged = FALSE;
    ctrl->delay = 1000;
    ctrl->timerid = 0;
    ctrl->errcnt = 0;
    ctrl->lastrxf = 0.0;
    ctrl->lasttxf = 0.0;
}

static void
gtk_rig_ctrl_destroy (GtkObject *object)
{
    GtkRigCtrl *ctrl = GTK_RIG_CTRL (object);
    
    /* stop timer */
    if (ctrl->timerid > 0) 
        g_source_remove (ctrl->timerid);

    /* free configuration */
    if (ctrl->conf != NULL) {
        g_free (ctrl->conf->name);
        g_free (ctrl->conf->host);
        g_free (ctrl->conf);
        ctrl->conf = NULL;
    }
    if (ctrl->conf2 != NULL) {
        g_free (ctrl->conf2->name);
        g_free (ctrl->conf2->host);
        g_free (ctrl->conf2);
        ctrl->conf2 = NULL;
    }
    
    /* free transponder */
    if (ctrl->trsplist != NULL) {
        free_transponders (ctrl->trsplist);
        ctrl->trsplist = NULL;   /* destroy might be called twice (?) so we need to NULL it */
    }

    (* GTK_OBJECT_CLASS (parent_class)->destroy) (object);
}



/** \brief Create a new rig control widget.
 * \return A new rig control window.
 * 
 */
GtkWidget *
gtk_rig_ctrl_new (GtkSatModule *module)
{
    GtkWidget *widget;
    GtkWidget *table;
    
    /* check that we have rig conf */
    if (!have_conf()) {
        return NULL;
    }

    widget = g_object_new (GTK_TYPE_RIG_CTRL, NULL);
    
    /* store satellites */
    g_hash_table_foreach (module->satellites, store_sats, widget);
    
    GTK_RIG_CTRL (widget)->target = SAT (g_slist_nth_data (GTK_RIG_CTRL (widget)->sats, 0));
    
    /* store QTH */
    GTK_RIG_CTRL (widget)->qth = module->qth;
    
    /* get next pass for target satellite */
    GTK_RIG_CTRL (widget)->pass = get_next_pass (GTK_RIG_CTRL (widget)->target,
                                                 GTK_RIG_CTRL (widget)->qth,
                                                 3.0);
    
    /* initialise custom colors */
    gdk_rgb_find_color (gtk_widget_get_colormap (widget), &ColBlack);
    gdk_rgb_find_color (gtk_widget_get_colormap (widget), &ColWhite);
    gdk_rgb_find_color (gtk_widget_get_colormap (widget), &ColRed);
    gdk_rgb_find_color (gtk_widget_get_colormap (widget), &ColGreen);

    /* create contents */
    table = gtk_table_new (3, 2, FALSE);
    gtk_table_set_row_spacings (GTK_TABLE (table), 5);
    gtk_table_set_col_spacings (GTK_TABLE (table), 5);
    gtk_container_set_border_width (GTK_CONTAINER (table), 10);
    gtk_table_attach (GTK_TABLE (table), create_downlink_widgets (GTK_RIG_CTRL (widget)),
                      0, 1, 0, 1, GTK_FILL|GTK_EXPAND, GTK_FILL|GTK_EXPAND, 0, 0);
    gtk_table_attach (GTK_TABLE (table), create_uplink_widgets (GTK_RIG_CTRL (widget)),
                      1, 2, 0, 1, GTK_FILL|GTK_EXPAND, GTK_FILL|GTK_EXPAND, 0, 0);
    gtk_table_attach (GTK_TABLE (table), create_target_widgets (GTK_RIG_CTRL (widget)),
                      0, 1, 1, 2, GTK_FILL|GTK_EXPAND, GTK_FILL|GTK_EXPAND, 0, 0);
    gtk_table_attach (GTK_TABLE (table), create_conf_widgets (GTK_RIG_CTRL (widget)),
                      1, 2, 1, 2, GTK_FILL|GTK_EXPAND, GTK_FILL|GTK_EXPAND, 0, 0);
    gtk_table_attach (GTK_TABLE (table), create_count_down_widgets (GTK_RIG_CTRL (widget)),
                      0, 2, 2, 3, GTK_FILL|GTK_EXPAND, GTK_FILL|GTK_EXPAND, 0, 0);

    gtk_container_add (GTK_CONTAINER (widget), table);
    
    GTK_RIG_CTRL (widget)->timerid = g_timeout_add (GTK_RIG_CTRL (widget)->delay,
                                                    rig_ctrl_timeout_cb,
                                                    GTK_RIG_CTRL (widget));
    
	return widget;
}


/** \brief Update rig control state.
 * \param ctrl Pointer to the GtkRigCtrl.
 * 
 * This function is called by the parent, i.e. GtkSatModule, indicating that
 * the satellite data has been updated. The function updates the internal state
 * of the controller and the rigator.
 */
void
gtk_rig_ctrl_update   (GtkRigCtrl *ctrl, gdouble t)
{
    gdouble satfreq;
    gchar *buff;
    
    if (ctrl->target) {
        
        /* update Az/El */
        buff = g_strdup_printf (AZEL_FMTSTR, ctrl->target->az);
        gtk_label_set_text (GTK_LABEL (ctrl->SatAz), buff);
        g_free (buff);
        buff = g_strdup_printf (AZEL_FMTSTR, ctrl->target->el);
        gtk_label_set_text (GTK_LABEL (ctrl->SatEl), buff);
        g_free (buff);
        
        update_count_down (ctrl, t);
        
        /* update range */
        if (sat_cfg_get_bool (SAT_CFG_BOOL_USE_IMPERIAL)) {
            buff = g_strdup_printf ("%.0f mi", KM_TO_MI (ctrl->target->range));
        }
        else {
            buff = g_strdup_printf ("%.0f km", ctrl->target->range);
        }
        gtk_label_set_text (GTK_LABEL (ctrl->SatRng), buff);
        g_free (buff);
        
        /* update range rate */
        if (sat_cfg_get_bool (SAT_CFG_BOOL_USE_IMPERIAL)) {
            buff = g_strdup_printf ("%.3f mi/s", KM_TO_MI (ctrl->target->range_rate));
        }
        else {
            buff = g_strdup_printf ("%.3f km/s", ctrl->target->range_rate);
        }
        gtk_label_set_text (GTK_LABEL (ctrl->SatRngRate), buff);
        g_free (buff);
        
        /* Doppler shift down */
        satfreq = gtk_freq_knob_get_value (GTK_FREQ_KNOB (ctrl->SatFreqDown));
        ctrl->dd = -satfreq * (ctrl->target->range_rate / 299792.4580); // Hz
        buff = g_strdup_printf ("%.0f Hz", ctrl->dd);
        gtk_label_set_text (GTK_LABEL (ctrl->SatDopDown), buff);
        g_free (buff);
        
        /* Doppler shift up */
        satfreq = gtk_freq_knob_get_value (GTK_FREQ_KNOB (ctrl->SatFreqUp));
        ctrl->du = -satfreq * (ctrl->target->range_rate / 299792.4580); // Hz
        buff = g_strdup_printf ("%.0f Hz", ctrl->du);
        gtk_label_set_text (GTK_LABEL (ctrl->SatDopUp), buff);
        g_free (buff);

        /* update next pass if necessary */
        if (ctrl->pass != NULL) {
            if (ctrl->target->aos > ctrl->pass->aos) {
                /* update pass */
                free_pass (ctrl->pass);
                ctrl->pass = get_next_pass (ctrl->target, ctrl->qth, 3.0);
            }
        }
        else {
            /* we don't have any current pass; store the current one */
            ctrl->pass = get_next_pass (ctrl->target, ctrl->qth, 3.0);
        }
    }
}


/** \brief Create freq control widgets for downlink.
 * \param ctrl Pointer to the GtkRigCtrl widget.
 * 
 * This function creates and initialises the widgets for controlling the
 * downlink frequency. It consists of a controller widget showing the
 * satellite frequency with the radio frequency below it.
 * 
 */
static
GtkWidget *create_downlink_widgets (GtkRigCtrl *ctrl)
{
    GtkWidget   *frame;
    GtkWidget   *vbox;
    GtkWidget   *hbox1,*hbox2;
    GtkWidget   *label;
    
    label = gtk_label_new (NULL);
    gtk_label_set_markup (GTK_LABEL (label), _("<b> Downlink </b>"));
    frame = gtk_frame_new (NULL);
    gtk_frame_set_label_align (GTK_FRAME (frame), 0.5, 0.5);
    gtk_frame_set_label_widget (GTK_FRAME (frame), label);
    
    vbox = gtk_vbox_new (FALSE, 5);
    hbox1 = gtk_hbox_new (FALSE, 5);
    hbox2 = gtk_hbox_new (FALSE, 5);
    
    /* satellite downlink frequency */
    ctrl->SatFreqDown = gtk_freq_knob_new (145890000.0, TRUE);
    g_signal_connect (ctrl->SatFreqDown, "freq-changed", G_CALLBACK (downlink_changed_cb), ctrl);
    gtk_box_pack_start_defaults (GTK_BOX (vbox), ctrl->SatFreqDown);
    
    /* Downlink doppler */
    label = gtk_label_new (_("Doppler:"));
    gtk_widget_set_tooltip_text (label,
                                 _("The Doppler shift according to the range rate and "\
                                   "the currently selected downlink frequency"));
    gtk_misc_set_alignment (GTK_MISC (label), 1.0, 0.5);
    gtk_box_pack_start (GTK_BOX (hbox1), label, FALSE, FALSE, 0);
    ctrl->SatDopDown = gtk_label_new ("---- Hz");
    gtk_misc_set_alignment (GTK_MISC (ctrl->SatDopDown), 0.0, 0.5);
    gtk_box_pack_start (GTK_BOX (hbox1), ctrl->SatDopDown, FALSE, TRUE, 0);
    
    /* Downconverter LO */
    ctrl->LoDown = gtk_label_new ("0 MHz");
    gtk_misc_set_alignment (GTK_MISC (ctrl->LoDown), 1.0, 0.5);
    gtk_box_pack_end (GTK_BOX (hbox1), ctrl->LoDown, FALSE, FALSE, 2);
    label = gtk_label_new (_("LO:"));
    gtk_misc_set_alignment (GTK_MISC (label), 1.0, 0.5);
    gtk_box_pack_end (GTK_BOX (hbox1), label, FALSE, FALSE, 0);
    
    /* Radio downlink frequency */
    label = gtk_label_new (NULL);
    gtk_label_set_markup (GTK_LABEL (label),"<span size='large'><b>Radio:</b></span>");
    gtk_misc_set_alignment (GTK_MISC (label), 1.0, 0.5);
    gtk_box_pack_start_defaults (GTK_BOX (hbox2), label); 
    ctrl->RigFreqDown = gtk_freq_knob_new (145890000.0, FALSE);
    gtk_box_pack_start_defaults (GTK_BOX (hbox2), ctrl->RigFreqDown);
    
    /* finish packing ... */
    gtk_box_pack_start_defaults (GTK_BOX (vbox), hbox1);
    gtk_box_pack_start_defaults (GTK_BOX (vbox), hbox2);
    gtk_container_add (GTK_CONTAINER (frame), vbox);
    
    return frame;
}


/** \brief Create uplink frequency display widgets.
 * \param ctrl Pointer to the GtkRigCtrl widget.
 * 
 * This function creates and initialises the widgets for displaying the
 * uplink frequency of the satellite and the radio.
 */
static
GtkWidget *create_uplink_widgets (GtkRigCtrl *ctrl)
{
    GtkWidget   *frame;
    GtkWidget   *vbox;
    GtkWidget   *hbox1,*hbox2;
    GtkWidget   *label;
    
    label = gtk_label_new (NULL);
    gtk_label_set_markup (GTK_LABEL (label), _("<b> Uplink </b>"));
    frame = gtk_frame_new (NULL);
    gtk_frame_set_label_align (GTK_FRAME (frame), 0.5, 0.5);
    gtk_frame_set_label_widget (GTK_FRAME (frame), label);
    
    vbox = gtk_vbox_new (FALSE, 5);
    hbox1 = gtk_hbox_new (FALSE, 5);
    hbox2 = gtk_hbox_new (FALSE, 5);
    
    /* satellite uplink frequency */
    ctrl->SatFreqUp = gtk_freq_knob_new (145890000.0, TRUE);
    g_signal_connect (ctrl->SatFreqUp, "freq-changed", G_CALLBACK (uplink_changed_cb), ctrl);
    gtk_box_pack_start_defaults (GTK_BOX (vbox), ctrl->SatFreqUp);
    
    /* Uplink doppler */
    label = gtk_label_new (_("Doppler:"));
    gtk_widget_set_tooltip_text (label,
                                 _("The Doppler shift according to the range rate and "\
                                   "the currently selected downlink frequency"));
    gtk_misc_set_alignment (GTK_MISC (label), 1.0, 0.5);
    gtk_box_pack_start (GTK_BOX (hbox1), label, FALSE, FALSE, 0);
    ctrl->SatDopUp = gtk_label_new ("---- Hz");
    gtk_misc_set_alignment (GTK_MISC (ctrl->SatDopUp), 0.0, 0.5);
    gtk_box_pack_start (GTK_BOX (hbox1), ctrl->SatDopUp, FALSE, TRUE, 0);
    
    /* Upconverter LO */
    ctrl->LoUp = gtk_label_new ("0 MHz");
    gtk_misc_set_alignment (GTK_MISC (ctrl->LoUp), 1.0, 0.5);
    gtk_box_pack_end (GTK_BOX (hbox1), ctrl->LoUp, FALSE, FALSE, 2);
    label = gtk_label_new (_("LO:"));
    gtk_misc_set_alignment (GTK_MISC (label), 1.0, 0.5);
    gtk_box_pack_end (GTK_BOX (hbox1), label, FALSE, FALSE, 0);
    
    /* Radio uplink frequency */
    label = gtk_label_new (NULL);
    gtk_label_set_markup (GTK_LABEL (label),"<span size='large'><b>Radio:</b></span>");
    gtk_misc_set_alignment (GTK_MISC (label), 1.0, 0.5);
    gtk_box_pack_start_defaults (GTK_BOX (hbox2), label); 
    ctrl->RigFreqUp = gtk_freq_knob_new (145890000.0, FALSE);
    gtk_box_pack_start_defaults (GTK_BOX (hbox2), ctrl->RigFreqUp);
    
   
    gtk_box_pack_start_defaults (GTK_BOX (vbox), hbox1);
    gtk_box_pack_start_defaults (GTK_BOX (vbox), hbox2);
    gtk_container_add (GTK_CONTAINER (frame), vbox);


    return frame;
}

/** \brief Create target widgets.
 * \param ctrl Pointer to the GtkRigCtrl widget.
 */
static
GtkWidget *create_target_widgets (GtkRigCtrl *ctrl)
{
    GtkWidget *frame,*table,*label,*satsel,*track;
    GtkWidget *tune,*trsplock,*hbox;
    gchar *buff;
    guint i, n;
    sat_t *sat = NULL;
    

    buff = g_strdup_printf (AZEL_FMTSTR, 0.0);
    
    table = gtk_table_new (4, 4, FALSE);
    gtk_container_set_border_width (GTK_CONTAINER (table), 5);
    gtk_table_set_col_spacings (GTK_TABLE (table), 5);
    gtk_table_set_row_spacings (GTK_TABLE (table), 5);

    /* sat selector */
    satsel = gtk_combo_box_new_text ();
    n = g_slist_length (ctrl->sats);
    for (i = 0; i < n; i++) {
        sat = SAT (g_slist_nth_data (ctrl->sats, i));
        if (sat) {
            gtk_combo_box_append_text (GTK_COMBO_BOX (satsel), sat->nickname);
        }
    }
    gtk_combo_box_set_active (GTK_COMBO_BOX (satsel), 0);
    gtk_widget_set_tooltip_text (satsel, _("Select target object"));
    g_signal_connect (satsel, "changed", G_CALLBACK (sat_selected_cb), ctrl);
    gtk_table_attach_defaults (GTK_TABLE (table), satsel, 0, 3, 0, 1);
    
    /* tracking button */
    track = gtk_toggle_button_new_with_label (_("Track"));
    gtk_widget_set_tooltip_text (track, _("Track the satellite transponder.\n"\
                                          "Enabling this button will apply Dopper correction "\
                                          "to the frequency of the radio."));
    gtk_table_attach_defaults (GTK_TABLE (table), track, 3, 4, 0, 1);
    g_signal_connect (track, "toggled", G_CALLBACK (track_toggle_cb), ctrl);
    
    /* Transponder selector, tune, and trsplock buttons */
    ctrl->TrspSel = gtk_combo_box_new_text ();
    gtk_widget_set_tooltip_text (ctrl->TrspSel, _("Select a transponder"));
    load_trsp_list (ctrl);
    //gtk_combo_box_set_active (GTK_COMBO_BOX (ctrl->TrspSel), 0);
    g_signal_connect (ctrl->TrspSel, "changed", G_CALLBACK (trsp_selected_cb), ctrl);
    gtk_table_attach_defaults (GTK_TABLE (table), ctrl->TrspSel, 0, 3, 1, 2);
    
    /* buttons */
    tune = gtk_button_new_with_label (_("T"));
    gtk_widget_set_tooltip_text (tune,
                                  _("Tune the radio to this transponder. "\
                                  "The uplink and downlink will be set to the center of "\
                                  "the transponder passband. In case of beacons, only the "\
                                  "downlink will be tuned to the beacon frequency."));
    g_signal_connect (tune, "clicked", G_CALLBACK (trsp_tune_cb), ctrl);
    
    trsplock = gtk_toggle_button_new_with_label (_("L"));
    gtk_widget_set_tooltip_text (trsplock,
                                  _("Lock the uplink and the downlink to each other. "\
                                  "Whenever you change the downlink (in the controller "\
                                  "or on the dial, the uplink will track it according "\
                                  "to whether the transponder is inverting or not. "\
                                  "Similarly, if you change the uplink the downlink "\
                                  "will track it automatically.\n\n"\
                                  "If the downlink and uplink are initially out of sync "\
                                  "when you enable this function, "\
                                  "the current downlink frequency will be used as "\
                                  "baseline for setting the new uplink frequency."));
    g_signal_connect (trsplock, "toggled", G_CALLBACK (trsp_lock_cb), ctrl);
    
    /* box for packing buttons */
    hbox = gtk_hbox_new (FALSE, 5);
    gtk_box_pack_start (GTK_BOX (hbox), tune, TRUE, TRUE, 0);
    gtk_box_pack_start (GTK_BOX (hbox), trsplock, TRUE, TRUE, 0);
    gtk_table_attach_defaults (GTK_TABLE (table), hbox, 3, 4, 1, 2);
    
    /* Azimuth */
    label = gtk_label_new (_("Az:"));
    gtk_misc_set_alignment (GTK_MISC (label), 1.0, 0.5);
    gtk_table_attach_defaults (GTK_TABLE (table), label, 0, 1, 2, 3);
    ctrl->SatAz = gtk_label_new (buff);
    gtk_misc_set_alignment (GTK_MISC (ctrl->SatAz), 1.0, 0.5);
    gtk_table_attach_defaults (GTK_TABLE (table), ctrl->SatAz, 1, 2, 2, 3);
    
    /* Elevation */
    label = gtk_label_new (_("El:"));
    gtk_misc_set_alignment (GTK_MISC (label), 1.0, 0.5);
    gtk_table_attach_defaults (GTK_TABLE (table), label, 0, 1, 3, 4);
    ctrl->SatEl = gtk_label_new (buff);
    gtk_misc_set_alignment (GTK_MISC (ctrl->SatEl), 1.0, 0.5);
    gtk_table_attach_defaults (GTK_TABLE (table), ctrl->SatEl, 1, 2, 3, 4);
    
    /* Range */
    label = gtk_label_new (_(" Range:"));
    gtk_misc_set_alignment (GTK_MISC (label), 1.0, 0.5);
    gtk_table_attach_defaults (GTK_TABLE (table), label, 2, 3, 2, 3);
    ctrl->SatRng = gtk_label_new ("0 km");
    gtk_misc_set_alignment (GTK_MISC (ctrl->SatRng), 0.0, 0.5);
    gtk_table_attach_defaults (GTK_TABLE (table), ctrl->SatRng, 3, 4, 2, 3);

    gtk_widget_set_tooltip_text (label,
                                 _("This is the current distance between the satellite "\
                                   "and the observer."));
    gtk_widget_set_tooltip_text (ctrl->SatRng,
                                 _("This is the current distance between the satellite "\
                                   "and the observer."));

    /* Range rate */
    label = gtk_label_new (_(" Rate:"));
    gtk_misc_set_alignment (GTK_MISC (label), 1.0, 0.5);
    gtk_table_attach_defaults (GTK_TABLE (table), label, 2, 3, 3, 4);
    ctrl->SatRngRate = gtk_label_new ("0.0 km/s");
    gtk_misc_set_alignment (GTK_MISC (ctrl->SatRngRate), 0.0, 0.5);
    gtk_table_attach_defaults (GTK_TABLE (table), ctrl->SatRngRate, 3, 4, 3, 4);

    gtk_widget_set_tooltip_text (label,
                                 _("The rate of change for the distance between "
                                   "the satellite and the observer."));
    gtk_widget_set_tooltip_text (ctrl->SatRngRate,
                                 _("The rate of change for the distance between "
                                   "the satellite and the observer."));

    frame = gtk_frame_new (_("Target"));

    gtk_container_add (GTK_CONTAINER (frame), table);
    
    g_free (buff);
    
    return frame;
}


static GtkWidget *
create_conf_widgets (GtkRigCtrl *ctrl)
{
    GtkWidget *frame,*table,*label,*timer;
    GDir        *dir = NULL;   /* directory handle */
    GError      *error = NULL; /* error flag and info */
    gchar       *dirname;      /* directory name */
    gchar      **vbuff;
    const gchar *filename;     /* file name */

    
    
    table = gtk_table_new (4, 3, FALSE);
    gtk_container_set_border_width (GTK_CONTAINER (table), 5);
    gtk_table_set_col_spacings (GTK_TABLE (table), 5);
    gtk_table_set_row_spacings (GTK_TABLE (table), 5);
    
    /* Primary device */
    label = gtk_label_new (_("1. Device:"));
    gtk_misc_set_alignment (GTK_MISC (label), 1.0, 0.5);
    gtk_table_attach_defaults (GTK_TABLE (table), label, 0, 1, 0, 1);
    
    ctrl->DevSel = gtk_combo_box_new_text ();
    gtk_widget_set_tooltip_text (ctrl->DevSel, _("Select primary radio device."\
                                                 "This device will be used for downlink and uplink "\
                                                 "unless you select a secondary device for uplink"));
    
    /* open configuration directory */
    dirname = get_hwconf_dir ();
    
    dir = g_dir_open (dirname, 0, &error);
    if (dir) {
        /* read each .rig file */
        while ((filename = g_dir_read_name (dir))) {
            
            if (g_strrstr (filename, ".rig")) {
                
                vbuff = g_strsplit (filename, ".rig", 0);
                gtk_combo_box_append_text (GTK_COMBO_BOX (ctrl->DevSel), vbuff[0]);
                g_strfreev (vbuff);
            }
        }
    }
    else {
        sat_log_log (SAT_LOG_LEVEL_ERROR,
                     _("%s:%d: Failed to open hwconf dir (%s)"),
                       __FILE__, __LINE__, error->message);
        g_clear_error (&error);
    }

    g_dir_close (dir);

    gtk_combo_box_set_active (GTK_COMBO_BOX (ctrl->DevSel), 0);
    g_signal_connect (ctrl->DevSel, "changed", G_CALLBACK (primary_rig_selected_cb), ctrl);
    gtk_table_attach_defaults (GTK_TABLE (table), ctrl->DevSel, 1, 2, 0, 1);
    /* config will be force-loaded after LO spin is created */

    /* Secondary device */
    label = gtk_label_new (_("2. Device:"));
    gtk_misc_set_alignment (GTK_MISC (label), 1.0, 0.5);
    gtk_table_attach_defaults (GTK_TABLE (table), label, 0, 1, 1, 2);
    
    ctrl->DevSel2 = gtk_combo_box_new_text ();
    gtk_widget_set_tooltip_text (ctrl->DevSel2, _("Select secondary radio device\n"\
                                                  "This device will be used for uplink"));
    
    /* load config */
    gtk_combo_box_append_text (GTK_COMBO_BOX (ctrl->DevSel2), _("None"));
    gtk_combo_box_set_active (GTK_COMBO_BOX (ctrl->DevSel2), 0);
    
    dir = g_dir_open (dirname, 0, &error);
    if (dir) {
        /* read each .rig file */
        while ((filename = g_dir_read_name (dir))) {
            
            if (g_strrstr (filename, ".rig")) {
                
                /* only add TX capable rigs */
                vbuff = g_strsplit (filename, ".rig", 0);
                if (is_rig_tx_capable (vbuff[0])) {
                    gtk_combo_box_append_text (GTK_COMBO_BOX (ctrl->DevSel2), vbuff[0]);
                }
                g_strfreev (vbuff);
            }
        }
    }
    else {
        sat_log_log (SAT_LOG_LEVEL_ERROR,
                     _("%s:%d: Failed to open hwconf dir (%s)"),
                       __FILE__, __LINE__, error->message);
        g_clear_error (&error);
    }

    g_free (dirname);
    g_dir_close (dir);
    
    
    //gtk_combo_box_set_active (GTK_COMBO_BOX (ctrl->DevSel), 0);
    g_signal_connect (ctrl->DevSel2, "changed", G_CALLBACK (secondary_rig_selected_cb), ctrl);
    gtk_table_attach_defaults (GTK_TABLE (table), ctrl->DevSel2, 1, 2, 1, 2);

    
    /* Engage button */
    ctrl->LockBut = gtk_toggle_button_new_with_label (_("Engage"));
    gtk_widget_set_tooltip_text (ctrl->LockBut, _("Engage the selected radio device"));
    g_signal_connect (ctrl->LockBut, "toggled", G_CALLBACK (rig_engaged_cb), ctrl);
    gtk_table_attach_defaults (GTK_TABLE (table), ctrl->LockBut, 2, 3, 0, 1);
    
    /* Now, load config*/
    primary_rig_selected_cb (GTK_COMBO_BOX (ctrl->DevSel), ctrl);
    
    /* Timeout */
    label = gtk_label_new (_("Cycle:"));
    gtk_misc_set_alignment (GTK_MISC (label), 1.0, 0.5);
    gtk_table_attach_defaults (GTK_TABLE (table), label, 0, 1, 3, 4);
    
    timer = gtk_spin_button_new_with_range (100, 5000, 10);
    gtk_spin_button_set_digits (GTK_SPIN_BUTTON (timer), 0);
    gtk_widget_set_tooltip_text (timer,
                                 _("This parameter controls the delay between "\
                                   "commands sent to the rig."));
    gtk_spin_button_set_value (GTK_SPIN_BUTTON (timer), ctrl->delay);
    g_signal_connect (timer, "value-changed", G_CALLBACK (delay_changed_cb), ctrl);
    gtk_table_attach (GTK_TABLE (table), timer, 1, 2, 3, 4,
                      GTK_FILL, GTK_FILL, 0, 0);
    
    label = gtk_label_new (_("msec"));
    gtk_misc_set_alignment (GTK_MISC (label), 0.0, 0.5);
    gtk_table_attach_defaults (GTK_TABLE (table), label, 2, 3, 3, 4);

    frame = gtk_frame_new (_("Settings"));
    gtk_container_add (GTK_CONTAINER (frame), table);
    
    return frame;
}


/** \brief Create count down widget */
static GtkWidget *
create_count_down_widgets (GtkRigCtrl *ctrl)
{
    GtkWidget *frame;
    
    /* create delta-t label */
    ctrl->SatCnt = gtk_label_new (NULL);
    gtk_misc_set_alignment (GTK_MISC (ctrl->SatCnt), 0.5, 0.5);
    gtk_label_set_markup (GTK_LABEL (ctrl->SatCnt),
                          _("<span size='large'><b>\316\224T: 00:00:00</b></span>"));
    gtk_widget_set_tooltip_text (ctrl->SatCnt,
                                 _("The time remaining until the next AOS or LOS event, "\
                                   "depending on which one comes first."));
    
    frame = gtk_frame_new (NULL);
    gtk_container_add (GTK_CONTAINER (frame), ctrl->SatCnt);
    
    return frame;
}



/** \brief Copy satellite from hash table to singly linked list.
 */
static void
store_sats (gpointer key, gpointer value, gpointer user_data)
{
    GtkRigCtrl *ctrl = GTK_RIG_CTRL( user_data);
    sat_t        *sat = SAT (value);

    ctrl->sats = g_slist_append (ctrl->sats, sat);
}


/** \brief Manage satellite selections
 * \param satsel Pointer to the GtkComboBox.
 * \param data Pointer to the GtkRigCtrl widget.
 * 
 * This function is called when the user selects a new satellite.
 */
static void
sat_selected_cb (GtkComboBox *satsel, gpointer data)
{
    GtkRigCtrl *ctrl = GTK_RIG_CTRL (data);
    gint i;
    
    i = gtk_combo_box_get_active (satsel);
    if (i >= 0) {
        ctrl->target = SAT (g_slist_nth_data (ctrl->sats, i));
        
        /* update next pass */
        if (ctrl->pass != NULL)
            free_pass (ctrl->pass);
        ctrl->pass = get_next_pass (ctrl->target, ctrl->qth, 3.0);
        
        /* read transponders for new target */
        load_trsp_list (ctrl);
    }
    else {
        sat_log_log (SAT_LOG_LEVEL_ERROR,
                     _("%s:%s: Invalid satellite selection: %d"),
                     __FILE__, __FUNCTION__, i);
        
        /* clear pass just in case... */
        if (ctrl->pass != NULL) {
            free_pass (ctrl->pass);
            ctrl->pass = NULL;
        }
        
    }
}


/** \brief Manage transponder selections.
 *  \param box Pointer to the transponder selector widget.
 *  \param data Pointer to the GtkRigCtrl structure
 *
 * This function is called when the user selects a new transponder.
 * It updates ctrl->trsp with the new selection.
 */
static void trsp_selected_cb (GtkComboBox *box, gpointer data)
{
    GtkRigCtrl *ctrl = GTK_RIG_CTRL (data);
    gint i, n;
    
    i = gtk_combo_box_get_active (box);
    n = g_slist_length (ctrl->trsplist);
    
    if (i == -1) {
        /* clear transponder data */
        ctrl->trsp = NULL;
    }
    else if (i < n) {
        ctrl->trsp = (trsp_t *) g_slist_nth_data (ctrl->trsplist, i);
    }
    else {
        sat_log_log (SAT_LOG_LEVEL_BUG,
                      _("%s: Inconsistency detected in internal transponder data (%d,%d)"),
                      __FUNCTION__, i, n);
    }
}


/** \brief Manage "Tune" events
 *  \param button Pointer to the GtkButton that received the signal.
 *  \param data Pointer to the GtkRigCtrl structure.
 *
 * This function is called when the user clicks on the Tune button next to the
 * transponder selector. When clicked, the radio controller will set the RX and TX
 * frequencies to the middle of the transponder uplink/downlink bands.
 *
 * To avoid conflicts with manual frequency changes on the radio, the sync between
 * RIG and GPREDICT is invalidated after the tuning operation is performed.
 *
 */
static void trsp_tune_cb (GtkButton *button, gpointer data)
{
    GtkRigCtrl *ctrl = GTK_RIG_CTRL (data);
    gdouble     freq;
    
    if (ctrl->trsp == NULL)
        return;
    
    /* tune downlink */
    if ((ctrl->trsp->downlow > 0.0) && (ctrl->trsp->downhigh > 0.0)) {
        
        freq = ctrl->trsp->downlow + fabs (ctrl->trsp->downhigh - ctrl->trsp->downlow) / 2;
        gtk_freq_knob_set_value (GTK_FREQ_KNOB (ctrl->SatFreqDown), freq);
        
        /* invalidate RIG<->GPREDICT sync */
        ctrl->lastrxf = 0.0;
    }
    
    /* tune uplink */
    if ((ctrl->trsp->uplow > 0.0) && (ctrl->trsp->uphigh > 0.0)) {
        
        freq = ctrl->trsp->uplow + fabs (ctrl->trsp->uphigh - ctrl->trsp->uplow) / 2;
        gtk_freq_knob_set_value (GTK_FREQ_KNOB (ctrl->SatFreqUp), freq);
        
        /* invalidate RIG<->GPREDICT sync */
        ctrl->lasttxf = 0.0;
    }
    
    
}


/** \brief Manage lock transponder signals.
 *  \param button Pointer to the GtkToggleButton that received the signal.
 *  \param data Pointer to the GtkRigCtrl structure.
 *
 * This function is called when the user toggles the "Lock Transponder" button.
 * When ON, the uplink and downlink are locked according to the current transponder
 * data, i.e. when user changes the downlink, the uplink will follow automatically
 * taking into account whether the transponder is inverting or not.
 */
static void trsp_lock_cb (GtkToggleButton *button, gpointer data)
{
    GtkRigCtrl *ctrl = GTK_RIG_CTRL (data);

    ctrl->trsplock = gtk_toggle_button_get_active (button);
    
    /* set uplink according to downlink */
    if (ctrl->trsplock) {
        track_downlink (ctrl);
    }
}




/** \brief Manage toggle signals (tracking)
 * \param button Pointer to the GtkToggle button.
 * \param data Pointer to the GtkRigCtrl widget.
 */
static void
track_toggle_cb (GtkToggleButton *button, gpointer data)
{
    GtkRigCtrl *ctrl = GTK_RIG_CTRL (data);
    
    ctrl->tracking = gtk_toggle_button_get_active (button);

    /* invalidate sync with radio */
    ctrl->lastrxf = 0.0;
    ctrl->lasttxf = 0.0;
}


/** \brief Manage cycle delay changes.
 * \param spin Pointer to the spin button.
 * \param data Pointer to the GtkRigCtrl widget.
 * 
 * This function is called when the user changes the value of the
 * cycle delay.
 */
static void
delay_changed_cb (GtkSpinButton *spin, gpointer data)
{
    GtkRigCtrl *ctrl = GTK_RIG_CTRL (data);
    
    
    ctrl->delay = (guint) gtk_spin_button_get_value (spin);

    if (ctrl->timerid > 0) 
        g_source_remove (ctrl->timerid);

    ctrl->timerid = g_timeout_add (ctrl->delay, rig_ctrl_timeout_cb, ctrl);
}




/** \brief New primary rig device selected.
 * \param box Pointer to the rigor selector combo box.
 * \param data Pointer to the GtkRigCtrl widget.
 * 
 * This function is called when the user selects a new rigor controller
 * device.
 *
 * BUG Doesn't prevent user to select same radio as in the secondary conf.
 */
static void
primary_rig_selected_cb (GtkComboBox *box, gpointer data)
{
    GtkRigCtrl *ctrl = GTK_RIG_CTRL (data);
    gchar      *buff;
    
    
    sat_log_log (SAT_LOG_LEVEL_DEBUG,
                 _("%s:%s: Primary device selected: %d"),
                 __FILE__, __FUNCTION__, gtk_combo_box_get_active (box));

    
    /* free previous configuration */
    if (ctrl->conf != NULL) {
        g_free (ctrl->conf->name);
        g_free (ctrl->conf->host);
        g_free (ctrl->conf);
    }
    
    ctrl->conf = g_try_new (radio_conf_t, 1);
    if (ctrl->conf == NULL) {
        sat_log_log (SAT_LOG_LEVEL_ERROR,
                     _("%s:%d: Failed to allocate memory for radio config"),
                     __FILE__, __LINE__);
        return;
    }
    
    /* load new configuration */
    ctrl->conf->name = gtk_combo_box_get_active_text (box);
    if (radio_conf_read (ctrl->conf)) {
        sat_log_log (SAT_LOG_LEVEL_MSG,
                     _("%s:%s: Loaded new radio configuration %s"),
                     __FILE__, __FUNCTION__, ctrl->conf->name);
        /* update LO widgets */
        buff = g_strdup_printf (_("%.0f MHz"), ctrl->conf->lo/1.0e6);
        gtk_label_set_text (GTK_LABEL (ctrl->LoDown), buff);
        g_free (buff);
        /* uplink LO only if single device */
        if (ctrl->conf2 == NULL) {
            buff = g_strdup_printf (_("%.0f MHz"), ctrl->conf->loup/1.0e6);
            gtk_label_set_text (GTK_LABEL (ctrl->LoUp), buff);
            g_free (buff);
        }
    }
    else {
        sat_log_log (SAT_LOG_LEVEL_ERROR,
                     _("%s:%s: Failed to load radio configuration %s"),
                     __FILE__, __FUNCTION__, ctrl->conf->name);

        g_free (ctrl->conf->name);
        if (ctrl->conf->host)
            g_free (ctrl->conf->host);
        g_free (ctrl->conf);
        ctrl->conf = NULL;
    }

}


/** \brief New secondary rig device selected.
 * \param box Pointer to the rigor selector combo box.
 * \param data Pointer to the GtkRigCtrl widget.
 * 
 * This function is called when the user selects a new rig controller
 * device for the secondary radio. This radio is used for uplink only.
 */
static void
secondary_rig_selected_cb (GtkComboBox *box, gpointer data)
{
    GtkRigCtrl *ctrl = GTK_RIG_CTRL (data);
    gchar      *buff;
    gchar      *name1, *name2;
    
    
    sat_log_log (SAT_LOG_LEVEL_DEBUG,
                 _("%s:%s: Secondary device selected: %d"),
                 __FILE__, __FUNCTION__, gtk_combo_box_get_active (box));
    
    /* free previous configuration */
    if (ctrl->conf2 != NULL) {
        g_free (ctrl->conf2->name);
        g_free (ctrl->conf2->host);
        g_free (ctrl->conf2);
        ctrl->conf2 = NULL;
    }
    
    if (gtk_combo_box_get_active (box) == 0) {
        /* first entry is "None" */
        
        /* reset uplink LO to what's in ctrl->conf */
        if (ctrl->conf != NULL) {
            buff = g_strdup_printf (_("%.0f MHz"), ctrl->conf->loup/1.0e6);
            gtk_label_set_text (GTK_LABEL (ctrl->LoUp), buff);
            g_free (buff);
        }
        
        return;
    }
    
    /* ensure that selected secondary rig is not the same as the primary */
    name1 = gtk_combo_box_get_active_text (GTK_COMBO_BOX (ctrl->DevSel));
    name2 = gtk_combo_box_get_active_text (GTK_COMBO_BOX (ctrl->DevSel2));
    if (!g_strcmp0 (name1, name2)) {
        /* selected conf is the same as the primary one */
        g_free (name1);
        g_free (name2);
        if (ctrl->conf != NULL) {
            buff = g_strdup_printf (_("%.0f MHz"), ctrl->conf->loup/1.0e6);
            gtk_label_set_text (GTK_LABEL (ctrl->LoUp), buff);
            g_free (buff);
        }
        gtk_combo_box_set_active (GTK_COMBO_BOX (ctrl->DevSel2), 0);
        
        return;
    }
    
    g_free (name1);
    g_free (name2);
    
    /* else load new device */
    ctrl->conf2 = g_try_new (radio_conf_t, 1);
    if (ctrl->conf2 == NULL) {
        sat_log_log (SAT_LOG_LEVEL_ERROR,
                     _("%s:%s: Failed to allocate memory for radio config"),
                     __FILE__, __FUNCTION__);
        return;
    }
    
    /* load new configuration */
    ctrl->conf2->name = gtk_combo_box_get_active_text (box);
    if (radio_conf_read (ctrl->conf2)) {
        sat_log_log (SAT_LOG_LEVEL_MSG,
                     _("%s:%s: Loaded new radio configuration %s"),
                     __FILE__, __FUNCTION__, ctrl->conf2->name);
        /* update LO widgets */
        buff = g_strdup_printf (_("%.0f MHz"), ctrl->conf2->loup/1.0e6);
        gtk_label_set_text (GTK_LABEL (ctrl->LoUp), buff);
        g_free (buff);
    }
    else {
        sat_log_log (SAT_LOG_LEVEL_ERROR,
                     _("%s:%s: Failed to load radio configuration %s"),
                     __FILE__, __FUNCTION__, ctrl->conf->name);

        g_free (ctrl->conf2->name);
        if (ctrl->conf2->host)
            g_free (ctrl->conf2->host);
        g_free (ctrl->conf2);
        ctrl->conf2 = NULL;
    }
}




/** \brief Manage Engage button signals.
 * \param button Pointer to the "Engage" button.
 * \param data Pointer to the GtkRigCtrl widget.
 * 
 * This function is called when the user toggles the "Engage" button.
 */
static void
rig_engaged_cb (GtkToggleButton *button, gpointer data)
{
    GtkRigCtrl *ctrl = GTK_RIG_CTRL (data);

    
    if (!gtk_toggle_button_get_active (button)) {
        /* close socket */
        gtk_widget_set_sensitive (ctrl->DevSel, TRUE);
        gtk_widget_set_sensitive (ctrl->DevSel2, TRUE);
        ctrl->engaged = FALSE;
        ctrl->lasttxf = 0.0;
        ctrl->lastrxf = 0.0;
    }
    else {
        if (ctrl->conf == NULL) {
            /* we don't have a working configuration */
            sat_log_log (SAT_LOG_LEVEL_ERROR,
                         _("%s: Controller does not have a valid configuration"),
                           __FUNCTION__);
            return;
        }
        
        gtk_widget_set_sensitive (ctrl->DevSel, FALSE);
        gtk_widget_set_sensitive (ctrl->DevSel2, FALSE);
        ctrl->engaged = TRUE;
        ctrl->wrops = 0;
        
        /* set initial frequency */
        if (ctrl->conf2 != NULL) {
            /* set initial dual mode */
            exec_dual_rig_cycle (ctrl);
        }
        else {
            switch (ctrl->conf->type) {
                
                case RIG_TYPE_RX:
                    exec_rx_cycle (ctrl);
                    break;
                    
                case RIG_TYPE_TX:
                    exec_tx_cycle (ctrl);
                    break;
                    
                case RIG_TYPE_TRX:
                    exec_trx_cycle (ctrl);
                    break;
                    
                case RIG_TYPE_DUPLEX:
                    exec_duplex_cycle (ctrl);
                    break;
                    
                default:
                    /* this is an error! */
                    ctrl->conf->type = RIG_TYPE_RX;
                    exec_rx_cycle (ctrl);
                    break;
            }
        }
    }
}


/** \brief Manage downlink frequency change callbacks.
 *  \param knob Pointer to the GtkFreqKnob widget that received the signal.
 *  \param data Pointer to the GtkRigCtrl structure.
 *
 * This function is called when the user changes the downlink frequency in the controller.
 * The function checks if the the transponder is locked, if yes it calls track_downlink().
 */
static void downlink_changed_cb (GtkFreqKnob *knob, gpointer data)
{
    GtkRigCtrl *ctrl = GTK_RIG_CTRL (data);
    
    if (ctrl->trsplock) {
        track_downlink (ctrl);
    }
}

/** \brief Manage uplink frequency change callbacks.
 *  \param knob Pointer to the GtkFreqKnob widget that received the signal.
 *  \param data Pointer to the GtkRigCtrl structure.
 *
 * This function is called when the user changes the uplink frequency in the controller.
 * The function checks if the the transponder is locked, if yes it calls track_uplink().
 */
static void uplink_changed_cb (GtkFreqKnob *knob, gpointer data)
{
    GtkRigCtrl *ctrl = GTK_RIG_CTRL (data);
    
    if (ctrl->trsplock) {
        track_uplink (ctrl);
    }
}



/** \brief Rigator controller timeout function
 * \param data Pointer to the GtkRigCtrl widget.
 * \return Always TRUE to let the timer continue.
 */
static gboolean
rig_ctrl_timeout_cb (gpointer data)
{
    GtkRigCtrl *ctrl = GTK_RIG_CTRL (data);
    
    
    if (ctrl->busy) {
        sat_log_log (SAT_LOG_LEVEL_ERROR,_("%s missed the deadline"),__FUNCTION__);
        return TRUE;
    }
    
    ctrl->busy = TRUE;
    
    if (ctrl->conf2 != NULL) {
        exec_dual_rig_cycle (ctrl);
    }
    else {
        /* Execute controller cycle depending on primary radio type */
        switch (ctrl->conf->type) {
        
            case RIG_TYPE_RX:
                exec_rx_cycle (ctrl);
                break;
            
            case RIG_TYPE_TX:
                exec_tx_cycle (ctrl);
                break;
            
            case RIG_TYPE_TRX:
                exec_trx_cycle (ctrl);
                break;
            
            case RIG_TYPE_DUPLEX:
                exec_duplex_cycle (ctrl);
                break;
            
            default:
                /* invalid mode */
                sat_log_log (SAT_LOG_LEVEL_ERROR,
                            _("%s: Invalid radio type %d. Setting type to RIG_TYPE_RX"),
                            __FUNCTION__, ctrl->conf->type);
                ctrl->conf->type = RIG_TYPE_RX;
            
        }
    }

    /* perform error count checking */
    if (ctrl->errcnt >= MAX_ERROR_COUNT) {
        /* disengage device */
        gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (ctrl->LockBut), FALSE);
        ctrl->engaged = FALSE;
        ctrl->errcnt = 0;
        sat_log_log (SAT_LOG_LEVEL_ERROR,
                        _("%s: MAX_ERROR_COUNT (%d) reached. Disengaging device!"),
                        __FUNCTION__, MAX_ERROR_COUNT);
        
        //g_print ("ERROR. WROPS = %d\n", ctrl->wrops);
    }
    
    //g_print ("       WROPS = %d\n", ctrl->wrops);
    
    ctrl->busy = FALSE;
    
    return TRUE;
}


/** \brief Execute RX mode cycle.
 *  \param ctrl Pointer to the GtkRigCtrl widget.
 *
 * This function executes a controller cycle when the device is of RIG_TYPE_RX.
 * This function is not used dual-rig mode.
 */
static void exec_rx_cycle (GtkRigCtrl *ctrl)
{
    gdouble readfreq=0.0, tmpfreq, satfreqd, satfrequ;
    gboolean ptt = FALSE;
    gboolean dialchanged = FALSE;
    
    /* get PTT status */
    if (ctrl->engaged && ctrl->conf->ptt)
        ptt = get_ptt (ctrl, ctrl->conf);
    
    
    /* Dial feedback:
       If radio device is engaged read frequency from radio and compare it to the
       last set frequency. If different, it means that user has changed frequency
       on the radio dial => update transponder knob
       
       Note: If ctrl->lastrxf = 0.0 the sync has been invalidated (e.g. user pressed "tune")
             and no need to execute the dial feedback.
    */
    if ((ctrl->engaged) && (ctrl->lastrxf > 0.0)) {
        
        if (ptt == FALSE) {
            if (!get_freq_simplex (ctrl, ctrl->conf, &readfreq)) {
                /* error => use a passive value */
                readfreq = ctrl->lastrxf;
                ctrl->errcnt++;
            }
        }
        else {
            readfreq = ctrl->lastrxf;
        }

        if (fabs (readfreq - ctrl->lastrxf) >= 1.0) {
            dialchanged = TRUE;
            
            /* user might have altered radio frequency => update transponder knob */
            gtk_freq_knob_set_value (GTK_FREQ_KNOB (ctrl->RigFreqDown), readfreq);
            ctrl->lastrxf = readfreq;
            
            /* doppler shift; only if we are tracking */
            if (ctrl->tracking) {
                satfreqd = (readfreq - ctrl->dd + ctrl->conf->lo);

            }
            else {
                satfreqd = readfreq + ctrl->conf->lo;
            }
            gtk_freq_knob_set_value (GTK_FREQ_KNOB (ctrl->SatFreqDown), satfreqd);
            
            /* Update uplink if locked to downlink */
            if (ctrl->trsplock) {
                track_downlink (ctrl);
            }
        }
    }

    /* now, forward tracking */
    if (dialchanged) {
        /* no need to forward track */
        return;
    }
    
    /* If we are tracking, calculate the radio freq by applying both dopper shift
       and tranverter LO frequency. If we are not tracking, apply only LO frequency.
    */
    satfreqd = gtk_freq_knob_get_value (GTK_FREQ_KNOB (ctrl->SatFreqDown));
    satfrequ = gtk_freq_knob_get_value (GTK_FREQ_KNOB (ctrl->SatFreqUp));
    if (ctrl->tracking) {
        /* downlink */
        gtk_freq_knob_set_value (GTK_FREQ_KNOB (ctrl->RigFreqDown),
                                  satfreqd + ctrl->dd - ctrl->conf->lo);
        /* uplink */
        gtk_freq_knob_set_value (GTK_FREQ_KNOB (ctrl->RigFreqUp),
                                  satfrequ + ctrl->du - ctrl->conf->loup);
    }
    else {
        gtk_freq_knob_set_value (GTK_FREQ_KNOB (ctrl->RigFreqDown),
                                  satfreqd - ctrl->conf->lo);
        gtk_freq_knob_set_value (GTK_FREQ_KNOB (ctrl->RigFreqUp),
                                  satfrequ - ctrl->conf->loup );
    }

    tmpfreq = gtk_freq_knob_get_value(GTK_FREQ_KNOB(ctrl->RigFreqDown));


    /* if device is engaged, send freq command to radio */
    if ((ctrl->engaged) && (ptt == FALSE) && (fabs(ctrl->lastrxf - tmpfreq) >= 1.0)) {
        if (set_freq_simplex (ctrl, ctrl->conf, tmpfreq)) {
            /* reset error counter */
            ctrl->errcnt = 0;
            
            /* give radio a chance to set frequency */
            g_usleep (WR_DEL);

            /* The actual frequency might be different from what we have set because
               the tuning step is larger than what we work with (e.g. FT-817 has a
               smallest tuning step of 10 Hz). Therefore we read back the actual
               frequency from the rig. */
            get_freq_simplex (ctrl, ctrl->conf, &tmpfreq);
            ctrl->lastrxf = tmpfreq;
        }
        else {
            ctrl->errcnt++;
        }
    }
    
}


/** \brief Execute TX mode cycle.
 *  \param ctrl Pointer to the GtkRigCtrl widget.
 *
 * This function executes a controller cycle when the primary device is of RIG_TYPE_TX.
 * This function is not used in dual-rig mode.
 */
static void exec_tx_cycle (GtkRigCtrl *ctrl)
{
    gdouble readfreq=0.0, tmpfreq, satfreqd, satfrequ;
    gboolean ptt = TRUE;
    gboolean dialchanged = FALSE;
    
    /* get PTT status */
    if (ctrl->engaged && ctrl->conf->ptt) {
        ptt = get_ptt (ctrl, ctrl->conf);
    }

    /* Dial feedback:
       If radio device is engaged read frequency from radio and compare it to the
       last set frequency. If different, it means that user has changed frequency
       on the radio dial => update transponder knob
       
       Note: If ctrl->lasttxf = 0.0 the sync has been invalidated (e.g. user pressed "tune")
             and no need to execute the dial feedback.
    */
    if ((ctrl->engaged) && (ctrl->lasttxf > 0.0)) {
        
        if (ptt == TRUE) {
            if (!get_freq_simplex (ctrl, ctrl->conf, &readfreq)) {
                /* error => use a passive value */
                readfreq = ctrl->lasttxf;
                ctrl->errcnt++;
            }
        }
        else {
            readfreq = ctrl->lasttxf;
        }
        
        if (fabs (readfreq - ctrl->lasttxf) >= 1.0) {
            dialchanged = TRUE;
            
            /* user might have altered radio frequency => update transponder knob */
            gtk_freq_knob_set_value (GTK_FREQ_KNOB (ctrl->RigFreqUp), readfreq);
            ctrl->lasttxf = readfreq;
            
            /* doppler shift; only if we are tracking */
            if (ctrl->tracking) {
                satfrequ = readfreq - ctrl->du + ctrl->conf->loup;
            }
            else {
                satfrequ = readfreq + ctrl->conf->loup;
            }
            gtk_freq_knob_set_value (GTK_FREQ_KNOB (ctrl->SatFreqUp), satfrequ);

            /* Follow with downlink if transponder is locked */
            if (ctrl->trsplock) {
                track_uplink (ctrl);
            }
        }
    }

    /* now, forward tracking */
    if (dialchanged) {
        /* no need to forward track */
        return;
    }
    
    /* If we are tracking, calculate the radio freq by applying both dopper shift
       and tranverter LO frequency. If we are not tracking, apply only LO frequency.
    */
    satfreqd = gtk_freq_knob_get_value (GTK_FREQ_KNOB (ctrl->SatFreqDown));
    satfrequ = gtk_freq_knob_get_value (GTK_FREQ_KNOB (ctrl->SatFreqUp));
    if (ctrl->tracking) {
        /* downlink */
        gtk_freq_knob_set_value (GTK_FREQ_KNOB (ctrl->RigFreqDown),
                                  satfreqd + ctrl->dd - ctrl->conf->lo);
        /* uplink */
        gtk_freq_knob_set_value (GTK_FREQ_KNOB (ctrl->RigFreqUp),
                                  satfrequ + ctrl->du - ctrl->conf->loup);
    }
    else {
        gtk_freq_knob_set_value (GTK_FREQ_KNOB (ctrl->RigFreqDown),
                                  satfreqd - ctrl->conf->lo);
        gtk_freq_knob_set_value (GTK_FREQ_KNOB (ctrl->RigFreqUp),
                                  satfrequ - ctrl->conf->loup);
    }

    tmpfreq = gtk_freq_knob_get_value(GTK_FREQ_KNOB(ctrl->RigFreqUp));

    /* if device is engaged, send freq command to radio */
    if ((ctrl->engaged) && (ptt == TRUE) && (fabs(ctrl->lasttxf - tmpfreq) >= 1.0)) {
        if (set_freq_simplex (ctrl, ctrl->conf, tmpfreq)) {
            /* reset error counter */
            ctrl->errcnt = 0;

            /* give radio a chance to set frequency */
            g_usleep (WR_DEL);

            /* The actual frequency migh be different from what we have set because
               the tuning step is larger than what we work with (e.g. FT-817 has a
               smallest tuning step of 10 Hz). Therefore we read back the actual
               frequency from the rig. */
            get_freq_simplex (ctrl, ctrl->conf, &tmpfreq);
            ctrl->lasttxf = tmpfreq;
        }
        else {
            ctrl->errcnt++;
        }
    }
    
}


/** \brief Execute simplex mode cycle.
 *  \param ctrl Pointer to the GtkRigCtrl widget.
 *
 * This function executes a controller cycle when the device is of RIG_TYPE_TRX (simplex).
 * Technically, the function simply checks the PTT status and executes either exec_tx_cycle()
 * or exec_rx_cycle().
 */
static void exec_trx_cycle (GtkRigCtrl *ctrl)
{
    exec_rx_cycle (ctrl);
    exec_tx_cycle (ctrl);
}


/** \brief Execute duplex mode cycle.
 *  \param ctrl Pointer to the GtkRigCtrl widget.
 *
 * This function executes a controller cycle when the device is of RIG_TYPE_DUPLEX.
 */
static void exec_duplex_cycle (GtkRigCtrl *ctrl)
{
	if (ctrl->engaged) {
	
		/* Downlink */
		set_vfo (ctrl, ctrl->conf->vfoDown);
		exec_rx_cycle (ctrl);
		
		/* Uplink */
		set_vfo (ctrl, ctrl->conf->vfoUp);
		exec_tx_cycle (ctrl);
    }
    else {
        /* still execute cycles to update UI widgets
           no data will be sent to RIG */
        exec_rx_cycle (ctrl);
        exec_tx_cycle (ctrl);
    }
}

/** \brief Execute dual-rig cycle.
 *  \param ctrl Pointer to the GtkRigCtrl widget.
 *
 * This function executes a controller cycle when we use a primary device for
 * downlink and a secondary device for uplink.
 */
static void exec_dual_rig_cycle (GtkRigCtrl *ctrl)
{
    gdouble tmpfreq,readfreq,satfreqd,satfrequ;
    gboolean dialchanged = FALSE;
    
    /* Execute downlink cycle using ctrl->conf */
    if (ctrl->engaged && (ctrl->lastrxf > 0.0)) {
        
        /* get frequency from receiver */
        if (!get_freq_simplex (ctrl, ctrl->conf, &readfreq)) {
            /* error => use a passive value */
            readfreq = ctrl->lastrxf;
            ctrl->errcnt++;
        }
        
        if (fabs (readfreq - ctrl->lastrxf) >= 1.0) {
            dialchanged = TRUE;
            
            /* user might have altered radio frequency => update transponder knob */
            gtk_freq_knob_set_value (GTK_FREQ_KNOB (ctrl->RigFreqDown), readfreq);
            ctrl->lastrxf = readfreq;
            
            /* doppler shift; only if we are tracking */
            if (ctrl->tracking) {
                satfreqd = readfreq - ctrl->dd + ctrl->conf->lo;
            }
            else {
                satfreqd = readfreq + ctrl->conf->lo;
            }
            gtk_freq_knob_set_value (GTK_FREQ_KNOB (ctrl->SatFreqDown), satfreqd);
            
            /* Update uplink if locked to downlink */
            if (ctrl->trsplock) {
                track_downlink (ctrl);
            }
        }
    }
    
    if (dialchanged) {
        /* update uplink */
        satfrequ = gtk_freq_knob_get_value (GTK_FREQ_KNOB (ctrl->SatFreqUp));
        if (ctrl->tracking) {
            gtk_freq_knob_set_value (GTK_FREQ_KNOB (ctrl->RigFreqUp),
                                     satfrequ + ctrl->du - ctrl->conf2->loup);
        }
        else {
            gtk_freq_knob_set_value (GTK_FREQ_KNOB (ctrl->RigFreqUp),
                                     satfrequ - ctrl->conf2->loup);
        }

        tmpfreq = gtk_freq_knob_get_value(GTK_FREQ_KNOB(ctrl->RigFreqUp));
        
        /* if device is engaged, send freq command to radio */
        if ((ctrl->engaged) && (fabs(ctrl->lasttxf - tmpfreq) >= 1.0)) {
            if (set_freq_simplex (ctrl, ctrl->conf2, tmpfreq)) {
                /* reset error counter */
                ctrl->errcnt = 0;

                /* give radio a chance to set frequency */
                g_usleep (WR_DEL);

                /* The actual frequency migh be different from what we have set */
                get_freq_simplex (ctrl, ctrl->conf2, &tmpfreq);
                ctrl->lasttxf = tmpfreq;
            }
            else {
                ctrl->errcnt++;
            }
        }
        
    }  /* dialchanged on downlink */
    else {
        /* if no dial change on downlink perform forward tracking on downlink
           and execute uplink controller too.
        */
    
        satfreqd = gtk_freq_knob_get_value (GTK_FREQ_KNOB (ctrl->SatFreqDown));
        if (ctrl->tracking) {
            /* downlink */
            gtk_freq_knob_set_value (GTK_FREQ_KNOB (ctrl->RigFreqDown),
                                     satfreqd + ctrl->dd - ctrl->conf->lo);
        }
        else {
            gtk_freq_knob_set_value (GTK_FREQ_KNOB (ctrl->RigFreqDown),
                                     satfreqd - ctrl->conf->lo);
        }

        tmpfreq = gtk_freq_knob_get_value(GTK_FREQ_KNOB(ctrl->RigFreqDown));

        /* if device is engaged, send freq command to radio */
        if ((ctrl->engaged) && (fabs(ctrl->lastrxf - tmpfreq) >= 1.0)) {
            if (set_freq_simplex (ctrl, ctrl->conf, tmpfreq)) {
                /* reset error counter */
                ctrl->errcnt = 0;

                /* give radio a chance to set frequency */
                g_usleep (WR_DEL);

                /* The actual frequency migh be different from what we have set */
                get_freq_simplex (ctrl, ctrl->conf, &tmpfreq);
                ctrl->lastrxf = tmpfreq;
            }
            else {
                ctrl->errcnt++;
            }
        }

        /*** Now execute uplink controller ***/
        
        /* check if uplink dial has changed */
        if ((ctrl->engaged) && (ctrl->lasttxf > 0.0)) {
        
            if (!get_freq_simplex (ctrl, ctrl->conf2, &readfreq)) {
                /* error => use a passive value */
                readfreq = ctrl->lasttxf;
                ctrl->errcnt++;
            }
        
            if (fabs (readfreq - ctrl->lasttxf) >= 1.0) {
                dialchanged = TRUE;
            
                gtk_freq_knob_set_value (GTK_FREQ_KNOB (ctrl->RigFreqUp), readfreq);
                ctrl->lasttxf = readfreq;
            
                /* doppler shift; only if we are tracking */
                if (ctrl->tracking) {
                    satfrequ = readfreq - ctrl->du + ctrl->conf2->loup;
                }
                else {
                    satfrequ = readfreq + ctrl->conf2->loup;
                }
                gtk_freq_knob_set_value (GTK_FREQ_KNOB (ctrl->SatFreqUp), satfrequ);

                /* Follow with downlink if transponder is locked */
                if (ctrl->trsplock) {
                    track_uplink (ctrl);
                }
            }
        }
        
        if (dialchanged) {  /* on uplink */
            /* update downlink */
            satfreqd = gtk_freq_knob_get_value (GTK_FREQ_KNOB (ctrl->SatFreqDown));
            if (ctrl->tracking) {
                gtk_freq_knob_set_value (GTK_FREQ_KNOB (ctrl->RigFreqDown),
                                         satfreqd + ctrl->dd - ctrl->conf->lo);
            }
            else {
                gtk_freq_knob_set_value (GTK_FREQ_KNOB (ctrl->RigFreqDown),
                                         satfreqd - ctrl->conf->lo);
            }

            tmpfreq = gtk_freq_knob_get_value(GTK_FREQ_KNOB(ctrl->RigFreqDown));
        
            /* if device is engaged, send freq command to radio */
            if ((ctrl->engaged) && (fabs(ctrl->lastrxf - tmpfreq) >= 1.0)) {
                if (set_freq_simplex (ctrl, ctrl->conf, tmpfreq)) {
                    /* reset error counter */
                    ctrl->errcnt = 0;

                    /* give radio a chance to set frequency */
                    g_usleep (WR_DEL);

                    /* The actual frequency migh be different from what we have set */
                    get_freq_simplex (ctrl, ctrl->conf, &tmpfreq);
                    ctrl->lastrxf = tmpfreq;
                }
                else {
                    ctrl->errcnt++;
                }
            }
        } /* dialchanged on uplink */
        else {
            /* perform forward tracking on uplink */
            satfrequ = gtk_freq_knob_get_value (GTK_FREQ_KNOB (ctrl->SatFreqUp));
            if (ctrl->tracking) {
                gtk_freq_knob_set_value (GTK_FREQ_KNOB (ctrl->RigFreqUp),
                                         satfrequ + ctrl->du - ctrl->conf2->loup);
            }
            else {
                gtk_freq_knob_set_value (GTK_FREQ_KNOB (ctrl->RigFreqUp),
                                         satfrequ - ctrl->conf2->loup);
            }

            tmpfreq = gtk_freq_knob_get_value(GTK_FREQ_KNOB(ctrl->RigFreqUp));

            /* if device is engaged, send freq command to radio */
            if ((ctrl->engaged) && (fabs(ctrl->lasttxf - tmpfreq) >= 1.0)) {
                if (set_freq_simplex (ctrl, ctrl->conf2, tmpfreq)) {
                    /* reset error counter */
                    ctrl->errcnt = 0;

                    /* give radio a chance to set frequency */
                    g_usleep (WR_DEL);

                    /* The actual frequency migh be different from what we have set. */
                    get_freq_simplex (ctrl, ctrl->conf2, &tmpfreq);
                    ctrl->lasttxf = tmpfreq;
                }
                else {
                    ctrl->errcnt++;
                }
            }
        } /* else dialchange on uplink */

    } /* else dialchange on downlink */


}


/** \brief Get PTT status
 *  \param ctrl Pointer to the GtkRigVtrl widget.
 *  \return TRUE if PTT is ON, FALSE if PTT is OFF or an error occurred.
 *
 */
static gboolean get_ptt (GtkRigCtrl *ctrl, radio_conf_t *conf)
{
    gchar  *buff,**vbuff;
    gint    written,size;
    gint    status;
    struct hostent *h;
    struct sockaddr_in ServAddr;
    gint  sock;          /*!< Network socket */
    guint64  pttstat = 0;

    
    /* create socket */
    sock = socket(PF_INET, SOCK_STREAM, IPPROTO_TCP);
    if (sock < 0) {
        sat_log_log (SAT_LOG_LEVEL_ERROR,
                     _("%s:%d: Failed to create socket"),
                       __FILE__, __LINE__);
        return FALSE;
    }
    else {
        sat_log_log (SAT_LOG_LEVEL_DEBUG,
                     _("%s:%d Network socket created successfully"),
                       __FILE__, __LINE__);
    }
        
    memset(&ServAddr, 0, sizeof(ServAddr));     /* Zero out structure */
    ServAddr.sin_family = AF_INET;             /* Internet address family */
    h = gethostbyname(conf->host);
    memcpy((char *) &ServAddr.sin_addr.s_addr, h->h_addr_list[0], h->h_length);
    ServAddr.sin_port = htons(conf->port); /* Server port */

    /* establish connection */
    status = connect(sock, (struct sockaddr *) &ServAddr, sizeof(ServAddr));
    if (status < 0) {
        sat_log_log (SAT_LOG_LEVEL_ERROR,
                     _("%s:%d: Failed to connect to %s:%d"),
                       __FILE__, __LINE__, conf->host, conf->port);
        return FALSE;
    }
    else {
        sat_log_log (SAT_LOG_LEVEL_DEBUG,
                     _("%s:%d: Connection opened to %s:%d"),
                       __FILE__, __LINE__, conf->host, conf->port);
    }
    
    if (conf->ptt == PTT_TYPE_CAT) {
        /* send command get_ptt (t) */
        buff = g_strdup_printf ("t");
        size = 1;
    }
    else {
        /* send command \get_dcd */
        buff = g_strdup_printf ("%c",0x8b);
        size = 1;
    }
    
    written = send(sock, buff, size, 0);
    if (written != size) {
        sat_log_log (SAT_LOG_LEVEL_ERROR,
                     _("%s:%d: SIZE ERROR %d / %d"),
                       __FILE__, __LINE__, written, size);
    }
    g_free (buff);
    
    
    /* try to read answer */
    buff = g_try_malloc (128);
    if (buff == NULL) {
        sat_log_log (SAT_LOG_LEVEL_ERROR,
                     _("%s:%s: Failed to allocate 128 bytes (yes, this means trouble)"),
                       __FILE__, __FUNCTION__);
        shutdown (sock, SHUT_RDWR);
        close (sock);
        return FALSE;
    }
        
    size = read (sock, buff, 127);
    if (size == 0) {
        sat_log_log (SAT_LOG_LEVEL_ERROR,
                     _("%s:%s: Got 0 bytes from rigctld"),
                       __FILE__, __FUNCTION__);
    }
    else {
        sat_log_log (SAT_LOG_LEVEL_DEBUG,
                     _("%s:%s: Read %d bytes from rigctld"),
                       __FILE__, __FUNCTION__, size);
        
        buff[size] = 0;
        vbuff = g_strsplit (buff, "\n", 3);
        pttstat = g_ascii_strtoull (vbuff[0], NULL, 0);  //FIXME base = 0 ok?
                
        g_free (buff);
        g_strfreev (vbuff);
    }
    
    shutdown (sock, SHUT_RDWR);
    close (sock);


    return (pttstat == 1) ? TRUE : FALSE;

}


/** \brief Set frequency in simplex mode
 * \param ctrl Pointer to the GtkRigCtrl structure.
 * \param freq The new frequency.
 * \return TRUE if the operation was successful, FALSE if a connection error
 *         occurred.
 * 
 * \note freq is not strictly necessary for normal use since we could have
 *       gotten the current frequency from the ctrl; however, the param
 *       might become useful in the future.
 */
static gboolean set_freq_simplex (GtkRigCtrl *ctrl, radio_conf_t *conf, gdouble freq)
{
    gchar  *buff;
    gint    written,size;
    gint    status;
    struct hostent *h;
    struct sockaddr_in ServAddr;
    gint  sock;          /*!< Network socket */
        
    /* create socket */
    sock = socket(PF_INET, SOCK_STREAM, IPPROTO_TCP);
    if (sock < 0) {
        sat_log_log (SAT_LOG_LEVEL_ERROR,
                     _("%s: Failed to create socket"),
                       __FUNCTION__);
        return FALSE;
    }
    else {
        sat_log_log (SAT_LOG_LEVEL_DEBUG,
                     _("%s: Network socket created successfully"),
                       __FUNCTION__);
    }
        
    memset(&ServAddr, 0, sizeof(ServAddr));     /* Zero out structure */
    ServAddr.sin_family = AF_INET;             /* Internet address family */
    h = gethostbyname(conf->host);
    memcpy((char *) &ServAddr.sin_addr.s_addr, h->h_addr_list[0], h->h_length);
    ServAddr.sin_port = htons(conf->port); /* Server port */

    /* establish connection */
    status = connect(sock, (struct sockaddr *) &ServAddr, sizeof(ServAddr));
    if (status < 0) {
        sat_log_log (SAT_LOG_LEVEL_ERROR,
                     _("%s: Failed to connect to %s:%d"),
                       __FUNCTION__, conf->host, conf->port);
        return FALSE;
    }
    else {
        sat_log_log (SAT_LOG_LEVEL_DEBUG,
                     _("%s: Connection opened to %s:%d"),
                       __FUNCTION__, conf->host, conf->port);
    }
    
    /* send command */
    buff = g_strdup_printf ("F %10.0f", freq);
    
    size = 12;
    written = send(sock, buff, size, 0);
    if (written != size) {
        sat_log_log (SAT_LOG_LEVEL_ERROR,
                     _("%s: SIZE ERROR %d / %d"),
                       __FUNCTION__, written, size);
    }
    g_free (buff);
    shutdown (sock, SHUT_RDWR);
    close (sock);
    
    ctrl->wrops++;
    
    return TRUE;
}


/** \brief Get frequency
 * \param ctrl Pointer to the GtkRigCtrl structure.
 * \param freq The current frequency of the radio.
 * \return TRUE if the operation was successful, FALSE if a connection error
 *         occurred.
 */
static gboolean get_freq_simplex (GtkRigCtrl *ctrl, radio_conf_t *conf, gdouble *freq)
{
    gchar  *buff,**vbuff;
    gint    written,size;
    gint    status;
    struct hostent *h;
    struct sockaddr_in ServAddr;
    gint  sock;          /*!< Network socket */


    if (freq == NULL) {
        sat_log_log (SAT_LOG_LEVEL_BUG,
                     _("%s:%d: NULL storage."),
                       __FILE__, __LINE__);
        return FALSE;
    }
    
    /* create socket */
    sock = socket(PF_INET, SOCK_STREAM, IPPROTO_TCP);
    if (sock < 0) {
        sat_log_log (SAT_LOG_LEVEL_ERROR,
                     _("%s:%d: Failed to create socket"),
                       __FILE__, __LINE__);
        return FALSE;
    }
    else {
        sat_log_log (SAT_LOG_LEVEL_DEBUG,
                     _("%s:%d Network socket created successfully"),
                       __FILE__, __LINE__);
    }
        
    memset(&ServAddr, 0, sizeof(ServAddr));     /* Zero out structure */
    ServAddr.sin_family = AF_INET;             /* Internet address family */
    h = gethostbyname(conf->host);
    memcpy((char *) &ServAddr.sin_addr.s_addr, h->h_addr_list[0], h->h_length);
    ServAddr.sin_port = htons(conf->port); /* Server port */

    /* establish connection */
    status = connect(sock, (struct sockaddr *) &ServAddr, sizeof(ServAddr));
    if (status < 0) {
        sat_log_log (SAT_LOG_LEVEL_ERROR,
                     _("%s:%d: Failed to connect to %s:%d"),
                       __FILE__, __LINE__, conf->host, conf->port);
        return FALSE;
    }
    else {
        sat_log_log (SAT_LOG_LEVEL_DEBUG,
                     _("%s:%d: Connection opened to %s:%d"),
                       __FILE__, __LINE__, conf->host, conf->port);
    }
    
    /* send command */
    buff = g_strdup_printf ("f");
    
    size = 1;
    written = send(sock, buff, size, 0);
    if (written != size) {
        sat_log_log (SAT_LOG_LEVEL_ERROR,
                     _("%s:%d: SIZE ERROR %d / %d"),
                       __FILE__, __LINE__, written, size);
    }
    g_free (buff);
    
    
    /* try to read answer */
    buff = g_try_malloc (128);
    if (buff == NULL) {
        sat_log_log (SAT_LOG_LEVEL_ERROR,
                     _("%s:%s: Failed to allocate 128 bytes (yes, this means trouble)"),
                       __FILE__, __FUNCTION__);
        shutdown (sock, SHUT_RDWR);
        close (sock);
        return FALSE;
    }
        
    size = read (sock, buff, 127);
    if (size == 0) {
        sat_log_log (SAT_LOG_LEVEL_ERROR,
                     _("%s:%s: Got 0 bytes from rigctld"),
                       __FILE__, __FUNCTION__);
    }
    else {
        sat_log_log (SAT_LOG_LEVEL_DEBUG,
                     _("%s:%s: Read %d bytes from rigctld"),
                       __FILE__, __FUNCTION__, size);
        
        buff[size] = 0;
        vbuff = g_strsplit (buff, "\n", 3);
        *freq = g_ascii_strtod (vbuff[0], NULL);
                
        g_free (buff);
        g_strfreev (vbuff);
    }
    
    shutdown (sock, SHUT_RDWR);
    close (sock);


    return TRUE;
}

/** \brief Select target VFO
 * \param ctrl Pointer to the GtkRigCtrl structure.
 * \param vfo The VFO to select
 * \return TRUE if the operation was successful, FALSE if a connection error
 *         occurred.
 * 
 */
static gboolean set_vfo (GtkRigCtrl *ctrl, vfo_t vfo)
{
    gchar  *buff;
    gint    written,size;
    gint    status;
    struct hostent *h;
    struct sockaddr_in ServAddr;
    gint  sock;          /*!< Network socket */
        
    /* create socket */
    sock = socket(PF_INET, SOCK_STREAM, IPPROTO_TCP);
    if (sock < 0) {
        sat_log_log (SAT_LOG_LEVEL_ERROR,
                     _("%s: Failed to create socket"),
                       __FUNCTION__);
        return FALSE;
    }
    else {
        sat_log_log (SAT_LOG_LEVEL_DEBUG,
                     _("%s: Network socket created successfully"),
                       __FUNCTION__);
    }
        
    memset(&ServAddr, 0, sizeof(ServAddr));     /* Zero out structure */
    ServAddr.sin_family = AF_INET;             /* Internet address family */
    h = gethostbyname(ctrl->conf->host);
    memcpy((char *) &ServAddr.sin_addr.s_addr, h->h_addr_list[0], h->h_length);
    ServAddr.sin_port = htons(ctrl->conf->port); /* Server port */

    /* establish connection */
    status = connect(sock, (struct sockaddr *) &ServAddr, sizeof(ServAddr));
    if (status < 0) {
        sat_log_log (SAT_LOG_LEVEL_ERROR,
                     _("%s: Failed to connect to %s:%d"),
                       __FUNCTION__, ctrl->conf->host, ctrl->conf->port);
        return FALSE;
    }
    else {
        sat_log_log (SAT_LOG_LEVEL_DEBUG,
                     _("%s: Connection opened to %s:%d"),
                       __FUNCTION__, ctrl->conf->host, ctrl->conf->port);
    }
    
    /* prepare command */
    switch (vfo) {
    case VFO_A:
    	buff = g_strdup_printf ("V VFOA");
        size = 6;
        break;
        
    case VFO_B:
    	buff = g_strdup_printf ("V VFOB");
        size = 6;
        break;
        
    case VFO_MAIN:
    	buff = g_strdup_printf ("V Main");
        size = 6;
        break;
        
    case VFO_SUB:
    	buff = g_strdup_printf ("V Sub");
        size = 5;
        break;
        
     default:
        sat_log_log (SAT_LOG_LEVEL_BUG,
                     _("%s: Invalid VFO argument. Using VFOA."),
                     __FUNCTION__);
    	buff = g_strdup_printf ("V VFOA");
        size = 6;
        break;
	}    
	
    /* send command */
    written = send(sock, buff, size, 0);
    if (written != size) {
        sat_log_log (SAT_LOG_LEVEL_ERROR,
                     _("%s: SIZE ERROR %d / %d"),
                       __FUNCTION__, written, size);
    }
    g_free (buff);
    shutdown (sock, SHUT_RDWR);
    close (sock);
    
    ctrl->wrops++;
    
    return TRUE;
}


/** \brief Update count down label.
 * \param[in] ctrl Pointer to the RigCtrl widget.
 * \param[in] t The current time.
 * 
 * This function calculates the new time to AOS/LOS of the currently
 * selected target and updates the ctrl->SatCnt label widget.
 */
static void update_count_down (GtkRigCtrl *ctrl, gdouble t)
{
    gdouble  targettime;
    gdouble  delta;
    gchar   *buff;
    guint    h,m,s;
    gchar   *ch,*cm,*cs;
    gchar   *aoslos;

    
    /* select AOS or LOS time depending on target elevation */
    if (ctrl->target->el < 0.0) {
        targettime = ctrl->target->aos;
        aoslos = g_strdup_printf (_("AOS in"));
    }
    else {
        targettime = ctrl->target->los;
        aoslos = g_strdup_printf (_("LOS in"));
    }
    
    delta = targettime - t;
    
    /* convert julian date to seconds */
    s = (guint) (delta * 86400);

    /* extract hours */
    h = (guint) floor (s/3600);
    s -= 3600*h;

    /* leading zero */
    if ((h > 0) && (h < 10))
        ch = g_strdup ("0");
    else
        ch = g_strdup ("");

    /* extract minutes */
    m = (guint) floor (s/60);
    s -= 60*m;

    /* leading zero */
    if (m < 10)
        cm = g_strdup ("0");
    else
        cm = g_strdup ("");

    /* leading zero */
    if (s < 10)
        cs = g_strdup (":0");
    else
        cs = g_strdup (":");

    if (h > 0) 
        buff = g_strdup_printf ("<span size='xx-large'><b>%s %s%d:%s%d%s%d</b></span>",
                                aoslos, ch, h, cm, m, cs, s);
    else
        buff = g_strdup_printf ("<span size='xx-large'><b>%s %s%d%s%d</b></span>",
                                aoslos, cm, m, cs, s);

    gtk_label_set_markup (GTK_LABEL (ctrl->SatCnt), buff);

    g_free (buff);
    g_free (aoslos);
    g_free (ch);
    g_free (cm);
    g_free (cs);

}


/** \brief Load the transponder list for the target satellite.
 *  \param ctrl Pointer to the GtkRigCtrl structure.
 *
 * This function loads the transponder list for the currently selected
 * satellite. The transponder list is loaded into ctrl->trsplist and the
 * transponder names are added to the ctrl->TrspSel combo box. If any of
 * these already contain data, it is cleared. The combo box is also cleared
 * if there are no transponders for the current target, or if there is no
 * target.
 */
static void load_trsp_list (GtkRigCtrl *ctrl)
{
    trsp_t *trsp = NULL;
    guint i,n;
    
    if (ctrl->trsplist != NULL) {
        /* clear combo box */
        n = g_slist_length (ctrl->trsplist);
        for (i = 0; i < n; i++) {
            gtk_combo_box_remove_text (GTK_COMBO_BOX (ctrl->TrspSel), 0);
        }
        
        /* clear transponder list */
        free_transponders (ctrl->trsplist);
        
        ctrl->trsp = NULL;
    }
    
    /* check if there is a target satellite */
    if G_UNLIKELY (ctrl->target == NULL) {
        sat_log_log (SAT_LOG_LEVEL_MSG,
                      _("%s:%s: GtkSatModule has no target satellite."),
                      __FILE__, __FUNCTION__);
        return;
    }

    /* read transponders for new target */
    ctrl->trsplist = read_transponders (ctrl->target->tle.catnr);
    
    /* append transponder names to combo box */
    n = g_slist_length (ctrl->trsplist);
    
    sat_log_log (SAT_LOG_LEVEL_DEBUG,
                 _("%s:%s: Satellite %d has %d transponder modes."),
                 __FILE__, __FUNCTION__, ctrl->target->tle.catnr, n);
    
    if (n == 0)
        return;
    
    for (i = 0; i < n; i++) {
        trsp = (trsp_t *) g_slist_nth_data (ctrl->trsplist, i);
        gtk_combo_box_append_text (GTK_COMBO_BOX (ctrl->TrspSel), trsp->name);
        
        sat_log_log (SAT_LOG_LEVEL_DEBUG,
                     _("%s:&s: Read transponder '%s' for satellite %d"),
                     __FILE__, __FUNCTION__, trsp->name, ctrl->target->tle.catnr);
    }
    
    /* make an initial selection */
    ctrl->trsp = (trsp_t *) g_slist_nth_data (ctrl->trsplist, 0);
    gtk_combo_box_set_active (GTK_COMBO_BOX (ctrl->TrspSel), 0);
}


/** \brief Check that we have at least one .rig file */
static gboolean have_conf ()
{
    GDir        *dir = NULL;   /* directory handle */
    GError      *error = NULL; /* error flag and info */
    gchar       *dirname;      /* directory name */
    const gchar *filename;     /* file name */
    gint         i = 0;

    
    /* open configuration directory */
    dirname = get_hwconf_dir ();
    
    dir = g_dir_open (dirname, 0, &error);
    if (dir) {
        /* read each .rig file */
        while ((filename = g_dir_read_name (dir))) {
            
            if (g_strrstr (filename, ".rig")) {
                i++;
            }
        }
    }
    else {
        sat_log_log (SAT_LOG_LEVEL_ERROR,
                     _("%s:%d: Failed to open hwconf dir (%s)"),
                       __FILE__, __LINE__, error->message);
        g_clear_error (&error);
    }

    g_free (dirname);
    g_dir_close (dir);
    
    return (i > 0) ? TRUE : FALSE;
}


/** \brief Track the downlink frequency.
 *  \param ctrl Pointer to the GtkRigCtrl structure.
 *
 * This function tracks the downlink frequency by setting the uplink frequency
 * according to the lower limit of the downlink passband.
 */
static void track_downlink (GtkRigCtrl *ctrl)
{
    gdouble delta,down,up;
    
    if (ctrl->trsp == NULL) {
        return;
    }
    
    /* ensure that we have a useable transponder config */
    if ((ctrl->trsp->downlow > 0.0) && (ctrl->trsp->uplow > 0.0)) {
    
        down = gtk_freq_knob_get_value (GTK_FREQ_KNOB (ctrl->SatFreqDown));
        delta = down - ctrl->trsp->downlow;
    
        if (ctrl->trsp->invert) {
            up = ctrl->trsp->uphigh - delta;
        }
        else {
            up = ctrl->trsp->uplow + delta;
        }
    
        gtk_freq_knob_set_value (GTK_FREQ_KNOB (ctrl->SatFreqUp), up);
    }
}


/** \brief Track the uplink frequency.
 *  \param ctrl Pointer to the GtkRigCtrl structure.
 *
 * This function tracks the uplink frequency by setting the downlink frequency
 * according to the offset from the lower limit on the uplink passband.
 */
static void track_uplink (GtkRigCtrl *ctrl)
{
    gdouble delta,down,up;
    
    if (ctrl->trsp == NULL) {
        return;
    }

    /* ensure that we have a useable transponder config */
    if ((ctrl->trsp->downlow > 0.0) && (ctrl->trsp->uplow > 0.0)) {

        up = gtk_freq_knob_get_value (GTK_FREQ_KNOB (ctrl->SatFreqUp));
        delta = up - ctrl->trsp->uplow;
    
        if (ctrl->trsp->invert) {
            down = ctrl->trsp->downhigh - delta;
        }
        else {
            down = ctrl->trsp->downlow + delta;
        }
    
        gtk_freq_knob_set_value (GTK_FREQ_KNOB (ctrl->SatFreqDown), down);
    }
}


/** \brief Check whether a radio configuration is TX capable.
 *  \param confname The name of the configuration to check.
 *  \return TRUE if the radio is TX capable, FALSE otherwise.
 */
static gboolean is_rig_tx_capable (const gchar *confname)
{
    radio_conf_t *conf = NULL;
    gboolean cantx = FALSE;
    
    conf = g_try_new (radio_conf_t, 1);
    if (conf == NULL) {
        sat_log_log (SAT_LOG_LEVEL_ERROR,
                     _("%s:%d: Failed to allocate memory for radio config"),
                     __FILE__, __LINE__);
        return FALSE;
    }
    
    /* load new configuration */
    conf->name = g_strdup (confname);
    if (radio_conf_read (conf)) {
        cantx = (conf->type == RIG_TYPE_RX) ? FALSE : TRUE;
    }
    else {
        sat_log_log (SAT_LOG_LEVEL_ERROR,
                     _("%s:%d: Error reading radio configuration %s"),
                     __FILE__, __LINE__, confname);
        
        
        cantx = FALSE;
    }
 
    g_free (conf->name);
    if (conf->host)
        g_free (conf->host);
    g_free (conf);
    
    return cantx;
}
