/* gxmessage - an xmessage clone using GTK2
 *
 * $Id: gxmessage.c,v 1.18 2005/02/22 09:22:38 trm Exp trm $
 *
 * Copyright Timothy Musson 2001 - 2005, <trmusson@ihug.co.nz>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * http://www.gnu.org/copyleft/gpl.html
 *
 */



#include <config.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <gtk/gtk.h>


#ifdef ENABLE_NLS
#  include <libintl.h>
#  define _(String) gettext(String)
#  ifdef gettext_noop
#    define N_(String) gettext_noop(String)
#  else
#    define N_(String) (String)
#  endif
#else
#  define _(String) (String)
#  define N_(String) (String)
#  define textdomain(String) (String)
#  define gettext(String) (String)
#  define dgettext(Domain,String) (String)
#  define dcgettext(Domain,String,Type) (String)
#  define bindtextdomain(Domain,Directory) (Domain)
#endif /* ENABLE_NLS */


#define FNAME_ICON    PACKAGE_DATA_DIR G_DIR_SEPARATOR_S \
                      "pixmaps" G_DIR_SEPARATOR_S "gxmessage.png"


typedef struct _ButtonList ButtonList;



struct {
	gchar             *message_text;
	gint              message_length;
	ButtonList        *button_list;
	const gchar       *default_str;
	const gchar       *title_str;
	const gchar       *geom_str;
	const gchar       *font_str;
	const gchar       *color_fg;
	const gchar       *color_bg;
	const gchar       *encoding;
	const gchar       *entry_str;
	gint              timeout;
	guint             timeout_id;
	gboolean          do_iconify;
	gboolean          do_print;
	gboolean          do_buttons;
	gboolean          do_borderless;
	gboolean          do_focus;
	GtkWrapMode       wrap_mode;
	GtkWindowPosition window_position;
	GtkWidget         *entry_widget;
	gint              exit_code;
} gx;


struct _ButtonList {
	gboolean          is_default;
	gint              value;
	const gchar       *label_str;
	ButtonList        *prev;
	ButtonList        *next;
};


struct Option {
	gint     min_len; /* support -bu, -but, -butt, etc., as with xmessage */
	gboolean requires_arg;
	gchar    *opt_str;
};


static struct Option option[] = { /* min_len, requires_arg, opt_str */
	{2, TRUE,  "buttons"},
	{1, FALSE, "center"},
	{2, TRUE,  "default"},
	{2, TRUE,  "file"},
	{2, FALSE, "nearmouse"},
	{1, FALSE, "print"},
	{3, TRUE,  "timeout"},
	{2, TRUE,  "fn"},
	{2, TRUE,  "font"},
	{1, TRUE,  "geometry"},
	{3, TRUE,  "title"},
	{2, TRUE,  "bg"},
	{2, TRUE,  "fg"},
	{2, TRUE,  "bd"},
	{2, TRUE,  "bw"},
	{1, FALSE, "iconic"},
	{2, TRUE,  "xrm"},
	{2, FALSE, "rv"},
	{2, FALSE, "reverse"},
	{2, TRUE,  "selectionTimeout"},
	{2, FALSE, "synchronous"},
	{2, TRUE,  "xnllanguage"},
	{2, FALSE, "borderless"},
	{1, FALSE, "wrap"},
	{3, TRUE,  "encoding"},
	{3, FALSE, "nofocus"},
	{6, TRUE,  "entrytext"},
	{3, FALSE, "entry"},
	{1, FALSE, "?"},
	{1, FALSE, "help"},
	{1, FALSE, "version"}
};

enum {
	OPT_IS_UNKNOWN = -2,
	OPT_IS_MISSING_ARG,
	OPT_BUTTONS,
	OPT_CENTER,
	OPT_DEFAULT,
	OPT_FILE,
	OPT_NEARMOUSE,
	OPT_PRINT,
	OPT_TIMEOUT,
	OPT_FN,
	OPT_FONT,
	OPT_GEOMETRY,
	OPT_TITLE,
	OPT_BG,
	OPT_FG,
	OPT_BD,
	OPT_BW,
	OPT_ICONIC,
	OPT_XRM,
	OPT_RV,
	OPT_REVERSE,
	OPT_SELECTIONTIMEOUT,
	OPT_SYNCHRONOUS,
	OPT_XNLLANGUAGE,
	OPT_BORDERLESS,
	OPT_WRAP,
	OPT_ENCODING,
	OPT_FOCUS,
	OPT_ENTRYTEXT,
	OPT_ENTRY,
	OPT_HELP_Q,
	OPT_HELP,
	OPT_VERSION,
	N_OPTS
};



void prog_cleanup (void);



ButtonList*
button_first (ButtonList *button)
{
	if (button != NULL) {
		while (button->prev != NULL) {
			button = button->prev;
		}
	}
	return button;
}



void
button_free (ButtonList *button)
{
	ButtonList *next;

	button = button_first (button);
	while (button != NULL) {
		next = button->next;
		g_free (button);
		button = next;
	}
}



ButtonList*
button_append (ButtonList *button, ButtonList *new_button)
{
	if (button != NULL) {
		new_button->prev = button;
		button->next = new_button;
	}
	return new_button;
}



ButtonList*
button_new_from_str (gchar *str, gint default_value, gint len)
{
	/* Given a string containing "label" or "label:value", fill in a
	 * new button structure and return it.
	 */

	ButtonList *button = g_new0 (ButtonList, 1);
	gboolean value_from_str = FALSE;
	gint i = len;


	/* parse "label:value" format in str */
	str[len] = '\0';
	while (i != 0 && !value_from_str) value_from_str = str[--i] == ':';
	if (value_from_str) {
		/* get label and value from str */
		str[i] = '\0';
		button->label_str = str;
		button->value = atoi (&str[i + 1]);
	}
	else {
		/* get label from str, value from default_value */
		button->label_str = str;
		button->value = default_value;
	}

	/* xmessage will only draw the buttons if at least one of
	 * them has a label. To emulate that, keep track of it here:
	 */
	gx.do_buttons = gx.do_buttons || len > 0;

	return button;
}



ButtonList*
button_list_from_str (gchar *str)
{
	/* Given a string containing a comma-separated list of "label" and/or
	 * "label:value", build and return a list of buttons. The return points
	 * to the last button in the list.
	 */

	ButtonList *button = NULL;
	ButtonList *b;
	gint len;
	gint i = 0;
	gint startpos = 0;
	gint value = 101;


	if (str == NULL) return NULL;

	len = strlen (str);
	while (i < len) {
		if (str[i] == ',') {
			b = button_new_from_str (&str[startpos], value, i - startpos);
			button = button_append (button, b);
			startpos = i + 1;
			value++;
		}
		i++;
	}

	b = button_new_from_str (&str[startpos], value, i - startpos);
	button = button_append (button, b);

	return button;
}



void
button_set_default (ButtonList *button, const gchar *str)
{
	/* Make button->is_default TRUE for each button whose label matches str */

	if (str == NULL) return;

	button = button_first (button);
	while (button != NULL) {
		button->is_default = strcmp (button->label_str, str) == 0;
		button = button->next;
	}
}



gint
my_get_opt (const gchar *str, gboolean not_last)
{
	/* Try to identify the command line option in str, returning a unique
	 * option/error code. The not_last variable specifies whether current
	 * option was followed by something else (a value or another option)
	 * on the command line.
	 */

	gint opt;
	gint len;
	gint offset = 1;


	if (strcmp (str, "+rv") == 0) return OPT_RV;
	if (str[0] != '-') return OPT_IS_UNKNOWN;

	len = strlen (&str[offset]);
	if (len > 0) {
		if (str[1] == '-') {
			offset++;
			len--;
		}
		if (len > 0) {
			for (opt = 0; opt < N_OPTS; opt++) {
				if (len >= option[opt].min_len &&
				    strncmp (&str[offset], option[opt].opt_str, len) == 0) {
					if (!option[opt].requires_arg || not_last) {
						return opt;
					}
					else {
						return OPT_IS_MISSING_ARG;
					}
				}
			}
		}
	}
	return OPT_IS_UNKNOWN;
}



void
cb_window_destroy (GtkWidget *widget, GdkEvent *event, gpointer data)
{
	gtk_main_quit ();
}



void
cb_button_clicked (GtkWidget *widget, gpointer data)
{
	gx.exit_code = ((ButtonList*)data)->value;

	if (gx.do_print) {
		g_print ("%s\n", ((ButtonList*)data)->label_str);
	}
	else if (gx.entry_str != NULL) {
		g_print ("%s\n", gtk_entry_get_text (GTK_ENTRY(gx.entry_widget)));
	}
	gtk_main_quit ();
}



void
cb_entry_activated (GtkWidget *widget, gpointer data)
{
	gx.exit_code = 0;
	g_print ("%s\n", gtk_entry_get_text (GTK_ENTRY(gx.entry_widget)));
	gtk_main_quit ();
}



gint
cb_timeout (gpointer data)
{
	static gint counter = 0;

	if (++counter >= (gint)data) {
		gx.exit_code = 0;
		gtk_main_quit ();
	}
	return TRUE;
}



gint
geometry_get_number (const gchar **geometry)
{
	gint value = 0;
	gint mult  = 1;


	if (**geometry == '-') {
		mult = -1;
		(*geometry)++;
	}
	while ( **geometry && isdigit(**geometry) ) {
		value = value * 10 + (**geometry - '0');
		(*geometry)++;
	}
	return value * mult;
}



gboolean
geometry_parse_string (const gchar *geometry,
                       gint *x, gint *y, gint *w, gint *h)
{
	/* Parse geometry string into x, y, w and h. Return TRUE if successful,
	 * else FALSE. (This function and geometry_get_number are taken from
	 * the GNOME 1.x gnome-geometry.c)
	 */

	gint subtract;


	g_return_val_if_fail (x != NULL, FALSE);
	g_return_val_if_fail (y != NULL, FALSE);
	g_return_val_if_fail (w != NULL, FALSE);
	g_return_val_if_fail (h != NULL, FALSE);

	*x = *y = *w = *h = -1;

	if (geometry == NULL) return FALSE;

	if (*geometry == '=') geometry++;
	if (!*geometry) return FALSE;

	if (isdigit (*geometry)) {
		*w = geometry_get_number (&geometry);
	}
	if (!*geometry) {
		return TRUE;
	}
	if (*geometry == 'x') {
		geometry++;
		*h = geometry_get_number (&geometry);
	}
	if (!*geometry) {
		return TRUE;
	}


	if (*geometry == '+') {
		subtract = 0;
		geometry++;
	}
	else if (*geometry == '-') {
		subtract = gdk_screen_width ();
		geometry++;
	}
	else {
		return FALSE;
	}


	*x = geometry_get_number (&geometry);
	if (subtract) {
		*x = subtract - *x;
	}
	if (!*geometry) {
		return TRUE;
	}


	if (*geometry == '+') {
		subtract = 0;
		geometry++;
	}
	else if (*geometry == '-') {
		subtract = gdk_screen_height ();
		geometry++;
	}
	else {
		return FALSE;
	}


	*y = geometry_get_number (&geometry);
	if (subtract) {
		*y = subtract - *y;
	}

	return TRUE;
}



void
window_create (void)
{
	GtkWidget  *window;
	GtkWidget  *vbox, *vbox2;
	GtkWidget  *scroller;
	GtkWidget  *button_box = NULL;
	GtkWidget  *widget = NULL;
	GtkWidget  *message_widget;
	ButtonList *button_list;
	GdkColor   color;
	PangoFontDescription *font_desc;
	GtkRequisition size_req;
	GtkTextBuffer *buf;
	gint win_w, win_h;


	window = gtk_window_new (GTK_WINDOW_TOPLEVEL);


	g_signal_connect (G_OBJECT(window),
	                  "destroy",
	                  G_CALLBACK(cb_window_destroy),
	                  NULL);


	if (gx.title_str != NULL) {
		gtk_window_set_title (GTK_WINDOW(window), gx.title_str);
	}

	if (gx.do_iconify) {
		gtk_window_iconify (GTK_WINDOW(window));
	}

	gtk_window_set_icon_from_file (GTK_WINDOW(window), FNAME_ICON, NULL);

	if (gx.do_borderless) {
		gtk_window_set_decorated (GTK_WINDOW(window), FALSE);
	}
#if _GTK_2_0
#else
	/* requires GTK >= 2.4.0 */
	gtk_window_set_accept_focus (GTK_WINDOW(window), gx.do_focus);
#endif


	/* window contents */
	gtk_container_set_border_width (GTK_CONTAINER(window), 12);

	vbox = gtk_vbox_new (FALSE, 12);
	gtk_container_add (GTK_CONTAINER(window), vbox);

	vbox2 = gtk_vbox_new (FALSE, 0);
	gtk_box_pack_start (GTK_BOX(vbox), vbox2, TRUE, TRUE, 0);
	gtk_container_set_border_width (GTK_CONTAINER(vbox2), 0);

	scroller = gtk_scrolled_window_new (NULL, NULL);
	gtk_container_set_border_width (GTK_CONTAINER(scroller), 0);
	gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW(scroller), GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
	gtk_scrolled_window_set_shadow_type (GTK_SCROLLED_WINDOW(scroller), GTK_SHADOW_ETCHED_IN);
	gtk_scrolled_window_set_placement (GTK_SCROLLED_WINDOW(scroller), GTK_CORNER_TOP_LEFT);
	gtk_box_pack_start (GTK_BOX(vbox2), scroller, TRUE, TRUE, 0);


	/* the message */
	message_widget = gtk_text_view_new ();
	gtk_text_view_set_editable (GTK_TEXT_VIEW(message_widget), FALSE);
	gtk_text_view_set_cursor_visible (GTK_TEXT_VIEW(message_widget), TRUE);
	gtk_text_view_set_wrap_mode (GTK_TEXT_VIEW(message_widget), gx.wrap_mode);

	if (gx.font_str != NULL) {
		font_desc = pango_font_description_from_string (gx.font_str);
	}
	else {
		/* default to a monospaced font */
		font_desc = pango_font_description_from_string ("monospace");
	}
	gtk_widget_modify_font (message_widget, font_desc);
	pango_font_description_free (font_desc);

	gtk_container_add (GTK_CONTAINER(scroller), message_widget);

	buf = gtk_text_view_get_buffer (GTK_TEXT_VIEW(message_widget));
	gtk_text_buffer_set_text (buf, gx.message_text, gx.message_length);

	if (gx.color_fg != NULL) {
		if (gdk_color_parse (gx.color_fg, &color)) {
			gtk_widget_modify_text (message_widget, GTK_STATE_NORMAL, &color);
		}
	}
	if (gx.color_bg != NULL) {
		if (gdk_color_parse (gx.color_bg, &color)) {
			gtk_widget_modify_base (message_widget, GTK_STATE_NORMAL, &color);
		}
	}


	/* text entry */
	if (gx.entry_str != NULL) {
		gx.entry_widget = gtk_entry_new ();
		gtk_editable_set_editable (GTK_EDITABLE(gx.entry_widget), TRUE);
		gtk_entry_set_text (GTK_ENTRY(gx.entry_widget), gx.entry_str);
		gtk_box_pack_start (GTK_BOX(vbox), gx.entry_widget, FALSE, FALSE, 5);
		gtk_widget_grab_focus (gx.entry_widget);
		if (!gx.do_buttons) {
			/* allow hitting <RETURN> to close the window */
			g_signal_connect (G_OBJECT(gx.entry_widget),
			                  "activate",
			                  G_CALLBACK(cb_entry_activated),
			                  (gpointer)0);
		}
	}


	/* add buttons */
	if (gx.do_buttons) {

		button_list = button_first (gx.button_list);

		button_box = gtk_hbutton_box_new ();
		gtk_button_box_set_layout (GTK_BUTTON_BOX(button_box),
		                           GTK_BUTTONBOX_END);
		gtk_box_set_spacing (GTK_BOX(button_box), 6);
		gtk_box_pack_end (GTK_BOX(vbox), button_box, FALSE, FALSE, 0);

		while (button_list != NULL) {

			if (g_str_has_prefix (button_list->label_str, "GTK_STOCK_")) {
				gchar *s;
				gchar *p;
				p = g_ascii_strdown (button_list->label_str + 10, -1);
				s = p;
				while (*s != '\0') {
					if (*s == '_') *s = '-';
					s++;
				}
				s = g_strconcat ("gtk-", p, NULL);
				widget = gtk_button_new_from_stock (s);
				g_free (s);
				g_free (p);
				if (widget == NULL) {
					widget = gtk_button_new_with_label (button_list->label_str);
				}
			}
			else {
				widget = gtk_button_new_with_mnemonic (button_list->label_str);
			}

			g_signal_connect (G_OBJECT(widget),
			                  "clicked",
			                  G_CALLBACK(cb_button_clicked),
			                  (gpointer)button_list);
			gtk_box_pack_start (GTK_BOX(button_box), widget, FALSE, FALSE, 0);

			if (button_list->is_default) {
				gtk_widget_grab_focus (widget);
			}

			button_list = button_list->next;
		}

	}


	/* window geometry */

	win_w = 330;  /* XXX */
	win_h = 90;   /* XXX */

	if (gx.entry_str != NULL) {
		gtk_widget_realize (gx.entry_widget);
		gtk_widget_size_request (gx.entry_widget, &size_req);
		win_h = win_h + size_req.height + 10;
	}
	if (gx.do_buttons) {
		gtk_widget_size_request (button_box, &size_req);
		win_h = win_h + size_req.height;
	}

	if (gx.geom_str != NULL) {
		gint tmp_x, tmp_y, tmp_w, tmp_h;
		if (geometry_parse_string (gx.geom_str, &tmp_x, &tmp_y, &tmp_w, &tmp_h)) {
			if (win_w != -1) {
				win_w = tmp_w;
				win_h = tmp_h;
			}
		}
	}

	gtk_window_set_default_size (GTK_WINDOW(window), win_w, win_h);
	gtk_window_set_position (GTK_WINDOW(window), gx.window_position);

	/* open the window */
	gtk_widget_show_all (window);

	/* begin timeout */
	if (gx.timeout != 0) {
		gx.timeout_id = gtk_timeout_add (1000, (GtkFunction)cb_timeout,
		                                 (gpointer)gx.timeout);
	}
}



gchar*
read_stdin (void)
{
	GString *text;
	gchar *str;
	gint ch;

	text = g_string_new ("");

	while ( (ch = getc (stdin)) != EOF ) {
		g_string_append_c (text, ch);
	}
	str = text->str;
	g_string_free (text, FALSE);
	return str;
}



gchar*
message_to_utf8 (const gchar *str)
{
	gchar *result;
	GError *error = NULL;

	if (gx.encoding == NULL) {
		/* assume message encoding matches current locale */
		result = g_locale_to_utf8 (str, -1, NULL, NULL, NULL);
	}
	else {
		/* use encoding specified on command line */
		result = g_convert_with_fallback (str, -1, "UTF-8", gx.encoding,
		                                  NULL, NULL, NULL, NULL);
	}

	if (result == NULL) {
		/* fall back to ISO-8859-1 as source encoding */
		result = g_convert_with_fallback (str, -1, "UTF-8", "ISO-8859-1",
		                                  NULL, NULL, NULL, &error);
		if (result == NULL) {
			if (error != NULL && error->message != NULL) {
				g_printerr (PACKAGE ": %s\n", error->message);
			}
			prog_cleanup ();
		}
	}
	return result;
}



void
option_kludge (gint *argc, gchar ***argv)
{
	/* Vet argv before gtk_init sees it */

	gint i, len;

	for (i = 1; i < *argc; i++) {
		len = strlen ((*argv)[i]);
		if (len > 2) {
			if (strncmp ("-display", (*argv)[i], len) == 0) {
				(*argv)[i] = "--display";
				if (i + 1 < *argc && (*argv)[i + 1] != NULL) i++;
			}
			else if (strncmp ("-name", (*argv)[i], len) == 0) {
				(*argv)[i] = "--name";
				if (i + 1 < *argc && (*argv)[i + 1] != NULL) i++;
			}
		}
	}

#if _GTK_2_6
	/* work around the removal of '-' by gtk_init >= 2.6.0 */
	for (i = 2; i < *argc; i++) {
		if (strcmp ((*argv)[i], "-") == 0 &&
		    (strcmp ((*argv)[i - 1], "-file") == 0 ||
		     strcmp ((*argv)[i - 1], "--file") == 0)) {
			(*argv)[i] = "gxmsg:stdin";
		}
	}
#endif
}



void
usage (void)
{
	g_print(_("\n%s - a GTK-based xmessage clone\n\n"), PACKAGE);
	g_print(_("Usage: %s [OPTIONS] message ...\n"), PACKAGE);
	g_print(_("       %s [OPTIONS] -file FILENAME\n\n"), PACKAGE);
	g_print(_("xmessage options:\n" \
	        "  -file FILENAME         Get message text from file, '-' for stdin\n" \
	        "  -buttons BUTTON_LIST   List of \"LABEL:EXIT_CODE\", comma separated\n" \
	        "  -default LABEL         Give keyboard focus to the specified button\n" \
	        "  -print                 Send the selected button's LABEL to stdout\n" \
	        "  -center                Try to open window in the centre of the screen\n" \
	        "  -nearmouse             Try to open window near the mouse pointer\n" \
	        "  -timeout SECONDS       Exit with code 0 after SECONDS seconds\n\n"));
	g_print(_("X-like options:\n" \
	        "  -display DISPLAY       X display to use\n" \
	        "  -fn FONT | -font FONT  Set message font (works with GTK font names)\n" \
	        "  -fg COLOUR             Set message font colour\n" \
	        "  -bg COLOUR             Set message background colour\n" \
	        "  -geometry GEOMETRY     Set window size (position will be ignored)\n" \
	        "  -iconic                Start iconified\n" \
	        "  -name NAME             Program name as used by the window manager\n" \
	        "  -title TITLE           Set window title to TITLE\n\n"));
	g_print(_("%s options:\n" \
	        "  -borderless            Try to open window without border decoration\n" \
	        "  -nofocus               Don't focus the window when it opens\n" \
	        "  -encoding CHARSET      Expect CHARSET as the message encoding\n" \
	        "  -entry                 Prompt for text to be sent to stdout\n" \
	        "  -entrytext TEXT        Same as -entry, but with TEXT as default text\n" \
	        "  -wrap                  Wrap lines of text to fit window width\n" \
	        "  -help | -?             Show this usage information\n" \
	        "  -version               Show gxmessage version information\n\n"), PACKAGE);

	prog_cleanup ();
}



void
prog_cleanup (void)
{
	button_free (gx.button_list);
	if (gx.message_text != NULL) {
		g_free (gx.message_text);
	}
	if (gx.timeout_id != 0) {
		gtk_timeout_remove (gx.timeout_id);
	}

	exit (gx.exit_code);
}



int
main (gint argc, gchar *argv[])
{
	GString *gstr = NULL;
	gchar *ch = NULL, *tmpstr;
	const gchar *fname = NULL;
	gint opt, arg = 1;
	gchar bu_default[] = "okay:0";

	/* The default "okay:0" string is intentionally hard-wired, to avoid
	 * breaking scripts that make use of xmessage's -print option.
	 * It must not be changed or gettextize'd. */


#ifdef ENABLE_NLS
	bindtextdomain (PACKAGE, PACKAGE_LOCALE_DIR);
	bind_textdomain_codeset (PACKAGE, "UTF-8");
	textdomain (PACKAGE);
#endif

	option_kludge (&argc, &argv);
	gtk_init (&argc, &argv);

	gx.exit_code       = 1;
	gx.do_focus        = TRUE;
	gx.wrap_mode       = GTK_WRAP_NONE;
	gx.window_position = GTK_WIN_POS_NONE;


	while (arg < argc) {
		opt = my_get_opt (argv[arg], arg + 1 < argc);
		switch (opt) {
		case OPT_HELP :
		case OPT_HELP_Q :
			gx.exit_code = 0;
			usage ();
			break;
		case OPT_VERSION :
			g_print (PACKAGE "-" VERSION "\n");
			exit (0);
			break;
		case OPT_ENTRY :
		case OPT_ENTRYTEXT :
			if (gx.do_print) {
				g_printerr (_("%s: can't have both -entry and -print\n"), PACKAGE);
				prog_cleanup ();
			}
			if (gx.timeout) {
				g_printerr (_("%s: can't have both -entry and -timeout\n"), PACKAGE);
				gx.timeout = 0;
			}
			if (opt == OPT_ENTRY) {
				gx.entry_str = "";
			}
			else {
				gx.entry_str = argv[++arg];
			}
			break;
		case OPT_BUTTONS :
			button_free (gx.button_list);
			gx.button_list = button_list_from_str (argv[++arg]);
			break;
		case OPT_CENTER :
			gx.window_position = GTK_WIN_POS_CENTER;
			break;
		case OPT_DEFAULT :
			gx.default_str = argv[++arg];
			break;
		case OPT_FILE :
			if (gstr != NULL) {
				g_printerr (_("%s: can't get message from both -file and command line\n"), PACKAGE);
				prog_cleanup ();
			}
			arg++;
			fname = argv[arg];
			break;
		case OPT_NEARMOUSE :
			/* -center takes priority over -nearmouse */
			if (gx.window_position != GTK_WIN_POS_CENTER) {
				gx.window_position = GTK_WIN_POS_MOUSE;
			}
			break;
		case OPT_PRINT :
			if (gx.entry_str != NULL) {
				g_printerr (_("%s: can't have both -entry and -print\n"), PACKAGE);
				prog_cleanup ();
			}
			gx.do_print = TRUE;
			break;
		case OPT_TIMEOUT :
			gx.timeout = strtol (argv[++arg], &ch, 10);
			if (*ch) {
				g_printerr (_("%s: integer -timeout value expected\n"), PACKAGE);
			}
			if (gx.timeout < 0) {
				/* xmessage doesn't complain here, just disables timeout */
				gx.timeout = 0;
			}
			if (gx.entry_str != NULL) {
				g_printerr (_("%s: can't have both -entry and -timeout\n"), PACKAGE);
				gx.timeout = 0;
			}
			break;
		case OPT_TITLE :
			gx.title_str = argv[++arg];
			break;
		case OPT_GEOMETRY :
			gx.geom_str = argv[++arg];
			break;
		case OPT_FN :
		case OPT_FONT :
			gx.font_str = argv[++arg];
			break;
		case OPT_RV :
		case OPT_REVERSE :
		case OPT_SYNCHRONOUS :
			/* not implemented - ignore */
			break;
		case OPT_BG :
			gx.color_bg = argv[++arg];
			break;
		case OPT_FG :
			gx.color_fg = argv[++arg];
			break;
		case OPT_BD :
		case OPT_BW :
		case OPT_XRM :
		case OPT_SELECTIONTIMEOUT :
		case OPT_XNLLANGUAGE :
			/* not implemented - ignore and skip arg */
			arg++;
			break;
		case OPT_ICONIC :
			gx.do_iconify = TRUE;
			break;
		case OPT_BORDERLESS :
			gx.do_borderless = TRUE;
			break;
		case OPT_WRAP :
			gx.wrap_mode = GTK_WRAP_WORD;
			break;
		case OPT_ENCODING :
			gx.encoding = argv[++arg];
			break;
		case OPT_FOCUS:
			gx.do_focus = FALSE;
			break;
		case OPT_IS_MISSING_ARG :
			/* in this case, xmessage treats the "option" as normal text */
		case OPT_IS_UNKNOWN :
		default:
			if (fname != NULL) {
				g_printerr (_("%s: can't get message from both -file and command line\n"), PACKAGE);
				prog_cleanup ();
			}
			if (gstr == NULL) {
				gstr = g_string_new ("");
			}
			else {
				gstr = g_string_append_c (gstr, ' ');
			}
			gstr = g_string_append (gstr, argv[arg]);
			break;
		}
		arg++;
	}

	if (fname != NULL) {
#if _GTK_2_6
		if (strcmp ("gxmsg:stdin", fname) == 0) {
#else
		if (strcmp ("-", fname) == 0) {
#endif
			tmpstr = read_stdin ();
		}
		else if (!g_file_get_contents (fname, &tmpstr, NULL, NULL)) {
			g_printerr (_("%s: unable to read file\n"), PACKAGE);
			prog_cleanup ();
		}
		gx.message_text = message_to_utf8 (tmpstr);
		gx.message_length = strlen (gx.message_text);
		g_free (tmpstr);
	}
	else if (gstr != NULL) {
		gx.message_text = message_to_utf8 (gstr->str);
		gx.message_length = strlen (gx.message_text);
		g_string_free (gstr, TRUE);
	}
	else {
		g_printerr (_("%s: message text is required\n"), PACKAGE);
		g_printerr (_("Try `%s --help' for more information\n"), PACKAGE);
		prog_cleanup ();
	}

	if (gx.button_list == NULL) {
		gx.button_list = button_append (NULL,
		    button_new_from_str (bu_default, 0, strlen (bu_default)));
	}


	button_set_default (gx.button_list, gx.default_str);


	window_create ();
	gtk_main ();


	prog_cleanup ();
	return 0;
}

