/*
 * Gmail. A Gnome email client.
 * Copyright (C) 1999 Wayne Schuller
 *
 * outgoing.c - stuff for composing new emails and sending them out. 
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 */

#include "main.h"
#include "smtp.h"


extern gboolean	DestroyRightPane(void);
extern gboolean add_msg_to_db(Message *message, DTimer *timer);
extern ssize_t recvline(int sock, void *vptr, size_t maxlen);
extern gint progress_timeout(DTimer *timerinfo);
extern void destroy_dtimeout(GtkWidget *widget, DTimer *timer);
extern void mark_as_matched(gint id, GString *vfolders, gboolean overwrite); 
extern 	Message * load_vfolder_msg(gint id);
extern void broken_cb (GtkWidget *widget, void *data);
extern void about_cb (GtkWidget *widget, void *data);
extern void free_message(Message *message);
extern void  entry_drag_data_received(GtkWidget *widget, GdkDragContext *context, gint x, gint y, GtkSelectionData *data, guint info, guint time);
extern  gboolean check_db(MYSQL *mysql);
extern void mark_readstatus(gint msg_id, gchar *val);
extern void vfolder_display_rightpane_by_row(gint row);
extern void vfolder_display_rightpane_by_mailbox(Mailbox *mailbox);
extern MailSetup *load_default_msetup(void);
extern void free_msetup(MailSetup *msetup);
extern void check_vfolder_matches(gint id, DTimer *timer);

extern _GmailApp_	*GmailApp;

void gmail_wrap_string(gchar *str, int width);
void  compose_cb (GtkWidget *widget, gchar *type);
void  send_cb (GtkWidget *widget, gboolean queue);
void	cancel_send_cb (GtkWidget *widget, gpointer *data);
void	IndentMsg (GString *msg);
gchar * strip_email(gchar *to);
gchar * strip_name(gchar *to);
gboolean gmime_smtp_send(GMimeMessage *msg);
void send_smtp_rcpt(GList *rcptlist, gint sock, GtkWidget *label);
void addressbook_cb (GtkWidget *widget, gpointer *data);
void queue_cb (GtkWidget *widget, gpointer *data);
void send_queued_cb (GtkWidget *widget, gpointer *data);
void add_file_attach (GtkWidget *widget, gpointer *data);
void remove_file_attach (GtkWidget *widget, gpointer *data);
void file_attach_selected(GtkWidget *widget, GtkFileSelection *fs);
void add_attachments(GMimePart *multi_part, GtkWidget *clist);
GString *insert_reply_to_all(GMimeMessage *message, gchar *replyto, gchar *from);
void attachment_drag_data_received (GtkWidget *widget, GdkDragContext *context, gint x, gint y, GtkSelectionData *data, guint info, guint time);
void add_attachment_to_clist(GtkWidget *clist, gchar *fullpath);

#define set_status_label(label, text) gtk_label_set_text((GtkLabel*)label, text); while (g_main_iteration(FALSE));


/* Standard DnD types */
enum
  {
    TARGET_EMAIL,
  };

GtkTargetEntry email_field_drop_types[] =
{
  {"x-application/x-email", 0, TARGET_EMAIL }
};

GtkTargetEntry attachment_drop_types[] =
{
  {"text/uri-list", 0, TARGET_EMAIL }
};

#define ELEMENTS(x) (sizeof (x) / sizeof (x[0]))

/* 
 * Routine to compose mail.
 * Is called by the compose/reply/reply to all/forward buttons.
 * type can be: "new", "reply", "replytoall", "forward".
 *
 */
void
compose_cb (GtkWidget *widget, gchar *type)
{
	GtkWidget *window, *to, *from, *subject, *cc, *bcc, *text, *clist, *wid;
	gchar *str, *real_name;
	Message *msg = NULL;
	FILE	*sig;
	gchar *glade_src;
	Mailbox	*mailbox = NULL;
	GladeXML *xml;
	GString *body;
	gint id;
	GString *fulltext;
	GMimeMessage *message;
	gchar *replyto = NULL, *frm = NULL;

	/* Get the xml tree from the msg widget. */
	xml = glade_get_widget_tree(GTK_WIDGET (GmailApp->msgpane));
	text = glade_xml_get_widget(xml, "textmsg");
	body = g_string_new(gtk_editable_get_chars(GTK_EDITABLE (text), 0, -1));

	id = (gint) gtk_object_get_data(GTK_OBJECT(GmailApp->msgpane), "id");

	/* Load the glade file. */
	glade_src = g_strdup_printf("%s/compose.glade", GMAIL_GLADEDIR);
	xml = glade_xml_new(glade_src, "compose");
	g_free(glade_src);

	if (!xml) {
		g_warning("Unable to load xml interface. Are the gmail glade files installed onto your system?\n");
		exit(1);
		}

	glade_xml_signal_autoconnect(xml);

	/* Grab our glade widgets for the compose window. */
	window = glade_xml_get_widget(xml, "compose");
	to = glade_xml_get_widget(xml, "to");
	from = glade_xml_get_widget(xml, "from");
	subject = glade_xml_get_widget(xml, "subject");
	cc = glade_xml_get_widget(xml, "cc");
	bcc = glade_xml_get_widget(xml, "bcc");
	text = glade_xml_get_widget(xml, "textmsg");
	clist = glade_xml_get_widget(xml, "attach_clist");

	gtk_widget_set_usize(GTK_WIDGET(window), 650, 500);	

	gtk_text_set_editable (GTK_TEXT (text), TRUE);
	gtk_text_set_word_wrap (GTK_TEXT (text), TRUE);
	gtk_widget_set_style (text, GmailApp->style);

	/* Set up drag and drop targets. */
	gtk_drag_dest_set (to, GTK_DEST_DEFAULT_ALL,
		     email_field_drop_types, ELEMENTS (email_field_drop_types),
		     GDK_ACTION_COPY); 
	gtk_signal_connect( GTK_OBJECT(to), "drag_data_received",
		      GTK_SIGNAL_FUNC( entry_drag_data_received), NULL); 
	gtk_drag_dest_set (cc, GTK_DEST_DEFAULT_ALL,
		     email_field_drop_types, ELEMENTS (email_field_drop_types),
		     GDK_ACTION_COPY); 
	gtk_signal_connect( GTK_OBJECT(cc), "drag_data_received",
		      GTK_SIGNAL_FUNC( entry_drag_data_received), NULL); 
	gtk_drag_dest_set (bcc, GTK_DEST_DEFAULT_ALL,
		     email_field_drop_types, ELEMENTS (email_field_drop_types),
		     GDK_ACTION_COPY); 
	gtk_signal_connect( GTK_OBJECT(bcc), "drag_data_received",
		      GTK_SIGNAL_FUNC( entry_drag_data_received), NULL); 

	/* Allow drag'n'drop into the attach clist, from either gmc
	 * or Nautilus.
	 */
	gtk_signal_connect( GTK_OBJECT(clist), "drag_data_received",
		      GTK_SIGNAL_FUNC( attachment_drag_data_received), NULL); 
	gtk_drag_dest_set (clist, GTK_DEST_DEFAULT_ALL,
		     attachment_drop_types, ELEMENTS (attachment_drop_types),
		     GDK_ACTION_COPY); 

	/* Put in a default email into the From field.
	 * Check for mailbox default, then global default.
	 */

	mailbox = gtk_ctree_node_get_row_data(GTK_CTREE (GmailApp->mblist), GTK_CTREE_NODE (GmailApp->node));

	if (mailbox != NULL && mailbox->email != NULL && strlen(mailbox->email) > 3 ) {
		frm = g_strdup(mailbox->email);
		} else {
		frm = gnome_config_get_string("/gmail/UserInfo/EmailAddress"); 
		if (frm == NULL) frm = g_strdup("no default email address set.");
		}
	/* Put the users name in front of the email address, like:
	 * Wayne Schuller <k_wayne@linuxpower.org">
	 */
	real_name = gnome_config_get_string("/gmail/UserInfo/RealName");
	str = g_strdup_printf("%s <%s>", real_name, frm);
	wid = glade_xml_get_widget(xml, "from_gnome_entry");
	gnome_entry_load_history(GNOME_ENTRY(wid));
	gnome_entry_append_history(GNOME_ENTRY (wid), TRUE, str);
	gtk_entry_set_text(GTK_ENTRY (from), str);	
	g_free(str);
	g_free(real_name);

	/* Load the details of the message we might be replying to. */	
	msg = load_vfolder_msg(id);

	/* Do extra stuff for reply/replytoall/forward buttons.
	 * FIXME: the below code should be re-written logically.
	 */
	if (!strcmp(type, "new") == 0 && msg != NULL) {

	/* Form the GMimeMessage. */
	fulltext = g_string_new(msg->headers->str);
	g_string_append(fulltext, msg->message->str);

	message = g_mime_parser_construct_message(fulltext->str, fulltext->len, TRUE);
	g_string_free(fulltext, TRUE);

	if (message) replyto = (gchar *) g_mime_message_get_reply_to(message);

	if (replyto == NULL) replyto = msg->from;

		/* Is this a reply or reply-to-all? */
		if (!strcmp(type, "forward")==0) {
			if (replyto)
				gtk_entry_set_text(GTK_ENTRY (to), replyto);	

			if (strcmp(type, "replytoall")==0) {
				GString *rta;
				rta = insert_reply_to_all(message, replyto, frm);
				gtk_entry_append_text(GTK_ENTRY (cc), rta->str);	
				g_string_free(rta, TRUE);
				}
		}

		/* If we have a subject header insert it. */
		if (msg->subject) {
			if (strncasecmp(msg->subject, "Re:", 3) != 0)
				gtk_entry_set_text(GTK_ENTRY (subject), "Re: ");	
			gtk_entry_append_text(GTK_ENTRY (subject), msg->subject);	
			} 

		IndentMsg(body);	 /* Indent the old message */

		gtk_text_insert (GTK_TEXT (text), NULL, NULL, NULL, body->str, -1);

		free_message(msg); /* Free the memory */
	}

	str = gnome_config_get_string("/gmail/UserInfo/Signature");

	if (str != NULL && strlen(str) > 0) {
		sig = fopen(str, "r");
		if (sig == NULL)
			g_print("Cant open signature file: %s\n", str);
		else {
			/* Lets insert signature file now */
			gchar *txt;
			gchar line[80];

			/* Insert two newlines. */
			gtk_text_insert (GTK_TEXT (text), NULL, NULL, NULL, "\n\n--\n", -1);
			do {
				txt = fgets(line, 80, sig);
				gtk_text_insert (GTK_TEXT (text), NULL, NULL, NULL, txt, -1);
			} while(txt != NULL);
		fclose(sig);
		}
	}
	g_free(str);

	/* Go to the top of the message. */
	gtk_text_set_point(GTK_TEXT (text), 0);

	gtk_widget_show_all(window);

	g_string_free(body, TRUE);
	g_free(frm);

	/* Let them begin typing the to field. */
	gtk_widget_grab_focus(GTK_WIDGET(to));

	return;
}

/* 
 * Callback for when they click on the 'send' button in the compose window. 
 * Will parse the compose window, and send the message.
 *
 * Unfortunately, we create the message in two different data structures.
 * We set it up in a GMimeMessage format used for the SMTP send.
 * We also set it up in a gmail Message format for adding to the database.
 * 
 * We use the GMime one as it handles parsing the recipients nicely.
 */
void
send_cb (GtkWidget *widget, gboolean queue)
{
	Message	*msg;
	GtkWidget *dialog; 
	gboolean result = TRUE; /* Did we successfully send? */ 
	GtkWidget *window, *to, *from, *sub, *cc, *bcc, *text, *clist;
	GtkWidget *gentry;
	GladeXML *xml = glade_get_widget_tree(GTK_WIDGET (widget));
	gchar *hdrs, *body, *ptr;
	gchar *tmp;
	GMimePart *multi_part, *text_part;	
	GMimeContentType *mime_type;
	GMimeMessage *message;

	/* Get a pointer to the window the button was in. */
	window = glade_xml_get_widget(xml, "compose");
	to = glade_xml_get_widget(xml, "to");
	from = glade_xml_get_widget(xml, "from");
	sub = glade_xml_get_widget(xml, "subject");
	cc = glade_xml_get_widget(xml, "cc");
	bcc = glade_xml_get_widget(xml, "bcc");
	text = glade_xml_get_widget(xml, "textmsg");
	clist = glade_xml_get_widget(xml, "attach_clist");
	gentry = glade_xml_get_widget(xml, "from_gnome_entry");

	/* Create the message to be sent. 
	 * We want all the data in our own memory space, so we g_strdup the info
	 * from the gtk_entry widgets.
	 */
	msg = g_malloc(sizeof(Message));
	msg->headers = g_string_new(NULL); /* We build headers later. */
	msg->message = g_string_new(NULL); /* We build the body later. */
	msg->date = NULL;
	msg->subject = g_strdup(gtk_entry_get_text(GTK_ENTRY(sub)));
	msg->from = g_strdup(gtk_entry_get_text(GTK_ENTRY(from)));
	msg->to = g_strdup(gtk_entry_get_text(GTK_ENTRY(to)));
	msg->cc = g_strdup(gtk_entry_get_text(GTK_ENTRY(cc)));
	msg->bcc = g_strdup(gtk_entry_get_text(GTK_ENTRY(bcc)));
	if (queue)
		msg->readstatus = g_strdup("Queued");
	else
		msg->readstatus = g_strdup("Sent");
	msg->direction = g_strdup("Outgoing");
	msg->uid = g_strdup("Outgoing messages have no UID.");

	/* g_print("To: %s\nFrom: %s\nSubject: %s\nCc: %s\n\n%s\n", to, from, subject, cc, body); */

	/* Lets do some primitive error checking. */

	/* Missing "To:" field. */
	if (strlen(msg->to) < 2) {  
		dialog = gnome_message_box_new("please fill out the \"To:\" field", GNOME_MESSAGE_BOX_ERROR, "ok", 0);
    	gtk_window_set_position(GTK_WINDOW(dialog), GTK_WIN_POS_CENTER);
			gtk_window_set_modal(GTK_WINDOW (dialog), TRUE);
    	gtk_widget_show(dialog);
			return;
		}

	/* Missing "Subject:" field. */
	if (strlen(msg->subject) < 2) {
		dialog = gnome_message_box_new("subject field missing\n(don't you realise how incredibly annoying emails without subjects are?)", GNOME_MESSAGE_BOX_WARNING, "ok", 0);
    	gtk_window_set_position(GTK_WINDOW(dialog), GTK_WIN_POS_CENTER);
			gtk_window_set_modal(GTK_WINDOW (dialog), TRUE);
    	gtk_widget_show(dialog);
	    return;
		}

	/* Currently we always create a MIME message, even if there is
	 * no attachment. Is this ok?
	 * What are the I18N issues?
	 */

	g_print("Creating MIME message...\n");

	/* create a gmime message structure. */
	message = g_mime_message_new ();

	g_assert(message != NULL); /* FATAL. */

	g_mime_message_set_sender (message, msg->from);
	g_mime_message_set_reply_to (message, msg->from);
	g_mime_message_add_recipients_from_string (message, GMIME_RECIPIENT_TYPE_TO, msg->to);
	g_mime_message_add_recipients_from_string (message, GMIME_RECIPIENT_TYPE_CC, msg->cc);
	g_mime_message_add_recipients_from_string (message, GMIME_RECIPIENT_TYPE_BCC, msg->bcc);
	g_mime_message_set_subject (message, msg->subject);
	tmp = g_strdup_printf("Gmail %s (http://gmail.linuxpower.org)", VERSION);
	g_mime_message_set_header (message, "X-Mailer", (const gchar *) tmp);
	g_free(tmp);

	/* Create the first MIME part: the message body. */
	multi_part  = g_mime_part_new_with_type ("multipart", "mixed");

	/* NOTE: Gmime will set the boundary for us. */

	text_part = g_mime_part_new ();
	mime_type = g_mime_content_type_new ("text", "plain");
	g_mime_part_set_content_type (text_part, mime_type);
	g_mime_part_set_content_id (text_part, "1");

	tmp = gtk_editable_get_chars(GTK_EDITABLE (text), 0, -1);

	/* Wrap the body text to a set line length... */
	if (gnome_config_get_bool("/gmail/UserInfo/wrap_text")) 
		gmail_wrap_string(tmp, gnome_config_get_int("/gmail/UserInfo/wrap_len"));

	g_mime_part_set_content (text_part, tmp, (guint) strlen(tmp));

	/* FIXME: next line crashes. how do we use this? 
	g_mime_part_set_content_disposition(text_part, "body"); */

	g_mime_part_add_child (multi_part, text_part);

	/* Check for any attachments in the attachment clist */
	if (GTK_CLIST(clist)->rows > 0) add_attachments(multi_part, clist); 
		

	g_mime_message_set_mime_part (message, multi_part);

	/* Create a Message structure here for sending.
	 * We need it to insert into the database.
 	 */

	/* Let gmime create the rfc822 message. */
	hdrs = g_mime_message_to_string(message);

	/* Split it into headers and message. \n\n is in the middle. */
	/* FIXME: what if it is \r\n\r\n in the middle? */
	ptr = strstr(hdrs, "\n\n");
	body = g_strdup(ptr);
	*ptr = '\0';
	g_string_append(msg->headers, hdrs);
	/* g_string_append(msg->headers, "\n\n");  Add newline? */
	g_string_append(msg->message, body);
	g_free(hdrs);
	g_free(body);

	/* The following code is #ifdef'ed out until we can fix gmime to
	 * include the MIME headers in g_mime_message_get_headers.
	 */
#if FALSE
	/* Create the message headers.  */
	g_string_append(msg->headers, g_mime_message_get_headers(message));

	/* Create the message body. */
	body = g_mime_message_to_string(message);
	ptr = body + msg->headers->len; /* Move to end of headers. */
	g_string_append(msg->message, ptr);
	g_free(body);
#endif

	if (!queue) result = gmime_smtp_send(message);

	/* We can now destroy the message. */
	g_mime_message_destroy(message);

	/* If the smtp failed we queue the message. */
	if (!result) {
		msg->readstatus = g_strdup("Queued");
		queue = TRUE;
	}

	/* Add the bcc field to the message.
	 * We do this so the user can see who they sent the msg to.
	 * We don't do it earlier as then it would become part of the msg
	 * and would defeat the point of being blind carbon copy.
	 */	
	if (msg->bcc && strlen(msg->bcc) > 0) g_string_sprintfa(msg->headers, "Bcc: %s\r\n", msg->bcc); 
	g_string_sprintfa(msg->headers, "\r\n");

	/* Add the msg to the database. */
	add_msg_to_db(msg, NULL);

	/* Update the matched index for this message. 
	 * This is useful if the user has a 'queued' vfolder.
	 */
	check_vfolder_matches(msg->id, NULL);

	/* Save the gnome entry data. */
	gnome_entry_save_history (GNOME_ENTRY(gentry));

	/* We can close the compose window now. */
	if (window != NULL) gtk_widget_destroy(GTK_WIDGET (window));

	/* Free the xml reference. */
	gtk_object_unref(GTK_OBJECT(xml));

	/* BELOW CODE COMMENTED OUT. Why do we need this?
	 * It makes the appbar message for 'message sent' get overwritten.
	 */
	/* Update the current vfolder, it's contents may be changed. 
	Mailbox *mailbox = gtk_ctree_node_get_row_data(GTK_CTREE (GmailApp->mblist), GTK_CTREE_NODE (GmailApp->node));
	vfolder_display_rightpane_by_mailbox(mailbox); */

	/* Free the memory from above. */
	free_message(msg); /* Free the memory */
}


/* Callback for when they click on the 'cancel' button in a compose window.
 * Need to close the relevant compose window.
 * 
 * FIXME: We need to check and remind them if they have a half composed
 * message open. 
 */
void
cancel_send_cb (GtkWidget *widget, gpointer *data)
{
	GtkWidget *window, *gentry;
	GladeXML *xml = glade_get_widget_tree(GTK_WIDGET (widget));

	/* Get a pointer to the window the button was in. */
	window = glade_xml_get_widget(xml, "compose");

	gentry = glade_xml_get_widget(xml, "from_gnome_entry");

	/* Save the gnome entry data. */
	gnome_entry_save_history (GNOME_ENTRY(gentry));

	if (window != NULL) gtk_widget_destroy(GTK_WIDGET (window));

	/* Free the xml reference. */
	gtk_object_unref(GTK_OBJECT(xml));
}

/* This routine is called to put indents at the start of each line of the
 * message in the static cache. 
 * FIXME: Should we add other stuff? like "thingy wrote:\n" first?
 */
void
IndentMsg (GString *msg) {
	gchar *tok;  /* tokenizer we use to find newlines. */
	gint diff=0; /* the number of chars until just after a newline. */

	while (diff < msg->len) {

		/* Insert the indent after the newline. */
		/* FIXME: Use a configurable indent from prefs. */
		g_string_insert(msg, diff, "> ");

		/* Move the token just past the newline. */
		tok = msg->str + diff;

		/* Work out the distance to the next newline + 1 */
		diff += strcspn(tok, "\n")+1; 
	}
	g_string_insert(msg, 0, "\n");
}

/* gmime_smtp_send - send a GMime message via smtp protocol 
 * 
 * This is very basic. Thanks to Spruce for the reference. It's improved a
 * a bit since then.
 *
 * FIXME/TODO:
 * 
 * - Make the progress bar work according to the contents of the message
 * being uploaded.
 * - Fix awful hack checking for the existance of the timer constantly.
 * - There is repeated code here from msg_match_popup.
 *
 */
gboolean
gmime_smtp_send(GMimeMessage *msg)
{
 	struct hostent *ptr_host;
	gchar	*hostname;
	struct sockaddr_in sin;
	port_t 	port = 25; /* Standard smtp port. FIXME: make configurable */
	gchar ip[17]; /* IPv4 dotted decimal and NULL */
	int sock;
	gchar buffer[256];
	gchar command[256];
	gchar localhost[256];
	gchar *body;
	const gchar *from;
	GtkWidget *vbox, *child, *label;
	gchar fejj[6]; /* Named in honour of Spruce author fejj. */
	DTimer *timer;
	gint i, size;
	MailSetup *msetup;
	gchar *address;

	timer = g_malloc(sizeof(DTimer));

	timer->clist = NULL; /* We're not using the clist. */
	timer->value = 0;	
	timer->total = 1;

	timer->dialog = gnome_dialog_new (
                 "Gmail Popup",
                 NULL);
	if (GmailApp->app)
             gnome_dialog_set_parent (GNOME_DIALOG (timer->dialog), GTK_WINDOW (GmailApp->app));

	gtk_widget_set_usize(GTK_WIDGET(timer->dialog), 200, 150);
	gtk_window_set_modal(GTK_WINDOW (timer->dialog), TRUE);

	gtk_signal_connect_after(GTK_OBJECT (timer->dialog), "destroy", GTK_SIGNAL_FUNC (destroy_dtimeout), timer);

	vbox = GNOME_DIALOG(timer->dialog)->vbox;

	label = gtk_label_new("Looking up smtp host...");
	gtk_box_pack_start (GTK_BOX (vbox), label, FALSE, FALSE, 5);
	gtk_widget_show(label);


	timer->progress = GTK_PROGRESS_BAR(gtk_progress_bar_new());

	timer->timeout = gtk_timeout_add (10, (GtkFunction) progress_timeout, timer);

	gtk_progress_set_show_text (GTK_PROGRESS (timer->progress), TRUE);
	gtk_box_pack_start (GTK_BOX (vbox), GTK_WIDGET(timer->progress), FALSE, FALSE, 5);
	gtk_widget_show (GTK_WIDGET(timer->progress));

	timer->button = gtk_button_new_with_label(_("Abort"));
	gtk_box_pack_start (GTK_BOX (vbox), timer->button, FALSE, FALSE, 5);
	gtk_signal_connect_after (GTK_OBJECT (timer->button), "clicked",
                                   GTK_SIGNAL_FUNC (destroy_dtimeout),
                                   timer);
	gtk_widget_show(timer->button);

 	gtk_widget_show_all(timer->dialog);
	while (g_main_iteration(FALSE));

	/* Get SMTP server for current default setup. */
	msetup = load_default_msetup();
	if (msetup != NULL) 
		hostname = g_strdup(msetup->smtpserver);
		else {
		g_print("No default mail setup configured!\n");
		hostname = NULL; /* This forces the error. */
		}
	free_msetup(msetup);
	if (hostname == NULL) {
  	gtk_label_set_text((GtkLabel *) label, _("No smtp host in config,\n or no default mail setup selected."));
	return(FALSE);
		}
   	if ( (ptr_host = gethostbyname(hostname)) == NULL) {
  	gtk_label_set_text((GtkLabel *) label, "Bad SMTP hostname");
	return(FALSE);
		}

   	memcpy(&sin.sin_addr, ptr_host->h_addr, ptr_host->h_length);
	g_snprintf(ip, sizeof(ip), "%s", inet_ntoa(sin.sin_addr));

	sock = socket(AF_INET, SOCK_STREAM, IPPROTO_TCP);
	if (sock == -1) {
		set_status_label(label, "SMTP socket failure");
		return(FALSE);
	}

	sin.sin_family = AF_INET;
   	sin.sin_port = htons(port);

	set_status_label(label, "Establishing connection."); 
	if (connect(sock, (struct sockaddr *) &sin, sizeof(sin)) < 0) {
		set_status_label(label, "no connection");
		return(FALSE);
	}

	g_return_val_if_fail(GTK_IS_WIDGET(timer->dialog), FALSE);

	if ( (recvline(sock, buffer, 255) < 0) || (strstr(buffer, ServiceRdy) == NULL) ) {
		set_status_label(label, "bad smtp server");
		close(sock);
      		return(FALSE);
   	}

	timer->value = 0.1;

	set_status_label(label, "Sending HELO"); 
	/* Send HELO */
	gethostname(localhost, 255);
	g_snprintf(command, 255, "HELO %s\r\n", localhost);
   	if (send (sock, command, strlen(command), 0) < 0) {
		set_status_label(label, "untalkative smtp");
      		return(FALSE);
   	}

	/* Check HELO reply. */		
	if ( (recvline(sock, buffer, 255) < 0) 
			|| (strstr(buffer, ReqMlActOK) == NULL) ) {
		g_warning("unfriendly smtp");
		set_status_label(label, "unfriendly smtp");
      		return(FALSE);
   	}

	g_return_val_if_fail(GTK_IS_WIDGET(timer->dialog), FALSE);
	timer->value = 0.2;

	/* set_status("Sending MAIL FROM"); */
	/* Send MAIL FROM (make gmime parse the address out) */
	from = g_mime_message_get_sender(msg);
	address = strip_email((gchar *) from);

	g_snprintf(command, 255, "MAIL FROM: %s\r\n", address);
   	if (send (sock, command, strlen(command), 0) < 0) {
		set_status_label(label, "smtp disliked MAIL FROM");
      		return(FALSE);
   	}

	/* Check MAIL FROM reply. */
	if ( (recvline(sock, buffer, 255) < 0) 
			|| (strstr(buffer, ReqMlActOK) == NULL) ) {
		gchar *str;
		str = g_strdup_printf("didn't like our mail from: %s", address);
		set_status_label(label, str);
		g_strdup(str);
      		return(FALSE);
   	}

	g_free(address);

	g_return_val_if_fail(GTK_IS_WIDGET(timer->dialog), FALSE);
	timer->value = 0.3;

	/* Send RCPT's to the addresses in the email. */
	send_smtp_rcpt(g_mime_message_get_recipients(msg, GMIME_RECIPIENT_TYPE_TO), sock, label);
	send_smtp_rcpt(g_mime_message_get_recipients(msg, GMIME_RECIPIENT_TYPE_CC), sock, label);
	send_smtp_rcpt(g_mime_message_get_recipients(msg, GMIME_RECIPIENT_TYPE_BCC), sock, label);

	g_return_val_if_fail(GTK_IS_WIDGET(timer->dialog), FALSE);
	timer->value = 0.4;

	/* set_status("Sending DATA"); */
	/* Send DATA command */
	g_snprintf(command, 255, "DATA\r\n");
   	if (send (sock, command, strlen(command), 0) < 0) {
		g_warning("didn't like our DATA\nMessage: %s", buffer);
		set_status_label(label, "smtp disliked DATA");
		return(FALSE);
   	}

	/* Check reply. */
	if ( (recvline(sock, buffer, 255) < 0) || (strstr(buffer, StrtMailInpt) == NULL) ) {
		gchar *txt;
		txt = g_strdup_printf("SMTP DATA error:\n%s", buffer);
		set_status_label(label, txt);
		g_free(txt);
		return(FALSE);
   	}

	g_return_val_if_fail(GTK_IS_WIDGET(timer->dialog), FALSE);

	/* FIXME: insert date here? */
	body = g_mime_message_to_string(msg);

	/* 
	 * To follow the RFC: make sure \n's become \r\n's
	 * Some SMTP servers are picky about this (eg: QMAIL)
	 * 
	 * This code basically sends the data one char at a time, because
	 * we need to be looking to make sure every \n has a \r before it.
	 *
	 * Code originally taken from Spruce. Variable named after fejj!
	 */
	timer->total = strlen(body);
	size = strlen(body);
	for (i = 0; i < size; i++) {
		if (body[i] == '\n' && body[i-1] != '\r') {
			strcpy(fejj, "\r\n");
			send(sock, fejj, 2, 0); /* Send \r\n */
		} else {
			fejj[0] = body[i];
			fejj[1] = '\0';
			send(sock, fejj, 1, 0); /* Just send the char */
		}
		timer->value = i;
	}

	/* Send the \r\n. to end the msg.  */
	send (sock, "\r\n.\r\n", 5, 0);

	if ( (recvline(sock, buffer, 255) < 0)
       		|| (strstr(buffer, ReqMlActOK) == NULL) ) {
		g_warning("didn't like our DATA\nMessage: %s", buffer);
		set_status_label(label, "smtp rejected DATA");
      		return(FALSE);
		}

	g_return_val_if_fail(GTK_IS_WIDGET(timer->dialog), FALSE);
	timer->value = 0.8;

	set_status_label(label, "Sending QUIT"); 
	/* Send QUIT */
	g_snprintf(command, 255, "QUIT\r\n");
   	if (send (sock, command, strlen(command), 0) < 0) {
		g_warning("smtp didn't like QUIT");
		set_status_label(label, "smtp disliked QUIT");
      	return(FALSE);
   	}

	/* Check reply. */
	if ( (recvline(sock, buffer, 255) < 0) 
			|| (strstr(buffer, ServClose) == NULL) ) {
		gchar *txt;
		txt = g_strdup_printf("SMTP QUIT error:\n%s", buffer);
		set_status_label(label, txt);
		g_free(txt);
      		return(FALSE);
   	}


	timer->value = 1;

	/* Send positive feedback to the user. */
	set_status_label(label, "Message sent!"); 

	/* Update the button text. */
	child = GTK_BIN(timer->button)->child;
	if (GTK_IS_LABEL(child))
		gtk_label_set(GTK_LABEL(child), "Close");

	while (g_main_iteration(FALSE));

	destroy_dtimeout(NULL, timer); /* Closes the dialog. */

	g_free(hostname);

	gnome_appbar_set_status (GNOME_APPBAR (GmailApp->appbar), "Message Sent!");
	while (g_main_iteration(FALSE));

	return(TRUE);
} /* SMTP HACK */

/* Strip email.
 *
 * It's ok to have names in the tofield, but we should strip
 * out everything but the bare address for the RCPT.
 * "Barry Smith" <b.smith@hotmail.com>
 * Should become:
 * b.smith@hotmail.com
 * 
 * If we can't find the email address return NULL.
 * Luckily, the gmime library does all the work for us.
 */
gchar *
strip_email(gchar *to)
{
	InternetAddress *ia = NULL;
	GList *addr;
	gchar *email_address = NULL;

	addr = internet_address_parse_string(to);
	if (addr == NULL) return(NULL);

	ia = addr->data;

	if (ia == NULL) return(NULL);

	if (ia->value.addr) email_address = g_strdup(ia->value.addr);

	internet_address_destroy(ia);

	return(email_address);
}

/* Like strip_email, but returns the name, not the email addy.
 * Calling function should free the memory.
 * If we can't find anything return NULL.
 */
gchar *
strip_name(gchar *to)
{
	InternetAddress *ia = NULL;
	GList *addr;
	gchar *name = NULL;

	addr = internet_address_parse_string(to);
	if (addr == NULL) return(NULL);

	ia = addr->data;
	if (ia == NULL) return(NULL);
	if (ia->name == NULL) {
		internet_address_destroy(ia);
		return(NULL);
	}
	name = g_strdup(ia->name);
	internet_address_destroy(ia);

	return(name);
}


/* send_smtp_rcpt - Send a RCPT TO to everyone in the message.
 *
 * Just print any errors to standard output and continue. This is so
 * we don't halt the entire smtp send just because one recipient failed.
 * If there are no working recipients then the SMTP server will tell us later.
 *
 * FIXME: 
 * Do each one in a seperate dialog, and continue even if one of them fails,
 * then you can see which ones failed. ???
 */
void send_smtp_rcpt(GList *rcptlist, gint sock, GtkWidget *label)
{
	gchar command[256];
	gchar buffer[256];

	while (rcptlist) {
		InternetAddress *ia = rcptlist->data;
		gchar *adr = g_strdup(ia->value.addr); 
		gchar *msg;

		msg = g_strdup_printf("RCPT TO: %s", adr);

		g_print("Recipient: %s\n", msg);

		set_status_label(label, msg); 

		/* RCPT TO for each recipient. */
		g_snprintf(command, 255, "RCPT TO: %s\r\n", adr);
   		if (send (sock, command, strlen(command), 0) < 0) {
			g_print("SMTP didn't like: RCPT TO %s\n", adr);
			rcptlist = rcptlist->next; continue;
   		}

		/* Check reply. */
		if ( (recvline(sock, buffer, 255) < 0) 
				|| (strstr(buffer, ReqMlActOK) == NULL) ) {
			g_print("SMTP error on RCPT TO '%s', %s", adr, buffer);
			rcptlist = rcptlist->next; continue;
   		}

		g_free(adr);
		rcptlist = rcptlist->next;
	}

	return;
}

/* This function will launch the users selected address book program. */
void
addressbook_cb (GtkWidget *widget, gpointer *data)
{
	char *argv[1];
	gchar *program = gnome_config_get_string("/gmail/UserInfo/AddressBook=gnomecard"); 

	g_return_if_fail(program != NULL);

	argv[0] = program;
	gnome_execute_async (NULL, 1, argv);
	g_free(program);
}

/* Queue_cb - Queue a composed message for later sending. 
 * Just call send_cb with the TRUE option to tell it to just add to the
 * database and not try and send it immediately.
 */
void
queue_cb (GtkWidget *widget, gpointer *data) 
{
	send_cb(widget, TRUE);
}

/* send_queued_cb - Send all queued messages via SMTP. 
 *
 */
void
send_queued_cb (GtkWidget *widget, gpointer *data) 
{
	MYSQL_RES *res;
	MYSQL_ROW mysqlrow;
	Message *msg;
	gint id, affected_rows;
	gchar *query, *note;
	Mailbox *mailbox = gtk_ctree_node_get_row_data(GTK_CTREE (GmailApp->mblist), GTK_CTREE_NODE (GmailApp->node));

	if (!check_db(&GmailApp->mysql)) { return; };

	/* Get all the messages that are queued. */
	query = g_strdup_printf("SELECT id FROM display WHERE readstatus=\"Queued\"");

	/* Execute query.. */
	if (mysql_query(&GmailApp->mysql, query) != 0) {
		g_warning("Query failed: %s\n", mysql_error(&GmailApp->mysql));
		g_print("query was: |%s|\n", query);
		g_free(query);
		return;
	} 

	/* Check the result. */	
	res = mysql_store_result(&GmailApp->mysql);

	if (res == NULL) {
		g_warning("Query failed: %s\n", mysql_error(&GmailApp->mysql));
		g_warning("vfolder match query returned NULL. :(");
		g_print("query was: |%s|\n", query);
		return;
		}

	affected_rows = mysql_num_rows(res); 

	while ((mysqlrow = mysql_fetch_row(res))) {
		gboolean result=FALSE;
		
		/* Get the msg id. */
		id = (gint) atoi(mysqlrow[0]);

		/* Load the message.  */
		msg = load_vfolder_msg(id); 

		if (msg != NULL) {
			GMimeMessage *message;

			/* Get the full message in one big GString. */
			g_string_append(msg->headers, msg->message->str);

			/* g_print("Sending Queued:\n%s", msg->headers->str); */

			/* Convert to gmime. */
			message = g_mime_parser_construct_message(msg->headers->str, msg->headers->len, TRUE);	

			/* SMTP send it. */
			result = gmime_smtp_send(message); 

			/* Free the data. */
			g_mime_message_destroy(message);
			free_message(msg);

			/* Mark the message as read in the database. */	
			if (result) mark_readstatus(id, "Sent");

			/* Update the matched index for this message. */
			check_vfolder_matches(id, NULL);
		}
	}	

	g_free(query);
	mysql_free_result(res); 

	/* Update the current vfolder, it's contents may be changed. */
	vfolder_display_rightpane_by_mailbox(mailbox);

	/* Report back to the user. */
	if (affected_rows == 1)
		note = g_strdup_printf("Sent one queued message.");
	else
		note = g_strdup_printf("Sent %i queued messages.", affected_rows);
	gnome_appbar_set_status (GNOME_APPBAR (GmailApp->appbar), note);
	g_free(note);
}


/* Adds a file to be attached to a composed message. */
void add_file_attach (GtkWidget *widget, gpointer *data) 
{
	GladeXML *xml = glade_get_widget_tree(GTK_WIDGET (widget));
	GtkWidget *clist, *file;

	/* Get a pointer to the clist. */
	clist = glade_xml_get_widget(xml, "attach_clist");

	/* Open the file selection widget.. */
	file = gtk_file_selection_new ("Gmail: Attach File");

	/* Connect the ok_button to file_ok_sel function */
	gtk_signal_connect (GTK_OBJECT (GTK_FILE_SELECTION (file)->ok_button), "clicked", (GtkSignalFunc) file_attach_selected, file);
       
	/* Connect the cancel_button to destroy the widget */
	gtk_signal_connect_object (GTK_OBJECT (GTK_FILE_SELECTION (file)->cancel_button), "clicked", (GtkSignalFunc) gtk_widget_destroy, GTK_OBJECT (file)); 

	/* We need to remember the clist it is associated with. */
	gtk_object_set_data(GTK_OBJECT(file), "clist", clist);

	gtk_widget_show_all(file);
}

/* Removes a file to be attached to a composed message. */
void remove_file_attach (GtkWidget *widget, gpointer *data) 
{
	GladeXML *xml = glade_get_widget_tree(GTK_WIDGET (widget));
	GtkWidget *clist;
	gint row = -1;

	/* Get a pointer to the clist. */
	clist = glade_xml_get_widget(xml, "attach_clist");

	/* Find out what row is selected, if any. */
	row = GTK_CLIST(clist)->focus_row; /* FIXME: is focus_row ok??? */

	if (row >= 0 && row <= GTK_CLIST(clist)->rows) {
		/* Delete that row. */
		gtk_clist_remove(GTK_CLIST(clist), row);	
		}

}

void file_attach_selected(GtkWidget *widget, GtkFileSelection *fs)
{
	gchar *fullpath;
	GtkWidget *clist = NULL;
	
	/* We used to get the plain name easily from the file chooser, but
	 * this is now deprecated because we could be getting the file path
	 * from a dnd source. So we just pass the full path to the utility
	 * function that will parse it out in both cases.
	 * name = gtk_entry_get_text (GTK_ENTRY (fs->selection_entry)); */

	fullpath = g_strdup(gtk_file_selection_get_filename (GTK_FILE_SELECTION (fs))); 

	clist = (GtkWidget *) gtk_object_get_data(GTK_OBJECT(fs), "clist");

	add_attachment_to_clist(clist, fullpath);

	/* Close the file selection widget.  */
	gtk_widget_destroy(GTK_WIDGET(fs)); 

}

/* Add a file to a messages attachment clist.
 * This can be called from using a file selector or from drag and drop.
 */
void add_attachment_to_clist(GtkWidget *clist, gchar *fullpath) {
	gchar *vals[2];
	gchar *name = g_basename(fullpath);
	struct stat st;
	gint ret, row;
	FILE *fp;

	/* Get the details of the file. */
	ret = stat(fullpath, &st);
	if (ret < 0) {
		g_print("Couldn't stat the file: %s.\n", fullpath);
		return;
		}

	/* check that it is a regular file. */
	if (!S_ISREG(st.st_mode)) return;

	/* check that we have read permissions. */
	fp = fopen(fullpath, "r");
	if (fp == NULL) {
		GtkWidget *dialog;
		gchar *msg;

		msg = g_strdup_printf("Read permission error on %s.", name);
		dialog = gnome_message_box_new(msg, GNOME_MESSAGE_BOX_ERROR, "ok", 0);
    		gtk_window_set_position(GTK_WINDOW(dialog), GTK_WIN_POS_CENTER);
    		gtk_widget_show(dialog);

		g_free(msg);
		return;
	}
	fclose(fp);

	vals[0] = g_strdup_printf("%s", name);
	vals[1] = g_strdup_printf("%i", (gint) st.st_size);

	/* Add it to the clist. */
	row = gtk_clist_append(GTK_CLIST (clist), vals);

	/* Attach the full path of the file as row data. */
	gtk_clist_set_row_data(GTK_CLIST (clist), row, fullpath);

	g_free(vals[0]);
	g_free(vals[1]);

}

/* Loop through the entries in the attachments clist and add them as children
 * to multi_part.
 */
void add_attachments(GMimePart *multi_part, GtkWidget *clist)
{
	GMimePart *attach_part;
	GMimeContentType *mime_type;
	gint inlen = 0, ret;
	guchar *in = NULL;
	FILE *fp;
	struct stat st;
	gchar *filename = NULL, *name = NULL, *tmp = NULL;
	gint row;

	/* Loop through every entry in the clist. */
	for (row = 0; row < GTK_CLIST(clist)->rows; row++) {	
		GMimePartEncodingType et;

		/* Get the plain name, for the content disposition. */
		gtk_clist_get_text(GTK_CLIST(clist), row, 0, &name);

		/* Get the fully pathed filename for attaching. */
		filename = gtk_clist_get_row_data(GTK_CLIST (clist), row);
	
		attach_part = g_mime_part_new ();

		/* FIXME: use the gnome mime func to work out mime type. */
		mime_type = g_mime_content_type_new ("application", "octet-stream");
		g_mime_part_set_content_type (attach_part, mime_type);


		/* Calculate the content id. The body is 1, and the attachment
		 * row starts at 0. So we add 2 for the first attachment
		 * to become the value '2'.
		 */
		tmp = g_strdup_printf("%i", row + 2);
		g_mime_part_set_content_id (attach_part, tmp);
		g_free(tmp);
		g_mime_part_set_content_disposition(attach_part, "attachment");
		g_mime_part_set_filename(attach_part, name);

		ret = stat(filename, &st);
		if (ret < 0) {
			g_warning("couldn't stat file %s\n", filename);
			break; /* Skip to the next attachment. */
		} else {

		/* Continue attaching the file. */

		inlen = (gint) st.st_size;	/* Work out the size. */

		/* Make enought room to store the binary file. */
		in = g_new0(guchar, inlen); 

		/* Read in the data. */
		fp = fopen(filename, "r");
		if (fp == NULL) {
			g_warning("Unable to read attachment: %s. Skipped.\n", filename);
			break; /* Skip to next message. */
		}
		fread(in, sizeof(guchar), inlen, fp);
		fclose(fp);

		g_print("Attaching file %s (size = %i)\n", filename, inlen);

		/* Let gmime decide whether to use BASE64 or QP encoding.
		 * It seems to get it right.
		 */
		et = g_mime_utils_best_encoding(in, inlen);
		g_mime_part_set_encoding(attach_part, et);

		g_mime_part_set_content (attach_part, (const char *) in, inlen);
		g_mime_part_add_child (multi_part, attach_part);

		g_free(in);
		}
	}
}

/* Put the reply-to-all addresses in the cc field.
 * This is tricky.
 * It returns the 'cc' string of addresses to go into the new message.
 * Basically, we put all recipients of the original message (to or cc)
 * in the cc field, except the current address in the 'to' or 'from' fields.
 * FIXME: Don't insert the same message more than once.
 */
GString * insert_reply_to_all(GMimeMessage *message, gchar *replyto, gchar *from)
{
	GList *rcp;
	GString *text = g_string_new(NULL);

	rcp = g_mime_message_get_recipients(message, GMIME_RECIPIENT_TYPE_TO);
	while (rcp) {
		gchar *adr = internet_address_to_string(rcp->data, FALSE);
		/* Don't insert our email address or the replyto */
		if (adr && (strcmp(adr, from) != 0) && (strcmp(adr, replyto) != 0)) {
			if (text->len != 0) g_string_append(text, ", ");
			g_string_append(text, adr);
			}
		g_free(adr);
		rcp = rcp->next;
	}

	rcp = g_mime_message_get_recipients(message, GMIME_RECIPIENT_TYPE_CC);
	while (rcp) {
		gchar *adr = internet_address_to_string(rcp->data, FALSE);
		/* Don't insert our email address or the replyto */
		if (adr && (strcmp(adr, from) != 0) && (strcmp(adr, replyto) != 0)) {
			g_string_append(text, ", ");
			g_string_append(text, adr);
			}
		g_free(adr);
		rcp = rcp->next;
	}
	return(text);
}

/* gmail_wrap_string
 * wraps given string replacing spaces with '\n'.  do changes in place.
 * lnbeg - line beginning position, sppos - space position, 
 * te - tab's extra space.
 * Unashamedly stolen and adapted for gstring instead of gchar
 * from balsa-1.0.1 by Peter Allen (:
 * (void libbalsa_wrap_string(gchar * str, int width))
 */
void gmail_wrap_string(gchar *str, int width)
{
    const int minl = width / 2;
    gchar *lnbeg, *sppos, *ptr;
    gint te = 0;

    g_return_if_fail(str != NULL);
    lnbeg = sppos = ptr = str;

    while (*ptr) {
	switch (*ptr) {
	case '\t':
	    te += 7;
	    break;
	case '\n':
	    lnbeg = ptr + 1;
	    te = 0;
	    break;
	case ' ':
	    sppos = ptr;
	    break;
	}

	if ((ptr - lnbeg >= width - te) && (sppos >= lnbeg + minl)) {
	    *sppos = '\n';
	    lnbeg = sppos + 1;
	    te = 0;
	}
	ptr++;
    }
}


/* Recieve attachments from file-browsers sending URI lists. */
void  
attachment_drag_data_received  (GtkWidget          *widget,
			    GdkDragContext     *context,
			    gint                x,
			    gint                y,
			    GtkSelectionData   *data,
			    guint               info,
			    guint               time)
{
	GtkWidget *clist = widget; /* Attachment clist of the composer widget.*/
	GList *uri = NULL;

	g_return_if_fail(GTK_IS_CLIST(widget));

	g_print("attachment received! (%s)\n", data->data);

	/* Parse the URI's that have been dropped.
	 * Add all the file ones to the composer clist.
	 */

	uri = gnome_uri_list_extract_filenames(data->data);


	while (uri != NULL) {
		/* fullpath = get_filename_from_uri; */
		add_attachment_to_clist(clist, uri->data);
		uri = uri->next;
	}
}
