/* -*- mode: C++; c-basic-offset: 2; indent-tabs-mode: nil -*- */
/*
 *  Main authors:
 *     Guido Tack <tack@gecode.org>
 *
 *  Copyright:
 *     Guido Tack, 2006
 *
 *  Last modified:
 *     $Date: 2010-03-31 11:23:42 +0200 (Wed, 31 Mar 2010) $ by $Author: tack $
 *     $Revision: 10619 $
 *
 *  This file is part of Gecode, the generic constraint
 *  development environment:
 *     http://www.gecode.org
 *
 * Permission is hereby granted, free of charge, to any person obtaining
 * a copy of this software and associated documentation files (the
 * "Software"), to deal in the Software without restriction, including
 * without limitation the rights to use, copy, modify, merge, publish,
 * distribute, sublicense, and/or sell copies of the Software, and to
 * permit persons to whom the Software is furnished to do so, subject to
 * the following conditions:
 *
 * The above copyright notice and this permission notice shall be
 * included in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE
 * LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
 * OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
 * WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 *
 */

#include <gecode/gist/visualnode.hh>

#include <gecode/gist/layoutcursor.hh>
#include <gecode/gist/nodevisitor.hh>

#include <utility>

namespace Gecode { namespace Gist {

  Shape* Shape::leaf;
  Shape* Shape::hidden;

  /// Allocate shapes statically
  class ShapeAllocator {
  public:
    /// Constructor
    ShapeAllocator(void) {
      Shape::leaf = Shape::allocate(Extent(Layout::extent));
      Shape::hidden = Shape::allocate(Extent(Layout::extent), Shape::leaf);
    }
    ~ShapeAllocator(void) {
      Shape::deallocate(Shape::leaf);
      Shape::deallocate(Shape::hidden);
    }
  };

  /// Allocate shapes statically
  ShapeAllocator shapeAllocator;

  VisualNode::VisualNode(void)
  : shape(NULL)
  , offset(0)
  , box(0,0)
  {
    setDirty(true);
    setChildrenLayoutDone(false);
    setHidden(false);
    setMarked(false);
    setOnPath(false);
    setBookmarked(false);
  }

  VisualNode::VisualNode(Space* root)
  : SpaceNode(root)
  , shape(NULL)
  , offset(0)
  , box(0,0)
  {
    setDirty(true);
    setChildrenLayoutDone(false);
    setHidden(false);
    setMarked(false);
    setOnPath(false);
    setBookmarked(false);
  }

  VisualNode::~VisualNode(void) {
    Shape::deallocate(shape);
  }

  void
  VisualNode::dirtyUp(void) {
    VisualNode* cur = this;
    while (!cur->isDirty()) {
      cur->setDirty(true);
      if (! cur->isRoot()) {
        cur = cur->getParent();
      }
    }
  }

  void
  VisualNode::layout(void) {
    LayoutCursor l(this);
    PostorderNodeVisitor<LayoutCursor> p(l);
    // int nodesLayouted = 1;
    // clock_t t0 = clock();
    while (p.next()) {}
    // while (p.next()) { nodesLayouted++; }
    // double t = (static_cast<double>(clock()-t0) / CLOCKS_PER_SEC) * 1000.0;
    // double nps = static_cast<double>(nodesLayouted) /
    //   (static_cast<double>(clock()-t0) / CLOCKS_PER_SEC);
    // std::cout << "Layout done. " << nodesLayouted << " nodes in "
    //   << t << " ms. " << nps << " nodes/s." << std::endl;
  }

  void VisualNode::pathUp(void) {
    VisualNode* cur = this;
    while (cur) {
      cur->setOnPath(true);
      cur = cur->getParent();
    }
  }

  void VisualNode::unPathUp(void) {
    VisualNode* cur = this;
    while (!cur->isRoot()) {
      cur->setOnPath(false);
      cur = cur->getParent();
    }
  }

  int
  VisualNode::getPathAlternative(void) {
    for (int i=getNumberOfChildren(); i--;) {
      if (getChild(i)->isOnPath())
        return i;
    }
    return -1;
  }

  void
  VisualNode::toggleHidden(void) {
    setHidden(!isHidden());
    dirtyUp();
  }

  void
  VisualNode::hideFailed(void) {
    HideFailedCursor c(this);
    PreorderNodeVisitor<HideFailedCursor> v(c);
    while (v.next()) {}
    dirtyUp();
  }

  void
  VisualNode::unhideAll(void) {
    UnhideAllCursor c(this);
    PreorderNodeVisitor<UnhideAllCursor> v(c);
    while (v.next()) {}
    dirtyUp();
  }

  void
  VisualNode::toggleStop(void) {
    if (getStatus() == STOP)
      setStatus(UNSTOP);
    else if (getStatus() == UNSTOP)
      setStatus(STOP);
    dirtyUp();
  }
  
  void
  VisualNode::unstopAll(void) {
    UnstopAllCursor c(this);
    PreorderNodeVisitor<UnstopAllCursor> v(c);
    while (v.next()) {}
    dirtyUp();
  }

  void
  VisualNode::changedStatus() { dirtyUp(); }

  bool
  VisualNode::containsCoordinateAtDepth(int x, int depth) {
    if (x < box.left ||
        x > box.right ||
        depth >= shape->depth()) {
      return false;
    }
    Extent theExtent;
    if (shape->getExtentAtDepth(depth, theExtent)) {
      return (theExtent.l <= x && x <= theExtent.r);
    } else {
      return false;
    }
  }

  VisualNode*
  VisualNode::findNode(int x, int y) {
    VisualNode* cur = this;
    int depth = y / Layout::dist_y;

    while (depth > 0 && cur != NULL) {
      if (cur->isHidden()) {
        break;
      }
      VisualNode* oldCur = cur;
      cur = NULL;
      for (unsigned int i=0; i<oldCur->getNumberOfChildren(); i++) {
        VisualNode* nextChild = oldCur->getChild(i);
        int newX = x - nextChild->getOffset();
        if (nextChild->containsCoordinateAtDepth(newX, depth - 1)) {
          cur = nextChild;
          x = newX;
          break;
        }
      }
      depth--;
      y -= Layout::dist_y;
    }

    if(cur == this && !cur->containsCoordinateAtDepth(x, 0)) {
      return NULL;
    }
    return cur;
  }

  std::string
  VisualNode::toolTip(BestNode*, int, int) {
    return "";
  }

  /// \brief Helper functions for the layout algorithm
  class Layouter {
  public:
    /// Compute distance needed between \a shape1 and \a shape2
    static int getAlpha(Extent* shape1, int depth1,
                        Extent* shape2, int depth2);
    /// Merge \a shape1 and \a shape2 into \a result with distance \a alpha
    static void merge(Extent* result,
                      Extent* shape1, int depth1,
                      Extent* shape2, int depth2, int alpha);
  };

  int
  Layouter::getAlpha(Extent* shape1, int depth1,
                     Extent* shape2, int depth2) {
    int alpha = Layout::minimalSeparation;
    int extentR = 0;
    int extentL = 0;
    for (int i=0; i<depth1 && i<depth2; i++) {
      extentR += shape1[i].r;
      extentL += shape2[i].l;
      alpha = std::max(alpha, extentR - extentL + Layout::minimalSeparation);
    }
    return alpha;
  }

  void
  Layouter::merge(Extent* result,
                  Extent* shape1, int depth1,
                  Extent* shape2, int depth2, int alpha) {
    if (depth1 == 0) {
      for (int i=depth2; i--;)
        result[i] = shape2[i];
    } else if (depth2 == 0) {
      for (int i=depth1; i--;)
        result[i] = shape1[i];
    } else {
      // Extend the topmost right extent by alpha.  This, in effect,
      // moves the second shape to the right by alpha units.
      int topmostL = shape1[0].l;
      int topmostR = shape2[0].r;
      int backoffTo1 =
        shape1[0].r - alpha - shape2[0].r;
      int backoffTo2 =
        shape2[0].l + alpha - shape1[0].l;

      result[0] = Extent(topmostL, topmostR+alpha);

      // Now, since extents are given in relative units, in order to
      // compute the extents of the merged shape, we can just collect the
      // extents of shape1 and shape2, until one of the shapes ends.  If
      // this happens, we need to "back-off" to the axis of the deeper
      // shape in order to properly determine the remaining extents.
      int i=1;
      for (; i<depth1 && i<depth2; i++) {
        Extent currentExtent1 = shape1[i];
        Extent currentExtent2 = shape2[i];
        int newExtentL = currentExtent1.l;
        int newExtentR = currentExtent2.r;
        result[i] = Extent(newExtentL, newExtentR);
        backoffTo1 += currentExtent1.r - currentExtent2.r;
        backoffTo2 += currentExtent2.l - currentExtent1.l;
      }

      // If shape1 is deeper than shape2, back off to the axis of shape1,
      // and process the remaining extents of shape1.
      if (i<depth1) {
        Extent currentExtent1 = shape1[i];
        int newExtentL = currentExtent1.l;
        int newExtentR = currentExtent1.r;
        result[i] = Extent(newExtentL, newExtentR+backoffTo1);
        ++i;
        for (; i<depth1; i++) {
          result[i] = shape1[i];
        }
      }

      // Vice versa, if shape2 is deeper than shape1, back off to the
      // axis of shape2, and process the remaining extents of shape2.
      if (i<depth2) {
        Extent currentExtent2 = shape2[i];
        int newExtentL = currentExtent2.l;
        int newExtentR = currentExtent2.r;
        result[i] = Extent(newExtentL+backoffTo2, newExtentR);
        ++i;
        for (; i<depth2; i++) {
          result[i] = shape2[i];
        }
      }
    }
  }

  void
  VisualNode::setShape(Shape* s) {
    Shape::deallocate(shape);
    shape = s;
    setBoundingBox(shape->getBoundingBox());
  }

  void
  VisualNode::computeShape(VisualNode* root) {
    Extent extent(Layout::extent);
    int numberOfShapes = getNumberOfChildren();
    if (numberOfShapes == 1) {
      getChild(0)->setOffset(0);
      Shape* result = Shape::allocate(extent, getChild(0)->getShape());
      (*result)[1].extend(- extent.l, - extent.r);
      setShape(result);
    } else {
      // alpha stores the necessary distances between the
      // axes of the shapes in the list: alpha[i].first gives the distance
      // between shape[i] and shape[i-1], when shape[i-1] and shape[i]
      // are merged left-to-right; alpha[i].second gives the distance between
      // shape[i] and shape[i+1], when shape[i] and shape[i+1] are merged
      // right-to-left.
      assert(root->copy != NULL);
      Region r(*root->copy);
      std::pair<int,int>* alpha =
        r.alloc<std::pair<int,int> >(numberOfShapes);

      // distance between the leftmost and the rightmost axis in the list
      int width = 0;

      int maxDepth = 0;
      for (int i = numberOfShapes; i--;)
        maxDepth = std::max(maxDepth, getChild(i)->getShape()->depth());

      Extent* currentShapeL = r.alloc<Extent>(maxDepth);
      int ldepth = getChild(0)->getShape()->depth();
      for (int i=ldepth; i--;)
        currentShapeL[i] = (*getChild(0)->getShape())[i];

      // After merging, we can pick the result of either merging left or right
      // Here we chose the result of merging right
      Shape* mergedShape = Shape::allocate(maxDepth+1);
      (*mergedShape)[0] = extent;
      Shape* rShape = getChild(numberOfShapes-1)->getShape();
      int rdepth = rShape->depth();
      for (int i=rdepth; i--;)
        (*mergedShape)[i+1] = (*rShape)[i];
      Extent* currentShapeR = &(*mergedShape)[1];

      for (int i = 1; i < numberOfShapes; i++) {
        // Merge left-to-right.  Note that due to the asymmetry of the
        // merge operation, nextAlphaL is the distance between the
        // *leftmost* axis in the shape list, and the axis of
        // nextShapeL; what we are really interested in is the distance
        // between the *previous* axis and the axis of nextShapeL.
        // This explains the correction.

        Shape* nextShapeL = getChild(i)->getShape();
        int nextAlphaL =
          Layouter::getAlpha(&currentShapeL[0], ldepth,
                             &(*nextShapeL)[0], nextShapeL->depth());
        Layouter::merge(&currentShapeL[0], &currentShapeL[0], ldepth,
                        &(*nextShapeL)[0], nextShapeL->depth(), nextAlphaL);
        ldepth = std::max(ldepth,nextShapeL->depth());
        alpha[i].first = nextAlphaL - width;
        width = nextAlphaL;

        // Merge right-to-left.  Here, a correction of nextAlphaR is
        // not required.
        Shape* nextShapeR = getChild(numberOfShapes-1-i)->getShape();
        int nextAlphaR =
          Layouter::getAlpha(&(*nextShapeR)[0], nextShapeR->depth(),
                               &currentShapeR[0], rdepth);
        Layouter::merge(&currentShapeR[0],
                        &(*nextShapeR)[0], nextShapeR->depth(),
                        &currentShapeR[0], rdepth, nextAlphaR);
        rdepth = std::max(rdepth,nextShapeR->depth());
        alpha[numberOfShapes - i].second = nextAlphaR;
      }

      // The merged shape has to be adjusted to its topmost extent
      (*mergedShape)[1].extend(- extent.l, - extent.r);

      // After the loop, the merged shape has the same axis as the
      // leftmost shape in the list.  What we want is to move the axis
      // such that it is the center of the axis of the leftmost shape in
      // the list and the axis of the rightmost shape.
      int halfWidth = false ? 0 : width / 2;
      (*mergedShape)[1].move(- halfWidth);

      // Finally, for the offset lists.  Now that the axis of the merged
      // shape is at the center of the two extreme axes, the first shape
      // needs to be offset by -halfWidth units with respect to the new
      // axis.  As for the offsets for the other shapes, we take the
      // median of the alphaL and alphaR values, as suggested in
      // Kennedy's paper.
      int offset = - halfWidth;
      getChild(0)->setOffset(offset);
      for (int i = 1; i < numberOfShapes; i++) {
        offset += (alpha[i].first + alpha[i].second) / 2;
        getChild(i)->setOffset(offset);
      }
      setShape(mergedShape);
    }
  }

  size_t
  VisualNode::size(void) const {
    size_t s = sizeof(*this);
    s += sizeof(Node*)*getNumberOfChildren();
    if (shape && shape != Shape::leaf && shape != Shape::hidden) {
      s += sizeof(Shape)+sizeof(Extent)*(shape->depth()-1);
    }
    if (copy)
      s += copy->allocated();
    if (workingSpace)
      s += workingSpace->allocated();
    s += (choice != NULL ? choice->size() : 0);
    return s;
  }

}}

// STATISTICS: gist-any
