/* 

                          Firewall Builder

                 Copyright (C) 2002 NetCitadel, LLC

  Author:  Vadim Kurland     vadim@vk.crocodile.org

  $Id: NATCompiler_pf_writers.cpp,v 1.4 2005/01/28 07:54:58 vkurland Exp $

  This program is free software which we release under the GNU General Public
  License. You may redistribute and/or modify this program under the terms
  of that license as published by the Free Software Foundation; either
  version 2 of the License, or (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
 
  To get a copy of the GNU General Public License, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/

#include "NATCompiler_pf.h"

#include "fwbuilder/RuleElement.h"
#include "fwbuilder/NAT.h"
#include "fwbuilder/AddressRange.h"
#include "fwbuilder/IPService.h"
#include "fwbuilder/ICMPService.h"
#include "fwbuilder/TCPService.h"
#include "fwbuilder/UDPService.h"
#include "fwbuilder/CustomService.h"
#include "fwbuilder/Host.h"
#include "fwbuilder/Network.h"
#include "fwbuilder/Interface.h"
#include "fwbuilder/IPv4.h"
#include "fwbuilder/Firewall.h"


#include <iostream>
#if __GNUC__ > 3 || \
    (__GNUC__ == 3 && (__GNUC_MINOR__ > 2 || (__GNUC_MINOR__ == 2 ) ) ) || \
    _MSC_VER
#  include <streambuf>
#else
#  include <streambuf.h>
#endif
#include <iomanip>
#include <fstream>
#include <sstream>

#include <assert.h>

using namespace libfwbuilder;
using namespace fwcompiler;
using namespace std;



/**
 *-----------------------------------------------------------------------
 *                    Methods for printing
 */




NATCompiler_pf::PrintRule::PrintRule(const std::string &name) : NATRuleProcessor(name) 
{ 
    init=true; 
}

bool NATCompiler_pf::PrintRule::processNext()
{
    NATRule *rule=getNext(); if (rule==NULL) return false;

    tmp_queue.push_back(rule);

    string rl=rule->getLabel();
    if (rl!=current_rule_label) {

        compiler->output << "# " << endl;
        compiler->output << "# Rule  " << rl << endl;

        string    comm=rule->getComment();
        string::size_type c1,c2;
        c1=0;
        while ( (c2=comm.find('\n',c1))!=string::npos ) {
            compiler->output << "# " << comm.substr(c1,c2-c1) << endl;
            c1=c2+1;
        }
        compiler->output << "# " << comm.substr(c1) << endl;
        compiler->output << "# " << endl;

        current_rule_label=rl;
    }



    Address  *osrc=compiler->getFirstOSrc(rule);  assert(osrc);
    Address  *odst=compiler->getFirstODst(rule);  assert(odst);
    Service  *osrv=compiler->getFirstOSrv(rule);  assert(osrv);
                                      
    Address  *tsrc=compiler->getFirstTSrc(rule);  assert(tsrc);
    Address  *tdst=compiler->getFirstTDst(rule);  assert(tdst);
    Service  *tsrv=compiler->getFirstTSrv(rule);  assert(tsrv);

    string     iface_name = rule->getInterfaceStr();
//    Interface *iface = compiler->getCachedFwInterface(iface_id);
//    string iface_name= (iface!=NULL) ? iface->getName() : "";
    if (iface_name=="nil") iface_name="";

    switch ( rule->getRuleType() ) {
    case NATRule::Continue:
    case NATRule::NONAT:
	compiler->output  << "no nat ";
        if (iface_name!="") compiler->output << "on " << iface_name << " ";
	_printProtocol(osrv);
	compiler->output  << "from ";
	_printAddr( osrc );
	compiler->output  << "to ";
	_printAddr( odst );

	compiler->output  << endl;
        break;

    case NATRule::SNAT:
	compiler->output  << "nat ";
        if (iface_name!="") compiler->output << "on " << iface_name << " ";
	_printProtocol(osrv);
	compiler->output  << "from ";
	_printAddr( osrc );
	compiler->output  << "to ";
	_printAddr( odst );
        _printPort( osrv );

	compiler->output  << "-> ";
	_printAddr( tsrc , false );

	compiler->output  << endl;
        break;

    case NATRule::DNAT:
	compiler->output  << "rdr ";
        if (iface_name!="") compiler->output << "on " << iface_name << " ";
	_printProtocol(osrv);
	compiler->output  << "from ";
	_printAddr( osrc );
	compiler->output  << "to ";
	_printAddr( odst );
	_printPort(osrv);
	compiler->output  << "-> ";
	_printAddr( tdst , false );
	_printPort(tsrv);
	compiler->output  << endl;
        break;

    case NATRule::Redirect:
	compiler->output  << "rdr ";
        if (iface_name!="") compiler->output << "on " << iface_name << " ";
	_printProtocol(osrv);
	compiler->output  << "from ";
	_printAddr( osrc );
	compiler->output  << "to ";
	_printAddr( odst );
	_printPort(osrv);
	compiler->output  << "-> ";
	_printAddr( tdst , false );
	_printPort(tsrv);
	compiler->output  << endl;
        break;
    default: break;
    }

    return true;
}

void NATCompiler_pf::PrintRule::_printProtocol(Service *srv)
{
    if ( ! CustomService::isA(srv))
    {
        string s=srv->getProtocolName();
        if (s=="ip") s="{tcp udp icmp}";
        compiler->output << "proto " <<  s << " ";
    }
}

void NATCompiler_pf::PrintRule::_printPort(Service *srv)
{
    if (TCPService::isA(srv) || UDPService::isA(srv)) {
	int drs=srv->getInt("dst_range_start");
	if (drs!=0)   compiler->output << "port " << drs << " ";
    }
}


void NATCompiler_pf::PrintRule::_printAddr(Address  *o, bool print_netmask)
{
    IPAddress addr=o->getAddress();
    Netmask   mask=o->getNetmask();

    if (Interface::cast(o)!=NULL) {
	Interface *interface_=Interface::cast(o);
	if (interface_->isDyn()) {
	    compiler->output << "(" << interface_->getName() << ") ";
	    return;
	}

	mask=Netmask("255.255.255.255");
    }

    if (IPv4::cast(o)!=NULL) {
	mask=Netmask("255.255.255.255");
    }

    if (addr.toString()=="0.0.0.0" && mask.toString()=="0.0.0.0") {
	compiler->output << "any ";
    } else {
	compiler->output << addr.toString();
	if (print_netmask && mask.toString()!="255.255.255.255") {
	    compiler->output << "/" << mask.getLength();
	}
	compiler->output  << " ";
    }
}

