/* 

                          Firewall Builder

                 Copyright (C) 2000,2001 Vadim Kurland

  Author:  Vadim Kurland     vadim@vk.crocodile.org

  $Id: iptables.c,v 1.93 2001/12/07 00:17:51 vkurland Exp $


  This program is free software which we release under the GNU General Public
  License. You may redistribute and/or modify this program under the terms
  of that license as published by the Free Software Foundation; either
  version 2 of the License, or (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
 
  To get a copy of the GNU General Puplic License, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA


  ************************************************************************
  
  This is policy compiler for iptables. Some features supported by
  Firewall Builder and this policy compiler require "patch-o-matic"
  support. These features are (as of July 2001) :

  NETLINK target - pending adding of new Action code "User Process"

  NETMAP  target - when orig. src and trans. src are networks of
                   the same size. All other combinations of object and
		   network, or networks of different size, in these two
		   rule elemets are considered an error

  dropped-table  - implemented via global firewall option

  iplimit        - pending adding of new Action code "Limit"

  ipv4options    - used if IP Service object has options turned on,
                   except fragmentation which is implemented via standard
		   iptables command line option "-f"
		   
  time           - used to implement rule's time frame support
  
*/
 
#include <config.h>
#include "iptables.h"

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <assert.h>
#include <time.h>
#include <unistd.h>
#include <pwd.h>

FILE *ofile;
int   dynamic_address=0;

char  global_log_parameters[256];

char  global_log_level[256];
char  global_log_prefix[256];
char  global_log_tcp_seq[256];
char  global_log_tcp_opt[256];
char  global_log_ip_opt[256];
int   global_use_numeric_log_levels;
char  global_log_limit[256];
char  global_action_on_reject[256];
int   global_no_optimisation=0;
int   global_no_state=0;
int   global_manage_virtual_addr=0;
int   global_use_ip=1;

int   generate_input_output_groups_for_any=1;


char rule_chain[64];
char temp_i_chain[64];
char temp_o_chain[64];
char temp_f_chain[64];

char temp_n_chain[64];

GSList  *ip_aliases=NULL;


void parseOptions(xmlNodePtr options,
		  char* log_level,
		  char* log_prefix,
		  char* log_tcp_seq,
		  char* log_tcp_opt,
		  char* log_ip_opt,
		  int * use_numeric_log_levels,
		  char* limit,
		  char* action_on_reject,
		  int * no_optimisation,
		  int * no_state,
		  int * manage_virtual_addr)

{
    const char* cptr;
    int         val;
    
    strcpy(log_level,"");
    strcpy(log_prefix,"");
    strcpy(log_tcp_seq,"");
    strcpy(log_tcp_opt,"");
    strcpy(log_ip_opt,"");
    strcpy(limit,"");
    strcpy(action_on_reject,"");

    cptr=getOptionStr(options,"log_level");
    if (cptr && strlen(cptr)) {
	sprintf(log_level," --log-level %s",cptr);
    }
    cptr=getOptionStr(options,"log_prefix");
    if (cptr && strlen(cptr)) {
	sprintf(log_prefix," --log-prefix \"%s\"",cptr);
    }
    cptr=getOptionStr(options,"log_tcp_seq");
    if (cptr && strcasecmp(cptr,"true")==0) {
	strcpy(log_tcp_seq," --log-tcp-sequence ");
    }
    cptr=getOptionStr(options,"log_tcp_opt");
    if (cptr && strcasecmp(cptr,"true")==0) {
	strcpy(log_tcp_opt," --log-tcp-options ");
    }
    cptr=getOptionStr(options,"log_ip_opt");
    if (cptr && strcasecmp(cptr,"true")==0) {
	strcpy(log_ip_opt," --log-ip-options ");
    }

    val=getOptionBool(options,"use_numeric_log_levels");
    if (val) {
	*use_numeric_log_levels=val;
    }

    val=getOptionInt(options,"limit_value");

    if (val!=0) {
	sprintf(limit," -m limit --limit %d",val);
	cptr=getOptionStr(options,"limit_suffix");
	if (cptr && strlen(cptr)) strcat(limit,cptr);
    }
    val=getOptionBool(options,"no_optimisation");
    if (val) {
	*no_optimisation=val;
    }
    val=getOptionBool(options,"stateless");
    if (val) {
	*no_state=val;
    }
    val=getOptionBool(options,"manage_virtual_addr");
    if (val) {
	*manage_virtual_addr=val;
    }
    cptr=getOptionStr(options,"action_on_reject");
    if (cptr && strlen(cptr)) {
	if (strstr(cptr,"TCP")) {
	    strcpy(action_on_reject,"-p tcp --reject-with tcp-reset");
	}

	if (strstr(cptr,"ICMP")) {
	    if (strstr(cptr,"unreachable")) {
		strcpy(action_on_reject," --reject-with ");
		if (strstr(cptr,"net"))
		    strcat(action_on_reject,"icmp-net-unreachable");
		if (strstr(cptr,"host"))
		    strcat(action_on_reject,"icmp-host-unreachable");
		if (strstr(cptr,"port"))
		    strcat(action_on_reject,"icmp-port-unreachable");
		if (strstr(cptr,"proto"))
		    strcat(action_on_reject,"icmp-proto-unreachable");
	    }
	    if (strstr(cptr,"prohibited")) {
		strcpy(action_on_reject," --reject-with ");
		if (strstr(cptr,"net"))
		    strcat(action_on_reject,"icmp-net-prohibited");
		if (strstr(cptr,"host"))
		    strcat(action_on_reject,"icmp-host-prohibited");
	    }
	}

    }

    if (*use_numeric_log_levels) {
	if (strcmp(log_level," --log-level alert")==SAME)   strcpy(log_level," --log-level 1");
	if (strcmp(log_level," --log-level crit")==SAME)    strcpy(log_level," --log-level 2");
	if (strcmp(log_level," --log-level error")==SAME)   strcpy(log_level," --log-level 3");
	if (strcmp(log_level," --log-level warning")==SAME) strcpy(log_level," --log-level 4");
	if (strcmp(log_level," --log-level notice")==SAME)  strcpy(log_level," --log-level 5");
	if (strcmp(log_level," --log-level info")==SAME)    strcpy(log_level," --log-level 6");
	if (strcmp(log_level," --log-level debug")==SAME)   strcpy(log_level," --log-level 7");
    }

}

void parseFirewallOptions(xmlNodePtr options)
{
    parseOptions(options,
		 global_log_level,    
		 global_log_prefix,
		 global_log_tcp_seq,
		 global_log_tcp_opt,
		 global_log_ip_opt,
		 &global_use_numeric_log_levels,
		 global_log_limit,
		 global_action_on_reject,
		 &global_no_optimisation,
		 &global_no_state,
		 &global_manage_virtual_addr);


    generate_input_output_groups_for_any=
	getOptionBool(options,"firewall_is_part_of_any_and_networks");

}

gint _compareAddr(gconstpointer  a, gconstpointer  b)
{
    return (*(int*)a != *(int*)b);
}


void printARPEntryCommands(xmlNodePtr firewall,
			   xmlNodePtr outside_ip,
			   xmlNodePtr inside_ip)
{
    char outs[80];
    char ins[80];
    xmlNodePtr c1;
    int   *outs_addr;
    int    iface_addr;

    if (outside_ip==NULL || inside_ip==NULL) return;
    
    processSrcAndDst(outside_ip,outs);
    processSrcAndDst(inside_ip ,ins);

    /*
     * do not use fwb_malloc here because memory allocated by fwb_malloc
     * is freed after each rule is processed, but we want this list to
     * stay throughout processing of the whole policy
     */
    outs_addr=malloc(sizeof(int));
    
    *outs_addr=convertAddress(outside_ip);

    if (g_slist_find_custom(ip_aliases,outs_addr,_compareAddr)==NULL) {

	ip_aliases=g_slist_append(ip_aliases,outs_addr);
	
	for(c1=firewall->xmlChildrenNode; c1; c1=c1->next) {
	    if ( xmlIsBlankNode(c1) ) continue;
	    if (isElement(c1,"Interface")) {
		if ( isDynamicInterface(c1) ) continue;
/* if outside IP address for arp entry coincides with address of one
 * of the firewall's interfaces, we do not need arp entry */
		iface_addr=convertAddress(c1);
		if (*outs_addr==iface_addr) break;
	    
		if (global_use_ip) {
		    if (belongsToNet( c1 , outs )) 
			fprintf(ofile,"ip addr add %s dev %s scope link\n",
				outs,getName(c1));

		} else {
		    if (belongsToNet( c1 , outs )) 
			fprintf(ofile,"arp -Ds %s %s pub\n", outs, getName(c1) );

		    if (belongsToNet( c1 , ins ) && !global_use_ip) 
			fprintf(ofile,"route add %s dev %s\n",outs,getName(c1));
		}


	    }
	}
    }
}



void prologue(xmlNodePtr firewall,xmlNodePtr fw_options)
{
    const char* cptr;
    char ofname[256], tmpstr[256];
    int    i;
    xmlNodePtr c1;

    struct passwd *pwd;
    time_t         tm;
    char           timestr[256];

/*    ip_aliases=NULL; */
    
    sprintf(ofname,"%s.fw",getName(firewall));
    
    if ( (ofile=fopen(ofname,"w"))==NULL ) {
	fprintf(stderr,"Error: Could not open output file %s for writing\n",
		ofname);
	exit(1);
    }
    
    parseFirewallOptions( fw_options );


    if (getOptionBool(fw_options,"debug"))  fprintf(ofile,"#!/bin/sh -x\n");
    else                                    fprintf(ofile,"#!/bin/sh\n");

    
    tm=time(NULL);
    strcpy(timestr,ctime(&tm));
    timestr[ strlen(timestr)-1 ]='\0';
    
    pwd=getpwuid( getuid() );
    if (pwd==NULL) {
	fprintf(stderr,"Can't figure out your user name, aborting\n");
	exit(1);
    }

    fprintf(ofile,"#
#  This is automatically generated file. DO NOT MODIFY !
#
#  Firewall Builder  fwb_iptables v"VERSION" 
#
#  Generated %s %s by %s
#
#  
#
#\n", timestr , tzname[0] , pwd->pw_name );

    tmpstr[0]='\0';
    cptr=getOptionStr(fw_options,"log_level");
    if (cptr && strlen(cptr)) {
	strcpy(tmpstr,"-p ");
	strcat(tmpstr,cptr);
    }
    fprintf(ofile,"
if [ -x /usr/bin/logger ]; then
  logger %s \"Activating firewall script %s generated %s %s by %s\"
fi\n",tmpstr, ofname, timestr , tzname[0] , pwd->pw_name );

    
    if ( (cptr=getOptionStr(fw_options,"script_env_path"))!=NULL &&
	 strlen(cptr)!=0) {
	if ( strstr(cptr,"PATH=")!=NULL || strstr(cptr,"path=")!=NULL )
	    fprintf(ofile,"%s\n",cptr);
	else
	    fprintf(ofile,"PATH=\"%s\"\n",cptr);

	fprintf(ofile,"\n");
    }

    
    if (getOptionBool(fw_options,"load_modules")) {
	fprintf(ofile,"modprobe ip_conntrack || exit 1\n");
        fprintf(ofile,"modprobe ip_conntrack_ftp || exit 1\n");
        fprintf(ofile,"modprobe ip_nat_ftp || exit 1\n");
    }

/*
 *    Turn off packet forwarding for now. We'll turn it on if needed in the end
 */
    fprintf(ofile,"\n\n");
    fprintf(ofile,"FWD=`cat /proc/sys/net/ipv4/ip_forward`\n");
    fprintf(ofile,"echo \"0\" > /proc/sys/net/ipv4/ip_forward\n\n");

    if (getOptionBool(fw_options,"linux24_ip_dynaddr")) 
	fprintf(ofile,"echo \"1\" > /proc/sys/net/ipv4/ip_dynaddr\n\n");


    if (getOptionBool(fw_options,"linux24_rp_filter")) 
	fprintf(ofile,"echo \"1\" > /proc/sys/net/ipv4/conf/all/rp_filter\n\n");

    if (getOptionBool(fw_options,"linux24_accept_source_route")) 
	fprintf(ofile,"echo \"1\" > /proc/sys/net/ipv4/conf/all/accept_source_route\n\n");

    if (getOptionBool(fw_options,"linux24_accept_redirects")) 
	fprintf(ofile,"echo \"1\" > /proc/sys/net/ipv4/conf/all/accept_redirects\n\n");

    if (getOptionBool(fw_options,"linux24_log_martians")) 
	fprintf(ofile,"echo \"1\" > /proc/sys/net/ipv4/conf/all/log_martians\n\n");


    
    if (getOptionBool(fw_options,"linux24_icmp_echo_ignore_broadcasts")) 
	fprintf(ofile,"echo \"1\" > /proc/sys/net/ipv4/icmp_echo_ignore_broadcasts\n\n");

    if (getOptionBool(fw_options,"linux24_icmp_echo_ignore_all")) 
	fprintf(ofile,"echo \"1\" > /proc/sys/net/ipv4/icmp_echo_ignore_all\n\n");

    if (getOptionBool(fw_options,"linux24_icmp_ignore_bogus_error_responses")) 
	fprintf(ofile,"echo \"1\" > /proc/sys/net/ipv4/icmp_ignore_bogus_error_responses\n\n");


    
    if ( (i=getOptionInt(fw_options,"linux24_tcp_fin_timeout"))!=0 )
	fprintf(ofile,"echo \"%d\" > /proc/sys/net/ipv4/tcp_fin_timeout\n\n",i);

    if ( (i=getOptionInt(fw_options,"linux24_tcp_keepalive_interval"))!=0 )
	fprintf(ofile,"echo \"%d\" > /proc/sys/net/ipv4/tcp_keepalive_intvl\n\n",i);

    if (getOptionBool(fw_options,"linux24_tcp_window_scaling")) 
	fprintf(ofile,"echo \"1\" > /proc/sys/net/ipv4/tcp_window_scaling\n\n");

    if (getOptionBool(fw_options,"linux24_tcp_sack")) 
	fprintf(ofile,"echo \"1\" > /proc/sys/net/ipv4/tcp_sack\n\n");

    if (getOptionBool(fw_options,"linux24_tcp_fack")) 
	fprintf(ofile,"echo \"1\" > /proc/sys/net/ipv4/tcp_fack\n\n");

    if (getOptionBool(fw_options,"linux24_tcp_syncookies")) 
	fprintf(ofile,"echo \"1\" > /proc/sys/net/ipv4/tcp_syncookies\n\n");

    if (getOptionBool(fw_options,"linux24_tcp_ecn")) 
	fprintf(ofile,"echo \"1\" > /proc/sys/net/ipv4/tcp_ecn\n\n");

    if (getOptionBool(fw_options,"linux24_tcp_timestamps")) 
	fprintf(ofile,"echo \"1\" > /proc/sys/net/ipv4/tcp_timestamps\n\n");


    fprintf(ofile,"iptables -P OUTPUT  DROP\n");
    fprintf(ofile,"iptables -P INPUT   DROP\n");
    fprintf(ofile,"iptables -P FORWARD DROP\n");

    fprintf(ofile,"\n\n");

/*
 *    flush all the rules which existed before
 */

    fprintf(ofile,"

cat /proc/net/ip_tables_names | while read table; do
  iptables -t $table -L -n | while read c chain rest; do
      if test \"X$c\" = \"XChain\" ; then
        iptables -t $table -F $chain
      fi
  done
  iptables -t $table -X
done

");
    
    fprintf(ofile,"\n");

/*
 *  Remove all host static routes and "pub" ARP entries if we are going to 
 *  create new ones
 */

    if (global_manage_virtual_addr) {

	if (global_use_ip) {
	    for(c1=firewall->xmlChildrenNode; c1; c1=c1->next) {
		if ( xmlIsBlankNode(c1) ) continue;
		if (isElement(c1,"Interface")) {
		    if ( isDynamicInterface(c1) ) continue;

		    fprintf(ofile,"ip addr flush dev %s scope link\n",
			    getName(c1));
		}
	    }
	} else {
	    fprintf(ofile,"
/bin/netstat -rn | /usr/bin/awk \'{ 
    if ($3==\"255.255.255.255\") { printf \"route delete %%s\\n\",$1;}
}\' | /bin/sh

arp -an |   /usr/bin/awk \'/PERM/ {
    iface=$NF; addr=$2;
    gsub(\"[()]\",\"\",addr);
    printf \"arp -i %%s -d %%s pub\\n\",iface,addr;
}\' | /bin/sh
");
	}

	fprintf(ofile,"\n");
    }



    if ( ! getOptionBool(fw_options,"accept_new_tcp_with_no_syn")) {
	fprintf(ofile,"iptables -A OUTPUT  -p tcp ! --syn -m state --state NEW -j LOG\n");
	fprintf(ofile,"iptables -A OUTPUT  -p tcp ! --syn -m state --state NEW -j DROP\n");

	fprintf(ofile,"iptables -A INPUT   -p tcp ! --syn -m state --state NEW -j LOG\n");
	fprintf(ofile,"iptables -A INPUT   -p tcp ! --syn -m state --state NEW -j DROP\n");
					  
	fprintf(ofile,"iptables -A FORWARD -p tcp ! --syn -m state --state NEW -j LOG\n");
	fprintf(ofile,"iptables -A FORWARD -p tcp ! --syn -m state --state NEW -j DROP\n");
    }

    

    if (getOptionBool(fw_options,"log_all_dropped")) {

	strcpy(tmpstr,global_log_level);
	strcat(tmpstr,global_log_prefix);
	strcat(tmpstr,global_log_tcp_seq);
	strcat(tmpstr,global_log_tcp_opt);
	strcat(tmpstr,global_log_ip_opt);
	strcat(tmpstr,global_log_limit);
	
	fprintf(ofile,"iptables -t drop -A DROPPING -j LOG %s %s\n\n",
		global_log_parameters,
		tmpstr);

	printf(
"WARNING: Logging of all dropped packets requires
          dropped-table patch from patch-o-matic\n");
    }
    
    if (getOptionBool(fw_options,"clamp_mss_to_mtu")) {
	fprintf(ofile, "iptables -A FORWARD -p tcp --tcp-flags SYN,RST SYN -j TCPMSS --clamp-mss-to-pmtu\n");
	printf(
"WARNING: Using \"clamp MSS to MTU\" feature requires
          tcp-MSS patch from patch-o-matic\n");

    }
    
    
/*    fprintf(ofile,"iptables -N nat\n"); */

    fprintf(ofile,"\n\n\n");
    
    /*
       --state state
              Where  state  is a comma separated list of the con
              nection  states  to  match.   Possible  states  are
              INVALID  meaning that the packet is associated with
              no known connection, ESTABLISHED meaning  that  the
              packet  is  associated  with a connection which has
              seen packets in both directions, NEW  meaning  that
              the  packet has started a new connection, or other
              wise associated with a  connection  which  has  not
              seen  packets in both directions, and RELATED mean
              ing that the packet is starting a  new  connection,
              but is associated with an existing connection, such
              as an FTP data transfer, or an ICMP error.
     */

    if (getOptionBool(fw_options,"accept_established")) {
    
	fprintf(ofile,
"iptables -A INPUT   -m state --state ESTABLISHED,RELATED -j ACCEPT\n");
	fprintf(ofile,
"iptables -A OUTPUT  -m state --state ESTABLISHED,RELATED -j ACCEPT\n");
	fprintf(ofile,
"iptables -A FORWARD -m state --state ESTABLISHED,RELATED -j ACCEPT\n");
    }
}

void epilogue(xmlNodePtr firewall,xmlNodePtr fw_options)
{
    char  ofname[256];
    const char *c;
    
    sprintf(ofname,"%s.fw",getName(firewall));



/*
 *   Catch all rules
 */
    fprintf(ofile,"#\n#  Final rules\n#\n");

    fprintf(ofile,"iptables -A INPUT      -j DROP\n");
    fprintf(ofile,"iptables -A OUTPUT     -j DROP\n");
    fprintf(ofile,"iptables -A FORWARD    -j DROP\n");
/*    fprintf(ofile,"iptables -A nat        -j DROP\n"); */

/*
 *    Turn on packet forwarding
 */
    c=getOptionStr(fw_options,"linux24_ip_forward");
    if (c!=NULL) {
	fprintf(ofile,"\necho \"%s\" > /proc/sys/net/ipv4/ip_forward\n\n",c);
    }
    else
	fprintf(ofile,"\necho \"$FWD\" > /proc/sys/net/ipv4/ip_forward\n\n");
	

    printf("Wrote file %s\n",ofname);

    fclose(ofile);

    chmod(ofname,S_IXUSR|S_IRUSR|S_IWUSR|S_IRGRP);
}


int processSrcAndDst(xmlNodePtr obj,char *buff)
{
    const char *addr;

    *buff='\0';
    
    addr=getStr(obj,"address");
    if (addr==NULL || strlen(addr)==0) return(0);
    
    if (isElement(obj,"Host")) {
	strcpy(buff,addr);
	if (strcmp(buff,"0.0.0.0")==0) strcpy(buff,"0/0");
    } else
	if (isElement(obj,"Network")) {
	    sprintf(buff,"%s/%s",getStr(obj,"address"),getStr(obj,"netmask"));
	    if (strcmp(buff,"0.0.0.0/0.0.0.0")==0) strcpy(buff,"0/0");
	} else
	    if (isAnyNetwork(obj)) {
		strcpy(buff,"0/0");
	    } else
		strcpy(buff,addr);

    return(1);
}


void processSrc(xmlNodePtr src,triplet *tr)
{
    char        str[256];
    const char *mac_addr;
    xmlNodePtr  c1;
    int         mac_filtering;
    const char *addr;
    const char *iface_addr;
    
    addr=getStr(src,"address");

    mac_filtering=0;
    
    if(src==NULL) { tr->p_src=NULL; return; }
    
    processSrcAndDst(src,str);

    tr->p_src=fwb_strdup(str);

    for(c1=src->xmlChildrenNode; c1; c1=c1->next) {
	if ( xmlIsBlankNode(c1) ) continue;
	if (isElement(c1,"HostOptions")) {
	    if ( getOptionBool(c1,"use_mac_addr_filter") ) {
		mac_filtering=1;
		break;
	    }
	}
    }
    if (mac_filtering) {
	for(c1=src->xmlChildrenNode; c1; c1=c1->next) {
	    if ( xmlIsBlankNode(c1) ) continue;
	    if (isElement(c1,"Interface")) {
		iface_addr=getStr(c1,"address");
		if (addr==NULL || iface_addr==NULL ||
		    strcmp(addr,"0.0.0.0")==SAME ||
		    strcmp(iface_addr,"0.0.0.0")==SAME ||
		    isDynamicInterface(c1) ||
		    strcmp(addr,iface_addr)==SAME) {
		    mac_addr=getStr(c1,"physAddress");
		    if (mac_addr && strlen(mac_addr)!=0)
			tr->p_mac_src=fwb_strdup(mac_addr);
		    break;
		}
	    }
	}

    }
}

void processDst(xmlNodePtr dst,triplet *tr)
{
    char str[80];

    if(dst==NULL) { tr->p_dst=NULL; return; }
    
    processSrcAndDst(dst,str);
    tr->p_dst=fwb_strdup(str);
}

/*
 *  Day of the week :
 *  0 - Sunday
 *  1 - Monday
 *    ..
 *  6 - Saturday
 */
void processTime(xmlNodePtr time,char *buff)
{
    int   from1,from2,to1,to2;
    char  str[80];
    char  res[80];
    int   dow1,dow2;
    char  *days[7]={"Sun","Mon","Tue","Wed","Thu","Fri","Sat"};

    res[0]='\0';
    buff[0]='\0';

    if (time==NULL) return;
    if (isAnyInterval(time)) return;
    
    from1=getInt(time,"from_hour");
    from2=getInt(time,"from_minute");

    to1=getInt(time,"to_hour");
    to2=getInt(time,"to_minute");

    dow1=getInt(time,"from_weekday");
    dow2=getInt(time,"to_weekday");

    if (from1==-1 && from2==-1 && to1==-1 && to2==-1 && dow1==-1 && dow2==-1)
	return;

    if (from1!=-1 && from2==-1) from2=0;
    if (from1==-1 && from2!=-1) from1=0;
    
    if (from1!=-1 && from2!=-1) {
	sprintf(str,"--timestart %02d:%02d ",from1,from2);
	strcat(res,str);
    }
    
    if (to1!=-1 && to2==-1) to2=0;
    if (to1==-1 && to2!=-1) to1=0;

    if (to1!=-1 && to2!=-1) {
	sprintf(str,"--timestop %02d:%02d ",to1,to2);
	strcat(res,str);
    }
    
    if (dow1==-1) dow1=0; /* from the beginning of the week */
    if (dow2==-1) dow2=6; /* till the end of the week */
    str[0]='\0';
    for (; dow1>=0 && dow1<=dow2 && dow1<=6; ++dow1) {
	strcat(str,days[dow1]);
	strcat(str,",");
    }
    str[ strlen(str)-1 ]='\0'; /* removes the very last comma */
    strcat(res," --days ");
    strcat(res,str);
    if (strlen(res)!=0) {
	strcpy(buff," -m time ");
	strcat(buff,res);

	printf(
"WARNING: Matching start and stop time requires
          time patch from patch-o-matic\n");
    }
}


void processSrv(xmlNodePtr srv,triplet *tr)
{
    char sres[128], dres[128];
    int  sps,spe,dps,dpe;
    int  syn_flag, ack_flag, fin_flag, rst_flag, urg_flag, psh_flag;
    int  z;

    xmlNodePtr cur;
    const char* cplatform;
    const char* c;
    
    sps=spe=dps=dpe=0;

    if (isElement(srv,"CustomService") ) {

	for(cur=srv->xmlChildrenNode; cur; cur=cur->next) {
	    if ( xmlIsBlankNode(cur) ) continue;
	    cplatform=getStr(cur,"platform");
	    if (cplatform && strcmp(cplatform,"iptables")==0) {

		c =  FROMXMLCAST( xmlNodeGetContent(cur) );
		if (c!=NULL && strlen(c)!=0) {
		    addOptionCode(tr,c);
		    tr->state_new=0;

		    return;

		}  else {
		    fprintf(stderr,"Code for custom service '%s' is not defined for iptables platform\n",getName(srv));
		    exit(1);
		}
	    }
	}

	return;
    }

    if (isElement(srv,"TCPService") || isElement(srv,"UDPService") ) {
    
	strcpy(sres,"");
	strcpy(dres,"");
    
	sps=getInt(srv,"src_range_start");
	spe=getInt(srv,"src_range_end");
	dps=getInt(srv,"dst_range_start");
	dpe=getInt(srv,"dst_range_end");

	if ( spe!=0 ) {
	    if (sps==spe)  sprintf(sres,"%d",sps);
	    else           sprintf(sres,"%d:%d",sps,spe);
	}
	else {
	    if (sps!=0)    sprintf(sres,"%d",sps);
	}    

	if ( dpe!=0 ) {
	    if (dps==dpe)  sprintf(dres,"%d",dps);
	    else           sprintf(dres,"%d:%d",dps,dpe);
	}
	else {
	    if (dps!=0)    sprintf(dres,"%d",dps);
	}    
    

	tr->p_sprt=fwb_strdup(sres);
	tr->p_dprt=fwb_strdup(dres);

	if (isElement(srv, "UDPService")) {
	    tr->p_proto      = "udp";
	}
    
	if (isElement(srv, "TCPService")) {
	    tr->p_proto      = "tcp";
	    syn_flag=getBool(srv,"syn_flag");
	    ack_flag=getBool(srv,"ack_flag");
	    fin_flag=getBool(srv,"fin_flag");
	    rst_flag=getBool(srv,"rst_flag");
	    psh_flag=getBool(srv,"psh_flag");
	    urg_flag=getBool(srv,"urg_flag");
	    if (syn_flag ||  ack_flag ||  fin_flag ||
		rst_flag ||  psh_flag ||  urg_flag) {
		if (syn_flag && (!ack_flag && !fin_flag &&
				 !rst_flag && !psh_flag && !urg_flag) )
		    addOptionCode(tr,"--syn");
		else {
		    z=0;
		    addOptionCode(tr,"--tcp-flags ALL ");
		    if (syn_flag) {
			addOptionCode(tr,"SYN");
			z=1;
		    }
		    if (ack_flag) {
			if (z++) addOptionCode(tr,",");
			addOptionCode(tr,"ACK");
		    }
		    if (fin_flag) {
			if (z++) addOptionCode(tr,",");
			addOptionCode(tr,"FIN");
		    }
		    if (rst_flag) {
			if (z++) addOptionCode(tr,",");
			addOptionCode(tr,"RST");
		    }
		    if (psh_flag) {
			if (z++) addOptionCode(tr,",");
			addOptionCode(tr,"PSH");
		    }
		    if (urg_flag) {
			if (z++) addOptionCode(tr,",");
			addOptionCode(tr,"URG");
		    }
		}
	    }
	}
    }

    if (isElement(srv,"IPService") ){
	if (getInt(srv,"protocol_num")==0) tr->p_proto=NULL;
	else  tr->p_proto = fwb_strdup( getStr(srv,"protocol_num") );

	if (getBool(srv,"fragm")) {
	    addOptionCode(tr,"-f");
	    tr->state_new=0;
	}
	if (getBool(srv,"short_fragm")) {
	    addOptionCode(tr,"-f");
	    tr->state_new=0;
	}

	if  (getBool(srv,"lsrr") ||
	     getBool(srv,"ssrr") ||
	     getBool(srv,"rr") ||
	     getBool(srv,"ts") ) {

	    addOptionCode(tr," -m ipv4options ");
	    printf("
WARNING: IPv4 options support requires
ipv4options patch from patch-o-matic\n");
	}
	
	if (getBool(srv,"lsrr")) {
	    addOptionCode(tr," --lsrr");
	    tr->state_new=0;
	}
	
	if (getBool(srv,"ssrr")) {
	    addOptionCode(tr," --ssrr");
	    tr->state_new=0;
	}
	
	if (getBool(srv,"rr")) {
	    addOptionCode(tr," --rr");
	    tr->state_new=0;
	}
	
	if (getBool(srv,"ts")) {
	    addOptionCode(tr," --ts");
	    tr->state_new=0;
	}
	
    }

    if (isElement(srv,"ICMPService") ) {
	tr->p_proto      = "icmp";
	tr->p_icmp_type  = fwb_strdup( getStr(srv,"type") );
	tr->p_icmp_code  = fwb_strdup( getStr(srv,"type") );
    }


}


