/* 

                          Firewall Builder

                 Copyright (C) 2000 Vadim Kurland

  Author:  Vadim Kurland     vadim@vk.crocodile.org

  $Id: main_window.cc,v 1.85 2001/12/29 10:06:36 vkurland Exp $


  This program is free software which we release under the GNU General Public
  License. You may redistribute and/or modify this program under the terms
  of that license as published by the Free Software Foundation; either
  version 2 of the License, or (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
 
  To get a copy of the GNU General Public License, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/

#include <config.h>

#include <iostream>
#include <fstream>
#include <cstdlib>
#include <vector>

#include <gtk--.h>

#include <gdk/gdkkeysyms.h>

#include "glademm_support.hh"


#include "main_window.hh"
#include "helpers.hh"

#include "fwbuilder/Firewall.hh"
#include "fwbuilder/Policy.hh"
#include "fwbuilder/NAT.hh"

#include "FWObjectDatabaseGUI.hh"
#include "fwbuilder/Network.hh"
#include "DialogPlugin.hh"
#include "PrintDialog.hh"
#include "PolicyDialog.hh"
#include "NATDialog.hh"
#include "FirewallDialog.hh"
#include "PolicyList.hh"
#include "FWObjectBook.hh"
#include "AboutDialog.hh"
#include "OptionsDlg.hh"
#include "ExecDialog.hh"
#include "MessageDialog.hh"
#include "fwbuilder/FWOptions.hh"

#include "fwbuilder/Constants.hh"

#include "StandardRulesDruid.hh"
#include "DiscoveryDruid.hh"

using namespace libfwbuilder;


main_window::main_window() :  Gtk::Window(GTK_WINDOW_TOPLEVEL)
{

    custom_item=NULL;
    policy_in_right_pane=false;
    current_right_pane_dialog=NULL;
    pane_docked=true;
    current_lib="";
    floating_window=NULL;

    left_pane_width=150;
    stored_divider_pos=left_pane_width;

    main_vb          = manage(new class Gtk::VBox());

    main_menubar1    = manage(new class Gtk::MenuBar());
    hpaned           = manage(new class Gtk::HPaned());
    main_statusbar   = manage(new class Gtk::Statusbar());
   

    hpaned->set_handle_size(10);
    hpaned->set_gutter_size(6);
    hpaned->set_position(-1);

    main_vb->pack_start(*main_menubar1 , false, false );
    main_vb->pack_start(*hpaned        , true , true  );
    main_vb->pack_start(*main_statusbar, false, false );
    


    int sw=gdk_screen_width();
    int sh=gdk_screen_height();

    if (sw<=1024)
	set_default_size( 800,600);
    else  
	set_default_size( int(float(sw)/1.1), int (float(sh)/1.4) );

    set_position(GTK_WIN_POS_CENTER);

    set_policy(false,true,false);

    add(*main_vb);

    hpaned->show();
    main_statusbar->show();
    main_vb->show_all();



    obook=new FWObjectBook();

/*
 *  do not manage() left_pane_port and right_pane_port because we will
 *  unparent them later
 */
    left_pane_port     = new Gtk::ScrolledWindow();
    Gtk::Viewport *vp  = manage( new Gtk::Viewport() );
    
    ((Gtk::ScrolledWindow*)left_pane_port)->set_policy(GTK_POLICY_AUTOMATIC, 
						       GTK_POLICY_AUTOMATIC);
    ((Gtk::ScrolledWindow*)left_pane_port)->add(*vp);
    vp->add( *obook );


    obook->setMode(
	Preferences::global_prefs->getOptStr(
	    "/FWBuilderPreferences/UI/ObjectTreeMode") );

    left_pane_port->set_usize(150,-1);
    left_pane_port->size_allocate.connect(SigC::slot(this, 
			  &main_window::on_left_pane_size_allocate_event));
    hpaned->pack1( *left_pane_port , false, false );

    right_pane_port = new Gtk::Viewport();
    right_pane_port->add( *(new Gtk::Label("")) ); // placeholder
    right_pane_port->size_allocate.connect(SigC::slot(this, 
			  &main_window::on_right_pane_size_allocate_event));
    hpaned->pack2( *right_pane_port , true, true);


    main_menubar1->set_shadow_type( GTK_SHADOW_NONE );  // glade bug?

    agr = (Gtk::AccelGroup*)Gtk::AccelGroup::get_default();
/*
 * Mask here __enables__ different (actually all) modifiers. If mask is
 * set to 0, then all accelerators silently ignore modifiers
 * (Ctrl, Shift etc)
 */
    agr->set_default_mod_mask( GDK_MODIFIER_MASK );
    add_accel_group( *agr );

    build_menu();  // needs AccelGroup object 


    delete_event.connect( SigC::slot(this,&main_window::on_delete_event));
}

void main_window::setCurrentLibrary(const string &lib) 
{ 
    current_lib=lib; 
}

void main_window::on_left_pane_size_allocate_event(GtkAllocation *all)
{
    left_pane_width  = all->width;
    left_pane_height = all->height;
}

void main_window::on_right_pane_size_allocate_event(GtkAllocation *all)
{
    right_pane_width  = all->width;
    right_pane_height = all->height;
}

void main_window::LoadObjDB(const string &data_file)
{   
    g_assert(FWObjectDatabaseGUI::getInstance() != NULL);

    try
    {
        FWObjectDatabaseGUI::getInstance()->load(data_file);
    } catch(FWException &ex)
    {
	MessageDialog::LongTextError("Error loading data file. ",
				     ex.toString());
    }
	
    obook->clearSelection();
    obook->build();
    setCurrentLibrary("");
}

void main_window::on_new1_activate()
{   
    if(safe_to_close_right_pane()) 
    {
        string dbTemplateFile = TEMPLATE_DIR  "/" + FWObjectDatabaseGUI::DB_TEMPLATE_FILE_NAME;
	ClearRightPane(true);
	LoadObjDB(dbTemplateFile);
        FWObjectDatabase::db->setFileName("");
    }
}

void main_window::on_open1_activate()
{   
    if (safe_to_close_right_pane()) {
	ClearRightPane(true);
	LoadObjDB("");
    }
}

void main_window::on_save1_activate()
{   
    assert (FWObjectDatabase::db!=NULL);
    if ( ! safe_to_close_right_pane()) return ;
    try
    {
        FWObjectDatabaseGUI::getInstance()->save(); 
    } catch(FWException &ex)
    {
	MessageDialog::Error("Error saving data file.",ex.toString());
    }
}

void main_window::on_saveas1_activate()
{   
    assert (FWObjectDatabase::db!=NULL);
    if ( ! safe_to_close_right_pane()) return ;
    try
    {
        FWObjectDatabaseGUI::getInstance()->saveAs(); 
    } catch(FWException &ex)
    {
	MessageDialog::Error("Error saving data file.",ex.toString());
    }
}

bool main_window::checkAndSaveData()
{
    if ( ! safe_to_close_right_pane()) return false;
    if (FWObjectDatabase::db->isDirty(true) ) {
        try  {
            if (! FWObjectDatabaseGUI::getInstance()->saveIfModified() ) return false;  // "Cancel"
        } catch(FWException &ex)  {
	    MessageDialog::Error("Error saving data file.",ex.toString());
        }
    }
    return true;
}

/*
 * This event is generated in responce to the window manager's signal
 * sent to the window when user clicks "close" button on window's
 * title bar 
 */
gint main_window::on_delete_event(GdkEventAny* ev)
{
    if ( checkAndSaveData() ) {
	Gtk::Main::quit();
	return false;
    }
    return true;
}

void main_window::on_exit1_activate()
{   
    if ( checkAndSaveData() )
	Gtk::Main::quit();
}

void main_window::on_print_activate()
{   
    if(checkAndSaveData())
    {
        PrintDialog *pd = 
	    new PrintDialog(obook->getCurrentSelection()!="" ?
			    obook->getCurrentSelection().c_str():NULL
        );
        try
        {
            pd->run();
        } catch(FWException &ex)
        {
            MessageDialog::Error("Print error.",ex.toString());
        }
	delete pd;
    }
}

void main_window::on_edit1_activate()
{   
}

void main_window::on_options_activate()
{   
    bool show_properties=Preferences::global_prefs->getOptBool(
	"/FWBuilderPreferences/UI/ShowObjectPropertiesInTree");
    bool hide_navbar=Preferences::global_prefs->getOptBool(
	"/FWBuilderPreferences/UI/HideNavigationBar");


    OptionsDlg *pd=new OptionsDlg();
    pd->run();
    delete pd;

    obook->setMode(
	Preferences::global_prefs->getOptStr(
	    "/FWBuilderPreferences/UI/ObjectTreeMode") );

    if (show_properties!=Preferences::global_prefs->getOptBool(
	    "/FWBuilderPreferences/UI/ShowObjectPropertiesInTree") )
	obook->build();

    if (hide_navbar!=Preferences::global_prefs->getOptBool(
	    "/FWBuilderPreferences/UI/HideNavigationBar") )
	main_w->schedule_open_object(main_w->nowShowingObjectId());
}

void main_window::on_about1_activate()
{   
    AboutDialog::showAboutDialog();
}

               

void main_window::on_release_notes_activate()
{
    string file=DOCDIR "/ReleaseNotes_" VERSION;
    std::ifstream rn( file.c_str() );
    if (!rn) {
	MessageDialog::Error("Can not open Release Notes file " + file );
	return;
    }
    string buf, s;

    while (getline(rn,s)) { buf+=s; buf+="\n"; }

    MessageDialog::LongTextInfo( buf );
}

void main_window::showStatusbarMsg(const string& msg)
{
  main_statusbar->push(1,msg);
}

void main_window::clearStatusbar()
{
  main_statusbar->pop(1);
}

PolicyList* main_window::extractPolicyList()
{
    PolicyList *pl=NULL;

    if (current_right_pane_dialog) {
	DialogPlugin *dp= current_right_pane_dialog->getDialog();
	
	if (dp->get_name()=="policy_dialog")
	    pl=((PolicyDialog*)dp)->getPolicyList();
	
	if (dp->get_name()=="nat_dialog")
	    pl=((NATDialog*)dp)->getPolicyList();

	if (dp->get_name()=="firewall_dialog")
	    pl=((FirewallDialog*)dp)->getPolicyList();
    }
    return pl;
}

/*
 * Submenu Policy
 */
void main_window::on_insert_top()    
{ 
    if (current_right_pane_dialog==NULL) return;
    PolicyList *pl=extractPolicyList();
    if (pl!=NULL) pl->insertRuleAtTop(); 
}

void main_window::on_append_bottom() 
{ 
    if (current_right_pane_dialog==NULL) return;
    PolicyList *pl=extractPolicyList();
    if (pl!=NULL) pl->appendRuleAtBottom();  
}

void main_window::on_add_rule()
{
    if (current_right_pane_dialog==NULL) return;
    PolicyList *pl=extractPolicyList();
    if (pl!=NULL) pl->appendRuleAfter(-1);
}

void main_window::on_remove_rule()   
{ 
    if (current_right_pane_dialog==NULL) return;
    PolicyList *pl=extractPolicyList();
    if (pl!=NULL) pl->delRule(-1);
}

void main_window::on_move_up()       
{ 
    if (current_right_pane_dialog==NULL) return;
    PolicyList *pl=extractPolicyList();
    if (pl!=NULL) pl->moveRuleUp(-1);
}

void main_window::on_move_down()     
{
    if (current_right_pane_dialog==NULL) return;
    PolicyList *pl=extractPolicyList();
    if (pl!=NULL) pl->moveRuleDown(-1);
}

void main_window::on_build_autorules()
{
    if (current_right_pane_dialog==NULL) return;

    if (safe_to_close_right_pane()) {

	FWObject   *obj=nowShowingObject();

	while (obj && ! Firewall::isA(obj)) obj=obj->getParent();
	if (obj) {
//		ClearRightPane(false);

	    StandardRulesDruid *dr=new StandardRulesDruid( obj->getId(),
						   "Standard policy rules",
		       Resources::global_res->getIconPath("RulesDruidLogo") );
	    dr->show_all();
	}

    }
}

void main_window::on_compile()
{
    if (current_right_pane_dialog==NULL) return;

    if (safe_to_close_right_pane()) {
	FWObject   *obj=nowShowingObject();
	if (obj) {
	    if (Policy::isA(obj)) obj=obj->getParent();
	    if (NAT::isA(obj))    obj=obj->getParent();

	    if (Firewall::isA(obj))   {
		on_save1_activate();
// call compiler now

		Firewall  *fw=Firewall::cast(obj);
		FWOptions *fwopt=fw->getOptionsObject();
		
		string compiler=fwopt->getStr("compiler");
		if (compiler=="") {
		    compiler=Resources::global_res->getCompiler(
			obj->getStr("platform"));
		}
            
		ExecDialog *cmpl=
		    new ExecDialog(compiler , 
				   fwopt->getStr("cmdline") ,
				   FWObjectDatabase::db->getFileName(),
				   obj->getName() ,
				   Preferences::global_prefs->getWdir()
		    );

		cmpl->execute();
		delete cmpl;
	    }
	}
    }
}

void main_window::on_install()
{
    if (current_right_pane_dialog==NULL) return;

    if (safe_to_close_right_pane()) {
	FWObject   *obj=nowShowingObject();
	if (obj) {
	    if (Policy::isA(obj)) obj=obj->getParent();
	    if (NAT::isA(obj))    obj=obj->getParent();

	    if (Firewall::isA(obj))   {
		on_save1_activate();
// call install script now

		Firewall  *fw           = Firewall::cast(obj);
		FWOptions *fwopt = fw->getOptionsObject();
		
		string inst_script=fwopt->getStr("inst_script");
		if (inst_script=="") {
// TODO: may be Error dialog "install script is not defined"
		    return ;
		}
            
		ExecDialog *inst=
		    new ExecDialog(inst_script , 
				   fwopt->getStr("inst_cmdline") ,
				   FWObjectDatabase::db->getFileName(),
				   obj->getName() ,
				   Preferences::global_prefs->getWdir()
		    );

		inst->execute();
		delete inst;
	    }
	}
    }
}

void main_window::RefreshRightPane()
{
    PolicyList    *pl=extractPolicyList();
    if (pl) pl->schedule_rebuild(0);
}

string main_window::nowShowingObjectId()
{
    FWObject   *obj=nowShowingObject();
    return (obj)?obj->getId():"";
}

FWObject* main_window::nowShowingObject()
{
    if (current_right_pane_dialog) {
	DialogPlugin *dp= current_right_pane_dialog->getDialog();
	return (dp)?dp->getObject():NULL;
    }
    return NULL;
}

void main_window::on_tools_scan()
{
    string icn=
	Resources::global_res->getResourceStr("/FWBuilderResources/Paths/Icndir")+"/"+
	Resources::global_res->getResourceStr("/FWBuilderResources/UI/Icons/Discover");

    DiscoveryDruid *dr=new DiscoveryDruid("Objects Discovery",icn);

    dr->show_all();
}

/*
 *  for reference:
 *
    struct windowGeometry {
	Gdk_Rectangle wnd;
	Gdk_Rectangle trwnd;
	Gdk_Rectangle objwnd;
	int           divider;
    };
*/
void main_window::readGeometry(windowGeometry &c)
{
    int x,y,w,h;

    x  =Preferences::global_prefs->getOptInt(
	"/FWBuilderPreferences/UI/MainWindowX");
    y  =Preferences::global_prefs->getOptInt(
	"/FWBuilderPreferences/UI/MainWindowY");
    w  =Preferences::global_prefs->getOptInt(
	"/FWBuilderPreferences/UI/MainWindowW");
    h  =Preferences::global_prefs->getOptInt(
	"/FWBuilderPreferences/UI/MainWindowH");
    c.wnd=Gdk_Rectangle(x,y,w,h);


    x=Preferences::global_prefs->getOptInt(
	"/FWBuilderPreferences/UI/TreeWindowX");
    y=Preferences::global_prefs->getOptInt(
	"/FWBuilderPreferences/UI/TreeWindowY");
    w=Preferences::global_prefs->getOptInt(
	"/FWBuilderPreferences/UI/TreeWindowW");
    h=Preferences::global_prefs->getOptInt(
	"/FWBuilderPreferences/UI/TreeWindowH");
    c.trwnd=Gdk_Rectangle(x,y,w,h);


    x=Preferences::global_prefs->getOptInt(
	"/FWBuilderPreferences/UI/ObjWindowX" );
    y=Preferences::global_prefs->getOptInt(
	"/FWBuilderPreferences/UI/ObjWindowY" );
    w=Preferences::global_prefs->getOptInt(
	"/FWBuilderPreferences/UI/ObjWindowW" );
    h=Preferences::global_prefs->getOptInt(
	"/FWBuilderPreferences/UI/ObjWindowH" );
    c.objwnd=Gdk_Rectangle(x,y,w,h);

    c.divider=Preferences::global_prefs->getOptInt(
	"/FWBuilderPreferences/UI/MainWindowDivider");

}

void main_window::writeGeometry(windowGeometry &c)
{
    Preferences::global_prefs->setOptInt(
	"/FWBuilderPreferences/UI/MainWindowX",c.wnd->x);
    Preferences::global_prefs->setOptInt(
	"/FWBuilderPreferences/UI/MainWindowY",c.wnd->y);
    Preferences::global_prefs->setOptInt(
	"/FWBuilderPreferences/UI/MainWindowW",c.wnd->width);
    Preferences::global_prefs->setOptInt(
	"/FWBuilderPreferences/UI/MainWindowH",c.wnd->height);

    Preferences::global_prefs->setOptInt(
	"/FWBuilderPreferences/UI/MainWindowDivider",c.divider);

    Preferences::global_prefs->setOptInt(
	"/FWBuilderPreferences/UI/TreeWindowX",c.trwnd->x);
    Preferences::global_prefs->setOptInt(
	"/FWBuilderPreferences/UI/TreeWindowY",c.trwnd->y);
    Preferences::global_prefs->setOptInt(
	"/FWBuilderPreferences/UI/TreeWindowW",c.trwnd->width);
    Preferences::global_prefs->setOptInt(
	"/FWBuilderPreferences/UI/TreeWindowH",c.trwnd->height);
    
    Preferences::global_prefs->setOptInt(
	"/FWBuilderPreferences/UI/ObjWindowX" ,c.objwnd->x);
    Preferences::global_prefs->setOptInt(
	"/FWBuilderPreferences/UI/ObjWindowY" ,c.objwnd->y);
    Preferences::global_prefs->setOptInt(
	"/FWBuilderPreferences/UI/ObjWindowW" ,c.objwnd->width);
    Preferences::global_prefs->setOptInt(
	"/FWBuilderPreferences/UI/ObjWindowH" ,c.objwnd->height);
}


void main_window::restoreWindowConfiguration()
{
    windowGeometry g;

//    gdk_window_get_geometry(NULL, &i, &i, &w_display, &h_display, &i);

    readGeometry(g);

    bool float_pane=Preferences::global_prefs->getOptBool(
	"/FWBuilderPreferences/UI/FloatObjectTree");

    if (float_pane==pane_docked)  on_float_pane();

    if (pane_docked) {
	setWindowConfiguration(g.wnd);
	setDividerPosition(g.divider);
    } else {
	setTreeWndConfiguration(g.trwnd);
	setObjWndConfiguration( g.objwnd);
    }
}

void main_window::restoreWindowPosition()
{
    int i, w_display, h_display;

    windowGeometry g;

    readGeometry(g);

    gdk_window_get_geometry(NULL, &i, &i, &w_display, &h_display, &i);

    if (g.wnd->x>0      && g.wnd->x<w_display-100 && 
	g.wnd->y>0      && g.wnd->y<h_display-100 &&
	g.trwnd->x>0    && g.trwnd->x<w_display-100 &&
	g.trwnd->y>0    && g.trwnd->y<h_display-100 &&
	g.objwnd->x>0   && g.objwnd->x<w_display-100 &&
	g.objwnd->y>0   && g.objwnd->y<h_display-100 ) {
	    
	if (pane_docked) {
	    setWindowPosition(g.wnd->x,g.wnd->y);
	} else {
	    setTreeWndPosition(g.trwnd->x  , g.trwnd->y );
	    setObjWndPosition( g.objwnd->x , g.objwnd->y);
	}
    }
}

void main_window::saveWindowConfiguration()
{
    windowGeometry g;


/*
 *  if user closed window by clicking on "x" (via "delete" event) and not
 *  via menu item Quit, then window is going to be invisible at this point.
 *  In this case its size and coordinates are going to be equal to 0 and we
 *  should not store them in preferences file
 */
    if (is_visible()) {
	getWindowConfiguration(g.wnd);
	getTreeWndConfiguration(g.trwnd);
	getObjWndConfiguration( g.objwnd);
	g.divider=getDividerPosition();
    } else {
	readGeometry(g);
    }

    writeGeometry(g);


    Preferences::global_prefs->setOptBool(
	"/FWBuilderPreferences/UI/FloatObjectTree" ,
	! pane_docked );
	
    try
    {
	Preferences::global_prefs->savePrefs();
    } catch(FWException &ex)
    {
	MessageDialog::Error("Error saving preferences.",ex.toString());
    }

}

void main_window::on_float_pane()
{
    int div;
    Gdk_Rectangle wndr, treer, objr;

    getWindowConfiguration(wndr);
    getTreeWndConfiguration(treer);
    getObjWndConfiguration( objr);
    div=getDividerPosition();

    if (pane_docked) {
	floating_window=manage(new Gtk::Window());
	floating_window->set_name("FloatingPane");
        floating_window->set_policy(false,true,false);


	right_pane_port->reparent( *floating_window );

	stored_divider_pos=treer->width;

	left_pane_port->hide();

	hpaned->remove( *left_pane_port );
	main_vb->remove( *hpaned );

	main_vb->pack_start(*left_pane_port        , true , true  );
	main_vb->reorder_child( *left_pane_port , 1 );

//	main_tbl->attach(*left_pane_port , 0,1,1,2 , 
//			 GTK_FILL|GTK_EXPAND, GTK_FILL|GTK_EXPAND);

	left_pane_port->show();

	Gtk::MenuItem *itm=(Gtk::MenuItem*)( find_widget("float_pane",
							 main_menubar1) );
	assert (itm!=NULL);
	Gtk::Label *lbl=(Gtk::Label*)(itm->get_child());
	assert(lbl!=NULL);
	lbl->set_text("Dock Object Dialog");

	if (is_visible()) 
	    floating_window->show();

	floating_window->delete_event.connect( SigC::slot(this,
			      &main_window::on_floated_pane_delete_event));
	pane_docked=false;

	setObjWndConfiguration(objr);
	setTreeWndConfiguration(treer);

    } else {
	on_dock_pane();
    }
}

gint main_window::on_dock_pane()
{
    int ww,wh;
    int sw,sh;

    assert(floating_window!=NULL);

    Gdk_Rectangle wndr, treer, objr;

    getWindowConfiguration(wndr);
    getTreeWndConfiguration(treer);
    getObjWndConfiguration( objr);


    sw=gdk_screen_width();
    sh=gdk_screen_height();

    GtkWidget *widget=GTK_WIDGET(floating_window->gtkobj());
    if (widget->window)
	gdk_window_get_size(widget->window, &ww, &wh);


    main_vb->remove( *left_pane_port );

    hpaned  = manage(new class Gtk::HPaned());

    hpaned->set_handle_size(10);
    hpaned->set_gutter_size(6);
    hpaned->set_position(-1);


    hpaned->pack1( *left_pane_port , false, false );

    right_pane_port->unparent();
    hpaned->pack2( *right_pane_port , true, true );

    main_vb->pack_start(*hpaned        , true , true  );
    main_vb->reorder_child( *hpaned , 1 );

    hpaned->show();

    int width;
    if (wndr->width+ww>=sw) width=wndr->width;// combined window would be too wide
    else                    width=wndr->width+ww;
    
    wndr->width=width;
    setWindowConfiguration(wndr);

    setDividerPosition( 150 );


    Gtk::MenuItem *itm=(Gtk::MenuItem*)( find_widget("float_pane",
							 main_menubar1) );
    assert (itm!=NULL);
    Gtk::Label *lbl=(Gtk::Label*)(itm->get_child());
    assert(lbl!=NULL);
    lbl->set_text("Float Object Dialog");

    floating_window->hide();

    pane_docked=true;
    floating_window=NULL;

    return false;
}

gint main_window::on_floated_pane_delete_event(GdkEventAny* ev)
{
    Gtk::Main::idle.connect(slot(this,&main_window::on_dock_pane));
    return true; // deny closing window now
}

void main_window::getWindowConfiguration(Gdk_Rectangle &r)
{
    int a,b;

    GtkWidget *widget=GTK_WIDGET(gtkobj());
    if (widget->window) {
	gdk_window_get_position(widget->window, &a, &b);
	r->x=a; r->y=b;
	gdk_window_get_size(widget->window, &a, &b);
	r->width=a; r->height=b;
    }
}

void main_window::setWindowConfiguration(Gdk_Rectangle &r)
{
    setWindowPosition(r->x,r->y);
    set_default_size( r->width,r->height );
}

void main_window::setWindowPosition(gint16 x , gint16 y)
{
    if (is_visible()) {
//	GtkWidget *widget=GTK_WIDGET(gtkobj());
//	gdk_window_move(widget->window, x, y);
	set_uposition(x,y);
    }
}


void main_window::getTreeWndConfiguration(Gdk_Rectangle &r)
{
    getWindowConfiguration(r);
    if (pane_docked) {
	r->width=left_pane_width;
    }
}

void main_window::setTreeWndConfiguration(Gdk_Rectangle &r)
{
    setWindowConfiguration(r);
}

void main_window::setTreeWndPosition(gint16 x , gint16 y)
{
    if (is_visible()) {
	GtkWidget *widget=GTK_WIDGET(gtkobj());
	gdk_window_move(widget->window, x, y);
    }
}


void main_window::getObjWndConfiguration(Gdk_Rectangle &r)
{
    if (pane_docked) {
	getWindowConfiguration(r);
	r->x     += left_pane_width;
	r->width -= left_pane_width;
    } else {
	GtkWidget *widget=GTK_WIDGET(floating_window->gtkobj());
	if (widget->window) {
	    int a,b;
	    gdk_window_get_position(widget->window, &a, &b);
	    r->x=a; r->y=b;
	    gdk_window_get_size(widget->window, &a, &b);
	    r->width=a; r->height=b;
	}
    }
}

void main_window::setObjWndConfiguration(Gdk_Rectangle &r)
{
    if (! pane_docked) {
	setObjWndPosition( r->x ,  r->y);
	floating_window->set_default_size( r->width , r->height );
    }
}

void main_window::setObjWndPosition(gint16 x , gint16 y)
{
    if (! pane_docked) {
	if (floating_window->is_visible()) {
	    GtkWidget *widget=GTK_WIDGET(floating_window->gtkobj());
	    gdk_window_move(widget->window, x, y);
	}
    }
}

int  main_window::getDividerPosition()         { return left_pane_width;     }
void main_window::setDividerPosition(int size) 
{ 
    left_pane_width=size;
    hpaned->set_position(size); 
}

void main_window::show_impl()
{
    Gtk::Widget::show_impl();
    if (floating_window!=NULL) floating_window->show();
}

void main_window::show_all_impl()
{
    Gtk::Widget::show_all_impl();
    if (floating_window!=NULL) floating_window->show_all();
}
