proc printStack {} {
    set level [info level]
    for {set i 1} {$i < $level} {incr i} {
	puts "Level $i: [info level $i]"
    }
}

proc powInitGlobals {} {
   ##############################
   # This routine sets up various global variables which should be initialized
   # when POW is loaded... Some are used before the TCL routine powInit 
   # is called... eg, in powSetupColormap.  Putting it here instead of in
   # the C initialization routine, PowInit, makes it easily updated/modified
   # by developers (and us).  PowInit must call this routine.
   ##############################

   global powMinColorcells
   global powPseudoImages
   global powbg
   global powFrameForTop
   global yellowLineWidth
   global powWCS
   global regionParam
   global powLutButton powROIButton
   global POWRC tcl_platform env
   global isMac
   global powIsDragging
   global powOrderedGraphList

   global powNotifications

   ##############################
   #  Notifications sent by POW:
   #       on graph creation/redraw:  graphHasFinishedDrawing
   #       on graph destruction:      graphHasBeenDestroyed
   #       on graph deselection:      graphHasBeenUnselected
   #       on graph selection:        graphHasBeenSelected
   #       on image selection:        imageHasBeenSelected
   #       on graph resize:           graphHasResized
   #       on graph moved:            graphHasMoved   dx  dy

   set powNotifications [gNotifications]

   #
   ##############################

   # Identify if on a Mac
   if { $tcl_platform(platform)=="macintosh" } {
      set isMac 1
   } else {
      set isMac 0
   }

   #   Try to grab 128 colorcells to hold POW's colormaps
   set powMinColorcells 128
   #   Use pseudocolor images by default 
   set powPseudoImages 1

   #   Also need this so powToplevel can get going in safe mode
   if { $isMac } {
      # Set default background to a more "Platinum" appearance
      set powbg "#eeeeee"
   } else {
      set powbg "#cccccc"
   }

   #   By default, assume we're not running safe (need to make a powSafeInit
   #   entry point to do this eventually
   set powFrameForTop 0
   #   How wide should the yellow line around the selected graph be?
   set yellowLineWidth 3

   #   Initialize a couple of region handling globals
   set regionParam(nItems) 0
   set regionParam(gn) ""


   # These two variables are here so developers can easily override them...
   # The first determines which mouse button "diddles" the image colortable,
   # the second drags out a Region Of Interest
   set powLutButton 1
   set powROIButton 3

   trace variable powWCS w powWCSCallback

   # Define location of the POW resource file

   switch $tcl_platform(platform) {
      "unix"      { set POWRC "~/.powrc" }
      "windows"   { set POWRC [file join $env(POW_LIBRARY) "pow.ini"] }
      "macintosh" { set POWRC [file join $env(PREF_FOLDER) "pow Preferences"] }
   }

   powXPA::init

   # This is just a state variable
   set powIsDragging 0

   # This lists the graphs created, in order
   set powOrderedGraphList {}
}

########################################################################

proc powSetGlobal_bg { val } {
   global powbg

   if { $val == "" } return
   set powbg $val
   powChangeBg
}

proc powSetGlobal_cursor { val } {
   global powcursor

   set powcursor $val
   .pow       configure -cursor $val
   .pow.pow   configure -cursor $val
   .pow.scope configure -cursor $val
}

proc powSetGlobal_ResizeMain { val } {
   global powResizeMain

   set powResizeMain $val
}

proc powSetGlobal_GUIposition { val } {
   global powGUIposition

   set powGUIposition $val
   powLayoutGUI
   powUpdateGeometry
}

proc powSetGlobal_ScopeSize { val } {
   eval powCmds::scope $val
}

########################################################################

proc powGetVersion { } {
   set revisionString {$Revision: 1.384 $}
   regexp {Revision: ([0-9.]+)} $revisionString blob powVersion
   return $powVersion
}

proc powWCSCallback {array element op} {
    global powWCS
    if {$powWCS($element) == ""} return
    if {![catch {image type $element}]} {
	eval [concat powWCSInitImage $element $powWCS($element)];
    } elseif { [lsearch -exact [powListCurves] $element]!=-1 } {
	eval [concat powWCSInitCurve $element $powWCS($element)];
    }
}

proc powInit { {powXColormapWindow none} {powContainerPath none} {powgui 1}} {
#This procedure creates the .pow window and sets up all the bindings
#  powXColormapWindow -  a Tk path specifying a window to use to find the
#                        X colormap to use.
#  powContainer - what window to "use" to put POW in
#  powgui - Do you want the POW user GUI header (usually yes, unless you're
#           embedding POW in some other application)
    global currgn currimg mag powLargeFont Private_Colormap env
    global cmap cmap_inv powResizeMain powcursor
    global powImageParam powCurveParam powMenuOption
    global powbg powScopeWidth powScopeHeight powPlotParam powFontParam
    global powMinHeight powMinWidth powMaxHeight powMaxWidth
    global powFirstTimeThroughFlag
    global powShowHandlesFlag powTrackText 
    global powGraphCoordsTracker powImagePixelTracker  powImageValueTracker
    global powHelpTopics curr_img powFirstPixel
    global powSqueeze powSlide powPseudoImages powMinColorcells
    global Pow_Allocated powGUIposition
    global powScopeMargin powScopeSize powShowScope powScopeGn
    global powGUI showlinks powContainer powFrameForTop powDWP
    global powLutButton powROIButton POWRC
    global tcl_platform

    set powGUI $powgui
    set powContainer $powContainerPath
    if {($powXColormapWindow == "safe" || $powFrameForTop == 1) && \
	    $powContainer == "none"} {
	set powContainer "."
    }

    event add <<powExit>> <Control-c>
    set powFirstTimeThroughFlag 1
    set powShowHandlesFlag 1


#    trace variable powPlotParam w debug_trace


#set defaults for options

    if {$powXColormapWindow == "safe"} {
#don't set up a colormap if we're running in a tclet
	set powFrameForTop 1
	if {[winfo visual $powContainer] == "pseudocolor"} {
#	    set powPseudoImages 1
#	    This seems to break the plugin, so for now
	    set powPseudoImages 0
	} else {
	    set powPseudoImages 0
	}
	powToplevel .pow safe
    } else {	
        if {$powXColormapWindow == "none" || $powXColormapWindow == "NULL"} {
	    if {$powContainer != "none" && $powContainer != "NULL"} {
		powSetupColormap .pow 0 [list -use [winfo id $powContainer]]
	    } else {
		powSetupColormap .pow 0 
	    }
	    set powXColormapWindow .pow 
	} else {
	    set visual [winfo visual $powXColormapWindow]
	    set depth [winfo depth $powXColormapWindow]
	    if {![regexp -nocase "pseudocolor" $visual] || $depth != 8} {
                # This doesn't seem so bad
		# puts stderr "Visual of window $powXColormapWindow is $visual $depth."
		set powPseudoImages 0
	    }
	    
	    if {$powContainer != "none" && $powContainer != "NULL"} {
		powToplevel .pow $powXColormapWindow [list -use [winfo id $powContainer]]
	    } else {
		powToplevel .pow $powXColormapWindow 
	    }
	}



#Check that there's still enough colors for POW to function (i.e. they haven't
#filled up the Colormap since we set it up, this should only happen if 
#they're using the default colormap).
	if {$powPseudoImages} {
	    set free_colors [powTestColormap $powXColormapWindow]
	    if {(($Pow_Allocated != 0) && ($free_colors < 10)) || \
		    ($Pow_Allocated == 0) && ($free_colors < 60) } {
		puts stderr "Colormap full";
		set powPseudoImages 0;
	    }
	}
    }


#Since a master window must be the parent of a slave window or the descendant
#of the slaves parent, we can't use .pow.whatever for "popup" dialogs 
#DWP stands for DialogWindowPrefix
    if $powFrameForTop {
	set powDWP ".pow"
    } else {
	set powDWP ".pow."
    }

#powPseudoImages is set by powSetupColormap
    if {!$powPseudoImages} {
#	puts stderr "Pseudocolor images disabled; using Truecolor images."
	powSetupPhotoImages
    }

    #  Huge lists of fonts can't be listed on screen, so just grab
    #  some common ones.

    set allFonts [lsort [font families]]
    set powFontParam(allFonts,powDef) {}
    foreach fnt [list \
          {[Aa]rial} \
          {[Cc]ourier} \
          {[Hh]elvet} \
          {[Pp]alat} \
          {[Tt]imes} \
          {[Ss]ymbol} \
          ] {
       set i [lsearch -regexp $allFonts "^${fnt}.*"]
       if { $i >= 0 } {
          lappend powFontParam(allFonts,powDef) [lindex $allFonts $i]
       }
    }
    set fnt {[Cc]ourier}
    set i [lsearch -regexp $powFontParam(allFonts,powDef) "^${fnt}.*"]
    if { $i < 0 } {set i 0}
    set fnt [lindex $powFontParam(allFonts,powDef) $i]

    set powFontParam(allTypes,powDef) [list title axis tick note]
    set powFontParam(allOpts,powDef)  [list Font Size Weight Slant Color]
    foreach lbl $powFontParam(allTypes,powDef) {
        set powFontParam(${lbl}Font,powDef)   $fnt
        set powFontParam(${lbl}Size,powDef)   12
        set powFontParam(${lbl}Weight,powDef) normal
        set powFontParam(${lbl}Slant,powDef)  roman
        set powFontParam(${lbl}Color,powDef)  "#000000"
    }
    set powFontParam(titleSize,powDef) 16

    set powImageParam(allOpts,powDef) [list colormap invert scale]
    set powImageParam(colormap,powDef) spectrum
    set powImageParam(invert,powDef) No
    set powImageParam(scale,powDef) log
    set powImageParam(allMaps,powDef) [list \
	  [list Continuous gray blkbdy hot cold spectrum inv_spec \
	                   color1 color2 color3] \
	  [list Ramps gray-ramp2 gray-ramp4 bgr-ramp bgr-ramp2 \
	                   rygcbm-ramp bowlerhat] \
	  [list Steps gray-step4 gray-step8 bgr-step bgr-step2 \
	                   rygcbm-step tophat] \
	  ]

    set currgn "powDef"
    set powScopeGn "-"

    set powCurveParam(allOpts,powDef) \
	    [list pDisp pShape pSizeErr pSize pFill pColor \
	          lDisp lStyle lWidth lStep lBoxFill lColor \
		  logX logY LOD]
    foreach opt $powCurveParam(allOpts,powDef) \
	    val [list Yes Cross No 4 No #000000 \
	              No " " 1 No No #000000 \
		      No No 0] {
	set powCurveParam($opt,powDef) $val
    }
    set powCurveParam(allColors,powDef) \
	  [list Black  #000000  Red   #FF0000  Blue    #0000FF  Green  #00FF00 \
	        Yellow #FFFF00  White #FFFFFF  Purple  #9900FF  Orange #FF9900 \
	        Aqua   #00FFFF  Grey  #999999  Fuchsia #FF00FF  ]



    set screenHeight [winfo screenheight .]
    if { $screenHeight > 850 } {
       set powScopeWidth 200
       set powScopeHeight 200
    } else {
       set powScopeWidth 100
       set powScopeHeight 100
    }
    set powMinHeight 300
    set powMinWidth  300
    set powGUIposition top
    set powShowScope 1

    set powcursor crosshair
    set powResizeMain 1
    set showlinks 0
    set powFirstPixel 1


    set powPlotParam(allOpts,powDef) [list \
	  tickLabels xTickScal yTickScal xNumTicks yNumTicks \
	  xTickLength yTickLength xLabelTicks yLabelTicks \
	  tickFormatCmdX tickFormatCmdY \
	  GridLines GridColor GridDash \
          xdimdisp ydimdisp ]
    set powPlotParam(tickLabels,powDef) "degrees"
    set powPlotParam(tickFormatCmdX,powDef) "format %.6lg"	   
    set powPlotParam(tickFormatCmdY,powDef) "format %.6lg"	   
    set powPlotParam(xTickScal,powDef) "linear"
    set powPlotParam(yTickScal,powDef) "linear"
    set powPlotParam(xNumTicks,powDef) 3
    set powPlotParam(yNumTicks,powDef) 3
    set powPlotParam(GridLines,powDef) Yes
    set powPlotParam(GridColor,powDef) "#FFFFFF"
    set powPlotParam(GridDash,powDef) " "
    set powPlotParam(xdimdisp,powDef) 200
    set powPlotParam(ydimdisp,powDef) 200

    # order is [lft rgt top bot]
    set powPlotParam(xTickLength,powDef) [list 10 10 10 10]
    set powPlotParam(yTickLength,powDef) [list 10 10 10 10]
    set powPlotParam(xLabelTicks,powDef) [list Yes No No Yes]
    set powPlotParam(yLabelTicks,powDef) [list Yes No No Yes]


    # Initialize other powDef variables to simplify creating new graphs

    set powPlotParam(curves,powDef)   NULL
    set powPlotParam(images,powDef)   NULL
    set powPlotParam(xunits,powDef)   NULL
    set powPlotParam(yunits,powDef)   NULL
    set powPlotParam(xlabel,powDef)   X
    set powPlotParam(ylabel,powDef)   Y
    set powPlotParam(xBot,powDef)     NULL
    set powPlotParam(yBot,powDef)     NULL
    set powPlotParam(xTop,powDef)     NULL
    set powPlotParam(yTop,powDef)     NULL


#read user's option file if present

    catch {if [file readable $POWRC] { source $POWRC } }

# test for obsolete variable powCurvetype

    if { [info exists powCurvetype] } {
	if { [string first Points $powCurvetype]==-1 } {
	    set powCurveParam(pDisp,powDef) No
	} else {
	    set powCurveParam(pDisp,powDef) Yes
	}
	if { [string first Line $powCurvetype]==-1 } {
	    set powCurveParam(lDisp,powDef) No
	} else {
	    set powCurveParam(lDisp,powDef) Yes
	}
	unset powCurvetype
    }

# Convert old 1/0 booleans to Yes/No strings

    foreach opt [list pDisp pSizeErr pFill lDisp lStep] {
       if { $powCurveParam($opt,powDef) } {
	  set powCurveParam($opt,powDef) Yes
       } else {
	  set powCurveParam($opt,powDef) No
       }
    }

    set powSqueeze 0.0
    set powSlide 0.0

# Convert old powNumTicks to powX/YNumTicks

    if { [info exists powNumTicks] } {
       set powPlotParam(xNumTicks,powDef) $powNumTicks
       set powPlotParam(yNumTicks,powDef) $powNumTicks
    }

# Convert old pow* options to powPlotParam(...)

    if { [info exists powGrid] } {
       if { $powGrid } {
	  set powPlotParam(GridLines,powDef) Yes
       } else {
	  set powPlotParam(GridLines,powDef) No
       }
    }
    if { [info exists powGridColor] } {
       set powPlotParam(GridColor,powDef) $powGridColor
    }
    if { [info exists powGridDash] } {
       set powPlotParam(GridDash,powDef) $powGridDash
    }
    if { [info exists powSixties] } {
       if { $powSixties } {
	  set powPlotParam(tickLabels,powDef) "degrees"
       } else {
	  set powPlotParam(tickLabels,powDef) "decimal"
       }
    }

    set powMenuOption(tickScal) \
	  "$powPlotParam(xTickScal,powDef)-$powPlotParam(yTickScal,powDef)"
    
# Convert text colors to hex...
    if { [info exists powCurveParam(Color,powDef)] } {
       set powCurveParam(lColor,powDef) $powCurveParam(Color,powDef)
       set powCurveParam(pColor,powDef) $powCurveParam(Color,powDef)
       unset powCurveParam(Color,powDef)
    }

    set powCurveParam(lColor,powDef) \
	  [powColorToHex $powCurveParam(lColor,powDef)]

    set powCurveParam(pColor,powDef) \
	  [powColorToHex $powCurveParam(pColor,powDef)]

    set powPlotParam(GridColor,powDef) \
	  [powColorToHex $powPlotParam(GridColor,powDef)]

# Calculate Scopebox margin/size from current Width/Height values

    set powScopeSize [list $powScopeWidth $powScopeHeight]
    if { !($powScopeWidth && $powScopeHeight) } {
       set powScopeWidth  10
       set powScopeHeight 10
       set powShowScope   0
    } else {
       set powShowScope 1
    }
    set powScopeMargin [expr ($powScopeWidth+$powScopeHeight)/20]

    set Private_Colormap 4
    
    if {!$powPseudoImages} {
#	powSetColorTable
    }

    .pow configure -bg $powbg
    catch {wm title .pow "POW [powGetVersion]"}

    if $powGUI {

       ##############################################################
       #      Start Menus
       ##############################################################
       global isMac

       bind .pow <<PostMenus>>   "powEvents::postMenus %W"
       bind .pow <<CloseWindow>> "powEvents::ExitPOW"

       .pow config -menu .pow.mbar
       menu .pow.mbar -postcommand "powEvents::generate <<PostMenus>>" -bg $powbg
       if { $isMac } {
          set cmdkey "Cmd"
          set bdVal 0
          .pow.mbar add cascade -menu .pow.mbar.apple
          .pow.mbar add cascade -menu .pow.mbar.file    -label "File"
          .pow.mbar add cascade -menu .pow.mbar.edit    -label "Edit"
          .pow.mbar add cascade -menu .pow.mbar.colors  -label "Colors"
          .pow.mbar add cascade -menu .pow.mbar.tools   -label "Tools"
          .pow.mbar add cascade -menu .pow.mbar.help    -label "Help"
          menu .pow.mbar.apple -tearoff False
          .pow.mbar.apple add command -label "About POW" \
                -command {powHelp About.html}
       } else {
          set cmdkey "Alt"
          set bdVal 2
          .pow.mbar add cascade -menu .pow.mbar.file    -label "File"
          .pow.mbar add cascade -menu .pow.mbar.edit    -label "Edit"
          .pow.mbar add cascade -menu .pow.mbar.colors  -label "Colors"
          .pow.mbar add cascade -menu .pow.mbar.tools   -label "Tools"
          .pow.mbar add cascade -menu .pow.mbar.help    -label "Help"
       }

       #
       # FILE
       #
       
       menu .pow.mbar.file -bg $powbg -bd $bdVal \
           -postcommand "powEvents::generate <<PostMenus>>"
       .pow.mbar.file add command -label "Print/Save Plot..." \
             -command {powPrintBox} \
             -accelerator "$cmdkey+P"
       .pow.mbar.file add command -label "Close" \
             -command "powEvents::generate <<CloseWindow>>" \
             -accelerator "$cmdkey+W"

       #
       # COLORS
       #

       menu .pow.mbar.colors -bg $powbg -bd $bdVal \
           -postcommand "powEvents::generate <<PostMenus>>"
       foreach colorGrp $powImageParam(allMaps,powDef) {
          set cName [lindex $colorGrp 0]
          menu .pow.mbar.colors.c$cName -bg $powbg -bd $bdVal
          .pow.mbar.colors add cascade -menu .pow.mbar.colors.c$cName \
                -label "$cName"
          foreach color [lrange $colorGrp 1 end] {
             .pow.mbar.colors.c$cName add radiobutton -label $color \
                   -value $color \
                   -variable powImageParam(colormap,powDef) \
                   -command "powCmds::colormap $color"
          }
       }

       .pow.mbar.colors add separator

       .pow.mbar.colors add checkbutton -label "Invert Colortable" \
             -variable powImageParam(invert,powDef) \
             -onvalue Yes -offvalue No \
             -command {powCmds::colormap invert $powImageParam(invert${currimg},$currgn)}

       .pow.mbar.colors add separator

        .pow.mbar.colors add radiobutton -label "linear"         -value linear \
              -variable powImageParam(scale,powDef) \
              -command "powCmds::colormap scale linear"
        .pow.mbar.colors add radiobutton -label "square root"    -value sqrt \
              -variable powImageParam(scale,powDef) \
              -command "powCmds::colormap scale sqrt"
        .pow.mbar.colors add radiobutton -label "logarithmic"    -value log \
              -variable powImageParam(scale,powDef) \
              -command "powCmds::colormap scale log"
        .pow.mbar.colors add radiobutton -label "Histo Equalize" -value histo \
              -variable powImageParam(scale,powDef) \
              -command "powCmds::colormap scale histo"

	.pow.mbar.colors add command -label "Rescale Image..." \
              -command {powRescaleBox}

	.pow.mbar.colors add separator

	.pow.mbar.colors add command -label "Create Colorbar" \
              -command {powColorbar}
    
        #
        # EDIT
        #

	menu .pow.mbar.edit -bg $powbg -bd $bdVal \
            -postcommand "powEvents::generate <<PostMenus>>"

        if { $isMac } {
            .pow.mbar.edit add command -label "Can't Undo" -state disabled
            .pow.mbar.edit add separator
        }
        
	.pow.mbar.edit add command -label "Edit Graph..." \
	    -command {powEditGraphDlg $currgn}
	.pow.mbar.edit add command -label "Add Text Label..." \
	    -command {powEditNoteDlg $currgn -1}
	.pow.mbar.edit add command -label "Choose Graph Size..." \
	    -command powSetGraphSize

        .pow.mbar.edit add separator

	.pow.mbar.edit add command -label "Duplicate Graph" \
	    -command {powEditCreateNewGraph $currgn}
	.pow.mbar.edit add command -label "Delete Graph" \
	    -command {powDeleteGraph $currgn}
	menu .pow.mbar.edit.merge -bg $powbg -bd $bdVal \
            -postcommand {powGenerateMergeCascade}
	.pow.mbar.edit add cascade -label "Merge Graphs" \
	    -menu .pow.mbar.edit.merge 
	menu .pow.mbar.edit.unmap -bg $powbg -bd $bdVal \
	    -postcommand {powGenerateUnmapCascade}
	.pow.mbar.edit add cascade -label "Hide Graph" \
	    -menu .pow.mbar.edit.unmap
	menu .pow.mbar.edit.replot -bg $powbg -bd $bdVal \
	    -postcommand {powGenerateReplotCascade}  
	.pow.mbar.edit add cascade -label "Replot Graph" \
	    -menu .pow.mbar.edit.replot

        .pow.mbar.edit add separator

	menu .pow.mbar.edit.log -bg $powbg -bd $bdVal
	.pow.mbar.edit add cascade -label "Axes Transforms" \
	      -menu .pow.mbar.edit.log
       .pow.mbar.edit.log add radiobutton -label "Linear-Linear" \
	    -variable powMenuOption(tickScal) -value "linear-linear" \
	    -command { powCmds::axes linear linear }
       .pow.mbar.edit.log add radiobutton -label "Linear-Log" \
	    -variable powMenuOption(tickScal) -value "linear-log" \
	    -command { powCmds::axes linear log }
       .pow.mbar.edit.log add radiobutton -label "Log-Linear" \
	    -variable powMenuOption(tickScal) -value "log-linear" \
	    -command { powCmds::axes log linear }
       .pow.mbar.edit.log add radiobutton -label "Log-Log" \
	    -variable powMenuOption(tickScal) -value "log-log" \
	    -command { powCmds::axes log log }

        #  Tick Label format

	menu .pow.mbar.edit.tlabels -bg $powbg -bd $bdVal
	.pow.mbar.edit add cascade -label "Tick Labels" \
	    -menu .pow.mbar.edit.tlabels
	.pow.mbar.edit.tlabels add radiobutton -label "Decimal" \
	    -variable powPlotParam(tickLabels,$currgn) -value "decimal"  \
	    -command {
		powAdornGraph $currgn .pow.pow;
	        powRedrawGraphHandles $currgn
	    }
	.pow.mbar.edit.tlabels add radiobutton -label "Base 60 (deg)" \
	    -variable powPlotParam(tickLabels,$currgn) -value "degrees"  \
	    -command {
		powAdornGraph $currgn .pow.pow
	        powRedrawGraphHandles $currgn
	    }

       # Grid Line Options

	menu .pow.mbar.edit.grid -bg $powbg -bd $bdVal
	.pow.mbar.edit add cascade -label "Grid Line Options" \
	    -menu .pow.mbar.edit.grid
	.pow.mbar.edit.grid add checkbutton -label "Show Grid Lines" \
	    -variable powPlotParam(GridLines,$currgn) \
	    -onvalue Yes -offvalue No \
	    -command {powChangeGrid 1}

	.pow.mbar.edit.grid add separator

        foreach {clr hex} [list White #FFFFFF Black #000000 \
	      Blue #0000FF Red #FF0000] {
	   .pow.mbar.edit.grid add radiobutton -label $clr \
		 -variable powPlotParam(GridColor,$currgn) -value $hex \
		 -command {powChangeGrid 0}
	}

	.pow.mbar.edit.grid add separator

	foreach [list opt val] \
	        [list Solid " " "Small Dash" 10 "Large Dash" 20] {
	   .pow.mbar.edit.grid add radiobutton -label $opt \
		 -variable powPlotParam(GridDash,$currgn) -value $val \
		 -command {powChangeGrid 0}
	}

	.pow.mbar.edit.grid add separator

	.pow.mbar.edit.grid add command -label "Fewer Lines" \
	    -command {
	       if {$powPlotParam(xNumTicks,$currgn)>0} {
		  incr powPlotParam(xNumTicks,$currgn) -1
	       }
	       if {$powPlotParam(yNumTicks,$currgn)>0} {
		  incr powPlotParam(yNumTicks,$currgn) -1
	       }
               powChangeGrid 1
        }
	.pow.mbar.edit.grid add command -label "More Lines" \
	    -command {
                incr powPlotParam(xNumTicks,$currgn)
                incr powPlotParam(yNumTicks,$currgn)
                powChangeGrid 1
        }

	.pow.mbar.edit add separator
	.pow.mbar.edit add command -label "Preferences..." \
		-command {powDefaultOptions}

# Eliminate in favor of Preferences box with "Get Current" button
#	.pow.mbar.edit add separator
#	.pow.mbar.edit add command -label "Save Current Settings" \
#	    -command {powGetCurrVariables; powSaveConfig}

       #
       # TOOLS
       #

       menu .pow.mbar.tools -bg $powbg -bd $bdVal \
           -postcommand "powEvents::generate <<PostMenus>>"
       .pow.mbar.tools add command -label "Blink Graphs..." \
             -command {powBlinkGraphDlg}
       .pow.mbar.tools add command -label "Blink Images..." \
             -command {powMovie}
       .pow.mbar.tools add command -label "Region Files..." \
             -command {powRegion}
       .pow.mbar.tools add command -label "Make Contour Map..." \
             -command {powContour}
       .pow.mbar.tools add command -label "Draw Profile..." \
             -command {ProfileDlg}
       .pow.mbar.tools add command -label "Ruler..." \
             -command {OpenRuler}
       .pow.mbar.tools add command -label "Image Probe" \
             -command {imgProbeDialog}

        #
        # HELP
        #

	menu .pow.mbar.help -bg $powbg -bd $bdVal \
            -postcommand "powEvents::generate <<PostMenus>>"
        if { $isMac } {
            .pow.mbar.help config -tearoff False
        }
	set powHelpTopics(About.html) About
	set powHelpTopics(Overview.html) Overview
	set powHelpTopics(Moving_Graphs.html) "Moving Graphs"
	set powHelpTopics(ROI.html) "Panning/Zooming"
	set powHelpTopics(File.html) "File Menu"
	set powHelpTopics(Color.html) "Colors"
	set powHelpTopics(Edit.html) "Edit Menu"
	set powHelpTopics(Tools.html) "Tool Menu"
	set powHelpTopics(Regions.html) "Region Files"
	set powHelpTopics(Blinking.html) "Blinking"
	set powHelpTopics(Contours.html) "Contours"
	set powHelpTopics(DefaultOptions.html) "Preferences"
	set powHelpTopics(EditGraphs.html) "Editting Graphs"
	set powHelpTopics(EditObjects.html) "Editting Objects"
	set powHelpTopics(Profile.html) "Drawing Profile"
	set powHelpTopics(Ruler.html) "Measure"
	set powHelpTopics(Probe.html) "Image Probe"
	foreach topic [lsort [array names powHelpTopics]] {
           .pow.mbar.help add command -label $powHelpTopics($topic) \
                 -command "powHelp $topic" \
        }

        ##############################################################
        #  Build Scope Objects
        ##############################################################

        frame .pow.scopebuttons -bg $powbg
        button .pow.scopein -bg $powbg -text "Zoom In" \
              -command "powScopeZoom in"
        button .pow.scopezoom1 -bg $powbg -text "Replot" \
              -command "powEndROI 1"
        button .pow.scopeout -bg $powbg -text "Zoom Out" \
              -command "powScopeZoom out"

        frame .pow.scopeframe -bg $powbg
        canvas .pow.scope -bg $powbg -cursor $powcursor \
              -width $powScopeWidth -height $powScopeHeight
        label .pow.currgn -textvariable powScopeGn -background yellow \
              -relief sunken -foreground black

        ##############################################################
        #  Build Tracker Objects
        ##############################################################

	set powTrackText(gn)   "NULL"
	set powTrackText(img)  "NULL"
	set powTrackText(rx)   "X"
	set powTrackText(ry)   "X"
	set powTrackText(imgx) "X"
	set powTrackText(imgy) "X"
	set powTrackText(imgz) "X"
	set powTrackText(zunits) " "
	powUpdateTrackVars
    
	set powTrackerWidth 30
	frame .pow.trackers -bg $powbg -width $powTrackerWidth
	label .pow.graphtrack  -textvariable powGraphCoordsTracker \
	    -background $powbg -relief sunken -anchor w -justify left \
            -width $powTrackerWidth
	label .pow.pixeltrack  -textvariable powImagePixelTracker \
	    -background $powbg -relief sunken -anchor w -justify left \
            -width $powTrackerWidth
	label .pow.pixvaltrack -textvariable powImageValueTracker \
	    -background $powbg -relief sunken -anchor w -justify left \
            -width $powTrackerWidth

        frame .pow.gui -bg $powbg

        powLayoutGUI
    } 
#end powGUI if block

    frame .pow.bottom -bg $powbg 
    canvas .pow.pow -bg $powbg -cursor $powcursor\
	-xscrollcommand ".pow.scrollx set" \
	-yscrollcommand ".pow.scrolly set" -scrollregion [list 0 0 50 50]
    scrollbar .pow.scrolly -command "powScrollMainCanvas Y" -orient vertical\
	-bg $powbg
    scrollbar .pow.scrollx -command "powScrollMainCanvas X" -orient horizontal\
	-bg $powbg

    grid configure .pow.bottom -row 1 -column 1 -sticky "n s e w"
    grid rowconfigure    .pow 1  -weight 1
    grid columnconfigure .pow 1  -weight 1


    grid configure .pow.pow -sticky "n s e w" -row 0 -column 0 \
	-in .pow.bottom
    grid configure .pow.scrolly -sticky "n s e"  -row 0 -column 1 \
	-in .pow.bottom
    grid configure .pow.scrollx -sticky "n e w" -row 1 -column 0 \
	-in .pow.bottom
    
    
	
    grid rowconfigure .pow.bottom 0 -weight 1
    grid columnconfigure .pow.bottom 0 -weight 1

#put .pow.top above .pow.pow in the stacking order, 
#This should prevent window items on the
#.pow.pow canvas from "bleeding" off the edges of the .pow.pow canvas
    lower .pow.bottom

    powSetGeometry
    if $powGUI {
    
       if { ! $powShowScope } {
	  grid remove .pow.scopeframe
       }

       bind .pow.pow   <Motion> {set_tracker_info %x %y .pow.pow}
       bind .pow.scope <Motion> {set_tracker_info %x %y .pow.scope}

       # The 'powProcessClick' is required here since both BtnPress 
       # and ROI is bound to canvas, but only 1 can be executed

       powBindBtn <<ROI>> "bind .pow.pow" \
	     {powBeginROI %x %y .pow.pow} \
	     {powDragROI  %x %y .pow.pow; set_tracker_info %x %y .pow.pow} \
	     {powEndROI   0     .pow.pow}
	   
       powBindBtn <<ROI>> "bind .pow.scope" \
	     {powBeginROI %x %y .pow.scope} \
	     {powDragROI  %x %y .pow.scope; set_tracker_info %x %y .pow.scope} \
	     {powEndROI   0     .pow.scope}

       powBindBtn <<ROI_Drag>> "bind .pow.scope"  \
	     {catch {powPanROI %x %y .pow.scope}} \
	     {catch {powPanROI %x %y .pow.scope}} \
	     {catch {powEndROI 0     .pow.scope}}
    }

    bind .pow <<powExit>> {destroy .pow}
    bind .pow.pow <Destroy> {powExit}

    bind .pow.pow <<ROI>>         {+powProcessClick %x %y B%b}
    bind .pow.pow <<BtnPress>>    {powProcessClick %x %y B%b}
    bind .pow.pow <<DblBtnPress>> {powProcessClick %x %y B%bD}

    bind .pow <Down> {event generate .pow <Motion> -warp yes -x %x \
	  -y [expr %y+1]}
    bind .pow <Up> {event generate .pow <Motion> -warp yes -x %x \
	  -y [expr %y-1]}
    bind .pow <Left> {event generate .pow <Motion> -warp yes -x [expr %x - 1] \
	  -y %y}
    bind .pow <Right> {event generate .pow <Motion> -warp yes \
	  -x [expr %x + 1] -y %y}


    .pow.pow bind graphDragable <Enter> {
       if { !$powIsDragging } {
          .pow.pow configure -cursor fleur
       }
    }
    .pow.pow bind graphDragable <Leave> {
       if { !$powIsDragging } {
          .pow.pow configure -cursor $powcursor
       }
    }

    powBindBtn <<Drag>> ".pow.pow bind graphDragable" \
          {powDragGraph start %X %Y} \
          {powDragGraph drag  %X %Y} \
          {powDragGraph end   %X %Y}

    ####
    # Create the virtual events and default bindings
    ####

    event add <<BtnPress>>     <ButtonPress-1> <ButtonPress-2> <ButtonPress-3>
    event add <<DblBtnPress>>  <Double-ButtonPress-1>

    if { $powLutButton != 0 && $powLutButton != "NULL" } {
       event add    <<LUT>>  <ButtonPress-$powLutButton>
       if { $powLutButton == 3  &&  $tcl_platform(platform) == "macintosh" } {
	  event add <<LUT>>  <Command-ButtonPress-1>
       }
    }

    if { $powROIButton != 0 && $powROIButton != "NULL" } {
       # Must delete BtnPress sequence to prevent it from hiding the ROI event
       event delete <<BtnPress>>  <ButtonPress-$powROIButton>
       event add    <<ROI>>       <ButtonPress-$powROIButton>
       event add    <<ROI_Drag>>  <ButtonPress-1>
       if { $powROIButton == 3  &&  $tcl_platform(platform) == "macintosh" } {
	  event add <<ROI>>       <Command-ButtonPress-1>
       }
    }


    if { $tcl_platform(platform) != "macintosh" } {
        event add <<RGN_Create>>   <ButtonPress-3> 
    } else {
        event add <<RGN_Create>>   <ButtonPress-3>  <Command-ButtonPress-1>
    }
    event add <<RGN_Drag>>     <ButtonPress-1>
    event add <<RGN_DragPt>>   <ButtonPress-1>
    if { $tcl_platform(platform) != "macintosh" } {
        event add <<RGN_InsertPt>> <ButtonPress-3>
    } else {
        event add <<RGN_InsertPt>> <ButtonPress-3> <Command-ButtonPress-1>
    }
    event add <<RGN_Rotate>>   <ButtonPress-1>

    event add <<Drag>>         <ButtonPress-1> 

    event add <<ProfileCreate>> <ButtonPress-3>
    event add <<ProfileMod>>    <ButtonPress-1>
}

# The next two functions prevent multiple bindings from being executed
# for Btn events.  Only the first set of events will be accepted.

proc powBindBtn { event bindCmd beginCmd dragCmd endCmd } {

   regsub -all % $dragCmd %% dragCmd
   regsub -all % $endCmd %% endCmd

   set c1 [concat powBtn Begin $event [list $beginCmd] ]
   set c2 [concat $bindCmd <B%b-Motion> \{ \
	 [list powBtn Drag $event $dragCmd] \
	 \}]
   set c3 [concat $bindCmd <ButtonRelease-%b> \{ \
	 [concat $bindCmd <B%b-Motion> \{\}] \; \
	 [concat $bindCmd <ButtonRelease-%b> \{\}] \; \
	 [list    powBtn End $event $endCmd] \; \
	 \}]

   eval $bindCmd $event {"$c3; $c2; $c1"}
}

proc powBtn { evt cntxt cmd } {
    global powBtnState powBtnContext

    if { ![info exists powBtnState] } {
	set powBtnState   none
	set powBtnContext none
    }

    # If this is a different event from last time, execute command and remember
    # context... otherwise execute only if same context

    if { $evt != $powBtnState || $cntxt==$powBtnContext } {
	set powBtnState $evt
	set powBtnContext $cntxt
	uplevel #0 $cmd
    }
}


proc powLayoutGUI { } {
   global powGUIposition powShowScope

   # Build Tracker Box
   grid configure .pow.graphtrack -row 0 -column 0 -sticky ew \
         -in .pow.trackers
   grid configure .pow.pixeltrack -row 1 -column 0 -sticky ew \
         -in .pow.trackers
   grid configure .pow.pixvaltrack -row 2 -column 0 -sticky ew \
         -in .pow.trackers 

   # Build ScopeWindow
   grid configure .pow.currgn -row 0 -column 0 -sticky s \
         -in .pow.scopeframe
   grid configure .pow.scope  -row 1 -column 0 -sticky n \
         -in .pow.scopeframe

   if { $powGUIposition == "top" || $powGUIposition=="bottom" } {
      # Build ScopeButtons
      grid configure .pow.scopein    -row 0 -column 1 -sticky ew \
            -in .pow.scopebuttons
      grid configure .pow.scopezoom1 -row 1 -column 1 -sticky ew \
            -in .pow.scopebuttons
      grid configure .pow.scopeout   -row 2 -column 1  -sticky ew \
            -in .pow.scopebuttons

      grid configure .pow.trackers      -in .pow.gui -row 1 -column 0 \
            -sticky w -padx 3 -pady 3
      if { $powShowScope } {
         grid configure .pow.scopeframe -in .pow.gui -row 1 -column 1 \
               -sticky e
      }
      grid configure .pow.scopebuttons  -in .pow.gui -row 1 -column 2 \
            -sticky e

      grid columnconfigure .pow.gui 1 -weight 1

      grid rowconfigure    .pow.gui 0 -weight 0
      grid rowconfigure    .pow.gui 1 -weight 1

   } else {
      # Build ScopeButtons
      grid configure .pow.scopein    -row 1 -column 0 -sticky ew \
            -in .pow.scopebuttons
      grid configure .pow.scopezoom1 -row 1 -column 1 -sticky ew \
            -in .pow.scopebuttons
      grid configure .pow.scopeout   -row 1 -column 2  -sticky ew \
            -in .pow.scopebuttons

      grid configure .pow.trackers      -in .pow.gui -row 2 -column 1 \
            -sticky n -padx 5 -pady 5
      if { $powShowScope } {
         grid configure .pow.scopeframe -in .pow.gui -row 0 -column 1 \
               -sticky n
      }
      grid configure .pow.scopebuttons  -in .pow.gui -row 1 -column 1 \
            -sticky n

      grid columnconfigure .pow.gui 1 -weight 1

      grid rowconfigure    .pow.gui 0 -weight 1
      grid rowconfigure    .pow.gui 1 -weight 0
   }

   switch $powGUIposition {
      "top" {
         grid configure .pow.gui -in .pow -row 0 -column 1 \
               -sticky news
      }
      "bottom" {
         grid configure .pow.gui -in .pow -row 2 -column 1 \
               -sticky news
      }
      "left" {
         grid configure .pow.gui -in .pow -row 1 -column 0 \
               -sticky n
      }
      "right" {
         grid configure .pow.gui -in .pow -row 1 -column 2 \
               -sticky n
      }
      default {
         grid remove .pow.gui
      }
   }

   lower .pow.gui
}


proc powScrollMainCanvas {x_or_y args} {
    global powPreScrollCallback powPostScrollCallback
    if [info exists powPreScrollCallback] {
	eval $powPreScrollCallback $x_or_y $args
    }
    
    if {$x_or_y == "X"} {
	eval .pow.pow xview $args
    } else {
	eval .pow.pow yview $args
    }

    if [info exists powPostScrollCallback] {
	eval $powPostScrollCallback $x_or_y $args
    }
}
 
			  

proc powSetGeometry { } {
   global powHeaderHeight powHeaderWidth powGUIposition
   global powMaxWidth powMaxHeight powMinWidth powMinHeight
   global powRealMinWidth powRealMinHeight
   global powGUI powScopeSize powContainer tcl_platform

   update idletasks

   if $powGUI {

      if { $powGUIposition=="top" || $powGUIposition=="bottom" } {
         set powHeaderHeight [lindex [lsort -integer -decreasing [list \
               [expr 3*[winfo height .pow.graphtrack]] \
               [expr [winfo height .pow.scope]+[winfo height .pow.currgn]] \
               [winfo height .pow.scopebuttons] \
               ] ] 0]
         if { $tcl_platform(platform) != "macintosh" } {
            # Add in the menubar
            incr powHeaderHeight 32
         }
         set powHeaderWidth  0
         set minGuiWidth [expr [winfo width .pow.graphtrack] \
                                 + [winfo width .pow.scopeout] \
                                 + [winfo width .pow.scope] \
                                 + 15]
         if { $minGuiWidth > $powMinWidth } {
            set powRealMinWidth $minGuiWidth
         } else {
            set powRealMinWidth $powMinWidth
         }
         set powRealMinHeight [expr $powMinHeight + $powHeaderHeight]

      } elseif { $powGUIposition=="left" || $powGUIposition=="right" } {

         set powHeaderWidth [lindex [lsort -integer -decreasing [list \
               [winfo width .pow.graphtrack] \
               [winfo width .pow.scope] \
               [winfo width .pow.scopebuttons] \
               ] ] 0]
         set powHeaderHeight 0
         set minGuiHeight [expr 3* [winfo height .pow.graphtrack] \
                                 + [winfo height .pow.scopeout]   \
                                 + [winfo height .pow.scope]      \
                                 + [winfo height .pow.currgn]     \
                                 + 10]
         if { $minGuiHeight > $powMinHeight } {
            set powRealMinHeight $minGuiHeight
         } else {
            set powRealMinHeight $powMinHeight
         }
         if { $tcl_platform(platform) != "macintosh" } {
            # Add in the menubar
            incr powRealMinHeight 32
            set powHeaderHeight 32
         }
         set powRealMinWidth [expr $powMinWidth + $powHeaderWidth]

      } else {

         set powHeaderHeight 0
         set powHeaderWidth  0

         set powRealMinWidth  $powMinWidth
         set powRealMinHeight $powMinHeight
      }

   } else {
      set powHeaderHeight 0
      set powHeaderWidth  0

      set powRealMinWidth  0
      set powRealMinHeight 0
   }

   set powMaxHeight  [expr [winfo screenheight .pow] - $powHeaderHeight]
   set powMaxWidth   [expr [winfo screenwidth  .pow] - $powHeaderWidth]
      
   if {$powContainer == "none" || $powContainer == "NULL" } {
      catch {
	 wm minsize .pow $powRealMinWidth $powRealMinHeight
      }
      catch {
	 wm maxsize .pow [expr $powMaxWidth + $powHeaderWidth] \
               [expr $powMaxHeight + $powHeaderHeight]
      }
   } else {
      catch {wm geometry .pow "[winfo geometry $powContainer]"}
   }
}

proc powGetGraphOptions {gn} {
#returns list of extra graph options
    global powPlotParam

#test for requested graph    
    if { ![powListGraphs $gn] } {
	error "Graph $gn does not exist"
	return
    }

    lappend optlist "bgcolor" $powPlotParam(bgcolor,$gn)
    lappend optlist "xmargin" $powPlotParam(xmargin,$gn)
    lappend optlist "ymargin" $powPlotParam(ymargin,$gn)

    return $optlist
}


proc powGraphOptions {gn args} {
#add extra graph options here.  Arguments are "option value" pairs.
    global powPlotParam currgn

#test for requested graph...  Drop this... allow calls prior to creation
#    if { ![powListGraphs $gn] } {
#	error "Graph $gn does not exist"
#	return
#    }

    set argc [llength $args]
    if { $argc == 1 } {
       if { [info exists powPlotParam($args,$gn)] } {
          return $powPlotParam($args,$gn)
       } else {
          return ""
       }
    } elseif { [expr $argc %2] != 0 } {
       error "Arguments must be in 'option value' pairs"
    }

    foreach [list option value] $args {
	switch $option {
	    bgcolor {
		set powPlotParam(bgcolor,$gn) $value
                if [winfo exists .pow.pow] {
                   set cleanName [powCleanName $gn]handle
                   .pow.pow itemconfigure ${gn}bkg -fill $value -outline $value
                   if [winfo exists .pow.s$cleanName] {
                      .pow.s$cleanName configure  -bg $value
                   }
                   if [winfo exists .pow.ms$cleanName] {
                      .pow.ms$cleanName configure -bg $value
                   }
                   .pow.pow lower ${gn}bkg
                }
	    }

            xdimdisp -
            ydimdisp -
	    xmargin -
	    ymargin -
	    FixedAspect {
		set powPlotParam($option,$gn) $value
	    }

	    titleString -
	    titleAnchor -
	    titlePosition -
	    xNumTicks -
	    yNumTicks -
	    xlabel -
	    ylabel -
	    xunits -
	    yunits {
	       set powPlotParam($option,$gn) $value
	       set doAction(adornGraph) 1
	    }

	    xTickLength -
	    yTickLength -
	    xLabelTicks -
	    yLabelTicks -
	    tickLabels	-
	    tickFormatCmdX -
	    tickFormatCmdY -
	    xTickScal -
	    yTickScal {
		set powPlotParam($option,$gn) $value
		set doAction(adornGraph) 1
		set doAction(redrawGraphHandles) 1
	    }

	    # Grid and GridLines are same option.  Grid is archaic.
	    Grid -
	    GridLines {
		set powPlotParam(GridLines,$gn) $value
		set doAction(adornGraph) 1
	    }
	    GridColor {
		set clr [powColorToHex $value]
		set powPlotParam(GridColor,$gn) $clr
		if [winfo exists .pow.pow] {
		   .pow.pow itemconfig ${gn}grid -fill $clr
		}
	    }
	    GridDash {
		set powPlotParam(GridDash,$gn) $value
		if [winfo exists .pow.pow] {
		   .pow.pow itemconfig ${gn}grid -dash $value
		}
	    }

	    default {
		error "Unknown option $option"
	    }
	}
    }

    # Perform delayed actions if graph (and window) already exists

    if { [powListGraphs $gn] && [winfo exists .pow.pow] } {

       if { [info exists doAction(adornGraph)] } {
          powAdornGraph $gn .pow.pow
       }
       if { [info exists doAction(redrawGraphHandles)] } {
          powRedrawGraphHandles $gn
       }

       if { $gn == $currgn } {
          powUpdateGraphMenuOptions
       }
    }

}

proc powColorToHex { color } {
   global powCurveParam

   if { [string index $color 0]!="#" } {
      set idx [lsearch $powCurveParam(allColors,powDef) $color]
      if { $idx>-1 } {
	 incr idx
	 set color [lindex $powCurveParam(allColors,powDef) $idx]
      }
   }
   return $color
}

proc powToplevel {topwin refwin {options ""}} {
#this implements what *should* be default behavior for X colormap handling.  
#Apparently the evil of Xlib colormap handling is contagious.
#if we're running in the plugin or in single window mode, this creates
#a frame instead of a toplevel.
    global powFrameForTop powbg powContainer

    
    if {!$powFrameForTop} {
       eval {toplevel $topwin -colormap $refwin \
             -visual [list [winfo visual $refwin] [winfo depth $refwin]]} \
             $options
       focus $topwin
    } else {
#in a tclet just use frames
	frame $topwin -bg $powbg 

	if [info exists powContainer] {
	    grid $topwin -sticky news -in $powContainer
	} else {
	    grid $topwin -sticky news
	}
    }
}
    

proc powReditherImage {gn img {canvas .pow.pow} } {
#This recalculates the colors/intensities in the base Photo image.
#The call to powMagImage is necessary to make them appear.
    global powPseudoImages powImageParam

#We could redither pseudocolors, but it isn't necessary
    if { ! $powPseudoImages } {
        powReditherPhotoBlock $img \
                $powImageParam(RBmin${img},$gn) $powImageParam(RBmax${img},$gn)
    }

    if { [$canvas find withtag ${img}disp${gn}]!="" } {
       powMagImage $gn $img $canvas
    }
}


proc powReditherImages {gn img} {
#This checks whether we need to redither both a color bar and an image
#It also redithers the scope box if it is the current graph
    global powPseudoImages powPlotParam powImageParam currgn powGUI

#We could redither pseudocolors, but it isn't necessary
#    if $powPseudoImages return 

# Redither this image

    powReditherImage $gn $img

# Redither Scope

    if { $powGUI && $currgn == $gn && \
	  [.pow.scope find withtag ${img}disp${gn}scope]!="" } {
       powReditherImage ${gn}scope $img .pow.scope
    }

# Redither original image or colorbar if other exists

    foreach {gn2 img2} [powGetColorbarLink $gn $img] {}
    if { $gn2 != "" } {
       powReditherImage $gn2 $img2
    }
}


proc powCmapInvert { } {
global powPseudoImages curr_img
    if $powPseudoImages {
	$curr_img colormap invert
    } else {
	powPhotoColorTable invert
    }
}

proc powSetRanges { gn img min max } {
   global powGUI currgn powPlotParam powImageParam

   powSetRange $gn $img $min $max

# Set Scope

    if { $powGUI && $currgn == $gn } {
	powSetRange ${gn}scope $img $min $max
    }

# Set original image or colorbar if other exists

    foreach {gn2 img2} [powGetColorbarLink $gn $img] {}

    if { $gn2 != "" } {

	powSetRange $gn2 $img2 $min $max

        # Also need to update RB for undisplayed scope so must break modularity!
        if { $powGUI } {
	   set powImageParam(RBmin${img2},${gn2}scope) $min
	   set powImageParam(RBmax${img2},${gn2}scope) $max
	}

    }
}

proc powSetRange { gn img min max } {
   global powPseudoImages powImageParam powGUI

   set powImageParam(RBmin${img},$gn) $min
   set powImageParam(RBmax${img},$gn) $max

   if {$powPseudoImages} {
      ${img}disp${gn} range $min $max
   } else {
      if { $powGUI && [regexp scope$ $gn] } {
	 powReditherImage $gn $img .pow.scope
      } else {
	 powReditherImage $gn $img .pow.pow
      }
   }
}

proc powSetColorTable { gn img } { 
    global powPseudoImages currimg currgn powImageParam

    powSetLut $gn $img $powImageParam(scale${img},$gn)
    if {$powPseudoImages}  {
       ${img}disp${gn} colormap $powImageParam(colormap${img},$gn)
    } else {
	powPhotoColorTable $powImageParam(colormap${img},$gn)
    }
    invert_cmap_if_flag_set $gn $img
}

proc powGenerateReplotCascade { } {
    global powPlotParam powOrderedGraphList

    set idx 0
    .pow.mbar.edit.replot delete 0 end
    foreach gn $powOrderedGraphList {
       incr idx
       if { $gn=="" } continue
       if { [.pow.pow find withtag ${gn}box]=="" } {
          set state normal
       } else {
          set state disabled
       }
       set title $powPlotParam(titleString,$gn)
       if { $title=="" } {
          set title "Untitled $idx"
       }
       .pow.mbar.edit.replot add command -label $title \
             -command "powMapGraph $gn" -state $state
    }
}

proc powGenerateUnmapCascade { } {
    global powPlotParam powOrderedGraphList

    set idx 0
    .pow.mbar.edit.unmap delete 0 end
    foreach gn $powOrderedGraphList {
       incr idx
       if { $gn=="" } continue
       if { [.pow.pow find withtag ${gn}box]=="" } {
          set state disabled
       } else {
          set state normal
       }
       set title $powPlotParam(titleString,$gn)
       if { $title=="" } {
          set title "Untitled $idx"
       }
       .pow.mbar.edit.unmap add command -label $title \
             -command "powUnmapGraph $gn" -state $state
    }
}

proc powGenerateMergeCascade { } {
    global currgn powPlotParam powOrderedGraphList

    set idx 0
    .pow.mbar.edit.merge delete 0 end
    foreach gn $powOrderedGraphList {
       incr idx
       if { $gn=="" } continue
       if { $gn==$currgn } {
          set state disabled
       } else {
          set state normal
       }
       set title $powPlotParam(titleString,$gn)
       if { $title=="" } {
          set title "Untitled $idx"
       }
       .pow.mbar.edit.merge add command -label $title \
             -command "powMergeGraphs $gn" -state $state
    }
}

proc powProcessClick { x y binding} {
    global powClickCallback

    set gn [powWhereAmI [.pow.pow canvasx $x] [.pow.pow canvasy $y]] 
    if {$gn != "NULL"} {
#	powSelectGraph $gn
	if [info exists powClickCallback] {
	    set gcoords [powCanvasToGraph $gn [.pow.pow canvasx $x] [.pow.pow canvasy $y]]
	    set rx [lindex $gcoords 0]
	    set ry [lindex $gcoords 1]
	    $powClickCallback $gn $rx $ry $binding
	}
    }
}
	
proc powPrint { type name orient } {
    set bbox [.pow.pow bbox all]
    set width  [expr [lindex $bbox 2]-[lindex $bbox 0]]
    set height [expr [lindex $bbox 3]-[lindex $bbox 1]]
    if [string match "Print*" $type] {
	set fnam "[pid]tmp.ps"
	.pow.pow postscript -colormode color -rotate $orient -file $fnam \
		-width $width -height $height \
		-x [lindex $bbox 0] -y [lindex $bbox 1]
	set comm "cat $fnam | $name"
	exec /bin/sh -c $comm
	catch {exec rm $fnam}
    } elseif [string match "File*" $type] {
	.pow.pow postscript -colormode color -rotate $orient -file $name \
		-width $width -height $height \
		-x [lindex $bbox 0] -y [lindex $bbox 1]
    }
    powShowHandles 1
}



proc powRescaleBox { } {
    global powbg curr_img currimg currgn powRBmin powRBmax powbg powDWP
    global powImageParam powHisto

    if { [winfo exists ${powDWP}powRB] } { destroy ${powDWP}powRB }
    if {![info exists curr_img]} {
	error "You must first select an image"
	return
    }

    powToplevel ${powDWP}powRB .pow "-bg $powbg -class \"POW Rescale\"" 
    bind ${powDWP}powRB <<CloseWindow>> "destroy ${powDWP}"
    bind ${powDWP}powRB <Tab> "+powUpdateHisto ?"
    bind ${powDWP}powRB <Return> "+powUpdateHisto ?"
    

    label ${powDWP}powRB.min -text "Image min: $powRBmin($currimg)" -bg $powbg
    label ${powDWP}powRB.max -text "Image max: $powRBmax($currimg)" -bg $powbg

    
    label ${powDWP}powRB.cmin -text "Current min:" -bg $powbg
    entry ${powDWP}powRB.ecmin -relief sunken -bg $powbg \
	  -textvariable powImageParam(RBmin${currimg},$currgn)

    label ${powDWP}powRB.cmax -text "Current max:" -bg $powbg
    entry ${powDWP}powRB.ecmax -relief sunken -bg $powbg \
	  -textvariable powImageParam(RBmax${currimg},$currgn)
    
    frame ${powDWP}powRB.buttonFrame -bg $powbg
    button ${powDWP}powRB.apply -text Apply -command {
       powUpdateHisto 0
#      powSetRanges $currgn $currimg
       powCmds::colormap scale $powImageParam(scale${currimg},$currgn) \
	     $powImageParam(RBmin${currimg},$currgn) \
	     $powImageParam(RBmax${currimg},$currgn)
    }  -bg $powbg
    button ${powDWP}powRB.reset -text Reset -command {
#      powSetRanges $currgn $currimg
       powCmds::colormap scale $powImageParam(scale${currimg},$currgn) \
	     $powRBmin($currimg) $powRBmax($currimg)
       powUpdateHisto ?
    }  -bg $powbg
    button ${powDWP}powRB.exit -text Exit -command {destroy ${powDWP}powRB} \
            -bg $powbg

    grid configure ${powDWP}powRB.min -row 0 -column 0 -columnspan 2 -sticky w
    grid configure ${powDWP}powRB.max -row 0 -column 2 -columnspan 2 -sticky w
    grid configure ${powDWP}powRB.cmin -row 1 -column 0  -sticky w
    grid configure ${powDWP}powRB.ecmin -row 1 -column 1 -sticky w
    grid configure ${powDWP}powRB.cmax -row 1 -column 2  -sticky w
    grid configure ${powDWP}powRB.ecmax -row 1 -column 3  -sticky w
    grid configure ${powDWP}powRB.buttonFrame -row 4 -column 0 -columnspan 4 -sticky ew
    grid configure ${powDWP}powRB.apply -row 0 -column 0 -in ${powDWP}powRB.buttonFrame -sticky w
    grid configure ${powDWP}powRB.reset -row 0 -column 1 -in ${powDWP}powRB.buttonFrame
    grid configure ${powDWP}powRB.exit -row 0 -column 2 -in ${powDWP}powRB.buttonFrame -sticky e


    frame ${powDWP}powRB.histo -bg $powbg
    grid ${powDWP}powRB.histo -row 2 -column 0 -columnspan 4 \
            -padx 3 -pady 5 -sticky news
    
    canvas ${powDWP}powRB.histo.grph -relief sunken -width 256 -height 150 \
            -bg $powbg -bd 3
    grid ${powDWP}powRB.histo.grph -row 1 -column 2 -rowspan 3
    powBindBtn <<BtnPress>> "bind ${powDWP}powRB.histo.grph " \
            { powDragHistoBounds b1 %x; powDragHistoBounds b2 %x } \
            { powDragHistoBounds b1 %x } \
            { powUpdateHisto ? }
    
    canvas ${powDWP}powRB.histo.bnds -relief flat -width 266 -height 6 \
            -bg $powbg -bd 0 -highlightthickness 0
    grid ${powDWP}powRB.histo.bnds -row 0 -column 2
    
    frame ${powDWP}powRB.histo.histbutt -bg $powbg
    grid ${powDWP}powRB.histo.histbutt -row 4 -column 2

    radiobutton ${powDWP}powRB.histo.histbutt.fullrange \
            -text "Full Range" -bg $powbg -variable powHisto(range) \
            -value full -highlightthickness 0 -command {
        after idle { powUpdateHisto 1 }
    }
    radiobutton ${powDWP}powRB.histo.histbutt.currrange \
            -text "Current Range" -bg $powbg -variable powHisto(range) \
            -value curr -highlightthickness 0 -command {
        after idle { powUpdateHisto 1 }
    }
    grid ${powDWP}powRB.histo.histbutt.fullrange -row 1 -column 1
    grid ${powDWP}powRB.histo.histbutt.currrange -row 1 -column 2

    set powHisto(image) $currimg
    set powHisto(graph) $currgn
    set powHisto(min) $powRBmin($currimg)
    set powHisto(max) $powRBmax($currimg)
    set powHisto(range) full
    powUpdateHisto 1
}

proc powUpdateHisto { new } {
    global powDWP powRBmin powRBmax
    global powImageParam powHisto

    if { ![winfo exists ${powDWP}powRB] } { return }

    set img $powHisto(image)
    set gn  $powHisto(graph)

    if { $new=="?" } {
        if { $powHisto(range)!="full" } {
            set new 1
        } else {
            set new 0
        }
    }

    if { $new } {
        if { $powHisto(range)=="full" } {
            set powHisto(min) $powRBmin($img)
            set powHisto(max) $powRBmax($img)
        } else {
            set powHisto(min) $powImageParam(RBmin${img},$gn)
            set powHisto(max) $powImageParam(RBmax${img},$gn)
        }
    }

    set min $powImageParam(RBmin${img},$gn)
    set max $powImageParam(RBmax${img},$gn)
    if { $min > $max } {
        set tmp $min
        set min $max
        set max $tmp
        set powImageParam(RBmin${img},$gn) $min
        set powImageParam(RBmax${img},$gn) $max
    }

    set scale [expr $powHisto(max) - $powHisto(min) ]
    if { $scale != 0.0 } {
        set scale [expr 255.0 / $scale]
        set minBounds [expr round($scale * ($min - $powHisto(min)))]
        set maxBounds [expr round($scale * ($max - $powHisto(min)))]
    } else {
        set minBounds 0
        set maxBounds 255
    }
    
    if { $new } {
        set powHisto(b1) $min
        set powHisto(b2) $max

        ${powDWP}powRB.histo.bnds delete all

        ${powDWP}powRB.histo.bnds create polygon \
                [expr $minBounds+0] 0 \
                [expr $minBounds+10] 0 \
                [expr $minBounds+5] 5 \
                -fill black -tags b1
    
        ${powDWP}powRB.histo.bnds create polygon \
                [expr $maxBounds+0] 0 \
                [expr $maxBounds+10] 0 \
                [expr $maxBounds+5] 5 \
                -fill black -tags b2

        powBindBtn <<BtnPress>> "${powDWP}powRB.histo.bnds bind b1" \
                { powDragHistoBounds b1 %x } \
                { powDragHistoBounds b1 %x } \
                { powUpdateHisto ? }
	   
        powBindBtn <<BtnPress>> "${powDWP}powRB.histo.bnds bind b2" \
                { powDragHistoBounds b2 %x } \
                { powDragHistoBounds b2 %x } \
                { powUpdateHisto ? }
	   
        ${powDWP}powRB.histo.bnds bind b1 <Enter> \
                { ${powDWP}powRB.histo.bnds itemconfig b1 -fill red }
        ${powDWP}powRB.histo.bnds bind b1 <Leave> \
                { ${powDWP}powRB.histo.bnds itemconfig b1 -fill black }
        ${powDWP}powRB.histo.bnds bind b2 <Enter> \
                { ${powDWP}powRB.histo.bnds itemconfig b2 -fill red }
        ${powDWP}powRB.histo.bnds bind b2 <Leave> \
                { ${powDWP}powRB.histo.bnds itemconfig b2 -fill black }

    } else {

        if { $powHisto(b1) < $powHisto(b2) } {
            set minTag b1
            set maxTag b2
        } else {
            set minTag b2
            set maxTag b1
        }
        set powHisto($minTag) $min
        set powHisto($maxTag) $max
        ${powDWP}powRB.histo.bnds coords $minTag \
                [expr $minBounds+0] 0 \
                [expr $minBounds+10] 0 \
                [expr $minBounds+5] 5

        ${powDWP}powRB.histo.bnds coords $maxTag \
                [expr $maxBounds+0] 0 \
                [expr $maxBounds+10] 0 \
                [expr $maxBounds+5] 5
    }
    
    if { $new } {
        set histo [powGetHisto $img $powHisto(min) $powHisto(max)]
        set pixmax 0
        foreach val [lrange $histo 1 254] {
            if { $val>$pixmax } {
                set pixmax $val
            }
        }
        if { $pixmax==0 } {
            set pixmax 1
        }
        ${powDWP}powRB.histo.grph delete all
        set i 4
        foreach val $histo {
            set scaledVal [expr $val * 149 / $pixmax]
            if { $scaledVal > 153 } {
                set scaledVal 153
            }
            ${powDWP}powRB.histo.grph create line $i 153 $i [expr 153-$scaledVal] -fill blue
            incr i
        }
    }

    ${powDWP}powRB.histo.grph addtag blackLines enclosed \
            0 0 [expr $minBounds+3.5] 160
    ${powDWP}powRB.histo.grph addtag greyLines enclosed \
            [expr $minBounds+3.5] 0 [expr $maxBounds+4.5] 160
    ${powDWP}powRB.histo.grph addtag whiteLines enclosed \
            [expr $maxBounds+4.5] 0 270 160

    ${powDWP}powRB.histo.grph itemconfig blackLines -fill black
    ${powDWP}powRB.histo.grph itemconfig greyLines  -fill blue
    ${powDWP}powRB.histo.grph itemconfig whiteLines -fill white

    ${powDWP}powRB.histo.grph dtag blackLines
    ${powDWP}powRB.histo.grph dtag greyLines
    ${powDWP}powRB.histo.grph dtag whiteLines
}

proc powDragHistoBounds { tag x } {
    global powHisto powImageParam

    set img $powHisto(image)
    set gn  $powHisto(graph)

    set scale [expr $powHisto(max) - $powHisto(min) ]
    if { $scale == 0.0 } {
        return
    }
    set scale [expr 255.0 / $scale]
    set val [expr ($x-5)/$scale + $powHisto(min)]
    set powHisto($tag) $val

    if { $powHisto(b1) < $powHisto(b2) } {
        set powImageParam(RBmin${img},$gn) $powHisto(b1)
        set powImageParam(RBmax${img},$gn) $powHisto(b2)
    } else {
        set powImageParam(RBmin${img},$gn) $powHisto(b2)
        set powImageParam(RBmax${img},$gn) $powHisto(b1)
    }

    powUpdateHisto 0
}

proc powSetMagstepBox { } {
    global powPlotParam currgn powbg powDWP
    global powXMagstep powYMagstep powSaveXMagstep powSaveYMagstep

    if { $currgn=="powDef" } {
	error "You must first select a graph"
	return
    }
    if { [winfo exists ${powDWP}magstep] } {
	raise ${powDWP}magstep
	return
    }
    powToplevel ${powDWP}magstep .pow "-bg $powbg -class \"POW Magstep\"" 
    bind ${powDWP}magstep <<CloseWindow>> "destroy ${powDWP}"

    set powXMagstep $powPlotParam(xmagstep,$currgn) 
    set powYMagstep $powPlotParam(ymagstep,$currgn) 
    set powSaveXMagstep $powXMagstep
    set powSaveYMagstep $powYMagstep

    label ${powDWP}magstep.label -text "Current magstep:" -bg $powbg
    label ${powDWP}magstep.xlabel -text "X " -bg $powbg
    label ${powDWP}magstep.ylabel -text "Y " -bg $powbg
    entry ${powDWP}magstep.xmagstep -textvariable powXMagstep \
	    -relief sunken -bg $powbg
    entry ${powDWP}magstep.ymagstep -textvariable powYMagstep \
	    -relief sunken -bg $powbg
    
    frame ${powDWP}magstep.buttonFrame -bg $powbg
    button ${powDWP}magstep.apply -text Apply -command \
	{powMagGraph $currgn $powXMagstep $powYMagstep; \
	     set powXMagstep $powPlotParam(xmagstep,$currgn); \
	     set powYMagstep $powPlotParam(ymagstep,$currgn)}  -bg $powbg
    button ${powDWP}magstep.reset -text Reset -command \
	{set powXMagstep $powSaveXMagstep; \
	 set powYMagstep $powSaveYMagstep; \
	 powMagGraph $currgn $powXMagstep $powYMagstep}  -bg $powbg
    button ${powDWP}magstep.exit -text Exit -command {destroy ${powDWP}magstep}  -bg $powbg

    grid configure ${powDWP}magstep.label -row 0 -column 0 -sticky w
    grid configure ${powDWP}magstep.xlabel -row 0 -column 1
    grid configure ${powDWP}magstep.ylabel -row 1 -column 1
    grid configure ${powDWP}magstep.xmagstep -row 0 -column 2 -sticky w
    grid configure ${powDWP}magstep.ymagstep -row 1 -column 2 -sticky w
    grid configure ${powDWP}magstep.buttonFrame -row 2 -column 0 -columnspan 3 -sticky ew
    grid configure ${powDWP}magstep.apply -row 0 -column 0 -in ${powDWP}magstep.buttonFrame -sticky w
    grid configure ${powDWP}magstep.reset -row 0 -column 1 -in ${powDWP}magstep.buttonFrame
    grid configure ${powDWP}magstep.exit -row 0 -column 2 -in ${powDWP}magstep.buttonFrame -sticky e
}


proc powSetGraphSize { } {
    global powPlotParam currgn powbg powFrameForTop
    global powXDim powYDim powSaveXDim powSaveYDim powDWP

    if { $currgn=="powDef" } {
	error "You must first select a graph"
	return
    }
    if { [winfo exists ${powDWP}dim] } {
	raise ${powDWP}dim
	return
    }
    powToplevel ${powDWP}dim .pow "-bg $powbg -class \"POW Dim\"" 
    bind ${powDWP}dim <<CloseWindow>> "destroy ${powDWP}"

    if {!$powFrameForTop} {
	wm title ${powDWP}dim "Set Graph Dimensions"
    }

    set powXDim [tagXdim .pow.pow ${currgn}box]
    set powYDim [tagYdim .pow.pow ${currgn}box]
    set powSaveXDim $powXDim
    set powSaveYDim $powYDim

    label ${powDWP}dim.label -text "Current Dim:" -bg $powbg
    label ${powDWP}dim.xlabel -text "X " -bg $powbg
    label ${powDWP}dim.ylabel -text "Y " -bg $powbg
    entry ${powDWP}dim.xDim -textvariable powXDim \
	    -relief sunken -bg $powbg -width 10
    entry ${powDWP}dim.yDim -textvariable powYDim \
	    -relief sunken -bg $powbg -width 10
    
    frame ${powDWP}dim.buttonFrame -bg $powbg
    button ${powDWP}dim.apply -text Apply -command \
	{powStretchGraphToSize $currgn $powXDim $powYDim}  -bg $powbg
    button ${powDWP}dim.reset -text Reset -command \
	{set powXDim $powSaveXDim; \
	 set powYDim $powSaveYDim; \
	 powStretchGraphToSize $currgn $powXDim $powYDim}  -bg $powbg
    button ${powDWP}dim.exit -text Exit -command {destroy ${powDWP}dim}  -bg $powbg

    grid configure ${powDWP}dim.label -row 0 -column 0 -sticky w
    grid configure ${powDWP}dim.xlabel -row 0 -column 1
    grid configure ${powDWP}dim.ylabel -row 1 -column 1
    grid configure ${powDWP}dim.xDim -row 0 -column 2 -sticky w
    grid configure ${powDWP}dim.yDim -row 1 -column 2 -sticky w
    grid configure ${powDWP}dim.buttonFrame -row 2 -column 0 -columnspan 3 -sticky ew
    grid configure ${powDWP}dim.apply -row 0 -column 0 -in ${powDWP}dim.buttonFrame -sticky w
    grid configure ${powDWP}dim.reset -row 0 -column 1 -in ${powDWP}dim.buttonFrame
    grid configure ${powDWP}dim.exit -row 0 -column 2 -in ${powDWP}dim.buttonFrame -sticky e
}


proc powPrintBox { } {
    global powbg pcom fname pcom_fname powOutputType pcom_fname powPostOrient
    global powHandles powDWP
    powToplevel ${powDWP}print .pow "-bg $powbg -class \"POW Print\""
    bind ${powDWP}print <<CloseWindow>> "destroy ${powDWP}"
    

    set powOutputType "Printer Command:"
#orientation is 0-portrait 1-landscape
    set powPostOrient 0
    set pcom_fname "lpr"
    set pcom "lpr"
    set fname "pow.ps"
    set powHandles 0
    
    

    
    
	frame ${powDWP}print.frame -bg $powbg

	label ${powDWP}print.olabel \
		-anchor e \
		-text Output: \
                -width 16 \
                -bg $powbg

	radiobutton ${powDWP}print.pbutton \
		-text Printer \
                -value {Printer Command:} \
		-variable powOutputType \
                -command {set pcom_fname $pcom} \
                -bg $powbg

	radiobutton ${powDWP}print.fbutton \
		-text File \
		-value {File Name:} \
		-variable powOutputType \
                -command {set pcom_fname $fname} \
                -bg $powbg

	button ${powDWP}print.printbutton \
	-command {destroy ${powDWP}print; update idletasks; powPrint $powOutputType $pcom_fname $powPostOrient;} \
		-text Print/Save \
                -bg $powbg

	button ${powDWP}print.exitbutton \
                -command {destroy ${powDWP}print; powShowHandles 1} \
		-text Exit \
                -bg $powbg

	label ${powDWP}print.ocomlabel \
                -anchor e \
		-textvariable powOutputType \
	        -width 16   \
                -bg $powbg

	entry ${powDWP}print.pcomentry \
		-textvariable pcom_fname \
                -bg $powbg

	label ${powDWP}print.handlelabel \
		-anchor e \
		-text Orientation: \
	        -width 16   \
                -bg $powbg

        powShowHandles $powHandles

	radiobutton ${powDWP}print.showbutton \
		-text "Show Handles" \
		-value 1 \
		-variable powHandles \
                -command {powShowHandles 1} \
                -bg $powbg

	radiobutton ${powDWP}print.hidebutton \
		-text "Hide Handles" \
		-value 0 \
		-variable powHandles \
                -command {powShowHandles 0} \
                -bg $powbg

	label ${powDWP}print.orlabel \
		-anchor e \
		-text Orientation: \
	        -width 16   \
                -bg $powbg


	radiobutton ${powDWP}print.porbutton \
		-text Portrait \
		-value 0 \
		-variable powPostOrient \
                -bg $powbg

	radiobutton ${powDWP}print.landbutton \
		-text Landscape \
		-value 1 \
		-variable powPostOrient \
                -bg $powbg


	# Geometry management

	grid ${powDWP}print.frame -in ${powDWP}print	-row 5 -column 1  \
		-columnspan 3
	grid ${powDWP}print.olabel -in ${powDWP}print	-row 1 -column 1  \
		-sticky ew
	grid ${powDWP}print.pbutton -in ${powDWP}print	-row 1 -column 2 
	grid ${powDWP}print.fbutton -in ${powDWP}print	-row 1 -column 3 
	grid ${powDWP}print.printbutton -in ${powDWP}print.frame	-row 1 -column 1 
	grid ${powDWP}print.exitbutton -in ${powDWP}print.frame	-row 1 -column 3 
	grid ${powDWP}print.ocomlabel -in ${powDWP}print	-row 2 -column 1  \
		-sticky ew
	grid ${powDWP}print.pcomentry -in ${powDWP}print	-row 2 -column 2  \
		-columnspan 2
	grid ${powDWP}print.handlelabel -in ${powDWP}print -row 3 -column 1  \
		-sticky ew
	grid ${powDWP}print.showbutton -in ${powDWP}print	-row 3 -column 2 
	grid ${powDWP}print.hidebutton -in ${powDWP}print	-row 3 -column 3 
	grid ${powDWP}print.orlabel -in ${powDWP}print	-row 4 -column 1  \
		-sticky ew
	grid ${powDWP}print.porbutton -in ${powDWP}print	-row 4 -column 2 
	grid ${powDWP}print.landbutton -in ${powDWP}print	-row 4 -column 3 

	# Resize behavior management

	grid rowconfigure ${powDWP}print 1 -weight 0 -minsize 30
	grid rowconfigure ${powDWP}print 2 -weight 0 -minsize 30
	grid rowconfigure ${powDWP}print 3 -weight 0 -minsize 30
	grid rowconfigure ${powDWP}print 4 -weight 0 -minsize 50
	grid columnconfigure ${powDWP}print 1 -weight 0 -minsize 30
	grid columnconfigure ${powDWP}print 2 -weight 0 -minsize 30
	grid columnconfigure ${powDWP}print 3 -weight 0 -minsize 15

	grid rowconfigure ${powDWP}print.frame 1 -weight 0 -minsize 30
	grid columnconfigure ${powDWP}print.frame 1 -weight 0 -minsize 30
	grid columnconfigure ${powDWP}print.frame 2 -weight 1 -minsize 30
	grid columnconfigure ${powDWP}print.frame 3 -weight 0 -minsize 96

}


proc powMergeGraphs { newgn } {
    global currgn powPlotParam powGUI

    powPlotImages $currgn $powPlotParam(images,$newgn) .pow.pow
    powPlotCurves $currgn $powPlotParam(curves,$newgn) .pow.pow
    if { $powGUI } { powRedrawScopebox }
}


proc powEditNoteDlg { gn idx {id ""} } {
   global powPlotParam powFontParam powNote powDWP powbg

   if { $idx=="" } {
      set idx [powFindNoteIdx $gn $id]
   }
   powSetupNoteVar $gn $idx

   #
   #  Build Dialog Window
   #

   set w ${powDWP}note
   if { [winfo exists $w] } {
      raise $w
      focus $w
      if { $powNote(idx)<0 } {
         $w.buttons.apply  config -state disabled
         $w.buttons.delete config -state disabled
      } else {
         $w.buttons.apply  config -state normal
         $w.buttons.delete config -state normal
      }
      return
   }
   powToplevel $w .pow "-bg $powbg"
   bind $w <<CloseWindow>> "destroy $w"
   wm title $w "Annotations"

   set row 1
   
   label $w.title -text "Edit/Add Annotations" -bg $powbg
   grid $w.title -row $row -column 1 -columnspan 2 -sticky n
   incr row

   grid rowconfigure $w $row -minsize 10
   incr row

   label $w.lbl  -text "Label:"     -bg $powbg
   entry $w.lblentry -width 30 -bg $powbg -textvariable powNote(string)
   grid $w.lbl      -row $row -column 1 -sticky e
   grid $w.lblentry -row $row -column 2 -sticky ew -padx 5
   incr row

   grid rowconfigure $w $row -minsize 3
   incr row

   label $w.fnt  -text "Font:"      -bg $powbg
   frame $w.fntframe -bg $powbg

   set mnu [eval tk_optionMenu $w.fntframe.fnt \
         powNote(Font) $powFontParam(allFonts,powDef)]
   $w.fntframe.fnt config -bg $powbg -highlightthickness 0 -width 20
   $mnu config -bg $powbg

   set mnu [tk_optionMenu $w.fntframe.siz \
         powNote(Size) 7 9 12 14 16 18 24 32 40]
   $w.fntframe.siz config -bg $powbg -highlightthickness 0 -width 3
   $mnu config -bg $powbg

   pack $w.fntframe.fnt -side left -padx 5
   pack $w.fntframe.siz -side left -padx 5
   grid $w.fnt       -row $row -column 1 -sticky e
   grid $w.fntframe  -row $row -column 2 -sticky w
   incr row

   grid rowconfigure $w $row -minsize 3
   incr row

   label $w.stl  -text "Style:"     -bg $powbg
   frame $w.stlframe -bg $powbg
   checkbutton $w.stlframe.bld -text Bold   -onvalue bold   -offvalue normal \
         -bg $powbg -variable powNote(Weight)  -highlightthickness 0
   checkbutton $w.stlframe.itl -text Italic -onvalue italic -offvalue roman \
         -bg $powbg -variable powNote(Slant)   -highlightthickness 0
   pack $w.stlframe.bld -side left -padx 5
   pack $w.stlframe.itl -side left -padx 5
   grid $w.stl      -row $row -column 1 -sticky e
   grid $w.stlframe -row $row -column 2 -sticky w
   incr row

   grid rowconfigure $w $row -minsize 3
   incr row

   label $w.clr -text "Color:" -bg $powbg
   button $w.clrbtn -textvariable powNote(Color) \
         -bg $powbg -highlightthickness 0 -width 7 \
         -command "powSelectColor powNote(Color)"
   grid $w.clr      -row $row -column 1 -sticky e
   grid $w.clrbtn   -row $row -column 2 -sticky w -padx 5
   incr row

   grid rowconfigure $w $row -minsize 3
   incr row

   label $w.pos  -text "Position:"   -bg $powbg
   frame $w.posframe -bg $powbg
   entry $w.posframe.x -width 14 -bg $powbg -textvariable powNote(xpos)
   entry $w.posframe.y -width 14 -bg $powbg -textvariable powNote(ypos)
   pack $w.posframe.x -side left -padx 5
   pack $w.posframe.y -side left -padx 5
   grid $w.pos      -row $row -column 1 -sticky e
   grid $w.posframe -row $row -column 2 -sticky w
   incr row
   
   grid rowconfigure $w $row -minsize 3
   incr row

   label $w.crd  -text "Attach To:" -bg $powbg
   frame $w.crdframe -bg $powbg
   radiobutton $w.crdframe.graph -text "Graph" \
         -variable powNote(coordSys) \
         -value "graph" -highlightthickness 0 -bg $powbg \
         -command powUpdateNoteCoord
   radiobutton $w.crdframe.coord -text "Coordinates" \
         -variable powNote(coordSys) \
         -value "coord" -highlightthickness 0 -bg $powbg \
         -command powUpdateNoteCoord
   pack $w.crdframe.graph -side left -padx 5
   pack $w.crdframe.coord -side left -padx 5
   grid $w.crd      -row $row -column 1 -sticky e
   grid $w.crdframe -row $row -column 2 -sticky w
   incr row

   grid rowconfigure $w $row -minsize 10
   incr row

   frame $w.buttons -bg $powbg
   button $w.buttons.apply  -text "Apply"  -bg $powbg -highlightthickness 0 \
         -command {powUpdateNote apply}
   button $w.buttons.add    -text "Add"    -bg $powbg -highlightthickness 0 \
         -command {powUpdateNote add}
   button $w.buttons.delete -text "Delete" -bg $powbg -highlightthickness 0 \
         -command {powUpdateNote delete}
   button $w.buttons.done  -text "Exit"    -bg $powbg -highlightthickness 0 \
         -command "destroy $w"
   pack $w.buttons.add    -side left -padx 5
   pack $w.buttons.delete -side left -padx 5
   pack $w.buttons.apply  -side left -padx 5
   pack $w.buttons.done   -side left -padx 5
   grid $w.buttons  -row $row -column 1 -columnspan 2
   incr row

   grid rowconfigure $w $row -minsize 5
   incr row

   if { $powNote(idx)<0 } {
      $w.buttons.apply  config -state disabled
      $w.buttons.delete config -state disabled
   } else {
      $w.buttons.apply  config -state normal
      $w.buttons.delete config -state normal
   }
}


proc powSetupNoteVar { gn idx } {
   global powNote powFontParam powPlotParam

   set powNote(gn) $gn
   if { $idx<0 && [llength $powPlotParam(Notes,$gn)]==0 } {
      # This is a new record
      # Grab font info from graph defaults

      foreach opt $powFontParam(allOpts,powDef) {
         set powNote($opt) $powFontParam(note${opt},$gn)
      }
      set powNote(xpos)     1.01
      set powNote(ypos)     0.9
      set powNote(string)   "Blank"
      set powNote(coordSys) "graph"

   } else {

      if { $idx<0 || $idx >= [llength $powPlotParam(Notes,$gn)] } {
         # Grab font info from last note
         set record [lindex $powPlotParam(Notes,$gn) end]
         set record [lreplace $record 0 0 "Blank"]
         set record [lreplace $record end end -1]
         set idx -1
      } else {
         set record [lindex $powPlotParam(Notes,$gn) $idx]
      }

      foreach [list string Font Size Weight Slant Color xpos ypos coordSys id] \
            $record {}
      foreach opt \
            [list string Font Size Weight Slant Color xpos ypos coordSys id] {
         set powNote($opt) [subst \$$opt]
      }
   }
   set powNote(idx) $idx
   set powNote(oldCoord) $powNote(coordSys)
}


proc powUpdateNote { method } {
   global powNote powPlotParam powDWP

   set gn $powNote(gn)
   if { $method=="delete" } {

      # Delete this Note

      if { $powNote(idx)>=0 } {
         powDeleteNote $gn $powNote(idx)
         set powNote(idx) -1
      }

   } elseif { $powNote(string)=="" } {

      error "Cannot Add/Apply an empty label"

   } else {

      # Update the Note

      set record {}
      foreach opt [list \
            string Font Size Weight Slant Color xpos ypos coordSys] {
         lappend record $powNote($opt)
      }

      if { $method=="add" || $powNote(idx)<0 } {

         # Create a new Note

         set powNote(idx) [llength $powPlotParam(Notes,$gn)]
         lappend record -1
         lappend powPlotParam(Notes,$gn) $record

      } else {  # Apply

         # Grab current ID number of the current Note

         set r [lindex $powPlotParam(Notes,$gn) $powNote(idx)]
         lappend record [lindex $r 9]

      }

      set newID [powDrawNote $gn $record]
      set record [lreplace $record 9 9 $newID]
      set powPlotParam(Notes,$gn) [lreplace \
            $powPlotParam(Notes,$gn) $powNote(idx) $powNote(idx) $record]

      powRedrawGraphHandles $gn
   }

   if { $powNote(idx)<0 } {
      ${powDWP}note.buttons.apply  config -state disabled
      ${powDWP}note.buttons.delete config -state disabled
   } else {
      ${powDWP}note.buttons.apply  config -state normal
      ${powDWP}note.buttons.delete config -state normal
   }
}


proc powUpdateNoteCoord { } {
   global powNote powPlotParam

   set gn   $powNote(gn)
   set xpos $powNote(xpos)
   set ypos $powNote(ypos)

   foreach [list x0 y1 x1 y0] [.pow.pow coord ${gn}box] {}

   if { $powNote(oldCoord)==$powNote(coordSys) } {
   
      set idx $powNote(idx)
      if { $idx>=0 } {
         set r [lindex $powPlotParam(Notes,$gn) $idx]
         set xpos [lindex $r 6]
         set ypos [lindex $r 7]
         set crd  [lindex $r 8]
         if { $crd==$powNote(coordSys) } {
            set powNote(xpos) $xpos
            set powNote(ypos) $ypos
            return
         }
      } else {
         return
      }

   }

   if { $powNote(coordSys)=="graph" } {

      foreach {x y} [powGraphToCanvas $gn $xpos $ypos .pow.pow] {}
      set xpos [expr ($x - $x0) / ($x1 - $x0)]
      set ypos [expr ($y - $y0) / ($y1 - $y0)]

   } elseif { $powNote(coordSys)=="coord" } {

      set x [expr $xpos * ($x1-$x0) + $x0]
      set y [expr $ypos * ($y1-$y0) + $y0]
      foreach {xpos ypos} [powCanvasToGraph $gn $x $y .pow.pow] {}

   }      
   set powNote(xpos) $xpos
   set powNote(ypos) $ypos
   set powNote(oldCoord) $powNote(coordSys)
}


proc powAddNote { gn xpos ypos string args } {
   global powPlotParam powFontParam

   array set opts [list \
         -font   $powFontParam(noteFont,powDef) \
         -size   $powFontParam(noteSize,powDef) \
         -weight $powFontParam(noteWeight,powDef) \
         -slant  $powFontParam(noteSlant,powDef) \
         -color  $powFontParam(noteColor,powDef) \
         -coord  graph \
         ]

   foreach {opt val} $args {
      if { [info exists opts($opt)] } {
         set opts($opt) $val
      } else {
         puts "Unrecognized option $opt"
      }
   }

   set record [list $string $opts(-font) $opts(-size) $opts(-weight) \
         $opts(-slant) $opts(-color) $xpos $ypos $opts(-coord) -1]
   set newID [powDrawNote $gn $record]
   set record [lreplace $record end end $newID]
   lappend powPlotParam(Notes,$gn) $record

   powRedrawGraphHandles $gn

   return $newID
}


proc powDeleteNote { gn idx } {
   global powPlotParam

   set r [lindex $powPlotParam(Notes,$gn) $idx]
   set id [lindex $r 9]
   if { $id>=0 && [.pow.pow find withtag $id]!="" } {
      .pow.pow delete $id
   }
   set powPlotParam(Notes,$gn) \
         [lreplace $powPlotParam(Notes,$gn) $idx $idx]

   powRedrawGraphHandles $gn
}


proc powDrawNote { gn record } {

   foreach [list string Font Size Weight Slant Color xpos ypos coordSys id] \
         $record {}

   foreach [list x0 y1 x1 y0] [.pow.pow coord ${gn}box] {}

   if { $coordSys=="graph" } {

      set x [expr $xpos * ($x1-$x0) + $x0]
      set y [expr $ypos * ($y1-$y0) + $y0]

   } elseif { $coordSys=="coord" } {

      foreach {x y} [powGraphToCanvas $gn $xpos $ypos .pow.pow] {}
      if { $x<$x0 || $x>$x1 || $y<$y1 || $y>$y0 } {
         # Note not inside graph box
         if { $id>=0 && [.pow.pow find withtag $id]!="" } {
            .pow.pow delete $id
         }
         return -1
      }
      
   } else {
      puts "Unsupported coordSys $coordSys"
      return -1
   }

   if { $id>=0 && [.pow.pow find withtag $id]!="" } {

      .pow.pow coords $id $x $y
      .pow.pow itemconfig $id -text $string -fill $Color \
            -font [list $Font $Size $Weight $Slant] -anchor sw \
            -tags "$gn ${gn}text"

   } else {

      set id [.pow.pow create text $x $y -text $string -fill $Color \
            -font [list $Font $Size $Weight $Slant] -anchor sw \
            -tags "$gn ${gn}text"]

      .pow.pow bind $id <<DblBtnPress>> "powEditNoteDlg $gn {} $id"
      powBindBtn <<Drag>> ".pow.pow bind $id" \
            " powDragNote start $gn $id %X %Y " \
            " powDragNote drag  $gn $id %X %Y " \
            " powDragNote end   $gn $id %X %Y "

   }

   return $id
}


proc powDragNote { mode gn id X Y } {
   global powMoveX powMoveY powIsDragging powResizeMain powNote powDWP

   switch -exact $mode {
      start {
         set powIsDragging 1
      }
      drag {
         set dx [expr $X - $powMoveX]
         set dy [expr $Y - $powMoveY]
         .pow.pow move $id $dx $dy
      }
      end {
         set idx [powFindNoteIdx $gn $id]
         powRepositionNote $gn $idx
         powRedrawGraphHandles $gn
         set powIsDragging 0
         if { [winfo exist ${powDWP}note] } {
            if { $gn==$powNote(gn) && $idx==$powNote(idx) } {
               powUpdateNoteCoord
            } else {
               powEditNoteDlg $gn {} $id
            }
         }
      }
   }
   set powMoveX $X
   set powMoveY $Y
}


proc powRepositionNote { gn idx } {
   # Calculate the appropriate x/y position for
   # the note's current canvas position

   global powPlotParam

   if { $idx<0 || $idx>[llength $powPlotParam(Notes,$gn)] } return

   set r     [lindex $powPlotParam(Notes,$gn) $idx]
   set coord [lindex $r 8]
   set id    [lindex $r 9]
   foreach {x y}         [.pow.pow coord $id] {}
   foreach {x0 y1 x1 y0} [.pow.pow coord ${gn}box] {}

   if { $coord=="graph" } {

      set xpos [expr ($x - $x0) / ($x1 - $x0)]
      set ypos [expr ($y - $y0) / ($y1 - $y0)]

   } elseif { $coord=="coord" } {

      foreach {xpos ypos} [powCanvasToGraph $gn $x $y .pow.pow] {}
      if { $x<$x0 || $x>$x1 || $y<$y1 || $y>$y0 } {
         # Note not inside graph box
         if { $id>=0 && [.pow.pow find withtag $id]!="" } {
            .pow.pow delete $id
         }
         set r [lreplace $r 9 9 -1]
      }

   } else {

      error "Unknown coordinate system: $coord"

   }

   set r [lreplace $r 6 7 $xpos $ypos]
   set powPlotParam(Notes,$gn) [lreplace $powPlotParam(Notes,$gn)\
         $idx $idx $r]
}


proc powFindNoteIdx { gn id } {
   global powPlotParam

   set idx 0
   foreach r $powPlotParam(Notes,$gn) {
      if { [lindex $r end]==$id } break
      incr idx
   }
   if { $idx >= [llength $powPlotParam(Notes,$gn)] } {
      set idx -1
   }
   return $idx
}


proc powRedrawNotes { gn } {
   global powPlotParam

   set i 0
   set notes {}
   foreach r $powPlotParam(Notes,$gn) {
      set newID [powDrawNote $gn $r]
      set r [lreplace $r end end $newID]
      lappend notes $r
   }
   set powPlotParam(Notes,$gn) $notes
}


proc powAddTextToGraphDoIt { } {
   global powDWP
   if [catch {selection get} gn] {
      set gn " "
   }
   set savebinding [bind .pow.pow <ButtonPress-1>]
   bind .pow.pow <ButtonPress-1> "\
         powPlaceText [list [${powDWP}addtext.text get 0.0 end] $gn %x %y .pow.pow]; destroy ${powDWP}addtext;\
         bind .pow.pow <ButtonPress-1> \{$savebinding\}"
}


proc powSelectColor { varName } {
   upvar #0 $varName var
   set newClr [tk_chooseColor -initialcolor $var]
   if {$newClr != ""} {set var $newClr}
}


proc swap { a b} {
    upvar $a one
    upvar $b two
    set tmp $one
    set one  $two
    set two $tmp
}

 
proc debug_trace {name element op} { if {$element != ""} { set name
    ${name}($element) } upvar $name x puts "Variable $name set to $x"
}


proc powUpdateTrackVars {} { 
    global powGraphCoordsTracker powImagePixelTracker powImageValueTracker
    global powTrackText powFirstPixel powPlotParam

    set gn $powTrackText(gn)
    if [regexp {[0-9]} $powTrackText(rx)] {
	if {$powPlotParam(tickLabels,$gn)=="degrees" \
	      && [powWCSexists $gn]} {
	    set rx  [powHourRA $powTrackText(rx)]
	    set ry  [powDegDec $powTrackText(ry)]
	    set powGraphCoordsTracker \
		"Graph coordinates:\n   ( $rx, $ry )"
	} else {

	   set rxVal $powTrackText(rx)
	   if {$powPlotParam(xTickScal,$gn)=="log"} {
	      # Make sure the log value isn't out-of-bounds
	      if { $rxVal>300 || $rxVal<-300 } {
		 set rxVal "***"
	      } else {
		 set rxVal [format "%.6lg" [expr pow(10.0,$rxVal)]]
	      }
	   } else {
	      set rxVal [format "%.6lg" $rxVal]
	   }

	   set ryVal $powTrackText(ry)
	   if {$powPlotParam(yTickScal,$gn)=="log"} {
	      # Make sure the log value isn't out-of-bounds
	      if { $ryVal>300 || $ryVal<-300 } {
		 set ryVal "***"
	      } else {
		 set ryVal [format "%.6lg" [expr pow(10.0,$ryVal)]]
	      }
	   } else {
	      set ryVal [format "%.6lg" $ryVal]
	   }

	   set powGraphCoordsTracker "Graph coordinates:\n   ( $rxVal, $ryVal )"
	}
    } else {
	set powGraphCoordsTracker "Graph coordinates:\n   ( X , X )"
    }


    if [regexp {[0-9]} $powTrackText(imgx)] {
	set powImagePixelTracker "Image pixel:\n   ( [expr $powTrackText(imgx) + $powFirstPixel], [expr $powTrackText(imgy) + $powFirstPixel] )"
    } else {
	set powImagePixelTracker "Image pixel:\n   ( X , X )"
    }
	
    if [regexp {[0-9]} $powTrackText(imgz)] {
	set pixval [format %.16lg [expr $powTrackText(imgz)]]
    } elseif { $powTrackText(imgz)=="NULL" } {
       set pixval "NULL"
    } else {
	set pixval "X"
    }
    set powImageValueTracker "Pixel value:\n    $pixval ($powTrackText(zunits))"

}



proc powHelp { topic } {
   global env tcl_platform powHelpTopics powbg powDWP

   if { [string match "*.html" $topic] } {
      set topic [string range $topic 0 end-5]
   }

   if { [winfo exist ${powDWP}hyperhelp] == 0} {

      if { $tcl_platform(platform)=="windows"  } {
         set size large
      } else {
         set size medium
      }
      set allTopics {}
      foreach aTopic [lsort [array names powHelpTopics]] {
         lappend allTopics [list $powHelpTopics($aTopic) $aTopic]
      }

      iwidgets::hyperhelp ${powDWP}hyperhelp -title "POW: Hyperhelp" \
          -topics      $allTopics \
          -fontname    courier \
          -fontsize    $size \
          -helpdir     $env(POW_HELPDIR) \
          -background  $powbg \
          -textbackground  $powbg
#        -helpdir     $env(POW_LIBRARY)
   }
   ${powDWP}hyperhelp showtopic $topic
   ${powDWP}hyperhelp activate
}


proc powMax { a b } {
    return [expr ($a > $b) ? $a : $b]
}

proc powMin { a b } {
    return [expr ($a < $b) ? $a : $b]
}

proc powFindFont { w {pointsizes 120} {weight medium} {slant r}} {
    foreach family {times courier helvetica } {
	foreach points $pointsizes {
	    if {[catch {$w config -font \
			    -*-$family-$weight-$slant-*-*-*-$points-*}] == 0} {
		return -*-$family-$weight-$slant-*-*-*-$points-*
	    }
	}
    }
    $w config -font fixed
    return fixed
}

proc powGetFontList { gn lbl } {
    global powFontParam
    
    return [list \
            $powFontParam(${lbl}Font,$gn) \
            $powFontParam(${lbl}Size,$gn) \
            $powFontParam(${lbl}Weight,$gn) \
            $powFontParam(${lbl}Slant,$gn)]
}
      

proc powScopeZoom { in_or_out } {
    global currgn saveROI

    if [regexp "in" $in_or_out] {
	set multiplier 0.5
    } else {
	set multiplier 2.0
    }


    if {! [string compare [.pow.scope find  withtag ROI] ""] || ![winfo ismapped .pow.scope]} {
	# no ROI currently, ROI is whole graph OR no scopebox at all
	set ROIbbox [.pow.pow coords ${currgn}box]
	set halfwidth [expr ([lindex $ROIbbox 2] - [lindex $ROIbbox 0])/2.0]
	set halfheight [expr ([lindex $ROIbbox 3] - [lindex $ROIbbox 1])/2.0]
	set x_center [expr [lindex $ROIbbox 0] + $halfwidth]
	set y_center [expr [lindex $ROIbbox 1] + $halfheight]
	set new_halfwidth [expr $halfwidth * $multiplier]
	set new_halfheight [expr $halfheight * $multiplier]
	.pow.pow create rectangle \
	    [expr $x_center - $new_halfwidth]  \
	    [expr $y_center - $new_halfheight] \
	    [expr $x_center + $new_halfwidth]  \
	    [expr $y_center + $new_halfheight] \
	    -tags ROI -outline blue
	powEndROI 0 .pow.pow
	return
    } else {
	set ROIbbox [.pow.scope coords ROI]
	set saveROI $ROIbbox
	.pow.scope delete ROI
    }
    set halfwidth [expr ([lindex $ROIbbox 2] - [lindex $ROIbbox 0])/2.0]
    set halfheight [expr ([lindex $ROIbbox 3] - [lindex $ROIbbox 1])/2.0]
    set x_center [expr [lindex $ROIbbox 0] + $halfwidth]
    set y_center [expr [lindex $ROIbbox 1] + $halfheight]
    set new_halfwidth [expr $halfwidth * $multiplier]
    set new_halfheight [expr $halfheight * $multiplier]
    .pow.scope create rectangle \
          [expr $x_center - $new_halfwidth]  \
          [expr $y_center - $new_halfheight] \
          [expr $x_center + $new_halfwidth]  \
          [expr $y_center + $new_halfheight] \
          -tags ROI -outline blue
    powEndROI 0 .pow.scope
}

proc powGetCurrVariables {} {
   global powPlotParam currgn currimg
   global powCurveParam
   global powImageParam
   global powFontParam

   if { ![info exists currgn] || $currgn=="powDef" } return

   foreach opt $powPlotParam(allOpts,powDef) {
      set powPlotParam(${opt},powDef) $powPlotParam(${opt},$currgn)
   }

   if { [info exists currimg] && $currimg != "" } {
      foreach opt $powImageParam(allOpts,powDef) {
	 set powImageParam(${opt},powDef) \
	       $powImageParam(${opt}${currimg},$currgn)
      }
   }

   set crv [lindex $powPlotParam(curves,$currgn) 0]
   if { $crv != "NULL" } {
      foreach opt $powCurveParam(allOpts,powDef) {
	 set powCurveParam(${opt},powDef) \
	       $powCurveParam(${opt}${crv},$currgn)
      }
   }

   foreach lbl $powFontParam(allTypes,powDef) {
      foreach opt $powFontParam(allOpts,powDef) {
         set powFontParam(${lbl}${opt},powDef) \
               $powFontParam(${lbl}${opt},$currgn)
      }
   }

}

proc powSaveConfig { } {
    global powbg powCurveParam powImageParam powFontParam
    global powcursor powResizeMain currgn
    global showlinks powScopeHeight powScopeWidth powMinHeight powMinWidth
    global powPlotParam powShowScope powGUIposition
    global POWRC

    if [catch {open $POWRC w} RCFILE] {
	error "Couldn't open $POWRC, not saving configuration"
    } else {
        puts $RCFILE "\n# Application parameters:"
	puts $RCFILE "set powbg \"$powbg\""
	puts $RCFILE "set powcursor \"$powcursor\""
	puts $RCFILE "set powResizeMain \"$powResizeMain\""
	puts $RCFILE "set showlinks \"$showlinks\""
        if { $powShowScope } {
	   puts $RCFILE "set powScopeWidth \"$powScopeWidth\""
	   puts $RCFILE "set powScopeHeight \"$powScopeHeight\""
	} else {
	   puts $RCFILE "set powScopeWidth \"0\""
	   puts $RCFILE "set powScopeHeight \"0\""
	}
	puts $RCFILE "set powMinHeight \"$powMinHeight\""
	puts $RCFILE "set powMinWidth \"$powMinWidth\""
        puts $RCFILE "set powGUIposition \"$powGUIposition\""

        puts $RCFILE "\n# Default Font Parameters:"

        foreach lbl $powFontParam(allTypes,powDef) {
           foreach opt $powFontParam(allOpts,powDef) {
              puts $RCFILE "set powFontParam(${lbl}${opt},powDef)\
                    \"$powFontParam(${lbl}${opt},powDef)\""
           }
        }

        puts $RCFILE "\n# Default Graph Parameters:"
	foreach opt $powPlotParam(allOpts,powDef) {
	    puts $RCFILE "set powPlotParam($opt,powDef)\
		  \"$powPlotParam($opt,powDef)\""
	}

        puts $RCFILE "\n# Default Curve Parameters:"
	foreach opt $powCurveParam(allOpts,powDef) {
	    puts $RCFILE "set powCurveParam($opt,powDef) \"$powCurveParam($opt,powDef)\""
	}

        puts $RCFILE "\n# Default Image Parameters:"
	foreach opt $powImageParam(allOpts,powDef) {
	    puts $RCFILE "set powImageParam($opt,powDef) \"$powImageParam($opt,powDef)\""
	}

	close $RCFILE
    }
}

proc powBreakAllLinks { } {
#deletes all link info
    global axisToChainHash chainToAxisHash nextchain
 
    if [info exists axisToChainHash] {
	unset axisToChainHash 
	unset chainToAxisHash 
	unset nextchain
    }
    .pow.pow delete link
}
    


proc powLinkAxes {gn1 axis1 gn2 axis2} {
    global axisToChainHash chainToAxisHash nextchain

#catch stupid input
    if {$gn1 == $gn2} {return}
    

    if {![info exists nextchain]} {
	set nextchain 1
    }
    set graphlist [powListGraphs] 

    if {[lsearch $graphlist $gn1] == -1} {return "graph $gn1 does not exist";}
    if {[lsearch $graphlist $gn2] == -1} {return "graph $gn2 does not exist";}

    set chain1 0
    set chain2 0
    
    if {[array names axisToChainHash $gn1$axis1] != ""} {set chain1 $axisToChainHash($gn1$axis1)}
    if {[array names axisToChainHash $gn2$axis2] != ""} {set chain2 $axisToChainHash($gn2$axis2)}
    if {$chain1 == 0 && $chain2 == 0} {
#new chain
	set axisToChainHash($gn1$axis1) $nextchain
	set axisToChainHash($gn2$axis2) $nextchain
	set chainToAxisHash($nextchain) [list $gn1$axis1 $gn2$axis2]
	incr nextchain
    } elseif {$chain1 != 0 && $chain2 !=0} {
#two chains, delete one
	powMergeChains $chain1 $chain2
    } elseif {$chain1 != 0} {
#add axis 2 to chain 1	
	set axisToChainHash($gn2$axis2) $chain1
	lappend chainToAxisHash($chain1) $gn2$axis2
    } else {
#add axis 1 to chain 2
	set axisToChainHash($gn1$axis1) $chain2
	lappend chainToAxisHash($chain2) $gn1$axis1
    }
}

proc powMergeChains {chain1 chain2} {
#delete two existing chains and make a new one consisting of all of their
#members    
    global axisToChainHash chainToAxisHash nextchain
    foreach axis [array names axisToChainHash] {
	if {$axisToChainHash($axis) == $chain1 || \
		$axisToChainHash($axis) == $chain2} {
	    set axisToChainHash($axis) $nextchain
	}
    }
    set chainToAxisHash($nextchain) [concat $chainToAxisHash($chain1) $chainToAxisHash($chain2)]
    unset chainToAxisHash($chain1)
    unset chainToAxisHash($chain2)
}
    
 
proc powBreakLink {gn axis} {
#removes graph 1 axis 1 from the chain it belongs to
    global axisToChainHash chainToAxisHash
    set chain 0
    if {[array names axisToChainHash $gn$axis] != ""} {
	set chain $axisToChainHash($gn$axis)
    } else {
	puts "Graph $gn axis $axis is not part of a chain"
	return
    }
    unset axisToChainHash($gn$axis)
    if {[llength $chainToAxisHash($chain)] <= 2} {
	unset chainToAxisHash($chain)
    } else {
	set chainToAxisHash($chain) [lreplace $chainToAxisHash($chain) [set bozo [lsearch $chainToAxisHash($chain) $gn$axis]] $bozo]
    }
}

proc chop {theString} {
    return [string range $theString 0 [expr [string length $theString]-2]]
}
    

proc powAlignChain {gn axis orient {gap default}} {
#stacks all graphs in a chain either (H)orizontally or (V)ertically
    global axisToChainHash chainToAxisHash powResizeMain powPlotParam   
    if {![info exists axisToChainHash($gn$axis)]} return;


    foreach graph $chainToAxisHash($axisToChainHash($gn$axis)) {
    #some (all?) graphs in a chain may be unmapped so loop until we get one
    #that is mapped
	set oldgraph [lindex $chainToAxisHash($axisToChainHash($gn$axis)) 0]
	set oldgraph [chop $oldgraph]
	if {[.pow.pow find withtag $oldgraph] != ""} break
    }
    foreach graph [lrange $chainToAxisHash($axisToChainHash($gn$axis)) 1 end] {
	set graph [chop $graph]
	if {[.pow.pow find withtag $graph] != ""} {
	    #good, this graph is mapped
	    set gbox [.pow.pow coords ${oldgraph}box]
	    set bbox [.pow.pow bbox ${oldgraph}]
	    if {[lindex $orient 0] == "H"} {
		if {$gap == "default"} {
		    set gap [expr 2 * $powPlotParam(xmargin,$graph)]
		}
		set toX [expr [lindex $bbox 2] \
			     + $gap ]
		set toY [lindex $gbox 1]
	    } else {
		if {$gap == "default"} {
		    set gap [expr 2 * $powPlotParam(ymargin,$graph)]
		}
		set toX [lindex $gbox 0]
		set toY [expr [lindex $bbox 3] + $gap ]
	    }
	    set coords [.pow.pow coords ${graph}box]
	    set fromX [lindex $coords 0] 
	    set fromY [lindex $coords 1] 
	    powMoveGraph $graph [expr int($toX - $fromX)] [expr int($toY - $fromY)]
	    set oldgraph $graph
	}
    }
    powReconfigureToplevel $powResizeMain
}


proc powReconfigureToplevel {{resizemain 1}  } {
    global powMinHeight powMinWidth powMaxWidth powMaxHeight
    global powRealMinHeight powRealMinWidth
    global powHeaderWidth powHeaderHeight
    global powContainer

#resize POW window if necessary
#    update idletasks
    set bigbbox [.pow.pow bbox all]
    if {$resizemain &&  ($powContainer == "none" || $powContainer == "NULL")} {
        set windowX [expr [lindex $bigbbox 2] - [lindex $bigbbox 0] + 50]
	set windowY [expr [lindex $bigbbox 3] - [lindex $bigbbox 1] + 50]
       
        set windowX [powMax $windowX $powMinWidth ]
        set windowY [powMax $windowY $powMinHeight]

        set windowX [powMin $windowX $powMaxWidth ]
        set windowY [powMin $windowY $powMaxHeight]

        incr windowX $powHeaderWidth
        incr windowY $powHeaderHeight

        set windowX [powMax $windowX $powRealMinWidth ]
        set windowY [powMax $windowY $powRealMinHeight]

        foreach {x y} [lrange [split [wm geometry .pow] {x+-}] 2 3] {}
	if {$x != 0 && $y != 0} {
	    catch {wm geometry .pow "${windowX}x${windowY}+$x+$y"}
	} else {
	    catch {wm geometry .pow "${windowX}x${windowY}"}
	}
    }

    # Check if scrollregion has significantly changed so that one doesn't
    # force a full-screen update

    set currBnds [.pow.pow cget -scrollregion]
    set newBnds [list [expr [lindex $bigbbox 0] - 20] \
	              [expr [lindex $bigbbox 1] - 20] \
	              [expr [lindex $bigbbox 2] + 20] \
	              [expr [lindex $bigbbox 3] + 20] ]

    if { [expr abs([lindex $currBnds 0]-[lindex $newBnds 0])]>10 \
	    || [expr abs([lindex $currBnds 1]-[lindex $newBnds 1])]>10 \
	    || [expr abs([lindex $currBnds 2]-[lindex $newBnds 2])]>10 \
	    || [expr abs([lindex $currBnds 3]-[lindex $newBnds 3])]>10 } {
	.pow.pow configure -scrollregion $newBnds
    }

    powShowLinks
}
    


proc powChangeBg {} {
    global powbg powShowHandlesFlag powPlotParam

    set oldpowbg [.pow.pow cget -background]

    foreach com [info commands .pow.*] {
	if {[$com cget -bg] == $oldpowbg} {
	    catch {$com configure  -bg $powbg}
	}
	if {[$com cget -background] == $oldpowbg} {
	    catch {$com configure -background $powbg}
	}
    }

#catch the next line in case no graphs yet    
    catch {.pow.currgn configure -background yellow}

    .pow configure -bg $powbg

    foreach gn [powListGraphs] {
	if {![regexp {scope$} $gn] && \
		     $powPlotParam(bgcolor,$gn) == $oldpowbg} {
	    powGraphOptions $gn bgcolor $powbg
	}
    }
    
}



proc powExit { }  {
    global axisToChainHash chainToAxisHash nextchain powGUI
    catch { unset axisToChainHash chainToAxisHash nextchain}
    powCleanUp
    event generate . <<powExit>> -when tail
    catch {destroy .pow}
}

proc powLogGraph { gn x y } {
   global powPlotParam powCurveParam

   # Cannot have log plots with WCS information
   if { [powWCSexists $gn] && ($x=="log" || $y=="log") } {
      error "Cannot apply log transforms to WCS graphs"
   }

   foreach bnd [list xBot yBot xTop yTop xTickScal yTickScal] {
      set $bnd $powPlotParam($bnd,$gn)
   }

   if { $powPlotParam(curves,$gn) != "NULL" } {

      # Try to preserve the bounding box region

      if { $x!=$xTickScal } {
	 if { $x=="log" } {
	    if { $xBot>0.0 } {
	       set xBot [expr log10($xBot)]
	    } else {
	       set xBot NULL
	    }
	    if { $xTop>0.0 } {
	       set xTop [expr log10($xTop)]
	    } else {
	       set xTop NULL
	    }
	 } else {
	    if { $xBot<-300 || $xBot>300 } {
	       set xBot NULL
	    } else {
	       set xBot [expr pow(10.0,$xBot)]
	    }
	    if { $xTop<-300 || $xTop>300 } {
	       set xTop NULL
	    } else {
	       set xTop [expr pow(10.0,$xTop)]
	    }
	 }
      }
      if { $y!=$yTickScal } {
	 if { $y=="log" } {
	    if { $yBot>0.0 } {
	       set yBot [expr log10($yBot)]
	    } else {
	       set yBot NULL
	    }
	    if { $yTop>0.0 } {
	       set yTop [expr log10($yTop)]
	    } else {
	       set yTop NULL
	    }
	 } else {
	    if { $yBot<-300 || $yBot>300 } {
	       set yBot NULL
	    } else {
	       set yBot [expr pow(10.0,$yBot)]
	    }
	    if { $yTop<-300 || $yTop>300 } {
	       set yTop NULL
	    } else {
	       set yTop [expr pow(10.0,$yTop)]
	    }
	 }
      }

      set powPlotParam(xBot,$gn) $xBot
      set powPlotParam(yBot,$gn) $yBot
      set powPlotParam(xTop,$gn) $xTop
      set powPlotParam(yTop,$gn) $yTop
   }

   set powPlotParam(xTickScal,$gn) $x
   set powPlotParam(yTickScal,$gn) $y

   if { $x=="log" } { set x Yes } else { set x No }
   if { $y=="log" } { set y Yes } else { set y No }
   foreach crv $powPlotParam(curves,$gn) {
      if { $crv == "NULL" } continue
      set powCurveParam(logX${crv},$gn) $x
      set powCurveParam(logY${crv},$gn) $y
   }
   powEraseGraph $gn 1
   powMapGraph $gn
#   powAdornGraph $gn .pow.pow
#   powRedrawGraphHandles $gn
}

proc powEraseGraph { gn {scope 1}} {
   global powGUI currgn

#Removes a graph from the display
   .pow.pow delete $gn
   .pow.pow delete ${gn}handles
   if { $currgn==$gn } {
      .pow.pow delete current_gn
      if {$powGUI && $scope} {
	 .pow.scope delete all
      }
   }
}

proc powDeleteGraph { gn } {

   set ans [tk_messageBox -icon warning -parent .pow -type yesno -default no \
         -message "This will permanantly delete the current graph.\nContinue?"]
   if { $ans == "no" } return

   powDestroyGraph $gn
}

proc powFreeGraph { gn } {
   # Called from powDestroyGraph
   global powOrderedGraphList

   set idx [lsearch -exact $powOrderedGraphList $gn]
   if { $idx >= 0 } {
      set powOrderedGraphList \
            [lreplace $powOrderedGraphList $idx $idx ""]
   }

   [gNotifications default] postMessage $gn graphHasBeenDestroyed
}

proc powUnmapGraph { gn } {
   global currgn powScopeGn

   powEraseGraph $gn 1
   if { $currgn == $gn } {
      set currgn "powDef"
      set powScopeGn "-"
      set otherGraphs [.pow.pow find withtag gbox]
      if {$otherGraphs != ""} {
	 set newGraph [lindex [.pow.pow gettags [lindex $otherGraphs end]] 0]
	 powSelectGraph $newGraph
      } else {
         powUpdateGraphMenuOptions
         powUpdateCurrentDialogs
      }
   }
}

proc powMapGraph { gn {restore_position 0}} {
#puts a graph back on the display
    global powPlotParam

    if $restore_position {
	set xo $powPlotParam(xo,$gn) 
	set yo $powPlotParam(yo,$gn)
    }


    powCreateGraph $gn $powPlotParam(curves,$gn) $powPlotParam(images,$gn) \
	$powPlotParam(xunits,$gn) $powPlotParam(yunits,$gn) \
	$powPlotParam(xlabel,$gn) $powPlotParam(ylabel,$gn) \
	$powPlotParam(xdimdisp,$gn) $powPlotParam(ydimdisp,$gn) \
	$powPlotParam(xBot,$gn) $powPlotParam(yBot,$gn) \
	$powPlotParam(xTop,$gn) $powPlotParam(yTop,$gn) 
	

    if $restore_position {
	powMoveGraphTo $gn $xo $yo .pow.pow
    }
}

proc powDeleteImage  {gn img} {
#deletes 
    global powPlotParam 
    .pow.pow delete ${img}disp$gn
    catch {.pow.scope delete ${img}disp${gn}scope}
    .pow.pow delete current_img
    set whichImage [lsearch $powPlotParam(images,$gn) $img] 
    set powPlotParam(images,$gn) [lreplace $powPlotParam(images,$gn) $whichImage $whichImage]
}

proc powDeleteCurve  {gn curve} {
#deletes 
    global powPlotParam 
    .pow.pow delete ${curve}$gn
    catch {.pow.scope delete ${curve}${gn}scope}
    set whichCurve [lsearch $powPlotParam(curves,$gn) $curve] 
    set powPlotParam(curves,$gn) [lreplace $powPlotParam(curves,$gn) $whichCurve $whichCurve]
}


proc invert_cmap_if_flag_set { gn img } {
   global cmap_inv powPseudoImages currimg currgn powImageParam

   if { $powImageParam(invert${img},$gn) } {
      if $powPseudoImages {
	 ${img}disp${gn} colormap invert
      } else {
	 powPhotoColorTable invert
      }
   }
}

proc powAddCustomLut { cmapName lut } {
   global powImageParam powbg

   if { [expr [llength $lut]%3]!=0 } {
      error "Lut must be list with multiple-of-3 elements (R G B)"
   }

   set powImageParam(cmapLUT_$cmapName,powDef) $lut

   set allMaps $powImageParam(allMaps,powDef)
   set map [lindex $allMaps end]
   if { [lindex $map 0] == "Custom" } {
      if { [lsearch -exact $map $cmapName]==-1 } {
         lappend map $cmapName
         # Add menu item to Custom menu
         .pow.mbar.colors.cCustom add radiobutton -label $cmapName \
               -value $cmapName \
               -variable powImageParam(colormap,powDef) \
               -command "powCmds::colormap $cmapName"
      }
      set powImageParam(allMaps,powDef) [lreplace $allMaps end end $map]
   } else {
      lappend powImageParam(allMaps,powDef) [list Custom $cmapName]
      #  Add Cust menu plus this item
      set bdVal [.pow.mbar.colors cget -bd]
      set idx [.pow.mbar.colors index [lindex $map 0]]
      incr idx
      menu .pow.mbar.colors.cCustom -bg $powbg -bd $bdVal
      .pow.mbar.colors insert $idx cascade -menu .pow.mbar.colors.cCustom \
            -label "Custom"
      .pow.mbar.colors.cCustom add radiobutton -label $cmapName \
            -value $cmapName \
            -variable powImageParam(colormap,powDef) \
            -command "powCmds::colormap $cmapName"
   }
   return
}

proc powSetLut { gn img scale {recalc {}} } {
    global powSqueeze powSlide powClen
    global currimg currgn powPlotParam powImageParam

    set powClen 255

    if {$scale == $powImageParam(scale${img},$gn) && $recalc=="" \
            && [info exists powImageParam(lut${img},$gn)] } {
       powSetImageScale $gn $img $scale
       powCmapStretchIntensity $gn $img $powClen $powClen \
               $powImageParam(lut${img},$gn)
       return
    }

    powSetImageScale $gn $img $scale

    set powSqueeze 0.0
    set powSlide 0.0
    powCmapStretchIntensity $gn $img $powClen $powClen \
            [list 0 0 $powClen $powClen]
}

proc powSetImageScale { gn img scale } {
    global powPseudoImages currimg currgn
    global powPlotParam powImageParam powGUI
    
    foreach {gn2 img2} [powGetColorbarLink $gn $img] {}
    set powImageParam(scale${img},$gn) $scale
    if { $gn2 != "" } {
       set powImageParam(scale${img2},$gn2) $scale
    }
    if { $powGUI && ![regexp scope$ $gn] } {
        set powImageParam(scale${img},${gn}scope) $scale
        if { $gn2 != "" } {
            set powImageParam(scale${img2},${gn2}scope) $scale
        }
    }

    #  Make sure we only do the equalization on original image, not colorbar
    #  nor colorbar's scope image

    if { $scale == "histo" } {
        if { $powGUI && [regexp scope$ $gn] } {
            set gn [string range $gn 0 [expr [string length $gn]-6]]
            foreach {gn2 img2} [powGetColorbarLink $gn $img] {}
        }
        if { [regexp _colorbar$ $img] } {
            if { $img2 != "" } {
                set gn $gn2
                set img $img2
            }
        }
    }

    if { $scale == "histo" } {
        set minmax [powImageScale $scale $img \
                $powImageParam(RBmin${img},$gn) \
                $powImageParam(RBmax${img},$gn)]
    } else {
        powImageScale $scale
    }
}


	
proc powCmapStretchIntensity { gn img cwidth clen lut } {
    global powPseudoImages
    global powPlotParam powImageParam powGUI
    
    foreach {gn2 img2} [powGetColorbarLink $gn $img] {}
    set powImageParam(lut${img},$gn) $lut
    if { $gn2 != "" } {
       set powImageParam(lut${img2},$gn2) $lut
    }
    if { $powGUI && ![regexp scope$ $gn] } {
       set powImageParam(lut${img},${gn}scope) $lut
       if { $gn2 != "" } {
           set powImageParam(lut${img2},${gn2}scope) $lut
       }
    }

    if $powPseudoImages {
       ${img}disp${gn} cmap_stretch intensity $cwidth $clen $lut
       if { $gn2 != "" } {
	  ${img2}disp${gn2} cmap_stretch intensity $cwidth $clen $lut
       }
    } else {
#       powPhotoColorTable $powImageParam(colormap${img},$gn)
       powPhotoCmapStretch $cwidth $clen $lut
    }
}

	
proc powBoundDiddleLut {gn img x y} {
    set cx [.pow.pow canvasx $x]
    set cy [.pow.pow canvasy $y]
    set bbox [.pow.pow coords ${gn}box]
    set lx [lindex $bbox 0]
    set ly [lindex $bbox 1]
    set ux [lindex $bbox 2]
    set uy [lindex $bbox 3]
#make range from -1 to 1
    set fx [expr (2.0 * ($cx - $lx)/($ux - $lx) - 1.0)]
    set fy [expr (2.0 * ($cy - $ly)/($uy - $ly) - 1.0)]

    powDiddleLut $gn $img $fx $fy

}
    


proc powDiddleLut { gn img slide squeeze } {
#$squeze and $slide should range from -1 to 1, not inclusive
    global powSqueeze powSlide currimg powClen
    global powPseudoImages powPlotParam powImageParam currgn

    set powSqueeze $squeeze
    set powSlide $slide

    if { ![info exist powClen] } return

#    foreach {gn2 img2} [powGetColorbarLink $currgn $currimg] {}
#    set powImageParam(scale${currimg},$currgn) current
#    if { $gn2 != "" } {
#       set powImageParam(scale${img2},$gn2) current
#    }

    set squeeze [expr double($powSqueeze)]
    set slide [expr double($powSlide)]
    set increment [expr int($powClen * $slide)]

    if {$squeeze > 0 && $squeeze < 1} {
	set factor [expr (1.0 - $squeeze)] 
	set increment [expr $increment + $squeeze*$powClen/5.0]
    } elseif {$squeeze < 0 && $squeeze > - 1} {
	set factor [expr 1.0/(1.0 + $squeeze)] 
	set increment [expr $increment - ($factor - 1.0) * $powClen/5.0]
    } else {
	set factor 1.0
    }

    for {set i 0} {$i <= $powClen} {incr i 5} {
	set newi [expr $i * $factor + $increment]
	set newi [expr ($newi < 0 ) ? 0 : $newi]
	set newi [expr ($newi > $powClen ) ? $powClen : $newi]
	lappend l2 [expr int(floor($newi))]
        lappend l2 $i
    }

    powCmapStretchIntensity $gn $img $powClen $powClen $l2
    if { !$powPseudoImages } {
        powReditherImages $gn $img
    }
}

proc powShowHandles {showhandles } {
    global powShowHandlesFlag currgn powbg powPlotParam tcl_platform
   global  powHiddenWindows

    set powShowHandlesFlag $showhandles
    if {$showhandles} then {
	.pow.pow raise current_img
        .pow.pow raise current_gn
	.pow.pow itemconfigure current_img -outline green
	.pow.pow itemconfigure current_gn -outline yellow
	.pow.pow raise buttonfg
	.pow.pow configure -bg $powbg
	foreach graph [powListGraphs] {
	    .pow.pow itemconfigure ${graph}bkg -fill $powPlotParam(bgcolor,$graph) \
		-outline $powPlotParam(bgcolor,$graph)
#for win32
	    if {[string match "Win*" $tcl_platform(os) ] &&\
		    ![regexp {scope$} $graph]} {
		.pow.pow delete deleteMe
		set images $powPlotParam(images,$graph)
		set powPlotParam(images,$graph) 'NULL'
		powPlotImages $graph  $images .pow.pow
	    }
	}
        foreach key [array names powHiddenWindows "*,loc"] {
           foreach [list wind k] [split $key ,] {}
           eval .pow.pow create window $powHiddenWindows($wind,loc) \
                 -tags {$powHiddenWindows($wind,tags)} -window $wind \
                 -anchor $powHiddenWindows($wind,anchor)
           unset powHiddenWindows($key)
        }
#	foreach wind [.pow.pow find withtag canvas_window] {
#	    set windowname [.pow.pow itemcget $wind -window]
#	    $windowname configure -foreground black -background \
#		$powPlotParam(bgcolor,$graph)  \
#		-highlightthickness 1 -relief raised
#	}
    } else {
        .pow.pow itemconfigure ohandle -outline {}
        .pow.pow lower ohandle
        .pow.pow itemconfigure graphbkg -fill {} -outline {}
#	.pow.pow configure -bg white
	.pow.pow raise ${currgn}line
	foreach wind [.pow.pow find withtag canvas_window] {
	    set windowname [.pow.pow itemcget $wind -window]
            set powHiddenWindows($windowname,loc) \
                  [.pow.pow coord $wind]
            set powHiddenWindows($windowname,tags) \
                  [.pow.pow itemcget $wind -tags]
            set powHiddenWindows($windowname,anchor) \
                  [.pow.pow itemcget $wind -anchor]
            .pow.pow delete $wind
#	    $windowname configure -foreground white -background white \
#		-highlightthickness 0 -relief flat
	}

    }
}

proc chopped {theString} {
    return [string index $theString [expr [string length $theString] - 1]]
}


proc powShowLinks { } {
    global showlinks axisToChainHash chainToAxisHash
    .pow.pow delete link
    if {$showlinks} then {
	foreach chain [array names chainToAxisHash] {
	    set oldaxis [chopped $chainToAxisHash($chain)]
	    set oldgraph [chop [lindex $chainToAxisHash($chain) 0]]
	    foreach graph [lrange $chainToAxisHash($chain) 1 end] {
		set axis [chopped $graph]
		set graph [chop $graph]
		set abox [.pow.pow coords ${oldgraph}box]
		set bbox [.pow.pow coords ${graph}box]
		if {$oldaxis == "X"} {
		    set fromX [expr int(([lindex $abox 2]+[lindex $abox 0])/2.0)]
		    set fromY [expr int([lindex $abox 3])]
		} else {
		    set fromY [expr int(([lindex $abox 3]+[lindex $abox 1])/2.0)]
		    set fromX [expr int([lindex $abox 0])]
		}
		if {$axis == "X"} {
		    set toX [expr int(([lindex $bbox 2]+[lindex $bbox 0])/2.0)]
		    set toY [expr int([lindex $bbox 3])]
		} else {
		    set toY [expr int(([lindex $bbox 3]+[lindex $bbox 1])/2.0)]
		    set toX [expr int([lindex $bbox 0])]
		}
		.pow.pow create line $fromX $fromY $toX $toY -tags "link" -fill pink
		set oldaxis $axis
		set oldgraph $graph
	    }
	}
    }
}


proc powGetColorbarLink { gn img } {
   global powPlotParam

   if { [info exists powPlotParam(Colorbar${img},$gn)] } {
      regexp "(.*)disp(.*)" $powPlotParam(Colorbar${img},$gn) z img2 gn2
   } elseif { [info exists \
	 powPlotParam(Colorbar${img}_colorbar,${gn}_colorbar)] } {
      set img2 ${img}_colorbar
      set gn2 ${gn}_colorbar
   } else {
      set img2 ""
      set gn2 ""
   }
   return [list $gn2 $img2]
}


proc powSetCurrImageOpts { args } {
   global powImageParam powPlotParam curr_img currimg currgn

   if { [info exists curr_img] && $currgn!="powDef" } {
      foreach img $powPlotParam(images,$currgn) {
	 eval powSetImageOptions $currgn $img $args
      }
   }
}

proc powSetImageOptions {gn image {args ""}} {
   global powPlotParam powImageParam powGUI currgn

   foreach {gn2 image2} [powGetColorbarLink $gn $image] {}

   if { $args == "" } {
      set lst ""
      foreach opt $powImageParam(allOpts,powDef) {
	 catch {lappend lst $opt $powImageParam(${opt}${image},$gn)}
      }
      return $lst
   } else {
      foreach {opt val} $args {
	 set idx [lsearch -exact $powImageParam(allOpts,powDef) $opt]
	 if { $idx != -1 } {
	    set powImageParam(${opt}${image},$gn) $val
	    if { $gn2 != "" } {
	       set powImageParam(${opt}${image2},$gn2) $val
	    }
	 }
	 if { $opt=="scale" } {
	    powSetLut $gn $image $val forceIt
	 }
      }
      if { [.pow.pow find withtag ${image}disp${gn}] != "" } {
	 powSetColorTable $gn $image
	 powReditherImages $gn $image
      }
   }
}

proc powPlotImages {gn images {canvas ".pow.pow"}} {
    global powPlotParam powImageParam
    global filename_array powcursor
    global powPseudoImages powRBmin powRBmax

    if [regexp "NULL" $images] return

    #   remove "NULL" from images list if present
    if [regexp "NULL" $powPlotParam(images,$gn)] {
       set powPlotParam(images,$gn) {}
    }

    set imgcnt 0
    foreach current_image "$images" {

#   if image is already in list, don't plot it
	if {[lsearch $powPlotParam(images,$gn) $current_image]>=0} continue

        # Check if image's WCS/scaling is consistent with this graph
       if { [catch {powTestImage $gn $current_image} err] } {
          tk_messageBox -icon error -type ok -parent .pow \
                -message "Couldn't place $current_image into graph...\
                \n\n\"$err\"\n\nSkipping image."
          continue
       }

	incr imgcnt

#	puts "\nplotting image: $current_image in $gn"

# Setup defaults... powDef for images, original for colorbars

       if { $canvas == ".pow.scope" } {
	  set trueGn [string range $gn 0 [expr [string length $gn]-6] ]
	  foreach opt $powImageParam(allOpts,powDef) {
	     set powImageParam(${opt}${current_image},$gn) \
		   $powImageParam(${opt}${current_image},$trueGn)
	  }
          set powImageParam(RBmin${current_image},$gn) \
                $powImageParam(RBmin${current_image},$trueGn)
          set powImageParam(RBmax${current_image},$gn) \
                $powImageParam(RBmax${current_image},$trueGn)
#	  set powImageParam(lut${current_image},$gn) \
#		$powImageParam(lut${current_image},$trueGn)
       } elseif { [string match "*_colorbar" $gn] && \
	      [string match "*_colorbar" $current_image] } {
	   regexp "(.*)disp(.*)" \
		 $powPlotParam(Colorbar${current_image},$gn) z orig_img orig_gn
	   foreach opt $powImageParam(allOpts,powDef) {
	      if { ![info exists powImageParam(${opt}${current_image},$gn)] } {
		 set powImageParam(${opt}${current_image},$gn) \
		       $powImageParam(${opt}${orig_img},$orig_gn)
	      }
	      set powImageParam(lut${current_image},$gn) \
		    $powImageParam(lut${orig_img},$orig_gn)
	   }
        } else {
	   foreach opt $powImageParam(allOpts,powDef) {
	      if { ![info exists powImageParam(${opt}${current_image},$gn)] } {
		 set powImageParam(${opt}${current_image},$gn) \
		       $powImageParam(${opt},powDef)
	      }
	   }
        }

#make a copy of the current image 
	if $powPseudoImages {
	    image create pict ${current_image}disp$gn
	} else {
	    image create photo ${current_image}disp$gn
	}

#This if block allows rescalings to persist through ROI zooms
#you could get very bizarre behavior is somebodies reusing image names....
#nothing I can think of to do about that though
	if {![info exists powImageParam(RBmin${current_image},$gn)]} {
	    set powImageParam(RBmin${current_image},$gn) \
		  $powRBmin($current_image)
	    set powImageParam(RBmax${current_image},$gn) \
		  $powRBmax($current_image)
	}


	set clipbox [powGetImageClipbox $gn $current_image $canvas]
	set powPlotParam(clipbox$current_image,$gn) $clipbox

        if {![regexp "clipped" $clipbox]}  {

	    # First two elements indicate location on graph to place image
	    set x0 [lindex $clipbox 0]
	    set y0 [lindex $clipbox 1]
	    set pcoords [powGraphToCanvas $gn $x0 $y0 $canvas]
	    set x0 [lindex $pcoords 0]
	    set y0 [lindex $pcoords 1]


	    set image_id [$canvas create image $x0 $y0 \
			      -image ${current_image}disp$gn -anchor sw \
			      -tags "$gn disp$gn\
			             ${current_image}disp$gn image_body\
			             img_$current_image"]

	    if {$canvas == ".pow.pow" } {
	       powBindBtn <<LUT>> ".pow.pow bind $image_id" \
		     "powSelectImage $gn $current_image" \
		     "powBoundDiddleLut $gn $current_image %x %y" \
		     {}

	       # The following prevents the <<LUT>> binding from executing if
	       # the <<ROI>> binding is more appropriate due to modifiers
	       powBindBtn <<ROI>> ".pow.pow bind $image_id" {} {} {}

	       if { ![info exists selImg] } {set selImg $current_image}
	    }
	}
	lappend powPlotParam(images,$gn) $current_image 

        powSetColorTable $gn $current_image
        powReditherImage $gn $current_image $canvas

    }
    if { [llength $powPlotParam(images,$gn)]==0 } {
       set powPlotParam(images,$gn) "NULL"
    }

    if { [info exists selImg] } {
       powSelectImage $gn $selImg
    }
}


proc powDeSelectImage { } {
    global curr_img currimg powPlotParam
    if {[string compare [.pow.pow find  withtag current_img] ""]} {
	.pow.pow delete current_img
    }
    catch {unset curr_img}
    catch {unset currimg}
}

proc powSelectImage {gn img} {
    global curr_img currimg powPlotParam powGUI currgn powPseudoImages

    set powPlotParam(currimg,$gn) $img

    if { $gn != $currgn } return

#delete previous bbox rectangle
    .pow.pow delete current_img 
#make all things visible
    .pow.pow raise ${gn}line

    if { [info exists currimg] } {
       set prevImg $currimg
       set prevGn  $currgn
    } else {
       set prevImg ""
       set prevGn  ""
    }

    set currimg  $img
    set curr_img ${img}disp$gn

    set tags [.pow.pow find withtag disp$gn]
    if { $tags != "" } {
	.pow.pow raise $curr_img [lindex $tags end]
    }
    if $powGUI {
	set scopeids [.pow.scope find withtag disp${gn}scope]
	if {$scopeids != ""} {
	    .pow.scope raise img_$img [lindex $scopeids end]
	}
    }

    set ibbox [.pow.pow bbox ${img}disp$gn]
    if {$ibbox != ""} {
	eval [concat .pow.pow create rectangle  $ibbox \
	      -tags [list "current_img $gn handle ohandle"] -outline green]
    }

    if { $currgn != $prevGn || $currimg != $prevImg } {
       powSetColorTable $currgn $currimg
       powUpdateGraphMenuOptions
       [gNotifications default] postMessage $currimg imageHasBeenSelected
    }
}

proc powSelectGraph {gn} {
    global currgn currimg mag powPlotParam
    global powGUI powScopeGn yellowLineWidth

#add a bit of slack around bbox
    foreach [list x0 y0 x1 y1] [.pow.pow bbox $gn] {}
    incr x0 -2
    incr y0 -2
    incr x1 2
    incr y1 2
    foreach [list ox0 oy0 ox1 oy1] [.pow.pow coord current_gn] {}
    
    if { ![info exists ox0] || \
          [expr abs( $x0 - $ox0 ) + abs( $y0 - $oy0 ) + \
          abs( $x1 - $ox1 ) + abs( $y1 - $oy1 ) ] > 1 } {
       #delete previous bbox rectangle
       .pow.pow delete current_gn
       .pow.pow create rectangle  $x0 $y0 $x1 $y1 \
             -tags "current_gn graphDragable ${gn}yhandle handle ohandle" \
             -outline yellow -width $yellowLineWidth
    }

# Rearrange graph layers and select current image if selecting new graph
    if { $currgn!=$gn } {

       [gNotifications default] postMessage $currgn graphHasBeenUnselected

       set currgn $gn
       if {[regexp "NULL" $powPlotParam(images,$gn)]} {
	  .pow.pow delete current_img
	  set powPlotParam(currimg,$gn) "NULL"
       }

       if $powGUI {
	  if {! [string compare [.pow.scope find withtag ${gn}scopebox] ""]} {
	     powRedrawScopebox
	  }
       }

       powUpdateCurrentDialogs

    } elseif $powGUI {
       if {! [string compare [.pow.scope find withtag ${gn}scopebox] ""]} {
	  powRedrawScopebox
       }
    }

    # Now restore current image for this graph
    if { [info exists powPlotParam(currimg,$gn)] \
	  && $powPlotParam(currimg,$gn)!="NULL" } {
       powSelectImage $gn $powPlotParam(currimg,$gn)
    } elseif [info exists currimg] {
       unset currimg
    }
       
    .pow.pow raise ${gn}handles
    .pow.pow lower ${gn}bkg
    .pow.pow raise ${gn}text
    .pow.pow raise $gn
    .pow.pow lower graphSelect_$gn $gn
    .pow.pow raise current_gn

    powUpdateGraphMenuOptions

    [gNotifications default] postMessage $gn graphHasBeenSelected
}

proc powUpdateGraphMenuOptions {} {
   global currgn currimg powGUI
   global powPlotParam powImageParam powMenuOption

   if $powGUI {

      set powMenuOption(tickScal) \
	 "$powPlotParam(xTickScal,$currgn)-$powPlotParam(yTickScal,$currgn)"

      .pow.mbar.edit.tlabels entryconfigure "Decimal" \
	    -variable powPlotParam(tickLabels,$currgn)
      .pow.mbar.edit.tlabels entryconfigure "Base 60 (deg)" \
	    -variable powPlotParam(tickLabels,$currgn)

      .pow.mbar.edit.grid entryconfigure "Show Grid Lines" \
	    -variable powPlotParam(GridLines,$currgn)

      foreach clr [list White Black Blue Red] {
	 .pow.mbar.edit.grid entryconfigure $clr \
	       -variable powPlotParam(GridColor,$currgn)
      }
      foreach opt [list Solid "Small Dash" "Large Dash"] {
	 .pow.mbar.edit.grid entryconfigure $opt \
	       -variable powPlotParam(GridDash,$currgn)
      }

      if { [info exists currimg] && $currimg != "" } {
	 set img $currimg
	 set gn $currgn
      } else {
	 set img ""
	 set gn powDef
      }

      foreach colorGrp $powImageParam(allMaps,powDef) {
	 set cName [lindex $colorGrp 0]
	 foreach color [lrange $colorGrp 1 end] {
	    .pow.mbar.colors.c$cName entryconfigure $color \
		  -variable powImageParam(colormap${img},$gn)
	 }
      }
      .pow.mbar.colors entryconfigure "Invert Colortable" \
	    -variable powImageParam(invert${img},$gn)
      .pow.mbar.colors entryconfigure linear \
	    -variable powImageParam(scale${img},$gn)
      .pow.mbar.colors entryconfigure "square root" \
	    -variable powImageParam(scale${img},$gn)
      .pow.mbar.colors entryconfigure logarithmic \
	    -variable powImageParam(scale${img},$gn)
      .pow.mbar.colors entryconfigure "Histo Equalize" \
	    -variable powImageParam(scale${img},$gn)
   }
}

proc powUpdateCurrentDialogs { } {
   global currgn powDWP

   if { [winfo exists ${powDWP}gEdit] } {
      powEditResetDialog
   }

}


proc powRedrawScopebox {  } {
   global currgn powPlotParam powScopeWidth powScopeHeight powScopeMargin
   global currimg powScopeGn powOrderedGraphList

   .pow.scope delete all

   # Make sure the scope's Title is up-to-date

   set title $powPlotParam(titleString,$currgn)
   if { $currgn=="powDef" } {
      set powScopeGn "-"
   } elseif { [string length $title] > 24 } {
      set ll [expr [string length $title]-10]
      set powScopeGn "[string range $title 0 11]...[string range $title $ll end]"
   } elseif { $title=="" } {
      set idx [expr [lsearch $powOrderedGraphList $currgn]+1]
      set powScopeGn "Untitled $idx"
   } else {
      set powScopeGn $title
   }
       
   # Do we need to go any further?  Any curves/images in graph?

   if {[regexp "NULL" $powPlotParam(curves,$currgn)] && \
	 [regexp "NULL" $powPlotParam(images,$currgn)] } return
   set width  [expr $powScopeWidth  - 2*$powScopeMargin]
   set height [expr $powScopeHeight - 2*$powScopeMargin]
   set width  [expr ($width < 10 ? 10 : $width )]
   set height [expr ($height< 10 ? 10 : $height)]
   powCreateGraph ${currgn}scope $powPlotParam(curves,$currgn) \
	 $powPlotParam(images,$currgn) $powPlotParam(xunits,$currgn) \
	 $powPlotParam(yunits,$currgn) $powPlotParam(xlabel,$currgn) \
	 $powPlotParam(ylabel,$currgn) \
	 $width $height \
	 NULL NULL NULL NULL .pow.scope

   # Raise current image
   if { [info exists currimg] && $currimg != "" } {
      set scopeids [.pow.scope find withtag disp${currgn}scope]
      if {$scopeids != ""} {
	 .pow.scope raise img_$currimg [lindex $scopeids end]
      }
   }

   powDrawScopeROI [list \
	 $powPlotParam(xBot,$currgn) \
	 $powPlotParam(yBot,$currgn) \
	 $powPlotParam(xTop,$currgn) \
	 $powPlotParam(yTop,$currgn)]

}

proc powResizeScope { width height } {
   global powScopeWidth powScopeHeight powScopeMargin currgn
   global powShowScope powScopeSize

   # Resize Scopebox window

   set powScopeSize [list $width $height]
   if { $width && $height } {
      set powShowScope   1
      set powScopeWidth  $width
      set powScopeHeight $height
      set powScopeMargin [expr ($width+$height)/20]
      .pow.scope configure -width $width -height $height
      powLayoutGUI
      powRedrawScopebox
   } else {
      set powShowScope 0
      grid remove .pow.scopeframe
      .pow.scope configure -width 1 -height 1
   }

   powUpdateGeometry
}

proc powUpdateGeometry {} {
   global powRealMinWidth powRealMinHeight powResizeMain

   # Update window geometry

   powSetGeometry

   set resize 0
   foreach {dx dy} [lrange [split [wm geometry .pow] {x+-}] 0 1] {}
   if { $powRealMinWidth>$dx } {
      set dx $powRealMinWidth
      set resize 1
   }
   if { $powRealMinHeight>$dy } {
      set dy $powRealMinHeight
      set resize 1
   }
   if { $resize } {
#      set x [winfo x .pow]
#      set y [winfo y .pow]
      wm geometry .pow "${dx}x${dy}"
   }

   powReconfigureToplevel $powResizeMain
}

proc powGetCurrentGraph { } {
   global currgn

   if { $currgn=="powDef" } { return "" }
   return $currgn
}

proc powMagImage {gn img {canvas .pow.pow}} {
#lowlevel routine, don't call this yourself
#resizes image to match the current magstep and ROI window
    global powPlotParam curr_img currimg
    global powPseudoImages powImageParam isMac
    
    set clipbox $powPlotParam(clipbox${img},$gn)
    if [regexp "clipped" $clipbox] return
    
    catch {image delete ${img}disp$gn}
    if $powPseudoImages {
	image create pict ${img}disp$gn
    } else {
	image create photo ${img}disp$gn
    }
    
    set width  [image width $img]
    set height [image height $img]

    #collect up the inputs to ship to Tk_(Pict||Photo)PutScaledBlock
    set x0 [lindex $clipbox 2]
    set y0 [lindex $clipbox 3]
    set x1 [lindex $clipbox 4]
    set y1 [lindex $clipbox 5]
    foreach {X0 Y0} [powPixelToCanvas $gn $img -0.5 -0.5 $canvas] {}
    foreach {X1 Y1} [powPixelToCanvas $gn $img \
	    [expr $width-0.5] [expr $height-0.5] $canvas] {}
    
    set zoomX [expr ($X1-$X0)/$width]
    set zoomY [expr ($Y0-$Y1)/$height]
		     
    set width  [expr int( ($x1 - $x0)*$zoomX + 0.5 )]
    set height [expr int( ($y1 - $y0)*$zoomY + 0.5 )]

    if { $isMac && ![powTestMacMemory [expr $width*$height]] } {
        tk_messageBox -type ok -icon error \
                -message "Not enough memory to display $img.  Will hide it\
                until memory becomes available."
    } else {
        powPutZoomedBlock $img $gn $x0 $y0 $width $height $zoomX $zoomY
    }
    
#    set curr_img ${img}disp$gn

    if { $powPseudoImages } {
       powSetRange $gn $img \
	     $powImageParam(RBmin${img},$gn) $powImageParam(RBmax${img},$gn)
    }
}

proc powSetCurveOptions {gn curve {args ""}} {
   global powCurveParam powGUI currgn

   if { $args == "" } {
      set lst ""
      foreach opt $powCurveParam(allOpts,powDef) {
	 catch {lappend lst $opt $powCurveParam(${opt}${curve},$gn)}
      }
      return $lst
   } elseif { [llength $args] == 1 } {
      set opt [lindex $args 0]
      set idx [lsearch -exact $powCurveParam(allOpts,powDef) $opt]
      if { $idx != -1 } {
         if { $opt=="pColor" || $opt=="lColor" } {
            set val [powColorToHex $val]
         }
         return $powCurveParam(${opt}${curve},$gn)
      } else {
         return ""
      }
   } else {
      foreach {opt val} $args {
	 set idx [lsearch -exact $powCurveParam(allOpts,powDef) $opt]
	 if { $idx != -1 } {
	    if { $opt=="pColor" || $opt=="lColor" } {
	       set val [powColorToHex $val]
	    }
	    set powCurveParam(${opt}${curve},$gn) $val
	 }
      }
      if { [.pow.pow find withtag ${curve}${gn}] != "" } {
	 .pow.pow delete ${curve}${gn}
	 powPlot1Curve $gn $curve .pow.pow
	  if {$powGUI && ($gn == $currgn)}  {
	    .pow.scope delete ${curve}${gn}scope
	    powPlot1Curve ${gn}scope $curve .pow.scope
	 }
      }
   }
}

proc powAddCurves {gn curves} {
   global powPlotParam currgn powGUI

   powPlotCurves $gn $curves .pow.pow
   if { $powGUI && $gn == $currgn } {
      powRedrawScopebox
   }
}

proc powAddImages {gn images} {
   global powPlotParam currgn powGUI

   powPlotImages $gn $images .pow.pow
   if { $powGUI && $gn == $currgn } {
      powRedrawScopebox
   }
}

proc powRemoveCurves {gn curves} {
   global powPlotParam currgn powGUI

   set hasChanged 0
   foreach c $curves {
      set idx [lsearch -exact $powPlotParam(curves,$gn) $c]
      if { $c != "NULL" && $idx != -1 } {
         set hasChanged 1
         .pow.pow delete $c$gn
         set powPlotParam(curves,$gn) \
               [lreplace $powPlotParam(curves,$gn) $idx $idx]
      }
   }
   if { [llength $powPlotParam(curves,$gn)]==0 } {
      set powPlotParam(curves,$gn) "NULL"
   }

   if { $powGUI && $gn == $currgn && $hasChanged } {
      powRedrawScopebox
   }
}

proc powRemoveImages {gn images} {
   global powPlotParam currgn powGUI

   set hasChanged 0
   foreach i $images {
      set idx [lsearch -exact $powPlotParam(images,$gn) $i]
      if { $i != "NULL" && $idx != -1 } {
         set hasChanged 1
         .pow.pow delete ${i}disp$gn
         set powPlotParam(images,$gn) \
               [lreplace $powPlotParam(images,$gn) $idx $idx]
      }
   }
   if { [llength $powPlotParam(images,$gn)]==0 } {
      set powPlotParam(images,$gn) "NULL"
   }

   if { $powGUI && $gn == $currgn && $hasChanged } {
      powRedrawScopebox
   }
}

proc powGetCurveLength { crv } {
   array set crvInfo [powFetchCurveInfoHash $crv]
   array set vecInfo [powFetchVectorInfoHash $crvInfo(X)]
   return [powFetchDataLength $vecInfo(data)]
}

proc powPlot1Curve {gn crv {canvas .pow.pow}} {
   global powCurveParam powScopeWidth powScopeHeight

   if { $canvas == ".pow.scope" } {
      set trueGn [string range $gn 0 [expr [string length $gn]-6] ]
      foreach opt $powCurveParam(allOpts,powDef) {
	 set powCurveParam(${opt}${crv},$gn) \
	       $powCurveParam(${opt}${crv},$trueGn)
      }
   } else {
      foreach opt $powCurveParam(allOpts,powDef) {
	 if {! [info exists powCurveParam(${opt}${crv},$gn)]} {
	    if { $opt == "pShape" && [powGetCurveLength $crv]>10000 \
                  && ($powCurveParam(LOD,powDef) == 0 || \
                  $powCurveParam(LOD,powDef)  > 10000) } {
	       set powCurveParam(${opt}${crv},$gn) Dot
            } elseif { [info exists powCurveParam(${opt}${crv},powDef)] } {
	       set powCurveParam(${opt}${crv},$gn) \
		     $powCurveParam(${opt}${crv},powDef)
	    } else {
	       set powCurveParam(${opt}${crv},$gn) \
		     $powCurveParam(${opt},powDef)
	    }
	 }
      }
   }
   
   foreach opt $powCurveParam(allOpts,powDef) {
      set $opt $powCurveParam(${opt}${crv},$gn)
   }
   if { $logX || $logY } {
      if { [powWCSexists $gn] } {
         # Cannot mix WCS and log, so change options to No
         set powCurveParam(logX${crv},$gn) No
         set powCurveParam(logY${crv},$gn) No
         set logX No
         set logY No
      }
   }
   
   if { $canvas == ".pow.scope" } {
      # Shrink point size if drawing in the scope window
      if { $pSize>0 } {
	 set pSize [expr round($pSize*($powScopeWidth+$powScopeHeight)/800.0)]
	 if { $pSize<=1 } {
	    set pShape Dot
	 }
      }
   }
   
   #   call curve plotting routine
   $canvas create powCurve $crv $gn \
	 -pointdisplay $pDisp \
	 -pointtype $pShape \
	 -pointsize $pSize \
	 -pointerror $pSizeErr \
	 -pointfill $pFill \
	 -linedisplay $lDisp \
	 -dash $lStyle \
	 -width $lWidth \
	 -stairstep $lStep \
         -boxfill $lBoxFill \
	 -lfill $lColor \
	 -pfill $pColor \
	 -logx $logX \
	 -logy $logY \
	 -tags "$gn $crv$gn" \
         -LOD $LOD
}

proc powPlotCurves {gn curves {canvas .pow.pow}} {
    global powPlotParam powCurveParam powcursor powResizeMain

    if [regexp "NULL" $curves] return

#remove "NULL" from curves list if present
    if [regexp "NULL" $powPlotParam(curves,$gn)] {set powPlotParam(curves,$gn) { }}

    set crvCnt [llength $powPlotParam(curves,$gn)]
    foreach current_curve $curves {

#   if curve is already in list, don't plot it
	if [regexp $current_curve $powPlotParam(curves,$gn)] continue

#   Check if we need to assign a new color to this curve
        if { ![info exists powCurveParam(lColor${current_curve},$gn)] && \
              ![info exists powCurveParam(lColor${current_curve},powDef)] } {
	   set colors $powCurveParam(allColors,powDef)
	   set nElem [lsearch $colors $powCurveParam(lColor,powDef)]
	   # Must increment by 2* because list contains COLOR #HEX COLOR #HEX
	   incr nElem [expr $crvCnt+$crvCnt]
	   if { $nElem<0 } {
	      set powCurveParam(lColor${current_curve},$gn) \
		    $powCurveParam(lColor,powDef)
	   } else {
	      while { $nElem >= [llength $colors] } {
		 incr nElem -[llength $colors]
	      }
	      set powCurveParam(lColor${current_curve},$gn) \
		    [lindex $colors $nElem]
	   }
	}
        if { ![info exists powCurveParam(pColor${current_curve},$gn)] && \
              ![info exists powCurveParam(pColor${current_curve},powDef)] } {
	   set colors $powCurveParam(allColors,powDef)
	   set nElem [lsearch $colors $powCurveParam(pColor,powDef)]
	   # Must increment by 2* because list contains COLOR #HEX COLOR #HEX
	   incr nElem [expr $crvCnt+$crvCnt]
	   if { $nElem<0 } {
	      set powCurveParam(pColor${current_curve},$gn) \
		    $powCurveParam(pColor,powDef)
	   } else {
	      while { $nElem >= [llength $colors] } {
		 incr nElem -[llength $colors]
	      }
	      set powCurveParam(pColor${current_curve},$gn) \
		    [lindex $colors $nElem]
	   }
	}
	       
        if { [catch {powPlot1Curve $gn $current_curve $canvas} err] } {
	   tk_messageBox -icon error -type ok -parent .pow \
		 -message "Couldn't place $current_curve into graph...\
                 \n\n\"$err\"\n\nSkipping curve."
	} else {
#        add name to list of curves
	   lappend powPlotParam(curves,$gn) $current_curve
	   incr crvCnt
	}
    }
}

proc powWhereAmI {x y {canvas ".pow.pow"}} {
    set boxes [$canvas find withtag gbox]
    set topbox -1
    set topgraph ""
    foreach graph [powListGraphs] {
	set gbox [$canvas coords ${graph}box]
	if { $gbox == "" } continue
	if { $x >= [lindex $gbox 0] && $x <= [lindex $gbox 2] \
		&& $y >= [lindex $gbox 1]  && $y <= [lindex $gbox 3] } {
	    set order [lsearch $boxes [$canvas find withtag ${graph}box]]
	    if {$order>$topbox} {set topbox $order; set topgraph $graph }
	}
    }
    if {$topbox>=0} {return $topgraph} else {return "NULL"}
}


proc powWhereAmI_img {gn x y {canvas ".pow.pow"}} {

   set images [$canvas find withtag disp$gn]
   set N [llength $images]
   if { $N==0 } { return "NULL" }

   for { set i $N } { $i>0 } {  } {
      incr i -1
      set img [lindex $images $i]
      set ibox [$canvas bbox $img]
      if { $ibox == "" } continue
      if { $x >= [lindex $ibox 0] && $x <= [lindex $ibox 2] \
            && $y >= [lindex $ibox 1]  && $y <= [lindex $ibox 3] } {
         set tags [$canvas gettags $img]
         set elem [lsearch -glob $tags ?*disp$gn]
         if { [regexp "^(.+)disp$gn$" [lindex $tags $elem] dmy theImage] } {
            return $theImage
         }
      }
   }
   return "NULL"
}


#  C  routines: CanvasToGraph, GraphToPixel
#               PixelToGraph, GraphToCanvas
# TCL routines: CanvasToPixel, PixelToCanvas

proc powPixelToCanvas {gn img x y {canvas .pow.pow}} {

    set ccoords [powPixelToGraph $img $x $y]
    set rx [lindex $ccoords 0]
    set ry [lindex $ccoords 1]
    set ccoords [powGraphToCanvas $gn $rx $ry $canvas]

#    puts "PixelToCanvas - $x $y $ccoords $gn"
    return $ccoords 
}

proc powCanvasToPixel {gn img x y {canvas .pow.pow}} {
    global powPlotParam

    set ccoords [powCanvasToGraph $gn $x $y $canvas]
    set rx [lindex $ccoords 0]
    set ry [lindex $ccoords 1]
    set ccoords [powGraphToPixel $img $rx $ry]

#    puts "CanvasToPixel - $x $y $ccoords $gn"
    return $ccoords 
}

proc set_tracker_info {x y {canvas ".pow.pow"}} {
    global powPlotParam powTrackText currimg currgn
    global powPseudoImages

    set cx [$canvas canvasx $x]
    set cy [$canvas canvasy $y]
    set gn [powWhereAmI $cx $cy $canvas]

    set powTrackText(gn) $gn
    if { $gn != "NULL" } {
       set gcoords [powCanvasToGraph $gn $cx $cy $canvas]
       set powTrackText(rx) [lindex $gcoords 0]
       set powTrackText(ry) [lindex $gcoords 1]

       set img [powWhereAmI_img $gn $cx $cy $canvas]
       set powTrackText(img) $img
       if { $img != "NULL" } {
          set icoords [powCanvasToPixel $gn $img $cx $cy $canvas]
          set imgx [expr int([lindex $icoords 0]+0.5)]
          set imgy [expr int([lindex $icoords 1]+0.5)]
          
          set width  [image width  $img]
          set height [image height $img]
	
          if { ($imgx < $width) && ($imgy < $height) \
                && ($imgx >= 0) && ($imgy >= 0) } {
             set powTrackText(imgx) $imgx
             set powTrackText(imgy) $imgy
             set powTrackText(imgz) [powGetImageZ $img $imgx $imgy]
          } else {
             set powTrackText(imgx) "X"
             set powTrackText(imgy) "X"
             set powTrackText(imgz) "X"
          }
          set powTrackText(zunits) [powGetImageUnits $img Z]
       } else {
          set powTrackText(imgx) "X"
          set powTrackText(imgy) "X"
          set powTrackText(imgz) "X"
          set powTrackText(zunits) " "
       }

    } else {

       set powTrackText(rx) X
       set powTrackText(ry) X
       set powTrackText(imgx) "X"
       set powTrackText(imgy) "X"
       set powTrackText(imgz) "X"
       set powTrackText(zunits) " "
    }
    powUpdateTrackVars
}


proc powStretchGraph {gn xfactor yfactor {canvas ".pow.pow"}} {
    global powPlotParam

    powResizeGraph $gn $xfactor $yfactor $canvas
}


proc powMagGraph {gn newxmagstep newymagstep {canvas ".pow.pow"}} {
    global powPlotParam

    set xfactor \
	    [expr double($newxmagstep) / double($powPlotParam(xmagstep,$gn))]
    set yfactor \
	    [expr double($newymagstep) / double($powPlotParam(ymagstep,$gn))]
    powResizeGraph $gn $xfactor $yfactor $canvas
}

proc powResizeGraph {gn xfactor yfactor {canvas ".pow.pow"}} {
#lowlevel routine, don't call this yourself.  Use powMagGraph or powStretchGraph
#all "resizings" of a graph are done here.  Don't you dare
#do them elsewhere or you'll regret it.
    global powPlotParam powcursor powResizeMain
    global currimg powScopeMargin

    foreach el  [array names powPlotParam]  {
	set p1 [lindex [split $el ,] 0]
	set p2 [lindex [split $el ,] 1]
	if { $p2 == $gn } {
	    set $p1 $powPlotParam($p1,$p2)
	}
    }
    
    
    set bbox [$canvas coords ${gn}box]

#    save initial coordinates

    if {$canvas == ".pow.scope"} {
	set ul [list $powScopeMargin $powScopeMargin]
    } else {
	set ul [list [lindex $bbox 0] [lindex $bbox 1]]
    }


    $canvas scale $gn [lindex $bbox 0] [lindex $bbox 1] $xfactor $yfactor


    set powPlotParam(xmagstep,$gn) [expr $xfactor * $powPlotParam(xmagstep,$gn)]
    set powPlotParam(ymagstep,$gn) [expr $yfactor * $powPlotParam(ymagstep,$gn)]

    powSetGraphMagstep $gn $powPlotParam(xmagstep,$gn) \
	    $powPlotParam(ymagstep,$gn) 
    
    if {![regexp "NULL" $images]} {
	foreach img $powPlotParam(images,$gn) {
	    powMagImage $gn $img $canvas
	}
    }
    
    
    if {$canvas == ".pow.pow"} {
       # Redraw all the adornments

	.pow.pow delete ${gn}handles
	.pow.pow delete ${gn}shandle
	.pow.pow delete ${gn}yhandle
	
	#draw new tick marks and numbers to go with and new labels
	powDrawTicks $gn $canvas
	
	#make new GraphHandles
	powMakeGraphLabels $gn
	powMakeGraphHandles $gn
	powSelectGraph $gn
    }

    [gNotifications default] postMessage $gn graphHasResized
}



proc powRestoreGraph {gn {canvas .pow.pow}} {
    set bbox [$canvas bbox $gn]
    set x [lindex $bbox 0]
    set y [lindex $bbox 1]
    set mx [expr ($x < 20) ? 20 - $x : 0]
    set my [expr ($y < 20) ? 20 - $y : 0]
    powMoveGraph $gn $mx $my $canvas
}
    


proc tagXdim {can tag} {
    set bbox [$can coords $tag]
    return [expr [lindex $bbox 2] - [lindex $bbox 0]]
}

proc tagYdim {can tag} {
    set bbox [$can coords $tag]
    return [expr [lindex $bbox 3] - [lindex $bbox 1]]
}



proc powStretchGraphToSize {gn xdim ydim {canvas ".pow.pow"}} {
#stretches/shrinks graph to fit in xdim/ydim size
    global  powPlotParam 

    set curr_xdim [tagXdim $canvas ${gn}box]
    set curr_ydim [tagYdim $canvas ${gn}box]

    set xfactor [expr double($xdim)/double($curr_xdim)]
    set yfactor [expr double($ydim)/double($curr_ydim)]
	
    powStretchGraph $gn $xfactor $yfactor $canvas
    
    #save requested current size of graph
    set powPlotParam(xdimdisp,$gn) $xdim
    set powPlotParam(ydimdisp,$gn) $ydim
    
    if {$canvas == ".pow.pow"} {
	powSelectGraph $gn
    }

}


proc powDragGraph { stage X Y } {
   global powMoveX powMoveY powIsDragging
   global currgn powResizeMain

   switch -exact $stage {
      start {
         set powMoveX $X
         set powMoveY $Y
         set powIsDragging 1
      }
      drag {
         powHideCurves $currgn
         powMoveHandle $currgn $X $Y
      }
      end {
         powShowCurves $currgn
         powReconfigureToplevel $powResizeMain
         set powIsDragging 0
      }
   }
}

#Plotting routines below here ...
proc powMoveGraph {gn xDist yDist {canvas ".pow.pow"}} {
    global powPlotParam 

    if { $xDist==0 && $yDist==0 } return
    $canvas move $gn $xDist $yDist
    $canvas move ${gn}handles $xDist $yDist
    $canvas move ${gn}yhandle $xDist $yDist
    incr powPlotParam(xo,$gn) $xDist
    incr powPlotParam(yo,$gn) $yDist

    [gNotifications default] postMessage $gn graphHasMoved $xDist $yDist
}

proc powMoveGraphTo {gn x y {canvas ".pow.pow"}} {
    global powPlotParam 
    set bbox [$canvas coords ${gn}box]
    set xDist [expr $x - [lindex $bbox 0]]
    set yDist [expr $y - [lindex $bbox 1]]

    $canvas move $gn $xDist $yDist
    $canvas move ${gn}handles $xDist $yDist
    $canvas move ${gn}yhandle $xDist $yDist
    set powPlotParam(xo,$gn) $x
    set powPlotParam(yo,$gn) $y
    # powRestoreGraph $gn $canvas
    
    [gNotifications default] postMessage $gn graphHasMoved $xDist $yDist
}


proc powRedrawGraphHandles {gn} {
   global currgn
   if { [.pow.pow find withtag ${gn}handles] != "" } {
      .pow.pow delete ${gn}handles
      .pow.pow delete ${gn}shandle
      powMakeGraphHandles $gn
      #  If this is current graph, call SelectGraph to update yellow box
      if { $gn == $currgn } {powSelectGraph $gn}
   }
}

proc powMakeGraphHandles {gn} {
    global powPlotParam env
    
#    update idletasks
    set bbox [.pow.pow bbox $gn]
    set left [lindex $bbox 0]
    set top [lindex $bbox 1]
    set right [lindex $bbox 2 ]
    set bot [lindex $bbox 3]

#Make "handle" for graph,  you can pick up the graph and 
#move it around by dragging this
    if [string match "*t*" $powPlotParam(handleposition,$gn)] {
	set y $top
    } elseif [string match "*b*" $powPlotParam(handleposition,$gn)] {
	set y $bot
    } else {
	set y [expr ($top + $bot)/2.0]
    }

    if [string match "*l*" $powPlotParam(handleposition,$gn)] {
	set x $left
    } elseif [string match "*r*" $powPlotParam(handleposition,$gn)] {
	set x $right
    } else {
	set x [expr ($left + $right)/2.0]
    }

# .pow.ms${gn}handle - the 'ms' stands for Move/select and is necessary
# to allow graph names to start with a capital 

    set msName "ms[powCleanName $gn]handle"
    if [winfo exists .pow.$msName] {
	.pow.$msName configure -bg $powPlotParam(bgcolor,$gn) \
	    -text $powPlotParam(handletext,$gn) -cursor fleur 
    } else {
	button .pow.$msName \
		-bg $powPlotParam(bgcolor,$gn) \
		-text $powPlotParam(handletext,$gn) -cursor fleur 
	bind .pow.$msName <ButtonPress-1> \
		"set powMoveX %X ; set powMoveY %Y"
	bind .pow.$msName <B1-Motion> \
		"powHideCurves $gn ; powMoveHandle $gn %X %Y"
	bind .pow.$msName <ButtonRelease-1> \
	    "+powShowCurves $gn; powSelectGraph $gn; \
             powReconfigureToplevel \$powResizeMain"
#the next binding lets us get the graph back if we drag it off the viewable
#area
#	bind .pow <ButtonRelease-1> {
#	    powReconfigureToplevel $powResizeMain
#	}

    }

    raise .pow.$msName .pow.pow

#    .pow.pow create window $x $y \
#	-tags "${gn}handle handle ghandle ${gn}handles canvas_window" \
#        -anchor $powPlotParam(handleanchor,$gn) \
#	-window .pow.$msName



#Make "stretch handle" for graph,  you will be able to expand the graph
#by dragging this.

    set sName "s[powCleanName $gn]handle"
    if [winfo exists .pow.$sName] {
	.pow.$sName configure -bg $powPlotParam(bgcolor,$gn) \
	    -bitmap stretcharrow\
	    -cursor bottom_right_corner
    } else {
	button .pow.$sName -bg $powPlotParam(bgcolor,$gn) \
	    -bitmap stretcharrow \
	    -cursor bottom_right_corner
	bind .pow.$sName <B1-Motion> \
	      "powHideCurves $gn; powStretch $gn %X %Y"
	bind .pow.$sName <ButtonPress-1> \
	    "powBeginStretch $gn %X %Y; \
             set fixedStretch \"yes\"; \
             powStretch $gn %X %Y; "
	bind .pow.$sName <Shift-ButtonPress-1> \
	    "powBeginStretch $gn %X %Y; \
             set fixedStretch \"no\"; \
             powStretch $gn %X %Y; "
	bind .pow.$sName <ButtonRelease-1> \
	    "powShowCurves $gn; powEndStretch $gn"
    }


    .pow.pow create window $right $bot\
	-tags " ${gn}shandle shandle handle ${gn}handles canvas_window" \
	-anchor se -window .pow.$sName
   

    raise .pow.$sName .pow.pow


#Make a colored background
    .pow.pow create rectangle $bbox \
	-tags "${gn}handles ${gn}bkg graphbkg" \
	-fill $powPlotParam(bgcolor,$gn) -outline $powPlotParam(bgcolor,$gn)
    .pow.pow lower ${gn}bkg

# Now create an underlying polygon with -fill {} to catch all clicks
       
    .pow.pow delete graphSelect_$gn
    foreach {x0 y0 x1 y1} $bbox {}
    .pow.pow create polygon $x0 $y0 $x0 $y1 $x1 $y1 $x1 $y0 $x0 $y0 \
          -outline {} -fill {} -tags "${gn}handles graphSelect_$gn"
    .pow.pow lower graphSelect_$gn $gn

#Store position of "Select" handle relative to graph box
#    update idletasks
    set hcoords [.pow.pow coords ${gn}handle]
    set bcoords [.pow.pow coords ${gn}box]
    set powPlotParam(handleoffsetx,$gn) \
	[expr [lindex $hcoords 0] - [lindex $bcoords 0]]
    set powPlotParam(handleoffsety,$gn) \
	[expr [lindex $hcoords 1] - [lindex $bcoords 1]]
}

proc powCleanName {gn} {
    regsub -all {\.} $gn {_} a
    return $a
}


proc powHideCurves { gn } {
    global powPlotParam

    foreach crv $powPlotParam(curves,$gn) {
	if {$crv=="NULL"} continue
#if the curve has string Z data, the next statement will fail
#since the string Z data is implemented as a separate
#canvas text item with the same tag and canvas text items don't
#have the -hidden option... so catch it
	catch {.pow.pow itemconfig ${crv}${gn} -hidden 1}
    }
}

proc powShowCurves { gn } {
    global powPlotParam

    foreach crv $powPlotParam(curves,$gn) {
	if {$crv=="NULL"} continue
#if the curve has string Z data, the next statement will fail
#since the string Z data is implemented as a separate
#canvas text item with the same tag and canvas text items don't
#have the -hidden option... so catch it
	catch {.pow.pow itemconfig ${crv}${gn} -hidden 0}
    }
}

proc powMoveHandle {gn x y} {
    global powMoveX powMoveY powPlotParam

    # Calculate root bounding box of .pow.pow canvas (- a little)

    set left  [expr [winfo rootx  .pow.pow]         + 10]
    set top   [expr [winfo rooty  .pow.pow]         + 10]
    set right [expr [winfo width  .pow.pow] + $left - 20]
    set bott  [expr [winfo height .pow.pow] + $top  - 20]
    
    # Check whether we have moved outside of the .pow.pow canvas

    if { $x < $left } {
	set x $left
    } elseif { $x > $right } {
	set x $right
    }

    if { $y < $top } {
	set y $top
    } elseif { $y > $bott } {
	set y $bott
    }

    set dx [expr $x - $powMoveX]
    set dy [expr $y - $powMoveY]

    powMoveGraph $gn [expr $x - $powMoveX] [expr $y - $powMoveY]

    set powMoveX $x 
    set powMoveY $y
}


proc powFindOverlapGraph { Lft Top Rgt Bot } {
   set gn ""
   foreach gnIdx [.pow.pow find withtag gbox] {
      foreach {lft top rgt bot} [.pow.pow bbox $gnIdx] {}
      if { $rgt<$Lft || $lft>$Rgt || $top>$Bot || $bot<$Top } continue
      set gn [lindex [.pow.pow gettags $gnIdx] 0]
   }
   return $gn
}


proc powBeginROI {x y {canvas .pow.pow}} {
    global roi_xo roi_yo saveROI

    if {[$canvas find withtag ROI] != ""} {
	set saveROI [$canvas coords ROI]
    }

    set roi_xo [$canvas canvasx $x]
    set roi_yo [$canvas canvasy $y]
    $canvas create rectangle $x $y $x $y -tags ROI -outline blue
}

proc powDragROI {x y {canvas .pow.pow}} {
    global roi_xo roi_yo
    $canvas delete ROI
    $canvas create rectangle $roi_xo $roi_yo \
	[$canvas canvasx $x] [$canvas canvasy $y] \
	-tags ROI -outline blue
}

proc powPanROI {x y {canvas .pow.pow}} {
    global roi_xo roi_yo
    set ROIbbox [$canvas coords ROI]
    set halfwidth [expr ([lindex $ROIbbox 2] - [lindex $ROIbbox 0])/2.0]
    set halfheight [expr ([lindex $ROIbbox 3] - [lindex $ROIbbox 1])/2.0]
    $canvas delete ROI
    $canvas create rectangle [expr $x - $halfwidth] [expr $y - $halfheight] [expr $x + $halfwidth] [expr $y + $halfheight] -tags ROI -outline blue
}


proc powEndROI { zoomback {canvas .pow.pow}} {
global saveROI powGUI
#if zoomback is true, we're restoring the "default size" of the graph
#otherwise, this is the end of a user dragging an ROI box
    global powPlotParam currgn axisToChainHash chainToAxisHash powResizeMain
    global powGraphsTagRangeList powTagsColorMap currimg
    global powGraphsTagRectList

    if {$canvas == ".pow.scope"} {
        set currgraph ${currgn}scope
        # If graph is empty, scopebox will be empty, so don't draw ROI
        if { [.pow.scope find withtag ${currgraph}box]=="" } {
           .pow.scope delete ROI
           return
        }
    } else {
	set currgraph  $currgn
    }

    if {!($zoomback)} {
	set ROIbbox [$canvas coords ROI]
	set x0 [lindex $ROIbbox 0]
	set x1 [lindex $ROIbbox 2]
	set y0 [lindex $ROIbbox 3]
	set y1 [lindex $ROIbbox 1]

#return if ROI is <= 1 screen pixel (they probably clicked without moving)
	if { abs($x0 -$x1) <= 1 || abs($y0 - $y1) <= 1} {
	    $canvas delete ROI
	    if [info exists saveROI] {
#redraw original ROI
		$canvas create rectangle $saveROI -tags ROI -outline blue
	    }
	    return
	}

	if { $canvas==".pow.pow" } {
	   # Find which graph ROI overlaps, if any

	   set overlap_gn [powFindOverlapGraph $x0 $y1 $x1 $y0]
	   if { $overlap_gn != "" && $overlap_gn!=$currgn } {
	      powSelectGraph $overlap_gn
	      set currgraph $currgn
	   }
	}

#get "real" coordinates of ROIbbox
	set gcoords  [powCanvasToGraph $currgraph $x0 $y0 $canvas] 
	set llx [lindex $gcoords 0]
	set lly [lindex $gcoords 1]
	set gcoords  [powCanvasToGraph $currgraph $x1 $y1 $canvas] 
	set urx [lindex $gcoords 0]
	set ury [lindex $gcoords 1]
	
    }

    set graphlist $currgn

    if {[array names axisToChainHash ${currgn}X] != ""} {
	set graphlist [concat $graphlist $chainToAxisHash($axisToChainHash(${currgn}X))]
    }

    if {[array names axisToChainHash ${currgn}Y] != ""} {
	set graphlist [concat $graphlist $chainToAxisHash($axisToChainHash(${currgn}Y))]
    }

    set principal 1
    
    foreach graph [concat $graphlist] {
	if {!$principal} {
		set axis [chopped $graph]
		set graph [chop $graph]
	}



	if {!$zoomback} {
	    if $principal {
#note rROIbbox is in "scientific" coordinate, other bboxs are in "X" coordinates (i.e. upper left origin)
		set rROIbbox [list $llx $lly $urx $ury]
		if { $canvas != ".pow.scope" } { $canvas delete ROI }
	    } else {
		if {$axis == "X"} {
		    set abox [.pow.pow coords ${graph}box]
		    set cllx $llx
		    set clly [lindex [powCanvasToGraph $graph \
			     [lindex $abox 0] [lindex $abox 3] .pow.pow] 1]
		    set curx $urx
		    set cury [lindex [powCanvasToGraph $graph \
                             [lindex $abox 2] [lindex $abox 1] .pow.pow] 1]
		    set rROIbbox [list $cllx $clly $curx $cury]
		} else {
		    set abox [.pow.pow coords ${graph}box]
		    set cllx [lindex [powCanvasToGraph $graph \
                             [lindex $abox 0] [lindex $abox 3] .pow.pow] 0]
		    set clly $lly
		    set curx [lindex [powCanvasToGraph $graph \
                             [lindex $abox 2] [lindex $abox 1] .pow.pow] 0]
		    set cury $ury
		    set rROIbbox [list $cllx $clly $curx $cury]
		}	    
	    }
	}

    
#get together everything you need for the next call to powCreateGraph
	set graph_position [.pow.pow coords ${graph}handle]
	set ROIcurves $powPlotParam(curves,$graph)
	set ROIimages $powPlotParam(images,$graph)
	set ROIunits [list $powPlotParam(xunits,$graph) \
		           $powPlotParam(yunits,$graph) \
			   $powPlotParam(xlabel,$graph) \
			   $powPlotParam(ylabel,$graph) ]

	set ROIgraphOptions [powGetGraphOptions $graph]


# Make new graph
#	powUnmapGraph $graph 0
	if {$zoomback} {

	    eval [concat powCreateGraph $graph \{$ROIcurves\} \{$ROIimages\} \
                                    $ROIunits $powPlotParam(xdimdisp,$graph) \
		                    $powPlotParam(ydimdisp,$graph)]
	    set rROIbbox [list $powPlotParam(xBot,$graph) \
                               $powPlotParam(yBot,$graph) \
                               $powPlotParam(xTop,$graph) \
                               $powPlotParam(yTop,$graph) ]
	    set powPlotParam(zoomed,$graph) 0

	} else {

	    eval [concat powCreateGraph $graph \{$ROIcurves\} \{$ROIimages\} \
		                    $ROIunits $powPlotParam(xdimdisp,$graph) \
		                    $powPlotParam(ydimdisp,$graph) $rROIbbox]
	    set powPlotParam(zoomed,$graph) 1
	}

	eval [concat powGraphOptions $graph $ROIgraphOptions]
	set principal 0
    }

    if {$powGUI && $canvas != ".pow.scope"} {
	.pow.scope delete ROI
	powDrawScopeROI $rROIbbox
    }

    if {[info exists powGraphsTagRangeList($currgn)]} {
	foreach tagrangelist [concat $powGraphsTagRangeList($currgn)] {
	    set tag [lindex $tagrangelist 3]
	    eval [concat powColorRange $currgn $tagrangelist $powTagsColorMap($tag) 1]
	}
    }
    if {[info exists powGraphsTagRectList($currgn)]} {
	foreach tagrectlist [concat $powGraphsTagRectList($currgn)] {
	    set tag [lindex $tagrectlist 4]
	    eval [concat powColorRect $currgn $tagrectlist $powTagsColorMap($tag) 1]
	}
    }

}


proc powDrawScopeROI { rROIbbox } {
    global currgn

    set gn ${currgn}scope

    # If graph is empty, scopebox will be empty, so don't draw ROI
    if { [.pow.scope find withtag ${gn}box]=="" } return

    set rllx [lindex $rROIbbox 0]
    set rlly [lindex $rROIbbox 1]
    set rurx [lindex $rROIbbox 2]
    set rury [lindex $rROIbbox 3]
    set ccoords [powGraphToCanvas $gn $rllx $rlly .pow.scope]
    set ulx [lindex $ccoords 0]
    set lry [lindex $ccoords 1]
    set ccoords [powGraphToCanvas $gn $rurx $rury .pow.scope]
    set lrx [lindex $ccoords 0]
    set uly [lindex $ccoords 1]
    .pow.scope create rectangle $ulx $uly $lrx $lry -tags ROI -outline blue
}




proc powStretch {gn x y} {
    global stretchX0 stretchY0
    global powPlotParam
    global fixedStretch new_xdim new_ydim
    global yellowLineWidth
    global ulx_yellow uly_yellow lrx_yellow lry_yellow
    global powHandX0 powHandY0
    global powGBWidth powGBHeight
    global powrootx powrooty

    # Calculate root bounding box of allowed area of the canvas (- a little)

    set cx [.pow.pow canvasx [expr $x - $powrootx]]
    set cy [.pow.pow canvasy [expr $y - $powrooty]]
    

    set left [expr $ulx_yellow + 30]
    set top [expr $uly_yellow + 30]
    

    set right [.pow.pow canvasx [expr [winfo width  .pow.pow] - 20]]
    set bott  [.pow.pow canvasy [expr [winfo height .pow.pow] - 20]]
    
    # Check whether we have moved outside of the .pow.pow canvas
    # or past the upper left corner of the current_gn box
   
    if { $cx < $left } {
	set cx $left
    } elseif { $cx > $right } {
	set cx $right
    }

    if { $cy < $top } {
	set cy $top
    } elseif { $cy > $bott } {
	set cy $bott
    }

    .pow.pow delete current_gn


#how far have we moved the stretch-handle?

    set dx [expr $cx - $stretchX0]
    set dy [expr $cy - $stretchY0]



#check magstep


    

# calculate new xfactor from change in size of graphbox

    if { [expr $powGBWidth  + $dx] < 1 } { set dx [expr 1 - $powGBWidth]  }
    if { [expr $powGBHeight + $dy] < 1 } { set dy [expr 1 - $powGBHeight] }

    set xfactor [expr double($powGBWidth  + $dx) / double($powGBWidth) ]
    set yfactor [expr double($powGBHeight + $dy) / double($powGBHeight)]

    if { $fixedStretch == "yes" } {
	if { $xfactor < $yfactor } {
	    set yfactor $xfactor
	    set dy [expr double($powGBHeight) * ($yfactor - 1.0)]
	} else {
	    set xfactor $yfactor
	    set dx [expr double($powGBWidth)  * ($xfactor - 1.0)]
	}
    }



#Move the stretch-handle

    .pow.pow coords ${gn}shandle [expr $powHandX0 + $dx] [expr $powHandY0 +$dy]
    
#make new current_gn

    .pow.pow create rectangle  $ulx_yellow $uly_yellow \
	[expr $lrx_yellow + $dx] [expr $lry_yellow + $dy] \
	-tags  "current_gn graphDragable ${gn}yhandle handle ohandle" \
        -outline yellow -width $yellowLineWidth

    set new_xdim [expr $xfactor * $powGBWidth]
    set new_ydim [expr $yfactor * $powGBHeight]

#make magstep label   
    set sizeText [format "%4d x %4d" [expr round($new_xdim)] \
                                     [expr round($new_ydim)]  ]
    .pow.ms[powCleanName ${gn}]handle configure -text "GraphSize: $sizeText" 

}

proc powBeginStretch {gn x y} {
        
    global powPlotParam ulx_yellow uly_yellow lrx_yellow lry_yellow
    global yellowLineWidth stretchX0 stretchY0 stretchGBWidth stretchGBHeight
    global powHandX0 powHandY0
    global powGBWidth powGBHeight
    global powrootx powrooty

    powSelectGraph $gn
    set bbox [.pow.pow coords current_gn]
    set ulx_yellow [lindex $bbox 0]
    set uly_yellow [lindex $bbox 1]
    set lrx_yellow [lindex $bbox 2]
    set lry_yellow [lindex $bbox 3]

    set handcoords [.pow.pow coords ${gn}shandle]
    
    set powHandX0 [lindex $handcoords 0]
    set powHandY0 [lindex $handcoords 1]

    set gbox [.pow.pow coords ${gn}box]
    
    set powGBWidth [expr [lindex $gbox 2] - [lindex $gbox 0]]
    set powGBHeight [expr [lindex $gbox 3] - [lindex $gbox 1]]

    set powrootx [winfo rootx .pow.pow]
    set powrooty [winfo rooty .pow.pow]

    set stretchX0 [.pow.pow canvasx [expr $x - $powrootx ]]
    set stretchY0 [.pow.pow canvasy [expr $y - $powrooty ]]




}


proc powEndStretch {gn} {
    global powcursor powResizeMain powPlotParam 
    global fixedStretch new_xdim new_ydim

    powStretchGraphToSize $gn $new_xdim $new_ydim
    powSelectGraph $gn
    powReconfigureToplevel $powResizeMain
}


proc powStartNewRow { } {
    global powOpenAreaTop
 
#    update idletasks
    set powOpenAreaTop [lindex [.pow.pow bbox all] 3]
}

proc powInitGraph {gn xMin xMax yMin yMax xunits yunits xLabel yLabel \
		       {canvas ".pow.pow"} \
		       {xDim 600} {yDim 400} \
		       {xDimDisp 600} {yDimDisp 400} {aspect yes} \
		       {xmargin 60} {ymargin 60} } {

# An array of plotting parameters
 global powPlotParam powOpenAreaTop powbg powScopeMargin powFontParam

#puts "gn $gn xMin $xMin xMax $xMax yMin $yMin yMax $yMax xunits $xunits yunits $yunits xLabel $xLabel yLabel $yLabel canvas $canvas xDim $xDim yDim $yDim xpixunit $xpixunit ypixunit $ypixunit aspect $aspect"

#puts "xmin xmax ymin ymax: $xMin $xMax $yMin $yMax"

#####################Plot
set powPlotParam(images,$gn) "NULL"
set powPlotParam(curves,$gn) "NULL"
set powPlotParam(zoomed,$gn) 0
 if {![info exists powOpenAreaTop]} {set powOpenAreaTop 10}
 if { $canvas == ".pow.scope" } {
     set powPlotParam(xo,$gn) $powScopeMargin
     set powPlotParam(yo,$gn) $powScopeMargin
 } elseif {[info exists powPlotParam(xo,$gn)]} {
#   Do nothing, thereby keeping the xo/yo values intact
 } elseif {$canvas == ".pow.pow"} then {
#     update idletasks
     set bbox [.pow.pow bbox all]
     if {$bbox != ""} {
	 set leftSide [lindex $bbox 0]
	 .pow.pow addtag currentRow enclosed $leftSide $powOpenAreaTop \
	     [lindex $bbox 2]  [lindex $bbox 3]
#	 update idletasks
	 set bbox [.pow.pow bbox currentRow]
	 .pow.pow dtag currentRow
	 if {$bbox != ""} then {
	     set powPlotParam(xo,$gn) [expr [lindex $bbox 2] + $xmargin]
	 } else {
	     set powPlotParam(xo,$gn) [expr $leftSide + $xmargin]
	 } 
     } else {
	 set powPlotParam(xo,$gn) $xmargin
     }
     set powPlotParam(yo,$gn) [expr $powOpenAreaTop  + $ymargin ]
 }
 set powPlotParam(graphHeight,$gn) $yDim
 set powPlotParam(graphWidth,$gn) $xDim
 set powPlotParam(xBot,$gn) $xMin
 set powPlotParam(xTop,$gn) $xMax
 set powPlotParam(xunits,$gn) $xunits
 set powPlotParam(xlabel,$gn) $xLabel
 set powPlotParam(yBot,$gn) $yMin
 set powPlotParam(yTop,$gn) $yMax 
 set powPlotParam(yunits,$gn) $yunits
 set powPlotParam(ylabel,$gn) $yLabel
 
###################defaults for optional graph params handled by
#                  powGraphOptions
if {![info exists powPlotParam(bgcolor,$gn)]} {
    set powPlotParam(bgcolor,$gn) $powbg
}
if {![info exists powPlotParam(xmargin,$gn)]} {
    set powPlotParam(xmargin,$gn) $xmargin
}
if {![info exists powPlotParam(ymargin,$gn)]} {
    set powPlotParam(ymargin,$gn) $ymargin
}
if {![info exists powPlotParam(handletext,$gn)]} {
    set powPlotParam(handletext,$gn) "Select/Move: $gn"
}
if {![info exists powPlotParam(handleanchor,$gn)]} {
    set powPlotParam(handleanchor,$gn) "sw"
}
if {![info exists powPlotParam(handleposition,$gn)]} {
    set powPlotParam(handleposition,$gn) "tl"
}
if {![info exists powPlotParam(titleString,$gn)]} {
    set powPlotParam(titleString,$gn) "$gn"
}
if {![info exists powPlotParam(titlePosition,$gn)]} {
    set powPlotParam(titlePosition,$gn) "n"
}
if {![info exists powPlotParam(titleAnchor,$gn)]} {
    set powPlotParam(titleAnchor,$gn) "s"
}

#      Axis tick and grid options...

if {![info exists powPlotParam(GridLines,$gn)]} {
    set powPlotParam(GridLines,$gn) $powPlotParam(GridLines,powDef)
}
if {![info exists powPlotParam(GridColor,$gn)]} {
    set powPlotParam(GridColor,$gn) $powPlotParam(GridColor,powDef)
}
if {![info exists powPlotParam(GridDash,$gn)]} {
    set powPlotParam(GridDash,$gn) $powPlotParam(GridDash,powDef)
}
if {![info exists powPlotParam(xNumTicks,$gn)]} {
    set powPlotParam(xNumTicks,$gn) $powPlotParam(xNumTicks,powDef)
}
if {![info exists powPlotParam(yNumTicks,$gn)]} {
    set powPlotParam(yNumTicks,$gn) $powPlotParam(yNumTicks,powDef)
}
if {![info exists powPlotParam(xTickLength,$gn)]} {
    # order is [lft rgt top bot]
    set powPlotParam(xTickLength,$gn) $powPlotParam(xTickLength,powDef)
}
if {![info exists powPlotParam(yTickLength,$gn)]} {
    # order is [lft rgt top bot]
    set powPlotParam(yTickLength,$gn) $powPlotParam(yTickLength,powDef)
}
if {![info exists powPlotParam(xLabelTicks,$gn)]} {
    # order is [lft rgt top bot]
    set powPlotParam(xLabelTicks,$gn) $powPlotParam(xLabelTicks,powDef)
}
if {![info exists powPlotParam(yLabelTicks,$gn)]} {
    # order is [lft rgt top bot]
    set powPlotParam(yLabelTicks,$gn) $powPlotParam(yLabelTicks,powDef)
}
if {![info exists powPlotParam(tickLabels,$gn)]} {
    set powPlotParam(tickLabels,$gn) $powPlotParam(tickLabels,powDef)
}
if {![info exists powPlotParam(tickFormatCmdX,$gn)]} {
    set powPlotParam(tickFormatCmdX,$gn) $powPlotParam(tickFormatCmdX,powDef)
}
if {![info exists powPlotParam(tickFormatCmdY,$gn)]} {
    set powPlotParam(tickFormatCmdY,$gn) $powPlotParam(tickFormatCmdY,powDef)
}
if {![info exists powPlotParam(xTickScal,$gn)]} {
    set powPlotParam(xTickScal,$gn) $powPlotParam(xTickScal,powDef)
}
if {![info exists powPlotParam(yTickScal,$gn)]} {
    set powPlotParam(yTickScal,$gn) $powPlotParam(yTickScal,powDef)
}
if {![info exists powPlotParam(Notes,$gn)]} {
    set powPlotParam(Notes,$gn) {}
}

#     Text Font Options...

foreach lbl $powFontParam(allTypes,powDef) {
    foreach opt $powFontParam(allOpts,powDef) {
        if { ![info exists powFontParam(${lbl}${opt},$gn)] } {
            set powFontParam(${lbl}${opt},$gn) \
                  $powFontParam(${lbl}${opt},powDef)
        }
    }
}

#     Graph size

if { $xDimDisp == "NULL" } {
   if { ![info exists powPlotParam(xdimdisp,$gn)] } {
      set powPlotParam(xdimdisp,$gn) $powPlotParam(xdimdisp,powDef)
   }
   set xDimDisp $powPlotParam(xdimdisp,$gn)
} else {
   set powPlotParam(xdimdisp,$gn) $xDimDisp
}

if { $yDimDisp == "NULL" } {
   if { ![info exists powPlotParam(ydimdisp,$gn)] } {
      set powPlotParam(ydimdisp,$gn) $powPlotParam(ydimdisp,powDef)
   }
   set yDimDisp $powPlotParam(ydimdisp,$gn)
} else {
   set powPlotParam(ydimdisp,$gn) $yDimDisp
}


if {![info exists powPlotParam(FixedAspect,$gn)]} {
    set powPlotParam(FixedAspect,$gn) $aspect
}

set xmagstep  [expr double($xDimDisp)/$xDim]
set ymagstep  [expr double($yDimDisp)/$yDim]
set newaspect [expr $xmagstep/$ymagstep]

if { $powPlotParam(FixedAspect,$gn) } {
    if { [info exists powPlotParam(xmagstep,$gn)] } {
	set aspect [expr $powPlotParam(xmagstep,$gn) \
		         / $powPlotParam(ymagstep,$gn) ]
	if { $newaspect > $aspect } {
	    set xmagstep [expr $ymagstep*$aspect]
	} else {
	    set ymagstep [expr $xmagstep/$aspect]
	}
    } else {
	if { $xmagstep<$ymagstep } {
	   if { [expr $xmagstep*$yDim] < [expr $yDimDisp/15.0] } {
	      set ymagstep [expr $yDimDisp/15.0/$yDim]
	      set powPlotParam(yNumTicks,$gn) \
		    [expr $powPlotParam(yNumTicks,$gn)/2+1]
	   } else {
	      set ymagstep $xmagstep
	   }
	} else {
	   if { [expr $ymagstep*$xDim] < [expr $xDimDisp/15.0] } {
	      set xmagstep [expr $xDimDisp/15.0/$xDim]
	      set powPlotParam(xNumTicks,$gn) \
		    [expr $powPlotParam(xNumTicks,$gn)/2+1]
	   } else {
	      set xmagstep $ymagstep
	   }
	}
	# Handle special 1D case even better...
	if { $xDim==1 } {
	   set powPlotParam(xNumTicks,$gn) 0
	}
	if { $yDim == 1 } {
	   set powPlotParam(yNumTicks,$gn) 0
	}
    }
}

set powPlotParam(xmagstep,$gn) $xmagstep
set powPlotParam(ymagstep,$gn) $ymagstep
powSetGraphMagstep $gn $xmagstep $ymagstep

#######   End of powInitGraph   ########
}

proc powBuildGraph { gn images curves canvas } {
   global powPlotParam powResizeMain powGUI currgn powOrderedGraphList
   global powcursor powbg  powFirstTimeThroughFlag

   foreach el [list xo yo graphWidth graphHeight xmagstep ymagstep] {
      set $el $powPlotParam($el,$gn)
   }

# Clean the canvas if there was a previous version of this graph
   $canvas delete $gn

# Plot graph box and other niceties

   $canvas  create rectangle $xo $yo \
	 [expr $graphWidth  * $xmagstep + $xo] \
	 [expr $graphHeight * $ymagstep + $yo] \
	 -tags "$gn ${gn}box ${gn}line gbox" -outline black

   if {$canvas == ".pow.pow"} {
	
      # Can't have an image from the previous graph interfering with a new one
       powDeSelectImage

       powAdornGraph $gn $canvas

      .pow.pow delete ${gn}handles
      .pow.pow delete ${gn}shandle
      powMakeGraphHandles $gn

      if {$powFirstTimeThroughFlag} {
	 powReconfigureToplevel 1
	 set powFirstTimeThroughFlag 0
      } else {
	 powReconfigureToplevel $powResizeMain
      }

# Scroll to new graph
      set cbbox [.pow.pow cget -scrollregion]
      set bbox1 [.pow.pow bbox $gn]
      set xloc [expr double( [lindex $bbox1 0]+[lindex $bbox1 2]) \
	    / [lindex $cbbox 2] / 2.0 ]
      set yloc [expr double( [lindex $bbox1 1]+[lindex $bbox1 3]) \
	    / [lindex $cbbox 3] / 2.0 ]
      set xv [.pow.pow xview]
      if {$xloc<[lindex $xv 0] || $xloc>[lindex $xv 1]} {
	 .pow.pow xview moveto [expr double([lindex $bbox1 0]-30) \
	       / [lindex $cbbox 2] ]
      }
      set yv [.pow.pow yview]
      if {$yloc<[lindex $yv 0] || $yloc>[lindex $yv 1]} {
	 .pow.pow yview moveto [expr double([lindex $bbox1 1]-30) \
	       / [lindex $cbbox 3] ]
      }

      if { $gn != $currgn } {
	 # Place a "working" message on graph, update screen, then continue
	 set gMidX [expr 0.5*($graphWidth  * $xmagstep) + $xo]
	 set gMidY [expr 0.5*($graphHeight * $ymagstep) + $yo]
	 .pow.pow create text $gMidX $gMidY \
	       -anchor center -tags deleteMe -text "Building graph..."
	 update idletasks
	 .pow.pow delete deleteMe
      }

      .pow.pow bind $gn <ButtonPress-1> "powSelectGraph $gn"
      .pow.pow bind graphSelect_$gn <ButtonPress-1> "powSelectGraph $gn"
   }

   powPlotImages $gn $images $canvas
   powPlotCurves $gn $curves $canvas
   if { $canvas==".pow.pow" \
         && [lsearch -exact $powOrderedGraphList $gn]==-1 } {
      lappend powOrderedGraphList $gn
   }

   [gNotifications default] postMessage $gn graphHasFinishedDrawing
}

proc powSetCursor { crsr } {
   global powSaveCursor

   if { $crsr == "reset" } {
      set crsr [lindex $powSaveCursor end]
      set powSaveCursor [lreplace $powSaveCursor end end]
   } else {
      lappend powSaveCursor [.pow.pow cget -cursor]
   }
   .pow configure -cursor $crsr
   .pow.pow configure -cursor $crsr
   catch {.pow.scope configure -cursor $crsr}
}

proc powOverlapTest {id {canvas .pow.pow}} {
    set bb [$canvas bbox $id]
    if {$bb != ""} {
	set olap [eval $canvas find overlapping $bb]
    } else {
	return 0
    }
    foreach item $olap {
	set tags [$canvas gettags $item]
	if {$item != $id && !([string match "*handle*" $tags ])} {
	    return 1
	}
    }
    return 0
}


proc powRedrawBox {gn {canvas .pow.pow}} {
    global powPlotParam currimg
    
    foreach el  [array names powPlotParam]  {
	set p1 [lindex [split $el ,] 0]
	set p2 [lindex [split $el ,] 1]
	if { $p2 == $gn } {
	    set $p1 $powPlotParam($p1,$p2)
        }
    }
    
#find corners of new box
    set x0 [lindex [$canvas coords ${gn}box] 0]
    set y0 [lindex [$canvas coords ${gn}box] 3]
    set ccoords [powGraphToCanvas $gn $xTop $yTop $canvas] 
    set x1 [lindex $ccoords 0]
    set y1 [lindex $ccoords 1]
    
#remove previous box    
    $canvas delete ${gn}box

# plot the new box
    
    $canvas  create rectangle $x0 $y1 $x1 $y0 \
	    -tags "$gn gbox ${gn}box ${gn}line" -outline black

}


proc powChangeGrid { {redraw 0} } {
    global powPlotParam currgn

    if { $currgn=="powDef" } {return}

    if {$redraw} {
	powAdornGraph $currgn .pow.pow
    } else {
	.pow.pow itemconfig ${currgn}grid \
	      -fill $powPlotParam(GridColor,$currgn) \
	      -dash $powPlotParam(GridDash,$currgn)
    }
}

proc powContour { } {
    global currimg powRBmin powRBmax powbg powContourParam currgn
    global powDWP

    if { ![info exists currimg] || $currimg=="NULL" || $currimg=="" } {
	tk_messageBox -message "Select a graph with an image first." \
		-title "No Image" -type ok -parent .pow
	return
    }

    set powContourParam(image)    $currimg
    set powContourParam(gn)       $currgn
    set powContourParam(res)      2
    set powContourParam(separate) no
    set powContourParam(nContrs)  10
    set lst [powGetTics $powRBmin($currimg) $powRBmax($currimg) 10 linear]
    set powContourParam(min)      [lindex $lst 0]
    set powContourParam(max)      [lindex $lst end]
    set powContourParam(scale)    linear

    if {[winfo exists ${powDWP}contour]} {destroy ${powDWP}contour}
    powToplevel ${powDWP}contour .pow "-bg $powbg"
    bind ${powDWP}contour <<CloseWindow>> "destroy ${powDWP}contour"

    catch {wm title ${powDWP}contour "Create Contours"}

    button ${powDWP}contour.help -text "Help" \
	    -command {powHelp Contours.html} \
	    -bg $powbg -takefocus 0
    
    label ${powDWP}contour.image -bg $powbg -text "Image:"
    label ${powDWP}contour.currimg -bg yellow -fg black -text $currimg

    label ${powDWP}contour.imgrng -bg $powbg -text "Image Range:"
    frame ${powDWP}contour.imgfrm -bg $powbg
    label ${powDWP}contour.imgfrm.min -bg $powbg -width 10 \
	    -text "$powRBmin($currimg)"
    label ${powDWP}contour.imgfrm.dash -bg $powbg -text " - "
    label ${powDWP}contour.imgfrm.max -bg $powbg -width 10 \
	    -text "$powRBmax($currimg)"
    pack ${powDWP}contour.imgfrm.min -in ${powDWP}contour.imgfrm -side left \
	    -padx 4 -pady 1 -fill x -expand 1
    pack ${powDWP}contour.imgfrm.dash -in ${powDWP}contour.imgfrm -side left \
	    -padx 4 -pady 1
    pack ${powDWP}contour.imgfrm.max -in ${powDWP}contour.imgfrm -side left \
	    -padx 4 -pady 1 -fill x -expand 1

    label ${powDWP}contour.pixrng -bg $powbg -text "Contour Range:"
    frame ${powDWP}contour.pixfrm -bg $powbg
    entry ${powDWP}contour.pixfrm.min -bg $powbg -width 10 \
	    -textvariable powContourParam(min) -takefocus 1
    label ${powDWP}contour.pixfrm.dash -bg $powbg -text " - "
    entry ${powDWP}contour.pixfrm.max -bg $powbg -width 10 \
	    -textvariable powContourParam(max) -takefocus 1
    pack ${powDWP}contour.pixfrm.min -in ${powDWP}contour.pixfrm -side left \
	    -padx 4 -pady 1 -fill x -expand 1
    pack ${powDWP}contour.pixfrm.dash -in ${powDWP}contour.pixfrm -side left \
	    -padx 4 -pady 1
    pack ${powDWP}contour.pixfrm.max -in ${powDWP}contour.pixfrm -side left \
	    -padx 4 -pady 1 -fill x -expand 1

    label ${powDWP}contour.scale -bg $powbg -text "Scale:"
    frame ${powDWP}contour.sclbutt -bg $powbg
    radiobutton ${powDWP}contour.sclbutt.linear -bg $powbg -text Linear \
	    -variable powContourParam(scale) -value linear \
	    -highlightthickness 0 -takefocus 0
    radiobutton ${powDWP}contour.sclbutt.sqrt -bg $powbg -text Sqrt \
	    -variable powContourParam(scale) -value sqrt \
	    -highlightthickness 0 -takefocus 0
    radiobutton ${powDWP}contour.sclbutt.log -bg $powbg -text Log \
	    -variable powContourParam(scale) -value log \
	    -highlightthickness 0 -takefocus 0
    pack ${powDWP}contour.sclbutt.linear -in ${powDWP}contour.sclbutt -side left \
	    -padx 4 -pady 1
    pack ${powDWP}contour.sclbutt.sqrt -in ${powDWP}contour.sclbutt -side left \
	    -padx 4 -pady 1
    pack ${powDWP}contour.sclbutt.log -in ${powDWP}contour.sclbutt -side left \
	    -padx 4 -pady 1

    label ${powDWP}contour.ncntrs -bg $powbg -text "# Contours:"
    frame ${powDWP}contour.ncntrsbutt -bg $powbg
    button ${powDWP}contour.ncntrsbutt.less -bg $powbg -text "<" \
	    -command { incr powContourParam(nContrs) -1 } -takefocus 0
    entry  ${powDWP}contour.ncntrsbutt.numb -bg $powbg \
	    -textvariable powContourParam(nContrs) -width 5 -takefocus 1
    button ${powDWP}contour.ncntrsbutt.more -bg $powbg -text ">" \
	    -command { incr powContourParam(nContrs) 1 } -takefocus 0
    pack ${powDWP}contour.ncntrsbutt.less -in ${powDWP}contour.ncntrsbutt -side left \
	    -pady 1
    pack ${powDWP}contour.ncntrsbutt.numb -in ${powDWP}contour.ncntrsbutt -side left \
	    -pady 1
    pack ${powDWP}contour.ncntrsbutt.more -in ${powDWP}contour.ncntrsbutt -side left \
	    -pady 1
    trace variable powContourParam(nContrs) w { powSetContours }
    trace variable powContourParam(min) w { powSetContours }
    trace variable powContourParam(max) w { powSetContours }
    trace variable powContourParam(scale) w { powSetContours }
	    
    label ${powDWP}contour.clist -bg $powbg -text "Contours:"

    frame ${powDWP}contour.cntrs -bg $powbg
    scrollbar ${powDWP}contour.cntrs.scrolly -orient vertical -takefocus 0 \
	    -command {${powDWP}contour.cntrs.lst yview} -bg $powbg
    text ${powDWP}contour.cntrs.lst -bg $powbg -width 20 -height 5 \
	    -yscrollcommand {${powDWP}contour.cntrs.scrolly set } \
	    -takefocus 0

    grid ${powDWP}contour.cntrs.lst     -in ${powDWP}contour.cntrs -row 1 -column 1 \
	    -sticky news
    grid ${powDWP}contour.cntrs.scrolly -in ${powDWP}contour.cntrs -row 1 -column 2 \
	    -sticky news
    grid rowconfigure    ${powDWP}contour.cntrs 1 -weight 1
    grid columnconfigure ${powDWP}contour.cntrs 1 -weight 1

    label ${powDWP}contour.res -bg $powbg -text "Resolution:"
    frame ${powDWP}contour.resbutt -bg $powbg
    radiobutton ${powDWP}contour.resbutt.high -bg $powbg -text High \
	    -variable powContourParam(res) -value 1 -highlightthickness 0 \
	    -takefocus 0
    radiobutton ${powDWP}contour.resbutt.med -bg $powbg -text Medium \
	    -variable powContourParam(res) -value 2 -highlightthickness 0 \
	    -takefocus 0
    radiobutton ${powDWP}contour.resbutt.low -bg $powbg -text Low \
	    -variable powContourParam(res) -value 3 -highlightthickness 0 \
	    -takefocus 0
    pack ${powDWP}contour.resbutt.high -in ${powDWP}contour.resbutt -side left \
	    -padx 4 -pady 1
    pack ${powDWP}contour.resbutt.med -in ${powDWP}contour.resbutt -side left \
	    -padx 4 -pady 1
    pack ${powDWP}contour.resbutt.low -in ${powDWP}contour.resbutt -side left \
	    -padx 4 -pady 1

    checkbutton ${powDWP}contour.separate -bg $powbg \
	    -text "Place contours in separate graph" \
	    -variable powContourParam(separate) -onvalue yes -offvalue no \
	    -highlightthickness 0 -takefocus 0

    frame ${powDWP}contour.buttons -bg $powbg
    button ${powDWP}contour.buttons.make -text "Make Contours" -bg $powbg \
	    -command { 
	powMakeContours $powContourParam(image) \
                        [${powDWP}contour.cntrs.lst get 1.0 end] \
	                $powContourParam(res)
    }
    button ${powDWP}contour.buttons.exit -text "Exit" -bg $powbg \
	    -command {destroy ${powDWP}contour}
    pack ${powDWP}contour.buttons.make -in ${powDWP}contour.buttons -side left \
	    -padx 4 -pady 3
    pack ${powDWP}contour.buttons.exit -in ${powDWP}contour.buttons -side left \
	    -padx 4 -pady 3

    grid ${powDWP}contour.help -in ${powDWP}contour -row 0 -column 2 -sticky ne
    grid ${powDWP}contour.image -in ${powDWP}contour -row 0 -column 0 -sticky e \
	    -pady 8
    grid ${powDWP}contour.currimg -in ${powDWP}contour -row 0 -column 1 -sticky ew \
	    -pady 8 -padx 6

    grid ${powDWP}contour.imgrng -in ${powDWP}contour -row 2 -column 0 -sticky e
    grid ${powDWP}contour.imgfrm -in ${powDWP}contour -row 2 -column 1 -sticky ew \
	    -columnspan 2
    grid ${powDWP}contour.pixrng -in ${powDWP}contour -row 3 -column 0 -sticky e
    grid ${powDWP}contour.pixfrm -in ${powDWP}contour -row 3 -column 1 -sticky ew \
	    -columnspan 2
    grid ${powDWP}contour.ncntrs -in ${powDWP}contour -row 4 -column 0 -sticky e
    grid ${powDWP}contour.ncntrsbutt -in ${powDWP}contour -row 4 -column 1 -sticky w \
	    -padx 4
    grid ${powDWP}contour.scale -in ${powDWP}contour -row 5 -column 0 -sticky e
    grid ${powDWP}contour.sclbutt -in ${powDWP}contour -row 5 -column 1 -sticky w
    grid ${powDWP}contour.clist -in ${powDWP}contour -row 6 -column 0 -sticky e
    grid ${powDWP}contour.cntrs -in ${powDWP}contour -row 6 -column 1 -sticky news \
	    -padx 5 -columnspan 2
    grid ${powDWP}contour.res -in ${powDWP}contour -row 8 -column 0 -sticky e
    grid ${powDWP}contour.resbutt -in ${powDWP}contour -row 8 -column 1 -sticky w \
	    -pady 8
    grid ${powDWP}contour.separate -in ${powDWP}contour -row 10 -column 0 -sticky ew \
	    -padx 6 -pady 8 -columnspan 3

    grid ${powDWP}contour.buttons -in ${powDWP}contour -row 11 -column 0 -columnspan 3 \
	    -pady 8

    grid columnconfigure ${powDWP}contour 1 -weight 1
    grid rowconfigure ${powDWP}contour 0 -weight 1
    grid rowconfigure ${powDWP}contour 1 -minsize 10
    grid rowconfigure ${powDWP}contour 7 -minsize 10
    grid rowconfigure ${powDWP}contour 6 -weight 1
    grid rowconfigure ${powDWP}contour 11 -weight 1

    powSetContours 0 0 0
}

proc powGetScale { min max scale nlvls } {
    global powDWP
    set offset 0.0
    if { $min<0.0 } {
	set offset [expr -2.0*$min]
	set min [expr $min+$offset]
	set max [expr $max+$offset]
    }
    set min [expr double($min)]
    set max [expr double($max)]
    set list ""

    switch $scale {
	linear {
	    set step [expr ($max-$min) / ($nlvls-1) ]
	    set val $min
	    for {set i 0} {$i<$nlvls} {incr i} {
		lappend list [expr $val-$offset]
		set val [expr $val + $step]
	    }
	}
	sqrt {
	    set step [expr ( sqrt($max) - sqrt($min) ) / ($nlvls-1) ]
	    set val [expr sqrt($min)]
	    for {set i 0} {$i<$nlvls} {incr i} {
		lappend list [expr $val*$val-$offset]
		set val [expr $val + $step]
	    }
	}
	log {
	    if {$min==0.0} {set min [expr 0.001*$max]}
	    set step [expr log( $max / $min ) / ($nlvls-1) ]
	    set val [expr log($min)]
	    for {set i 0} {$i<$nlvls} {incr i} {
		lappend list [expr exp($val)-$offset]
		set val [expr $val + $step]
	    }
	}
	exp {}
    }
    return $list
}

proc powSetContours { a b c } {
    global powContourParam powDWP

    set nContrs $powContourParam(nContrs)
    if { $nContrs == "" } return
    if { !($nContrs > 2) } { set nContrs 2 }
    set powContourParam(list) ""

    set powContourParam(list) \
	    [powGetScale $powContourParam(min) $powContourParam(max) \
                         $powContourParam(scale) $nContrs]

    if { [winfo exists ${powDWP}contour.cntrs.lst] } {
	${powDWP}contour.cntrs.lst delete 1.0 end
	${powDWP}contour.cntrs.lst insert end [join $powContourParam(list) "\n"]
    }
}

proc powMakeContours { img list res } {
    global powContourParam powPlotParam
    global powWCS powCurveParam

    set gn $powContourParam(gn)
    set cntr ${img}_contour

    if { [info exists powWCS($img)] && $powWCS($img)!="" } {
	set powWCS($cntr) $powWCS($img)
    }
    powCreateContour $cntr $img $list $res

    if { $powContourParam(separate) == "yes" } {
	set graph $cntr
	set images NULL
	set curves $cntr
    } else {
	set graph $gn
	set images $powPlotParam(images,$gn)
	set curves $powPlotParam(curves,$gn)
	if {$curves=="NULL"} {
	    set curves $cntr
	} else {
	    lappend curves $cntr
	}
    }

    # Find the true width of the of the graph box
    set width  [tagXdim .pow.pow ${gn}box]
    set height [tagYdim .pow.pow ${gn}box]

    if { [lsearch -exact [powListGraphs] $graph]>=0 } {
#	powUnmapGraph $graph
    }

    set powCurveParam(lStyle${cntr},$graph) " "
    set powCurveParam(lDisp${cntr},$graph) Yes
    set powCurveParam(pDisp${cntr},$graph) No

    set fixed $powPlotParam(FixedAspect,$gn)

    powCreateGraph $graph $curves $images \
	    $powPlotParam(xunits,$gn) $powPlotParam(yunits,$gn) \
	    $powPlotParam(xlabel,$gn) $powPlotParam(ylabel,$gn) \
	    $width $height \
	    $powPlotParam(xBot,$gn) $powPlotParam(yBot,$gn) \
	    $powPlotParam(xTop,$gn) $powPlotParam(yTop,$gn)

    set powPlotParam(FixedAspect,$graph) $fixed
}

proc powAdornGraph {gn {canvas ".pow.pow"}} {
   global powPlotParam

   if {$canvas != ".pow.pow" || $gn=="powDef" } {return}

   foreach par [list xNumTicks yNumTicks GridColor GridDash GridLines \
	             xTickScal yTickScal tickLabels] {
      set $par $powPlotParam($par,$gn)
   }

   if { $tickLabels=="degrees" && [powWCSexists $gn] } {
      # Convert "wcs" scaling to ra/dec to distinguish x/y axes
      set xTickScal "ra"
      set yTickScal "dec"
   }

   #  Make tick frequency non-linear
   set xNumTicks [expr $xNumTicks + int(exp($xNumTicks/3.0)) - 1]
   set yNumTicks [expr $yNumTicks + int(exp($yNumTicks/3.0)) - 1]

   .pow.pow delete ${gn}grid
   set powPlotParam(tickList,$gn) \
	 [powDrawGridLines $gn $canvas $xTickScal $yTickScal \
	                   $GridColor $xNumTicks $yNumTicks $GridDash \
			   $GridLines]

   powDrawTicks $gn $canvas

   powMakeGraphLabels $gn

}

proc powDrawTicks { gn {canvas .pow.pow} } {
   global powPlotParam powFontParam
   global powTicksPerAxis

   foreach par [list xTickLength xLabelTicks yTickLength yLabelTicks \
         xmargin xTickScal yTickScal tickLabels tickFormatCmdX \
		   tickFormatCmdY] {
      set $par $powPlotParam($par,$gn)
   }
   set sideOrder [list lft rgt top bot]

   if { $tickLabels=="degrees" && [powWCSexists $gn] } {
      set xTickScal "ra"
      set yTickScal "dec"
   }

   .pow.pow delete ${gn}ticks ${gn}nums ${gn}label

   foreach axis [list x y] {
      foreach side [list top lft rgt bot none] {
         set powTicksPerAxis($axis$side,$gn) 0
      }
   }

   ################################################################
   #
   #  Analyze tick values to identify required precision on labels
   #

   set xValues {}
   set yValues {}
   foreach {x y val axis side} $powPlotParam(tickList,$gn) {
      if { $axis=="x" } {
         lappend xValues $val
      } elseif { $axis=="y" } {
         lappend yValues $val
      }
   }

   set xValues [lsort -unique -real $xValues]
   set xLabelFmt [powBuildAxisFormat $xValues $xTickScal \
         $powPlotParam(tickFormatCmdX,$gn)]

   set yValues [lsort -unique -real $yValues]
   set yLabelFmt [powBuildAxisFormat $yValues $yTickScal \
         $powPlotParam(tickFormatCmdY,$gn)]

   #
   #
   ################################################################

   foreach {x y val axis side} $powPlotParam(tickList,$gn) {
      incr powTicksPerAxis($axis$side,$gn)
      foreach {x y} [powGraphToCanvas $gn $x $y $canvas] {}
      if {$axis=="x"} {
	 if {$xTickScal=="ra"} {
	    set label [powHourRA $val $xLabelFmt]
            if { [llength $xValues]<2 } {
               # string will be of format xxhxxmxx.xxxxs
               regsub {\.*0*s$} $label "s" label
               regsub {00s$} $label "" label
            } else {
               regsub {X.*$} $label "" label
            }
	 } elseif {$xTickScal=="log"} {
	    set label [eval $xLabelFmt [expr pow(10.0,$val)] ]
	 } else {
	    set label [eval $xLabelFmt $val ]
	 }
      } elseif {$axis=="y"} {
	 if {$yTickScal=="dec"} {
	    set label [powDegDec $val $yLabelFmt]
            if { [llength $yValues]<2 } {
               # string will be of format xx:xx:xx.xxxx
               regsub {(:00)?\.*0*$} $label "" label
            } else {
               regsub {X.*$} $label "" label
            }
	 } elseif {$yTickScal=="log"} {
	    set label [eval $yLabelFmt [expr pow(10.0,$val)] ]
	 } else {
	    set label [eval $yLabelFmt $val ]
	 }   
      }

      switch $side {
         lft {
	    set tckLen [eval lindex \$${axis}TickLength 0]
	    set tckLab [eval lindex \$${axis}LabelTicks 0]
	    if { $tckLen != 0 } {
	       $canvas create line $x $y [expr $x - $tckLen] $y \
              -tags "$gn ${gn}line ${gn}ticks ${gn}lftticks ${gn}${axis}ticks" \
              -fill black
	    }
	    if { $tckLab } {
	       $canvas create text [expr $x - 5 - ($tckLen>0?$tckLen:0)] $y \
              -text $label -anchor e -font [powGetFontList $gn tick] \
              -fill $powFontParam(tickColor,$gn) \
              -tags "$gn ${gn}text ${gn}nums ${gn}lftnums ${gn}${gn}nums"
	    }
         }
         rgt {
	    set tckLen [eval lindex \$${axis}TickLength 1]
	    set tckLab [eval lindex \$${axis}LabelTicks 1]
	    if { $tckLen != 0 } {
	       $canvas create line $x $y [expr $x + $tckLen] $y \
              -tags "$gn ${gn}line ${gn}ticks ${gn}rgtticks ${gn}${axis}ticks" \
              -fill black
	    }
	    if { $tckLab } {
	       $canvas create text [expr $x + 5 + ($tckLen>0?$tckLen:0)] $y \
              -text $label -anchor w -font [powGetFontList $gn tick] \
              -fill $powFontParam(tickColor,$gn) \
              -tags "$gn ${gn}text ${gn}nums ${gn}rgtnums ${gn}${gn}nums"
	    }
         }
         top {
	    set tckLen [eval lindex \$${axis}TickLength 2]
	    set tckLab [eval lindex \$${axis}LabelTicks 2]
	    if { $tckLen != 0 } {
	       $canvas create line $x $y $x [expr $y - $tckLen] \
              -tags "$gn ${gn}line ${gn}ticks ${gn}topticks ${gn}${axis}ticks" \
              -fill black
	    }
	    if { $tckLab } {
	       $canvas create text $x [expr $y - 5 - ($tckLen>0?$tckLen:0)] \
              -text $label -anchor s -font [powGetFontList $gn tick] \
              -fill $powFontParam(tickColor,$gn) \
              -tags "$gn ${gn}text ${gn}nums ${gn}topnums ${gn}${gn}nums"
	    }
         }
         bot {
	    set tckLen [eval lindex \$${axis}TickLength 3]
	    set tckLab [eval lindex \$${axis}LabelTicks 3]
	    if { $tckLen != 0 } {
	       $canvas create line $x $y $x [expr $y + $tckLen] \
              -tags "$gn ${gn}line ${gn}ticks ${gn}botticks ${gn}${axis}ticks" \
              -fill black
	    }
	    if { $tckLab } {
	       $canvas create text $x [expr $y + 5 + ($tckLen>0?$tckLen:0)] \
              -text $label -anchor n -font [powGetFontList $gn tick] \
              -fill $powFontParam(tickColor,$gn) \
              -tags "$gn ${gn}text ${gn}nums ${gn}botnums ${gn}${axis}nums"
	    }
         }
      }
   }
       
    


#   .pow.pow bind ${gn}nums <Enter> \
#         ".pow.pow itemconfigure ${gn}nums -fill yellow"
#   .pow.pow bind ${gn}nums <Leave> \
#         ".pow.pow itemconfigure ${gn}nums -fill black"
   .pow.pow bind ${gn}nums <<DblBtnPress>> \
         "powEditGraphDlg $gn; powEditSelectPage Ticks"
}


proc powMakeGraphLabels { gn {canvas ".pow.pow"} } {
   global powPlotParam powFontParam
   global powTicksPerAxis

   foreach par [list xTickLength xLabelTicks yTickLength yLabelTicks \
         xlabel ylabel xunits yunits titleString titlePosition titleAnchor \
         xmargin xTickScal yTickScal tickLabels] {
      set $par $powPlotParam($par,$gn)
   }

   foreach [list lft top rgt bot] [$canvas coords ${gn}box] {}

# put the X and Y labels 

   if { $xunits=="" || [regexp -nocase NULL $xunits] } {
      set xString "$xlabel"     
   } else {
      set xString "$xlabel ($xunits)"     
   }
   if { $yunits=="" || [regexp -nocase NULL $yunits] } {
      set yString "$ylabel"     
   } else {
      set yString "$ylabel ($yunits)"     
   }

   # Should we swap the Axis labels?
   if { [powWCSisSwapped $gn] && \
         $powTicksPerAxis(xlft,$gn) < $powTicksPerAxis(xbot,$gn) && \
         $powTicksPerAxis(ylft,$gn) > $powTicksPerAxis(ybot,$gn) } {
      set tmp     $xString
      set xString $yString
      set yString $tmp
   } elseif { ![powWCSisSwapped $gn] && \
         $powTicksPerAxis(xlft,$gn) > $powTicksPerAxis(xbot,$gn) && \
         $powTicksPerAxis(ylft,$gn) < $powTicksPerAxis(ybot,$gn) } {
      set tmp     $xString
      set xString $yString
      set yString $tmp
   }
         
   
   set lineSpace [font metrics [powGetFontList $gn axis] -linespace]
   incr lineSpace 5
   set topMarg [powMax [lindex $xTickLength 2] [lindex $yTickLength 2]]
   set botMarg [powMax [lindex $xTickLength 3] [lindex $yTickLength 3]]
   if { $botMarg<0 } {set botMarg 0}
   if { $topMarg<0 } {set topMarg 0}

   if [regexp {[^ ]} $xString] {
      $canvas create text [expr ($lft + $rgt)/2 ] \
	    [expr $bot + $botMarg + $lineSpace] -text $xString -anchor n \
	    -tags "$gn ${gn}label ${gn}xlabel ${gn}text" \
            -font [powGetFontList $gn axis] \
            -fill $powFontParam(axisColor,$gn)
   }

   if [regexp {[^ ]} $yString] {
      $canvas create text [expr $lft - $xmargin/2] \
	    [expr $top - $topMarg] -text $yString -anchor sw \
            -justify left -tags "$gn ${gn}label ${gn}ylabel ${gn}text"\
            -font [powGetFontList $gn axis] \
            -fill $powFontParam(axisColor,$gn)
   }

   
# Now do the titleString

   if [regexp {[^ ]} $titleString] {

      if [string match "*w*" $titlePosition] {
         set x $lft
      } elseif [string match "*e*" $titlePosition] {
         set x $rgt
      } else {
         set x [expr ($lft + $rgt)*0.5]
      }

      if [string match "*n*" $titlePosition] {
         set y [expr $top - $topMarg]
         if { [lindex $xLabelTicks 2] || [lindex $yLabelTicks 2] } {
            set y [expr $y - $lineSpace]
         }
         if [regexp {[^ ]} $yString] {
            set y [expr $y - $lineSpace]
         }
      } elseif [string match "*s*" $titlePosition] {
         set y [expr $bot + $botMarg]
         if { [lindex $xLabelTicks 3] || [lindex $yLabelTicks 3] } {
            set y [expr $y + $lineSpace]
         }
         if [regexp {[^ ]} $xString] {
            set y [expr $y + $lineSpace]
         }
      } else {
         set y [expr ($top + $bot)*0.5]
      }

      $canvas create text $x $y -anchor $titleAnchor -text $titleString \
	    -tags "$gn graphDragable ${gn}label ${gn}tlabel ${gn}text" \
            -font [powGetFontList $gn title] \
            -fill $powFontParam(titleColor,$gn)
   }

#   $canvas bind ${gn}label <Enter> \
#         "$canvas itemconfigure ${gn}label -fill yellow"
#   $canvas bind ${gn}label <Leave> \
#         "$canvas itemconfigure ${gn}label -fill black"
   $canvas bind ${gn}label <<DblBtnPress>> \
         "powEditGraphDlg $gn; powEditSelectPage Graph"

   # Now do any extra graph labels

   powRedrawNotes $gn
}


proc powDummyRangeCallback { gn x0 x1} {
    puts "You have selected the ordered pair: ( $x0 , $x1) on the graph $gn"
}


proc powDragRange { x_or_y {tag highlight} {color red} {callback powDummyRangeCallback}} {
    global currgn powRangeX0 powRangeX1 powRangeX0C powRangeXC 
    global powRangeY0 powRangeY1 powRangeY0C powRangeYC powRangeTag powRangeColor
    global powRangeCallback  powRangeSaveBinding
    
    set powRangeCallback $callback
    set powRangeTag $tag
    set powRangeColor $color
    
    set powRangeSaveBinding(ButtonPress-1) [bind .pow.pow <ButtonPress-1>]
    set powRangeSaveBinding(B1-Motion) [bind .pow.pow <B1-Motion>]
    set powRangeSaveBinding(ButtonRelease-1) [bind .pow.pow <ButtonRelease-1>]
    
    bind .pow.pow <ButtonPress-1> {
	set gn [powWhereAmI %x %y]; 
	if {$gn == $currgn} {
	    set powRangeX0C [.pow.pow canvasx %x];
	    set powRangeY0C [.pow.pow canvasy %y];
	    set gcoords [powCanvasToGraph $currgn $powRangeX0C $powRangeY0C \
			     .pow.pow];
	    set powRangeX0 [lindex gcoords 0]
	    set powRangeY0 [lindex gcoords 1]
	    .pow.pow create line $powRangeX0C $powRangeY0C $powRangeX0C $powRangeY0C \
		-tags Range -fill $powRangeColor
	}
    }
    bind .pow.pow <B1-Motion> {
	set gn [powWhereAmI %x %y]; 
	if {$gn == $currgn} {
	    if {![info exists powRangeX0C]} {
		set powRangeX0C [.pow.pow canvasx %x];
		set powRangeY0C [.pow.pow canvasy %y];
		set gcoords [powCanvasToGraph $currgn $powRangeX0C $powRangeY0C \
			     .pow.pow];
		set powRangeX0 [lindex gcoords 0]
		set powRangeY0 [lindex gcoords 1]
	    } else {
		.pow.pow delete Range; 
	    }
	    set powRangeXC [.pow.pow canvasx %x];
	    set powRangeYC [.pow.pow canvasy %y];
	    .pow.pow create line $powRangeX0C $powRangeY0C $powRangeXC $powRangeYC \
		-tags Range -fill $powRangeColor
	}
    }
    if {$x_or_y == "X"} { 
	bind .pow.pow <ButtonRelease-1> {
	    if {[info exists powRangeX0C]} {
		set range_coords [.pow.pow coords Range]
		.pow.pow delete Range
		set powRangeXC [lindex $range_coords 0];
		set powRangeYC [lindex $range_coords 1];
		set powRangeCoords [powCanvasToGraph $currgn \
                               $powRangeXC $powRangeYC .pow.pow]
		set powRangeX0 [lindex $powRangeCoords 0]
		set powRangeY0 [lindex $powRangeCoords 1]
		set powRangeXC [lindex $range_coords 2];
		set powRangeYC [lindex $range_coords 3];
		set powRangeCoords [powCanvasToGraph $currgn \
                               $powRangeXC $powRangeYC .pow.pow]
		set powRangeX1 [lindex $powRangeCoords 0]
		set powRangeY1 [lindex $powRangeCoords 1]
		powColorRange $currgn X $powRangeX0 $powRangeX1 $powRangeY0 $powRangeY1 $powRangeTag $powRangeColor 0;
		$powRangeCallback $currgn $powRangeX0 $powRangeX1; 
		bind .pow.pow <ButtonPress-1> \
		    "$powRangeSaveBinding(ButtonPress-1)";
		bind .pow.pow <B1-Motion> \
		    "$powRangeSaveBinding(B1-Motion)";
		bind .pow.pow <ButtonRelease-1> \
		    "$powRangeSaveBinding(ButtonRelease-1)";
		#clear start point for next Drag
		unset powRangeX0C
	    }
	}
    } else {
	bind .pow.pow <ButtonRelease-1> {
	    if {[info exists powRangeX0C]} {
		set range_coords [.pow.pow coords Range]
		.pow.pow delete Range
		set powRangeXC [lindex $range_coords 0];
		set powRangeYC [lindex $range_coords 1];
		set powRangeCoords [powCanvasToGraph $currgn \
                               $powRangeXC $powRangeYC .pow.pow]
		set powRangeX0 [lindex $powRangeCoords 0]
		set powRangeY0 [lindex $powRangeCoords 1]
		set powRangeXC [lindex $range_coords 2];
		set powRangeYC [lindex $range_coords 3];
		set powRangeCoords [powCanvasToGraph $currgn \
                               $powRangeXC $powRangeYC .pow.pow]
		set powRangeX1 [lindex $powRangeCoords 0]
		set powRangeY1 [lindex $powRangeCoords 1]
		powColorRange $currgn Y $powRangeX0 $powRangeX1 $powRangeY0 $powRangeY1 $powRangeTag $powRangeColor 0;
		$powRangeCallback $currgn $powRangeY0 $powRangeY1; 
		bind .pow.pow <ButtonPress-1> \
		    "$powRangeSaveBinding(ButtonPress-1)";
		bind .pow.pow <B1-Motion> \
		    "$powRangeSaveBinding(B1-Motion)";
		bind .pow.pow <ButtonRelease-1> \
		    "$powRangeSaveBinding(ButtonRelease-1)";
		#clear start point for next Drag
		unset powRangeX0C
	    }
	}
    }
}

proc powColorRange { gn x_or_y x0 x1 y0 y1 {tag highlight} {color red} {redrawing 0}} {
    global powGraphsTagRangeList powTagsColorMap
    global chainToAxisHash axisToChainHash
#    puts "powColorRange: $gn $x_or_y $x0 $x1 $y0 $y1 $tag $color $redrawing"

    if {$x_or_y == "X"} {
	set a0 $x0
	set a1 $x1
    } else {
	set a0 $y0
	set a1 $y1
    }



    if {$a0 > $a1} {
	set tmp $a0
	set a0 $a1
	set a1 $tmp
    }
    if {!$redrawing} {
	set powTagsColorMap($tag) $color
	lappend powGraphsTagRangeList($gn) "$x_or_y $a0 $a1 $tag"
    }	

    set graphlist ${gn}$x_or_y


    if {$x_or_y =="X"} {
	if {[array names axisToChainHash ${gn}X] != ""} {
	    set graphlist [concat $graphlist $chainToAxisHash($axisToChainHash(${gn}X))]
	}
    } else {
	if {[array names axisToChainHash ${gn}Y] != ""} {
	    set graphlist [concat $graphlist $chainToAxisHash($axisToChainHash(${gn}Y))]
	}
    }
    

    foreach graph [concat $graphlist] {
	set axis [chopped $graph]
	set graph [chop $graph]
	powTagRange $graph  $x_or_y $x0 $x1 $y0 $y1 $tag
    }

    .pow.pow itemconfigure $tag -fill $color
}


proc powTagRange { gn x_or_y x0 x1 y0 y1 tag } {
#    puts "powTagRange: $gn $x_or_y $x0 $x1 $tag"
    set gbox [.pow.pow coords ${gn}box]
    set gx0 [lindex $gbox 0]
    set gx1 [lindex $gbox 2]
    set gy0 [lindex $gbox 1]
    set gy1 [lindex $gbox 3]
    if {$x_or_y == "X"} {
	set xa [lindex [powGraphToCanvas $gn $x0 $y0 .pow.pow] 0]
	set xb [lindex [powGraphToCanvas $gn $x1 $y1 .pow.pow] 0]
	if {($xa < $gx0 && $xb < $gx0) || ($xa > $gx1 && $xb > $gx1)} {
	    #range is entirely off the graph
	    return
	} 
	if {$xa < $gx0} {set xa $gx0 }
	if {$xb > $gx1} {set xb $gx1 }
	set ya $gy0
	set yb $gy1
    } else {
	set xa $gx0
	set xb $gx1
	set ya [lindex [powGraphToCanvas $gn $x1 $y1 .pow.pow] 1]
	set yb [lindex [powGraphToCanvas $gn $x0 $y0 .pow.pow] 1]
	if {($ya < $gy0 && $yb < $gy0) || ($ya > $gy1 && $yb > $gy1)} {
	    #range is entirely off the graph
	    return
	} 
	if {$ya < $gy0} {set ya $gy0 }
	if {$yb > $gy1} {set yb $gy1 }
    }
    .pow.pow addtag $tag enclosed $xa $ya $xb $yb
}

proc powTagRect { gn x0 y0 x1 y1 tag } {
    if {$x0 > $x1} {
	set tmp $x0
	set x0 $x1
	set x1 $tmp
    }
    if {$y0 > $y1} {
	set tmp $y0
	set y0 $y1
	set y1 $tmp
    }
    set ccoords [powGraphToCanvas $gn $x0 $y0 .pow.pow]
    set xa [lindex $ccoords 0]
    set yb [lindex $ccoords 1]
    set ccoords [powGraphToCanvas $gn $x1 $y1 .pow.pow]
    set xb [lindex $ccoords 0]
    set ya [lindex $ccoords 1]
    set gbox [.pow.pow coords ${gn}box]
    set gx0 [lindex $gbox 0]
    set gx1 [lindex $gbox 2]
    set gy0 [lindex $gbox 1]
    set gy1 [lindex $gbox 3]
    if {($xa < $gx0 && $xb < $gx0) || ($xa > $gx1 && $xb > $gx1) || \
	    ($ya < $gy0 && $yb < $gy0) || ($ya > $gy1 && $yb > $gy1)} {
#rect is entirely off the displayed graph
	return
    } 
    if {$xa < $gx0} {set xa $gx0 }
    if {$xb > $gx1} {set xb $gx1 }
    if {$ya < $gy0} {set ya $gy0 }
    if {$yb > $gy1} {set yb $gy1 }
    
    .pow.pow  addtag $tag enclosed $xa $ya $xb $yb
}





proc powDummyRectCallback { gn  x0 y0 x1 y1} {
    puts "You have selected the rectangle: ( $x0 , $y0 , $x1, $y1) on the graph "
}


proc powDragRect { {tag highlight} {color red} {callback powDummyRectCallback}} {
    global currgn powRectX0 powRectX1 powRectX0C powRectXC 
    global powRectY0 powRectY1 powRectY0C powRectYC powRectTag powRectColor
    global powRectCallback powRectSaveBinding

    set powRectCallback $callback
    set powRectTag $tag
    set powRectColor $color

    set powRectSaveBinding(ButtonPress-1) [bind .pow.pow <ButtonPress-1>]
    set powRectSaveBinding(B1-Motion) [bind .pow.pow <B1-Motion>]
    set powRectSaveBinding(ButtonRelease-1) [bind .pow.pow <ButtonRelease-1>]

    
    bind .pow.pow <ButtonPress-1> {
	set gn [powWhereAmI %x %y]; 
	if {$gn == $currgn} {
	    set powRectX0C [.pow.pow canvasx %x];
	    set powRectY0C [.pow.pow canvasy %y];
	    set gcoords [powCanvasToGraph $currgn $powRectX0C $powRectY0C .pow.pow];

	    set powRectX0 [lindex $gcoords 0]
	    set powRectY0 [lindex $gcoords 1]

	    .pow.pow create rectangle $powRectX0C $powRectY0C $powRectX0C $powRectY0C \
		-tags Rect -outline $powRectColor
	}
    }
    bind .pow.pow <B1-Motion> {
	set gn [powWhereAmI %x %y]; 
	if {$gn == $currgn} {
	    if {![info exists powRectX0C]} {
		set powRectX0C [.pow.pow canvasx %x];
		set powRectX0 [powCanvasToGraph $currgn X $powRectX0C .pow.pow];
		set powRectY0C [.pow.pow canvasy %y];
		set powRectY0 [powCanvasToGraph $currgn Y $powRectY0C .pow.pow];
	    } else {
		.pow.pow delete Rect; 
	    }
	    set powRectXC [.pow.pow canvasx %x];
	    set powRectYC [.pow.pow canvasy %y];
	    .pow.pow create rectangle $powRectX0C $powRectY0C $powRectXC $powRectYC \
		-tags Rect -outline $powRectColor
	}
    }
    bind .pow.pow <ButtonRelease-1> {
	if {[info exists powRectX0C]} {
	    set rect_coords [.pow.pow coords Rect]
	    .pow.pow delete Rect
	    set powRectXC [lindex $rect_coords 0]
	    set powRectYC [lindex $rect_coords 1];
	    set gcoords [powCanvasToGraph $currgn $powRectXC $powRectYC .pow.pow];
	    set powRectX0 [lindex $gcoords 0]
	    set powRectY0 [lindex $gcoords 1]

	    set powRectXC [lindex $rect_coords 2]
	    set powRectYC [lindex $rect_coords 3];
	    set gcoords [powCanvasToGraph $currgn $powRectXC $powRectYC .pow.pow]
	    set powRectX1 [lindex $gcoords 0]
	    set powRectY1 [lindex $gcoords 1]
	    powColorRect $currgn $powRectX0 $powRectY0 $powRectX1 $powRectY1 $powRectTag $powRectColor 0;
	    bind .pow.pow <ButtonPress-1> \
		"$powRectSaveBinding(ButtonPress-1)";
	    bind .pow.pow <B1-Motion> \
		"$powRectSaveBinding(B1-Motion)";
	    bind .pow.pow <ButtonRelease-1> \
		"$powRectSaveBinding(ButtonRelease-1)";
	    $powRectCallback $currgn $powRectX0 $powRectY0 $powRectX1 $powRectY1;
	}
    }
}

proc powColorRect { gn  x0 y0 x1 y1 {tag highlight} {color red} {redrawing 0}} {
    global powGraphsTagRectList powTagsColorMap
#    puts "powColorRect: $gn $x_or_y $x0 $x1 $tag $color $redrawing"
    if {$x0 > $x1} {
	set tmp $x0
	set x0 $x1
	set x1 $tmp
    }
    if {!$redrawing} {
	set powTagsColorMap($tag) $color
	lappend powGraphsTagRectList($gn) "$x0 $y0 $x1 $y1 $tag"
    }	
    powTagRect $gn $x0 $y0 $x1 $y1 $tag

# Unless someone can come up with a good reason, powColorRect doesn't follow
# linked axes because it's unclear what to do with the unlinked axis
#    set graphlist $gn

#    if {[array names axisToChainHash ${gn}X] != ""} {
#	set graphlist [concat $graphlist $chainToAxisHash($axisToChainHash(${gn}X))]
#    }
#
#    if {[array names axisToChainHash ${gn}Y] != ""} {
#	set graphlist [concat $graphlist $chainToAxisHash($axisToChainHash(${gn}Y))]
#    }
    
#    foreach graph [concat $graphlist] {
#	set axis [chopped $graph]
#	set graph [chop $graph]
#	powTagRange $graph  $x0 $y0 $x1 $y1 $tag
	
#    }
    .pow.pow itemconfigure $tag -fill $color
}

proc powColorbar { } {
    global currimg currgn powResizeMain
    global powPlotParam powImageParam

    if { [regexp {_colorbar$} $currgn] } {
       tk_messageBox -icon warning \
	     -message "Cannot create colorbar of\nanother colorbar" \
	     -parent .pow -title "Colorbar Warning" -type ok
       return
    }
    if { ![info exists currimg] || $currimg == "" } {
       tk_messageBox -icon warning \
	     -message "Select an image first." \
	     -parent .pow -title "Colorbar Warning" -type ok
       return
    }

    set saveimg $currimg
    set savegn $currgn

    if {[.pow.pow find withtag ${currimg}disp${currgn}] == ""} {
	puts "Your selected image must be on the selected graph to make a colorbar"
	return
    }

    set colorbarGn  ${currgn}_colorbar
    set colorbarImg ${currimg}_colorbar

    set width 2048.0
    set min $powImageParam(RBmin${currimg},$currgn)
    set max $powImageParam(RBmax${currimg},$currgn)
    if { $min==$max } {
       if { $min==0.0 } {
	  set min -1
	  set max 1
       } else {
	  set min [expr $min-abs(0.1*$min)]
	  set max [expr $max+abs(0.1*$max)]
       }
    }
    set increment [expr ($max - $min) / ($width-1)]

    set x $min
    for {set j 0} {$j < $width} {incr j} {
	lappend color_list $x
	set x [expr $x + $increment]
    }

    powCreateDataFromList $colorbarImg $color_list
    
    set zunits [powGetImageUnits $currimg Z]

    powCreateImage $colorbarImg $colorbarImg 0 0 \
	    [expr int($width)] 1 $min $increment 0.5 1.0 \
	    $zunits " " $zunits

    
    set powPlotParam(xo,$colorbarGn) [lindex [.pow.pow bbox ${savegn}box] 0]
    set powPlotParam(yo,$colorbarGn) \
	  [expr 20 + [lindex [.pow.pow bbox $savegn] 3] ]
    set powPlotParam(Colorbar${colorbarImg},$colorbarGn) ${currimg}disp${currgn}
    set powPlotParam(FixedAspect,$colorbarGn) No
    set powPlotParam(FixedAspect,${colorbarGn}scope) No

    set powPlotParam(handletext,$colorbarGn) "$savegn Colorbar"
    set powPlotParam(handleposition,$colorbarGn) bl
    set powPlotParam(handleanchor,$colorbarGn) nw
    set powPlotParam(titleString,$colorbarGn) "$savegn Colorbar"
    set powPlotParam(titlePosition,$colorbarGn) sw
    set powPlotParam(titleAnchor,$colorbarGn) nw
    set powPlotParam(GridLines,$colorbarGn) No
    set powPlotParam(yTickLength,$colorbarGn) [list 0 0 0 0]
    set powPlotParam(yLabelTicks,$colorbarGn) [list No No No No]

    powCreateGraph $colorbarGn NULL $colorbarImg $zunits NULL " " " " \
	    [tagXdim .pow.pow ${currgn}box] 20

#    powGraphOptions $colorbarGn \
#	  handletext "$savegn Colorbar" \
#	  handleposition bl \
#	  handleanchor nw \
#	  GridLines No \
#	  yTickLength [list 0 0 0 0] \
#	  yLabelTicks [list No No No No]

    powReconfigureToplevel $powResizeMain


#reselect original image    
    powSelectGraph $savegn
    powSelectImage $savegn $saveimg
}
    

#  axisValues need to contain unique values in ascending order

proc powBuildAxisFormat { axisValues axisScale defaultFmt } {

   if { [llength $axisValues]<2 } {
      set axisDiff 0
   } else {
      set axisDiff [expr [lindex $axisValues 1] - [lindex $axisValues 0]]
      for { set i 2 } { $i<[llength $axisValues] } { incr i } {
         set j [expr $i-1]
         set diff [expr [lindex $axisValues $i] - [lindex $axisValues $j]]
         if { $diff < $axisDiff } {
            set axisDiff $diff
         }
      }
   }
   switch $axisScale {
      "ra" {
         set labelFmt "%dh"
         if { $axisDiff==0 } {
            append labelFmt "%02dm%07.4fs"
         } else {
            set axisDiff [expr $axisDiff * 3600.0 / 15.0]
            if { $axisDiff < 3600.0 } {
               # Need Minutes
               append labelFmt "%02dm"
               if { $axisDiff < 59.99 } {
                  # Need Seconds
                  if { $axisDiff < .99 } {
                     set axisDiffPrec  [expr int(-log10($axisDiff)+1)]
                     set axisDiffWidth [expr 3+$axisDiffPrec]
                     append labelFmt "%0${axisDiffWidth}.${axisDiffPrec}fs"
                  } else {
                     append labelFmt "%02.0fs"
                  }
               } else {
                  append labelFmt "X%02fs"
               }
            } else {
               append labelFmt "X%02dm%02fs"
            }
         }
      }
      "dec" {
         set labelFmt "%d:%02d"
         if { $axisDiff==0 } {
            append labelFmt ":%07.4f"
         } else {
            set axisDiff [expr $axisDiff * 3600.0]
            if { $axisDiff < 59.99 } {
               # Need Seconds
               if { $axisDiff < 0.99 } {
                  set axisDiffPrec [expr int(-log10($axisDiff)+1)]
                  set axisDiffWidth [expr 3+$axisDiffPrec]
                  append labelFmt ":%0${axisDiffWidth}.${axisDiffPrec}f"
               } else {
                  append labelFmt ":%02.0f"
               }
            } else {
               append labelFmt "X:%02.0f"
            }
         }
      }
      "log" {
         set labelFmt $defaultFmt
      }
      default {
         set labelFmt $defaultFmt
      }
   }

   return $labelFmt
}

# convert a decimal degree to HH MM SS.S
# the optional fmtStr needs to have 3 value placeholders (%'s) in h m s order

proc powHourRA { deciValue {fmtStr "%dh%02dm%05.2fs"} } {
#Written by J. Xu
    if { $deciValue < 0} {
	set deciValue [expr $deciValue + 360]
    }
    set hourValue [expr $deciValue/15.0 + 1e-13]
    set hour [expr int($hourValue)]
    set minuValue [expr ($hourValue - $hour)*60.0]
    set minu [expr int($minuValue)]
    set scndValue [expr ($minuValue - $minu)*60.0]
    set scnd $scndValue
    while {$hour >= 24} {set hour [expr $hour - 24]}
    while {$hour < 0} {set hour [expr $hour + 24]}

    # Check if we are rounding seconds to next value

    set scndFmt [lindex [split $fmtStr %] 3]
    set scndStr [format %$scndFmt $scnd]
    if { [regexp {^ *60} $scndStr] } {
	set scnd 0
	incr minu
	if {$minu == 60} {
           set minu 0
           incr hour
           if { $hour==24 } {
              set hour 0
           }
        }
    }
    return [format $fmtStr $hour $minu $scnd]
}

# convert a decimal degree to DD MM SS.S
# the optional fmtStr needs to have 3 value placeholders (%'s) in h m s order

proc powDegDec { deciValue {fmtStr "%d:%02d:%05.2f"} } {
#Written by J. Xu
    if { $deciValue < 0} {
	set isNeg 1
    } else {
	set isNeg 0
    }
    set deciValue [expr abs($deciValue) + 1e-13]
    set deg [expr int($deciValue)]
    while {$deg > 360} {set deg [expr $deg - 360]}
    while {$deg < -360} {set deg [expr $deg + 360]}
    set minuValue [expr ($deciValue - $deg)*60.0]
    set minu [expr int($minuValue)]
    set scndValue [expr ($minuValue - $minu)*60.0]
    set scnd $scndValue

    # Check if we are rounding seconds to next value

    set scndFmt [lindex [split $fmtStr %] 3]
    set scndStr [format %$scndFmt $scnd]
    if { [regexp {^ *60} $scndStr] } {
	set scnd 0
	incr minu
	if {$minu == 60} {
           set minu 0
           incr deg
           if { $deg==360 } {
              set deg 0
           }
        }
    }
    if { $isNeg } {
	return [format "-$fmtStr" $deg $minu $scnd]
    } else {
	return [format $fmtStr $deg $minu $scnd]
    }
}

proc powLoadFitsImage {url imagename} {
    global powWCS
#While POW, technically, should not know anything about FITS files, 
#loading an image using fitsTcl is *way* too complicated at the moment
#so this routine is provided as a public service

#This routine takes a url or filename and creates the POW Image object
#with the requested name.  The POW Data object is available under the
#name ${imagename}_data

#fitsTcl must be loaded to use this routine and an error will be thrown
#if it isn't

    if {[lsearch [info loaded] *Fitstcl*] == -1} {
	error "You must load fitsTcl to use powLoadFitsImage"
    }


#if you're running under Windows or MacOS, you can't use
#a URL, just a local file name because cfitsio's network drivers don't work
#outside of UNIX.
#open the fits file (readonly)    
    if [catch {set infilehandle [fits open $url 0]}] {
	error "Couldn't open file: $url"
    }

#load the image data into memory
set imghandle [$infilehandle load image]
    
#get the dimensions of the image
set dims [$infilehandle info imgdim]
set n1 [lindex $dims 0]
set n2 [lindex $dims 1]

#get the data type of the image 
set data_type [lindex [lindex [$infilehandle get keyword BITPIX] 0] 1]

#Now a bit of Voodoo to deal with possible wierd file types:

#If the image has BZERO or BSCALE keywords in the header, fitsTcl will
#do the appropriate thing with them automatically, but the datatype returned
#will be floating point doubles (isn't FITS fun:)
if { ([catch {$infilehandle get keyword BZERO}] == 0) ||
     ([catch {$infilehandle get keyword BSCALE}] == 0) } {
    set data_type 4
}

#make a POW DATA object

powCreateData ${imagename}_data $imghandle $data_type [expr $n1 * $n2] 0

#make a POW IMAGE object; the units (pixels, intensity) are arbitrary; since
#this is a general application, we don't know what they are

powCreateImage $imagename ${imagename}_data 0 0 $n1 $n2 0 1 0 1 pixels pixels intensity

#This will setup POW to use the Astronomical coordinate information
#in the file (if there is any) 
global powWCS
if { ! [catch  {$infilehandle get imgwcs} wcsString] } {
    set powWCS($imagename) $wcsString
}

#we're done reading the file now
$infilehandle close


}


proc powExpr { outDataName inputExpression } {
   #  Make sure fitsTcl is loaded first
   if {[lsearch [info loaded] *Fitstcl*] == -1} {
      error "You must load fitsTcl to use powExpr"
   }

   #  Evaluate the expression
   set res [vexpr -ptr -use powExprGetData $inputExpression]

   #  Create the powData item; have it create its own copy of the data
   eval powCreateData $outDataName $res 1

   #  Don't forget to free the pointer returned by vexpr
   fits free [lindex $res 0]
}


namespace eval powEvents {
    variable lastEventWndw ""
    
    proc generate { evt {evtWndw ""} } {
        variable lastEventWndw

        # This evtWndw messiness is necessary due to LinuxPPC's (and others?) problems
        # in tracking the focus when selecting menu items.  Can't tell if it is a
        # Window Manager problem or Tk problem.

        if { $evt == "<<PostMenus>>" } {
            set whn "now"
            set lastEventWndw $evtWndw
        } else {
            set whn "tail"
        }
        if { $evtWndw=="" } {
            if { $lastEventWndw=="" } {
                set evtWndw [focus]
            } else {
                set evtWndw $lastEventWndw
            }
        }
#        puts "Evt: $evt in $evtWndw"
        if { $evtWndw != "" && [winfo exists $evtWndw] } {
            event generate $evtWndw $evt -when $whn
        } else {
            event generate . $evt -when $whn
        }
    }

    proc postMenus { w } {
#      puts "Posting menus for $w"
       if { $w == ".pow" } {
          .pow.mbar.file entryconfig "Close*" -label "Close POW"
       } else {
          .pow.mbar.file entryconfig "Close*" -label "Close Window"
       }
    }

    proc ExitPOW { } {
        destroy .pow
        set Pow_Done 1
    }
    
}
