/* C Mode */

/* htmlgen.c
   Implements html output for FDScript
   Originally implemented by Ken Haase in the Machine Understanding Group
     at the MIT Media Laboratory.

   Copyright (C) 1994-2001 Massachusetts Institute of Technology
   Copyright (C) 2001-2002 beingmeta, inc. (A Delaware Corporation)

   This program comes with absolutely NO WARRANTY, including implied
   warranties of merchantability or fitness for any particular purpose.

    Use, modification, and redistribution of this program is permitted
    under the terms of either (at the developer's discretion) the GNU
    General Public License (GPL) Version 2, the GNU Lesser General Public
    License.

    This program is based on the FramerD library released in Fall 2001 by
    MIT under both the GPL and the LGPL licenses, both of which accompany
    this distribution.  Subsequent modifications by beingmeta, inc. are
    also released under both the GPL and LGPL licenses (at the developer's
    discretion).
*/ 

static char vcid[] = "$Id: htmlgen.c,v 1.32 2002/07/16 15:26:34 haase Exp $";

#define FD_INLINE_STRING_OPS 1
#define FD_INLINE_CHARACTER_OPS 1
#define FD_SOURCE 1
#define HTMLGEN_EXPORT EXPORTED
#include "fdtext.h"
#include "framerd/fdwww.h"

static char *default_doctype=
  "<!DOCTYPE html PUBLIC \"-//W3C//DTD HTML 4.0//EN\">\n";

static enum display_level oid_display_level=get_value;
HTMLGEN_EXPORT fd_lispenv fd_html_env, fd_xml_env;
fd_lispenv fd_html_env, fd_xml_env;

#define EMPTY_TAG_CLOSE(x) (((x != NULL) && (x->is_xml)) ? "/>" : ">")

static fd_lisp tag_slotid;
static fd_lisp name_symbol, value_symbol, size_symbol;
static fd_lisp cols_symbol, rows_symbol, content_symbol;

static void html_puts(fd_u8char *string);
static void html_puts_noescape(fd_u8char *string);

static int is_asciip(fd_u8char *string)
{
  while (*string)
    if (*string < 0x80) return 0;
    else string++;
  return 1;
}

static char *get_ascii(fd_u8char *string)
{
  struct FD_STRING_STREAM out; 
  int c=fd_sgetc(&string);
  FD_INITIALIZE_STRING_STREAM(&out,64);
  while (c >= 0) {
    if (c < 0x80) fd_sputc(&out,c);
    else {
      char buf[8]; sprintf(buf,"\\u%04x",c); fd_sputs(&out,buf);}
    c=fd_sgetc(&string);}
  return out.ptr;
}

/* Generic CGI stream operations */

static void (*_http_puts)(char *s,void *f)=NULL;
static void (*_http_putc)(int c,void *f)=NULL;
static void (*_http_putn)(char *data,int n,void *f)=NULL;

static void http_puts(char *s,fd_htstream *f)
{
  if (f == NULL) fd_fputs_encoded(s,strlen(s),stdout);
  else if (f->stream_type == stdio)
    fd_fputs_encoded(s,strlen(s),f->stream.fileptr);
  else if (f->stream_type == sstream) fd_sputs(f->stream.sstream,s);
  else if (_http_puts) _http_puts(s,f);
  else fd_raise_exception("Weird HTTP stream");
}
static void http_putc(int ch,fd_htstream *f)
{
  if (f == NULL) fputc(ch,stdout);
  else if (f->stream_type == stdio) fputc(ch,f->stream.fileptr);
  else if (f->stream_type == sstream) fd_sputc(f->stream.sstream,ch);
  else if (_http_putc) _http_putc(ch,f);
  else fd_raise_exception("Weird HTTP stream");
}
static void http_printf1s(fd_htstream *f,char *format,char *arg)
{
  if (f == NULL) printf(format,arg);
  else if (f->stream_type == stdio)
    fprintf(f->stream.fileptr,format,arg);
  else if (f->stream_type == sstream)
    fd_printf(f->stream.sstream,format,arg);
  else if (_http_puts) {
    char tmp_buf[512]; sprintf(tmp_buf,format,arg);
    _http_puts(tmp_buf,f);}
  else fd_raise_exception("Weird HTTP stream");
}
static void http_printf1i(fd_htstream *f,char *format,int i)
{
  if (f == NULL) printf(format,i);
  else if (f->stream_type == stdio)
    fprintf(f->stream.fileptr,format,i);
  else if (f->stream_type == sstream)
    fd_printf(f->stream.sstream,format,i);
  else if (_http_puts) {
    char tmp_buf[512]; sprintf(tmp_buf,format,i);
    _http_puts(tmp_buf,f);}
  else fd_raise_exception("Weird HTTP stream");
}
static void http_write_bytes(char *s,int n,fd_htstream *f)
{
  if (f == NULL)
    fwrite(s,sizeof(char),n,stdout);
  else if (f->stream_type == stdio)
    fwrite(s,sizeof(char),n,f->stream.fileptr);
  else if (f->stream_type == sstream)
    fd_sputn(f->stream.sstream,s,n);
  else if (_http_putn) _http_putn(s,n,f);
  else fd_raise_exception("Weird HTTP stream");
}

static void http_puts_quoted(char *string,fd_htstream *f)
{
  char *next=strchr(string,'"');
  while (next) {
    http_write_bytes(string,next-string,f);
    http_puts("&quot;",f);
    string=next+1; next=strchr(string,'"');}
  http_puts(string,f);
}

static void html_write_param
  (fd_lisp tag,lisp value,char *postfix,fd_htstream *f)
{
  fd_lisp lname=fd_xmltag_name(tag), lnamespace=fd_xmltag_namespace(tag);
  fd_u8char *name=((FD_SYMBOLP(lname)) ? (FD_SYMBOL_NAME(lname)) :
		   (FD_STRINGP(lname)) ? (FD_STRING_DATA(lname)) :
		   (fd_type_error(_("bad tag name"),lname),NULL));
  fd_u8char *namespace=
    ((FD_FALSEP(lnamespace)) ? (NULL) :
     (FD_SYMBOLP(lnamespace)) ? (FD_SYMBOL_NAME(lnamespace)) :
     (FD_STRINGP(lnamespace)) ? (FD_STRING_DATA(lnamespace)) : (NULL));
  if (namespace) {
    if (is_asciip(namespace)) http_puts(namespace,f);
    else {
      char *aname=get_ascii(namespace);
      http_puts(aname,f); fd_xfree(aname);}
    http_puts(":",f);}
  if (is_asciip(name)) http_puts(name,f);
  else {
    char *aname=get_ascii(name); http_puts(aname,f); free(aname);}
  http_puts("=\"",f);
  if (STRINGP(value)) 
    if (is_asciip(STRING_DATA(value)))
      http_puts_quoted(STRING_DATA(value),f);
    else {
      char *astring=get_ascii(STRING_DATA(value));
      http_puts_quoted(astring,f); free(astring);}
  else if (SYMBOLP(value))
    if (is_asciip(SYMBOL_NAME(value)))
      http_puts_quoted(SYMBOL_NAME(value),f);
    else {
      char *astring=get_ascii(SYMBOL_NAME(value));
      http_puts_quoted(astring,f); free(astring);}
  else {
    struct FD_STRING_STREAM ss;
    FD_INITIALIZE_STRING_STREAM(&ss,1024); ss.fancy_oids=0;
    fd_print_lisp_to_string(value,&ss);
    if (is_asciip(ss.ptr)) http_puts_quoted(ss.ptr,f);
    else {
      char *astring=get_ascii(ss.ptr);
      http_puts_quoted(astring,f); free(astring);}
    free(ss.ptr);}
  http_puts("\"",f); if (postfix) http_puts(postfix,f);
}

HTMLGEN_EXPORT
/* fd_set_http_output_methods:
     Arguments: three functions 
     Returns: void

  Sets the methods used for HTTP output.  The first function
is for outputting null-terminated strings in their entirety; the
second function is for outputing single characters; and the third function
is for outputing substrings given a start and a length.
*/
void fd_set_http_output_methods
  (void (*_puts)(char *,void *),void (*_putc)(int,void *),
   void (*_putn)(char *,int,void *))
{
  _http_puts=_puts; _http_putc=_putc; _http_putn=_putn;
}

/** State utility functions **/

#define FRAMERD_URL \
  "<A HREF=\"http://www.framerd.org/\">Framer<sub><em><strong>D</strong></em></sub></A>"
#define FRAMERD_CREDIT \
   "<P>This page was dynamically generated by FDScript, the scripting \
language for <A HREF=\"http://www.framerd.org/\">Framer<sub><strong><em>D</em></strong></sub></A> databases and applications."

static lisp html_name_symbol, timestamp_symbol, qmark_symbol, doctype_symbol;
static lisp xmltag_tag;

static lisp obj_name_symbol, anonymous_symbol;
static lisp html_methods_symbol, frame_symbol;
static lisp quote_symbol, current_file_symbol;
static lisp body_style_symbol, head_symbol, body_symbol;

FDSCRIPT_EXPORT lisp fd_get_method(lisp obj,lisp slot);
FDSCRIPT_EXPORT fd_exception fd_SyntaxError;

/** HTTP generation infrastructure **/

#if (FD_USING_THREADS)
#define thget(var,key) fd_tld_get(key)
#define thset(var,key,val) fd_tld_set(key,(void *)val)
static fd_tld_key doctype_key;
static fd_tld_key local_frames_key;
static fd_tld_key http_output_key;
static fd_tld_key http_phase_key;
static fd_tld_key cookie_key;
#else
#define thget(var,key) (var)
#define thset(var,key,val) var=val
static char *doctype=NULL;
static fd_hashset local_frames=NULL;
static struct FD_HTTP_STREAM *http_output=NULL;
static enum http_generation_phase http_phase=http_any;
static char *cookie=NULL;
#endif

/** Looking up browse URLS */

static struct FDWWW_BROWSE_URLS {
  fd_pool pool; fd_u8char *url;
  struct FDWWW_BROWSE_URLS *next;} *browse_urls;

#if (FD_USING_THREADS)
static fd_mutex browse_url_lock;
static char *browse_url=NULL;
#else
static char *browse_url=NULL;
#endif

STATIC_INLINE fd_u8char *get_browse_url(fd_pool p)
{
  struct FDWWW_BROWSE_URLS *scan; char *url;
  fd_lock_mutex(&browse_url_lock);
  if (p) scan=browse_urls; else scan=NULL;
  while (scan)
    if (scan->pool == p) break; else scan=scan->next;
  url=browse_url;
  fd_unlock_mutex(&browse_url_lock);
  if (scan) return scan->url;
  else if (url) return url;
  else return "browse.fdcgi";
}

HTMLGEN_EXPORT
void fd_set_browse_url(fd_u8char *url,fd_pool p)
{
  struct FDWWW_BROWSE_URLS *scan;
  fd_lock_mutex(&browse_url_lock);
  /* A NULL pool means set the default entry */
  if (p == NULL) {
    if (browse_url) fd_xfree(browse_url);
    browse_url=fd_strdup(url);
    fd_unlock_mutex(&browse_url_lock);
    return;}
  else scan=browse_urls;
  /* Find any existing entry */
  while (scan)
    if (scan->pool == p) break; else scan=scan->next;
  /* Change the existing entry or make a new one. */
  if (scan) {
    fd_xfree(scan->url); scan->url=fd_strdup(url);}
  else {
    scan=fd_xmalloc(sizeof(struct FDWWW_BROWSE_URLS));
    scan->url=fd_strdup(url); scan->pool=p;
    scan->next=browse_urls; browse_urls=scan;}
  fd_unlock_mutex(&browse_url_lock);
}

/* Getting HTTP state variables */

struct FD_HTTP_STREAM stdout_htstream;

static fd_htstream *get_http_output()
{
  fd_htstream *tfile=thget(http_output,http_output_key);
  if (tfile) return tfile;
  else {
    stdout_htstream.stream_type=stdio;
    stdout_htstream.is_xml=0;
    stdout_htstream.stream.fileptr=stdout;
    return &(stdout_htstream);}
}

static enum http_generation_phase get_http_phase() 
{
  return (enum http_generation_phase)
    (thget(http_phase,http_phase_key));
}
static char *get_cookie() 
{
  return (char *) (thget(cookie,cookie_key));
}

static fd_hashset get_local_frames()
{
  fd_hashset h=thget(local_frames,local_frames_key);
  if (h) return h; else return NULL;
}

static char *get_doctype() 
{
  return (char *) (thget(doctype,doctype_key));
}

/* Setting HTTP state variables */

void set_http_output(fd_htstream *f)
{
  thset(http_output,http_output_key,f);
}

static void set_http_phase(enum http_generation_phase p)
{
  thset(http_phase,http_phase_key,p);
}

HTMLGEN_EXPORT
/* fd_start_http_output:
     Arguments: a pointer to an fd_htstream
     Returns: void
  Begins output to the designated htstream, setting it
as the thread-local HTTP output stream and setting the initial
phasse of the HTTP output process.
*/
void fd_start_http_output(fd_htstream *s)
{
  set_http_output(s);
  set_http_phase(http_head);
}

HTMLGEN_EXPORT
/* fd_http_puts:
     Arguments: a pointer to a null-terminated string and an FD_HTTP_STREAM struct
     Returns: void
  Writes the string to the stream.
*/
void fd_http_puts(char *s,fd_htstream *stream)
{
  http_puts(s,stream);
}

HTMLGEN_EXPORT
/* fd_http_puts:
     Arguments: a pointer to a null-terminated string and an FD_HTTP_STREAM struct
     Returns: void
  Writes the string to the stream.
*/
void fd_http_write_bytes(char *s,int n,fd_htstream *stream)
{
  http_write_bytes(s,n,stream);
}

static fd_lisp lisp_http_flush_cproc()
{
  struct FD_HTTP_STREAM *hts=get_http_output();
  if ((hts) && (hts->stream_type == stdio)) fflush(hts->stream.fileptr);
  return FD_VOID;
}

void fd_set_cookie(char *cookie)
{
  enum http_generation_phase ph=get_http_phase();
  if (ph == http_any) {
    set_http_phase(http_head); ph=http_head;}
  if (ph == http_head) {
    fd_htstream *out=get_http_output(); 
    http_printf1s(out,"Set-Cookie: %s\n",cookie);}
  else fd_raise_exception("Too late to set cookie");
}

static lisp lisp_set_doctype_cproc(lisp name,lisp location)
{
  enum http_generation_phase ph=get_http_phase();
  if (ph == http_any) {ph=http_head; set_http_phase(ph);}
  if (!(SYMBOLP(name))) fd_type_error("Doctype ID not a symbol",name);
  else if (!(STRINGP(location)))
    fd_type_error("Doctype location not a string",location);
  else if ((ph == http_head) || (ph == html_start)) {
    char *buf=fd_malloc(64+strlen(SYMBOL_NAME(name))+STRING_LENGTH(location));
    char first_char=STRING_DATA(location)[0];
    if ((first_char == '+') || (first_char == '-'))
      sprintf(buf,"<!DOCTYPE %s PUBLIC \"%s\">\n",
	      SYMBOL_NAME(name),STRING_DATA(location));
    else sprintf(buf,"<!DOCTYPE %s SYSTEM \"%s\">\n",
		 SYMBOL_NAME(name),STRING_DATA(location));
    thset(doctype,doctype_key,buf);}
  else fd_raise_exception("Too late to set document type");
  return FD_VOID;
}

/* Adds a frame to the local frames of the current document */
lisp fd_declare_local_frame(lisp frame)
{
  fd_hashset local=get_local_frames();
  if (local == NULL) {
    local=fd_make_hashset(128);
    thset(local_frames,local_frames_key,local);}
  fd_hashset_add(local,frame);
  return FD_TRUE;
}

/** HTTP generation functions **/

static void http_header(char *s)
{
  fd_htstream *out=get_http_output();
  if ((get_http_phase()) == http_head) http_puts(s,out);
  else if ((get_http_phase()) == http_any) {
    set_http_phase(http_head);
    http_puts(s,out);}
  else fd_raise_exception("Too late to specify http headers");
}

static void finish_http_header()
{
  fd_htstream *out=get_http_output();
  if ((get_http_phase()) == http_any) return;
  if ((get_http_phase()) == html_start) return;
  http_puts("\r\n",out);
  set_http_phase(html_start);
}

static void html_header(char *s)
{
  fd_htstream *out=get_http_output();
  enum http_generation_phase ph=(get_http_phase());
  if (ph == http_any) return;
  if ((ph == http_head) || (ph == html_start)) {
    char *doctype=get_doctype();
    if (ph == http_head) finish_http_header();
    if (doctype) http_puts(doctype,out);
    else http_puts(default_doctype,out);
    http_puts("<HTML><HEAD>\n",out);
    thset(http_phase,http_phase_key,html_head);
    ph=html_head;}
  if (ph == html_head) html_puts_noescape(s);
  else fd_raise_exception("Too late to generate HTML head");
}

static void start_body()
{
  fd_htstream *out=get_http_output();
  enum http_generation_phase ph=(get_http_phase());
  if (ph == http_any) return;
  if (ph == html_body) return;
  if ((ph == http_head) || (ph == html_start)) {
    char *doctype=get_doctype();
    finish_http_header();
    if (doctype) http_puts(doctype,out);
    else http_puts(default_doctype,out);
    http_puts("<HTML><HEAD>\n<TITLE>FramerD Generated Page</TITLE>\n",out);
    thset(http_phase,http_phase_key,html_head);}
  if ((get_http_phase()) == html_head) {
    lisp style_info=fd_thread_symeval(body_style_symbol); char *sinfo;
    if (STRINGP(style_info)) sinfo=STRING_DATA(style_info); else sinfo=NULL;
    if (sinfo) http_printf1s(out,"</HEAD>\n<BODY STYLE=\"%s\">\n",sinfo);
    else http_puts("</HEAD>\n<BODY>\n",out);
    decref(style_info);
    thset(http_phase,http_phase_key,html_body);}    
}

/* This really just encodes in latin1, making a parorchial assumption
   which may be neccessary (at this point in history) for URL paths. */
static void uri_encode_local(fd_u8char *input,struct FD_STRING_STREAM *ss)
{
  fd_u8char *scan=input;
  while (*scan)
    if (*scan == ' ') {scan++; fd_sputc(ss,'+');}
    else if ((*scan == '"') ||
	     (*scan == '&') || (*scan == '=') ||
	     (*scan == '+') || (*scan == '#')) {
      char buf[4]; sprintf(buf,"%%%x",*scan++); fd_sputs(ss,buf);}
    else if (*scan >= 0x80) {
      char buf[8]; int c=fd_sgetc(&scan);
      if (c < 0x100) {
	sprintf(buf,"%%%02x",c); fd_sputs(ss,buf);}
      else {
	sprintf(buf,"\\u%04x",c);
	fd_raise_detailed_exception(fd_NoLocalChar,buf);}}
    else {int c=*scan++; fd_sputc(ss,c);}
}

static void uri_encode(fd_u8char *input,struct FD_STRING_STREAM *ss)
{
  fd_u8char *scan=input;
  while (*scan)
    if (*scan == ' ') {scan++; fd_sputc(ss,'+');}
    else if ((*scan == '"') ||
	     (*scan == '&') || (*scan == '=') ||
	     (*scan == '+') || (*scan == '#')) {
      char buf[4]; sprintf(buf,"%%%x",*scan++); fd_sputs(ss,buf);}
    else if (*scan >= 0x80) {
      char buf[8]; int c=fd_sgetc(&scan);
      if (c < 0x100) {
	sprintf(buf,"%%%02x",c); fd_sputs(ss,buf);}
      else {sprintf(buf,"\\u%04x",c); fd_sputs(ss,buf);}}
    else {int c=*scan++; fd_sputc(ss,c);}
}

/* HTML Generation */

#define html_special_char(c) \
   (((c)>=0x80) || (c == '<') || (c == '>') || (c == '&'))
#define oid_start(string) \
   (((*string) == '@') && ((string[1] == '/') | (isxdigit(string[1]))))


static lisp parse_literal_oid_ref(fd_u8char **string);
static void html_display_oid(lisp oid,fd_htstream *out);

/* html_display_string_internal:
     Arguments: a UTF-8 string, an htstream, and two flags
     Returns: void
  This outputs the string to the specified ports, doing 3 special
operations:
    1. HTML special characters (<>&) are emitted as &; escapes
    2. Non-ASCII characters are also emitted as &; escapes
    3. OID references are turned into ANCHORS if possible
 If the first flag argument is non-zero, the conversion to anchors
is not done; if the second flag argument is non zero, spaces will
be output as non-breakable spaces.
*/
static void html_display_string_internal
   (fd_u8char *string,fd_htstream *out,int no_oids,int nbsp)
{
  fd_u8char *scan=string, *start=scan;
  while (*scan) {
    int c;
    if (nbsp)
      while ((*scan) && (*scan != ' ') &&
	     (!(html_special_char(*scan))) &&
	     ((no_oids) || (!(oid_start(scan)))))
	scan++;
    else while ((*scan) && 
		(!(html_special_char(*scan))) &&
		((no_oids) || (!(oid_start(scan)))))
      scan++;
    if (scan > start) http_write_bytes(start,scan-start,out);
    if (*scan == '\0') return;
    if (*scan>=0x80) c=fd_sgetc(&scan); else c=*scan++;
    start=scan;
    switch (c) {
    case '<': http_puts("&lt;",out); break;
    case '>': http_puts("&gt;",out); break;
    case '&': http_puts("&amp;",out); break;
    case ' ':
      if (nbsp) http_puts("&nbsp;",out);
      else http_puts(" ",out);
      break;
    case '@': {
      lisp oid=parse_literal_oid_ref(&scan);
      if (OIDP(oid)) html_display_oid(oid,out);
      else {http_puts("@",out); break;}
      start=scan;
      break;}
    default: http_printf1i(out,"&#%d;",c); break;}}
}

static lisp parse_literal_oid_ref(fd_u8char **string)
{
  FD_OID id;
  fd_u8char *start=*string, tmp_buf[32];
  while (isxdigit(**string)) (*string)++;
  if ((*string == start) || (**string != '/')) {
    *string=start; return FD_VOID;}
  /* Don't run out of space on malformed OIDS */
  if (((*string)-start) >= 32) {*string=start; return FD_VOID;}
  /* Copy the high part and parse it */
  strncpy(tmp_buf,start,(*string)-start); tmp_buf[(*string)-start]=0;
  FD_SET_OID_HIGH(id,strtoul(tmp_buf,NULL,16));
  start=(*string)+1; (*string)++; while (isxdigit(**string)) (*string)++;
  if (*string == start) {*string=start; return FD_VOID;}
  /* Don't run out of space on malformed OIDS */
  if (((*string)-start) >= 32) {*string=start; return FD_VOID;}
  /* Copy the low part and parse it */
  strncpy(tmp_buf,start,(*string)-start); tmp_buf[(*string)-start]=0;
  FD_SET_OID_LOW(id,strtoul(tmp_buf,NULL,16));
  /* Return the OID */
  return fd_make_oid(id);
}

static void html_display_oid(lisp oid,fd_htstream *out)
{
  FD_OID id=FD_OID_ADDR(oid); lisp value; char buf[128];
  fd_hashset local_frames=get_local_frames();
  if ((local_frames) && (fd_hashset_get(local_frames,oid))) 
    sprintf(buf,"<A HREF=\"#%x/%x\">",FD_OID_HIGH(id),FD_OID_LOW(id));
  else sprintf(buf,"<A HREF=\"%s?@%x/%x\">",
	       get_browse_url(FD_GET_POOL(oid)),FD_OID_HIGH(id),FD_OID_LOW(id));
  http_puts(buf,out);
  if (oid_display_level == hide_value) value=FD_VOID;
  else value=fd_oid_value(oid);
  if (SLOTMAPP(value)) {
    lisp name=fd_prim_get(value,html_name_symbol);
    if (FD_EMPTYP(name))
      name=fd_prim_get(value,obj_name_symbol);      
    if (FD_EMPTYP(name)) {
      sprintf(buf,"@%x/%x",FD_OID_HIGH(id),FD_OID_LOW(id));
      http_puts(buf,out);}
    else if (FD_STRINGP(name)) {
      if (FD_STRING_LENGTH(name) < 32)
	html_display_string_internal(FD_STRING_DATA(name),out,1,1);
      else html_display_string_internal(FD_STRING_DATA(name),out,1,0);}
    else {
      fd_u8char *string=fd_object_to_string(name); 
      if (strlen(string) < 32)
	html_display_string_internal(string,out,1,1);
      else html_display_string_internal(string,out,1,0);
      free(string);}
    fd_decref(name);}
  else {
      sprintf(buf,"@%x/%x",FD_OID_HIGH(id),FD_OID_LOW(id));
      http_puts(buf,out);}
  fd_decref(value);
  http_puts("</A>",out);
}

static void html_puts(fd_u8char *string)
{
  enum http_generation_phase phase=get_http_phase();
  if (phase == html_body)
    html_display_string_internal(string,get_http_output(),0,0);
  else html_display_string_internal(string,get_http_output(),1,0);
}

static void html_puts_noescape(fd_u8char *string)
{
  int c; fd_htstream *out=get_http_output();
  while ((c=fd_sgetc(&string))>=0) {
    if (c < 0x100) http_putc(c,out);
    else http_printf1i(out,"&#%d;",c);}
}

static void html_puts_noescape_nobreak(fd_u8char *string)
{
  int c; fd_htstream *out=get_http_output();
  while ((c=fd_sgetc(&string))>=0) {
    if (c == ' ') http_printf1i(out,"&nbsp;",c);
    else if (c < 0x100) http_putc(c,out);
    else http_printf1i(out,"&#%d;",c);}
}

static void html_puts_param(fd_u8char *string)
{
  int c; fd_htstream *out=get_http_output();
  while ((c=fd_sgetc(&string))>=0) {
    if (c == '"') http_puts("&34;",out);
    else if (c < 0x80) http_putc(c,out);
    else http_printf1i(out,"\\u%04x",c);}
}

static void html_printout(lisp args,lispenv env)
{
  int eval=1;
  DOLIST(arg,args) {
    lisp value=((eval) ? (fd_eval_in_env(arg,env)) : (incref(arg)));
    if (FD_VOIDP(value)) {}
    else if (STRINGP(value))
      html_puts_noescape(STRING_DATA(value));
    else {
      struct FD_STRING_STREAM ss;
      FD_INITIALIZE_STRING_STREAM(&ss,1024); ss.fancy_oids=0;
      fd_print_lisp_to_string(value,&ss);
      html_puts(ss.ptr); free(ss.ptr);}
    decref(value);}
}

static void html_write(lisp value)
{
  struct FD_STRING_STREAM ss;
  FD_INITIALIZE_STRING_STREAM(&ss,1024); ss.fancy_oids=0;
  fd_print_lisp_to_string(value,&ss);
  html_puts(ss.ptr); free(ss.ptr);
}

static void html_url_puts(fd_u8char *s,fd_htstream *out)
{
  while (*s) {
    int c=*s++;
    if (c < 0x80) http_putc(c,out);
    else http_printf1i(out,"%%%x",c);}
}

/* Generating HTML commands */

/* Outputs an HTML environment start tag, with particular arguments,
   returning the string for the actual name, converting a name FOO*
   into simply FOO.  The name returned in this way can be used to
   close the environment. */
static void output_envopen
       (char *namestring,lisp params,lispenv params_env)
{
  fd_htstream *out=get_http_output();
  start_body();
  http_printf1s(out,"<%s",namestring);
  if (FD_FALSEP(params)) http_puts(">",out);
  if (FD_EMPTY_LISTP(params)) http_puts(">",out);
  else if (STRINGP(params)) {
    http_putc(' ',out); html_url_puts(STRING_DATA(params),out);
    http_putc('>',out);}
  else if (PAIRP(params)) {
    while (PAIRP(params)) {
      lisp var=fd_get_arg(params,0,FD_VOID);
      if (STRINGP(var)) {
	http_putc(' ',out);
	html_puts_param(STRING_DATA(var));
	http_putc(' ',out);
      	params=CDR(params);}
      else if (SYMBOLP(var)) {
	lisp val_expr=fd_get_arg(params,1,FD_VOID);
	lisp val=fd_eval_in_env(val_expr,params_env);
	params=CDR(CDR(params));
	http_printf1s(out," %s=\"",SYMBOL_NAME(var));
	if (OIDP(val)) {
	  char buf[64];
	  sprintf(buf,"@%x/%x\"",OID_ADDR_HIGH(val),OID_ADDR_LOW(val));
	  http_puts(buf,out);}
	else if (STRINGP(val)) {
	  html_puts_param(STRING_DATA(val)); http_putc('"',out);}
	else {
	  fd_u8char *string=fd_object_to_string(val);
	  html_puts_param(string); http_putc('"',out);
	  free(string); decref(val);}}
      else fd_type_error("not a valid HTML/XML environent parameter",var);}
    http_puts(">",out);}
  else fd_raise_exception("Weird parameter list");
}

char *stripped_env_name(lisp env_name,char *buf)
{
  if (!(SYMBOLP(env_name)))
    fd_raise_lisp_exception
      (fd_SyntaxError,"ENV-NAME must be symbol",env_name);
  else {
    char *namestring=SYMBOL_NAME(env_name);
    int len=strlen(namestring);
    int off=0;
    if (len>100) fd_raise_exception("HTML env name is too long");
    else strcpy(buf,namestring+off);
    if ((len>1) && ((namestring[len-1]) == '*')) {
      buf[(len-off)-1]='\0';
      return buf;}
    else return namestring;}
}

static void output_htmlenv
  (fd_lisp env_symbol,fd_lisp env_params,fd_lisp contents,fd_lispenv env,int newline)
{
  char buf[128], *env_name=stripped_env_name(env_symbol,buf);
  fd_htstream *out=get_http_output();
  UNWIND_PROTECT {
    output_envopen(env_name,env_params,env);
    if (newline) http_putc('\n',out);
    html_printout(contents,env);
    if (newline) http_putc('\n',out);}
  ON_UNWIND
    if (newline) http_printf1s(out,"</%s>\n",env_name);
    else http_printf1s(out,"</%s>",env_name);
  END_UNWIND;
}

/** Top level handlers **/

static lisp html_standalone_handler(lisp expr,lispenv env)
{
  lisp env_name=fd_get_arg(expr,0,FD_FALSE);
  lisp contents=fd_get_body(expr,1);
  char buf[128], *namestring=stripped_env_name(env_name,buf);
  output_envopen(namestring,contents,env);
  return FD_VOID;
}

static lisp html_printout_handler(lisp expr,lispenv env)
{
  lisp env_name=fd_get_arg(expr,0,FD_FALSE);
  lisp contents=fd_get_body(expr,1);
  output_htmlenv(env_name,FD_EMPTY_LIST,contents,env,0);
  return FD_VOID;
}

static lisp html_lineout_handler(lisp expr,lispenv env)
{
  lisp env_name=fd_get_arg(expr,0,FD_FALSE);
  lisp contents=fd_get_body(expr,1);
  output_htmlenv(env_name,FD_EMPTY_LIST,contents,env,1);
  return FD_VOID;
}

static lisp htmlstar_printout_handler(lisp expr,lispenv env)
{
  lisp env_name=fd_get_arg(expr,0,FD_FALSE);
  lisp env_params=fd_get_arg(expr,1,FD_FALSE);
  lisp contents=fd_get_body(expr,2);
  output_htmlenv(env_name,env_params,contents,env,0);
  return FD_VOID;
}

static lisp htmlstar_lineout_handler(lisp expr,lispenv env)
{
  lisp env_name=fd_get_arg(expr,0,FD_FALSE);
  lisp env_params=fd_get_arg(expr,1,FD_FALSE);
  lisp contents=fd_get_body(expr,2);
  output_htmlenv(env_name,env_params,contents,env,1);
  return FD_VOID;
}

static int valid_env_specp(lisp expr)
{
  if ((FD_PAIRP(expr)) && (FD_SYMBOLP(FD_CAR(expr))) &&
      (FD_PAIRP(FD_CDR(expr)))) {
    fd_lisp params=FD_CDR(expr);
    while (FD_PAIRP(params))
      if ((FD_SYMBOLP(params)) && (FD_PAIRP(FD_CDR(params)))) {
	params=FD_CDR(FD_CDR(params));}
      else if ((FD_STRINGP(params)) || (FD_NUMBERP(params)))
	params=FD_CDR(params);
      else return 0;
    if (FD_EMPTY_LISTP(params)) return 1; else return 0;}
  else if ((FD_PAIRP(expr)) && (FD_SYMBOLP(FD_CAR(expr))) &&
	   (FD_EMPTY_LISTP(FD_CDR(expr))))
    return 1;
  else return 0;
}

static lisp markup_handler(lisp expr,lispenv env)
{
  lisp envspec=fd_eval_in_env(fd_get_arg(expr,1,FD_VOID),env);
  lisp env_name, env_params;
  if (FD_SYMBOLP(envspec))  {
    env_name=envspec; env_params=FD_EMPTY_LIST;}
  else if (valid_env_specp(envspec)) {
    env_name=fd_get_arg(envspec,0,FD_VOID);
    env_params=fd_get_body(envspec,1);}
  else fd_type_error(_("Invalid env spec"),envspec);
  output_htmlenv(env_name,env_params,fd_get_body(expr,2),env,1);
  decref(envspec);
  return FD_VOID;
}

static lisp htmlenv_handler(lisp expr,lispenv env)
{
  lisp envspec=fd_get_arg(expr,1,FD_VOID);
  lisp env_name, env_params;
  if (FD_SYMBOLP(envspec)) {
    env_name=envspec; env_params=FD_EMPTY_LIST;}
  if (valid_env_specp(envspec)) {
    env_name=fd_get_arg(envspec,0,FD_VOID);
    env_params=fd_get_body(envspec,1);}
  else fd_type_error(_("Invalid env spec"),envspec);
  output_htmlenv(env_name,env_params,fd_get_body(expr,2),env,1);
  return FD_VOID;
}

static lisp output_empty_tag(lisp args,char *prefix,char *suffix)
{
  lisp env_name=CAR(args), params=CDR(args); fd_u8char *envstring;
  fd_htstream *out=get_http_output();
  if (FD_SYMBOLP(env_name)) envstring=FD_SYMBOL_NAME(env_name);
  else if (FD_STRINGP(env_name)) envstring=FD_STRING_DATA(env_name);
  else fd_type_error(_("Element name is not a string or symbol"),
		     env_name);
  start_body();
  http_puts(prefix,out); http_printf1s(out,"%s",envstring);
  if (!(FD_EMPTY_LISTP(params))) http_puts(" ",out);
  while (PAIRP(params))
    if (STRINGP(CAR(params))) {
      http_printf1s(out," %s ",STRING_DATA(CAR(params)));
      params=CDR(params);}
    else if (SYMBOLP(CAR(params))) {
      lisp prop=CAR(params), value=fd_car_noref(CDR(params));
      params=CDR(CDR(params));
      if (PAIRP(params))
	html_write_param(prop,value," ",out);
      else html_write_param(prop,value,"",out);}
    else fd_type_error("XMLTAG arg",CAR(params));
  http_puts(suffix,out);
  return FD_VOID;
}

static lisp htmltag_lexpr(lisp args)
{
  return output_empty_tag(args,"<",">");
}

static lisp xmltag_lexpr(lisp args)
{
  return output_empty_tag(args,"<","/>");
}

static lisp xmlpi_lexpr(lisp args)
{
  return output_empty_tag(args,"<?","?>");
}

static lisp interpret_variable(lisp expr,lispenv env)
{
  if (STRINGP(expr)) return fd_make_symbol(STRING_DATA(expr));
  else return fd_eval_in_env(expr,env);
}

static lisp html_submit_handler(lisp expr,lispenv env)
{
  lisp variable=interpret_variable(fd_get_arg(expr,1,FD_VOID),env);
  lisp value=fd_eval_in_env(fd_get_arg(expr,2,FD_FALSE),env);
  fd_htstream *out=get_http_output();
  if (!(SYMBOLP(variable))) fd_raise_exception("CGI-SUBMIT needs symbol for var name");
  http_puts("<INPUT TYPE=SUBMIT ",out); 
  if (FD_FALSEP(value)) {
    html_write_param(name_symbol,variable,"",out);
    http_puts(EMPTY_TAG_CLOSE(out),out);}
  else {
    html_write_param(name_symbol,variable," ",out);
    html_write_param(value_symbol,value,EMPTY_TAG_CLOSE(out),out);}
  decref(value);
  return FD_VOID;
}

static lisp html_checkbox_handler(lisp expr,lispenv env)
{
  lisp variable=interpret_variable(fd_get_arg(expr,1,FD_VOID),env);
  lisp value=fd_eval_in_env(fd_get_arg(expr,2,FD_VOID),env);
  lisp checked=fd_eval_in_env(fd_get_arg(expr,3,FD_VOID),env);
  lisp body=fd_get_body(expr,4);
  fd_htstream *out=get_http_output();
  if (!((SYMBOLP(variable)) || (STRINGP(variable))))
    fd_raise_exception("CGI-CHECKBOX needs symbol for var name");
  http_puts("<INPUT TYPE=CHECKBOX ",out);
  html_write_param(name_symbol,variable," ",out);
  html_write_param(value_symbol,value,"",out);
  if (!(FD_FALSEP(checked))) http_puts(" CHECKED>",out);
  else http_puts(EMPTY_TAG_CLOSE(out),out);
  decref(value); decref(checked);
  html_printout(body,env);
  return FD_VOID;
}

static lisp html_radiobutton_handler(lisp expr,lispenv env)
{
  lisp variable=interpret_variable(fd_get_arg(expr,1,FD_VOID),env);
  lisp value=fd_eval_in_env(fd_get_arg(expr,2,FD_VOID),env);
  lisp checked=fd_eval_in_env(fd_get_arg(expr,3,FD_VOID),env);
  lisp body=fd_get_body(expr,4);
  fd_htstream *out=get_http_output();
  if (!(SYMBOLP(variable)))
    fd_raise_exception("CGI-RADIOBUTTON needs symbol for var name");
  http_puts("<INPUT TYPE=RADIO ",out);
  html_write_param(name_symbol,variable," ",out);
  html_write_param(value_symbol,value,"",out);
  if (!(FD_FALSEP(checked))) http_puts(" CHECKED>",out);
  else http_puts(EMPTY_TAG_CLOSE(out),out);
  decref(value); decref(checked);
  html_printout(body,env);
  return FD_VOID;
}

static lisp html_cgipass_handler(lisp expr,lispenv env)
{
  int l=fd_list_length(expr);
  lisp variable=interpret_variable(fd_get_arg(expr,1,FD_VOID),env), value;
  fd_htstream *out=get_http_output();
  if (!(SYMBOLP(variable))) fd_raise_exception("CGIPASS needs symbol for var name");
  if (l == 2) value=fd_symeval(variable,env);
  else if (l == 3) value=fd_eval_in_env(fd_get_arg(expr,2,FD_VOID),env);
  else fd_raise_exception("CGIPASS Syntax error");
  http_puts("<INPUT TYPE=HIDDEN ",out);
  html_write_param(name_symbol,variable," ",out);
  html_write_param(value_symbol,value,EMPTY_TAG_CLOSE(out),out);
  decref(value);
  return FD_VOID;
}

static lisp html_textfield_handler(lisp expr,lispenv env)
{
  lisp variable=interpret_variable(fd_get_arg(expr,1,FD_VOID),env);
  lisp size=fd_eval_in_env(fd_get_arg(expr,2,LISPFIX(40)),env);
  lisp initial_contents=fd_eval_in_env(fd_get_arg(expr,3,FD_FALSE),env);
  fd_htstream *out=get_http_output();
  if (STRINGP(variable)) variable=fd_make_symbol(STRING_DATA(variable));
  if (!(SYMBOLP(variable))) fd_raise_exception("CGI-TEXTINPUT needs symbol for var name");
  http_puts("<INPUT TYPE=TEXT ",out); 
  html_write_param(name_symbol,variable," ",out);
  html_write_param(size_symbol,size," ",out);
  if (FD_FALSEP(initial_contents)) http_puts(EMPTY_TAG_CLOSE(out),out);
  else html_write_param
	 (value_symbol,initial_contents,EMPTY_TAG_CLOSE(out),out);
  decref(initial_contents);
  return FD_VOID;
}

static lisp html_textarea_handler(lisp expr,lispenv env)
{
  lisp variable=interpret_variable(fd_get_arg(expr,1,FD_VOID),env);
  lisp cols=fd_eval_in_env(fd_get_arg(expr,2,LISPFIX(40)),env);
  lisp rows=fd_eval_in_env(fd_get_arg(expr,3,LISPFIX(8)),env);
  lisp initial_contents=fd_eval_in_env(fd_get_arg(expr,4,FD_FALSE),env);
  fd_htstream *out=get_http_output();
  if (STRINGP(variable)) variable=fd_make_symbol(STRING_DATA(variable));
  if (!(SYMBOLP(variable))) fd_raise_exception("CGI-TEXTINPUT needs symbol for var name");
  http_puts("<TEXTAREA ",out); 
  html_write_param(name_symbol,variable," ",out);
  html_write_param(cols_symbol,cols," ",out);
  html_write_param(rows_symbol,rows," ",out);
  http_puts(">",out);
  if (!(FD_FALSEP(initial_contents))) {
    http_puts(fd_strdata(initial_contents),out);
    http_puts("</TEXTAREA>\n",out);}
  else http_puts("</TEXTAREA>\n",out);
  decref(initial_contents);
  return FD_VOID;
}

static lisp html_selection_handler(lisp expr,lispenv env)
{
  lisp head=fd_get_arg(expr,1,FD_VOID);
  lisp body=fd_get_body(expr,2);
  output_htmlenv(fd_make_symbol("SELECT"),head,body,env,1);
  return FD_VOID;
}

static lisp html_option_handler(lisp expr,lispenv env)
{
  lisp arg=fd_eval_in_env(fd_get_arg(expr,1,FD_VOID),env);
  lisp body=fd_get_body(expr,2);
  fd_htstream *out=get_http_output();
  http_puts("<OPTION ",out);
  html_write_param(value_symbol,arg,"",out);
  http_puts(EMPTY_TAG_CLOSE(out),out);
  html_printout(body,env);
  return FD_VOID;
}


/* Declaring HTML functions */

void fd_add_html_tag(char *name)
{
  char *primary=fd_xmalloc(strlen(name)+2+5);
  char *alternate=fd_xmalloc(strlen(name)+2+5);
  sprintf(primary,"%s",name);
  sprintf(alternate,"%s*",name);
  fd_add_special_form(fd_html_env,primary,html_printout_handler);
  fd_add_special_form(fd_html_env,alternate,htmlstar_printout_handler);
}

void fd_add_html_line_tag(char *name)
{
  char *primary=fd_xmalloc(strlen(name)+2+5);
  char *alternate=fd_xmalloc(strlen(name)+2+5);
  sprintf(primary,"%s",name);
  sprintf(alternate,"%s*",name);
  fd_add_special_form(fd_html_env,primary,html_lineout_handler);
  fd_add_special_form(fd_html_env,alternate,htmlstar_lineout_handler);
}

void fd_add_htmlenv_tag(char *name)
{
  char *primary=fd_xmalloc(strlen(name)+2+5);
  sprintf(primary,"%s",name);
  fd_add_special_form(fd_html_env,primary,htmlstar_printout_handler);
}

void fd_add_htmlenv_line_tag(char *name)
{
  char *primary=fd_xmalloc(strlen(name)+2+5);
  sprintf(primary,"%s",name);
  fd_add_special_form(fd_html_env,primary,htmlstar_lineout_handler);
}

void fd_add_html_standalone_tag(char *name)
{
  char *primary=fd_xmalloc(strlen(name)+2+5);
  sprintf(primary,"%s",name);
  fd_add_special_form(fd_html_env,primary,html_standalone_handler);
}


/* Anchor expressions */

/* For simple anchor expressions, the anchor argument can be either:
     + a string (indicating a URL)
     + a symbol (indicating an internal tag)
     + an OID (indicating a recursive browse)
*/
static void generate_anchor(lisp ref,u8char *target,lisp body,fd_lispenv env)
{
  fd_htstream *out=get_http_output();
  if (target) {
    if (FD_OIDP(ref)) {
      FD_OID id=FD_OID_ADDR(ref); char buf[128];
      sprintf(buf,"<A HREF=\"%s?@%x/%x\" TARGET=\"%s\">",
	      get_browse_url(FD_GET_POOL(ref)),FD_OID_HIGH(id),FD_OID_LOW(id),target);
      http_puts(buf,out);}
    else if (STRINGP(ref)) {
      http_printf1s(out,"<A HREF=\"%s\"",STRING_DATA(ref));
      http_printf1s(out," TARGET=\"%s\">",target);}
    else if (SYMBOLP(ref)) {
      http_printf1s(out,"<A HREF=\"#%s\"",SYMBOL_NAME(ref));
      http_printf1s(out," TARGET=\"%s\">",target);}
    else fd_type_error("not an anchor object",ref);}
  else {
    if (FD_OIDP(ref)) {
      FD_OID id=FD_OID_ADDR(ref); char buf[128];
      sprintf(buf,"<A HREF=\"%s?@%x/%x\">",
	      get_browse_url(FD_GET_POOL(ref)),FD_OID_HIGH(id),FD_OID_LOW(id));
      http_puts(buf,out);}
    else if (STRINGP(ref)) {
      http_printf1s(out,"<A HREF=\"%s\">",STRING_DATA(ref));}
    else if (SYMBOLP(ref)) {
      http_printf1s(out,"<A HREF=\"#%s\">",SYMBOL_NAME(ref));}
    else fd_type_error(_("not an anchor object"),ref);}
  html_printout(body,env);
  http_puts("</A>",out);
}

static lisp htmlanchor(lisp expr,lispenv env)
{
  lisp ref=fd_eval_in_env(fd_get_arg(expr,1,FD_VOID),env);
  lisp body=fd_get_body(expr,2);
  start_body();
  generate_anchor(ref,NULL,body,env);
  decref(ref);
  return FD_VOID;
}

static lisp htmltanchor(lisp expr,lispenv env)
{
  lisp target=fd_eval_in_env(fd_get_arg(expr,1,FD_VOID),env);
  lisp ref=fd_eval_in_env(fd_get_arg(expr,2,FD_VOID),env);
  lisp body=fd_get_body(expr,3);
  start_body();
  if (FD_STRINGP(target))
    generate_anchor(ref,FD_STRING_DATA(target),body,env);
  else if (FD_SYMBOLP(target))
    generate_anchor(ref,FD_SYMBOL_NAME(target),body,env);
  else fd_type_error(_("target is not a string or a symbol"),target);
  decref(ref); decref(target);
  return FD_VOID;
}


/* Forms for tags, images, inclusion, and code */

static lisp htmltag(lisp expr,lispenv env)
{
  fd_lisp head=fd_eval_in_env(fd_get_arg(expr,1,FD_VOID),env);
  fd_lisp tag, params;
  if (FD_SYMBOLP(head)) tag=fd_copy_string(FD_SYMBOL_NAME(head));
  else if (FD_STRINGP(head)) tag=fd_incref(head);
  else if (FD_OIDP(head)) {
    fd_u8char buf[32]; sprintf(buf,"%x/%x",OID_ADDR_HIGH(head),OID_ADDR_LOW(head));
    tag=fd_make_string(buf);}
  params=FD_MAKE_LIST(2,fd_make_symbol("NAME"),tag);
  output_htmlenv(fd_make_symbol("A"),params,fd_get_body(expr,2),env,1);
  decref(params); decref(head);
  return FD_VOID;
}

static lisp htmlimage(lisp expr,lispenv env)
{
  lisp body=fd_get_body(expr,2);
  lisp params=FD_MAKE_PAIR
    (fd_make_symbol("SRC"),
     FD_MAKE_PAIR(fd_eval_in_env(fd_get_arg(expr,1,FD_VOID),env),
		  incref(body)));
  output_envopen("IMG",params,env);
  decref(params);
  return FD_VOID;
}

static lisp htmlinclude(lisp expr,lispenv env)
{
  FILE *f;
  lisp filename=fd_get_arg(expr,1,FD_VOID);
  char *fname=fd_strdata(filename), *data;
  if (strchr(fname,':')) {
    fd_lisp retrieved=fd_urlget(fname);
    if (FD_STRINGP(retrieved))
      data=fd_strdup(FD_STRING_DATA(retrieved));
    else if (FD_SLOTMAPP(retrieved)) {
      fd_lisp content=fd_prim_get(retrieved,fd_make_symbol("CONTENT"));
      if (FD_STRINGP(content)) {
	data=fd_strdup(FD_STRING_DATA(retrieved)); fd_decref(content);}}
    else return retrieved;
    fd_decref(retrieved);}
  else {
    if (!((*fname == '/') || (*fname == '\\'))) {
      lisp root=fd_thread_symeval(current_file_symbol); 
      if (STRINGP(root)) {
	int new_size=STRING_LENGTH(filename)+STRING_LENGTH(root)+1;
	char *scan=STRING_DATA(root), *read=scan, *sep=scan;
	char *new=fd_xmalloc(sizeof(char)*new_size), *write=new;
	while (*read) {
	  if ((*read == '/') || (*read == '\\')) sep=write;
	  *write++=*read++;}
	write=sep+1; read=fname;
	while (*read) *write++=*read++; *write++='\0';
	decref(root);
	fname=new;}}
    f=fd_fopen(fname,"r");
    if (f == NULL)
      fd_raise_lisp_exception(fd_Cant_Read_File,fname,filename);
    else {
      char buf[1024];
      struct FD_STRING_STREAM ss;
      FD_INITIALIZE_STRING_STREAM(&ss,1024);
      while ((fgets(buf,1024,f)) != NULL) fd_sputs(&ss,buf);
      data=ss.ptr;}}
  http_puts(data,get_http_output());
  free(data);
  return FD_VOID;
}

static lisp htmlcodeblock(lisp expr,lispenv env)
{
  lisp lwidth=fd_eval_in_env(fd_get_arg(expr,1,FD_VOID),env);
  lisp contents=fd_eval_in_env(fd_get_arg(expr,2,FD_VOID),env);
  fd_htstream *out=get_http_output();
  int width=fd_fixlisp(lwidth);
  struct FD_STRING_STREAM ss; 
  FD_INITIALIZE_STRING_STREAM(&ss,1024);
  fd_pprint_lisp_to_string(contents,&ss,0,0,width); 
  http_puts("<PRE>\n",out);
  html_puts(ss.ptr);
  http_puts("\n</PRE>\n",out);
  free(ss.ptr); decref(contents); decref(lwidth);
  return FD_VOID;
}


/** Describing OIDs and frames in HTML **/

static void html_slot_row(lisp frame,lisp slot,lisp value)
{
  fd_htstream *out=get_http_output();
  http_puts
    ("<TR><TH VALIGN=TOP ALIGN=RIGHT STYLE='font-family: sans-serif;'>",out);
  if (SYMBOLP(slot)) {
    http_puts("<STRONG>",out);
    html_puts(SYMBOL_NAME(slot));
    http_puts("</STRONG>",out);}
  else html_write(slot);
  if (FD_EMPTYP(value)) 
    http_puts("</TH><TD><EM>No Values</EM></TD></TR>\n",out);
  else {
    int first_value=1;
    http_puts
      ("</TH><TD VALIGN=TOP ALIGN=LEFT STYLE='font-family: fixed;'>",out);
    {DO_CHOICES(v,value) {
      if (first_value) first_value=0; 
      else if ((OIDP(v)) || (FD_PAIRP(v)) || (FD_VECTORP(v)))
	http_puts("<BR>",out); 
      else http_puts(" . ",out);
      if ((FD_PAIRP(v)) || (FD_VECTORP(v))) {
	fd_u8char *string=fd_ppstring(v,100);
	html_puts(string); fd_xfree(string);}
      else html_write(v);}
    END_DO_CHOICES;}}
  http_puts("</TD></TR>\n",out);
}

static void html_frame_description(lisp frame)
{
  fd_htstream *out=get_http_output();
  lisp slotmap=fd_oid_value(frame);
  char buf[256];
  http_puts("<TABLE BORDER=3>\n",out);

  /* Output the caption */
  
  sprintf(buf,"<TR><TH COLSPAN=2 ALIGN=LEFT><font size=\"+2\"><A NAME=\"%x/%x\">The frame ",
	  OID_ADDR_HIGH(frame),OID_ADDR_LOW(frame));
  http_puts(buf,out);
  sprintf(buf,"@%x/%x",OID_ADDR_HIGH(frame),OID_ADDR_LOW(frame));
  http_puts(buf,out);

  sprintf(buf," has %d slots</A></FONT></TH></TR>\n",
	  (SLOTMAP_PTR(slotmap))->size);
  http_puts(buf,out);

  fd_for_slots(html_slot_row,frame);

  http_puts("</TABLE>\n",out);

  fd_decref(slotmap);
}

void fd_describe_oid_in_html(lisp frame)
{
  fd_htstream *out=get_http_output();
  lisp value=fd_oid_value(frame);
  if (get_http_phase() != html_body) {
    html_header("<TITLE>\nDescription of frame ");
    html_write(frame);
    html_header("</TITLE>");
    start_body();}
  if (SLOTMAPP(value)) {
    lisp html_methods=fd_overlay_get(frame,html_methods_symbol);
    FD_WITH_LEXICAL_ENV(html_env,NULL,2) {
      fd_bind_value(frame_symbol,frame,html_env);
      if (!(FD_EMPTYP(html_methods))) {
	DO_CHOICES(script,html_methods) {
	  DOLIST(method,script) {
	    lisp value=fd_eval_in_env(method,html_env); decref(value);}
	  http_puts("<HR>\n",out);;}
	END_DO_CHOICES;}
      else html_frame_description(frame);}
    FD_END_WITH_LEXICAL_ENV_NOVALUE();}
  else {
    char buf[128];
    sprintf
      (buf,"<P><A NAME=\"%x/%x\">The value of %x/%x</A> is</P></A>\n",
       OID_ADDR_HIGH(frame),OID_ADDR_LOW(frame),
       OID_ADDR_HIGH(frame),OID_ADDR_LOW(frame));
    http_puts(buf,out); html_write(value);}
  fd_decref(value);
}

static lisp describe_oid_in_html(lisp obj)
{
  fd_describe_oid_in_html(obj);
  return FD_VOID;
}

/** Top level stuff **/

static lisp htmlexpr(lisp expr,lispenv env)
{
  DOLIST(e,fd_get_body(expr,1)) {
    lisp val=fd_eval_in_env(e,env);
    if (STRINGP(val)) html_puts_noescape(STRING_DATA(val));
    else if (FD_VOIDP(val)) {}
    else {
      fd_htstream *out=get_http_output();
      enum http_generation_phase phase=get_http_phase();
      struct FD_STRING_STREAM ss;
      FD_INITIALIZE_STRING_STREAM(&ss,1024); ss.fancy_oids=0;
      fd_print_lisp_to_string(val,&ss);
      if (phase == html_body)
	html_display_string_internal(ss.ptr,out,0,0);
      else html_display_string_internal(ss.ptr,out,1,0);
      free(ss.ptr);}
    decref(val);}
  return FD_VOID;
}
    
static lisp htmlexpr_nobreak(lisp expr,lispenv env)
{
  DOLIST(e,fd_get_body(expr,1)) {
    lisp val=fd_eval_in_env(e,env);
    if (STRINGP(val))
      html_puts_noescape_nobreak(STRING_DATA(val));
    else if (FD_VOIDP(val)) {}
    else {
      fd_htstream *out=get_http_output();
      enum http_generation_phase phase=get_http_phase();
      struct FD_STRING_STREAM ss;
      FD_INITIALIZE_STRING_STREAM(&ss,1024); ss.fancy_oids=0;
      fd_print_lisp_to_string(val,&ss);
      if (phase == html_body)
	html_display_string_internal(ss.ptr,out,0,1);
      else html_display_string_internal(ss.ptr,out,1,1);
      free(ss.ptr);}
    decref(val);}
  return FD_VOID;
}
    
static lisp lisp_http_header_cproc(lisp name,lisp content)
{
  fd_htstream *out=get_http_output();  
  if ((get_http_phase() == http_any)) set_http_phase(http_head);
  if ((get_http_phase()) != http_head)
    fd_raise_exception("HTTP-HEADER: too late to specify headers");
  if (SYMBOLP(name)) http_puts(SYMBOL_NAME(name),out);
  else if (ASCII_STRINGP(name)) http_puts(STRING_DATA(name),out);
  else fd_type_error("HTTP-HEADER: Not a valid header",name);
  http_puts(": ",out); 
  if (STRINGP(content)) http_puts(STRING_DATA(content),out);
  else fd_type_error("HTTP-HEADER: Not valid header content",content);
  http_puts("\n",out);
  return FD_VOID;
}

static lisp lisp_html_insert_head_cproc(lisp string)
{
  html_header(fd_strdata(string));
  return FD_VOID;
}

static lisp lisp_html_start_body_cproc(lisp string)
{
  set_http_phase(html_body);
  html_puts_noescape(fd_strdata(string));
  return FD_VOID;
}

static lisp html_title_cproc(lisp expr,lispenv env)
{
  html_header("<TITLE>");
  html_printout(fd_get_body(expr,1),env);
  html_header("</TITLE>\n");
  return FD_VOID;
}

static lisp html_meta_cproc(lisp name,lisp content)
{
  fd_htstream *out=get_http_output();  
  http_puts("<META NAME=\"",out);
  if (STRINGP(name)) http_puts(STRING_DATA(name),out);
  else if (SYMBOLP(name)) http_puts(SYMBOL_NAME(name),out);
  else fd_type_error("HTML META name not a string or symbol",name);
  http_puts("\" ",out);
  html_write_param(content_symbol,content,">\n",out);
  return FD_VOID;
}

static lisp html_stylesheet_cproc(lisp arg)
{
  char *buf=fd_xmalloc(fd_strlen(arg)+128);
  sprintf(buf,"<LINK REL='stylesheet' TYPE='text/css' HREF='%s'>\n",
	  fd_strdata(arg));
  html_header(buf); free(buf);
  return FD_VOID;
}

HTMLGEN_EXPORT
/* fd_start_http:
     Arguments: a string containing a mime type specification
     Returns: void
  Outputs a content-type header field, including a charset specifier
which refers to the current default character encoding.
*/
void fd_start_http(char *mime)
{
  struct FD_TEXT_ENCODING *enc=fd_get_default_encoding(); char buf[128];
  set_http_phase(http_head);
  sprintf(buf,"Content-type: %s; charset=%s;\r\n\r\n",mime,enc->names[0]);
  http_header(buf);
}

static lisp httpdoc(lisp expr,lispenv env)
{
  struct FD_TEXT_ENCODING *enc=fd_get_default_encoding(); char buf[128];
  set_http_phase(http_head);
  sprintf(buf,"Content-type: text/html; charset=%s;\r\n",enc->names[0]);
  http_header(buf);
  html_printout(fd_get_body(expr,1),env);
  if (get_http_phase() != html_end) {
    html_puts_noescape("\n</BODY>\n</HTML>");
    set_http_phase(html_end);}
  return FD_VOID;
}

static lisp write_html_file_handler(lisp expr,lispenv env)
{
  lisp destination=fd_eval_in_env(fd_get_arg(expr,1,FD_VOID),env);
  char *filename=fd_localize_utf8
    (fd_strdata(destination),fd_get_default_encoding());
  struct FD_HTTP_STREAM hts;
  FILE *f=fd_fopen(filename,"w");
  if (f == NULL) fd_raise_detailed_exception(fd_FileOpenWFailed, filename);
  set_http_phase(html_start);
  hts.stream_type=stdio; hts.is_xml=0; hts.stream.fileptr=f;
  set_http_output(&hts);
  html_printout(fd_get_body(expr,2),env);
  if (get_http_phase() != html_end) {
    html_puts_noescape("\n</BODY></HTML>\n");
    set_http_phase(html_end);}
  fclose(f); set_http_output(NULL);
  free(filename); decref(destination);
  return FD_VOID;
}

static lisp htmlstring_handler(lisp expr,lispenv env)
{
  struct FD_STRING_STREAM ss; struct FD_HTTP_STREAM hts;
  FD_INITIALIZE_STRING_STREAM(&ss,8192);
  hts.stream_type=sstream; hts.is_xml=0; hts.stream.sstream=&ss;
  set_http_output(&hts); set_http_phase(http_any);
  html_printout(fd_get_body(expr,1),env);
  set_http_output(NULL);
  return fd_init_string(ss.ptr,ss.size);
}

static lisp htmlfragment_handler(lisp expr,lispenv env)
{
  struct FD_STRING_STREAM ss; struct FD_HTTP_STREAM hts;
  FD_INITIALIZE_STRING_STREAM(&ss,8192);
  hts.stream_type=sstream; hts.is_xml=0; hts.stream.sstream=&ss;
  set_http_output(&hts); set_http_phase(html_body);
  html_printout(fd_get_body(expr,1),env);
  set_http_phase(html_end);
  set_http_output(NULL);
  return fd_init_string(ss.ptr,ss.size);
}

static lisp http_redirect_cproc(lisp url)
{
  if (STRINGP(url)) {
    http_header("Location: ");
    http_header(STRING_DATA(url));
    http_header("\n");
    return FD_VOID;}
  else fd_type_error("HTTP-REDIRECT: URL not a string",url);
}

static lisp html_redirect_handler(lisp expr,lispenv env)
{
  struct FD_TEXT_ENCODING *enc=fd_get_default_encoding();
  lisp destination=fd_eval_in_env(fd_get_arg(expr,1,FD_VOID),env);
  lisp mime_type=fd_eval_in_env(fd_get_arg(expr,2,FD_VOID),env);  
  char *buf=fd_strdata(destination);
  set_http_phase(http_head);
  if (STRINGP(mime_type)) {
    char buf[512];
    sprintf(buf,"Content-type: %s\n",STRING_DATA(mime_type));
    http_header(buf);}
  else {
    char tbuf[128];
    sprintf(tbuf,"Content-type: text/html; charset=%s;\n",enc->names[0]);
    http_header(tbuf);}
  http_header("Location: "); http_header(buf); http_header("\n");
  html_printout(fd_get_body(expr,3),env);
  decref(destination); decref(mime_type);
  if (get_http_phase() != html_end) {
    html_puts_noescape("\n</BODY>\n");
    set_http_phase(html_end);}
  return FD_VOID;
}
    
static lisp http_splash_lexpr(lisp args)
{
  fd_lisp url, wait;
  fd_get_args("HTTP-SPLASH",args,&url,FD_VOID,&wait,FD_LISPFIX(10),NULL);
  if (!(FIXNUMP(wait)))
    fd_type_error("HTTP-SPLASH: WAIT not a fixnum",wait);
  else if (!(STRINGP(url)))
    fd_type_error("HTTP-SPLASH: URL not a string",url);
  else {
    char buf[64]; sprintf(buf,"Refresh: #%d; URL=",FIXLISP(wait));
    http_header(buf);
    http_header(STRING_DATA(url));
    http_header("\n");
    return FD_VOID;}
}

static lisp html_splash_handler(lisp expr,lispenv env)
{
  struct FD_TEXT_ENCODING *enc=fd_get_default_encoding(); char tbuf[128];
  char buf[64];
  lisp interval=fd_eval_in_env(fd_get_arg(expr,1,FD_VOID),env);
  lisp destination=fd_eval_in_env(fd_get_arg(expr,2,FD_VOID),env);
  set_http_phase(http_head);
  sprintf(buf,"Refresh: #%d; URL=",fd_lisp2int(interval));
  sprintf(tbuf,"Content-type: text/html; charset=%s;\n",enc->names[0]); http_header(tbuf);
  http_header(buf); http_header(fd_strdata(destination));
  http_header("\n");
  html_printout(fd_get_body(expr,3),env);
  decref(interval); decref(destination);
  if (get_http_phase() != html_end) {
    html_puts_noescape("\n</BODY>\n");
    set_http_phase(html_end);}
  return FD_VOID;
}
    
static lisp http_return_file(lisp mime_type,lisp filename)
{
  if (!(STRINGP(mime_type)))
    fd_type_error("mime-type is not a string",mime_type);
  else {
    fd_htstream *out=get_http_output();
    FILE *f=fd_fopen(STRING_DATA(filename),"rb");
    int bytes_read, bytes_total=0;
    char buf[1024];
    set_http_phase(http_head);
    if (f == NULL)
      fd_raise_lisp_exception
	("No Such File","for HTTP return file",filename);
    fseek(f,0,SEEK_END); bytes_total=ftell(f); fseek(f,0,SEEK_SET);
    http_printf1s(out,"Content-type: %s\r\n",STRING_DATA(mime_type));
    http_printf1i(out,"Content-length: %d\r\n",bytes_total);
    finish_http_header();
    while (((bytes_read=fread(buf,sizeof(unsigned char),1024,f)) > 0) ||
	   (!(feof(f))))
      http_write_bytes(buf,bytes_read,out);
    fclose(f); 
    return FD_VOID;}
}

static lisp http_return_data(lisp mime_type,lisp data)
{
  if (!(STRINGP(mime_type)))
    fd_type_error("mime-type is not a string",mime_type);
  else {
    fd_htstream *out=get_http_output();
    int size;
    if (FD_STRINGP(data)) size=FD_STRING_LENGTH(data);
    else if (FD_PACKETP(data)) size=FD_PACKET_LENGTH(data);
    else fd_type_error(_("not a string or packet"),data);
    set_http_phase(http_head);
    http_printf1s(out,"Content-type: %s\r\n",STRING_DATA(mime_type));
    http_printf1i(out,"Content-length: %d\r\n",size);
    finish_http_header();
    if (FD_STRINGP(data))
      http_write_bytes(FD_STRING_DATA(data),size,out);
    else http_write_bytes(FD_PACKET_DATA(data),size,out);
    return FD_VOID;}
}


/* Invoking remote scripts from FDScript */

static lisp index_url(lisp index_name,lisp index_key)
{
  if (!(STRINGP(index_name)))
    fd_raise_exception("index name isn't string");
  else if (STRINGP(index_key)) {
    struct FD_STRING_STREAM ks;
    FD_INITIALIZE_STRING_STREAM(&ks,256);
    uri_encode_local(STRING_DATA(index_name),&ks); fd_sputs(&ks,"?");
    uri_encode(STRING_DATA(index_key),&ks);
    return fd_init_string(ks.ptr,ks.size);}
  else if (OIDP(index_key)) {
    FD_OID id=FD_OID_ADDR(index_key); char buf[32];
    struct FD_STRING_STREAM ks;
    FD_INITIALIZE_STRING_STREAM(&ks,256);
    uri_encode_local(STRING_DATA(index_name),&ks); 
    sprintf(buf,"?@%x/%x",FD_OID_HIGH(id),FD_OID_LOW(id)); fd_sputs(&ks,buf);
    return fd_init_string(ks.ptr,ks.size);}
  else fd_raise_exception("Invalid index arg");
}

static lisp scripturl(lisp args)
{
  lisp script_name=fd_get_arg(args,0,FD_VOID);
  lisp script_args=fd_get_body(args,1);
  struct FD_STRING_STREAM qs;
  if (!(STRINGP(script_name))) 
    fd_raise_exception("Script name should be string");
  else {
    FD_INITIALIZE_STRING_STREAM(&qs,256);
    qs.fancy_oids=0; qs.escape=0;
    uri_encode_local(STRING_DATA(script_name),&qs);
    fd_sputs(&qs,"?");}
  if (FD_EMPTY_LISTP(script_args))
    return fd_init_string(qs.ptr,qs.size);
  else if (!(PAIRP(script_args)))
    fd_raise_exception("Weird args to SCRIPTURL");
  else if (FD_EMPTY_LISTP(CDR(script_args)))
    if (OIDP(CAR(script_args))) {
      FD_OID id=FD_OID_ADDR(CAR(script_args));
      char buf[32]; sprintf(buf,"@%x/%x",FD_OID_HIGH(id),FD_OID_LOW(id));
      uri_encode(buf,&qs);}
    else if (STRINGP(CAR(script_args))) {
      uri_encode_local(STRING_DATA(CAR(script_args)),&qs);}
    else  {
      struct FD_STRING_STREAM vs;
      FD_INITIALIZE_STRING_STREAM(&vs,256);
      vs.fancy_oids=0; vs.escape=1;
      fd_print_lisp_to_string(CAR(script_args),&vs);
      uri_encode(vs.ptr,&qs); free(vs.ptr);}
  else while (PAIRP(script_args)) {
    lisp query_var=fd_get_arg(script_args,0,FD_VOID);
    lisp query_val=fd_get_arg(script_args,1,FD_VOID);
    uri_encode(fd_symbol_name(query_var),&qs);
    fd_sputs(&qs,"=");
    if (OIDP(query_val)) {
      char buf[32];
      sprintf(buf,"@%x/%x",OID_ADDR_HIGH(query_val),OID_ADDR_LOW(query_val));
      fd_sputs(&qs,buf);}
    else if (STRINGP(query_val)) {
      uri_encode(STRING_DATA(query_val),&qs);}
    else {
      struct FD_STRING_STREAM vs;
      FD_INITIALIZE_STRING_STREAM(&vs,256);
      vs.fancy_oids=0; vs.escape=1;
      fd_print_lisp_to_string(query_val,&vs);
      uri_encode(vs.ptr,&qs); free(vs.ptr);}
    fd_sputs(&qs,"&");
    script_args=CDR(CDR(script_args));}
  return fd_init_string(qs.ptr,qs.size);
}

static lisp uri_encode_string(lisp string)
{
  struct FD_STRING_STREAM ss;
  FD_INITIALIZE_STRING_STREAM(&ss,64);
  uri_encode(fd_strdata(string),&ss);
  return fd_init_string(ss.ptr,ss.size);
}

/* Accessing parsed markup */

static int xml_emptyp(fd_lisp x)
{
  lisp content=fd_xml_content(x);
  if (FD_PAIRP(content))
    if (FD_EMPTY_LISTP(FD_CAR(content))) return 1;
    else return 0;
  else return 1;
}

static int html_emptyp(fd_lisp x)
{
  lisp content=fd_xml_content(x);
  if (FD_PAIRP(content)) return 0;
  else return 1;
}

/* Generating from parses */

static void output_tag
  (fd_u8char *prefix,fd_lisp tag_arg,fd_lisp attributes,fd_u8char *suffix,
   fd_htstream *hs)
{
  fd_lisp tag;
  http_puts(prefix,hs);
  if (FD_OIDP(tag_arg)) tag=fd_prim_get(tag_arg,tag_slotid);
  else tag=fd_incref(tag_arg);
  if (FD_SYMBOLP(tag)) 
    http_puts(FD_SYMBOL_NAME(tag),hs);
  else if (FD_STRINGP(tag))
    http_puts(FD_STRING_DATA(tag),hs);
  else if (FD_LRECORD_TYPEP(tag,xmltag_tag)) {
    fd_lisp ns=fd_xmltag_namespace(tag);
    fd_lisp name=fd_xmltag_name(tag);
    if (FD_SYMBOLP(ns)) {
      http_puts(FD_SYMBOL_NAME(ns),hs);
      http_puts(":",hs);}
    if (FD_STRINGP(name))
      http_puts(FD_STRING_DATA(name),hs);
    else if (FD_SYMBOLP(name))
      http_puts(FD_SYMBOL_NAME(name),hs);}
  else fd_type_error(_("Invalid XML tag"),tag_arg);
  {DOLIST(attr,attributes)
     if (STRINGP(attr)) {
       http_puts(" ",hs); http_puts(FD_STRING_DATA(attr),hs);}
     else if (PAIRP(attr)) {
       lisp var=FD_CAR(attr), val=fd_car_noref(FD_CDR(attr));
       http_puts(" ",hs); 
       html_write_param(var,val,"",hs);}
     else fd_raise_exception("Weird attribute");}
  http_puts(suffix,hs);       
}

static void unparse_xml_oid(fd_lisp oid,fd_lispenv env,fd_htstream *hs)
{
  fd_lisp tag=fd_xml_tag(oid);
  fd_lisp attributes=fd_xml_attributes(oid);
  if ((hs->is_xml == 0) && (html_emptyp(oid)))
    output_tag("<",tag,attributes,">",hs);
  else if (xml_emptyp(oid)) 
    output_tag("<",tag,attributes,EMPTY_TAG_CLOSE(hs),hs);
  else {
    fd_lisp content=fd_xml_content(oid);
    output_tag("<",tag,attributes,">",hs);
    {DOLIST(elt,content) fd_unparse_xml(elt,env,hs);}
    if (FD_SYMBOLP(tag)) 
      http_printf1s(hs,"</%s>",fd_symbol_name(tag));
    else output_tag("</",tag,FD_EMPTY_LIST,">",hs);}  
  fd_decref(tag); fd_decref(attributes);
}

/* We're breaking an abstraction barrier here by
   accessing the module slot of the env structure.
   It may mean more work if we change how environments are
   implemented. */
static fd_lisp get_exported(fd_lisp sym,fd_lispenv env)
{
  fd_lispenv scan=env; struct FD_MODULE *m=scan->module; 
  while ((m == NULL) && (scan != NULL)) {
    if (scan=scan->parent) m=scan->module;}
  if ((m) && (fd_hashset_get(&(m->exports),sym)))
    return fd_symeval(sym,env);
  else return FD_VOID;
}

static int get_handler(fd_lisp tag,fd_lispenv env,fd_lisp *handler)
{
  fd_lisp ns=fd_xmltag_namespace(tag);
  fd_lisp name=fd_xmltag_name(tag), namesym, v;
  fd_lispenv lookup_env;
  if (FD_FALSEP(ns)) lookup_env=env;
  else if (FD_SYMBOLP(ns)) {
    lookup_env=fd_get_module(FD_SYMBOL_NAME(ns),NULL,0);}
  else lookup_env=NULL;
  if (lookup_env == NULL) return 0;
  if (FD_SYMBOLP(name)) namesym=name;
  else if (FD_STRINGP(name))
    namesym=fd_intern(FD_STRING_DATA(name),FD_STRING_LENGTH(name));
  else return 0;
  /* Now do the lookup */
  v=get_exported(namesym,lookup_env);
  if (FD_VOIDP(v)) return 0;
  else if (FD_SPECIAL_FORMP(v)) return 0;
  else {*handler=v; return 1;}
}

HTMLGEN_EXPORT
/* fd_unparse_xml:
     Arguments: an xml element rep, an environment, and an http stream
     Returns: void

 Outputs a text representation of the xml/html structure represented by the first
  argument.  Any tags which have bindings in the environment are interpreted as function
  calls.  If the html flag is true, the generation process will try and generate
  HTML rather than XML. */
void fd_unparse_xml(fd_lisp expr,fd_lispenv env,fd_htstream *hs)
{
  if (STRINGP(expr)) http_puts(STRING_DATA(expr),hs);
  else if (PAIRP(expr)) {
    fd_lisp tag=fd_xml_tag(expr);
    if (FD_LISP_EQ(tag,qmark_symbol)) {
      fd_lisp attributes=fd_xml_attributes(expr);
      output_tag("<?",FD_CAR(attributes),FD_CDR(attributes),"?>",hs);
      fd_decref(attributes);}
    else if (FD_LISP_EQ(tag,doctype_symbol)) {
      lisp root=fd_get_arg(expr,1,FD_VOID);
      lisp sysarg=fd_get_arg(expr,2,FD_VOID);
      lisp uri=fd_get_arg(expr,3,FD_VOID);
      lisp idtd=fd_get_arg(expr,4,FD_VOID);
      http_puts("<!DOCTYPE ",hs);
      http_puts(fd_symbol_name(root),hs); http_puts(" ",hs);
      http_puts(fd_strdata(sysarg),hs); http_puts(" ",hs);
      http_printf1s(hs,"\"%s\"",fd_strdata(uri));
      if (!(FD_EMPTY_LISTP(idtd)))
	http_printf1s(hs," [%s] ",fd_strdata(uri));
      http_puts(">",hs);}
    else if ((SYMBOLP(tag)) ||
	     (FD_LRECORD_TYPEP(tag,xmltag_tag))) {
      fd_lisp handler=FD_VOID;
      fd_lisp attributes=fd_xml_attributes(expr);
      if (get_handler(tag,env,&handler)) {
	fd_lisp args, value;
	args=FD_MAKE_LIST1(fd_incref(expr));
	value=fd_apply(handler,args);
	value=fd_finish_value(value);
	fd_decref(handler);
	if (FD_VOIDP(value)) {}
	else if (FD_STRINGP(value)) {
	  http_puts(STRING_DATA(value),hs);}
	else {
	  fd_u8char *string=fd_object_to_string(value);
	  http_puts(string,hs); fd_xfree(string);}
	fd_decref(value); fd_decref(args);
	/* If we have a binding and run it, we just return; otherwise,
	   we go on and otuput the XML in the normal way. */
	return;}
      if (FD_LISP_EQ(tag,body_symbol)) set_http_phase(html_body);
      if (FD_LISP_EQ(tag,head_symbol)) set_http_phase(html_head);
      if (((hs == NULL) || (hs->is_xml == 0)) && (html_emptyp(expr)))
	output_tag("<",tag,attributes,EMPTY_TAG_CLOSE(hs),hs);
      else if (xml_emptyp(expr)) 
	output_tag("<",tag,attributes,EMPTY_TAG_CLOSE(hs),hs);
      else {
	fd_lisp content=fd_xml_content(expr);
	output_tag("<",tag,attributes,">",hs);
	{DOLIST(elt,content) fd_unparse_xml(elt,env,hs);}
	if (FD_SYMBOLP(tag)) 
	  http_printf1s(hs,"</%s>",fd_symbol_name(tag));
	else output_tag("</",tag,FD_EMPTY_LIST,">",hs);}
      if (FD_LISP_EQ(tag,body_symbol)) set_http_phase(html_end);
      fd_decref(attributes);}
    else {
      DOLIST(item,expr)	fd_unparse_xml(item,env,hs);}}
  else if (FD_OIDP(expr)) {
    fd_lisp tag=fd_xml_tag(expr);
    if (FD_EMPTYP(tag)) {
      fd_u8char *rep=fd_object_to_string(expr);
      http_puts(rep,hs); fd_xfree(rep);}
    else unparse_xml_oid(expr,env,hs);
    fd_decref(tag);}
  else {
    fd_u8char *rep=fd_object_to_string(expr);
    http_puts(rep,hs); fd_xfree(rep);}
}

static lisp unparse_xml(fd_lisp expr)
{
  fd_htstream *hs=get_http_output();
  int was_xml=hs->is_xml;
  hs->is_xml=1; fd_unparse_xml(expr,NULL,hs); hs->is_xml=was_xml;
  return FD_VOID;
}

static lisp unparse_html(fd_lisp expr)
{
  fd_unparse_xml(expr,NULL,get_http_output());
  return FD_VOID;
}

/* Access from Lisp */

static lisp use_browse_url_lexpr(lisp args)
{
  fd_lisp url, pool_id;
  fd_get_args("USE-BROWSE-URL!",args,&url,FD_VOID,&pool_id,FD_FALSE,NULL);
  if (FD_FALSEP(pool_id))
    fd_set_browse_url(fd_strdata(url),NULL);
  else {
    fd_pool p=fd_interpret_pool(pool_id);
    fd_set_browse_url(fd_strdata(url),p);}
  return FD_VOID;
}

static fd_u8char *value2string(fd_lisp object)
{
  struct FD_STRING_STREAM ss; FD_INITIALIZE_STRING_STREAM(&ss,1024);
  ss.escape=1; ss.fancy_oids=0;
  fd_print_lisp_to_string(object,&ss);
  return ss.ptr;
}

static lisp lisp_set_cookie_lexpr(lisp args)
{
  struct FD_STRING_STREAM cstream;
  fd_lisp name, value, path, expires;
  fd_u8char *valuestring;
  fd_get_args("SET-COOKIE!",args,&name,FD_VOID,&value,FD_VOID,
	      &path,FD_FALSE,&expires,FD_FALSE,
	      NULL);
  if (!(FD_SYMBOLP(name)))
    fd_type_error(_("Cookie name is not a symbol"),name);
  if (FD_STRINGP(value))
    valuestring=fd_strdup(STRING_DATA(value));
  else valuestring=value2string(value);
  FD_INITIALIZE_STRING_STREAM(&cstream,256);
  uri_encode(SYMBOL_NAME(name),&cstream); fd_sputc(&cstream,'=');
  uri_encode(valuestring,&cstream); fd_sputs(&cstream,"; "); free(valuestring);
  if (FD_FALSEP(expires)) {}
  else if (STRINGP(expires))
    fd_printf(&cstream,"expires=%s ",fd_strdata(expires));
  else if (LRECORD_TYPEP(expires,timestamp_symbol)) {
    time_t now=fd_timestamp_time(expires); struct tm expiration;
    char buf[128];
    fd_breakup_time(&expiration,now,0);
    strftime(buf,128,"%A, %d-%b-%Y %H:%M:%S GMT",&expiration);
    fd_printf(&cstream,"expires=%s;",buf);}
  else fd_type_error("Invalid expiration date",expires);
  if (STRINGP(path)) {
    fd_u8char *sdata=FD_STRING_DATA(path);
    if (strchr(sdata,':')) {
      fd_u8char *colon=strchr(sdata,':');
      fd_u8char *copy=fd_xmalloc((colon-sdata)+1);
      strncpy(copy,sdata,colon-sdata); copy[colon-sdata]=NUL;
      fd_sputs(&cstream,"domain="); uri_encode(copy,&cstream);
      fd_sputs(&cstream,"; "); fd_xfree(copy);
      fd_sputs(&cstream,"path="); uri_encode(colon+1,&cstream);
      fd_sputs(&cstream,"; ");}
    else {
      fd_sputs(&cstream,"path="); uri_encode(sdata,&cstream);
      fd_sputs(&cstream,"; ");}}
  fd_set_cookie(cstream.ptr); fd_xfree(cstream.ptr);
  return FD_VOID;
}


/* Generating frames */

static void start_env
  (fd_htstream *out,u8char *namestring,fd_lisp params,fd_lispenv params_env)
{
  http_printf1s(out,"<%s",namestring);
  if (FD_FALSEP(params)) http_puts(">",out);
  if (FD_EMPTY_LISTP(params)) http_puts(">",out);
  else if (STRINGP(params)) {
    http_putc(' ',out); html_url_puts(STRING_DATA(params),out);
    http_putc('>',out);}
  else if (PAIRP(params)) {
    while (PAIRP(params)) {
      lisp var=fd_get_arg(params,0,FD_VOID);
      if (STRINGP(var)) {
	http_putc(' ',out);
	html_puts_param(STRING_DATA(var));
	http_putc(' ',out);
      	params=CDR(params);}
      else if (FD_SYMBOLP(var)) {
	lisp val_expr=fd_get_arg(params,1,FD_VOID);
	lisp val=fd_eval_in_env(val_expr,params_env);
	params=CDR(CDR(params));
	http_printf1s(out," %s=\"",SYMBOL_NAME(var));
	if (OIDP(val)) {
	  char buf[64];
	  sprintf(buf,"@%x/%x\"",OID_ADDR_HIGH(val),OID_ADDR_LOW(val));
	  http_puts(buf,out);}
	else if (STRINGP(val)) {
	  html_puts_param(STRING_DATA(val)); http_putc('"',out);}
	else {
	  fd_u8char *string=fd_object_to_string(val);
	  html_puts_param(string); http_putc('"',out);
	  free(string); decref(val);}}
      else fd_type_error("not a valid HTML/XML param",var);}
    http_puts(">\n",out);}
  else fd_raise_exception("Weird parameter list");
}

static fd_lisp html_frameset_handler(fd_lisp expr,fd_lispenv env)
{
  fd_lisp frameset_attribs=fd_eval_in_env(fd_get_arg(expr,1,FD_VOID),env);
  fd_htstream *out=get_http_output();
  enum http_generation_phase ph=(get_http_phase());
  if (ph == html_body)
    fd_raise_exception("Can't generate frameset within body");
  set_http_phase(html_frameset);
  if ((ph == http_head) || (ph == html_start)) {
    char *doctype=get_doctype();
    finish_http_header();
    if (doctype) http_puts(doctype,out);
    else http_puts(default_doctype,out);
    http_puts("<HTML><HEAD>\n<TITLE>FramerD Generated Page</TITLE>\n",out);
    thset(http_phase,http_phase_key,html_head);}
  http_puts("</HEAD>\n",out);
  start_env(out,"FRAMESET",frameset_attribs,env);
  {DOLIST(frame_spec,fd_get_body(expr,2))
     start_env(out,"FRAME",frame_spec,env);}
  http_puts("</FRAMESET>\n",out);
  return FD_VOID;
}



/* Initializing the module */

void initialize_htmlgen_c()
{

  lisp framerd_url_symbol=fd_make_symbol("FRAMERD-URL");
  lisp framerd_credit_symbol=fd_make_symbol("FRAMERD-CREDIT");
  
  fd_lispenv menv=fd_make_module();
  fd_xml_env=fd_make_module();
  fd_html_env=menv;

#if (FD_USING_THREADS)
  fd_init_mutex(&browse_url_lock);
  fd_new_tld_key(&local_frames_key,NULL);
  fd_new_tld_key(&http_output_key,NULL);
  fd_new_tld_key(&http_phase_key,NULL);
  fd_new_tld_key(&cookie_key,NULL);
  fd_new_tld_key(&doctype_key,NULL);
#endif
  
  tag_slotid=fd_make_symbol("%TAG");
  xmltag_tag=fd_make_symbol("XMLTAG");

  qmark_symbol=fd_make_symbol("?");
  doctype_symbol=fd_make_symbol("!DOCTYPE");

  html_name_symbol=fd_make_symbol("HTML-NAME");
  obj_name_symbol=fd_make_symbol("OBJ-NAME");
  body_style_symbol=fd_make_symbol("$BODY-STYLE$");
  timestamp_symbol=fd_make_symbol("TIMESTAMP0");
  
  name_symbol=fd_make_symbol("NAME");
  value_symbol=fd_make_symbol("VALUE");
  size_symbol=fd_make_symbol("SIZE");
  cols_symbol=fd_make_symbol("COLS");
  rows_symbol=fd_make_symbol("ROWS");
  content_symbol=fd_make_symbol("CONTENT");

  FD_SET_SYMBOL_VALUE(framerd_url_symbol,fd_make_string(FRAMERD_URL));
  FD_SET_SYMBOL_VALUE(framerd_credit_symbol,fd_make_string(FRAMERD_CREDIT));
  obj_name_symbol=fd_make_symbol("OBJ-NAME"); 
  frame_symbol=fd_make_symbol("FRAME"); 
  anonymous_symbol=fd_make_symbol("Anonymous");
  html_methods_symbol=fd_make_symbol("HTML-SCRIPT");
  head_symbol=fd_make_symbol("HEAD");
  body_symbol=fd_make_symbol("BODY");
  quote_symbol=fd_make_symbol("QUOTE");
  current_file_symbol=fd_make_symbol("*CURRENT-FILE*");
  
  fd_add_cproc(menv,"HTTP-FLUSH!",0,lisp_http_flush_cproc);
  fd_add_lexpr(menv,"SET-COOKIE!",FD_ND_LEXPR,lisp_set_cookie_lexpr);
  fd_add_cproc(menv,"SET-DOCTYPE!",2,lisp_set_doctype_cproc);
  fd_add_lexpr(menv,"USE-BROWSE-SCRIPT!",FD_NORMAL_LEXPR,use_browse_url_lexpr);
  fd_add_cproc(menv,"DECLARE-LOCAL-FRAME!",1,fd_declare_local_frame);

  fd_add_cproc(menv,"HEADER",2,lisp_http_header_cproc);
  fd_add_cproc(menv,"HTML-INSERT-HEAD",1,lisp_html_insert_head_cproc);
  fd_add_cproc(menv,"HTML-START-BODY",1,lisp_html_start_body_cproc);

  fd_add_cproc(menv,"HTTP-REDIRECT",1,http_redirect_cproc);
  fd_add_lexpr(menv,"HTTP-SPLASH",FD_NORMAL_LEXPR,http_splash_lexpr);
  fd_add_cproc(menv,"STYLESHEET!",1,html_stylesheet_cproc);
  fd_add_cproc(menv,"META",2,html_meta_cproc);
  fd_add_special_form(menv,"TITLE",html_title_cproc);

  fd_add_special_form(menv,"WRITE-HTML-FILE",write_html_file_handler);
  fd_add_special_form(menv,"HTMLFRAGMENT",htmlfragment_handler);
  fd_add_special_form(menv,"HTMLSTRING",htmlstring_handler);
  fd_add_special_form(menv,"HTTPDOC",httpdoc);
  /* Obsolete */
  fd_add_special_form(menv,"HTMLDOC",httpdoc);

  fd_add_special_form(menv,"REDIRECT",html_redirect_handler);
  fd_add_special_form(menv,"SPLASH",html_splash_handler);
  fd_add_cproc(menv,"MIME-RETURN-FILE",2,http_return_file);
  fd_add_cproc(menv,"MIME-RETURN-DATA",2,http_return_data);

  fd_add_special_form(fd_xml_env,"MARKUP",markup_handler);
  fd_add_special_form(menv,"MARKUP",markup_handler);
  fd_add_lexpr(fd_xml_env,"XMLTAG",FD_NORMAL_LEXPR,xmltag_lexpr);
  fd_add_lexpr(fd_xml_env,"XMLPI",FD_NORMAL_LEXPR,xmlpi_lexpr);
  fd_add_lexpr(menv,"XMLTAG",FD_NORMAL_LEXPR,xmltag_lexpr);
  fd_add_special_form(fd_xml_env,"XMLENV",htmlenv_handler);
  fd_add_special_form(menv,"XMLENV",htmlenv_handler);

  fd_add_lexpr(menv,"HTMLTAG",FD_NORMAL_LEXPR,htmltag_lexpr);
  fd_add_special_form(menv,"HTMLENV",htmlenv_handler);

  fd_add_special_form(menv,"HTML",htmlexpr);
  fd_add_special_form(menv,"NOBREAK",htmlexpr_nobreak);

  /* Structural formatting directives */
  fd_add_html_line_tag("H1"); fd_add_html_line_tag("H2");
  fd_add_html_line_tag("H3"); fd_add_html_line_tag("H4");
  fd_add_html_line_tag("CENTER"); fd_add_html_line_tag("TABLE"); 
  fd_add_html_line_tag("P"); fd_add_html_line_tag("BLOCKQUOTE"); 

  fd_add_html_line_tag("OL"); fd_add_html_line_tag("UL"); 
  fd_add_html_line_tag("LI");

  fd_add_html_standalone_tag("HR"); fd_add_html_standalone_tag("BR");
  fd_add_html_standalone_tag("IMG");

  fd_add_special_form(menv,"CODEBLOCK",htmlcodeblock);

  /* Typeface formatting directives */
  fd_add_html_tag("EM"); fd_add_html_tag("STRONG");
  fd_add_html_tag("DEFN"); fd_add_html_tag("TT");
  fd_add_html_tag("SUB"); fd_add_html_tag("SUP"); 

  fd_add_htmlenv_tag("A"); 
  fd_add_htmlenv_tag("A*"); 
  fd_add_htmlenv_tag("FONT"); 
  fd_add_htmlenv_tag("FONT*"); 
  fd_add_htmlenv_tag("SPAN"); 

  /* Form stuff */
  fd_add_htmlenv_line_tag("FORM");
  fd_add_htmlenv_line_tag("DIV");

  fd_add_html_standalone_tag("INPUT");
  /* fd_add_htmlenv_tag("TEXTAREA"); */

  fd_add_special_form(menv,"CHECKBOX",html_checkbox_handler);
  fd_add_special_form(menv,"RADIOBUTTON",html_radiobutton_handler);
  fd_add_special_form(menv,"TEXTFIELD",html_textfield_handler);
  fd_add_special_form(menv,"TEXTAREA",html_textarea_handler);
  fd_add_special_form(menv,"SELECTION",html_selection_handler);
  fd_add_special_form(menv,"OPTION",html_option_handler);
  fd_add_special_form(menv,"SUBMIT",html_submit_handler);
  fd_add_special_form(menv,"CGIPASS",html_cgipass_handler);
  fd_add_alias(menv,"PASS","CGIPASS");
  fd_add_htmlenv_tag("BUTTON"); 
  
  fd_add_special_form(fd_html_env,"OPTION*",htmlstar_printout_handler);

  /* Tabular output */
  fd_add_html_line_tag("TR");
  fd_add_html_tag("TD");
  fd_add_html_tag("TH");

  /* Some special forms */
  fd_add_special_form(menv,"ANCHOR",htmlanchor);
  fd_add_special_form(menv,"ANCHOR@",htmltanchor);
  fd_add_special_form(menv,"TAGGED",htmltag);
  fd_add_alias(menv,"TAG","TAGGED");
  fd_add_special_form(menv,"IMAGE",htmlimage);
  fd_add_special_form(menv,"IMG*",htmlstar_printout_handler);
  fd_add_special_form(menv,"HTMLINCLUDE",htmlinclude);
  fd_add_special_form(menv,"FRAMESET",html_frameset_handler);

  fd_add_cproc(menv,"FRAME->HTML",1,describe_oid_in_html);
  fd_add_alias(menv,"OID->HTML","FRAME->HTML");
  /* Obsolete */
  fd_add_cproc(menv,"DESCRIBE-OID",1,describe_oid_in_html);
  
  /* Invoking remote scripts */
  fd_add_lexpr(fd_xml_env,"SCRIPTURL",FD_ND_LEXPR,scripturl);
  fd_add_lexpr(menv,"SCRIPTURL",FD_ND_LEXPR,scripturl);
  fd_add_cproc(menv,"URI-ENCODE",1,uri_encode_string);

  fd_add_cproc(menv,"INDEX-URL",2,index_url);
  fd_add_cproc(menv,"WEBINDEX",2,index_url);

  fd_add_cproc(fd_xml_env,"UNPARSE-XML",1,unparse_xml);
  fd_add_cproc(fd_xml_env,"UNPARSE-HTML",1,unparse_html);
  fd_add_cproc(menv,"UNPARSE-XML",1,unparse_xml);
  fd_add_cproc(menv,"UNPARSE-HTML",1,unparse_html);

  fd_register_module("HTMLGEN",menv);
  fd_register_module("XMLGEN",fd_xml_env);

  fd_register_source_file("htmlgen",__DATE__,vcid);
}



/* File specific stuff */

/* The CVS log for this file
   $Log: htmlgen.c,v $
   Revision 1.32  2002/07/16 15:26:34  haase
   Fixed TAGGED to evaluate its argument (in case its an OID, for instance)

   Revision 1.31  2002/07/01 02:53:38  haase
   Minor, inusufficient, changes to oid->html

   Revision 1.30  2002/06/24 16:29:42  haase
   Made xml parsing do case preservation, complicating the implementation xmltags

   Revision 1.29  2002/06/15 14:52:20  haase
   Made PASS into an alias for CGIPASS

   Revision 1.28  2002/05/27 18:16:34  haase
   Added abstraction layer for thread-local data

   Revision 1.27  2002/05/27 13:06:06  haase
   Added external API for http output from C

   Revision 1.26  2002/05/27 00:57:24  haase
   Fixed SEGV problem in looking for XML handlers

   Revision 1.25  2002/05/26 04:53:16  haase
   Added fdservlet executable and mod_fdserv module

   Revision 1.24  2002/05/13 07:26:18  haase
   Fixes to argument checking in htmlgen primitives

   Revision 1.23  2002/05/11 13:41:47  haase
   Added MIME-RETURN-DATA for returning packets or strings directly

   Revision 1.22  2002/05/07 08:07:25  haase
   Made HTMLGEN write namespaces in generated XML

   Revision 1.21  2002/05/01 21:46:31  haase
   Renamed mutex/condvar/rwlock types to have fd_ prefixes

   Revision 1.20  2002/04/27 17:48:11  haase
   Moved mutex/lock init and destroy into FramerD abstraction layer

   Revision 1.19  2002/04/19 00:18:14  haase
   Fixed some calls to fd_get_args to be null-terminated

   Revision 1.18  2002/04/19 00:14:40  haase
   Fix comment

   Revision 1.17  2002/04/03 13:27:05  haase
   Update new browse-script customization for non-threaded environments

   Revision 1.16  2002/04/02 21:41:08  haase
   Added log and emacs init entries to C source files

*/

/* Emacs local variables
;;;  Local variables: ***
;;;  compile-command: "cd ../..; make" ***
;;;  End: ***
*/
