/* FIGARO'S PASSWORD MANAGER (FPM)
 * Copyright (C) 2000 John Conneely
 * 
 * FPM is open source / free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * FPM is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 *
 * passfile.c -- Routines to save and load XML files with FPM data.
 */
#include <gnome.h>
#include <gnome-xml/parser.h>
#include <gnome-xml/tree.h>
#include "fpm.h"
#include "passfile.h"
#include "fpm_crypt.h"


static void
passfile_update_key()
/* This routine goes through the entire password list and will decrypt the
 * passwords with an old key and encrypt it again with a new key.  This is
 * needed in two situations: First, if the user changes a password.  Secondly,
 * (and much more common) if the salt changes.  In practice, this routine gets
 * called whenever we save a file.
 */
{

  fpm_data* data;
  gchar plaintext[FPM_PASSWORD_LEN+1];
  GList *list;


  if (old_context!=new_context)
  {
    list=g_list_first(glb_pass_list);
    while(list!=NULL)
    {
      data = list->data;
      fpm_decrypt_field(old_context, plaintext,
	 data->password, FPM_PASSWORD_LEN);
      fpm_encrypt_field(new_context, data->password,
	 plaintext, FPM_PASSWORD_LEN);
      list = g_list_next(list);
    }

    memset(plaintext, 0, FPM_PASSWORD_LEN);
    old_context = new_context;
    old_salt = new_salt;
  }
}

static void
move_backup(gchar* file_name)
/* In case of bugs, etc. I want to save old password files.  If you would
 * like to prevent this, you can set FPM_BACKUP_NUM to 0.  This is
 * HIGHLY UNRECOMMENDED while we are in beta!
 */
{
  gchar file1[512];
  gchar file2[512];
  gint i;


  for(i=FPM_BACKUP_NUM;i>0;i--)
  {
    g_snprintf(file2, 512, "%s.%02d", file_name, i);
    if (i>1)
      g_snprintf(file1, 512, "%s.%02d", file_name, i-1);
    else
      g_snprintf(file1, 512, "%s", file_name);
  
    rename(file1, file2);

  }
}

static void
passfile_upd_attr(xmlNodePtr node, char* cond, char** data_ptr)
{
  
  if (!strcmp(node->name, cond))
  {
    g_free(*data_ptr);
    if(node->childs != NULL)
      *data_ptr=g_strdup(node->childs->content);
    else
      *data_ptr=g_strdup("");
  }
}

static void new_leaf(xmlDocPtr doc, xmlNodePtr item, const char *name, char *text)
{
  char* tmp;

  tmp = xmlEncodeEntitiesReentrant(doc, text);
  xmlNewChild(item, NULL, name, tmp);
  free(tmp);
}
  
static void new_leaf_encrypt
	(xmlDocPtr doc, xmlNodePtr item, const char *name, char *text)
{
  gchar* cipher_text;
  cipher_text=fpm_encrypt_field_var(new_context, text);
  new_leaf(doc, item, name, cipher_text);
  g_free(cipher_text);
}

void
passfile_save(gchar* file_name)
{
  xmlDocPtr doc;
  xmlNodePtr nodelist, item;
  fpm_data* data;
  fpm_launcher* launcher;
  FILE * file;
  gchar* vstring_cipher;
  gchar* vstring_plain;
  gchar* cmd;
  gchar* tmp;
  GList * list;
  gint i;

  move_backup(file_name);
  passfile_update_key();

  vstring_cipher=g_malloc0(17);
  vstring_plain=g_malloc0(9);

  doc = xmlNewDoc("1.0");
  doc->root = xmlNewDocNode(doc, NULL, "FPM", NULL);
  xmlSetProp(doc->root, "full_version", FULL_VERSION);
  xmlSetProp(doc->root, "min_version", MIN_VERSION);
  xmlSetProp(doc->root, "display_version", DISPLAY_VERSION);
  item = xmlNewChild(doc->root, NULL, "KeyInfo", NULL);
  xmlSetProp(item, "salt", new_salt);
  strncpy(vstring_plain, "FIGARO", 8);
  fpm_encrypt_field(new_context, vstring_cipher, vstring_plain, 8);
  xmlSetProp(item, "vstring", vstring_cipher);

  fpm_decrypt_field(new_context, vstring_plain, vstring_cipher, 8);



  nodelist = xmlNewChild(doc->root, NULL, "LauncherList", NULL);
  list = g_list_first(glb_launcher_list);
  
  while (list!=NULL)
  {
    launcher=list->data;
    item=xmlNewChild(nodelist, NULL, "LauncerItem", NULL);
    new_leaf(doc, item, "title", launcher->title);
    new_leaf(doc, item, "cmdline", launcher->cmdline);
    tmp=g_strdup_printf("%d", launcher->copy_user);
    new_leaf(doc, item, "copy_user", tmp);
    g_free(tmp);
    tmp=g_strdup_printf("%d", launcher->copy_password);
    new_leaf(doc, item, "copy_password", tmp);
    g_free(tmp);

    list=g_list_next(list);
  }
  
  nodelist = xmlNewChild(doc->root, NULL, "PasswordList", NULL);
  list = g_list_first(glb_pass_list);
  i=0;
  while (list!=NULL)
  {
    data = list->data;
    item = xmlNewChild(nodelist, NULL, "PasswordItem", NULL);
    new_leaf_encrypt(doc, item, "title", data->title);
    new_leaf_encrypt(doc, item, "user", data->user);
    new_leaf_encrypt(doc, item, "url", data->arg);
    new_leaf(doc, item, "password", data->password);
    new_leaf_encrypt(doc, item, "notes", data->notes);
    new_leaf_encrypt(doc, item, "category", data->category);
    new_leaf_encrypt(doc, item, "launcher", data->launcher);
  
    if (data->default_list) xmlNewChild(item, NULL, "default", NULL);

    list=g_list_next(list);
    i++;
  }
  
  file=fopen(file_name, "w");
  xmlDocDump(file, doc);
  fclose(file);
  cmd = g_strdup_printf("chmod 0600 %s", file_name);
  gnome_execute_shell(NULL, cmd);
  g_free(cmd);

  printf("Saved %d passwords.\n", i);
  glb_dirty = FALSE;
  g_free(vstring_plain);
  g_free(vstring_cipher);
}

gint
passfile_load(gchar* file_name)
{
  xmlDocPtr doc;
  xmlNodePtr list, item, attr;
  fpm_data* data;
  fpm_launcher* launcher;
  gint i;

  /* Start from scratch */
  if(glb_pass_list != NULL) fpm_clear_list();

  doc=xmlParseFile(file_name);
  if (doc==NULL)
  {
    /* If we can't read the doc, then assume we are running for first time.*/
    old_salt=get_new_salt();
    new_salt=old_salt;
    glb_pass_list=NULL;
    glb_dirty=TRUE;
    fpm_init_launchers();
    
    return(-1);
  }

  /* Check if document is one of ours */
  g_return_val_if_fail(!strcmp(doc->root->name, "FPM"), -2);

  file_version = xmlGetProp(doc->root, "min_version");
  if (strcmp(file_version, FULL_VERSION) > 0)
  {
    g_error("Sorry, the password file cannot be read because it uses a future file format.  Please download the latest version of FPM and try again.\n");
    gtk_main_quit(); 
  }

  file_version = xmlGetProp(doc->root, "full_version");
 
/* Current versions of FPM encrypt all fields.  Fields other than the 
 * password are decrypted when the program reads the password file, 
 * and the password is decrypted as it is needed (to try to prevent it
 * from showing up in coredumps, etc.)  The global variable  glb_need_decrypt
 * is set by the passfile loading routines to specify that the passfile
 * was created with a version of FPM that requires this decryption.  
 * This allows backward compatibility with previous versions of FPM which
 * only encrypted passwords.
 */
 
  glb_need_decrypt = (strcmp(file_version, ENCRYPT_VERSION) >= 0);

  list=doc->root->childs;
  old_salt = xmlGetProp(list, "salt");
  vstring = xmlGetProp(list, "vstring");
  new_salt = get_new_salt();
  glb_using_defaults=FALSE;
  
  list=list->next;

  if (list==NULL || ( strcmp(list->name, "PasswordList") && strcmp(list->name, "LauncherList") ))
  {
    g_error("Invalid password file.");
    gtk_main_quit();
  }

  if(!strcmp(list->name, "LauncherList"))
  {
    printf("Loading launchers...\n");
    glb_launcher_list=NULL;
    item=list->childs;
    while(item!=NULL)
    {
      launcher=g_malloc0(sizeof(fpm_launcher));
      launcher->title=g_strdup("");
      launcher->cmdline=g_strdup("");
      launcher->copy_user=0;
      launcher->copy_password=0;
      attr=item->childs;
      while(attr!=NULL)
      {
	if(!strcmp(attr->name, "title") && attr->childs && attr->childs->content)
	{
	  g_free(launcher->title);
	  launcher->title=g_strdup(attr->childs->content);
	}
	if(!strcmp(attr->name, "cmdline") && attr->childs && attr->childs->content)
	{
	  g_free(launcher->cmdline);
	  launcher->cmdline=g_strdup(attr->childs->content);
	}
	if(!strcmp(attr->name, "copy_user"))
	{
	  if(!strcmp(attr->childs->content, "1")) launcher->copy_user=1;
	  if(!strcmp(attr->childs->content, "2")) launcher->copy_user=2;
	}
	if(!strcmp(attr->name, "copy_password"))
	{
	  if(!strcmp(attr->childs->content, "1")) launcher->copy_password=1;
	  if(!strcmp(attr->childs->content, "2")) launcher->copy_password=2;
	}
      attr=attr->next;
      }
      glb_launcher_list=g_list_append(glb_launcher_list, launcher);

      item=item->next;
    }
    fpm_create_launcher_string_list();

    /* Incurement top-level list from launcher list to password list. */
    list=list->next; 
  }
  else
  {
    fpm_init_launchers();
  }

  if (list==NULL || strcmp(list->name, "PasswordList"))
  {
    g_error("Invalid password file.");
    gtk_main_quit();
  }


  i=0;
  
  item=list->childs;
  while(item!=NULL)
  {

    /* Start with blank data record */
    data = g_malloc0(sizeof(fpm_data));
    data->title=g_strdup("");
    data->arg=g_strdup("");
    data->user=g_strdup("");
    data->notes=g_strdup("");
    data->category=g_strdup("");
    data->launcher=g_strdup("");
    data->default_list=0;
   
    /* Update data record with each type of attribute */
    attr=item->childs;
    while(attr!=NULL)
    {
      passfile_upd_attr(attr, "title", &data->title);
      passfile_upd_attr(attr, "url", &data->arg);
      passfile_upd_attr(attr, "arg", &data->arg);
      passfile_upd_attr(attr, "user", &data->user);
      passfile_upd_attr(attr, "notes", &data->notes);
      passfile_upd_attr(attr, "category", &data->category);
      passfile_upd_attr(attr, "launcher", &data->launcher);
      if(!strcmp(attr->name, "default"))
      {
        data->default_list=1;
	glb_using_defaults=TRUE;
      }
 
      if(!strcmp(attr->name, "password"))
        strncpy(data->password, attr->childs->content, FPM_PASSWORD_LEN*2);

      attr=attr->next;
    }

    /* Insert item into GList */

    glb_pass_list=g_list_append(glb_pass_list, data);

    item=item->next;
    i++;
  }
  printf("Loaded %d passwords.\n", i);
  glb_dirty = FALSE;

  return(0);

}   

