/*
  Copyright 2005, 2006, 2007 David Cad, Damien Stehl.

  This program is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the
  Free Software Foundation; either version 2 of the License, or (at your
  option) any later version.

  This program is distributed in the hope that it will be useful, but WITHOUT
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
  more details.

  You should have received a copy of the GNU General Public License along
  with this program; see the file COPYING.  If not, write to the Free
  Software Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
  02111-1307, USA.

  This program implements ideas from the paper "Floating-point LLL Revisited", 
  by Phong Nguyen and Damien Stehl, in the Proceedings of Eurocrypt'2005, 
  Springer-Verlag; and was partly inspired by Shoup's NTL library: 
  http://www.shoup.net/ntl/

*/

#ifndef PROVED_CPP
#define PROVED_CPP

#include "proved.h"
#include "util.h"

#ifdef DEBUG
#define LOOPS_BABAI 1000
#endif

template<class ZT,class FT> inline ZZ_mat<ZT>* proved<ZT,FT>::GetBase()
{
  return B;
}
/*TODO : derived class with WITH_TRANSFORM and delete ifdef here*/
template<class ZT,class FT> inline void proved<ZT,FT>::helper1U(
         int j,int xx,int kappa,int d, ZZ_mat<ZT>*U)
{
#ifdef WITH_TRANSFORM
  for (int i=0; i<n; i++)  
    {
      if (xx > 0)
	B->Get(kappa,i).submul_ui(B->Get(j,i),(unsigned long int) xx);
      else
	B->Get(kappa,i).addmul_ui(B->Get(j,i),(unsigned long int) -xx);
    }
#endif
}


template<class ZT,class FT> inline void proved<ZT,FT>::helper0XP(int zeros,int kappa,int j,
						   Z_NR<ZT>&ztmp,int n,int d,int kappamax)
{
  int i;
  d=d;
  for (int k=zeros+1; k<j; k++)
    mu->Get(kappa,k).sub(mu->Get(kappa,k), mu->Get(j,k));
  
  for (i=0; i<n; i++)    
    B->Get(kappa,i).sub(B->Get(kappa,i),B->Get(j,i));
		      
#ifdef WITH_TRANSFORM
  for (i=0; i<d; i++) 
    U->Get(kappa,i).sub(U->Get(kappa,i), U->Get(j,i));
#endif
  
  ztmp.mul_ui( G->Get(kappa,j), 2);
  ztmp.sub( G->Get(j,j), ztmp);
  G->Get(kappa,kappa).add(G->Get(kappa,kappa), ztmp);
  
  
  for (i=0; i<=j; i++)
    G->Get(kappa,i).sub(G->Get(kappa,i), G->Get(j,i));
					    
  for (i=j+1; i<kappa; i++)
    G->Get(kappa,i).sub(G->Get(kappa,i), G->Get(i,j));
  
  for (i=kappa+1; i<=kappamax; i++)
    G->Get(i,kappa).sub( G->Get(i,kappa), G->Get(i,j));
}
template<class ZT,class FT> inline void proved<ZT,FT>::helper0XM(int zeros,int kappa,int j,
							   Z_NR<ZT>& ztmp,int n,int d,int kappamax)
{
  d=d;
  for (int k=zeros+1; k<j; k++)
    mu->Get(kappa,k).add(mu->Get(kappa,k), mu->Get(j,k));

  int i;
  for (i=0; i<n; i++)    
    B->Get(kappa,i).add(B->Get(kappa,i), B->Get(j,i));

#ifdef WITH_TRANSFORM
  for (i=0; i<d; i++)    
    U->Get(kappa,i).add(U->Get(kappa,i), U->Get(j,i));
#endif
  
  ztmp.mul_ui( G->Get(kappa,j), 2);
  ztmp.add( G->Get(j,j), ztmp);
  G->Get(kappa,kappa).add(G->Get(kappa,kappa), ztmp);
  
  for (i=0; i<=j; i++)
    G->Get(kappa,i).add(G->Get(kappa,i), G->Get(j,i));
  
  for (i=j+1; i<kappa; i++)
    G->Get(kappa,i).add(G->Get(kappa,i),G->Get(i,j));
  
  for (i=kappa+1; i<=kappamax; i++)
    G->Get(i,kappa).add(G->Get(i,kappa), G->Get(i,j));
}
template<class ZT,class FT> inline void
proved<ZT,FT>::GSO(int a, int zeros, int kappamax, int n,
		   Z_NR<ZT>& ztmp, FP_NR<FT>& tmp, FP_NR<FT>& rtmp,FP_NR<FT>& max,
		   int aa)
{

  a=a; kappamax=kappamax; n=n;

  max.set(0.0);
  for (int j=aa; j<kappa; j++)
    {
      if (j > zeros+2)
	{
	  tmp.mul(mu->Get(j,zeros+1), r->Get(kappa,zeros+1));
	  set_fz(rtmp, G->Get(kappa,j));
	  rtmp.sub(rtmp, tmp);
	  for (int k=zeros+2; k<j-1; k++)
	    {
	      tmp.mul(mu->Get(j,k), r->Get(kappa,k));
	      rtmp.sub( rtmp, tmp);
	    }
	  tmp.mul( mu->Get(j,j-1), r->Get(kappa,j-1));
	  (r->Get(kappa,j)).sub(rtmp, tmp);
	}
      else if (j==zeros+2)
	{	      
	  tmp.mul( mu->Get(j,zeros+1), r->Get(kappa,zeros+1));
	  set_fz(rtmp, G->Get(kappa,j));
	  (r->Get(kappa,j)).sub( rtmp, tmp);
	}
      else
	set_fz(r->Get(kappa,j), G->Get(kappa,j));
      
      mu->Get(kappa,j).div( r->Get(kappa,j), r->Get(j,j));
      rtmp.abs(mu->Get(kappa,j));
      if (max.cmp(rtmp)<0) 
	max.set(rtmp);
    }
}


template<class ZT,class FT> int
proved<ZT,FT>::Babai (int a, int zeros, int kappamax, int n,
		      Z_NR<ZT>& ztmp, FP_NR<FT>& tmp, FP_NR<FT>& rtmp,FP_NR<FT>& max,FP_NR<FT>&max2,FP_NR<FT>&max3)
{
  int d, i, j, k, test, aa, sg,ll;
  signed long xx;
  long int expo;

  typename FP_NR<FT>::assoc_int X;

  d = G->GetNumRows();
#ifdef DEBUG
  int loops=0;
#endif 

  aa = (a > zeros) ? a : zeros+1;

#ifdef DEBUG
  cout<<"\nr: \n";
  cout.flush();
  r->print( kappamax+1, kappamax+1);
  fflush(stdout);
  cout<<"\n          STARTING BABAI WITH k="<<kappa<<" AND dim="<<d<<"\n";
  loops = 0;
  cout<<"\nmu: \n";
  mu->print( kappamax+1, kappamax+1);
  cout<<"a is "<<a<<", zeros is "<<zeros<<", aa is "<<aa<<"\n";
  B->print(kappamax+1, n);
#endif
  
  
  ll=0;
  do
    {
      ll++;
      test=0;
#ifdef DEBUG     
      loops++; 
      if (loops>LOOPS_BABAI){	  
	cout<<"INFINITE LOOP?\n";
	abort();
      }
#endif     
      /* ************************************** */
      /* Step2: compute the GSO for stage kappa */
      /* ************************************** */
      max3.set(max2);
      max2.set(max);
      GSO(a,zeros,kappamax,n,ztmp,tmp,rtmp,max,aa); 
      
      if(ll>=3)
	{
	  rtmp.mul_2ui(max2,SIZE_RED_FAILURE_THRESH);
	  if( max3.is_nan() || max3.cmp(rtmp)<=0)
	    {
#ifdef VERBOSE
	      cerr << "unexpected behaviour -> exit";
#endif
	      return kappa;
	    }
#ifdef DEBUG
	  cout << "\nrtmp=";rtmp.print();cout<<"\nmax2=";max2.print();cout<<"\nmax3=";max3.print();
	  cout<<"\n";
#endif
	  
	}

#ifdef DEBUG
      if (loops <=LOOPS_BABAI){
	cout<<"\nmu :\n";
	mu->print(kappa+1, kappa+1);
	cout<<"\nr :\n";
	r->print(kappa+1, kappa+1);

	if (max2.cmp(0.0)==0)
	  cout<<"\nmax=====================\n";
	cout << "\nmax=";
	max.print();
	cout << "\n";

      }
#endif
      
      /* **************************** */
      /* Step3--5: compute the X_j's  */
      /* **************************** */
      
      for (j=kappa-1; j>zeros; j--)
	{
	  /* test of the relaxed size-reduction condition */
	  
	  tmp.abs(mu->Get(kappa,j));

#ifdef DEBUG
	  if (loops<=LOOPS_BABAI)
	    {
	      printf( "tmp is : "); 
	      tmp.print(); printf( "\n");
	    }
#endif	  
	  if (tmp.cmp(halfplus) > 0) 
	    {
	      test = 1;
	      /* we consider separately the case X = +-1 */     
	      if (tmp.cmp(onedothalfplus)<=0 )   
		{
		  sg =  mu->Get(kappa,j).sgn();
		  if ( sg >=0 )   /* in this case, X is 1 */
		    {
		      helper0XP(zeros,kappa,j,ztmp,n,d,kappamax);
  		    }
		  
		  else          /* otherwise X is -1 */ 
		    {
		      helper0XM(zeros,kappa,j,ztmp,n,d,kappamax);
		    }
		}
	      
	      else   /* we must have |X| >= 2 */
		{		

		  tmp.rnd( mu->Get(kappa,j));
		  
		  for (k=zeros+1; k<j; k++)
		    {
		      rtmp.mul( tmp, mu->Get(j,k));
		      mu->Get(kappa,k).sub( mu->Get(kappa,k), rtmp);
		    }
		  
		  if (tmp.exp() < CPU_SIZE-2)   
		    /* X is stored in a long signed int */		  
		    {		      
		      xx=tmp.get_si();
#ifdef DEBUG
		      if (loops<=LOOPS_BABAI)
			{
			  printf("          xx[%d] is %ld\n", j, xx);
			  printf("          and tmp was ");
			  tmp.print(); printf("\n");
			}
#endif
		      
		      for (i=0; i<n; i++)  
			{
			  if (xx > 0)
			    B->Get(kappa,i).submul_ui(B->Get(j,i),(unsigned long int) xx);
			  else
			    B->Get(kappa,i).addmul_ui(B->Get(j,i),(unsigned long int) -xx);
			} 
		      helper1U(j,xx,kappa,d,U);		      


		      if (xx>0)
			G->Get(kappa,kappa).submul_ui(G->Get(kappa,j),(unsigned long int) 2*xx);
		      else
			G->Get(kappa,kappa).addmul_ui(G->Get(kappa,j),(unsigned long int) -2*xx);
		      
		      		      
		      ztmp.mul_si( G->Get(j,j), xx);
		      ztmp.mul_si( ztmp, xx);
		      G->Get(kappa,kappa).add(G->Get(kappa,kappa), ztmp);	

 
		      for (i=0; i<=j; i++)  
			{
			  if (xx > 0)
			    G->Get(kappa,i).submul_ui(G->Get(j,i),(unsigned long int) xx);
			  else
			    G->Get(kappa,i).addmul_ui(G->Get(j,i),(unsigned long int) -xx);
			} 
		      for (i=j+1; i<kappa; i++)  
			{
			  if (xx > 0)
			    G->Get(kappa,i).submul_ui(G->Get(i,j),(unsigned long int) xx);
			  else
			    G->Get(kappa,i).addmul_ui(G->Get(i,j),(unsigned long int) -xx);
			} 
		      for (i=kappa+1; i<=kappamax; i++)  
			{
			  if (xx > 0)
			    G->Get(i,kappa).submul_ui(G->Get(i,j),(unsigned long int) xx);
			  else
			    G->Get(i,kappa).addmul_ui(G->Get(i,j),(unsigned long int) -xx);
			} 
		    }
		  
		  else
		    {
		      expo = get_z_exp(X,tmp);
		      
		      for (i=0; i<n; i++)  
			{
			  ztmp.mul_2exp( B->Get(j,i), expo); 
			  B->Get(kappa,i).submul(ztmp,X);
			}
#ifdef WITH_TRANSFORM
		      for (i=0; i<d; i++)  
			{
			  ztmp.mul_2exp( U->Get(j,i), expo); 
			  U->Get(kappa,i).submul(ztmp,X);
			}
#endif
		      ztmp.mul_2exp( G->Get(kappa,j), expo+1);
		      G->Get(kappa,kappa).submul(ztmp,X);
		      ztmp.mul( G->Get(j,j), X);
		      ztmp.mul( ztmp, X);
 		      ztmp.mul_2exp( ztmp, 2*expo);
		      G->Get(kappa,kappa).add(
			    G->Get(kappa,kappa), ztmp);

		      for (i=0; i<=j; i++)
			{
			  ztmp.mul_2exp( G->Get(j,i), expo);
			  G->Get(kappa,i).submul(ztmp,X);
			}
		      
		      for (i=j+1; i<kappa; i++)
			{
			  ztmp.mul_2exp( G->Get(i,j), expo);
			  G->Get(kappa,i).submul(ztmp,X);
			} 
		      
		      for (i=kappa+1; i<=kappamax; i++)
			{
			  ztmp.mul_2exp( G->Get(i,j), expo);
			  G->Get(i,kappa).submul(ztmp,X);
			}
		    }
		}
	    }
	}

      
      /* Anything happened? */
      if (test) aa = zeros+1;
#ifdef DEBUG
      if (loops<=LOOPS_BABAI)
	{
	  printf("          test is %d\n", test);
	  printf("\nmu: \n");
	  mu->print(kappa+1, kappa+1);
	  printf("\nr: \n");
	  r->print(kappa+1, kappa+1);
	}
#endif
    }
  while (test);
  

  set_fz(s[zeros+1], G->Get(kappa,kappa));
  /* the last s[kappa]=r[kappa,kappa] is computed only if kappa increases */
  for (k=zeros+1; k<kappa-1; k++){
    tmp.mul( mu->Get(kappa,k), r->Get(kappa,k));
    s[k+1].sub( s[k], tmp);
  }
#ifdef DEBUG
  printf("          Number of loops is %d\n", loops);
  G->print( kappamax+1, kappamax+1);
#endif
  return 0;
}

template<class ZT,class FT>
proved<ZT,FT>::proved (ZZ_mat<ZT>* B_,int preci,double et,double delt)
{
  prec=preci;
  eta=et;
  delta=delt;

  halfplus.set(eta);
  onedothalfplus.set(1.0+eta);
  ctt.set(delta);

  B=B_;
  double rho = (1.0 + eta) * (1.0+eta) / (delta - eta*eta);
  unsigned int d= B->GetNumRows();
  unsigned int goodprec = (unsigned int) (7.0 + 0.2*d + d* log(rho)/ log(2.0) 
			     +  2.0 * log ( (double) d )  
			     - log( (eta-0.5)*(1.0-delta) ) / log(2.));

#ifdef VERBOSE
  cerr << "Setting precision at "<< (!prec?goodprec:prec) <<" if the type allows it."<<endl;
#endif
  
  if (prec!=0)
    FP_NR<FT>::setprec(prec);
  else
    FP_NR<FT>::setprec(goodprec);
 
  G=new ZZ_mat<ZT>(B->GetNumRows(),B->GetNumRows());
#ifdef WITH_TRANSFORM
 
  U=new ZZ_mat<ZT>(B->GetNumRows(),B->GetNumRows());
  for (int i=0;i<B->GetNumRows();i++)
    for (int j=0;j<B->GetNumRows();j++)
      if (i==j) U->Get(i,i).set(1);
      else U->Get(i,j).set(0);
#else
  U=new ZZ_mat<ZT>(0,0); 
#endif
  shift=B->getShift();
}

template<class ZT,class FT>
proved<ZT,FT>::~proved ()
{
  if (U) delete U;
  if (G) delete G;
}



/* ****************** */
/* ****************** */
/* ****************** */
/* The LLL Algorithm  */
/* ****************** */
/* ****************** */
/* ****************** */

/* LLL-reduces the integer matrix(ces) (G,B,U)? "in place" */



template<class ZT,class FT>
int 
proved<ZT,FT>::LLL ()
{
  int  kappa2, d, n=0, i, j, zeros, kappamax;
  //  FP_mat<FT> mu, r;
  //FP_NR<FT> *s;
  FP_NR<FT> *mutmp;
  FP_NR<FT> tmp, rtmp;
  Z_NR<ZT> ztmp;
  Z_NR<ZT> *SPtmp;
  int *alpha;
  FP_NR<FT> max,max2,max3;


  /*
#ifdef USE_DPE
#ifdef FORCE_LONGDOUBLE
  long double tmp2, y;
#endif
#endif
  */
  Z_NR<ZT> *Btmp;

#ifdef VERBOSE
  int newkappa, start, loops, lovasz;
#endif


  d = B->GetNumRows();
  n = B->GetNumCols();

  unsigned long long int iter=0;
  unsigned long long int maxIter=(unsigned long long int)(d - 2*d*(d+1)*((B->getMaxExp()+3)/log (delta)));



#ifdef DEBUG
  cerr<<"d = "<<d<<", n="<<n<<"\n";
#endif
  
  
#ifdef VERBOSE
  cerr<<   "\nEntering fpLLL:\nLLL-reduction factors (";
  ctt.printerr();
  cerr<< ", ";
  halfplus.printerr(); 
  cerr<< ")\n";
  cerr<<  "With MPFR and without forcing a precision, the reduction is guaranteed.\n\n";
#endif
  
  alpha = new int [d]; 
  /*INIT (tmp);
  INIT (rtmp);
  ZINIT (ztmp);
  */

  mu = new (FP_mat<FT>)(d,d);//init_matrixf (&mu, d, d);
  r = new FP_mat<FT>(d,d);//init_matrixf (&r, d, d);

  s = new FP_NR<FT> [d];//init_vecf (d+1);
  SPtmp = new Z_NR<ZT> [d];//init_vec (d+1);
  
#ifdef VERBOSE  
  start = cputime();
#endif
  
#ifdef DEBUG 
  B->print(d, n);
  cout<<"Step 1 is over\n";
  cout.flush();
#endif 
  

  /* ********************************* */
  /* Step2: Initializing the main loop */
  /* ********************************* */   
  

#ifdef VERBOSE  
  newkappa = 0;
  loops = lovasz = 0;
#endif

  kappamax = 0;
  i = 0; 

  do {
    ScalarProduct (G->Get(i,i), B->GetVec(i), B->GetVec(i), n);
    set_fz (r->Get(i,i), G->Get(i,i));
  }
  while ((G->Get(i,i).sgn() == 0)&&(++i <d));
  zeros = i-1; /* all vectors B[i] with i <= zeros are zero vectors */
  kappa = i+1;

  if (zeros < d-1) set_fz (r->Get(i,i), G->Get(i,i));
  
  for (i=zeros+1; i<d; i++)
    alpha[i]=0;
  
#ifdef DEBUG
  cout<<"Step 2 is over\n"; 
  cout<<"kappa is "<<kappa<<" and d is "<<d<<"\n";
  cout.flush();
#endif  
  
  while (kappa < d)
    {

      
      if (kappa>kappamax) {
	for (i=zeros+1; i<=kappa; i++)
	  ScalarProduct (G->Get(kappa,i), B->GetVec(kappa), B->GetVec(i), n);
	kappamax++; 
      }


      
#ifdef VERBOSE
      loops++;
      if (kappa>newkappa)
	{
	  newkappa++;
	  cerr<< "Discovering vector k = "<<kappa+1<<", iterations = "<<loops<<" cputime is "<<(cputime()-start)<<" ms\n";
	}
#endif

#ifdef DEBUG
      cout<<"alpha= ";
      for (i=zeros+1; i<d; i++) 
	cout<<alpha[i]<<" ";
      cout<<"\n";
      cout<<"entering the while loop with k="<<kappa<<"\n";
       B->print( d, n);
#endif


      /* ********************************** */
      /* Step3: Call to the Babai algorithm */
      /* ********************************** */   


       if (Babai (alpha[kappa], zeros, kappamax,(kappamax +1+ shift <= n) ? (kappamax+1+shift):n, 
		  ztmp, tmp, rtmp,max,max2,max3))
	 {
	   delete[] alpha;
	   
	   delete mu;
	   delete r;
	   
	   delete[] s;
	   delete[] SPtmp;
	   return kappa;
	 }

      /*      
      if (Babai (alpha[kappa], zeros, kappamax, n, 
		 ztmp, tmp, rtmp,max,max2,max3))
	{
	  return kappa;
	}
      */

      
#ifdef DEBUG      
      cout<<"Step 3 is over\n";  
      B->print( kappamax+1, n);
      r->print( kappamax+1, kappamax+1);
#endif
      
      /* ************************************ */
      /* Step4: Success of Lovasz's condition */
      /* ************************************ */  
      /* xtt * r.coeff[kappa-1][kappa-1] <= s[kappa-2] ?? */
      
      tmp.mul(r->Get(kappa-1,kappa-1), ctt);

#ifdef DEBUG
      cout<<"s["<<kappa-1<<"] is ";
      s[kappa-1].print();
      cout<<"\n";
      r->Get(kappa-1,kappa-1).print();
      cout<<"\n";
#endif

#ifdef VERBOSE
      lovasz++;
#endif


      if( tmp.cmp(s[kappa-1]) <=0 ) {
	alpha[kappa] = kappa;
	tmp.mul( mu->Get(kappa,kappa-1), r->Get(kappa,kappa-1));
	r->Get(kappa,kappa).sub( s[kappa-1], tmp);
	kappa++;
      } 
      else
	{

	  /* ******************************************* */
	  /* Step5: Find the right insertion index kappa */
          /* kappa2 remains the initial kappa            */
	  /* ******************************************* */  

	  kappa2 = kappa;
	  do
	    {
#ifdef VERBOSE
	      lovasz++;
#endif
	      kappa--;
	      if (kappa>zeros+1) tmp.mul( r->Get(kappa-1,kappa-1), ctt);
	    }
	  while ((kappa>=zeros+2) && (s[kappa-1].cmp(tmp) <=0 ));

#ifdef DEBUG
	  cout<< "Index of insertion: "<<kappa<<"\n";
	  cout<<"Step 5 is over\n";
	  cout<<"alpha= ";
	  for (i=0; i<=kappamax; i++) 
	    cout<<alpha[i]<<" ";
	  cout<<"\n";
#endif
	  
	  for (i=kappa; i<kappa2; i++)
	    if (kappa <= alpha[i]) alpha[i] = kappa;

	  for (i=kappa2; i>kappa; i--)
	    alpha[i] = alpha[i-1];

	  for (i=kappa2+1; i<=kappamax; i++)
	    if (kappa < alpha[i]) alpha[i] = kappa;
	  
	  alpha[kappa] = kappa;
	  
#ifdef DEBUG
	  cout<<"alpha= ";
	  for (i=0; i<d; i++) 
	    cout<<alpha[i]<< " " ;
	  cout<<"\n";
#endif

	  /* ****************************** */
	  /* Step6: Update the mu's and r's */
	  /* ****************************** */  
	  
	  mutmp = mu->GetVec(kappa2);
	  for (i=kappa2; i>kappa; i--)
	    mu->GetVec(i) = mu->GetVec(i-1);
	  mu->GetVec(kappa) = mutmp;
	  
	  mutmp = r->GetVec(kappa2);
	  for (i=kappa2; i>kappa; i--)
	    r->GetVec(i) = r->GetVec(i-1);
	  r->GetVec(kappa) = mutmp;
	  
	  r->Set(kappa,kappa,s[kappa]);
	  
#ifdef DEBUG 
	  cout<<"Step 6 is over\n";
#endif
	  
	  /* ********************* */
	  /* Step7: Update B, G, U */
	  /* ********************* */  	  
	  
	  Btmp = B->GetVec(kappa2);
	  for (i=kappa2; i>kappa; i--)
	    B->GetVec(i) = B->GetVec(i-1);
	  B->GetVec(kappa) = Btmp;

#ifdef WITH_TRANSFORM
	  Btmp = U->GetVec(kappa2);
	  for (i=kappa2; i>kappa; i--)
	    U->GetVec(i) = U->GetVec(i-1);
	  U->GetVec(kappa) = Btmp;
#endif

	  for (i=0; i<=kappa2; i++)
	    SPtmp[i].set(G->Get(kappa2,i));

	  for (i=kappa2+1; i<=kappamax; i++)
	    SPtmp[i].set(G->Get(i,kappa2));
	  
	  for (i=kappa2; i>kappa; i--)
	    {
	      for (j=0; j<kappa; j++)
		G->Set(i,j,G->Get(i-1,j));
	      
	      G->Set(i,kappa, SPtmp[i-1]);
	      
	      for (j=kappa+1; j<=i; j++)
		G->Set(i,j, G->Get(i-1,j-1));
	      
	      for (j=kappa2+1; j<=kappamax; j++)
		G->Set(j,i, G->Get(j,i-1));     
	    }
	  
	  for (i=0; i<kappa; i++)
	    G->Set(kappa,i, SPtmp[i]);

	  G->Set(kappa,kappa, SPtmp[kappa2]);

	  for (i=kappa2+1; i<=kappamax; i++)
	    G->Set(i,kappa, SPtmp[i]);

#ifdef DEBUG
	  B->print( kappamax+1, n);
	  G->print( kappamax+1, kappamax+1);
	  cout<<"Step 7 is over\n";
#endif
	  


	  /* ************************************** */
	  /* Step8: Prepare the next loop iteration */
	  /* ************************************** */  	  


	  if ( (kappa==zeros+1) && (G->Get(kappa,kappa).sgn()==0) )
	    {
	      zeros++;
	      kappa++;
	      set_fz(r->Get(kappa,kappa), G->Get(kappa,kappa));
	    }
	  
	  kappa++;
	  
#ifdef DEBUG	  
	  B->print( kappamax+1, n);
	  G->print( kappamax+1, kappamax+1);
	  cout<<"Step 8 is over, new kappa="<<kappa<<"\n";
#endif
	  
	}
      iter++;
      if (iter> maxIter)
	{
	  cerr<<"Too many loop iterations";
	  delete[] alpha;

	  delete mu;
	  delete r;
	  
	  delete[] s;
	  delete[] SPtmp;
  
	  return -1;
	}
      
    };  
 
#ifdef VERBOSE
  tmp.set( 1.0);
  for (i = zeros+1; i<d; i++)
    tmp.mul( tmp, r->Get(i,i));
  tmp.sqrt( tmp);

  cerr<<"\nLoop iterations = "<<loops<<"\n";
  cerr<<"Lovasz tests = "<<lovasz<<"\n";
  cerr<<"Cputime is "<<(cputime()-start)<<" ms\n";
  if (zeros < d-1)
    {
      cerr<<"Vol(L) is "; tmp.printerr(); cerr<<"\n";      
      tmp.sqrt(r->Get(zeros+1,zeros+1));
      cerr<<"||b_1|| is "; 
      tmp.printerr(); 
      cerr<<"\n"<<endl;
    }
#endif
  

  delete[] alpha;

  delete mu;
  delete r;

  delete[] s;
  delete[] SPtmp;
  return 0;
}

#endif
