#!/usr/bin/env python

# Free Open FTP Face (C)2006 Jeffrey Bakker
#
# FOFF is a graphical FTP client written in pyGTK.

# FOFF is compact, multiplatform (any system that supports GTK+ and python),
# has a simple and friendly interface, and has built-in convenience features
# including an image viewer, text viewer, one-click compress/decompress with
# gzip, and a mini console that takes both FTP commands and operating system
# commands.


# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.


import md5
import time
import ftplib

import threading
import os, re
import gtk

COMPRESS_GZIP=1
COMPRESS_BZIP2=2
COMPRESS_ZIP=3
COMPRESS_NONE=4

# progress bar for indefinite waits ============================================
class Pulse(threading.Thread):

	def __init__(self, progress):

		self.progress = progress
		self.Finished = False

		threading.Thread.__init__(self)

	def set_finished(self):

		self.Finished = True

		time.sleep(0.4)
		gtk.threads_enter()
		self.progress.set_fraction(0.0)
		gtk.threads_leave()

	def run(self):

		gtk.threads_enter()
		self.progress.set_fraction(0.0)
		gtk.threads_leave()

		while(not self.Finished):

			time.sleep(0.1)
			gtk.threads_enter()
			self.progress.pulse()
			gtk.threads_leave()

# terminal commands ============================================================
class Command(threading.Thread):

	def __init__(self, textview, command, pulse, reload):

		self.textvw = textview
		self.buffer = self.textvw.get_buffer()

		self.command = command
		self.pulse = pulse
		self.reload = reload

		threading.Thread.__init__(self)

	def run(self):

		self.read_output(self.buffer, self.command)

		self.pulse.set_finished()
		gtk.threads_enter()
		self.reload()
		gtk.threads_leave()

	def read_output(self, buffer, command):

		stdin, stdouterr = os.popen4(command)

		for line in stdouterr.readlines():

			gtk.threads_enter()
			buffer.insert(buffer.get_end_iter(), line)
			self.textvw.scroll_mark_onscreen(buffer.get_insert())
			gtk.threads_leave()

# compress with gzip ===========================================================
class Compress(threading.Thread):

	def __init__(self, filename, pulse, reload):

		self.filename = filename
		self.pulse = pulse
		self.reload = reload

		threading.Thread.__init__(self)


	def bzip(self):

		import bz2

		infile = open(self.filename, 'rb')
#		infile = open(self.filenames[0], 'rb')
		bzFile = bz2.BZ2File(self.filename + '.bz2', 'wb')

		while True:
			chunk = infile.read(65536)
			if not chunk:
				break

			bzFile.write(chunk)

		infile.close()
		bzFile.close()

	def gzip(self):

		import gzip

		infile = open(self.filename, 'rb')
#		infile = open(self.filenames[0], 'rb')
		gzFile = gzip.GzipFile(self.filename + '.gz', 'wb')

		while True:
			chunk = infile.read(65536)
			if not chunk:
				break

			gzFile.write(chunk)

		infile.close()
		gzFile.close()


	def run(self):

		self.gzip()

		self.pulse.set_finished()
		gtk.threads_enter()
		self.reload()
		gtk.threads_leave()

# decompress with gunzip =======================================================
class Decompress(threading.Thread):

	def __init__(self, filename, pulse, reload, txview):

		self.filename = filename
		self.pulse = pulse
		self.reload = reload
		self.txview = txview
		self.txbuff = txview.get_buffer()

		threading.Thread.__init__(self)


	def tarball(self):

		import tarfile

		ZFile = tarfile.TarFileCompat(self.filename, 'r', tarfile.TAR_GZIPPED)
		self.filename = self.filename.split('.tar.gz')[0]

#		os.mkdir(os.path.join(os.getcwd(), self.filename))

		for name in ZFile.namelist():

			gtk.threads_enter()
			img = self.txview.render_icon("gtk-file", gtk.ICON_SIZE_MENU)
			self.txbuff.insert_pixbuf(self.txbuff.get_end_iter(),img)
			self.txbuff.insert(self.txbuff.get_end_iter(), name + '\n')
			self.txview.scroll_mark_onscreen(self.txbuff.get_insert())
			gtk.threads_leave()

			rootdir = os.getcwd()

			if not os.path.exists(os.path.split(name)[0]):

				try:
					curpath = os.path.join(rootdir, os.path.split(name)[0])
					os.makedirs(curpath)
					time.sleep(0.2)
				except:
					pass

			file = open(os.path.join(rootdir, name), 'wb')
			file.write(ZFile.read(name))
			file.close()

		ZFile.close()

	def unzip(self):

		import zipfile
		ZFile = zipfile.ZipFile(self.filename, 'r')
		self.filename = self.filename.split('.zip')[0]

#		os.mkdir(os.path.join(os.getcwd(), self.filename))

		for name in ZFile.namelist():

			gtk.threads_enter()
			img = self.txview.render_icon("gtk-file", gtk.ICON_SIZE_MENU)
			self.txbuff.insert_pixbuf(self.txbuff.get_end_iter(),img)
			self.txbuff.insert(self.txbuff.get_end_iter(), name + '\n')
			self.txview.scroll_mark_onscreen(self.txbuff.get_insert())
			gtk.threads_leave()

			curpath = os.getcwd()

			if not os.path.exists(os.path.split(name)[0]):

				try:
					curpath = os.path.join(curpath, os.path.split(name)[0])
					os.makedirs(curpath)
					time.sleep(0.2)
				except:
					pass

			if name.endswith('/'):

				continue

			file = open(os.path.join(os.getcwd(), name), 'wb')
			file.write(ZFile.read(name))
			file.close()

		ZFile.close()

	def untar(self):

		import tarfile
		ZFile = tarfile.TarFile(self.filename, 'r')

		for member in ZFile.getmembers():

			gtk.threads_enter()
			img = self.txview.render_icon("gtk-file", gtk.ICON_SIZE_MENU)
			self.txbuff.insert_pixbuf(self.txbuff.get_end_iter(),img)
			self.txbuff.insert(self.txbuff.get_end_iter(), member.name + '\n')
			self.txview.scroll_mark_onscreen(self.txbuff.get_insert())
			gtk.threads_leave()

			ZFile.extract(member, os.getcwd())

		ZFile.close()

	def gunzip(self):

		import gzip
		ZFile = gzip.GzipFile(self.filename, 'rb')
		self.filename = self.filename.split('.gz')[0]

		outfile = open(self.filename, 'wb')

		while True:
			chunk = ZFile.read(65536)
			if not chunk:
				break

			outfile.write(chunk)

		ZFile.close()
		outfile.close()

	def bunzip(self):

		import bz2
		ZFile = bz2.BZ2File(self.filename, 'r')
		self.filename = self.filename.split('.bz2')[0]

		outfile = open(self.filename, 'wb')

		while True:
			chunk = ZFile.read(65536)
			if not chunk:
				break

			outfile.write(chunk)

		ZFile.close()
		outfile.close()


	def run(self):

		gtk.threads_enter()
		img = self.txview.render_icon("gtk-sort-descending", gtk.ICON_SIZE_MENU)
		self.txbuff.insert_pixbuf(self.txbuff.get_end_iter(),img)
		self.txbuff.insert(self.txbuff.get_end_iter(), 'Unarchiving ' + self.filename + '...\n')
		self.txview.scroll_mark_onscreen(self.txbuff.get_insert())
		gtk.threads_leave()

		if   self.filename.endswith('.tar.gz'):

			self.tarball()

		elif self.filename.endswith('.tar.bz2'):

			self.bunzip()
			self.untar()

			os.remove(self.filename)

		elif self.filename.endswith('.gz'):

			self.gunzip()

		elif self.filename.endswith('.bz2'):

			self.bunzip()

		elif self.filename.endswith('.zip'):

			self.unzip()

		elif self.filename.endswith('.tar'):

			self.untar()

		self.pulse.set_finished()
		gtk.threads_enter()
		self.reload()
		gtk.threads_leave()

# create md5 checksum ==========================================================
class CreateMD5(threading.Thread):

	def __init__(self, filename, pulse, reload):

		self.filename = filename
		self.pulse = pulse
		self.reload = reload

		threading.Thread.__init__(self)

	def run(self):

		cwd = os.getcwd()
		filename = self.filename

		check = open(filename,"rb")

		m = md5.new()
		while True:
			chunk = check.read(65536)
			if not chunk:
				break

			m.update(chunk)

		check.close()
		hash = m.hexdigest()

		hashfile = open(os.path.join(cwd, filename + ".md5"), "w")
		hashfile.write(hash + ' ' + filename)
		hashfile.close()

		self.pulse.set_finished()
		gtk.threads_enter()
		self.reload()
		gtk.threads_leave()

# upload queue and handling ====================================================
class Upload(threading.Thread):

# TODO: handle exceptions with user feedback

	def __init__(self, ftp, ulmodel, pulse, reload):

		self.Cancel = False

		self.ftp = ftp
		self.iter = None
		self.ul_model = ulmodel
		self.pulse = pulse
	
		self.reload = reload

		self.bytes_read = 0

		threading.Thread.__init__(self)

	def run(self):

		self.ul_model.foreach(self.put)
		gtk.threads_enter()
		self.ul_model.clear()
		gtk.threads_leave()

		self.pulse.set_finished()
		gtk.threads_enter()
		self.reload()
		gtk.threads_leave()

	def put(self, model, path, iter, user_data=None):

		if self.Cancel:
			return

		if self.ul_model.get_value(iter,4) == "gtk-stop":
			return

		ulfile = model.get_value(iter, 1)
		self.iter = iter

		gtk.threads_enter()
		self.ul_model.set(iter,3,"?? Unknown ??")
		gtk.threads_leave()

		if model.get_value(iter, 0) == "gtk-directory":

			loccwd = os.getcwd()
			remcwd = self.ftp.pwd()

			self.ftp.mkd(ulfile)
			self.ftp.cwd(ulfile)
			os.chdir(ulfile)

			print "chdir: " + ulfile + "\n"

			gtk.threads_enter()
			self.reload()
			gtk.threads_leave()

			for root, dirs, files in os.walk(os.getcwd()):

				if self.ul_model.get_value(self.iter,4) == "gtk-stop":
					return

				for name in dirs:

					if self.ul_model.get_value(self.iter,4) == "gtk-stop":
						return

					print "mkdir: " + name + "\n"

					self.ftp.mkd(name)
					self.ftp.cwd(name)
					os.chdir(name)

					for r2, d2, f2 in os.walk(os.getcwd()):

						if self.ul_model.get_value(self.iter,4) == "gtk-stop":
							return

						for ulleaf in f2:

							if self.ul_model.get_value(self.iter,4) == "gtk-stop":
								return

							print "upload: " + ulleaf + "\n"
							gtk.threads_enter()
							self.ul_model.set(self.iter,3,ulleaf)
							gtk.threads_leave()
							self.put_file(ulleaf)

				print "chdir to root: " + root + '\n'

				os.chdir(root)
				self.ftp.cwd(remcwd)
				self.ftp.cwd(ulfile)

				print "ftp dir: " + self.ftp.pwd() + "\n"

				for name in files:

					if self.ul_model.get_value(self.iter,4) == "gtk-stop":
						return

					print "upload: " + name + "\n"
					gtk.threads_enter()
					self.ul_model.set(self.iter,3,name)
					gtk.threads_leave()
					self.put_file(name)

			self.ftp.cwd(remcwd)
			os.chdir(loccwd)
		else:
			self.put_file(ulfile)

	def put_file(self, ulfile):

		if not self.ul_model.iter_is_valid(self.iter):
			return

		if self.ul_model.get_value(self.iter,4) == "gtk-stop":
			return

		match_text = re.compile(r".*\.(txt|log|htm|html|css|js|c|cpp|cc|php|pl|py|nfo|java|xml|glade|sh|inc|ini|int)$", re.IGNORECASE)

		for i in range(2):

			try:
				gtk.threads_enter()
				self.ul_model.set(self.iter,4,"gtk-go-up")
				gtk.threads_leave()

				if match_text.match(ulfile):

					self.ftp.storlines("STOR " + ulfile, open(ulfile))
				else:
					self.ftp.storbinary("STOR " + ulfile, open(ulfile, "rb"))

				gtk.threads_enter()
				self.ul_model.set(self.iter,3,"Upload Complete")
				self.ul_model.set(self.iter,4,"gtk-apply")
				gtk.threads_leave()
				time.sleep(0.3)

				break
			except:
				print "Error in upload attempt #" + str(i + 1) + " on " + ulfile

				gtk.threads_enter()
				self.ul_model.set(self.iter,3,"Error Uploading")
				self.ul_model.set(self.iter,4,"gtk-dialog-error")
				gtk.threads_leave()
				time.sleep(0.6)

	def set_cancel(self):

		self.Cancel = True

# download queue and handling ==================================================
class Download(threading.Thread):

	def __init__(self, ftp, dlmodel, reload):

		self.Cancel = False

		self.ftp = ftp
		self.dl_model = dlmodel
		self.outfile = None
		self.diter = None

		self.reload = reload
		self.bytes_read = 0
		self.cwd = os.getcwd()

		threading.Thread.__init__(self)

	def run(self):

		self.dl_model.foreach(self.get_file)

		gtk.threads_enter()
		self.dl_model.clear()
		self.reload()
		gtk.threads_leave()
		time.sleep(0.3)

	def get_file(self, model, path, iter, user_data=None):

		if self.Cancel:
			return

		if model.get_value(iter,5) == "gtk-stop":
			return

		if model.get_value(iter, 3) > 0:
			return

		if self.outfile != None:
			return

		self.bytes_read = 0
		self.diter = iter

		dlfile = model.get_value(iter, 1)
		match_text = re.compile(r".*\.(txt|log|htm|html|css|js|c|cpp|cc|php|pl|py|nfo|java|xml|glade|sh|inc|ini|int)$", re.IGNORECASE)

		if os.path.split(dlfile)[0] != '':

			dirs = os.path.split(dlfile)[0].split('/')
			cd = os.path.join(*dirs)

#			print cd

			if not os.path.exists(cd):

				try:
					os.makedirs(cd)
				except:
					pass

		if match_text.match(dlfile):

			self.get_text(dlfile)

		else:
			self.get_binary(dlfile)

		del self.outfile
		self.outfile = None

		time.sleep(0.3)


	def down_text(self, buffer):

		if not self.dl_model.iter_is_valid(self.diter):
			return

		if self.dl_model.get_value(self.diter,5) == "gtk-stop":
			return

		final_size = long(self.dl_model.get_value(self.diter, 2))
		self.bytes_read += len(buffer) + 1

# TODO: +2 above for DOS style CR text

		percent = int( float(float(self.bytes_read) / float(final_size)) * 100)

		gtk.threads_enter()
		self.dl_model.set(self.diter,3, percent)
		self.dl_model.set(self.diter,4, str(self.bytes_read))
		gtk.threads_leave()

		self.outfile.write(buffer + '\n')


	def get_text(self, filename):

		if not self.dl_model.iter_is_valid(self.diter):
			return

		if self.outfile is None:
			self.outfile = open(filename, "w")

		for i in range(2):

			try:

				gtk.threads_enter()
				self.dl_model.set(self.diter,5,"gtk-go-down")
				gtk.threads_leave()

				self.ftp.retrlines("RETR " + filename, lambda s, w=self.down_text: w(s))
				self.outfile.close()

				gtk.threads_enter()
				self.dl_model.set(self.diter,5,"gtk-apply")
				gtk.threads_leave()

				break

			# permission error
			except: # ftplib.error_perm:

				gtk.threads_enter()
				self.dl_model.set(self.diter,5,"gtk-dialog-error")
				gtk.threads_leave()

				print "Error in text download attempt #" + str(i + 1) + " on " + filename
				self.outfile.close()
				if os.path.exists(filename):
					os.remove(filename)
				time.sleep(0.5)

	def down_binary(self, buffer):

		if not self.dl_model.iter_is_valid(self.diter):
			return

		if self.dl_model.get_value(self.diter,5) == "gtk-stop":
			return


		final_size = long(self.dl_model.get_value(self.diter, 2))
		self.bytes_read += len(buffer)

		percent = int( float(float(self.bytes_read) / float(final_size)) * 100)

		gtk.threads_enter()
		self.dl_model.set(self.diter,3, percent)
		self.dl_model.set(self.diter,4, str(self.bytes_read))
		gtk.threads_leave()

		self.outfile.write(buffer)


	def get_binary(self, filename):

		if not self.dl_model.iter_is_valid(self.diter):
			return

		if self.outfile is None:

			self.outfile = open(filename, "wb")

		for i in range(2):

			try:

				gtk.threads_enter()
				self.dl_model.set(self.diter,5,"gtk-go-down")
				gtk.threads_leave()

				self.ftp.retrbinary("RETR " + filename, self.down_binary)
				self.outfile.close()

				gtk.threads_enter()
				self.dl_model.set(self.diter,5,"gtk-apply")
				gtk.threads_leave()

				break

			# permission error
			except: # ftplib.error_perm:

				gtk.threads_enter()
				self.dl_model.set(self.diter,5,"gtk-dialog-error")
				gtk.threads_leave()

				print "Error in binary download attempt #" + str(i + 1) + " on " + filename
				self.outfile.close()

				try:
					if os.path.exists(filename):
						os.remove(filename)
				except:
					time.sleep(0.5)
					continue

				time.sleep(0.5)

	def set_cancel(self):

		self.Cancel = True

# ==============================================================================