(* $Id: frontend.ml,v 1.27 2001/12/15 18:01:33 gerd Exp $
 * ----------------------------------------------------------------------
 *
 *)

open Findlib;;

exception Usage;;

type mode =
    M_use | M_query | M_install | M_remove | M_compiler of string | M_dep
  | M_printconf | M_guess | M_list
;;


type psubst =
    Const of string
  | Percent of char
;;


let percent_subst spec s =
  (* spec = [ 'c', [ "ctext1"; "ctext2"; ... ];
   *          'd', [ "dtext1"; "dtext2"; ... ] ]
   * All occurrences of %c in the string s are replaced as specified in spec.
   * spec is an association list with the characters of the %-notation as keys
   * and lists of strings as values. The result is a list of strings containing
   * every combination of substituted values.
   *)

  let l = String.length s in
  let rec preprocess i j =
    if j<l then begin
      match s.[j] with
	'%' ->
	  if j+1<l then begin
	    let c = s.[j+1] in
	    Const(String.sub s i (j-i)) :: Percent c :: preprocess (j+2) (j+2)
	  end
	  else failwith "bad format string"
      |	_ ->
	  preprocess i (j+1)
    end
    else
      if i<j then
	[Const(String.sub s i (j-i))]
      else
	[]
  in

  let rec subst prefix l =
    match l with
      [] -> [prefix]
    | Const s :: l' ->
	subst (prefix ^ s) l'
    | Percent c :: l' ->
	let replacements =
	  try List.assoc c spec
	  with Not_found -> failwith "bad format string" in
	List.flatten
	  (List.map
	     (fun replacement ->
	       subst (prefix ^ replacement) l')
	     replacements)
  in

  subst "" (preprocess 0 0)
;;


let rec remove_dups l =
  match l with
    x :: l' ->
      if List.mem x l' then remove_dups l' else x::remove_dups l'
  | [] -> []
;;


let arg n =
  if n < Array.length Sys.argv then Sys.argv.(n) else raise Not_found
;;


let use_package prefix pkgnames =
  let pdirs =
    List.map
      (fun pname ->
        try
          "-I " ^ package_directory pname
        with
	  Not_found -> failwith ("Cannot find package " ^ pname))
      pkgnames
  in

  print_endline (prefix ^ String.concat " " pdirs)
;;


let conflict_report incpath =
  (* First check whether there are several definitions for packages
   * in the current path
   *)
  Metacache.package_conflict_report();

  (* Second check whether there are module conflicts *)
  Metacache_unix.module_conflict_report incpath
;;


let check_package_list l =
  List.iter
    (fun pkg ->
       try
	 let _ = package_directory pkg in
	 ()
       with
	   Not_found ->
	     failwith ("package '" ^ pkg ^ "' not found"))
    l
;;


let run_command verbose cmd args =
  if verbose then
    print_string ("+ " ^ cmd ^ " " ^
		  String.concat " " args ^ "\n");
  
  flush stdout;

  let pid =
    Unix.create_process
      cmd
      (Array.of_list (cmd :: args))
      Unix.stdin
      Unix.stdout
      Unix.stderr
  in

  let (_,status) = Unix.waitpid [] pid in
  begin
    match status with
      Unix.WEXITED 0 -> ()
    | Unix.WEXITED n ->
	if verbose then
	  print_string (cmd ^ " returned with exit code " ^ string_of_int n ^ "\n");
	exit n
    | Unix.WSIGNALED _ ->
	print_string (cmd ^ " got signal and exited\n");
	exit 2
    | Unix.WSTOPPED _ ->
	failwith "Your operating system does not work correctly"
  end
;;


(************************* format expansion *************************)


let expand predicates eff_packages format =

    (* format:
     * %p         package name
     * %d         package directory
     * %v         version
     * %a         archive file(s)
     * %A         archive files as single string
     * %o         link option(s)
     * %O         link options as single string
     *)

  List.flatten
    (List.map
       (fun pkg ->
	 let dir =
	   try package_directory pkg
	   with Not_found -> failwith ("package '" ^ pkg ^ "' not found")
	 in
	 let spec =
	   [ 'p',  [pkg];
             'd',  [dir];
	     'v',  [try package_property predicates pkg "version"
	            with Not_found -> "[unspecified]"];
	     'a',  Split.in_words
	             (try package_property predicates pkg "archive"
		      with Not_found -> "");
	     'A',  [String.concat " "
		       (Split.in_words
		          (try package_property predicates pkg "archive"
			   with Not_found -> ""))];
	     'o',  Split.in_words_ws
	             (try package_property predicates pkg "linkopts"
		      with Not_found -> "");
	     'O',  [String.concat " "
		       (Split.in_words_ws
		          (try package_property predicates pkg "linkopts"
			   with Not_found -> ""))];
	   ]
	 in
	 percent_subst spec format)
       eff_packages)
;;


(************************** QUERY SUBCOMMAND ***************************)

let query_package () =
  Arg.current := 1;

  let long_format =
    "package:    %p\nversion:    %v\narchive(s): %A\nlinkopts:   %O\nlocation:   %d\n" in
  let i_format =
    "-I %d" in
  let l_format =
    "-ccopt -L%d" in
  let a_format =
    "%a" in
  let o_format =
    "%o" in
  let p_format =
    "%p" in


  let predicates = ref [] in
  let format = ref "%d" in
  let separator = ref "\n" in
  let prefix = ref "" in
  let suffix = ref "\n" in
  let recursive = ref false in
  let descendants = ref false in

  let packages = ref [] in

  let append_predicate s =
    let pl = Split.in_words s in
    predicates := !predicates @ pl
  in


  Arg.parse
    [ "-predicates", Arg.String append_predicate,
                  "      specifies comma-separated list of assumed predicates";
      "-format", Arg.String (fun s -> format := s),
              "          specifies the output format";
      "-separator", Arg.String (fun s -> separator := s),
                 "       specifies the string that separates multiple answers";
      "-prefix", Arg.String (fun s -> prefix := s),
              "          a string printed before the first answer";
      "-suffix", Arg.String (fun s -> suffix := s),
              "          a string printed after the last answer";
      "-recursive", Arg.Set recursive,
                 "       select direct and indirect ancestors/descendants, too";
      "-descendants", Arg.Set descendants,
                   "     query descendants instead of ancestors (the default)";
      "-long-format", Arg.Unit (fun () -> format := long_format),
                   "     specifies long output format";
      "-i-format", Arg.Unit (fun () -> format := i_format),
                "        prints -I options for ocamlc";
      "-l-format", Arg.Unit (fun () -> format := l_format),
                "        prints -ccopt -L options for ocamlc";
      "-a-format", Arg.Unit (fun () -> format := a_format),
                "        prints names of archives to be linked in for ocamlc";
      "-o-format", Arg.Unit (fun () -> format := o_format),
                "        prints link options for ocamlc";
      "-p-format", Arg.Unit (fun () -> format := p_format),
                "        prints package names";
    ]
    (fun p -> packages := !packages @ [p])
"usage: ocamlfind query [ -predicates <p>  | -format <f> |
                         -long-format     | -i-format   |
                         -l-format        | -a-format   |
			 -o-format        | -p-format   |
                         -prefix <p>      | -suffix <s> |
                         -separator <s>   | 
                         -descendants     | -recursive  ] package ...";

    (* check packages: *)
    List.iter
      (fun pkg ->
	try
	  let _ = package_directory pkg in
	  ()
	with
	  Not_found ->
	    failwith ("package '" ^ pkg ^ "' not found"))
      !packages;


    let eff_packages =
      if !recursive then begin
	if !descendants then
	  Metacache_unix.users !packages
	else
	  package_deep_ancestors !predicates !packages
      end
      else
	!packages
    in

    let answers = expand !predicates eff_packages !format in

     print_string !prefix;
     print_string (String.concat !separator answers);
     print_string !suffix;
;;


(**************** preprocessor ******************************************)

let process_pp_spec syntax_preds packages pp_opts =
  (* Returns: pp_command *)

  let pp_packages =
    package_deep_ancestors syntax_preds packages in
  (* the packages used for the preprocessor *)

  let preprocessor_cmds =
    List.flatten
      (List.map (fun pname ->
		   try
		     [ pname,
		       package_property syntax_preds pname "preprocessor"
		     ]
		   with
		       Not_found -> []
		)
	        pp_packages
      )
  in
    
  let preprocessor_cmd =
    if syntax_preds <> [] then
      match preprocessor_cmds with
	  [] ->
	    failwith("When using -syntax, the META variable 'preprocessor' must be set")
	| [_, cmd] -> Some cmd
	| _ ->
	    failwith("No unique value for the META variable 'preprocessor': " ^
		     String.concat ", "
		       (List.map
			  (fun (n,v) ->
			  "package " ^ n ^ " defines `" ^ v ^ "'")
			  preprocessor_cmds
		       )
		    )
    else
      None
  in

  let pp_i_options =
    List.flatten
      (List.map
	 (fun pkg ->
	    let pkgdir = package_directory pkg in
	      [ "-I"; pkgdir ]
	 )
	 pp_packages) in
  
  let pp_archives =
    if preprocessor_cmd = None then
      []
    else
      List.flatten
	(List.map
	   (fun pkg ->
	      let al = try package_property syntax_preds pkg "archive"
	               with Not_found -> "" in
	      Split.in_words al
	   )
	   pp_packages) in
  
  match preprocessor_cmd with
      None -> []
    | Some cmd ->
	["-pp";
	 cmd ^ " " ^
	 String.concat " " (List.map Filename.quote pp_i_options) ^ " " ^
	 String.concat " " (List.map Filename.quote pp_archives) ^ " " ^
	 String.concat " " (List.map Filename.quote pp_opts)]
;;


(**************** OCAMLC/OCAMLMKTOP/OCAMLOPT subcommands ****************)

type pass_file_t =
    Pass of string
  | Impl of string
  | Intf of string
;;


let ocamlc which () =
  Arg.current := 1;

  let destdir = ref (default_location()) in

  let switches = ref [] in
  let pass_options = ref [] in
  let pass_files = ref [] in
  let incpath = ref [] in

  let dll_pkgs = ref [] in
  let dll_pkgs_all = ref false in

  let linkpkg = ref false in

  let packages = ref [] in
  let predicates = ref [] in
  let dontlink = ref [] in

  let syntax_preds = ref [] in
  let pp_opts = ref [] in
  let pp_specified = ref false in

  let add_switch name =
    Arg.Unit (fun () ->
                switches := name :: !switches;
                pass_options := !pass_options @ [name]) in
  let add_spec_fn name s =
    pass_options := !pass_options @ [name; s] in
  let add_spec name = Arg.String (add_spec_fn name) in
  let add_pkg =
    Arg.String (fun s -> packages := !packages @ (Split.in_words s)) in
  let add_pred =
    Arg.String (fun s -> predicates := !predicates @ (Split.in_words s)) in
  let add_dontlink =
    Arg.String (fun s -> dontlink := !dontlink @ (Split.in_words s)) in
  let add_syntax_pred =
    Arg.String (fun s -> syntax_preds := !syntax_preds @ (Split.in_words s)) in
  let add_pp_opt =
    Arg.String (fun s -> pp_opts := !pp_opts @ [s]) in
  let add_dll_pkg =
    Arg.String (fun s -> dll_pkgs := !dll_pkgs @ (Split.in_words s)) in


  Arg.parse
    (List.flatten
    [ [
      "-package", add_pkg,
               " <name>   Refer to package when compiling";
      "-linkpkg", Arg.Set linkpkg,
               "          Link the packages in";
      "-predicates", add_pred,
                  " <p>   Add predicate <p> when resolving package properties";
      "-dontlink", add_dontlink,
                " <name>  Do not link in package <name> and its ancestors";
      "-syntax", add_syntax_pred,
              " <p>       Use preprocessor with predicate <p>";
      "-ppopt", add_pp_opt,
             " <opt>      Append option <opt> to preprocessor invocation";
      "-dllpath-pkg", add_dll_pkg,
                   "<pkg> Add -dllpath for this package";
      "-dllpath-all", Arg.Set dll_pkgs_all,
                   "      Add -dllpath for all linked packages";
      "-passopt", Arg.String (fun s -> pass_options := !pass_options @ [s]),
               " <opt>    Pass option <opt> directly to ocamlc/opt/mktop\nSTANDARD OPTIONS:";

      "-a", add_switch "-a",
         "                Build a library";
      "-c", add_switch "-c",
         "                Compile only (do not link)";
      "-cc", add_spec "-cc",
          " <comp>        Use <comp> as the C compiler and linker";
      "-cclib", add_spec "-cclib",
             " <opt>      Pass option <opt> to the C linker";
      "-ccopt", add_spec "-ccopt",
             " <opt>      Pass option <opt> to the C compiler and linker";
      ];
      if which = "ocamlopt" then  [
      "-compact", add_switch "-compact",
               "          Optimize code size rather than speed"
      ]
      else [];
      if which <> "ocamlopt" then [
      "-custom", add_switch "-custom",
              "           Link in custom mode";
      "-dllib", add_spec "-dllib",
	     " <lib>      Use the dynamically-loaded library <lib>";
      ] else [];
      [
      "-dllpath", add_spec "-dllpath",
               " <dir>    Add <dir> to the run-time search path for shared libraries"];
      if which <> "ocamlopt" then [
      "-g", add_switch "-g",
         "                Save debugging information";
      ] else [];
      [
      "-i", add_switch "-i",
         "                Print the types";
      "-I", (Arg.String
	       (fun s ->
		  incpath := s :: !incpath;
		  add_spec_fn "-I" s)),
         " <dir>          Add <dir> to the list of include directories";
      "-impl", Arg.String (fun s -> pass_files := !pass_files @ [ Impl s ]),
            " <file>      Compile <file> as a .ml file";
      ]	;
      if which = "ocamlopt" then [
      "-inline", add_spec "-inline",
              " <n>       Set aggressiveness of inlining to <n>";
      ]	else [];
      [
      "-intf", Arg.String (fun s -> pass_files := !pass_files @ [ Intf s ]),
            " <file>      Compile <file> as a .mli file";
      "-intf-suffix", add_spec "-intf-suffix",
                   " <s>  Suffix for interface file (default: .mli)";
      "-intf_suffix", add_spec "-intf_suffix",
                   " <s>  same as -intf-suffix";
      "-labels", add_switch "-labels",
              "           Use commuting label mode";
      "-linkall", add_switch "-linkall",
               "          Link all modules, even unused ones";
      ]	;
      if which <> "ocamlopt" then [
      "-make-runtime", add_switch "-make-runtime",
                    "     Build a runtime system";
      "-make_runtime", add_switch "-make_runtime",
                    "     same as -make-runtime";
      ]	else [];
      [
      "-noautolink", add_switch "-noautolink",
                  "       Don't automatically link C libraries specif'd in .cma files";
      "-noassert", add_switch "-noassert",
                "         Do not compile assertion checks";
      "-nolabels", add_switch "-nolabels",
              "           Ignore non-optional labels in types";
      "-o", add_spec "-o",
         " <file>         Set output file name to <file>";
      "-output-obj", add_switch "-output-obj",
                  "       Output a C object file instead of an executable";
      ];
      if which = "ocamlopt" then [
      "-p", add_switch "-p",
         "                Compile/link with profiling support for \"gprof\"
                       (implies -predicates gprof)";
      ]	else if which = "ocamlcp" then [
      "-p", add_spec "-p",
	 " [afilmt]       Profile constructs specified by argument:
      a  Everything
      f  Function calls
      i  if ... then ... else
      l  while, for
      m  match ... with
      t  try ... with";
      ]	else [];
      [
      "-pp", Arg.String (fun s -> pp_specified := true;
			          add_spec_fn "-pp" s),
          " <command>     Pipe sources through preprocessor <command>";
      "-rectypes", add_switch "-rectypes",
                "         Allow arbitrary recursive types";
      ]	;
      if which = "ocamlopt" then [
      "-S", add_switch "-S",
         "                Keep intermediate assembly file";
      ]	 else [];
      [
      "-thread", add_switch "-thread",
              "           Use thread-safe standard library (implies -predicate mt)";
      "-unsafe", add_switch "-unsafe",
              "           No bounds checking on array and string access";
      ]	;
      if which <> "ocamlopt" then [
      "-use-runtime", add_spec "-use-runtime",
                   " <path>   Generate bytecode for the given runtime system";
      "-use_runtime", add_spec "-use_runtime",
                   "          same as -use-runtime";
      ]	 else [];
      [
      "-v", add_switch "-v",
         "                Print compiler version number";
      "-verbose", add_switch "-verbose",
               "          Print calls to external commands";
      "-w", add_spec "-w",
         " <flags>        Enable or disable warnings according to <flags>:
     A/a enable/disable all warnings
     C/c enable/disable suspicious comment
     D/d enable/disable deprecated features
     F/f enable/disable partially applied function
     L/l enable/disable labels omitted in application
     M/m enable/disable overriden methods
     P/p enable/disable partial match
     S/s enable/disable non-unit statement
     U/u enable/disable unused match case
     V/v enable/disable hidden instance variables
     X/x enable/disable all other warnings
     default setting is Al (all warnings enabled but labels enabled)";
      "-warn-error", add_spec "-warn-error",
                  "       Turn these warnings into errors";
      "-where", add_switch "-where",
             "            Print standard library directory";
      "-", Arg.String (fun s -> pass_files := !pass_files @  [ Pass s ]),
	 " <file>          Treat <file> as a file name (even if it starts with `-')";
       ]
    ])
    (fun s -> pass_files := !pass_files @ [ Pass s])
    ("usage: ocamlfind " ^ which ^ " [options] file ...");

  begin match which with
    "ocamlc"     -> predicates := "byte" :: !predicates;
  | "ocamlcp"    -> predicates := "byte" :: !predicates;
  | "ocamlmktop" -> predicates := "byte" :: "toploop" :: !predicates;
  | "ocamlopt"   -> predicates := "native" :: !predicates;
  | _            -> failwith "unsupported backend"
  end;

  if List.mem "-thread" !switches then begin
    predicates := "mt" :: !predicates;
    (* check type of threads: posix or bytecode *)
    try
      let type_of_threads = package_property [] "threads" "type_of_threads" in
      if type_of_threads = "posix" then
	predicates := "mt_posix" :: !predicates;
    with
      Not_found -> ()
  end;

  if List.mem "-p" !switches then
    predicates := "gprof" :: !predicates;

  if Findlib_config.ocaml_has_autolinking &&
     not (List.mem "-noautolink" !switches)
  then
    predicates := "autolink" :: !predicates;

  if !syntax_preds <> [] then begin
    predicates := "syntax" :: !predicates;
    syntax_preds := "preprocessor" :: "syntax" :: !syntax_preds;
  end;

  let verbose = List.mem "-verbose" !switches in

  if verbose then begin
    if !syntax_preds <> [] then
      print_string ("Effective set of preprocessor predicates: " ^
		    String.concat "," !syntax_preds ^ "\n");
    print_string ("Effective set of compiler predicates: " ^
		  String.concat "," !predicates ^ "\n");
  end;

  if !pp_specified && !syntax_preds <> [] then
    prerr_endline("Warning: -pp overrides the effect of -syntax partly");

  (* check packages: *)
  check_package_list !packages;
  check_package_list !dontlink;

  let eff_packages0 =
    package_deep_ancestors !predicates !packages in

  (* Reorder eff_packages, such that "threads" is always first *)
  (* TODO: find a better way *)
  let eff_packages =
    if List.mem "threads" eff_packages0 then begin
      [ "unix"; "threads" ] @ 
      List.filter (fun n -> n <> "unix" && n <> "threads") eff_packages0
    end
    else 
      eff_packages0
  in

  let eff_dontlink =
    package_deep_ancestors !predicates !dontlink in

  let eff_link =
    List.flatten
      (List.map
	 (fun pkg -> if List.mem pkg !dontlink then [] else [pkg])
	 eff_packages) in


  let eff_packages_dl =
    remove_dups (List.map package_directory eff_packages) in

  let eff_link_dl =
    remove_dups (List.map package_directory eff_link) in

  (* Conflict report: *)
  conflict_report (!incpath @ ["."; Findlib.ocaml_stdlib() ]);
(*
  let pp_packages =
    package_deep_ancestors !syntax_preds !packages in
  (* the packages used for the preprocessor *)

  let preprocessor_cmds =
    List.flatten
      (List.map (fun pname ->
		   try
		     [ pname,
		       package_property !syntax_preds pname "preprocessor"
		     ]
		   with
		       Not_found -> []
		)
	        pp_packages
      )
  in

  let preprocessor_cmd =
    if !syntax_preds <> [] then
      match preprocessor_cmds with
	  [] ->
	    failwith("When using -syntax, the META variable 'preprocessor' must be set")
	| [_, cmd] -> Some cmd
	| _ ->
	    failwith("No unique value for the META variable 'preprocessor': " ^
		     String.concat ", "
		       (List.map
			  (fun (n,v) ->
			  "package " ^ n ^ " defines `" ^ v ^ "'")
			  preprocessor_cmds
		       )
		    )
    else
      None
  in
*)
  let initf_file_needed =
    List.mem "toploop" !predicates in
  let initl_file_needed =
    List.mem "toploop" !predicates && List.mem "findlib" eff_link in

  let initf_file_name =
    if initf_file_needed then
      Filename.temp_file "findlib_initf" ".ml" 
    else
      ""
  in
  let initl_file_name =
    if initl_file_needed then
      Filename.temp_file "findlib_initl" ".ml"
    else
      ""
  in

  (* initf_file_name: the initialization code inserted at the beginning of
   *   the cma/cmo list (initf = init first)
   * initl_file_name: the initialization code inserted at the end of
   *   the cma/cmo list (initl = init last)
   *)

  if initf_file_needed then begin
    (* Extend list of -I directories *)
    let initf = open_out_gen
		  [Open_wronly; Open_trunc; Open_text]
		  0o777
		  initf_file_name in
    try
      List.iter
	(fun d ->
	   output_string initf ("Topdirs.dir_directory \"" ^ 
				String.escaped d ^ "\";;\n")
	)
	eff_link_dl;
      close_out initf;
    with
      any ->
	close_out initf;
	Sys.remove initf_file_name;
	raise any
  end;

  if initl_file_needed then begin
    (* Generate initializer for "findlib_top.cma" *)
    let initl = open_out_gen
		  [Open_wronly; Open_trunc; Open_text]
		  0o777
		  initl_file_name in
    try
      output_string initl
	("Topfind.don't_load [" ^
	 String.concat ";"
	   (List.map
	      (fun pkg -> "\"" ^ String.escaped pkg ^ "\"")
	      eff_link) ^
	 "];;\n");
      output_string initl
	("Topfind.predicates := [" ^
	 String.concat ";"
	   (List.map
	      (fun pred -> "\"" ^ String.escaped pred ^ "\"")
	      !predicates) ^
	 "];;\n");
      close_out initl;
    with
      any ->
	close_out initl;
	Sys.remove initl_file_name;
	raise any
  end;

  if initf_file_needed then
    at_exit
      (fun () ->
	let tr f x = try f x with _ -> () in
	tr Sys.remove initf_file_name;
	tr Sys.remove (Filename.chop_extension initf_file_name ^ ".cmi");
	tr Sys.remove (Filename.chop_extension initf_file_name ^ ".cmo");
      );
  if initl_file_needed then
    at_exit
      (fun () ->
	let tr f x = try f x with _ -> () in
	tr Sys.remove initl_file_name;
	tr Sys.remove (Filename.chop_extension initl_file_name ^ ".cmi");
	tr Sys.remove (Filename.chop_extension initl_file_name ^ ".cmo");
      );

  let stdlibdir = Split.norm_dir (Findlib.ocaml_stdlib()) in

  let i_options =
    List.flatten
      (List.map
	 (fun pkgdir ->
	    if Split.norm_dir pkgdir = stdlibdir then
	      []
	    else
	      [ "-I"; pkgdir;
		"-ccopt"; "-I" ^ pkgdir; ])
	 eff_packages_dl) in

  let l_options =
    List.flatten
      (List.map
	 (fun pkgdir ->
	    if Split.norm_dir pkgdir = stdlibdir then
	      []
	    else
	      [ "-ccopt"; "-L" ^ pkgdir; ])
	 eff_link_dl) in

  let archives =
    (if initf_file_needed then
       [ initf_file_name ]
     else 
       []
    ) 
    @
    List.flatten
      (List.map
	 (fun pkg ->
	   let al = try package_property !predicates pkg "archive"
	            with Not_found -> "" in
	   List.map
	     (fun a -> Filename.concat (package_directory pkg) a)
	     (Split.in_words al)
	 )
	 eff_link) 
    @
    (if initl_file_needed then
       [ initl_file_name ]
     else
       []
    )
  in

  let linkopts =
    List.flatten
      (List.map
	 (fun pkg ->
	   let ol = try package_property !predicates pkg "linkopts"
	            with Not_found -> "" in
	   Split.in_words_ws ol)
	 (List.rev eff_link)) in

  let pp_command = 
    if !pp_specified then
      []
    else
      process_pp_spec !syntax_preds !packages !pp_opts
  in

  let pass_files' =
    List.flatten
      (List.map
	 (function
	      Pass s ->
		if s.[0] = '-'
		then [ "-"; String.sub s 1 (String.length s - 1) ]
		else [ s ]
	    | Impl s ->
		[ "-impl"; s ]
	    | Intf s ->
		[ "-intf"; s ]
	 )
	 !pass_files)
  in

  let dll_dirs =
    remove_dups
      ((List.map package_directory !dll_pkgs) @   (* XXX *)
       (if !dll_pkgs_all then eff_link_dl else [])) in

  let dll_options =
    List.flatten
      (List.map
	 (fun pkg -> ["-dllpath";  pkg] )
	 dll_dirs) in

  let arguments =
    !pass_options @
    i_options @
    pp_command @
    (if !linkpkg then l_options else []) @
    (if !linkpkg then archives else []) @
    pass_files' @
    (if !linkpkg then linkopts else []) @
    dll_options
  in

  let actual_command =
    match which with
	"ocamlc"     -> Findlib.command `ocamlc
      | "ocamlopt"   -> Findlib.command `ocamlopt
      | "ocamlcp"    -> Findlib.command `ocamlcp
      | "ocamlmktop" -> Findlib.command `ocamlmktop
      | _            -> assert false
  in

  run_command verbose actual_command arguments
;;


(************************************************************************)

let ocamldep () =
  Arg.current := 1;

  let switches = ref [] in
  let pass_options = ref [] in
  let pass_files = ref [] in

  let packages = ref [] in
  let syntax_preds = ref [] in
  let pp_opts = ref [] in
  let pp_specified = ref false in

  let verbose = ref false in

  let add_switch name =
    Arg.Unit (fun () ->
                switches := name :: !switches;
                pass_options := !pass_options @ [name]) in
  let add_spec_fn name s =
    pass_options := !pass_options @ [name; s] in
  let add_spec name = Arg.String (add_spec_fn name) in
  let add_syntax_pred =
    Arg.String (fun s -> syntax_preds := !syntax_preds @ (Split.in_words s)) in
  let add_pp_opt =
    Arg.String (fun s -> pp_opts := !pp_opts @ [s]) in
  let add_pkg =
    Arg.String (fun s -> packages := !packages @ (Split.in_words s)) in

  Arg.parse
      [
	"-syntax", add_syntax_pred,
                " <p>       Use preprocessor with predicate <p>";
	"-package", add_pkg,
	         " <p>      Add preprocessor package <p>";
	"-ppopt", add_pp_opt,
               " <opt>      Append option <opt> to preprocessor invocation";
	"-passopt", Arg.String (fun s -> pass_options := !pass_options @ [s]),
                 " <opt>    Pass option <opt> directly to ocamlc/opt/mktop";
	"-verbose", Arg.Set verbose,
	         "          Print calls to external commands\nSTANDARD OPTIONS:";
	"-I", add_spec "-I",
           " <dir>          Add <dir> to the list of include directories";
	"-native", add_switch "-native",
                "           Generate dependencies for a pure native-code project";
	"-pp", Arg.String (fun s -> pp_specified := true;
		 	            add_spec_fn "-pp" s),
            " <command>     Pipe sources through preprocessor <command>";
      ]
      (fun s -> pass_files := !pass_files @ [ s])
      ("usage: ocamlfind ocamldep [options] file ...");

  check_package_list !packages;
  
  if !syntax_preds <> [] then
    syntax_preds := "preprocessor" :: "syntax" :: !syntax_preds;
  
  if !verbose && !syntax_preds <> [] then
    print_string ("Effective set of preprocessor predicates: " ^
		  String.concat "," !syntax_preds ^ "\n");

  if !pp_specified && !syntax_preds <> [] then
    prerr_endline("Warning: -pp overrides the effect of -syntax partly");

  let pp_command = 
    if !pp_specified then
      []
    else
      process_pp_spec !syntax_preds !packages !pp_opts
  in

  let arguments =
    !pass_options @
    pp_command @
    !pass_files
  in

  let actual_command = Findlib.command `ocamldep in

  run_command !verbose actual_command arguments
;;


(************************************************************************)


let copy_file ?(rename = (fun name -> name)) ?(append = "") src dstdir =
  (* A system-independent function to copy the file src to dstdir *)
  let outname = rename (Filename.basename src) in
  let ch_in = open_in_bin src in
  (* Determine the permissions of the file: the permissions of the
   * user bits are extended to all groups (user, group, world bits),
   * and the umask is applied to the result.
   *)
  let s = Unix.stat src in
  let perm = s.Unix.st_perm in
  let user_perm = (perm land 0o700) lsr 6 in
  let perm' = user_perm lor (user_perm lsl 3) lor (user_perm lsl 6) in
  try
    let outpath = Filename.concat dstdir outname in
    if Sys.file_exists outpath then
      prerr_endline ("ocamlfind: [WARNING] Overwriting file " ^ outpath);
    let ch_out = open_out_gen 
		   [Open_wronly; Open_creat; Open_trunc; Open_binary]
		   perm'
		   outpath in
    try
      let buflen = 4096 in
      let buf = String.create buflen in
      let pos = ref 0 in
      let len = ref (input ch_in buf 0 buflen) in
      while !len > 0 do
	output ch_out buf !pos !len;
	len := input ch_in buf !pos buflen;
      done;
      output_string ch_out append;
      close_out ch_out;
      close_in ch_in;

      prerr_endline("Installed " ^ outpath);
    with
	exc -> close_out ch_out; raise exc
  with
      exc -> close_in ch_in; raise exc
;;


let install_create_directory pkgname dstdir =
  try
    Unix.mkdir dstdir 0o777
  with
      Unix.Unix_error(Unix.EEXIST,_,_) ->
	failwith ("Package " ^ pkgname ^ " is already installed; please remove it first - (directory " ^ dstdir ^ " already exists)")
    | Unix.Unix_error(Unix.ENOENT,_,_)
    | Unix.Unix_error(Unix.ENOTDIR,_,_) ->
	failwith ("Bad configuration: Cannot mkdir " ^ dstdir ^ " because a path component does not exist or is not a directory")
    | Unix.Unix_error(e,_,_) ->
	failwith ("Cannot mkdir " ^ dstdir ^ ": " ^
		  Unix.error_message e)
;;


let read_ldconf() =
  let filename = Filename.concat (Findlib.ocaml_stdlib()) "ld.conf" in
  let lines = ref [] in
  let f = open_in filename in
  try
    while true do
      lines := (input_line f) :: !lines
    done;
    assert false
  with
      End_of_file ->
	close_in f;
	List.rev !lines
    | other ->
	close_in f;
	raise other
;;


let write_ldconf lines new_lines =
  let filename = Filename.concat (Findlib.ocaml_stdlib()) "ld.conf" in
  let f = open_out filename in
  try
    List.iter
      (fun line -> output_string f (line ^ "\n"))
      (lines @ new_lines);
    close_out f;
    prerr_endline("Updated " ^ filename);
  with
      Sys_error e ->
	prerr_endline ("ocamlfind: [WARNING] Cannot write " ^ filename);
	prerr_endline ("Reason: " ^ e);
	prerr_endline ("This file contains the directories with DLLs.");
	if new_lines <> [] then begin
	  prerr_endline ("It is recommended to add the following line(s) to this file:");
	  List.iter prerr_endline new_lines
	end
;;


exception Skip_file;;

let install_package () =
  let destdir = ref (default_location()) in
  let metadir = ref (meta_directory()) in
  let don't_add_directory_directive = ref false in
  let don't_modify_ldconf = ref false in
  let pkgname = ref "" in
  let files = ref [] in

  let keywords =
    [ "-destdir", (Arg.String (fun s -> destdir := s)),
              ("<path>    Set the destination directory (default: " ^ 
	       !destdir ^ ")");
      "-metadir", (Arg.String (fun s -> metadir := s)),
              ("<path>    Install the META file into this directory (default: "^
	       (if !metadir = "" then "none" else !metadir) ^ ")");
      "-dont-add-directory-directive", (Arg.Set don't_add_directory_directive),
               " never append directory='...' to META";
      "-dont-modify-ldconf", (Arg.Set don't_modify_ldconf),
               " do not update the ld.conf file";
    ] in
  let errmsg = "usage: ocamlfind install [options] <package_name> <file> ..." in

  Arg.current := 1;
  Arg.parse
        keywords
	(fun s ->
	   if !pkgname = ""
	   then pkgname := s
	   else files := s :: !files
	)
	errmsg;
  if !pkgname = "" then (Arg.usage keywords errmsg; exit 1);

  (* Check whether META exists: *)
  let meta_dot_pkg = "META." ^ !pkgname in
  let has_meta =
    List.exists
      (fun p ->
	 let b = Filename.basename p in
	 b = "META" || b = meta_dot_pkg)
      !files
  in
  if not has_meta then
    failwith "The META file is missing";

  (* Check whether there are DLLs: *)
  let contains_dlls =
    List.exists
      (fun p ->
	 Filename.check_suffix p ".so" || Filename.check_suffix p ".dll")
      !files
  in

  (* Check that there is no meta_dot_pkg: *)
  if Sys.file_exists (Filename.concat !metadir meta_dot_pkg) then
    failwith ("Package " ^ !pkgname ^ " is already installed; please remove it first - (file " ^ Filename.concat !metadir meta_dot_pkg ^ " already exists)");

  (* Create the package directory: *)
  let pkgdir = Filename.concat !destdir !pkgname in
  install_create_directory !pkgname pkgdir;

  (* Now copy the files into the package directory: *)
  let has_metadir = !metadir <> "" in
  List.iter
    (fun p ->
       try
	 copy_file
	   ~rename: (fun f ->
		       if has_metadir then begin
			 if f = "META" || f = meta_dot_pkg
			 then raise Skip_file
			 else f
		       end
		       else
			 if f = meta_dot_pkg then "META" else f)
	   p
	   pkgdir
       with
	   Skip_file -> ()
    )
    !files;

  (* Copy META into metadir, if this has been requested *)
  if has_metadir then begin
    List.iter
      (fun p ->
	 let b = Filename.basename p in
	 if b = "META" || b = meta_dot_pkg then
	   copy_file
	     ~rename: (fun f ->
			 if f = "META" then meta_dot_pkg else f)
	     ~append: ("\ndirectory=\"" ^ pkgdir ^ "\" # auto-added by ocamlfind\n")
	     p
	     !metadir
      )
      !files
  end;

  (* Extend ld.conf if necessary: *)
  if contains_dlls && not !don't_modify_ldconf then begin
    if Sys.file_exists (Filename.concat (Findlib.ocaml_stdlib()) "ld.conf") then
      begin
	let lines = read_ldconf() in
	write_ldconf lines [ pkgdir ]
      end 
    else
      prerr_endline("ocamlfind: [WARNING] You have installed DLLs but there is no ld.conf")
  end
;;


let remove_package () =
  let destdir = ref (default_location()) in
  let destdir_set = ref false in
  let metadir = ref (meta_directory()) in
  let don't_modify_ldconf = ref false in
  let pkgname = ref "" in

  let keywords =
    [ "-destdir", (Arg.String (fun s -> destdir := s; destdir_set := true)),
              ("<path>    Set the destination directory (default: " ^ 
	       !destdir ^ ")");
      "-metadir", (Arg.String (fun s -> metadir := s)),
              ("<path>    Remove the META file from this directory (default: " ^
	       (if !metadir = "" then "none" else !metadir) ^ ")");
      "-dont-modify-ldconf", (Arg.Set don't_modify_ldconf),
              " do not update the ld.conf file";
    ] in
  let errmsg = "usage: ocamlfind remove [options] <package_name>" in

  Arg.current := 1;
  Arg.parse
        keywords
	(fun s ->
	   if !pkgname = ""
	   then pkgname := s
	   else raise (Arg.Bad "too many arguments")
	)
	errmsg;
  if !pkgname = "" then (Arg.usage keywords errmsg; exit 1);

  let meta_dot_pkg = "META." ^ !pkgname in
  let has_metadir = !metadir <> "" in
  let pkgdir = Filename.concat !destdir !pkgname in

  (* Warn if there is another package with the same name: *)
  let other_pkgdir = 
    try Findlib.package_directory !pkgname with Not_found -> "" in
  if other_pkgdir <> "" && not !destdir_set then begin
    (* Is pkgdir = other_pkgdir? - We check physical identity: *)
    try
      let s_other_pkgdir = Unix.stat other_pkgdir in
      try
	let s_pkgdir = Unix.stat pkgdir in
	if (s_pkgdir.Unix.st_dev <> s_other_pkgdir.Unix.st_dev) ||
	   (s_pkgdir.Unix.st_ino <> s_other_pkgdir.Unix.st_ino) 
	then
	  prerr_endline("ocamlfind: [WARNING] You are removing the package from " ^ pkgdir ^ " but the currently visible package is at " ^ other_pkgdir ^ "; you may want to specify the -destdir option");
      with
	  Unix.Unix_error(Unix.ENOENT,_,_) ->
	    prerr_endline("ocamlfind: [WARNING] You are trying to remove the package from " ^ pkgdir ^ " but the currently visible package is at " ^ other_pkgdir ^ "; you may want to specify the -destdir option");
    with
	Unix.Unix_error(_,_,_) -> ()    (* ignore, it's only a warning *)
  end;

  (* If there is a metadir, remove the META file from it: *)
  if has_metadir then begin
    let f = Filename.concat !metadir meta_dot_pkg in
    if Sys.file_exists f then begin
      Sys.remove f;
      prerr_endline ("Removed " ^ f);
    end
    else
      prerr_endline ("ocamlfind: [WARNING] No such file: " ^ f)
  end;

  (* Remove the files from the package directory: *)
  if Sys.file_exists pkgdir then begin
    try
      let files = Metacache_unix.list_dir pkgdir in
      List.iter (fun f -> Sys.remove (Filename.concat pkgdir f)) files;
      Unix.rmdir pkgdir;
      prerr_endline ("Removed " ^ pkgdir)
    with
	Unix.Unix_error(e,_,s) ->
	  failwith
	    ((if s <> "" then s ^ ": " else "") ^
	     Unix.error_message e)
  end
  else
    prerr_endline("ocamlfind: [WARNING] No such directory: " ^ pkgdir);

  (* Modify ld.conf *)
  if not !don't_modify_ldconf then begin
    if Sys.file_exists (Filename.concat (Findlib.ocaml_stdlib()) "ld.conf") then
      begin
	let lines = read_ldconf() in
	let d = Split.norm_dir pkgdir in
	let exists = List.exists (fun p -> Split.norm_dir p = d) lines in
	if exists then begin
	  let lines' = List.filter (fun p -> Split.norm_dir p <> d) lines in
	  write_ldconf lines' []
	end
      end 
  end
;;


let guess_meta_file () =
  let pkgname = ref "" in
  let files = ref [] in

  let keywords = [] in
  let errmsg = "usage: ocamlfind guess [options] <package_name> <file> ..." in

  Arg.current := 1;
  Arg.parse
        keywords
	(fun s ->
	   if !pkgname = ""
	   then pkgname := s
	   else files := s :: !files
	)
	errmsg;
  if !pkgname = "" then (Arg.usage keywords errmsg; exit 1);

  Findlib_guess.guess_meta_file !pkgname !files
;;


let list_packages() =
  let packages = Metacache_unix.list_packages() in
  let packages_sorted = List.sort compare packages in

  Metacache.package_conflict_report();

  let n = 20 in
  List.iter
    (fun p ->
       let v_string =
	 try
	   let v = Findlib.package_property [] p "version" in
	   let spaces = String.make (max 1 (n-String.length p)) ' ' in
	   spaces ^ "(version: " ^ v ^ ")"
	 with
	     Not_found -> ""
       in
       print_endline (p ^ v_string)
    )
    packages_sorted
;;


let print_configuration() =
  let dir s =
    if Sys.file_exists s then
      s
    else
      s ^ " (not found)"
  in

  print_endline "Effective configuration:";
  Printf.printf "Configuration file:\n    %s\n"
    (dir Findlib_config.config_file);
  Printf.printf "Search path:\n";
  List.iter
    (fun p -> Printf.printf "    %s\n" (dir p))
    (Findlib.search_path());
  Printf.printf "Packages will be installed in/removed from:\n    %s\n"
    (dir (Findlib.default_location()));
  Printf.printf "META files will be installed in/removed from:\n    %s\n"
    (let md = Findlib.meta_directory() in
     if md = "" then "the corresponding package directories" else dir md
    );
  Printf.printf "The standard library is assumed to reside in:\n    %s\n"
    (Findlib.ocaml_stdlib());
  flush stdout
;;


let select_mode() =
  let m_string = try arg 1 with Not_found -> raise Usage in
  let m =
    match m_string with
      ("use"|"-use") ->          M_use
    | ("query"|"-query") ->      M_query
    | ("install"|"-install") ->  M_install
    | ("remove"|"-remove") ->    M_remove
    | ("ocamlc"|"-ocamlc") ->    M_compiler "ocamlc"
    | ("ocamlcp"|"-ocamlcp") ->  M_compiler "ocamlcp"
    | ("ocamlmktop"|"-ocamlmktop") -> M_compiler "ocamlmktop"
    | ("ocamlopt"|"-ocamlopt") -> M_compiler "ocamlopt"
    | ("ocamldep"|"-ocamldep") -> M_dep 
    | ("printconf"|"-printconf") -> M_printconf
    | ("guess"|"-guess") ->      M_guess
    | ("list"|"-list") ->        M_list
    | _ -> raise Usage
  in

  m
;;


let main() =
  try
    let m = select_mode() in
    let l = Array.length Sys.argv in
    let rest = Array.sub Sys.argv 2 (l-2) in
    match m with
      M_use      -> if rest = [| |] then raise Usage;
                    if rest.(0) = "-p" then begin
	              if l<4 then raise Usage;
	              use_package rest.(1)
	                          (List.tl(List.tl(Array.to_list rest)))
	            end
	            else
                      use_package "" (Array.to_list rest)
    | M_query    -> query_package ()
    | M_install  -> install_package()
    | M_remove   -> remove_package ()
    | M_printconf      -> print_configuration ()
    | M_guess ->          guess_meta_file()
    | M_list ->           list_packages()
    | M_compiler which -> ocamlc which ()
    | M_dep ->            ocamldep()    
  with
    Usage ->
      prerr_endline "usage: ocamlfind query      [-help | other options] <package_name> ...";
      prerr_endline "   or: ocamlfind ocamlc     [-help | other options] <file> ...";
      prerr_endline "   or: ocamlfind ocamlcp    [-help | other options] <file> ...";
      prerr_endline "   or: ocamlfind ocamlmktop [-help | other options] <file> ...";
      prerr_endline "   or: ocamlfind ocamlopt   [-help | other options] <file> ...";
      prerr_endline "   or: ocamlfind ocamldep   [-help | other options] <file> ...";
      prerr_endline "   or: ocamlfind install    [-help | other options] <package_name> <file> ...";
      prerr_endline "   or: ocamlfind remove     [-help | other options] <package_name>";
      prerr_endline "   or: ocamlfind guess      [-help] <package_name> <file> ...";
      prerr_endline "   or: ocamlfind list";
      prerr_endline "   or: ocamlfind printconf";
      exit 2
  | Failure f ->
      prerr_endline ("ocamlfind: " ^ f);
      exit 2
;;


try
  Sys.catch_break true;
  main()
with
  any ->
    prerr_endline ("Uncaught exception: " ^ Printexc.to_string any);
    let raise_again =
      try ignore(Sys.getenv "OCAMLFIND_DEBUG"); true
      with Not_found -> false
    in
    if raise_again then raise any;
    exit 3
;;


(* ======================================================================
 * History:
 *
 * $Log: frontend.ml,v $
 * Revision 1.27  2001/12/15 18:01:33  gerd
 * 	Fix: /tmp/findlib_initf* is cleaned up.
 * 	Change: Support for new O'Caml 3.04 arguments.
 * 	Change: The thread library is now always the first linked library.
 *
 * Revision 1.26  2001/10/13 13:16:43  gerd
 * 	New -dllpath-pkg, -dllpath-all options.
 * 	ld.conf is automatically kept in synch with packages.
 *
 * Revision 1.25  2001/10/12 20:16:41  gerd
 * 	When directory names are compared, they are now normalized.
 *
 * Revision 1.24  2001/10/12 15:04:15  gerd
 * 	ocaml-3.03
 *
 * Revision 1.23  2001/09/04 16:12:32  gerd
 * 	Splitted the init code for ocamlmktop in an early part and a late
 * part. The early init code section sets up the include path (-I).
 * 	Added ocamlfind ocamldep.
 *
 * Revision 1.22  2001/07/24 20:05:19  gerd
 * 	printconf prints the standard library
 *
 * Revision 1.21  2001/07/24 19:59:22  gerd
 * 	New query option -p-format.
 * 	Install/remove: usage message includes default values
 * 	Remove: Warning if the removed package is not the visible package
 * 	Overall: Using Findlib.ocaml_stdlib instead of Findlib_config.
 * The stdlib location can now be changed
 *
 * Revision 1.20  2001/03/27 20:22:34  gerd
 * 	copy_file: sets the permissions of the installed files
 * according to the umask and the originial permissions.
 *
 * Revision 1.19  2001/03/10 08:15:22  gerd
 * 	-warn-error
 *
 * Revision 1.18  2001/03/06 20:14:26  gerd
 * 	Added -where (for O'Caml 3.01).
 *
 * Revision 1.17  2001/03/04 23:00:56  gerd
 * 	Fix
 *
 * Revision 1.16  2001/03/03 19:28:34  gerd
 * 	Added conflict reports.
 *
 * Revision 1.15  2001/02/24 20:23:28  gerd
 * 	New subcommands: guess, list, printconf.
 * 	Improved subcommands: install, remove.
 *
 * Revision 1.14  2000/07/31 01:37:37  gerd
 * 	Added options -syntax, -ppopt.
 * 	Added support for OCAMLFIND_COMMANDS.
 *
 * Revision 1.13  2000/04/28 13:45:25  gerd
 * 	The compiler frontends do not produce -I and -L options for the
 * standard library directory anymore.
 *
 * Revision 1.12  2000/04/26 00:09:20  gerd
 * 	O'Caml 3 changes.
 *
 * Revision 1.11  2000/02/28 20:23:53  gerd
 * 	Bugfix: option -output-obj works now.
 *
 * Revision 1.10  2000/02/28 20:22:05  gerd
 * 	Minor change.
 *
 * Revision 1.8  2000/01/10 22:48:04  gerd
 * 	The relative order of files passed using -intf <file> and
 * -impl <file> is not changed.
 *
 * Revision 1.7  1999/11/10 23:45:17  gerd
 * 	The -dontlink option can now list several packages.
 *
 * Revision 1.6  1999/07/09 15:28:38  gerd
 * 	Added automatic recognition of POSIX threads. The META file in
 * the "threads" package has now a variable "type_of_threads" with
 * possible values "bytecode" and "posix". This variable is set when
 * "findlib" is configured. The compiler frontends query this variable
 * (with an empty predicate list), and add "mt_posix" to the predicate
 * list if the variable has the value "posix".
 *
 * Revision 1.5  1999/06/26 15:44:33  gerd
 * 	Some minor changes; the /tmp files are now removed always;
 * multiple directories are only passed once to the underlying compiler.
 *
 * Revision 1.4  1999/06/26 15:01:49  gerd
 * 	Added the -descendants option.
 *
 * Revision 1.3  1999/06/24 20:17:50  gerd
 * 	Further modifications (dont know which...)
 *
 * Revision 1.2  1999/06/20 19:26:24  gerd
 * 	Major change: Added support for META files. In META files, knowlege
 * about compilation options, and dependencies on other packages can be stored.
 * The "ocamlfind query" subcommand has been extended in order to have a
 * direct interface for that. "ocamlfind ocamlc/ocamlopt/ocamlmktop/ocamlcp"
 * subcommands have been added to simplify the invocation of the compiler.
 *
 * Revision 1.1  1999/03/26 00:02:47  gerd
 * 	Initial release.
 *
 *
 *)
