/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */

/*
 *  File-Roller
 *
 *  Copyright (C) 2001 The Free Software Foundation, Inc.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Street #330, Boston, MA 02111-1307, USA.
 */

#include <config.h>

#include <string.h>
#include <sys/types.h>
#include <unistd.h>

#include <gtk/gtk.h>
#include <gnome.h>
#include <glade/glade.h>
#include <libgnomevfs/gnome-vfs-mime.h>
#include <libgnomevfs/gnome-vfs-mime-handlers.h>
#include "file-utils.h"
#include "window.h"


typedef struct {
	FRWindow *   window;
	GtkWidget *  dialog;
	GtkWidget *  include_subfold_checkbutton;
	GtkWidget *  add_if_newer_checkbutton;
	GtkWidget *  exclude_symlinks;
	GtkWidget *  exclude_other_fs;
	GtkWidget *  exclude_backup_files;
	GtkWidget *  exclude_dot_files;
	GtkWidget *  ignore_case;
} DialogData;


static void
open_file_destroy_cb (GtkWidget *w,
		      GtkWidget *file_sel)
{
	DialogData *data;
	data = gtk_object_get_data (GTK_OBJECT (file_sel), "fr_dialog_data");
	g_free (data);
}


static int
open_file_ok_cb (GtkWidget *w,
                 GtkWidget *file_sel)
{
	FRWindow *window;
	char * file_sel_path, *path;
	DialogData *data;
	gboolean update;
	gchar *base_dir;

	data = gtk_object_get_data (GTK_OBJECT (file_sel), "fr_dialog_data");
	window = data->window;

	file_sel_path = gtk_file_selection_get_filename (GTK_FILE_SELECTION (file_sel));

        if (file_sel_path == NULL)
                return FALSE;

	path = remove_ending_separator (file_sel_path);
	base_dir = remove_level_from_path (path);

	if (window->add_default_dir != NULL) 
		g_free (window->add_default_dir);
	window->add_default_dir = g_strdup (base_dir);

	update = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (data->add_if_newer_checkbutton));

	if ((strchr (path, '*') == NULL) && (strchr (path, '?') == NULL)) {
		GList *files, *scan;
		GtkCList *file_clist;

		file_clist = GTK_CLIST (GTK_FILE_SELECTION (file_sel)->file_list);

		files = NULL;
		if (file_clist->selection == NULL)
			files = g_list_prepend (files, 
						file_name_from_path (path));

		for (scan = file_clist->selection; scan; scan = scan->next) {
			char *filename;
			gint row;
			
			row = GPOINTER_TO_INT (scan->data);
			gtk_clist_get_text (file_clist, row, 0, &filename);
			files = g_list_prepend (files, filename);
#ifdef DEBUG
			g_print ("add %s\n", filename);
#endif
		}

#ifdef DEBUG
		g_print ("basedir : %s\n", base_dir);
#endif

		if (files != NULL) {
			gchar *first = files->data;
			if (path_is_dir (first))
				fr_archive_add_directory (data->window->archive, file_name_from_path (first), base_dir, FALSE);
			else 
				fr_archive_add (data->window->archive,
						files,
						base_dir,
						update);
		}
		
		g_list_free (files);
	} else {
		gboolean recursive;
		gboolean no_symlinks;
		gboolean same_fs;
		gboolean no_backup_files;
		gboolean no_dot_files;
		gboolean ignore_case;

		recursive = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (data->include_subfold_checkbutton));
		no_symlinks = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (data->exclude_symlinks));
		same_fs = ! gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (data->exclude_other_fs));
		no_backup_files = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (data->exclude_backup_files));
		no_dot_files = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (data->exclude_dot_files));
		ignore_case = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (data->ignore_case));
		
		fr_archive_add_with_wildcard (data->window->archive,
					      file_name_from_path (path),
					      base_dir,
					      update,
					      recursive,
					      ! no_symlinks,
					      same_fs,
					      no_backup_files,
					      no_dot_files,
					      ignore_case);
	}

	g_free (path);
	g_free (base_dir);
	gtk_widget_destroy (file_sel);

	return TRUE;
}


static void
selection_entry_changed (GtkWidget *widget, 
			 DialogData *data)
{
	char *path;
	gboolean wildcard;

	path = gtk_entry_get_text (GTK_ENTRY (GTK_FILE_SELECTION (data->dialog)->selection_entry));

	wildcard = (strchr (path, '*') != NULL) || (strchr (path, '?') != NULL);
	gtk_widget_set_sensitive (data->include_subfold_checkbutton, wildcard);
	gtk_widget_set_sensitive (data->exclude_symlinks, GTK_TOGGLE_BUTTON (data->include_subfold_checkbutton)->active && wildcard);
	gtk_widget_set_sensitive (data->exclude_other_fs, GTK_TOGGLE_BUTTON (data->include_subfold_checkbutton)->active && wildcard);
	gtk_widget_set_sensitive (data->exclude_backup_files, wildcard);
	gtk_widget_set_sensitive (data->exclude_dot_files, wildcard);
	gtk_widget_set_sensitive (data->include_subfold_checkbutton, wildcard);
	gtk_widget_set_sensitive (data->ignore_case, wildcard);
}


static int
include_subfold_toggled_cb (GtkWidget *widget, 
			    gpointer callback_data)
{
	DialogData *data = callback_data;
	gtk_widget_set_sensitive (data->exclude_symlinks,
				  GTK_TOGGLE_BUTTON (widget)->active);
	gtk_widget_set_sensitive (data->exclude_other_fs,
				  GTK_TOGGLE_BUTTON (widget)->active);
	return FALSE;
}


/* create the "add" dialog. */
void
add_cb (GtkWidget *widget, 
	void *callback_data)
{
	GtkWidget *file_sel;
	DialogData *data;
	gchar *dir;
	GtkWidget *vbox, *hbox;
	GtkWidget *frame;

	data = g_new (DialogData, 1);
	data->window = callback_data;
	data->dialog = file_sel = gtk_file_selection_new (_("Add files"));
	gtk_file_selection_hide_fileop_buttons (GTK_FILE_SELECTION (file_sel));

	gtk_clist_set_selection_mode (GTK_CLIST (GTK_FILE_SELECTION (file_sel)->file_list), GTK_SELECTION_EXTENDED);

	data->add_if_newer_checkbutton = gtk_check_button_new_with_label (_("Add only if newer"));
	data->include_subfold_checkbutton = gtk_check_button_new_with_label (_("Include sub-directories"));
	data->exclude_symlinks = gtk_check_button_new_with_label (_("Exclude directories that are symbolic links"));
	data->exclude_other_fs = gtk_check_button_new_with_label (_("Exclude other file systems"));
	data->exclude_backup_files = gtk_check_button_new_with_label (_("Exclude backup files (*~)"));
	data->exclude_dot_files = gtk_check_button_new_with_label (_("Exclude hidden files (.*)"));
	data->ignore_case = gtk_check_button_new_with_label (_("Ignore case"));

	frame = gtk_frame_new (_("Options"));
	gtk_frame_set_shadow_type (GTK_FRAME (frame), GTK_SHADOW_ETCHED_IN);
	gtk_box_pack_start (GTK_BOX (GTK_FILE_SELECTION (file_sel)->action_area), frame, TRUE, TRUE, 0);

	vbox = gtk_vbox_new (FALSE, 3);
	gtk_container_set_border_width (GTK_CONTAINER (vbox), 5);
	gtk_container_add (GTK_CONTAINER (frame), vbox);

	gtk_box_pack_start (GTK_BOX (vbox), data->add_if_newer_checkbutton, 
			    TRUE, TRUE, 0);

	gtk_box_pack_start (GTK_BOX (vbox), gtk_hseparator_new (),
			    TRUE, TRUE, 0);

	gtk_box_pack_start (GTK_BOX (vbox), data->include_subfold_checkbutton,
			    TRUE, TRUE, 0);

	hbox = gtk_hbox_new (FALSE, 0);
	gtk_box_pack_start (GTK_BOX (hbox), data->exclude_symlinks,
			    TRUE, TRUE, 15);
	gtk_box_pack_start (GTK_BOX (vbox), hbox,
			    TRUE, TRUE, 0);

	hbox = gtk_hbox_new (FALSE, 0);
	gtk_box_pack_start (GTK_BOX (hbox), data->exclude_other_fs,
			    TRUE, TRUE, 15);
	gtk_box_pack_start (GTK_BOX (vbox), hbox,
			    TRUE, TRUE, 0);

	gtk_box_pack_start (GTK_BOX (vbox), gtk_hseparator_new (),
			    TRUE, TRUE, 0);

	hbox = gtk_hbox_new (FALSE, 0);
	gtk_box_pack_start (GTK_BOX (vbox), hbox,
			    TRUE, TRUE, 0);
	gtk_box_pack_start (GTK_BOX (hbox), data->exclude_backup_files,
			    TRUE, TRUE, 0);
	gtk_box_pack_start (GTK_BOX (hbox), data->exclude_dot_files,
			    TRUE, TRUE, 0);

	gtk_box_pack_start (GTK_BOX (vbox), gtk_hseparator_new (),
			    TRUE, TRUE, 0);

	gtk_box_pack_start (GTK_BOX (vbox), data->ignore_case,
			    TRUE, TRUE, 0);
	gtk_widget_show_all (frame);

	/* set data */

	dir = g_strconcat (data->window->add_default_dir, "/", "*", NULL);

	gtk_file_selection_set_filename (GTK_FILE_SELECTION (file_sel), dir);
	g_free (dir);
	gtk_editable_select_region (GTK_EDITABLE (GTK_FILE_SELECTION (data->dialog)->selection_entry), 0, -1);

	gtk_object_set_data (GTK_OBJECT (file_sel), "fr_dialog_data", data);

	gtk_toggle_button_set_state (GTK_TOGGLE_BUTTON (data->include_subfold_checkbutton), FALSE);
	gtk_toggle_button_set_state (GTK_TOGGLE_BUTTON (data->exclude_backup_files), TRUE);
	gtk_toggle_button_set_state (GTK_TOGGLE_BUTTON (data->exclude_dot_files), TRUE);

	/* signals */
	
        gtk_signal_connect (GTK_OBJECT (file_sel),
                            "destroy", 
                            (GtkSignalFunc) open_file_destroy_cb,
                            file_sel);

        gtk_signal_connect_object (GTK_OBJECT (GTK_FILE_SELECTION (file_sel)->cancel_button),
                            "clicked", 
                            (GtkSignalFunc) gtk_widget_destroy,
                            GTK_OBJECT (file_sel));

	gtk_signal_connect_after (GTK_OBJECT (GTK_FILE_SELECTION (file_sel)->ok_button),
			    "clicked",
                            (GtkSignalFunc) open_file_ok_cb, 
                            file_sel);

	gtk_signal_connect (GTK_OBJECT (GTK_FILE_SELECTION (file_sel)->selection_entry), "changed",
			    GTK_SIGNAL_FUNC (selection_entry_changed),
			    data);

	gtk_signal_connect (GTK_OBJECT (data->include_subfold_checkbutton),
                            "toggled", 
                            (GtkSignalFunc) include_subfold_toggled_cb,
			    data);

	gtk_window_set_modal (GTK_WINDOW (file_sel),TRUE);
	gtk_widget_show (file_sel);
}
