/* $Id: arch_pic.c,v 1.24 2009-01-28 12:59:16 potyra Exp $ 
 *
 * Copyright (C) 2005-2009 FAUmachine Team <info@faumachine.org>.
 * This program is free software. You can redistribute it and/or modify it
 * under the terms of the GNU General Public License, either version 2 of
 * the License, or (at your option) any later version. See COPYING.
 */

#ifdef STATE

struct {
	unsigned char step;
	unsigned char instate;
	unsigned char special_fully_nested_mode;
	unsigned char auto_eoi;
	unsigned char special_mask;
	unsigned char poll;
	unsigned char imr;
	unsigned char irr;
	unsigned char isr;
	unsigned char priority_add;
	unsigned char rotate_on_auto_eoi;
	unsigned char vec;
	unsigned char icw3;
	unsigned char ic4;
	unsigned char ris;
	unsigned char elcr;
	unsigned int state_sp;
	unsigned int state_cas;

	unsigned char last_gone;
} NAME;

#endif /* STATE */
#ifdef BEHAVIOR

#define DEBUG_CONTROL_FLOW 0

static int
NAME_(get_irq)(struct cpssp *css)
{
	unsigned char pending;
	unsigned int n;

#if DEBUG_CONTROL_FLOW
	fprintf(stderr, "%s irr=%02x imr=%02x isr=%02x priadd=%d spec=%d\n",
			__FUNCTION__,
			css->NAME.irr, css->NAME.imr, css->NAME.isr,
			css->NAME.priority_add, css->NAME.special_mask);
#endif

	pending = css->NAME.irr & ~css->NAME.imr;

	for (n = 0; ; n++) {
		unsigned int irq;

		if (n == 8) {
			/* No interrupt pending or interrupt masked. */
			return -1;
		}
		irq = (n + css->NAME.priority_add) & 7;
		if (css->NAME.special_fully_nested_mode && (css->NAME.icw3 & (1 << irq))) {
			/* Ignore interrupts from slave. */
			continue;
		}
		if (css->NAME.isr & (1 << irq)) {
			/* Interrupt with higher priority is `in service'. */
			if (css->NAME.special_mask) {
				/* Ignore it... */
				continue;
			} else {
				/* It blocks all lower level interrupts. */
				return -1;
			}
		}
		if (pending & (1 << irq)) {
			/* Interrupt pending and not masked. */
			return irq;
		}
	}
}

/*forward*/ static void
NAME_(update_irq)(struct cpssp *css);

static void
NAME_(irq_set)(struct cpssp *css, unsigned int irq, unsigned int val)
{
	assert(/* 0 <= irq && */ irq < 8);
	assert(/* 0 <= val && */ val < 2);

#if DEBUG_CONTROL_FLOW
	fprintf(stderr, "%s %d %d\n", __FUNCTION__, irq, val);
#endif

	if (((css->NAME.instate >> irq) & 1) == val) {
		/* Nothing changed. */
		return;
	}
	css->NAME.instate &= ~(1 << irq);
	css->NAME.instate |= val << irq;

	if (val == 0) {
		css->NAME.last_gone = irq;
	}
#ifdef ELCR_MASK
	if ((css->NAME.elcr >> irq) & 1) {
		/* Level triggered interrupt. */
		css->NAME.irr &= ~(1 << irq);
		css->NAME.irr |= val << irq;

	} else
#endif
	{
		/* Edge triggered interrupt. */
		if (val) {
			/* Set interrupt request bit. */
			css->NAME.irr |= val << irq;
		}
	}

	NAME_(update_irq)(css);
}

#ifndef MASTER
static void
NAME_(sp_set)(struct cpssp *css, unsigned int val)
{
	css->NAME.state_sp = val;
}
#endif

#if ! defined(MASTER) || ! MASTER
static void
NAME_(cas_in_set)(struct cpssp *css, unsigned int val)
{
	css->NAME.state_cas = val;
}
#endif

static void
NAME_(update_irq)(struct cpssp *css)
{
	int irq;
	unsigned int val;

	irq = NAME_(get_irq)(css);

#ifdef MASTER
	val = MASTER;
#else
	val = css->NAME.state_sp;
#endif
	if (val) {
		/* Set cascade output. */
		if (0 <= irq) {
			NAME_(cas_out_set)(css, irq);
		} else {
			NAME_(cas_out_set)(css, 0);
		}
	}

	/* Set interrupt output. */
	if (0 <= irq) {
		NAME_(irq_out_set)(css, 1);
	} else {
		NAME_(irq_out_set)(css, 0);
	}
}

static void
NAME_(intack)(struct cpssp *css, unsigned int irq)
{
	assert(/* 0 <= irq && */ irq < 8);

	if (css->NAME.auto_eoi) {
		if (css->NAME.rotate_on_auto_eoi) {
			css->NAME.priority_add = (irq + 1) & 7;
		}
	} else {
		css->NAME.isr |= 1 << irq;
	}
#ifdef ELCR_MASK
	if (! ((css->NAME.elcr >> irq) & 1))
#endif
	{
		/* If not level triggered re-set interrupt request. */
		css->NAME.irr &= ~(1 << irq);
	}
}

static void
NAME_(inta_begin)(struct cpssp *css, uint8_t *valp)
{
	int irq;
	unsigned int val;

#if DEBUG_CONTROL_FLOW
	fprintf(stderr, "%s called()\n", __FUNCTION__);
#endif

#ifdef MASTER
	val = MASTER;
#else
	val = css->NAME.state_sp;
#endif

	if (val) {
		/*
		 * Master Mode
		 */
		irq = NAME_(get_irq)(css);
		if (irq < 0) {
			/* Spurious interrupt. */
			fprintf(stderr, "WARNING: " SNAME ": spurious interrupt (%d).\n",
					css->NAME.last_gone);
			*valp = css->NAME.vec + 7;

		} else {
			if (! ((css->NAME.icw3 >> irq) & 1)) {
				/* Master's interrupt. */
				*valp = css->NAME.vec + irq;
			}
		}
	} else {
		/*
		 * Slave Mode
		 */
		if (css->NAME.state_cas == css->NAME.icw3) {
			irq = NAME_(get_irq)(css);
			if (irq < 0) {
				/* Spurious interrupt. */
				fprintf(stderr, "WARNING: " SNAME ": spurious interrupt (%d).\n",
						css->NAME.last_gone);
				*valp = css->NAME.vec + 7;

			} else {
				*valp = css->NAME.vec + irq;
			}
		}
	}

#if DEBUG_CONTROL_FLOW
	fprintf(stderr, "%s: Reading 0x%02x\n",
			__FUNCTION__, *valp);
#endif
}

static void
NAME_(inta_end)(struct cpssp *css)
{
	int irq;
	unsigned int val;

#if DEBUG_CONTROL_FLOW
	fprintf(stderr, "%s called()\n", __FUNCTION__);
#endif

#ifdef MASTER
	val = MASTER;
#else
	val = css->NAME.state_sp;
#endif

	if (val) {
		/*
		 * Master Mode
		 */
		irq = NAME_(get_irq)(css);
		if (irq < 0) {
			/* Spurious interrupt. */
			/* No acknowledge. */

		} else {
			NAME_(intack)(css, irq);
			NAME_(update_irq)(css);
		}
	} else {
		/*
		 * Slave Mode
		 */
		if (css->NAME.state_cas == css->NAME.icw3) {
			irq = NAME_(get_irq)(css);
			if (irq < 0) {
				/* Spurious interrupt. */
				/* No acknowledge. */

			} else {
				NAME_(intack)(css, irq);
				NAME_(update_irq)(css);
			}
		}
	}

#if DEBUG_CONTROL_FLOW
	fprintf(stderr, "%s\n", __FUNCTION__);
#endif
}

static void
NAME_(inb)(struct cpssp *css, unsigned char *valp, unsigned short port)
{
	if (css->NAME.poll) {
		/*
		 * Polling active.
		 * Acknowledge interrupt.
		 */
		css->NAME.poll = 0;
		NAME_(inta_begin)(css, valp);
		NAME_(inta_end)(css);

	} else {
		/*
		 * No polling active.
		 * Do normal work.
		 */
		switch (port) {
		case 0:
			if (css->NAME.ris) {
				*valp = css->NAME.isr;
			} else {
				*valp = css->NAME.irr;
			}
			break;
		case 1:
			*valp = css->NAME.imr;
			break;
#ifdef ELCR_MASK			
		case 2:
			/* ELCR - Edge/Level Control Register */
			*valp = css->NAME.elcr;
			break;
#endif			
		default:
			assert(0); /* Cannot happen. */
		}
	}

#if DEBUG_CONTROL_FLOW
	fprintf(stderr, "%s: Reading 0x%02x from 0x%04x\n",
			__FUNCTION__, *valp, port);
#endif
}

static void
NAME_(outb)(struct cpssp *css, unsigned char value, unsigned short port)
{
	unsigned int irq;
	unsigned int n;

#if DEBUG_CONTROL_FLOW
	fprintf(stderr, "%s: Writing 0x%02x to 0x%04x\n",
			__FUNCTION__, value, port);
#endif

	switch (port) {
	case 0:
		if (value & 0x10) {
			/*
			 * ICW1
			 */
			/*
			 * Check value written.
			 * Level sensitive IRQ not supported.
			 * Single mode not supported.
			 */
			if (((value >> 3) & 1) == 1
			 || ((value >> 1) & 1) == 1) {
				fprintf(stderr,
					"WARNING: ICW1 of PIC set to 0x%02x.\n",
					value);
			}

			/* Reset PIC? FIXME VOSSI */

			css->NAME.ic4 = (value >> 0) & 1;

			css->NAME.step = 1;

		} else if (! (value & 0x08)) {
			/*
			 * OCW2
			 */
			/* Check value written. */
			if ((value & 0x18) != 0x00) {
				fprintf(stderr,
					"WARNING: OCW2 of PIC set to 0x%02x.\n",
					value);
			}

			/*
			 * To be able to generate a new pulse if more
			 * interrupts pending. INT is low for some cycles.
			 */
			NAME_(irq_out_set)(css, 0);

			switch ((value >> 5) & 0x7) {
			case 0x0:
				/* Rotate in automatic EOI mode (clear) */
			case 0x4:
				/* Rotate in automatic EOI mode (set) */
				css->NAME.rotate_on_auto_eoi = (value >> 7) & 1;
				break;

			case 0x1:
				/* Non-specific EOI command */
			case 0x5:
				/* Rotate on non-specific EOI command */
				for (n = 0; ; n++) {
					if (n == 8) {
						/* No interrupt pending. */
						break;
					}
					irq = (n + css->NAME.priority_add) & 7;
					if (css->NAME.isr & (1 << irq)) {
						css->NAME.isr &= ~(1 << irq);
						if ((value >> 7) & 1) {
							css->NAME.priority_add = (irq + 1) & 7;
						}
						break;
					}
				}
				break;

			case 0x2:
				/* No operation */
			case 0x6:
				/* Set priority command */
				irq = value & 7;
				if ((value >> 7) & 1) {
					css->NAME.priority_add = (irq + 1) & 7;
				}
				break;

			case 0x3:
				/* Specific EOI command */
			case 0x7:
				/* Rotate on specific EOI command */
				irq = value & 7;
				css->NAME.isr &= ~(1 << irq);
				if ((value >> 7) & 1) {
					css->NAME.priority_add = (irq + 1) & 7;
				}
				break;

			default:
				assert(0);	/* Cannot happen. */
			}
		} else {
			/*
			 * OCW3
			 */
			/*
			 * Check value written.
			 */
			if ((value & 0x98) != 0x08) {
				fprintf(stderr,
					"WARNING: OCW3 of PIC set to 0x%02x.\n",
					value);
			}

			if ((value >> 6) & 1) {
				css->NAME.special_mask = (value >> 5) & 1;
			}
			css->NAME.poll = (value >> 2) & 1;
			if ((value >> 1) & 1) {
				css->NAME.ris = (value >> 0) & 1;
			}
		}

		NAME_(update_irq)(css);
		break;

	case 1:
		switch (css->NAME.step) {
		case 0:
			/*
			 * OCW1
			 */
			css->NAME.imr = value;
			break;

		case 1:
			/* ICW2 */
			if ((value & 0x07) != 0x00) {
				fprintf(stderr,
					"WARNING: ICW2 of PIC set to 0x%02x.\n",
					value);
			}

			/* We assume 8086/8088 mode. */
			css->NAME.vec = value & 0xf8;

			css->NAME.step = 2;
			break;

		case 2:
			/*
			 * ICW3
			 */
			css->NAME.icw3 = value;

			/* Ignore value written. */

			if (css->NAME.ic4) {
				css->NAME.step = 3;
				break;
			} else {
				value = 0;
				/*FALLTRHOUGH*/
			}

		case 3:
			/*
			 * ICW4
			 */
			/*
			 * Check config byte.
			 * Bit 7-5 should be zero (reserved).
			 * Bit 0 should be one (8086/8088 mode).
			 */
			if ((value & 0xe1) != 0x01) {
				fprintf(stderr,
					"WARNING: ICW4 of PIC set to 0x%02x.\n",
					value);
			}

			css->NAME.special_fully_nested_mode
				= (value >> 4) & 1;
			/* Bit 2/3: Buffered mode? FIXME VOSSI */
			css->NAME.auto_eoi
				= (value >> 1) & 1;

			if (css->NAME.special_fully_nested_mode) {
				fprintf(stderr,
					"WARNING: PIC entered special fully nested mode.\n");
				fprintf(stderr,
					"WARNING: Not implemented yet.\n");
			}

			css->NAME.irr = 0x00;
			css->NAME.imr = 0x00;
			css->NAME.priority_add = 0;
			css->NAME.special_mask = 0;
			css->NAME.ris = 0;

			css->NAME.step = 0;
			break;

		default:
			assert(0);
		}

		NAME_(update_irq)(css);
		break;
#ifdef ELCR_MASK
	case 2:
		/* ELCR - Edge/Level Control Register */
		if (value & ~ELCR_MASK) {
			fprintf(stderr, "WARNING: PIC: Writing 0x%02x to edge/level control register.\n", value);
			value &= ELCR_MASK;
		}

		css->NAME.elcr = value;

		NAME_(update_irq)(css);
		break;
#endif
	default:
		assert(0); /* Cannot happen. */
	}
}

extern void
NAME_(isa_bus_int0_set)(struct cpssp *css, unsigned int val)
{
	NAME_(irq_set)(css, 0x0, val);
}

extern void
NAME_(isa_bus_int1_set)(struct cpssp *css, unsigned int val)
{
	NAME_(irq_set)(css, 0x1, val);
}

static void
NAME_(isa_bus_int2_set)(struct cpssp *css, unsigned int val)
{
	NAME_(irq_set)(css, 0x2, val);
}

extern void
NAME_(isa_bus_int3_set)(struct cpssp *css, unsigned int val)
{
	NAME_(irq_set)(css, 0x3, val);
}

extern void
NAME_(isa_bus_int4_set)(struct cpssp *css, unsigned int val)
{
	NAME_(irq_set)(css, 0x4, val);
}

extern void
NAME_(isa_bus_int5_set)(struct cpssp *css, unsigned int val)
{
	NAME_(irq_set)(css, 0x5, val);
}

extern void
NAME_(isa_bus_int6_set)(struct cpssp *css, unsigned int val)
{
	NAME_(irq_set)(css, 0x6, val);
}

extern void
NAME_(isa_bus_int7_set)(struct cpssp *css, unsigned int val)
{
	NAME_(irq_set)(css, 0x7, val);
}

static void
NAME_(reset)(struct cpssp *css)
{
	/* More to reset? - FIXME VOSSI */
	css->NAME.step = 0;
	css->NAME.instate = 0;
	css->NAME.special_mask = 0;
	css->NAME.imr = 0xff;
	css->NAME.irr = 0x00;
	css->NAME.isr = 0x00;
	css->NAME.vec = 0;
	css->NAME.ic4 = 0;
	css->NAME.ris = 0;
	css->NAME.elcr = 0;
}

static void
NAME_(init)(struct cpssp *css)
{
	/* Nothing to do... */
}

#undef DEBUG_CONTROL_FLOW

#endif /* BEHAVIOR */
