/* $Id: arch_kbd.c,v 1.42 2009-05-09 19:01:11 vrsieh Exp $ 
 *
 * Copyright (C) 2006-2009 FAUmachine Team <info@faumachine.org>.
 * This program is free software. You can redistribute it and/or modify it
 * under the terms of the GNU General Public License, either version 2 of
 * the License, or (at your option) any later version. See COPYING.
 */

#if defined(STATE)

struct {
	int n_reset_state;
	int a20gate_state;
	int f0_received;
	int translate;
	int auxdis;
	int kbddis;
	int selftest_done;
	int auxint;
	int kbdint;

	enum {
		KEYBOARD,
		MOUSE,
		OUT_PORT,
		WRITE_MODE,
		AUX_OBUF,
		/* Controller input buffer */
		IN_BUF, 
	} redir;

	uint8_t data[2];
	int data_full[2];

	unsigned char lastdata;

	unsigned char lastwrittento; /* Which port was last written
				      * to - used in the status reg */

	enum {
		AT,
		MCA	/* PS/2 */
	} ctrl_mode;
} NAME;

#elif defined(BEHAVIOR)

#define DEBUG_CONTROL_FLOW	0
#define DEBUG_IRQ		0

/* Keyboard Controller Commands */
/* see PC-Hardware 5th revision, pp. 1034 */
#define KBD_CCMD_READ_MODE	0x20 /* Read mode bits */
#define KBD_CCMD_WRITE_MODE	0x60 /* Write mode bits */
#define KBD_CCMD_GET_VERSION	0xa1 /* Get controller version */
#define KBD_CCMD_RSV1		0xa4 /* reserved */
#define KBD_CCMD_RSV2		0xa5 /* reserved */
#define KBD_CCMD_RSV3		0xa6 /* reserved */
#define KBD_CCMD_MOUSE_DISABLE	0xa7 /* Disable mouse interface */
#define KBD_CCMD_MOUSE_ENABLE	0xa8 /* Enable mouse interface */
#define KBD_CCMD_TEST_MOUSE	0xa9 /* Mouse interface test */
#define KBD_CCMD_SELF_TEST	0xaA /* Controller self test */
#define KBD_CCMD_KBD_TEST	0xaB /* Keyboard interface test */
#define KBD_CCMD_KBD_DISABLE	0xaD /* Keyboard interface disable */
#define KBD_CCMD_KBD_ENABLE	0xaE /* Keyboard interface enable */
#define KBD_CCMD_RESET_CLINE	0xb0 /* reset controller line, bits 2-0 select line */
#define KBD_CCMD_SET_CLINE	0xb8 /* set controller line, bits 2-0 select line  */
#define KBD_CCMD_RDIN		0xc0 /* read input port */
#define KBD_CCMD_PLINH		0xc2 /* poll input port high */
#define KBD_CCMD_PLINL		0xc3 /* poll input port low */
#define KBD_CCMD_READ_CTRL_MODE	0xcA /* Read keyboard controller mode */
#define KBD_CCMD_RDOUT		0xd0 /* read output port */
#define KBD_CCMD_WROUT		0xd1 /* write output port */
#define KBD_CCMD_WRKOR		0xd2 /* write keyboard output register */
#define KBD_CCMD_WRITE_AUX_OBUF	0xd3 /* write auxiliary device output register*/
#define KBD_CCMD_WRITE_MOUSE	0xd4 /* write the following byte to the mouse */
#define KBD_CCMD_RDTST		0xe0 /* read test input port */
#define KBD_CCMD_PLSE		0xf0 /* pulse: bits 3-0 select which bit */

static const uint8_t NAME_(table)[256] = {
	0xff,0x43,0x41,0x3f,0x3d,0x3b,0x3c,0x58,
	0x64,0x44,0x42,0x40,0x3e,0x0f,0x29,0x59,
	0x65,0x38,0x2a,0x70,0x1d,0x10,0x02,0x5a,
	0x66,0x71,0x2c,0x1f,0x1e,0x11,0x03,0x5b,
	0x67,0x2e,0x2d,0x20,0x12,0x05,0x04,0x5c,
	0x68,0x39,0x2f,0x21,0x14,0x13,0x06,0x5d,
	0x69,0x31,0x30,0x23,0x22,0x15,0x07,0x5e,
	0x6a,0x72,0x32,0x24,0x16,0x08,0x09,0x5f,
	0x6b,0x33,0x25,0x17,0x18,0x0b,0x0a,0x60,
	0x6c,0x34,0x35,0x26,0x27,0x19,0x0c,0x61,
	0x6d,0x73,0x28,0x74,0x1a,0x0d,0x62,0x6e,
	0x3a,0x36,0x1c,0x1b,0x75,0x2b,0x63,0x76,
	0x55,0x56,0x77,0x78,0x79,0x7a,0x0e,0x7b,
	0x7c,0x4f,0x7d,0x4b,0x47,0x7e,0x7f,0x6f,
	0x52,0x53,0x50,0x4c,0x4d,0x48,0x01,0x45,
	0x57,0x4e,0x51,0x4a,0x37,0x49,0x46,0x54,
	0x80,0x81,0x82,0x41,0x54,0x85,0x86,0x87,
	0x88,0x89,0x8a,0x8b,0x8c,0x8d,0x8e,0x8f,
	0x90,0x91,0x92,0x93,0x94,0x95,0x96,0x97,
	0x98,0x99,0x9a,0x9b,0x9c,0x9d,0x9e,0x9f,
	0xa0,0xa1,0xa2,0xa3,0xa4,0xa5,0xa6,0xa7,
	0xa8,0xa9,0xaa,0xab,0xac,0xad,0xae,0xaf,
	0xb0,0xb1,0xb2,0xb3,0xb4,0xb5,0xb6,0xb7,
	0xb8,0xb9,0xba,0xbb,0xbc,0xbd,0xbe,0xbf,
	0xc0,0xc1,0xc2,0xc3,0xc4,0xc5,0xc6,0xc7,
	0xc8,0xc9,0xca,0xcb,0xcc,0xcd,0xce,0xcf,
	0xd0,0xd1,0xd2,0xd3,0xd4,0xd5,0xd6,0xd7,
	0xd8,0xd9,0xda,0xdb,0xdc,0xdd,0xde,0xdf,
	0xe0,0xe1,0xe2,0xe3,0xe4,0xe5,0xe6,0xe7,
	0xe8,0xe9,0xea,0xeb,0xec,0xed,0xee,0xef,
	0xf0,0xf1,0xf2,0xf3,0xf4,0xf5,0xf6,0xf7,
	0xf8,0xf9,0xfa,0xfb,0xfc,0xfd,0xfe,0xff
};

static void
NAME_(irqs_update)(struct cpssp *cpssp)
{
	/* See note in inb function! */
	if (cpssp->NAME.data_full[0]
	 && cpssp->NAME.kbdint) {
		/* Signal interrupt. */
		NAME_(int1_set)(cpssp, 1);
		NAME_(intC_set)(cpssp, 0);
	} else {
		NAME_(int1_set)(cpssp, 0);
		if (cpssp->NAME.data_full[1]
		 && cpssp->NAME.auxint) {
			/* Signal interrupt. */
			NAME_(intC_set)(cpssp, 1);
		} else {
			NAME_(intC_set)(cpssp, 0);
		}
	}
}

static void
NAME_(recalculate_clkrunning)(struct cpssp *cpssp, unsigned int line)
{
	int clkrunning;


	switch (line) {
	case 0:
		clkrunning = ! cpssp->NAME.data_full[line]
			&& ! cpssp->NAME.kbddis;
		NAME_(kbd_clkrunning)(cpssp, clkrunning);
		break;
	case 1:
		clkrunning = ! cpssp->NAME.data_full[line]
			&& ! cpssp->NAME.auxdis;
		NAME_(mouse_clkrunning)(cpssp, clkrunning);
		break;
	default:
		assert(0);
	}
}

static void
NAME_(push)(struct cpssp *cpssp, unsigned int buf, unsigned char val)
{
	if (cpssp->NAME.data_full[buf]) {
		faum_log(FAUM_LOG_WARNING, SNAME,
			 (buf ? "mouse" : "keyboard"),
			 "Keyboard controller: %s buffer overflow\n",
			 (buf ? "mouse" : "keyboard"));
		return;
	}

	cpssp->NAME.data[buf] = val;
	cpssp->NAME.data_full[buf] = 1;

	NAME_(recalculate_clkrunning)(cpssp, buf);

	NAME_(irqs_update)(cpssp);
}

static unsigned char
NAME_(pop)(struct cpssp *cpssp, unsigned int buf)
{
	unsigned char value;

	assert(cpssp->NAME.data_full[buf]);

	value = cpssp->NAME.data[buf];
	cpssp->NAME.data_full[buf] = 0;

	NAME_(recalculate_clkrunning)(cpssp, buf);

	NAME_(irqs_update)(cpssp);
	
	return value;
}

static void
NAME_(do_keyboard_command)(struct cpssp *cpssp, unsigned char value)
{
	if (DEBUG_CONTROL_FLOW) {
		fprintf(stderr, "%s: %llu: value=0x%02x\n", __FUNCTION__,
				time_virt(), value);
	}

	/* Any keyboard command enables the keyboard again. */
	cpssp->NAME.kbddis = 0;
	NAME_(recalculate_clkrunning)(cpssp, 0);

	NAME_(kbd_send)(cpssp, value);

	if (DEBUG_CONTROL_FLOW) {
		fprintf(stderr, "%s: %llu: return\n", __FUNCTION__,
				time_virt());
	}
}

/*
 * See PC Hardware (5. Auflage), S. 1048.
 */
static void
NAME_(do_mouse_command)(struct cpssp *cpssp, unsigned char value)
{
	if (DEBUG_CONTROL_FLOW) {
		fprintf(stderr, "%s: %llu: value=0x%02x\n", __FUNCTION__,
				time_virt(), value);
	}

	/* Unlike with the keyboard, a mouse command does not reenable the
	 * mouse automatically */

	NAME_(mouse_send)(cpssp, value);

	if (DEBUG_CONTROL_FLOW) {
		fprintf(stderr, "%s: %llu: return\n", __FUNCTION__,
				time_virt());
	}
}

static void
NAME_(do_controller_command)(struct cpssp *cpssp, unsigned char value)
{
	if (DEBUG_CONTROL_FLOW) {
		fprintf(stderr, "%s: %llu: value=0x%02x\n", __FUNCTION__,
				time_virt(), value);
	}

	/*
	 * Commands used for keyboard initialization in initialize_keyboard()
	 * in drivers/char/pc_keyb.c
	 */
	switch (value) {
	case KBD_CCMD_READ_MODE:	/* 0x20: read mode register */
		value = 0;

		/* Bit 7: Unused */
		value |= 0 << 7;

		/* Translate? */
		value |= cpssp->NAME.translate << 6;

		/* Mouse disabled? */
		value |= cpssp->NAME.auxdis << 5;

		/* Keyboard disabled? */
		value |= cpssp->NAME.kbddis << 4;

		/* Ignore Keyboard Lock */
		value |= 0 << 3;

		/* Selftest done? */
		value |= cpssp->NAME.selftest_done << 2;

		/* Mouse interrupt enabled? */
		value |= cpssp->NAME.auxint << 1;

		/* Keyboard interrupt enabled? */
		value |= cpssp->NAME.kbdint << 0;

		NAME_(push)(cpssp, 0, value);
		break;

	case KBD_CCMD_WRITE_MODE:	/* 0x60: write mode register */
		cpssp->NAME.redir = WRITE_MODE;
		break;

	case KBD_CCMD_MOUSE_DISABLE:	/* 0xa7: disable mouse */
		cpssp->NAME.auxdis = 1;
		NAME_(recalculate_clkrunning)(cpssp, 1);
		break;

	case KBD_CCMD_MOUSE_ENABLE:	/* 0xa8: enable mouse */
		cpssp->NAME.auxdis = 0;
		NAME_(recalculate_clkrunning)(cpssp, 1);
		break;

	case KBD_CCMD_TEST_MOUSE:	/* 0xa9: Mouse Interface Test */
		NAME_(push)(cpssp, 0, 0x00); /* "All OK" */
		break;

	case KBD_CCMD_SELF_TEST:	/* 0xaa: Controller Self Test */
		/*
		 * Bit 0 of the controller status register is set to 1 upon
		 * completion of self test. Linux driver doesn't check this.
		 */
		cpssp->NAME.selftest_done = 1;
		NAME_(push)(cpssp, 0, 0x55); /* "All OK" */
		break;

	case KBD_CCMD_KBD_TEST:		/* 0xab: Keyboard Interface Test */
		NAME_(push)(cpssp, 0, 0x00); /* "All OK" */
		break;

	case KBD_CCMD_KBD_DISABLE:	/* 0xad: Disable Keyboard */
		cpssp->NAME.kbddis = 1;
		NAME_(recalculate_clkrunning)(cpssp, 0);
		break;

	case KBD_CCMD_KBD_ENABLE:	/* 0xae: Enable Keyboard */
		cpssp->NAME.kbddis = 0;
		NAME_(recalculate_clkrunning)(cpssp, 0);
		break;

	case KBD_CCMD_RESET_CLINE ... (KBD_CCMD_RESET_CLINE | 0x07):
				/* 0xb0 ... 0xb7 write 0 to controller lines */
				/* order is: I0, I1, I2, I3, O2, O3, I4, I5 */

		/* FIXME _really_ reset lines here */
		faum_log(FAUM_LOG_WARNING, SNAME, "",
				"Trying to reset controller port: 0x%02x\n",
				value);

		NAME_(push)(cpssp, 0, 0); /* (one byte of garbage) */
		break;

	case KBD_CCMD_SET_CLINE ... (KBD_CCMD_SET_CLINE | 0x07):
				/* 0xb8 ... 0xbf write 1 to controller lines */
				/* (order: see above) */

		/* FIXME _really_ set lines here */
		faum_log(FAUM_LOG_WARNING, SNAME, "",
				"Trying to set controller port: 0x%02x\n",
				value);

		NAME_(push)(cpssp, 0, 0); /* (one byte of garbage) */
		break;

	case KBD_CCMD_RDIN:	/* 0xc0: Read input register. */
		value = 0;

		/* Keyboard unlocked? */
		value |= 1 << 7; /* Yes */

		/* Display */
		value |= 0 << 6; /* CGA -- FIXME */

		/* Diagnostic loop jumper? */
		value |= 1 << 5; /* No */

		/* RAM on motherboard? */
		value |= 0 << 4; /* 512KB */

		/* Bit 3: Unused */
		value |= 0 << 3;

		/* Keyboard power failure? */
		value |= 0 << 2; /* No */

		/* Mouse Data */
		value |= 0 << 1; /* FIXME */

		/* Keyboard Data */
		value |= 0 << 0; /* FIXME */

		NAME_(push)(cpssp, 0, value);
		break;

	case KBD_CCMD_READ_CTRL_MODE:	/* 0xca Read controller mode */
		NAME_(push)(cpssp, 0, cpssp->NAME.ctrl_mode);
		break;

	case KBD_CCMD_RDOUT:	/* 0xd0: read output port */
		value = 0;

		/* Keyboard Data */
		value |= 0 << 7; /* High-Z -- FIXME */

		/* Keyboard Clk */
		value |= cpssp->NAME.kbddis << 6;

		/* IRQ 12 active? */
		value |= 0 << 5; /* FIXME */

		/* IRQ 1 active? */
		value |= 0 << 4; /* FIXME */

		/* Mouse Clk */
		value |= cpssp->NAME.auxdis << 3;

		/* Mouse Data */
		value |= 0 << 2; /* High-Z -- FIXME */

		/* A20 Gate */
		value |= cpssp->NAME.a20gate_state << 1;

		/* System Reset Pin */
		value |= cpssp->NAME.n_reset_state << 0;

		NAME_(push)(cpssp, 0, value);
		break;

	case KBD_CCMD_WROUT:	/* 0xd1: write output port */
		cpssp->NAME.redir = OUT_PORT;
		break;

	case KBD_CCMD_WRKOR:	/* 0xd2: Write keyboard buffer */
		cpssp->NAME.redir = IN_BUF;
		break;

	case KBD_CCMD_WRITE_AUX_OBUF:	/* 0xd3: write to output buffer as if */
					/* initiated by the auxiliary device */
		cpssp->NAME.redir = AUX_OBUF;
		break;

	case KBD_CCMD_WRITE_MOUSE:	/* 0xd4 - write to aux device FIXME */
		cpssp->NAME.redir = MOUSE;
		break;

	case KBD_CCMD_PLSE | 0x0e:	/* 0xfe: pulse reset line */
		cpssp->NAME.n_reset_state = 0;
		NAME_(n_reset_set)(cpssp, 0);
		cpssp->NAME.n_reset_state = 1;
		NAME_(n_reset_set)(cpssp, 1);
		break;

	case KBD_CCMD_PLSE | 0x0f:	/* 0xff: possibly sortof a "reset" */
		/* Ignore that - I don't know what is its use. */
		/* Ignored by QEMU as well. Seems to be OK to ignore it. */
		/* "the internet" says it generates a short pulse on all the
		 * controllers output lines? - probably some sort of reset */
		break;
	
	case KBD_CCMD_PLINH: /* Other commands: */
	case KBD_CCMD_PLINL:
	case KBD_CCMD_RDTST:
	case KBD_CCMD_PLSE | 0x00:
	case KBD_CCMD_PLSE | 0x01:
	case KBD_CCMD_PLSE | 0x02:
	case KBD_CCMD_PLSE | 0x03:
	case KBD_CCMD_PLSE | 0x04:
	case KBD_CCMD_PLSE | 0x05:
	case KBD_CCMD_PLSE | 0x06:
	case KBD_CCMD_PLSE | 0x07:
	case KBD_CCMD_PLSE | 0x08:
	case KBD_CCMD_PLSE | 0x09:
	case KBD_CCMD_PLSE | 0x0a:
	case KBD_CCMD_PLSE | 0x0b:
	case KBD_CCMD_PLSE | 0x0c:
	case KBD_CCMD_PLSE | 0x0d:
	/* case KBD_CCMD_PLSE | 0x0e: see above */
	/* case KBD_CCMD_PLSE | 0x0f: see above */
		/* a7 */
		faum_log(FAUM_LOG_WARNING, SNAME, "",
			 "Unimplemented keyboard controller command: 0x%02x\n",
			 value);
		break;

	default:
		faum_log(FAUM_LOG_WARNING, SNAME, "",
			 "Unimplemented keyboard controller command: 0x%02x\n",
			 value);
		break;
	}

	if (DEBUG_CONTROL_FLOW) {
		fprintf(stderr, "%s: %llu: return\n", __FUNCTION__,
				time_virt());
	}
}

static uint8_t
NAME_(inb)(struct cpssp *cpssp, unsigned short addr)
{
	uint8_t value;

	if (DEBUG_CONTROL_FLOW) {
		fprintf(stderr, "%s: %llu: addr=0x%02x\n", __FUNCTION__,
				time_virt(), addr);
	}

	/*
	 * Note:
	 * First look for data in keyboard buffer.
	 * After that look for data in mouse buffer.
	 * Otherwise OpenBSD will get wrong data if mouse is moved.
	 */

	switch (addr) {
	case 0:
		/* See note above! */
		if (cpssp->NAME.data_full[0]) {
			cpssp->NAME.lastdata = NAME_(pop)(cpssp, 0);
		} else if (cpssp->NAME.data_full[1]) {
			cpssp->NAME.lastdata = NAME_(pop)(cpssp, 1);
		}
		value = cpssp->NAME.lastdata;
		break;

	case 1:
		value = 0;

		/* Parity Error */
		value |= 0 << 7; /* No */

		/* Timeout Error */
		value |= 0 << 6; /* No */

		/* AUX buffer full? (See note above!) */
		if (! cpssp->NAME.data_full[0]
		 && cpssp->NAME.data_full[1]) {
			value |= 1 << 5;
		}

		/* Keyboard unlocked? */
		value |= 1 << 4;

		/* Last write was a command write (0=data). */
		value |= cpssp->NAME.lastwrittento << 3;

		/* Selftest done? */
		value |= cpssp->NAME.selftest_done << 2;

		/* Input buffer full? */
		value |= 0 << 1; /* FIXME */

		/* Output buffer full? (See note above!) */
		if (cpssp->NAME.data_full[0]
		 || cpssp->NAME.data_full[1]) {
			value |= 1 << 0;
		}
		break;

	default:
		assert(0);
	}

	if (DEBUG_CONTROL_FLOW) {
		fprintf(stderr, "%s: %llu: value=0x%02x\n", __FUNCTION__,
				time_virt(), value);
	}

	return value;
}

static void
NAME_(outb)(struct cpssp *cpssp, unsigned char value, unsigned short addr)
{
	if (DEBUG_CONTROL_FLOW) {
		fprintf(stderr, "%s: %llu: addr=0x%02x, value=0x%02x\n",
				__FUNCTION__, time_virt(), addr, value);
	}

	cpssp->NAME.lastwrittento = addr;

	switch (addr) {
	case 0:
		switch (cpssp->NAME.redir) {
		case WRITE_MODE:
			cpssp->NAME.translate = (value >> 6) & 1;

			cpssp->NAME.auxdis = (value >> 5) & 1;

			cpssp->NAME.kbddis = (value >> 4) & 1;

			/* Bit 3: always '0'. */

			cpssp->NAME.selftest_done = (value >> 2) & 1;

			cpssp->NAME.auxint = (value >> 1) & 1;

			cpssp->NAME.kbdint = (value >> 0) & 1;

			NAME_(recalculate_clkrunning)(cpssp, 1);
			NAME_(recalculate_clkrunning)(cpssp, 0);
			NAME_(irqs_update)(cpssp);
			break;

		case OUT_PORT:
			/* Several other bits -- FIXME */
			cpssp->NAME.a20gate_state = (value >> 1) & 1;
			NAME_(a20gate_set)(cpssp, cpssp->NAME.a20gate_state);
			break;

		case AUX_OBUF:
			NAME_(push)(cpssp, 1, value);
			break;

		case KEYBOARD:
			NAME_(do_keyboard_command)(cpssp, value);
			break;

		case MOUSE:
			NAME_(do_mouse_command)(cpssp, value);
			break;

		case IN_BUF:
			NAME_(push)(cpssp, 0, value);
			break;

		default:
			assert(0);
		}
		cpssp->NAME.redir = KEYBOARD;
		break;

	case 1:
		NAME_(do_controller_command)(cpssp, value);
		break;

	default:
		assert(0);
	}

	if (DEBUG_CONTROL_FLOW) {
		fprintf(stderr, "%s: %llu: return\n",
				__FUNCTION__, time_virt());
	}
}

static void
NAME_(kbd_recv)(struct cpssp *cpssp, uint8_t byte)
{
#if DEBUG_IRQ
	faum_log(FAUM_LOG_DEBUG, SNAME, "keyboard",
			 "byte for keyboard: %02x\n", byte);
#endif
	if (cpssp->NAME.translate) {
		if (cpssp->NAME.f0_received) {
			byte = NAME_(table)[byte];
			byte |= 0x80;
			cpssp->NAME.f0_received = 0;
		} else if (byte == 0xf0) {
			cpssp->NAME.f0_received = 1;
			return;
		} else {
			byte = NAME_(table)[byte];
		}
	}

	NAME_(push)(cpssp, 0, byte);
}

static void
NAME_(mouse_recv)(struct cpssp *cpssp, uint8_t byte)
{
#if DEBUG_IRQ
	faum_log(FAUM_LOG_DEBUG, SNAME, "mouse",
			 "byte for mouse: %02x\n", byte);
#endif

	NAME_(push)(cpssp, 1, byte);
}

static void
NAME_(reset)(struct cpssp *cpssp)
{
	cpssp->NAME.f0_received = 1;
	cpssp->NAME.translate = 1;
	cpssp->NAME.selftest_done = 0;
	cpssp->NAME.ctrl_mode = MCA;
#if 0
	cpssp->NAME.kbddis = 1;
	cpssp->NAME.auxdis = 1;
	NAME_(recalculate_clkrunning)(cpssp, 0);
	NAME_(recalculate_clkrunning)(cpssp, 1);
#endif
}

#undef DEBUG_IRQ
#undef DEBUG_CONTROL_FLOW

#endif /* defined(BEHAVIOR) */
